/**************************************************************************

   fotoxx      digital photo edit program

   Copyright 2007, 2008, 2009  Michael Cornelison
   source URL:  kornelix.squarespace.com
   contact: kornelix@yahoo.de
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

***************************************************************************/

#include <FreeImage.h>
#include "zfuncs.h"

#define fversion "fotoxx v.8.7  2009.11.25"                                //  version and release date
#define flicense "Free software - GNU General Public License v.3"
#define fhomepage "http://kornelix.squarespace.com/fotoxx"
#define ftranslators "Translators:"                                        \
   "\n Stanislas Zeller, Antonio Sánchez, Miguel Bouzada,"                 \
   "\n The Hamsters, Helge Soencksen, Jie Luo, Pavel Cidlina"              \
   "\n Eugenio Baldi, Santiago Torres Batán"
#define fcredits "Programs used: FreeImage, ufraw, exiftool"
#define fcontact "Bug reports: kornelix@yahoo.de"

//  GTK definitions

#define PXB GdkPixbuf
#define textwin GTK_TEXT_WINDOW_TEXT                                       //  GDK window of GTK text view
#define nodither GDK_RGB_DITHER_NONE
#define ALWAYS GTK_POLICY_ALWAYS
#define NEVER GTK_POLICY_NEVER
#define colorspace GDK_COLORSPACE_RGB
#define lineattributes GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER

//  FREEIMAGE definitions

#define FIF FREE_IMAGE_FORMAT                                              //  file types, jpeg or tiff
#define FIB FIBITMAP                                                       //  FreeImage bitmap
#define RR FI_RGBA_RED                                                     //  FIB-24 pixel color order
#define GG FI_RGBA_GREEN
#define BB FI_RGBA_BLUE

//  fotoxx definitions

#define mwinww 800                                                         //  main window default size
#define mwinhh 600
#define mega (1024 * 1024)
#define track(name) printf("track: %s \n",#name);                          //  debug path tracker
#define ftrash "Desktop/fotoxx-trash"                                      //  trash folder location
#define Pundo_max 50                                                       //  max undo/redo files
#define def_jpeg_quality "90"                                              //  default jpeg quality   v.8.3

#define max_images 100000                                                  //  max. images (m_index_tt())  v.8.4
#define maxtag1 30                                                         //  max tag cc for one tag
#define maxtag2 300                                                        //  max tag cc for one image file
#define maxtag3 20000                                                      //  max tag cc for all image files
#define maxtag4 200                                                        //  max tag cc for search tags
#define maxtag5 200                                                        //  max tag cc for recent tags
#define maxntags 1000                                                      //  max tag count for all images
#define maxtagF 300                                                        //  max image search /path*/file*

#define exif_tags_key "UserComment"                                        //  EXIF key for image tags
#define exif_date_key "DateTimeOriginal"                                   //  EXIF key for image date
#define exif_width_key "ExifImageWidth"                                    //  EXIF key for image width
#define exif_height_key "ExifImageHeight"                                  //  EXIF key for image height
#define exif_orientation_key "Orientation"                                 //  EXIF key for image orientation

#define bmpixel(rgb,px,py) ((uint16 *) rgb->bmp+((py)*(rgb->ww)+(px))*3)   //  return RGB pixel[3] at (px,py)
#define brightness(pix) (0.25*pix[0]+0.65*pix[1]+0.10*pix[2])              //  pixel brightness, 0-64K
#define redness(pix) (25 * pix[0] / (brightness(pix)+1))                   //  pixel redness, 0-100%

#define pixed_undomaxmem (100 * mega)                                      //  pixel edit max. memory alloc.
#define pixed_undomaxpix (mega)                                            //  pixel edit max. pixel blocks

namespace      image_navi {                                                //  zfuncs: image_gallery() etc.
   extern int     xwinW, xwinH;                                            //  image gallery window size
   extern int     thumbsize;                                               //  thumbnail image size
}

namespace      zfuncs {
   extern char    zlanguage[8];                                            //  current language lc_RC  v.8.5
}

GtkWidget      *mWin, *drWin, *mVbox;                                      //  main and drawing window
GtkWidget      *mMbar, *mTbar, *STbar;                                     //  menu bar, tool bar, status bar
GtkWidget      *brightgraph = 0;                                           //  brightness distribution graph
GtkWidget      *drbrightgraph = 0;                                         //  drawing area
GError         **gerror = 0;
GdkGC          *gdkgc = 0;                                                 //  GDK graphics context
GdkColor       black, white, red, green;
GdkColormap    *colormap = 0;
uint           maxcolor = 0xffff;
GdkCursor      *arrowcursor = 0;
GdkCursor      *dragcursor = 0;
GdkCursor      *drawcursor = 0;
GdkCursor      *busycursor = 0;

double      pi = 3.141592654;
char        PIDstring[12];                                                 //  process PID as string
pthread_t   tid_fmain = 0;                                                 //  thread ID for main()
int         NWthreads = 0;                                                 //  working threads to use
int         maxWthreads = 4;                                               //  max. worker threads to use
int         wtindex[4] = { 0, 1, 2, 3 };                                   //  internal thread ID

int         Fexiftool = 0;                                                 //  exiftool program available
int         Fufraw = 0;                                                    //  ufraw program is available
int         Fprintoxx = 0;                                                 //  printoxx program available
int         Fxdgopen = 0;                                                  //  xdg-open program available
int         Fshutdown = 0;                                                 //  app shutdown underway
int         Fdebug = 0;                                                    //  debug flag
int         Wrepaint = 0;                                                  //  request window paint
int         Wpainted = 0;                                                  //  window was repainted
int         Fmodified = 0;                                                 //  image was edited/modified
int         Fsaved = 0;                                                    //  image saved since last mod
int         Fpreview = 0;                                                  //  use window image for edits
int         Fblowup = 0;                                                   //  zoom small images to window size
int         Fshowarea = 0;                                                 //  show selected area outline
int         SBupdate = 0;                                                  //  request to update status bar
int         PercentDone = 0;                                               //  % completion in status bar
int         Fautolens = 0;                                                 //  lens parameter search underway
int         Fsearchlist = 0;                                               //  file list via search tags
int         Fimageturned = 0;                                              //  image was turned when loaded
int         Fslideshow = 0;                                                //  slide show mode is active
int         SS_interval = 3;                                               //  slide show interval
int         SS_timer = 0;                                                  //  slide show timer

char        *image_file = 0;                                               //  current image file
char        *recentfiles[40];                                              //  40 most recent image files
int         Nrecentfiles = 40;                                             //  recent file list size
double      file_MB;                                                       //  disk file size, MB
int         file_bpp;                                                      //  disk file bits/pixel
char        file_type[8];                                                  //  "jpeg" or "tiff" or "other"
char        *asstagsfile = 0;                                              //  assigned tags file
char        *topdirk = 0;                                                  //  top-level image directory
char        *topmenu;                                                      //  latest top-menu selection
char        jpeg_quality[8] = def_jpeg_quality;                            //  jpeg file compression quality

//  fotoxx RGB pixmaps                                                     //  v.6.5

typedef struct  {                                                          //  RGB pixmap
   char     wmi[8];
   int      ww, hh, bpp;                                                   //  width, height, bits per pixel
   void     *bmp;                                                          //  uint8*/uint16* (bpp=24/48)
}  RGB;

RGB         *Frgb24 = 0;                                                   //  input file 1 pixmap, RGB-24
RGB         *Frgb48 = 0;                                                   //  input file 1 pixmap, RGB-48
RGB         *Grgb48 = 0;                                                   //  input file 2 pixmap, RGB-48
RGB         *E1rgb48 = 0;                                                  //  edit pixmap, base image
RGB         *E3rgb48 = 0;                                                  //  edit pixmap, edited image
RGB         *E9rgb48 = 0;                                                  //  scratch image for some functions
RGB         *Drgb24 = 0;                                                   //  drawing window pixmap
RGB         *A1rgb48 = 0, *A2rgb48 = 0;                                    //  align image pixmaps (HDR, pano)

int         Fww, Fhh, Gww, Ghh;                                            //  input image dimensions
int         E1ww, E1hh, E3ww, E3hh;                                        //  edit image dimensions
int         Dww = mwinww, Dhh = mwinhh;                                    //  drawing window size (defaults)
int         A1ww, A1hh, A2ww, A2hh;                                        //  alignment image dimensions

double      Fzoom = 0;                                                     //  image zoom scale (0 = fit window)
int         zoomx = 0, zoomy = 0;                                          //  req. zoom center of window
double      Mscale = 1;                                                    //  scale factor, file to window
int         Iww, Ihh;                                                      //  current image size at 1x
int         iww, ihh;                                                      //  area in drawing window at Mscale
int         dww, dhh;                                                      //  drawing window image, iww * Mscale
int         Iorgx, Iorgy;                                                  //  drawing window origin in image
int         Dorgx, Dorgy;                                                  //  image origin in drawing window

mutex       pixmaps_lock;                                                  //  lock for accessing RGB pixmaps
int         menu_lock = 0;                                                 //  lock for some menu functions

char        *undo_files = 0;                                               //  undo/redo stack, image files
int         Pundo = 0;                                                     //  undo/redo stack position
int         Pumax = 0;                                                     //  undo/redo stack depth

int         Mbutton = 0;                                                   //  mouse button, 1/3 = left/right
int         LMclick = 0, RMclick = 0;                                      //  mouse left, right click
int         Mxclick, Myclick;                                              //  mouse click position
int         Mxposn, Myposn;                                                //  mouse move position
int         Mxdown, Mydown, Mxdrag, Mydrag;                                //  mouse drag vector
int         Mdrag = 0;                                                     //  mouse drag underway
int         Mcapture = 0;                                                  //  mouse captured by edit function

int         KBcapture = 0;                                                 //  KB key captured by edit function
int         KBkey = 0;                                                     //  active keyboard key

int         Ntoplines = 0, Nptoplines = 0;                                 //  lines overlayed on image in window
int         toplinex1[4], topliney1[4], toplinex2[4], topliney2[4];
int         ptoplinex1[4], ptopliney1[4], ptoplinex2[4], ptopliney2[4];

int         toparc = 0, ptoparc = 0;                                       //  arc (circle/ellipse) on top
int         toparcx,toparcy,toparcw,toparch;                               //    of image in window
int         ptoparcx,ptoparcy,ptoparcw,ptoparch;

zdialog     *zdtags = null;                                                //  edit tags zdialog
zdialog     *zdrename = null;                                              //  rename file zdialog
zdialog     *zdRGB = null;                                                 //  show RGB dialog
zdialog     *zdsela = null;                                                //  select area dialog
zdialog     *zdedit = null;                                                //  image edit dialog
zdialog     *zdburn = null;                                                //  burn CD/DVD dialog

//  pano and HDR control data

int      curr_lens = 0;                                                    //  current lens, 0-3
int      lens_cc = 19;                                                     //  lens name cc limit
char     *lens4_name[4];                                                   //  names for 4 lenses
double   lens4_mm[4], lens4_bow[4];                                        //  characteristics for 4 lenses
double   lens_mm, lens_bow;                                                //  current lens characteristics
double   pano_prealign_size = 500;                                         //  pre-align image height
double   pano_image_increase = 1.6;                                        //  image size increase per stage
double   pano_blend_decrease = 0.8;                                        //  blend width decrease per stage
double   pano_min_alignwidth = 0.10;                                       //  min. align area, 10% width
double   pano_max_alignwidth = 0.20;                                       //  max. align area, 20%
double   pano_ycurveF = 1.41;                                              //  image curve, y-adjust factor

int      fullSize, alignSize;                                              //  full and align image sizes
int      showRedpix = 0;                                                   //  flag, highlight alignment pixels
int      xshrink, yshrink;                                                 //  image shrinkage (pano, HDF)
int      pxL, pxH, pyL, pyH;                                               //  image overlap area
int      pxM, pxmL, pxmH;                                                  //  align area width mid/low/high
int      pyM, pymL, pymH;                                                  //  align area height
int      Nalign = 0;                                                       //  counts alignment cycles
int      aligntype = 0;                                                    //  align type (HDR, HDF, pano)
int      alignWidth, alignHeight;                                          //  pano/HDR/HDF alignment area
int      overlapixs = 0;                                                   //  overlapped pixels count
int      pixsamp = 5000;                                                   //  pixel sample size
char     *redpixels = 0;                                                   //  flags edge pixels for image align

double   Bratios1[3][256], Bratios2[3][256];                               //  brightness ratios/color/brightness
double   R12match[65536], G12match[65536], B12match[65536];                //  image1/2 color matching factors 
double   R21match[65536], G21match[65536], B21match[65536];                //  image2/1 color matching factors
double   Radjust, Gadjust, Badjust;                                        //  RGB manual adjustmants
double   xoff, yoff, toff;                                                 //  align data: x, y, theta
double   warpxu, warpyu, warpxl, warpyl;                                   //  pano image2 warp: upper/lower corners
double   xoffB, yoffB, toffB;                                              //  align data: current best values
double   warpxuB, warpyuB, warpxlB, warpylB;                               //  pano image2 warp: current best values
double   matchlev, matchB;                                                 //  image alignment match level

//  GTK functions

int   main(int argc, char * argv[]);                                       //  main program
int   gtkinitfunc(void *data);                                             //  GTK initz. function
int   gtimefunc(void *arg);                                                //  periodic function
int   delete_event();                                                      //  window delete event function
void  destroy_event();                                                     //  window destroy event function
int   mwpaint();                                                           //  window repaint - expose event
void  mwpaint2();                                                          //  window repaint - image modified
void  update_statusbar();                                                  //  update main window status bar
void  mouse_event(GtkWidget *, GdkEventButton *, void *);                  //  mouse event function
int   KBpress(GtkWidget *, GdkEventKey *, void *);                         //  KB key press event function
int   KBrelease(GtkWidget *, GdkEventKey *, void *);                       //  KB key release event
void  paint_toplines(int arg);                                             //  paint lines on image
void  paint_toparc(int arg);                                               //  paint arc on image
void  draw_line(int x1, int y1, int x2, int y2);                           //  draw line, image space
void  erase_line(int x1, int y1, int x2, int y2);                          //  erase line
void  topmenufunc(GtkWidget *, const char *menu);                          //  menu function

typedef void CBfunc();                                                     //  callback function type
CBfunc   *mouseCBfunc = 0;                                                 //  current mouse handler function
CBfunc   *KBkeyCBfunc = 0;                                                 //  current KB handler function

//  file functions

void  m_gallery(GtkWidget *, const char *);                                //  show image gallery window
void  m_gallery2(char *);                                                  //  converts function types
void  m_open(GtkWidget *, const char *);                                   //  open image file (menu)
void  m_open_drag(int x, int y, char *file);                               //  open drag-drop file
void  m_recent(GtkWidget *, const char *);                                 //  open recently accessed file
void  add_recent_file(const char *file);                                   //  add file to recent file list
void  f_open(const char *file);                                            //  open file helper function
void  m_raw(GtkWidget *, const char *);                                    //  open RAW file
void  m_prev(GtkWidget *, const char *);                                   //  open previous file
void  m_next(GtkWidget *, const char *);                                   //  open next file
void  m_save(GtkWidget *, const char *);                                   //  save modified image to same file
void  m_saveas(GtkWidget *, const char *);                                 //  save modified image to another file
void  f_save(cchar *outfile, cchar *format);                               //  save file helper function
void  m_print(GtkWidget *, const char *);                                  //  print image file(s)
void  m_trash(GtkWidget *, const char *);                                  //  move image to trash
void  m_rename(GtkWidget *, const char *);                                 //  rename file menu function
void  rename_dialog();                                                     //  start rename file dialog
void  m_quit(GtkWidget *, const char *);                                   //  exit application

//  tools functions

void  m_zoom(GtkWidget *, const char *);                                   //  zoom image +/-
void  m_montest(GtkWidget *, const char *);                                //  check monitor
void  m_index_tt(GtkWidget *, const char *);                               //  index tags and thumbnails
void  m_brightgraph(GtkWidget *, const char *);                            //  start brightness dist. graph
void  brightgraph_paint();                                                 //  update brightness dist. graph
void  brightgraph_destroy();                                               //  remove graph window
void  m_clone(GtkWidget *, const char *);                                  //  start another fotoxx instance
void  m_slideshow(GtkWidget *, const char *);                              //  slideshow mode
void  m_showRGB(GtkWidget *, const char *);                                //  show RGB values at mouse click
void  m_parms(GtkWidget *, const char *);                                  //  edit parameters
void  m_lang(GtkWidget *, const char *);                                   //  change language
void  m_launcher(GtkWidget *, const char *);                               //  make desktop icon/launcher
void  m_multiraw(GtkWidget *, const char *);                               //  convert multiple raw files to tiff
void  m_burn(GtkWidget *, const char *);                                   //  burn images to CD/DVD
void  burn_insert_file(const char *);                                      //  called from image gallery window

//  tags and EXIF functions

void  m_edit_tags(GtkWidget *, const char *);                              //  edit tags menu function
void  edit_tags_dialog();                                                  //  start edit tags dialog
void  load_filetags(const char *file);                                     //  load tags from an image file
void  update_filetags(const char *file);                                   //  write updated tags to image file
void  load_asstags();                                                      //  load all assigned tags
void  update_asstags(const char *file, int del = 0);                       //  update assigned tags file
void  m_search_tags(GtkWidget *, const char *);                            //  search images for matching tags
void  m_exif_list(GtkWidget *, const char *);                              //  list EXIF data to popup window
char  ** exif_get(cchar *file, cchar **keys, int nkeys);                   //  get EXIF data for given key(s)
int   exif_set(cchar *file, cchar **keys, cchar **text, int nkeys);        //  set EXIF data for given key(s)
int   exif_copy(cchar *f1, cchar *f2, cchar **k, cchar **t, int nk);       //  copy EXIF data + opt. updates

//  select area functions

void  m_select_mouse(GtkWidget *, const char *);                           //  select area to edit - mouse
void  m_select_color(GtkWidget *, const char *);                           //  select area to edit - color
void  m_select_show(GtkWidget *, const char *);                            //  show area outline
void  m_select_hide(GtkWidget *, const char *);                            //  hide area outline
void  m_select_finish(GtkWidget *, const char *);                          //  finish - find inside pixels
void  m_select_edgecalc(GtkWidget *, const char *);                        //  calculate distances from edge
void  m_select_invert(GtkWidget *, const char *);                          //  invert area
void  m_select_disable(GtkWidget *, const char *);                         //  disable area (menu)
void  m_select_delete(GtkWidget *, const char *);                          //  delete area (menu)
void  select_delete();                                                     //  delete area (callable)
void  select_drawpix1(int px, int py);                                     //  draw 1 pixel if not already drawn
void  select_drawpix2(int px, int py);                                     //  draw 1 pixel unconditionally
void  m_select_copy(GtkWidget *, const char *);                            //  copy and save area
void  m_select_paste(GtkWidget *, const char *);                           //  paste saved area into another image
void  select_paste_copy();

//  edit functions

void  m_whitebal(GtkWidget *, const char *);                               //  adjust white balance
void  m_flatten(GtkWidget *, const char *);                                //  flatten brightness distribution
void  m_tune(GtkWidget *, const char *);                                   //  brightness / color adjustments
void  m_redeye(GtkWidget *, const char *);                                 //  red-eye removal
void  m_blur(GtkWidget *, const char *);                                   //  blur image
void  m_sharpen(GtkWidget *, const char *);                                //  sharpen image
void  m_denoise(GtkWidget *, const char *);                                //  image noise reduction
void  m_resize(GtkWidget *, const char *);                                 //  resize image
void  m_trim(GtkWidget *, const char *);                                   //  trim image
void  m_rotate(GtkWidget *, const char *);                                 //  rotate image
void  m_unbend(GtkWidget *, const char *);                                 //  fix perspective problems
void  m_WarpA(GtkWidget *, const char *);                                  //  warp image area
void  m_WarpI(GtkWidget *, const char *);                                  //  warp image globally
void  m_colordep(GtkWidget *, const char *);                               //  set color depth 1-16 bits/color
void  m_draw(GtkWidget *, const char *);                                   //  make simulated drawing
void  m_emboss(GtkWidget *, const char *);                                 //  make simulated embossing
void  m_tiles(GtkWidget *, const char *);                                  //  make simulated tiles (pixelate)
void  m_painting(GtkWidget *, const char *);                               //  make simulated painting
void  m_pixedit(GtkWidget *, const char *);                                //  edit individual pixels
void  m_HDR(GtkWidget *, const char *);                                    //  make HDR image
void  m_HDF(GtkWidget *, const char *);                                    //  make HDF image
void  m_pano(GtkWidget *, const char *);                                   //  make panorama image

//  pano and HDR common functions

int      sigdiff(double d1, double d2, double signf);                      //  test for significant difference
void     getAlignArea();                                                   //  get image overlap area
void     getBrightRatios();                                                //  get color brightness ratios
void     setColorfixFactors(int state);                                    //  set color matching factors
void     flagEdgePixels();                                                 //  flag high-contrast pixels
double   matchImages();                                                    //  match images in overlap region
double   matchPixels(uint16 *pix1, uint16 *pix2);                          //  match two pixels
int      vpixel(RGB *rgb, double px, double py, uint16 *vpix);             //  get virtual pixel at (px,py)

//  edit support functions

typedef void * threadfunc(void *);                                         //  edit thread function

int   edit_setup(int preview, int delsa);                                  //  start new edit transaction
void  edit_cancel();                                                       //  cancel edit
void  edit_done();                                                         //  commit edit, add undo stack
void  edit_undo();                                                         //  undo edit, revert
void  edit_redo();                                                         //  redo the last undo
void  edit_fullsize();                                                     //  convert to full-size pixmaps
void  edit_progress(int done, int goal);                                   //  update status bar progress

void  start_thread(threadfunc func, void *arg);                            //  start a working thread
void  signal_thread();                                                     //  signal work is pending
void  wait_thread_idle();                                                  //  wait for work complete
void  wrapup_thread(int command);                                          //  wait for exit or command exit
void  thread_idle_loop();                                                  //  wait for work or exit command
void  exit_thread();                                                       //  exit thread unconditionally
int   thread_working();                                                    //  thread is working on edit function

void  m_undo(GtkWidget *, const char *);                                   //  undo one edit
void  m_redo(GtkWidget *, const char *);                                   //  redo one edit
void  save_undo();                                                         //  undo/redo save function
void  load_undo();                                                         //  undo/redo read function

//  other support functions

void  m_help(GtkWidget *, const char *);                                   //  various help functions
int   load_fotoxx_state();                                                 //  load state from prior session
int   save_fotoxx_state();                                                 //  save state for next session
void  free_resources();                                                    //  free all allocated resources
int   mod_keep();                                                          //  query keep/discard edited image
int   menulock(int lock);                                                  //  lock/unlock menu for some funcs
void  turn_image(int angle);                                               //  turn image (upright)
void  FI_error(FIF fif, const char *message);                              //  catch FreeImage error messages

//  RGB pixmap and FI bitmap conversion functions                          //  v.6.5

RGB * RGB_make(int ww, int hh, int bpp);                                   //  initialize RGB pixmap
void  RGB_free(RGB *rgb);                                                  //  free RGB pixmap
RGB * RGB_copy(RGB *rgb);                                                  //  copy RGB pixmap
RGB * RGB_copy_area(RGB *rgb, int orgx, int orgy, int ww, int hh);         //  copy section of RGB pixmap
RGB * RGB_convbpp(RGB *rgb);                                               //  convert from 24/48 to 48/24 bpp
FIB * RGB_FIB(RGB *rgb);                                                   //  convert RGB to FI bitmap
RGB * FIB_RGB(FIB *fib);                                                   //  convert FI to RGB pixmap
PXB * RGB_PXB(RGB *rgb);                                                   //  convert RGB pixmap to pixbuf 
RGB * PXB_RGB(PXB *pxb);                                                   //  convert pixbuf to RGB pixmap
RGB * RGB_rescale(RGB *rgb, int ww, int hh);                               //  rescale RGB pixmap (ww/hh)
RGB * RGB_rotate(RGB *rgb, double angle);                                  //  rotate RGB pixmap
RGB * image_load(const char *filespec, int bpp);                           //  image file >> RGB-24/48 pixmap

//  translatable strings used in multiple dialogs

const char  *Bsavetoedit;
const char  *Bexiftoolmissing;
const char  *Bopenrawfile;
const char  *BOK;
const char  *Bcancel;
const char  *Bdone;
const char  *Bclear;
const char  *Bapply;
const char  *Bundo;
const char  *Bundolast;
const char  *Bundoall;
const char  *Bredo;
const char  *Bsearch;
const char  *Binsert;
const char  *Baddall;
const char  *Bstart;
const char  *Bfinish;
const char  *Bsuspend;
const char  *Bresume;
const char  *Bshow;
const char  *Bhide;
const char  *Bdelete;
const char  *Binvert;
const char  *Bedgecalc;
const char  *Bblendwidth;
const char  *Bdeletearea;
const char  *Bwidth;
const char  *Bheight;
const char  *Bpercent;
const char  *Bpreset;
const char  *Bproceed;
const char  *Bred;
const char  *Bgreen;
const char  *Bblue;
const char  *Bbrightness;
const char  *Bdarker;
const char  *Blighter;
const char  *Breduce;


/**************************************************************************
      main program and GTK/GDK functions
***************************************************************************/

int main(int argc, char *argv[])
{
   char           lang[8] = "";
   
   printf(fversion "\n");                                                  //  print version
   if (argc > 1 && strEqu(argv[1],"-v")) return 0;

   gtk_init(&argc,&argv);                                                  //  initz. GTK
   zlockInit();

   initz_appfiles("fotoxx",null);                                          //  get app directories
   load_fotoxx_state();                                                    //  restore data from last session

   for (int ii = 1; ii < argc; ii++)                                       //  command line options
   {
      if (strEqu(argv[ii],"-d"))                                           //  -d (debug flag)
            Fdebug = 1;
      else if (strEqu(argv[ii],"-l") && argc > ii+1)                       //  -l language code
            strncpy0(lang,argv[++ii],7);
      else  image_file = strdupz(argv[ii]);                                //  initial file or directory
   }

   ZTXinit(lang);                                                          //  setup translations

   mWin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                             //  create main window
   gtk_window_set_title(GTK_WINDOW(mWin),fversion);
   gtk_window_set_position(GTK_WINDOW(mWin),GTK_WIN_POS_CENTER);
   gtk_window_set_default_size(GTK_WINDOW(mWin),Dww,Dhh);

   mVbox = gtk_vbox_new(0,0);                                              //  add vert. packing box
   gtk_container_add(GTK_CONTAINER(mWin),mVbox);

   mMbar = create_menubar(mVbox,16);                                       //  menus / sub-menus

   GtkWidget *mFile = add_menubar_item(mMbar,ZTX("File"),topmenufunc);
      add_submenu_item(mFile,    ZTX("Image Gallery"),         "m-gallery.png",     m_gallery);
      add_submenu_item(mFile,    ZTX("Open RAW File"),         "m-open.png",        m_raw);
      add_submenu_item(mFile,    ZTX("Open Image File"),       "m-open.png",        m_open);
      add_submenu_item(mFile,    ZTX("Open Recent File"),      "m-open.png",        m_recent);
      add_submenu_item(mFile,    ZTX("Save to Same File"),     "m-save.png",        m_save);
      add_submenu_item(mFile,    ZTX("Save to New File"),      "m-save.png",        m_saveas);
      add_submenu_item(mFile,    ZTX("Print Image File"),      "m-print.png",       m_print);
      add_submenu_item(mFile,    ZTX("Trash Image File"),      "m-trash.png",       m_trash);
      add_submenu_item(mFile,    ZTX("Rename Image File"),     "m-rename.png",      m_rename);
      add_submenu_item(mFile,    ZTX("Quit fotoxx"),           "m-quit.png",        m_quit);

   GtkWidget *mTools = add_menubar_item(mMbar,ZTX("Tools"),topmenufunc);
      add_submenu_item(mTools,   ZTX("Check Monitor"),         "m-montest.png",     m_montest);
      add_submenu_item(mTools,   ZTX("Index Tags and Thumbs"), "m-index.png",       m_index_tt);
      add_submenu_item(mTools,   ZTX("Brightness Graph"),      "m-distr.png",       m_brightgraph);
      add_submenu_item(mTools,   ZTX("Clone fotoxx"),          "m-clone.png",       m_clone);
      add_submenu_item(mTools,   ZTX("Slide Show"),            "m-slideshow.png",   m_slideshow);
      add_submenu_item(mTools,   ZTX("Show RGB"),              "m-RGB.png",         m_showRGB);
      add_submenu_item(mTools,   ZTX("Lens Parameters"),       "m-parms.png",       m_parms);
      add_submenu_item(mTools,   ZTX("Change Language"),       "m-lang.png",        m_lang);
      add_submenu_item(mTools,   ZTX("Create Launcher"),       "m-launcher.png",    m_launcher);
      add_submenu_item(mTools,   ZTX("Convert multiple RAWs"), "m-multiraw.png",    m_multiraw);
      add_submenu_item(mTools,   ZTX("Burn Images to CD/DVD"), "m-burn.png",        m_burn);

   GtkWidget *mTags = add_menubar_item(mMbar,ZTX("Tags"),topmenufunc);
      add_submenu_item(mTags,    ZTX("Edit Tags"),             "m-tags.png",        m_edit_tags);
      add_submenu_item(mTags,    ZTX("Search Tags"),           "m-tags.png",        m_search_tags);
      add_submenu_item(mTags,    ZTX("Basic EXIF data"),       "m-exif.png",        m_exif_list);
      add_submenu_item(mTags,    ZTX("All EXIF data"),         "m-exif.png",        m_exif_list);

   GtkWidget *mArea = add_menubar_item(mMbar,ZTX("Area"),topmenufunc);
      add_submenu_item(mArea,    ZTX("Select Area -mouse"),    "m-select.png",      m_select_mouse);
      add_submenu_item(mArea,    ZTX("Select Area -color"),    "m-select.png",      m_select_color);
      add_submenu_item(mArea,    ZTX("Show Area"),             "m-select.png",      m_select_show);
      add_submenu_item(mArea,    ZTX("Hide Area"),             "m-select.png",      m_select_hide);
      add_submenu_item(mArea,    ZTX("Area Edge Calc"),        "m-select.png",      m_select_edgecalc);
      add_submenu_item(mArea,    ZTX("Invert Area"),           "m-select.png",      m_select_invert);
      add_submenu_item(mArea,    ZTX("Disable Area"),          "m-select.png",      m_select_disable);
      add_submenu_item(mArea,    ZTX("Delete Area"),           "m-select.png",      m_select_delete);
      add_submenu_item(mArea,    ZTX("Copy Area"),             "m-select.png",      m_select_copy);
      add_submenu_item(mArea,    ZTX("Paste Area"),            "m-select.png",      m_select_paste);

   GtkWidget *mLight = add_menubar_item(mMbar,ZTX("Retouch"),topmenufunc);
      add_submenu_item(mLight,   ZTX("White Balance"),         "m-whitebal.png",    m_whitebal);
      add_submenu_item(mLight,   ZTX("Flatten Brightness"),    "m-flatten.png",     m_flatten);
      add_submenu_item(mLight,   ZTX("Brightness/Color"),      "m-tune.png",        m_tune);
      add_submenu_item(mLight,   ZTX("Red Eyes"),              "m-redeye.png",      m_redeye);

   GtkWidget *mSharp = add_menubar_item(mMbar,ZTX("Sharp"),topmenufunc);
      add_submenu_item(mSharp,   ZTX("Blur Image"),            "m-blur.png",        m_blur);
      add_submenu_item(mSharp,   ZTX("Sharpen Image"),         "m-sharpen.png",     m_sharpen);
      add_submenu_item(mSharp,   ZTX("Reduce Noise"),          "m-denoise.png",     m_denoise);

   GtkWidget *mSize = add_menubar_item(mMbar,ZTX("Size"),topmenufunc);
      add_submenu_item(mSize,    ZTX("Trim Image"),            "m-trim.png",        m_trim);
      add_submenu_item(mSize,    ZTX("Resize Image"),          "m-resize.png",      m_resize);
      add_submenu_item(mSize,    ZTX("Rotate Image"),          "m-rotate.png",      m_rotate);

   GtkWidget *mBend = add_menubar_item(mMbar,ZTX("Bend"),topmenufunc);
      add_submenu_item(mBend,    ZTX("Unbend Image"),          "m-unbend.png",      m_unbend);
      add_submenu_item(mBend,    ZTX("Warp Area"),             "m-warp.png",        m_WarpA);
      add_submenu_item(mBend,    ZTX("Warp Image"),            "m-warp.png",        m_WarpI);

   GtkWidget *mArt = add_menubar_item(mMbar,ZTX("Art"),topmenufunc);
      add_submenu_item(mArt,    ZTX("Color Depth"),            "m-colordep.png",    m_colordep);
      add_submenu_item(mArt,    ZTX("Simulate Drawing"),       "m-draw.png",        m_draw);
      add_submenu_item(mArt,    ZTX("Simulate Embossing"),     "m-emboss.png",      m_emboss);
      add_submenu_item(mArt,    ZTX("Simulate Tiles"),         "m-tiles.png",       m_tiles);
      add_submenu_item(mArt,    ZTX("Simulate Painting"),      "m-painting.png",    m_painting);
      add_submenu_item(mArt,    ZTX("Edit Pixels"),            "m-pixedit.png",     m_pixedit);

   GtkWidget *mComb = add_menubar_item(mMbar,ZTX("Combine"),topmenufunc);
      add_submenu_item(mComb,    ZTX("Make HDR Image"),        "m-hdr.png",         m_HDR);
      add_submenu_item(mComb,    ZTX("Make HDF Image"),        "m-hdf.png",         m_HDF);
      add_submenu_item(mComb,    ZTX("Make Panorama"),         "m-pano.png",        m_pano);

   GtkWidget *mHelp = add_menubar_item(mMbar,ZTX("Help"),topmenufunc);
      add_submenu_item(mHelp,    ZTX("About"),                 "m-about.png",       m_help);
      add_submenu_item(mHelp,    ZTX("User Guide"),            "m-userguide.png",   m_help);
      add_submenu_item(mHelp,    "README",                     "m-readme.png",      m_help);
      add_submenu_item(mHelp,    ZTX("Change Log"),            "m-changelog.png",   m_help);
      add_submenu_item(mHelp,    ZTX("Translate"),             "m-translate.png",   m_help);
      add_submenu_item(mHelp,    "FreeImage",                  "m-FI.png",          m_help);
      add_submenu_item(mHelp,    ZTX("Home Page"),             "m-fotoxx.png",      m_help);

   mTbar = create_toolbar(mVbox,24);                                       //  toolbar buttons
      add_toolbar_button(mTbar,  ZTX("Gallery"),   ZTX("Image Gallery"),         "gallery.png", m_gallery);
      add_toolbar_button(mTbar,  ZTX("Open"),      ZTX("Open Image File"),       "open.png",    m_open);
      add_toolbar_button(mTbar,  ZTX("Prev"),      ZTX("Open Previous File"),    "prev.png",    m_prev);
      add_toolbar_button(mTbar,  ZTX("Next"),      ZTX("Open Next File"),        "next.png",    m_next);
      add_toolbar_button(mTbar,  ZTX("Save"),      ZTX("Save to Same File"),     "save.png",    m_save);
      add_toolbar_button(mTbar,  ZTX("Save As"),   ZTX("Save to New File"),      "save.png",    m_saveas);
      add_toolbar_button(mTbar,  ZTX("Undo"),      ZTX("Undo One Edit"),         "undo.png",    m_undo);
      add_toolbar_button(mTbar,  ZTX("Redo"),      ZTX("Redo One Edit"),         "redo.png",    m_redo);
      add_toolbar_button(mTbar,  "Zoom+",          ZTX("Zoom-in (bigger)"),      "zoom+.png",   m_zoom);
      add_toolbar_button(mTbar,  "Zoom-",          ZTX("Zoom-out (smaller)"),    "zoom-.png",   m_zoom);
      add_toolbar_button(mTbar,  ZTX("Trash"),     ZTX("Move Image to Trash"),   "trash.png",   m_trash);
      add_toolbar_button(mTbar,  ZTX("Quit"),      ZTX("Quit fotoxx"),           "quit.png",    m_quit);

   drWin = gtk_drawing_area_new();                                         //  add drawing window
   gtk_box_pack_start(GTK_BOX(mVbox),drWin,1,1,0);

   STbar = create_stbar(mVbox);                                            //  add status bar

   G_SIGNAL(mWin,"delete_event",delete_event,0)                            //  connect signals to windows
   G_SIGNAL(mWin,"destroy",destroy_event,0)
   G_SIGNAL(drWin,"expose-event",mwpaint,0)

   gtk_widget_add_events(drWin,GDK_BUTTON_PRESS_MASK);                     //  connect mouse events
   gtk_widget_add_events(drWin,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(drWin,GDK_BUTTON_MOTION_MASK);                    //  all buttons         v.6.8
   gtk_widget_add_events(drWin,GDK_POINTER_MOTION_MASK);                   //  pointer motion      v.8.3
   G_SIGNAL(drWin,"button-press-event",mouse_event,0)
   G_SIGNAL(drWin,"button-release-event",mouse_event,0)
   G_SIGNAL(drWin,"motion-notify-event",mouse_event,0)

   G_SIGNAL(mWin,"key-press-event",KBpress,0)                           	//  connect KB events
   G_SIGNAL(mWin,"key-release-event",KBrelease,0)
   
   drag_drop_connect(drWin,m_open_drag);                                   //  connect drag-drop event   v.7.3

   gtk_widget_show_all(mWin);                                              //  show all widgets

   gdkgc = gdk_gc_new(drWin->window);                                      //  initz. graphics context

   black.red = black.green = black.blue = 0;                               //  set up colors
   white.red = white.green = white.blue = maxcolor;
   red.red = maxcolor;  red.green = red.blue = 0;
   green.green = maxcolor; green.red = green.blue = 0;

   colormap = gtk_widget_get_colormap(drWin);
   gdk_rgb_find_color(colormap,&black);
   gdk_rgb_find_color(colormap,&white);
   gdk_rgb_find_color(colormap,&red);
   gdk_rgb_find_color(colormap,&green);

   gdk_gc_set_foreground(gdkgc,&black);
   gdk_gc_set_background(gdkgc,&white);
   gdk_gc_set_line_attributes(gdkgc,1,lineattributes);

   arrowcursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);                       //  cursor for selection
   dragcursor = gdk_cursor_new(GDK_CROSSHAIR);                             //  cursor for dragging
   drawcursor = gdk_cursor_new(GDK_PENCIL);                                //  cursor for drawing lines
   busycursor = gdk_cursor_new(GDK_WATCH);                                 //  cursor for function busy

   gtk_init_add((GtkFunction) gtkinitfunc,0);                              //  set initz. call from gtk_main()

   gtk_main();                                                             //  process window events
   return 0;
}


/**************************************************************************/

//  initial function called from gtk_main() at startup

int gtkinitfunc(void * data)
{
   int               err, flag, npid;
   char              procfile[20], *pp;
   char              printoxx[200], command[200];
   const char        *ppc, *ppc2;
   struct stat       statb;

   Bsavetoedit = ZTX("Unknown file type, save as tiff or jpeg to edit");
   Bexiftoolmissing = ZTX("Package exiftool is missing");
   Bopenrawfile = ZTX("Open RAW File");
   BOK = ZTX("OK");
   Bcancel = ZTX("Cancel");
   Bdone = ZTX("Done");
   Bclear = ZTX("Clear");
   Bapply = ZTX("Apply");
   Bundo = ZTX("Undo");
   Bundolast = ZTX("Undo Last");
   Bundoall = ZTX("Undo All");
   Bredo = ZTX("Redo");
   Bsearch = ZTX("Search");
   Binsert = ZTX("Insert");
   Baddall = ZTX("Add All");
   Bstart = ZTX("Start");
   Bfinish = ZTX("Finish");
   Bsuspend = ZTX("Suspend");
   Bresume = ZTX("Resume");
   Bshow = ZTX("Show");
   Bhide = ZTX("Hide");
   Bdelete = ZTX("Delete");
   Binvert = ZTX("Invert");
   Bedgecalc = ZTX("Edge Calc");
   Bblendwidth = ZTX("Blend Width");
   Bdeletearea = ZTX("Delete selected area?");
   Bwidth = ZTX("Width");
   Bheight = ZTX("Height");
   Bpercent = ZTX("Percent");
   Bpreset = ZTX("Presets");
   Bproceed = ZTX("Proceed");
   Bred = ZTX("Red");
   Bgreen = ZTX("Green");
   Bblue = ZTX("Blue");
   Bbrightness = ZTX("Brightness");
   Bdarker = ZTX("Darker Areas");
   Blighter = ZTX("Lighter Areas");
   Breduce = ZTX("Reduce");

   tid_fmain = pthread_self();                                             //  get main() thread ID
   snprintf(PIDstring,11,"%06d",getpid());                                 //  get fotoxx process PID

   NWthreads = sysconf(_SC_NPROCESSORS_ONLN);                              //  get SMP CPU count   v.7.1
   if (! NWthreads) NWthreads = 1;
   if (NWthreads > maxWthreads) NWthreads = maxWthreads;                   //  compile time limit
   printf("using %d threads \n",NWthreads);

   FreeImage_Initialise(FALSE);                                            //  FREEIMAGE setup
   FreeImage_SetOutputMessage(FI_error);
   printf("FreeImage %s \n",FreeImage_GetVersion());

   err = system("echo -n \"exiftool \"; exiftool -ver");                   //  check for exiftool  v.6.9.1
   if (! err) Fexiftool = 1;
   err = system("xdg-open --version");                                     //  check for xdg-open
   if (! err) Fxdgopen = 1;
   err = system("ufraw --version");                                        //  check for xdg-open
   if (! err) Fufraw = 1;

   err = system("printoxx -v");                                            //  check for printoxx
   if (err) {
      pp = getenv("_");                                                    //  look in same place as me
      if (pp) {                                                            //  v.6.2
         strncpy0(printoxx,pp,188);
         pp = (char *) strrchr(printoxx,'/');
         if (pp) {
            strcpy(pp,"/printoxx -v");
            err = system(printoxx);
         }
      }
   }
   if (! err) Fprintoxx = 1;
   
   if (! Fexiftool) zmessageACK(ZTX("exiftool is not installed \n"         //  warn user
                               "edited images will lose EXIF data"));

   asstagsfile = zmalloc(200);                                             //  setup assigned tags file
   strncatv(asstagsfile,199,get_zuserdir(),"/assigned_tags",null);         //    home/user/.fotoxx/assigned_tags

   undo_files = zmalloc(200);
   *undo_files = 0;
   strncatv(undo_files,199,get_zuserdir(),"/*_undo_*",null);               //  home/user/.fotoxx/pppppp_undo_nn

   flag = 1;
   while ((ppc = SearchWild(undo_files,flag)))                             //  look for orphaned undo files
   {
      ppc2 = strstr(ppc,".fotoxx/");
      if (! ppc2) continue;
      npid = atoi(ppc2+8);                                                 //  pid of file owner
      snprintf(procfile,19,"/proc/%d",npid);
      err = stat(procfile,&statb);
      if (! err) continue;                                                 //  pid is active, keep file
      printf("orphaned undo file deleted: %s \n",ppc);
      snprintf(command,199,"rm -f %s",ppc);                                //  delete orphaned files     v.8.0
      err = system(command);
   }

   *undo_files = 0;                                                        //  setup undo stack files
   strncatv(undo_files,199,get_zuserdir(),"/",PIDstring,"_undo_nn",null);  //  home/user/.fotoxx/pppppp_undo_nn
   
   mutex_init(&pixmaps_lock,0);                                            //  setup lock for edit pixmaps
   
   g_timeout_add(20,gtimefunc,0);                                          //  start periodic function (20 ms)
   
   if (image_file) {
      char * pp = canonicalize_file_name(image_file);                      //  add cwd if needed
      zfree(image_file);
      image_file = null;
      if (pp) image_file = strdupz(pp);                                    //  change pp to zmalloc
      if (pp) free(pp);
   }
      
   if (image_file) {
      err = stat(image_file,&statb);
      if (! err && S_ISREG(statb.st_mode)) f_open(image_file);             //  open initial file
   }

   return 0;
}


/**************************************************************************/

//  Periodic function - runs every few milliseconds.
//  Avoid any thread usage of gtk/gdk functions.

int gtimefunc(void *arg)
{
   double      secs;
   static int  fbusy = 0;

   if (Wrepaint) {                                                         //  update drawing window
      Wrepaint = 0;
      mwpaint();
   }

   if (SBupdate)                                                           //  status bar update 
   {
      SBupdate = 0;
      update_statusbar();
   }
   
   if (Fslideshow) {
      secs = get_seconds();                                                //  show next slide
      if (secs > SS_timer) {
         SS_timer = secs + SS_interval;
         m_next(0,0);  
      }
   }
   
   if (! fbusy && thread_working()) {                                      //  v.8.4
      fbusy = 1;
      gdk_window_set_cursor(drWin->window,busycursor);                     //  set function busy cursor
   }
   
   if (fbusy && ! thread_working()) {
      fbusy = 0;
      gdk_window_set_cursor(drWin->window,0);                              //  restore normal cursor
   }

   return 1;
}


/**************************************************************************/

//  main window delete_event and destroy signals

int delete_event()                                                         //  main window closed
{
   if (mod_keep()) return 1;                                               //  allow user bailout
   Fshutdown++;                                                            //  shutdown in progress
   save_fotoxx_state();                                                    //  save state for next session
   free_resources();                                                       //  delete undo files
   return 0;
}

void destroy_event()                                                       //  main window destroyed
{
   Fshutdown++;
   printf("main window destroyed \n");
   exit(1);                                                                //  instead of gtk_main_quit();
   return;
}


/**************************************************************************/

//  cause (modified) output image to get repainted immediately
//  this function may be called from threads

void mwpaint2()
{
   Wrepaint++;
   return;
}


/**************************************************************************/

//  paint window when created, exposed, resized, or modified (edited)      //  overhauled  v.7.4

int mwpaint()
{
   GdkRectangle   wrect;
   int            incrx, incry;                                            //  mouse drag
   static int     pincrx = 0, pincry = 0;                                  //  prior mouse drag
   double         wscale, hscale;
   static double  pscale = 1;                                              //  prior scale
   RGB            *rgbtemp1, *rgbtemp2;
   
   if (Fshutdown) return 1;                                                //  shutdown underway
   if (! Frgb24) {
      gdk_window_clear(drWin->window);
      return 1;                                                            //  no image
   }

   Dww = drWin->allocation.width;                                          //  (new) drawing window size
   Dhh = drWin->allocation.height;
   if (Dww < 20 || Dhh < 20) return 1;                                     //  v.7.5

   if (mutex_trylock(&pixmaps_lock) != 0) {                                //  lock pixmaps
      Wrepaint++;                                                          //  cannot, return later
      return 1;
   }
   
   if (E3rgb48) {                                                          //  get image size
      Iww = E3ww;
      Ihh = E3hh;                                                          //  edit in progress
   }
   else {
      Iww = Fww;                                                           //  no edit
      Ihh = Fhh;
   }
   
   if (Fzoom == 0) {                                                       //  scale to fit window
      wscale = 1.0 * Dww / Iww;
      hscale = 1.0 * Dhh / Ihh;
      if (wscale < hscale) Mscale = wscale;                                //  use greatest ww/hh ratio
      else  Mscale = hscale;
      if (Iww < Dww && Ihh < Dhh && ! Fblowup) Mscale = 1.0;               //  small image 1x unless Fblowup
      zoomx = zoomy = 0;
   }
   else Mscale = Fzoom;                                                    //  scale to Fzoom level

   if (Mscale > pscale) {                                                  //  zoom increased
      Iorgx += iww * 0.5 * (1.0 - pscale / Mscale);                        //  keep current image center  v.7.5
      Iorgy += ihh * 0.5 * (1.0 - pscale / Mscale);
   }
   pscale = Mscale;
   
   iww = Dww / Mscale;                                                     //  image space fitting in window
   if (iww > Iww) iww = Iww;
   ihh = Dhh / Mscale;
   if (ihh > Ihh) ihh = Ihh;

   if (zoomx || zoomy) {                                                   //  req. zoom center       v.7.5
      Iorgx = zoomx - 0.5 * iww;                                           //  corresp. image origin
      Iorgy = zoomy - 0.5 * ihh;
      zoomx = zoomy = 0;
   }

   if ((Mxdrag || Mydrag) && ! Mcapture) {                                 //  scroll via mouse drag
      incrx = (Mxdrag - Mxdown) * 1.3 * Iww / iww;                         //  scale   v.7.5
      incry = (Mydrag - Mydown) * 1.3 * Ihh / ihh;
      if (pincrx > 0 && incrx < 0) incrx = 0;                              //  stop bounce at extremes
      if (pincrx < 0 && incrx > 0) incrx = 0;
      pincrx = incrx;
      if (pincry > 0 && incry < 0) incry = 0;
      if (pincry < 0 && incry > 0) incry = 0;
      pincry = incry;
      Iorgx += incrx;                                                      //  new image origin after scroll
      Iorgy += incry;
      Mxdown = Mxdrag + incrx;                                             //  new drag origin
      Mydown = Mydrag + incry;
      Mxdrag = Mydrag = 0;
   }

   if (iww == Iww) {                                                       //  scaled image <= window width
      Iorgx = 0;                                                           //  center image in window
      Dorgx = 0.5 * (Dww - Iww * Mscale);
   }
   else Dorgx = 0;                                                         //  image > window, use entire window

   if (ihh == Ihh) {                                                       //  same for image height
      Iorgy = 0;
      Dorgy = 0.5 * (Dhh - Ihh * Mscale);
   }
   else Dorgy = 0;
   
   if (Iorgx + iww > Iww) Iorgx = Iww - iww;                               //  set limits
   if (Iorgy + ihh > Ihh) Iorgy = Ihh - ihh;
   if (Iorgx < 0) Iorgx = 0;
   if (Iorgy < 0) Iorgy = 0;

   if (E3rgb48) {                                                          //  edit in progress
      rgbtemp1 = RGB_copy_area(E3rgb48,Iorgx,Iorgy,iww,ihh);               //  copy RGB-48
      rgbtemp2 = RGB_convbpp(rgbtemp1);                                    //  convert to RGB-24
      RGB_free(rgbtemp1);
   }
   else rgbtemp2 = RGB_copy_area(Frgb24,Iorgx,Iorgy,iww,ihh);              //  no edit, copy RGB-24

   dww = iww * Mscale;                                                     //  scale to window
   dhh = ihh * Mscale;
   RGB_free(Drgb24);
   Drgb24 = RGB_rescale(rgbtemp2,dww,dhh);
   RGB_free(rgbtemp2);

   wrect.x = wrect.y = 0;                                                  //  stop flicker
   wrect.width = Dww;
   wrect.height = Dhh;
   gdk_window_begin_paint_rect(drWin->window,&wrect);

   gdk_window_clear(drWin->window);                                        //  clear window

   gdk_draw_rgb_image(drWin->window, gdkgc, Dorgx, Dorgy, dww, dhh,        //  draw scaled image to window
                      nodither, (uint8 *) Drgb24->bmp, dww*3);

   if (Ntoplines) paint_toplines(1);                                       //  draw line overlays    v.8.3
   if (toparc) paint_toparc(1);                                            //  draw arc overlay
   if (Fshowarea) m_select_show(0,0);                                      //  draw select area outline
   
   gdk_window_end_paint(drWin->window);                                    //  release all window updates

   mutex_unlock(&pixmaps_lock);                                            //  unlock pixmaps
   Wpainted++;                                                             //  notify edit function of repaint
   if (brightgraph) brightgraph_paint();                                   //  update brightness graph
   SBupdate++;                                                             //  update status bar
   return 1;
}


/**************************************************************************/

//  update status bar with image data and status

void update_statusbar()
{
   static char    text1[200], text2[100];
   int            ww, hh, bpp, scale;

   if (! image_file) return;
   
   *text1 = *text2 = 0;
   
   if (E3rgb48) {                                                          //  v.8.5
      ww = E3ww;
      hh = E3hh;
      bpp = 48;
   }
   else if (Frgb48) {
      ww = Fww;
      hh = Fhh;
      bpp = 48;
   }
   else {
      ww = Fww;
      hh = Fhh;
      bpp = file_bpp;
   }
   
   snprintf(text1,199,"%dx%dx%d",ww,hh,bpp);                               //  2345x1234x48 (preview) 0.56MB 45%
   if (Fpreview) strcat(text1," (preview)");                               //       ... (turned)
   sprintf(text2," %.2fMB",file_MB);
   if (Fmodified || Pundo > Fsaved) sprintf(text2," ?? MB");               //  file size TBD
   strcat(text1,text2);
   scale = int(Mscale * 100 + 0.5);
   sprintf(text2," %d%c",scale,'%');
   strcat(text1,text2);
   if (Fimageturned) strcat(text1," (turned)");
   
   if (Pundo)                                                              //  edit undo stack depth 
   {
      snprintf(text2,99,"  edits: %d",Pundo);
      strcat(text1,text2);
   }

   if (Nalign && aligntype == 1)                                           //  HDR alignment data
   {
      snprintf(text2,99,"  align: %d  offsets: %+.1f %+.1f %+.4f  match: %.5f",
                     Nalign,xoffB,yoffB,toffB,matchB);
      strcat(text1,text2);
   }
   
   if (Nalign && aligntype == 2)                                           //  HDF alignment data
   {
      snprintf(text2,99,"  align: %d  offsets: %+.1f %+.1f %+.4f  match: %.5f",
                     Nalign,xoffB,yoffB,toffB,matchB);
      strcat(text1,text2);
   }

   if (Nalign && aligntype == 3)                                           //  pano alignment data
   {
      snprintf(text2,99,"  align: %d  offsets: %+.1f %+.1f %+.4f %+.1f %+.1f %+.1f %+.1f  match: %.5f",
                     Nalign,xoffB,yoffB,toffB,warpxuB,warpyuB,warpxlB,warpylB,matchB);
      strcat(text1,text2);
   }

   if (Fautolens)                                                          //  lens parameters (search status)
   {
      snprintf(text2,99,"  lens: %.1f %.2f",lens_mm,lens_bow);
      strcat(text1,text2);
   }

   if (PercentDone > 0 && PercentDone < 100)                               //  % completion       v.6.3
   {
      snprintf(text2,99,"  done: %d%c",PercentDone,'%');
      strcat(text1,text2);
   }

   stbar_message(STbar,text1);
   return;
}


/**************************************************************************/

//  mouse event function - capture buttons and drag movements

void mouse_event(GtkWidget *, GdkEventButton *event, void *)
{
   void mouse_convert(int &xpos, int &ypos);

   static int     bdtime = 0, butime = 0, mbusy = 0;
   int            button, time, type;

   type = event->type;
   button = event->button;                                                 //  button, 1/3 = left/right
   time = event->time;
   Mxposn = int(event->x);                                                 //  mouse position in window
   Myposn = int(event->y);

   mouse_convert(Mxposn,Myposn);                                           //  convert to image space  v.8.4

   if (type == GDK_MOTION_NOTIFY) {
      if (mbusy) return;                                                   //  discard excess motion events
      mbusy++;
      zmainloop();
      mbusy = 0;
   }

   if (type == GDK_BUTTON_PRESS) {                                         //  button down
      bdtime = time;                                                       //  time of button down
      Mxdown = Mxposn;                                                     //  position at button down time
      Mydown = Myposn;
      if (button) {
         Mdrag++;                                                          //  possible drag start
         Mbutton = button;
      }
      Mxdrag = Mydrag = 0;
   }

   if (type == GDK_BUTTON_RELEASE) {                                       //  button up
      Mxclick = Myclick  = 0;                                              //  reset click status
      butime = time;                                                       //  time of button up
      if (butime - bdtime < 400)                                           //  less than 0.4 secs down
         if (Mxposn == Mxdown && Myposn == Mydown) {                       //       and not moving          v.8.6.1
            if (Mbutton == 1) LMclick++;                                   //  left mouse click
            if (Mbutton == 3) RMclick++;                                   //  right mouse click
            Mxclick = Mxdown;                                              //  click = button down position
            Myclick = Mydown;
         }
      Mxdown = Mydown = Mxdrag = Mydrag = Mdrag = Mbutton = 0;             //  forget buttons and drag
   }
   
   if (type == GDK_MOTION_NOTIFY && Mdrag) {                               //  drag underway
      Mxdrag = Mxposn;
      Mydrag = Myposn;
   }
   
   if (mouseCBfunc) {                                                      //  pass to handler function
      (* mouseCBfunc)();
      return;
   }

   if (LMclick && ! Mcapture) {                                            //  left click = zoom request
      LMclick = 0;
      zoomx = Mxclick;                                                     //  zoom center = mouse   v.7.5
      zoomy = Myclick;
      m_zoom(null, (char *) "+");
   }

   if (RMclick && ! Mcapture) {                                            //  right click = reset zoom
      RMclick = 0;
      zoomx = zoomy = 0;                                                   //  v.7.5
      m_zoom(null, (char *) "-");
   }

   if ((Mxdrag || Mydrag) && ! Mcapture) mwpaint();                        //  drag = scroll
   return;
}


//  convert mouse position from window space to image space

void mouse_convert(int &xpos, int &ypos)
{
   xpos = int((xpos - Dorgx) / Mscale + Iorgx + 0.5);
   ypos = int((ypos - Dorgy) / Mscale + Iorgy + 0.5);

   if (xpos < 0) xpos = 0;                                                 //  if outside image put at edge
   if (ypos < 0) ypos = 0;                                                 //                   v.8.4

   if (E3rgb48) { 
      if (xpos >= E3ww) xpos = E3ww-1;
      if (ypos >= E3hh) ypos = E3hh-1;
   }
   else {
      if (xpos >= Fww) xpos = Fww-1;
      if (ypos >= Fhh) ypos = Fhh-1;
   }
   
   return;
}


/**************************************************************************/

//  keyboard event function - some toolbar buttons have KB equivalents
//  GDK key symbols: /usr/include/gtk-2.0/gdk/gdkkeysyms.h

int   KBcontrolkey = 0;

int KBpress(GtkWidget *win, GdkEventKey *event, void *)                    //  prevent propagation of key-press
{                                                                          //    events to toolbar buttons
   KBkey = event->keyval;
   if (KBkey == 65507) KBcontrolkey = 1;                                   //  Ctrl key is pressed    v.8.3
   return 1;
}

int KBrelease(GtkWidget *win, GdkEventKey *event, void *)
{
   KBkey = event->keyval;

   if (KBkeyCBfunc) {                                                      //  pass to handler function
      (* KBkeyCBfunc)();                                                   //  v.6.5
      KBkey = 0;
      return 1;
   }

   if (KBcapture) return 1;                                                //  let function handle it

   if (KBkey == 65507) KBcontrolkey = 0;                                   //  Ctrk key released   v.8.3
   
   if (KBcontrolkey) {
      if (KBkey == GDK_s) m_save(0,0);                                     //  Ctrl-* shortcuts   v.8.3
      if (KBkey == GDK_S) m_saveas(0,0);  
      if (KBkey == GDK_q) m_quit(0,0);  
      if (KBkey == GDK_Q) m_quit(0,0);  
   }

   if (KBkey == GDK_G) m_gallery(0,0);                                     //  key G  >>  image gallery   v.8.4.2
   if (KBkey == GDK_g) m_gallery(0,0);  

   if (KBkey == GDK_Left) m_prev(0,0);                                     //  arrow keys  >>  prev/next image
   if (KBkey == GDK_Right) m_next(0,0);  

   if (KBkey == GDK_plus) m_zoom(null, (char *) "+");                      //  +/- keys  >>  zoom in/out
   if (KBkey == GDK_equal) m_zoom(null, (char *) "+");                     //  = key: same as +
   if (KBkey == GDK_minus) m_zoom(null, (char *) "-");
   if (KBkey == GDK_KP_Add) m_zoom(null, (char *) "+");                    //  keypad +
   if (KBkey == GDK_KP_Subtract) m_zoom(null, (char *) "-");               //  keypad -
   
   if (KBkey == GDK_Z) m_zoom(null, (char *) "Z");                         //  Z key: zoom to 100%
   if (KBkey == GDK_z) m_zoom(null, (char *) "Z");

   if (KBkey == GDK_Escape) {                                              //  escape                 v.7.0
      if (Fslideshow) m_slideshow(0,0);                                    //  exit slideshow mode
      Fslideshow = 0;                                                      //  v.8.6
   }

   if (KBkey == GDK_Delete) m_trash(0,0);                                  //  delete  >>  trash

   if (KBkey == GDK_R) turn_image(+90);                                    //  keys L, R  >>  rotate
   if (KBkey == GDK_r) turn_image(+90);
   if (KBkey == GDK_L) turn_image(-90);
   if (KBkey == GDK_l) turn_image(-90);
   
   KBkey = 0;                                                              //  v.7.0
   return 1;
}


/**************************************************************************/

//  refresh overlay lines on top of image
//  arg = 1:   paint lines only (because window repainted)
//        2:   erase lines and forget them
//        3:   erase old lines, paint new lines, save new in old

void paint_toplines(int arg)                                               //  v.8.3
{
   int      ii;

   if (arg == 2 || arg == 3)                                               //  erase old lines
      for (ii = 0; ii < Nptoplines; ii++)
         erase_line(ptoplinex1[ii],ptopliney1[ii],ptoplinex2[ii],ptopliney2[ii]);
   
   if (arg == 1 || arg == 3)                                               //  draw new lines
      for (ii = 0; ii < Ntoplines; ii++)
         draw_line(toplinex1[ii],topliney1[ii],toplinex2[ii],topliney2[ii]);

   if (arg == 2) {
      Nptoplines = Ntoplines = 0;                                          //  forget lines
      return;
   }

   for (ii = 0; ii < Ntoplines; ii++)                                      //  save for future erase
   {
      ptoplinex1[ii] = toplinex1[ii];
      ptopliney1[ii] = topliney1[ii];
      ptoplinex2[ii] = toplinex2[ii];
      ptopliney2[ii] = topliney2[ii];
   }

   Nptoplines = Ntoplines;

   return;
}


/**************************************************************************/

//  refresh overlay arc (circle/ellipse) on top of image
//  arg = 1:   paint arc only (because window repainted)
//        2:   erase arc and forget it
//        3:   erase old arc, paint new arc, save new in old

void paint_toparc(int arg)                                                 //  v.8.3
{
   int      arcx, arcy, arcw, arch;

   if (ptoparc && (arg == 2 || arg == 3)) {                                //  erase old arc
      arcx = int((ptoparcx-Iorgx) * Mscale + Dorgx + 0.5);                 //  image to window space
      arcy = int((ptoparcy-Iorgy) * Mscale + Dorgy + 0.5);
      arcw = int(ptoparcw * Mscale);
      arch = int(ptoparch * Mscale);

      gdk_gc_set_function(gdkgc,GDK_INVERT);                               //  invert pixels
      gdk_draw_arc(drWin->window,gdkgc,0,arcx,arcy,arcw,arch,0,64*360);    //  draw arc
      gdk_gc_set_function(gdkgc,GDK_COPY);
   }
   
   if (toparc && (arg == 1 || arg == 3)) {                                 //  draw new arc
      arcx = int((toparcx-Iorgx) * Mscale + Dorgx + 0.5);                  //  image to window space
      arcy = int((toparcy-Iorgy) * Mscale + Dorgy + 0.5);
      arcw = int(toparcw * Mscale);
      arch = int(toparch * Mscale);

      gdk_gc_set_function(gdkgc,GDK_INVERT);                               //  invert pixels
      gdk_draw_arc(drWin->window,gdkgc,0,arcx,arcy,arcw,arch,0,64*360);    //  draw arc
      gdk_gc_set_function(gdkgc,GDK_COPY);
   }

   if (arg == 2) {
      toparc = ptoparc = 0;                                                //  forget arcs
      return;
   }
   
   ptoparc = toparc;                                                       //  save for future erase
   ptoparcx = toparcx;
   ptoparcy = toparcy;
   ptoparcw = toparcw;
   ptoparch = toparch;

   return;
}


/**************************************************************************/

//  draw red/green line. coordinates are in image space.                   //  overhauled   v.8.4.2

void draw_line(int ix1, int iy1, int ix2, int iy2)
{
   void draw_pixel(double pxm, double pym);

   double      x1, y1, x2, y2;   
   double      pxm, pym, slope;
   
   x1 = Mscale * (ix1-Iorgx);                                              //  image to window space
   y1 = Mscale * (iy1-Iorgy);
   x2 = Mscale * (ix2-Iorgx);
   y2 = Mscale * (iy2-Iorgy);
   
   if (abs(y2 - y1) > abs(x2 - x1)) {
      slope = 1.0 * (x2 - x1) / (y2 - y1);
      if (y2 > y1) {
         for (pym = y1; pym <= y2; pym++) {
            pxm = round(x1 + slope * (pym - y1));
            draw_pixel(pxm,pym);
         }
      }
      else {
         for (pym = y1; pym >= y2; pym--) {
            pxm = round(x1 + slope * (pym - y1));
            draw_pixel(pxm,pym);
         }
      }
   }
   else {
      slope = 1.0 * (y2 - y1) / (x2 - x1);
      if (x2 > x1) {
         for (pxm = x1; pxm <= x2; pxm++) {
            pym = round(y1 + slope * (pxm - x1));
            draw_pixel(pxm,pym);
         }
      }
      else {
         for (pxm = x1; pxm >= x2; pxm--) {
            pym = round(y1 + slope * (pxm - x1));
            draw_pixel(pxm,pym);
         }
      }
   }

   gdk_gc_set_foreground(gdkgc,&black);
   return;
}

void draw_pixel(double px, double py)
{
   int            pxn, pyn;
   static int     flip = 0;
   
   pxn = int(px);
   pyn = int(py);
   
   if (pxn < 0 || pxn > dww-1) return;
   if (pyn < 0 || pyn > dhh-1) return;
   
   if (++flip > 2) flip = -3;
   if (flip < 0) gdk_gc_set_foreground(gdkgc,&red);
   else gdk_gc_set_foreground(gdkgc,&green);
   gdk_draw_point(drWin->window, gdkgc, pxn + Dorgx, pyn + Dorgy);
   
   return;
}


//  erase line. refresh line path from Drgb24 pixels.

void erase_line(int ix1, int iy1, int ix2, int iy2)
{
   void erase_pixel(double pxm, double pym);

   double      x1, y1, x2, y2;   
   double      pxm, pym, slope;
   
   x1 = Mscale * (ix1-Iorgx);
   y1 = Mscale * (iy1-Iorgy);
   x2 = Mscale * (ix2-Iorgx);
   y2 = Mscale * (iy2-Iorgy);
   
   if (abs(y2 - y1) > abs(x2 - x1)) {
      slope = 1.0 * (x2 - x1) / (y2 - y1);
      if (y2 > y1) {
         for (pym = y1; pym <= y2; pym++) {
            pxm = x1 + slope * (pym - y1);
            erase_pixel(pxm,pym);
         }
      }
      else {
         for (pym = y1; pym >= y2; pym--) {
            pxm = x1 + slope * (pym - y1);
            erase_pixel(pxm,pym);
         }
      }
   }
   else {
      slope = 1.0 * (y2 - y1) / (x2 - x1);
      if (x2 > x1) {
         for (pxm = x1; pxm <= x2; pxm++) {
            pym = y1 + slope * (pxm - x1);
            erase_pixel(pxm,pym);
         }
      }
      else {
         for (pxm = x1; pxm >= x2; pxm--) {
            pym = y1 + slope * (pxm - x1);
            erase_pixel(pxm,pym);
         }
      }
   }

   return;
}

void erase_pixel(double px, double py)
{
   int            pxn, pyn;
   
   pxn = int(px);
   pyn = int(py);
   
   if (pxn < 0 || pxn > dww-1) return;
   if (pyn < 0 || pyn > dhh-1) return;

   uint8 *pixel = (uint8 *) Drgb24->bmp + (pyn * dww + pxn) * 3;
   gdk_draw_rgb_image(drWin->window, gdkgc, pxn + Dorgx, pyn + Dorgy, 
                                 1, 1, nodither, pixel, dww * 3);         
   return;
}


/**************************************************************************/

//  process top-level menu entry

void topmenufunc(GtkWidget *, const char *menu)
{
   topmenu = (char *) menu;                                                //  remember top-level menu in
   return;                                                                 //    case this is needed somewhere
}


/**************************************************************************
      file functions
***************************************************************************/

//  display image gallery (thumbnails) in a separate window

void m_gallery(GtkWidget *, const char *)
{
   if (image_file) image_gallery(image_file,"paint1",0,m_gallery2);        //  show image gallery window
   else {
      char *pp = get_current_dir_name();                                   //  initz. gallery file list and show
      if (pp) {
         image_gallery(pp,"init",0,m_gallery2);
         image_gallery(0,"paint1");
         free(pp);
         Fsearchlist = 0;
      }
   }

   return;
}


//  clicked thumbnails will call this function

void  m_gallery2(char *file)
{
   if (zdburn) {                                                           //  add file for CD burn  v.7.2
      burn_insert_file(file);
      return;
   }

   f_open((const char *) file);
   return;
}


/**************************************************************************/

//  open menu function

void m_open(GtkWidget *, const char *)
{
   Fsearchlist = 0;
   f_open(null);
   return;
}


/**************************************************************************/

//  open drag-drop file

void  m_open_drag(int x, int y, char *file)                                //  v.7.3
{
   Fsearchlist = 0;                                                        //  v.8.1
   f_open(file);
   zfree(file);
   return;
}


/**************************************************************************/

//  open an image file from the list of recent image files

char     recent_selection[300];

void  m_recent(GtkWidget *, const char *)                                  //  v.8.1
{
   int  recent_dialog_event(zdialog *zd, const char *event);

   zdialog     *zd;
   int         ii, zstat;
   
   if (! recentfiles[0]) return;

   zd = zdialog_new(ZTX("Open Recent File"),mWin,BOK,Bcancel,0);
   zdialog_add_widget(zd,"combo","combo","dialog",0);

   for (ii = 0; ii < Nrecentfiles && recentfiles[ii]; ii++)                //  stuff files into combo box list
      zdialog_cb_app(zd,"combo",recentfiles[ii]);
   zdialog_stuff(zd,"combo",recentfiles[0]);

   *recent_selection = 0;
   zstat = zdialog_run(zd,recent_dialog_event,0);                          //  run dialog, blocking
   zdialog_free(zd);

   if (zstat == 1 && *recent_selection == '/') 
      f_open(recent_selection);                                            //  open selected file
   return;
}

int  recent_dialog_event(zdialog *zd, const char *event)                   //  dialog event function  v.8.2
{
   if (strNeq(event,"combo")) return 0;
   zdialog_fetch(zd,"combo",recent_selection,299);                         //  get user selection
   zdialog_send_response(zd,1);                                            //  complete dialog with OK status
   return 1;
}


//  add a file to the list of recent files                                 //  v.8.4

void  add_recent_file(const char *file)
{
   int      ii;

   for (ii = 0; ii < Nrecentfiles-1 && recentfiles[ii]; ii++)              //  find file in recent list   v.8.1
      if (strEqu(file,recentfiles[ii])) break;                             //    (or find last entry in list)
   if (recentfiles[ii]) zfree(recentfiles[ii]);                            //  free this slot in list
   for (; ii > 0; ii--) recentfiles[ii] = recentfiles[ii-1];               //  move list down to fill hole
   recentfiles[0] = strdupz(file);                                         //  current file >> first in list
   return;
}


/**************************************************************************/

//  open a file and initialize FI bitmap

void f_open(const char *filespec)
{
   int            cc, fposn, fcount;
   char           *pp, wtitle[200], fname[100], fdirk[100];
   RGB            *temp24;
   char           **orientation;
   const char     *orientationkey[1] = { exif_orientation_key };
   
   zmondirk("close",null,null);                                            //  reset monitored directory

   if (! menulock(1)) return;                                              //  lock menu
   if (mod_keep()) goto openfail;
   
   if (filespec) filespec = strdupz(filespec);
   else filespec = zgetfile(ZTX("Open Image File"),image_file,"open");
   if (! filespec) goto openfail;
   
   temp24 = image_load(filespec,24);                                       //  load image as RGB-24 pixmap
   if (! temp24) goto openfail;

   free_resources();                                                       //  free resources for old image file

   mutex_lock(&pixmaps_lock);                                              //  lock pixmaps

   image_file = (char *) filespec;                                         //  setup new image file
   Frgb24 = temp24;
   Fww = Frgb24->ww;
   Fhh = Frgb24->hh;
   
   pp = (char *) strrchr(image_file,'/');                                  //  get image file name
   strncpy0(fname,pp+1,99);
   cc = pp - image_file;
   if (cc < 99) strncpy0(fdirk,image_file,cc+2);                           //  get dirk/path/ if short enough
   else {
      strncpy(fdirk,image_file,96);                                        //  or use /dirk/path...
      strcpy(fdirk+95,"...");
   }
   image_position(image_file,fposn,fcount);                                //  position and count in gallery list

   Fzoom = 0;                                                              //  zoom level = fit window
   zoomx = zoomy = 0;                                                      //  no zoom center      v.7.5

   snprintf(wtitle,199,"%s  %d/%d  %s",fname,fposn,fcount,fdirk);          //  set window title
   gtk_window_set_title(GTK_WINDOW(mWin),wtitle);
   gtk_window_present(GTK_WINDOW(mWin));                                   //  bring main window to front
   if (zdtags) edit_tags_dialog();                                         //  update active tags dialog
   if (zdrename) rename_dialog();                                          //  update active rename dialog

   mutex_unlock(&pixmaps_lock);                                            //  unlock pixmaps

   Fimageturned = 0;
   orientation = exif_get(image_file,orientationkey,1);                    //  upright turned image
   if (*orientation) {
      if (strstr(*orientation,"90")) turn_image(90);
      if (strstr(*orientation,"270")) turn_image(270);
   }

   mwpaint2();                                                             //  refresh main window

   pp = image_gallery(image_file,"find",0);                                //  file in current image gallery ?
   if (! pp) {
      image_gallery(image_file,"init");                                    //  no, reset gallery file list
      image_gallery(0,"paint2");                                           //  refresh gallery window if active
      Fsearchlist = 0;
   }
   else zfree(pp);

   if (! Fsearchlist) {                                                    //  start monitoring this directory
      pp = (char *) strrchr(image_file,'/');                               //    if not from search tags
      *pp = 0;                                                             //                v.6.4
      zmondirk("open",image_file,null);
      *pp = '/';
   }
   
   add_recent_file(image_file);                                            //  first in recent files list

openfail:
   menulock(0);
   mwpaint2();
   return;
}


/**************************************************************************/

//  open a raw image file and convert to tiff-48 

void m_raw(GtkWidget *, const char *)
{
   char           *pp, *rawfile, *outfile;
   char           command[1000];
   int            yn, err;
   struct stat    fstat;
   
   if (! menulock(1)) return;                                              //  lock menu
   if (mod_keep()) goto openfail;
   
   if (! Fufraw) {
      zmessageACK(ZTX("Package ufraw required for this function"));
      goto openfail;
   }
   
   rawfile = zgetfile(Bopenrawfile,image_file,"open");
   if (! rawfile) goto openfail;
   
   err = stat(rawfile,&fstat);
   if (err) {
      zmessageACK(strerror(errno));
      zfree(rawfile);
      goto openfail;
   }

   if (S_ISDIR(fstat.st_mode)) {
      pp = zgetfile(Bopenrawfile,rawfile,"open");
      zfree(rawfile);
      if (! pp) goto openfail;
      rawfile = pp;
   }

   outfile = strdupz(rawfile,5);
   pp = (char *) strrchr(rawfile,'.');
   pp = outfile + (pp - rawfile);
   strcpy(pp,".tiff");

   yn = zmessageYN(ZTX("Convert raw file to 48-bit tiff format? \n"
                       " (this may take a while) "));
   if (! yn) {
      zfree(outfile);
      goto openfail;
   }
   
   gdk_window_set_cursor(drWin->window,busycursor);                        //  set function busy cursor  v.8.4
   zmainloop();

   snprintf(command,999,"ufraw-batch --out-type=tiff --out-depth=16"       //  new ufraw   v.6.5
               " --overwrite --output=\"%s\" \"%s\" ",outfile,rawfile);
   err = system(command);

   if (err) {
      snprintf(command,999,"ufraw-batch --out-type=tiff16"                 //  old ufraw
               " --overwrite --output=\"%s\" \"%s\" ",outfile,rawfile);
      err = system(command);
   }

   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor

   if (err) {
      zmessageACK(wstrerror(err));
      zfree(outfile);
      goto openfail;
   }

   menulock(0);                                                            //  bugfix  v.6.3
   image_gallery(outfile,"init");                                          //  update image gallery file list
   image_gallery(0,"paint2");                                              //  refresh gallery window if active
   Fsearchlist = 0;
   f_open(outfile);                                                        //  open converted file
   return;

openfail:
   menulock(0);
   mwpaint2();
   return;
}


/**************************************************************************/

//  open previous or next file in same gallery as last file opened

void m_prev(GtkWidget *, const char *)
{
   if (! image_file) return;

   int mods = 0;
   while (zmondirk("event",null,null) > 0) mods++;                         //  detect directory mods  v.6.4
   if (mods) image_gallery(image_file,"init");                             //  refresh gallery file list

   char *pp = image_gallery(image_file,"prev");
   if (! pp) return;
   if (image_file_type(pp) == 2) f_open(pp);
   zfree(pp);
   return;
}

void m_next(GtkWidget *, const char *)
{
   if (! image_file) return;

   int mods = 0;
   while (zmondirk("event",null,null) > 0) mods++;                         //  v.6.4
   if (mods) image_gallery(image_file,"init");

   char *pp = image_gallery(image_file,"next");
   if (! pp) return;
   if (image_file_type(pp) == 2) f_open(pp);
   zfree(pp);
   return;
}


/**************************************************************************/

//  save (modified) image to same file - no confirmation of overwrite.

void m_save(GtkWidget *, const char *)                                     //  v.8.3
{   
   char           *outfile, *pext;
   const char     *format;
   
   if (! image_file) return;
   
   format = "jpeg";                                                        //  use jpeg unless tiff file
   if (strEqu(file_type,"tiff")) format = "tiff-24";                       //    or 48-bits/pixel file
   if (file_bpp == 48) format = "tiff-48";

   strcpy(jpeg_quality,def_jpeg_quality);                                  //  set default jpeg quality

   outfile = strdupz(image_file,8);                                        //  use input file name

   pext = (char *) strrchr(outfile,'/');                                   //  force compatible file .ext
   if (pext) pext = (char *) strrchr(pext,'.');                            //    if not already
   if (! pext) pext = outfile + strlen(outfile);
   if (strEqu(format,"jpeg") && ! strcmpv(pext,".jpg",".JPG",".jpeg",".JPEG",0))
      strcpy(pext,".jpeg");
   if (strnEqu(format,"tiff",4) && ! strcmpv(pext,".tif",".TIF",".tiff",".TIFF",0))
      strcpy(pext,".tiff");

   f_save(outfile,format);
   zfree(outfile);
   return;
}


/**************************************************************************/

//  save (modified) image to new file, confirm if overwrite existing file.

void m_saveas(GtkWidget *, const char *)
{   
   GtkWidget      *fdialog, *fchooser, *hbox;
   GtkWidget      *label1, *tiff2, *tiff4, *jpeg, *jqlab, *jqval;
   char           *outfile = 0, *outfile2 = 0, *pext;
   const char     *format;
   int            ii, err, yn, status;
   struct stat    fstat;

   if (! image_file) return;

   fdialog = gtk_dialog_new_with_buttons(ZTX("Save File"),                 //  build file save dialog
                           GTK_WINDOW(mWin), GTK_DIALOG_MODAL,
                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
                           GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, null);

   gtk_window_set_default_size(GTK_WINDOW(fdialog),500,400);

   fchooser = gtk_file_chooser_widget_new(GTK_FILE_CHOOSER_ACTION_SAVE);
   gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(fchooser),image_file);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(fdialog)->vbox),fchooser);
   
   hbox = gtk_hbox_new(0,0);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(fdialog)->vbox),hbox);
   gtk_box_set_child_packing(GTK_BOX(GTK_DIALOG(fdialog)->vbox),hbox,0,0,10,GTK_PACK_END);

   label1 = gtk_label_new("file type");                                    //  add file type options
   tiff2 = gtk_radio_button_new_with_label(null,"tiff-24");
   tiff4 = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tiff2),"tiff-48");
   jpeg = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tiff2),"jpeg");
   jqlab = gtk_label_new("jpeg quality");
   jqval = gtk_entry_new();
   gtk_entry_set_width_chars(GTK_ENTRY(jqval),3);
   gtk_box_pack_start(GTK_BOX(hbox),label1,0,0,5);
   gtk_box_pack_start(GTK_BOX(hbox),tiff2,0,0,5);                          //  simplified    v.6.7.1
   gtk_box_pack_start(GTK_BOX(hbox),tiff4,0,0,5);
   gtk_box_pack_start(GTK_BOX(hbox),jpeg,0,0,10);                          //  jpeg quality   v.6.7.2
   gtk_box_pack_start(GTK_BOX(hbox),jqlab,0,0,5);
   gtk_box_pack_start(GTK_BOX(hbox),jqval,0,0,5);
   
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(jpeg),1);                //  set default output file type
   gtk_entry_set_text(GTK_ENTRY(jqval),def_jpeg_quality);                  //  default jpeg quality

   if (strEqu(file_type,"tiff"))
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tiff2),1);
   if (file_bpp == 48)
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tiff4),1);

dialog_run:

   gtk_widget_show_all(fdialog);                                           //  run dialog
   status = gtk_dialog_run(GTK_DIALOG(fdialog));
   if (status != GTK_RESPONSE_ACCEPT) {                                    //  user cancelled
      gtk_widget_destroy(fdialog);
      return;
   }

   outfile2 = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(fchooser));   //  get user inputs
   if (! outfile2) goto dialog_run;
   outfile = strdupz(outfile2,8);
   g_free(outfile2);

   format = "jpeg";
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tiff2)))
      format = "tiff-24";
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tiff4)))
      format = "tiff-48";

   if (strEqu(format,"jpeg")) {                                            //  get jpeg quality   v.6.7.2
      ii = atoi(gtk_entry_get_text(GTK_ENTRY(jqval)));
      if (ii < 1 || ii > 100) {
         zmessageACK("jpeg quality must be 1-100");
         goto dialog_run;
      }
      sprintf(jpeg_quality,"%d",ii);
   }

   gtk_widget_destroy(fdialog);                                            //  kill dialog

   pext = (char *) strrchr(outfile,'/');                                   //  force compatible file .ext
   if (pext) pext = (char *) strrchr(pext,'.');                            //    if not already
   if (! pext) pext = outfile + strlen(outfile);
   if (strEqu(format,"jpeg") && ! strcmpv(pext,".jpg",".JPG",".jpeg",".JPEG",0))
      strcpy(pext,".jpeg");
   if (strnEqu(format,"tiff",4) && ! strcmpv(pext,".tif",".TIF",".tiff",".TIFF",0))
      strcpy(pext,".tiff");

   err = stat(outfile,&fstat);                                             //  check if file exists
   if (! err) {
      yn = zmessageYN(ZTX("Overwrite file? \n %s"),outfile);               //  confirm overwrite
      if (! yn) {
         zfree(outfile);
         return;
      }
   }
   
   f_save(outfile,format);
   zfree(outfile);
   return;
}


/**************************************************************************/

//  file save helper function - do the actual file save

void f_save(const char *outfile, const char *format)                       //  code simplification   v.8.6
{
   FIB            *fib;
   PXB            *pxb;
   RGB            *rgb24, *rgb48;
   const char     *exifkey[3] = { exif_width_key, exif_height_key, exif_orientation_key };
   const char     *exifdata[3];
   char           wwchar[8], hhchar[8];
   char           *tempfile, command[2000];
   int            err, status;
   struct stat    fstat;

   update_filetags(image_file);                                            //  commit poss. tag changes   v.8.3

   tempfile = strdupz(get_zuserdir(),24);                                  //  use temp output file
   strcat(tempfile,"/temp_");                                              //    ~/.fotoxx/temp_ppppp.ext
   strcat(tempfile,PIDstring);
   strcat(tempfile,".");
   strncat(tempfile,format,4);

   status = 0;
   fib = 0;
   pxb = 0;
   
   gdk_window_set_cursor(drWin->window,busycursor);                        //  set function busy cursor  v.8.4
   zmainloop();
   
   if (strEqu(format,"jpeg")) {                                            //  save as JPEG file
      if (Frgb48) {
         rgb24 = RGB_convbpp(Frgb48);
         pxb = RGB_PXB(rgb24);
         RGB_free(rgb24);
      }                                                                    //  use GDK pixbuf for jpeg
      else  pxb = RGB_PXB(Frgb24);
      status = gdk_pixbuf_save(pxb,tempfile,"jpeg",gerror,"quality",jpeg_quality,null);
   }
   
   if (strEqu(format,"tiff-24")) {                                         //  save as TIFF-24 file
      if (Frgb48) {
         rgb24 = RGB_convbpp(Frgb48);
         fib = RGB_FIB(rgb24);
         RGB_free(rgb24);
      }
      else  fib = RGB_FIB(Frgb24);
      status = FreeImage_Save(FIF_TIFF,fib,tempfile,0);
   }
   
   if (strEqu(format,"tiff-48")) {                                         //  save as TIFF-48 file
      if (Frgb48) fib = RGB_FIB(Frgb48);
      else {
         if (file_bpp == 48) rgb48 = image_load(image_file,48);            //  use original 48 bpp file
         else  rgb48 = RGB_convbpp(Frgb24);
         if (rgb48) {
            fib = RGB_FIB(rgb48);
            RGB_free(rgb48);
         }
      }
      status = FreeImage_Save(FIF_TIFF,fib,tempfile,0);
   }

   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor

   if (fib) FreeImage_Unload(fib);
   if (pxb) g_object_unref(pxb);

   if (! status) {
      zmessageACK(ZTX("Unable to save image"));
      snprintf(command,1999,"rm \"%s\"",tempfile);                         //  clean up
      err = system(command);
      zfree(tempfile);
      return;
   }

   snprintf(wwchar,6,"%d",Fww);
   snprintf(hhchar,6,"%d",Fhh);                                            //  copy EXIF data from source file
   exifdata[0] = wwchar;                                                   //    with possible new dimensions
   exifdata[1] = hhchar;
   exifdata[2] = "";                                                       //  assume saved upright     v.6.2
   err = exif_copy(image_file,tempfile,exifkey,exifdata,3);
   if (err) zmessageACK(ZTX("Unable to copy EXIF data"));
   
   snprintf(command,1999,"cp -f \"%s\" \"%s\" ",tempfile,outfile);         //  copy to final destination
   err = system(command);
   if (err) zmessageACK(ZTX("Unable to save image: %s"),wstrerror(err));

   snprintf(command,1999,"rm \"%s\"",tempfile);                            //  clean up
   err = system(command);
   zfree(tempfile);
   
   load_filetags(outfile);                                                 //  update assigned tags file
   update_asstags(outfile);
   
   Fmodified = Fimageturned = 0;                                           //  reset file modified status
   Fsaved = Pundo;                                                         //  note which mods are saved    v.8.3

   if (! Fsearchlist && samedirk(image_file,outfile)) {                    //  update image gallery file list
      image_gallery(image_file,"init");                                    //    if output in same directory
      image_gallery(0,"paint2");                                           //  refresh gallery window if active
      gtk_window_present(GTK_WINDOW(mWin));                                //  bring main to foreground
   }

   stat(outfile,&fstat);                                                   //  update file size    v.8.3
   file_MB = 1.0 * fstat.st_size / mega;
   SBupdate++;                                                             //  update status bar

   add_recent_file(outfile);                                               //  first in recent files list   v.8.4
   return;
}


/**************************************************************************/

//  print image files

void m_print(GtkWidget *, const char *)
{
   int      err;
   char     command[1000];
   
   if (! image_file) return;

   if (! Fprintoxx) {
      zmessageACK(ZTX("printoxx program not found (see user guide)"));
      return;
   }

   snprintf(command,999,"printoxx \"%s\" &",image_file);                   //  send curr. file to printoxx
   err = system(command);
   if (err) zmessLogACK(wstrerror(err));
   return;
}


/**************************************************************************/

//  Delete image file - move image_file to trash.
//  Trash has no standard location, so use a trash folder on the desktop.
//  User must delete or move to the official trash bin.

void m_trash(GtkWidget *, const char *)
{
   int            err, yn;
   char           command[1000], trashdir[100];
   struct stat    trstat;

   if (! image_file) return;                                               //  nothing to trash
   
   err = stat(image_file,&trstat);                                         //  get file status
   if (err) {
      zmessLogACK(strerror(errno));
      return;
   }

   if (! (trstat.st_mode & S_IWUSR)) {                                     //  check permission
      yn = zmessageYN(ZTX("Move read-only file to trash?"));
      if (! yn) return;
      trstat.st_mode |= S_IWUSR;
      chmod(image_file,trstat.st_mode);
   }
   
   snprintf(trashdir,99,"%s/%s",getenv("HOME"),ftrash);                    //  get full fotoxx trash file
   
   trstat.st_mode = 0;
   err = stat(trashdir,&trstat);
   if (! S_ISDIR(trstat.st_mode)) {
      snprintf(command,999,"mkdir -m 0750 \"%s\"",trashdir);
      err = system(command);
      if (err) {
         zmessLogACK(ZTX("Cannot create trash folder: %s"),wstrerror(err));
         return;
      }
   }

   snprintf(command,999,"cp \"%s\" \"%s\" ",image_file,trashdir);          //  copy image file to trash
   err = system(command);
   if (err) {
      zmessLogACK(ZTX("error: %s"),wstrerror(err));
      return;
   }

   snprintf(command,999,"rm \"%s\"",image_file);                           //  delete image file
   err = system(command);
   if (err) {
      zmessLogACK(ZTX("error: %s"),wstrerror(err));
      return;
   }
   
   update_asstags(image_file,1);                                           //  delete in assigned tags file
   if (! Fsearchlist) image_gallery(image_file,"init");                    //  reset image gallery file list
   image_gallery(0,"paint2");                                              //  refresh gallery window if active
   m_next(0,0);                                                            //  step to next file if there

   return;
}


/**************************************************************************/

//  rename menu function

char     rename_old[100] = "";
char     rename_new[100] = "";

void  m_rename(GtkWidget *, const char *)
{
   rename_dialog();                                                        //  activate rename dialog
   return;
}
   

//  activate rename dialog, stuff data from current file

void rename_dialog()
{
   int rename_dialog_event(zdialog *zd, const char *event);
   int rename_dialog_compl(zdialog *zd, int zstat);
   
   char     *pdir, *pfile, *pext;

   if (! image_file) return;

   if (! zdrename)                                                         //  restart dialog
   {
      zdrename = zdialog_new(ZTX("Rename Image File"),mWin,Bcancel,0);
      zdialog_add_widget(zdrename,"hbox","hb1","dialog",0,"space=10");
      zdialog_add_widget(zdrename,"vbox","vb1","hb1",0,"homog|space=5");
      zdialog_add_widget(zdrename,"vbox","vb2","hb1",0,"homog|expand");

      zdialog_add_widget(zdrename,"button","Bold","vb1",ZTX("old name"));
      zdialog_add_widget(zdrename,"button","Bnew","vb1",ZTX("rename to"));
      zdialog_add_widget(zdrename,"button","Bprev","vb1",ZTX("previous"));

      zdialog_add_widget(zdrename,"hbox","hb21","vb2",0);                  //  [ old name ] [ oldname  ]
      zdialog_add_widget(zdrename,"hbox","hb22","vb2",0);                  //  [ new name ] [ newname  ] [+1]
      zdialog_add_widget(zdrename,"hbox","hb23","vb2",0);                  //  [ previous ] [ prevname ]

      zdialog_add_widget(zdrename,"label","Lold","hb21");
      zdialog_add_widget(zdrename,"entry","Enew","hb22",0,"expand|scc=30");
      zdialog_add_widget(zdrename,"button","B+1","hb22"," +1 ","space=5");
      zdialog_add_widget(zdrename,"label","Lprev","hb23");

      zdialog_run(zdrename,rename_dialog_event,rename_dialog_compl);       //  start dialog
   }

   parsefile(image_file,&pdir,&pfile,&pext);
   strncpy0(rename_old,pfile,99);
   strncpy0(rename_new,pfile,99);
   zdialog_stuff(zdrename,"Lold",rename_old);                              //  current file name
   zdialog_stuff(zdrename,"Enew",rename_new);                              //  entered file name

   return;
}


//  rename dialog event and completion functions

int rename_dialog_event(zdialog *zd, const char *event)
{
   char           *pp, *pdir, *pfile, *pext, *pnew, command[2000];
   int            nseq, digits, ccp, ccn, ccx, err;
   struct stat    statb;
   
   if (strEqu(event,"Bold"))                                               //  reset to current file name
      zdialog_stuff(zd,"Enew",rename_old);

   if (strEqu(event,"Bprev")) {                                            //  previous name >> new name
      zdialog_fetch(zd,"Lprev",rename_new,99);
      zdialog_stuff(zd,"Enew",rename_new);
   }

   if (strEqu(event,"B+1"))                                                //  increment sequence number
   {
      zdialog_fetch(zd,"Enew",rename_new,94);                              //  get entered filename
      pp = rename_new + strlen(rename_new);
      digits = 0;
      while (pp[-1] >= '0' && pp[-1] <= '9') {
         pp--;                                                             //  look for NNN in filenameNNN
         digits++;
      }
      nseq = 1 + atoi(pp);                                                 //  NNN + 1
      if (nseq > 9999) nseq = 0;
      if (digits < 2) digits = 2;                                          //  keep digit count if enough
      if (nseq > 99 && digits < 3) digits = 3;                             //  use leading zeros
      if (nseq > 999 && digits < 4) digits = 4;
      snprintf(pp,digits+1,"%0*d",digits,nseq);
      zdialog_stuff(zd,"Enew",rename_new);
   }

   if (strEqu(event,"Bnew")) 
   {
      parsefile(image_file,&pdir,&pfile,&pext);                            //  existing /directories/file.ext

      zdialog_fetch(zd,"Enew",rename_new,94);                              //  new file name from user

      ccp = strlen(pdir);                                                  //  length of /directories/
      ccn = strlen(rename_new);                                            //  length of file
      if (pext) ccx = strlen(pext);                                        //  length of .ext
      else ccx = 0;

      pnew = zmalloc(ccp + ccn + ccx + 1);                                 //  put it all together
      strncpy(pnew,image_file,ccp);                                        //   /directories/file.ext
      strcpy(pnew+ccp,rename_new);
      if (ccx) strcpy(pnew+ccp+ccn,pext);
      
      err = stat(pnew,&statb);                                             //  check for new name exists
      if (! err) {
         zmessageACK(ZTX("The target file already exists"));
         zfree(pnew);
         return 0;
      }
      
      snprintf(command,1999,"cp \"%s\" \"%s\"",image_file,pnew);           //  copy to new file
      err = system(command);
      if (err) {
         zmessageACK(ZTX("Rename failed \n %s"),wstrerror(err));
         zfree(pnew);
         return 0;
      }

      zdialog_stuff(zd,"Lprev",rename_new);                                //  set previous name in dialog

      load_filetags(pnew);                                                 //  update assigned tags file
      update_asstags(pnew);
      zfree(pnew);

      pnew = strdupz(image_file);                                          //  save file name to be deleted
      m_next(0,0);                                                         //  move to image_file + 1
      snprintf(command,999,"rm \"%s\"",pnew);                              //  delete old file
      err = system(command);
      zfree(pnew);

      if (! Fsearchlist) image_gallery(image_file,"init");                 //  update image gallery file list
      image_gallery(0,"paint2");                                           //  refresh gallery window if active
      gtk_window_present(GTK_WINDOW(mWin));                                //  bring main to foreground
   }

   return 0;
}


int rename_dialog_compl(zdialog *zd, int zstat)
{
   zdialog_free(zdrename);                                                 //  kill dialog
   zdrename = null;
   return 0;
}


/**************************************************************************/

//  forced quit - can cause running function to crash

void m_quit(GtkWidget *, const char *)
{
   if (image_file) update_filetags(image_file);                            //  commit tag changes, if any
   if (mod_keep()) return;                                                 //  keep or discard pending changes
   printf("quit \n");
   Fshutdown++;
   save_fotoxx_state();                                                    //  save state for next session
   free_resources();                                                       //  delete temp files
   gtk_main_quit();                                                        //  gone forever
   return;
}


/**************************************************************************
      tools functions
***************************************************************************/

//  set new image zoom level or magnification

void m_zoom(GtkWidget *, const char *menu)
{
   int      ii, iww, ihh, Dww, Dhh;
   char     zoom;
   double   scalew, scaleh, fitscale;
   double   scales[9] = { 0.125, 0.176, 0.25, 0.354, 0.5, 0.71, 1.0, 1.41, 2.0 };
   
   if (strnEqu(menu,"Zoom",4)) zoom = menu[4];                             //  get + or -
   else  zoom = *menu;
   
   Dww = drWin->allocation.width;                                          //  drawing window size
   Dhh = drWin->allocation.height;
   
   if (E3rgb48) {                                                          //  bugfix  v.8.1
      iww = E3ww;
      ihh = E3hh;
   }
   else  {
      iww = Fww;
      ihh = Fhh;
   }

   if (iww > Dww || ihh > Dhh) {                                           //  get window fit scale
      scalew = 1.0 * Dww / iww;
      scaleh = 1.0 * Dhh / ihh;
      if (scalew < scaleh) fitscale = scalew;
      else fitscale = scaleh;
   }
   else fitscale = 1.0;                                                    //  if image < window use 100%
   
   if (zoom == '+') {                                                      //  zoom bigger
      if (! Fzoom) Fzoom = fitscale / 1.2;
      Fzoom = Fzoom * sqrt(2.0);                                           //  new scale: 41% bigger
      for (ii = 0; ii < 9; ii++)
         if (Fzoom < 1.01 * scales[ii]) break;                             //  next higher scale in table
      if (ii == 9) ii = 8;
      Fzoom = scales[ii];
      if (Fzoom < fitscale) Fzoom = 0;                                     //  image < window
   }

   if (zoom == '-') Fzoom = 0;                                             //  zoom to fit window

   if (zoom == 'Z') {
      if (Fzoom != 0) Fzoom = 0;                                           //  toggle 100% and fit window
      else  Fzoom = 1;
   }
   
   if (! Fzoom) zoomx = zoomy = 0;                                         //  no req. zoom center    v.7.5
   
   mwpaint2();                                                             //  refresh window
   return;
}


/**************************************************************************/

//  monitor test function

void m_montest(GtkWidget *, const char *)
{
   uint8       *pixel;
   int         red, green, blue;
   int         row, col, row1, row2;
   int         ww = 800, hh = 500;
   
   if (mod_keep()) return;
   if (! menulock(1)) return;

   mutex_lock(&pixmaps_lock);

   RGB_free(Frgb24);
   Frgb24 = RGB_make(ww,hh,24);
   Fww = ww;
   Fhh = hh;
   file_bpp = 24;
   file_MB = 0;

   for (red = 0; red <= 1; red++)
   for (green = 0; green <= 1; green++)
   for (blue = 0; blue <= 1; blue++)
   {
      row1 = 4 * red + 2 * green + blue;                                   //  row 0 to 7
      row1 = row1 * hh / 8;                                                //  stripe, 1/8 of image
      row2 = row1 + hh / 8;
      
      for (row = row1; row < row2; row++)
      for (col = 0; col < ww; col++)
      {
         pixel = (uint8 *) Frgb24->bmp + (row * ww + col) * 3;
         pixel[0] = red * 256 * col / ww;
         pixel[1] = green * 256 * col / ww;
         pixel[2] = blue * 256 * col / ww;
      }
   }

   Fzoom = 0;                                                              //  scale to window
   gtk_window_set_title(GTK_WINDOW(mWin),"monitor check");
   mutex_unlock(&pixmaps_lock);
   mwpaint2();                                                             //  repaint window
   menulock(0);
   return;
}


/**************************************************************************/

//  Rebuild assigned tags index and thumbnail files.
//  Process all image files within given top-level directory.              //  overhauled   v.8.4
//  Works incrementally and is very fast after the first run.

struct tt_tagrec {
   char        *file;                                                      //  image file filespec
   char        *tags;                                                      //  image file tags
   char        imagedate[12], filedate[16];                                //  both in one rec.
   int         update;
};

tt_tagrec   tt_old[max_images];
tt_tagrec   tt_new[max_images];


void m_index_tt(GtkWidget *, const char *)
{
   FILE           *fid;
   int            err, contx, fcount;
   char           buff[1000], stbartext[200];
   char           *subdirk, *pp, **ppv;
   char           *filespec1, *filespec2;
   char           *imagedate, *imagetags;
   cchar          *exifkeys[2] = { exif_date_key, exif_tags_key };
   cchar          *ppc;
   int            Nold = 0, Nnew = 0;
   int            comp, orec, nrec;
   struct tm      bdt;
   struct stat    statb;

   
   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(Bexiftoolmissing);
      return;
   }

   pp = zgetfile(ZTX("Select top image directory"),topdirk,"folder");
   if (! pp) return;
   if (topdirk) zfree(topdirk);
   topdirk = pp;

   if (! menulock(1)) return;

   gdk_window_set_cursor(drWin->window,busycursor);                        //  set function busy cursor  v.8.4
   zmainloop();

//  read current assigned tags file and build "oldlist" of tags

   fid = fopen(asstagsfile,"r");                                           //  open assigned tags file
   if (fid) 
   {
      while (true)
      {
         pp = fgets_trim(buff,999,fid);                                    //  read tag and file dates in one rec.
         if (! pp) break;
         
         tt_old[Nold].file = 0;
         tt_old[Nold].tags = 0;
         tt_old[Nold].imagedate[0] = 0;
         tt_old[Nold].filedate[0] = 0;

         ppc = strField(buff,' ',2);                                       //  date: yyyy:mm:dd yyyymmddhhmmss
         if (ppc) strncpy0(tt_old[Nold].imagedate,ppc,12);

         ppc = strField(buff,' ',3);
         if (ppc) strncpy0(tt_old[Nold].filedate,ppc,16);

         pp = fgets_trim(buff,999,fid);                                    //  tags: xxxxx xxxxx xxxxxxx xxxx
         if (! pp) break;
         tt_old[Nold].tags = strdupz(pp+6);
   
         pp = fgets_trim(buff,999,fid);                                    //  file: /directory/.../filename.jpg
         if (! pp) break;
         tt_old[Nold].file = strdupz(pp+6);
         
         fgets_trim(buff,999,fid);                                         //  read blank separator rec.
         
         if (++Nold == max_images) 
            zappcrash("more than %d image files: %d",max_images);
      }
      
      fclose(fid);
   }

   printf("%d current tag records found \n",Nold);
   
//  find all image files and create "newlist" with no tags

   snprintf(buff,999,"find \"%s\" -type d",topdirk);
   contx = 0;

   while ((subdirk = command_output(contx,buff)))                          //  find directories under top directory
   {
      pp = (char *) strrchr(subdirk,'/');
      if (pp && strEqu(pp,"/.thumbnails")) {                               //  ignore .thumbnails
         zfree(subdirk);
         continue;
      }

      image_gallery(subdirk,"init");                                       //  get all image files in directory
      filespec1 = image_gallery(subdirk,"first");

      while (filespec1)
      {
         if (image_file_type(filespec1) == 2) {                            //  construct new tag record
            err = stat(filespec1,&statb);
            if (err) continue;
            tt_new[Nnew].file = strdupz(filespec1);                        //  filespec
            tt_new[Nnew].tags = 0;                                         //  tags = empty
            tt_new[Nnew].imagedate[0] = 0;                                 //  tag date = empty
            gmtime_r(&statb.st_mtime,&bdt);
            sprintf(tt_new[Nnew].filedate,"%04d%02d%02d%02d%02d%02d",      //  file date = yyyymmddhhmmss
                     bdt.tm_year + 1900, bdt.tm_mon + 1, bdt.tm_mday,
                     bdt.tm_hour, bdt.tm_min, bdt.tm_sec);
            if (++Nnew == max_images) 
               zappcrash("more than %d image files: %d",max_images);
         }
         
         filespec2 = image_gallery(filespec1,"next");                      //  next image file
         zfree(filespec1);
         filespec1 = filespec2;
      }

      zfree(subdirk);
   }
   
   printf("found %d image files \n",Nnew);

//  sort old and new lists by filespec in preparation for merging them

   int index_tt_comp(cchar *rec1, cchar *rec2);

   HeapSort((char *) tt_old,sizeof(tt_tagrec),Nold,index_tt_comp);
   HeapSort((char *) tt_new,sizeof(tt_tagrec),Nnew,index_tt_comp);

//  merge and compare lists
//  if filespecs match and have the same date, then "tt_old" tags are OK

   for (orec = nrec = 0; nrec < Nnew; )
   {
      tt_new[nrec].update = 1;

      if (orec == Nold) comp = +1;
      else comp = strcmp(tt_old[orec].file, tt_new[nrec].file);
      
      if (comp > 0) nrec++;
      else if (comp < 0) orec++;

      else {
         if (strEqu(tt_new[nrec].filedate, tt_old[orec].filedate)) {
            tt_new[nrec].tags = tt_old[orec].tags;                         //  copy tags and tag date
            tt_old[orec].tags = 0;                                         //    from old to new
            strcpy(tt_new[nrec].imagedate, tt_old[orec].imagedate);
            tt_new[nrec].update = 0;
         }
         nrec++;
         orec++;
      }
   }

//  release old list memory

   for (orec = 0; orec < Nold; orec++)
   {
      zfree(tt_old[orec].file);
      if (tt_old[orec].tags) zfree(tt_old[orec].tags);
   }

//  process entries needing update in new list, get updated tags from image file EXIF data

   for (fcount = nrec = 0; nrec < Nnew; nrec++)
   {
      if (tt_new[nrec].update == 0) continue;

      ppv = exif_get(tt_new[nrec].file,exifkeys,2);
      imagedate = ppv[0];
      imagetags = ppv[1];
      
      if (imagedate && strlen(imagedate)) {
         if (strlen(imagedate) > 9) imagedate[10] = 0;                     //  truncate to yyyy:mm:dd
         strcpy(tt_new[nrec].imagedate,imagedate);
      }
      else strcpy(tt_new[nrec].imagedate,"null");
      
      if (imagetags && strlen(imagetags))
         tt_new[nrec].tags = strdupz(imagetags);
      else tt_new[nrec].tags = strdupz("null");

      if (imagedate) zfree(imagedate);
      if (imagetags) zfree(imagetags);

      snprintf(stbartext,199,"%5d %s",++fcount,tt_new[nrec].file);         //  update status bar
      stbar_message(STbar,stbartext);
   }

//  write new assigned tags file

   fid = fopen(asstagsfile,"w");                                           //  open assigned tags file
   if (! fid) zappcrash("cannot write tags file");

   for (nrec = 0; nrec < Nnew; nrec++)
   {
      fprintf(fid,"date: %s  %s""\n", tt_new[nrec].imagedate, tt_new[nrec].filedate);
      fprintf(fid,"tags: %s""\n",tt_new[nrec].tags);
      fprintf(fid,"file: %s""\n",tt_new[nrec].file);
      fprintf(fid,"\n");
   }
   
   fclose(fid);

//  look for missing thumbnails and create them

   for (fcount = nrec = 0; nrec < Nnew; nrec++)
   {
      pp = image_thumbfile(tt_new[nrec].file);                             //  find/update/create thumbnail
      if (pp) zfree(pp);
      snprintf(stbartext,199,"%5d %s",++fcount,tt_new[nrec].file);         //  update status bar
      stbar_message(STbar,stbartext);
   }

//  release new list memory

   for (nrec = 0; nrec < Nnew; nrec++)
   {
      zfree(tt_new[nrec].file);
      if (tt_new[nrec].tags) zfree(tt_new[nrec].tags);
   }

   image_gallery(image_file,"init");                                       //  reset image gallery file list
   image_gallery(0,"paint2");                                              //  refresh gallery window if active
   Fsearchlist = 0;

   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor
   menulock(0);
   return;
}


//  sort compare function - compare tag record filespecs and return
//   <0 | 0 | >0   for   file1 < | == | > file2

int index_tt_comp(cchar *rec1, cchar *rec2)
{
   char * file1 = ((tt_tagrec *) rec1)->file;
   char * file2 = ((tt_tagrec *) rec2)->file;
   return strcmp(file1,file2);
}


/**************************************************************************/

//  create or update brightness distribution graph

void m_brightgraph(GtkWidget *, const char *)                              //  menu function
{
   if (! Drgb24) return;

   if (brightgraph) {
      brightgraph_paint();
      return;
   }

   brightgraph = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(brightgraph),ZTX("Brightness Distribution"));
   gtk_window_set_transient_for(GTK_WINDOW(brightgraph),GTK_WINDOW(mWin));
   gtk_window_set_default_size(GTK_WINDOW(brightgraph),300,200);
   gtk_window_set_position(GTK_WINDOW(brightgraph),GTK_WIN_POS_MOUSE);
   
   drbrightgraph = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(brightgraph),drbrightgraph);

   G_SIGNAL(brightgraph,"destroy",brightgraph_destroy,0)
   G_SIGNAL(drbrightgraph,"expose-event",brightgraph_paint,0)

   gtk_widget_show_all(brightgraph);
   
   return;
}


void brightgraph_paint()                                                   //  paint graph window
{
   GdkGC       *gdkgc = 0;                                                 //  GDK graphics context
   int         brdist[20], nbins = 20;
   int         px, py, ii;
   int         winww, winhh;
   int         ww, hh, orgx, orgy;
   int         dist_maxbin = 0;
   uint8       *pixel;
   double      bright;
   
   gdkgc = gdk_gc_new(drbrightgraph->window);                              //  use separate graphics context  v.8.7
                                                                           //  (compensate new GDK bug)
   if (! brightgraph) return;
   if (! Drgb24) return;

   for (ii = 0; ii < nbins; ii++)                                          //  clear brightness distribution
      brdist[ii] = 0;

   mutex_lock(&pixmaps_lock);

   for (py = 0; py < dhh; py++)                                            //  compute brightness distribution
   for (px = 0; px < dww; px++)                                            //    for image in visible window
   {                                                                       //  Dww/hh -> dww/hh   bugfix v.7.4.2
      pixel = (uint8 *) Drgb24->bmp + (py * dww + px) * 3;
      bright = brightness(pixel);                                          //  0 to 255
      brdist[int(bright / 256 * nbins)]++;                                 //  0 to nbins
   }

   mutex_unlock(&pixmaps_lock);

   gdk_window_clear(drbrightgraph->window);

   winww = drbrightgraph->allocation.width;
   winhh = drbrightgraph->allocation.height;
   
   for (ii = 0; ii < nbins; ii++)
      if (brdist[ii] > dist_maxbin) dist_maxbin = brdist[ii];

   for (ii = 0; ii < nbins; ii++)
   {
      ww = winww / nbins;
      hh = int(0.9 * winhh * brdist[ii] / dist_maxbin);
      orgx = ii * ww;
      orgy = winhh - hh;
      gdk_draw_rectangle(drbrightgraph->window,gdkgc,1,orgx,orgy,ww,hh);
   }
   
   return;
}


void brightgraph_destroy()                                                 //  delete window
{
   if (brightgraph) gtk_widget_destroy(brightgraph);
   brightgraph = 0;
   return;
}


/**************************************************************************/

//  start a new instance of fotoxx in parallel

void m_clone(GtkWidget *, const char *)
{
   char     command[300];
   int      ignore;

   snprintf(command,299,"fotoxx -l %s",zfuncs::zlanguage);                 //  keep language   v.8.5 
   if (image_file) strncatv(command,299," \"",image_file,"\"",null);
   strcat(command," &");
   ignore = system(command);
   return;
}


/**************************************************************************/

//  enter or leave slideshow mode

void m_slideshow(GtkWidget *, const char *)
{
   static int     ww, hh;
   zdialog        *zd;
   int            zstat, secs;
   
   if (! Fslideshow)
   {
      gtk_window_get_size(GTK_WINDOW(mWin),&ww,&hh);
      gtk_widget_hide_all(GTK_WIDGET(mMbar));                              //  enter slide show mode
      gtk_widget_hide_all(GTK_WIDGET(mTbar));                              //  (full screen, no extras)
      gtk_widget_hide_all(GTK_WIDGET(STbar));
      gtk_window_fullscreen(GTK_WINDOW(mWin));

      zd = zdialog_new(ZTX("Time Interval"),mWin,Bapply,Bcancel,0);        //  dialog to get interval   v.8.4
      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
      zdialog_add_widget(zd,"label","lab1","hb1",ZTX("seconds"));
      zdialog_add_widget(zd,"entry","secs","hb1",0,"scc=5");
      zdialog_stuff(zd,"secs",SS_interval);
      zstat = zdialog_run(zd,0,0);
      zdialog_fetch(zd,"secs",secs);
      zdialog_free(zd);
      SS_interval = secs;                                                  //  interval between slides
      if (zstat != 1) secs = 9999;                                         //  cancel, use huge interval
      SS_timer = get_seconds() + secs + 1;                                 //  set timer for next slide
      Fslideshow = 1;
   }

   else
   {
      gtk_window_unfullscreen(GTK_WINDOW(mWin));                           //  leave slide show mode
      gtk_window_resize(GTK_WINDOW(mWin),ww,hh);
      gtk_widget_show_all(GTK_WIDGET(mMbar));
      gtk_widget_show_all(GTK_WIDGET(mTbar));
      gtk_widget_show_all(GTK_WIDGET(STbar));
      Fslideshow = 0;
   }

   Fzoom = 0;                                                              //  fit image to window
   Fblowup = Fslideshow;                                                   //  blow-up small images if SS mode
   mwpaint2(); 
   return;
}


/**************************************************************************/

//  show RGB values for pixel at mouse click

void m_showRGB(GtkWidget *, const char *)                                  //  menu function
{
   int   RGB_dialog_compl(zdialog *zd, int zstat);
   void  RGB_mousefunc();
   
   const char  *rgbmess = ZTX("click on window to show RGB");
   const char  *format = "Pixel: 0 0   RGB: 0.0  0.0  0.0";

   if (! Frgb24) return;                                                   //  no image

   mouseCBfunc = RGB_mousefunc;                                            //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks
   gdk_window_set_cursor(drWin->window,0);                                 //  set normal cursor

   if (zdRGB) return;                                                      //  already active

   zdRGB = zdialog_new(ZTX("Show RGB"),mWin,Bcancel,null);                 //  dialog to show RGB data
   zdialog_add_widget(zdRGB,"label","lab1","dialog",rgbmess,"space=5");
   zdialog_add_widget(zdRGB,"label","labrgb","dialog",format,"space=5");

   zdialog_run(zdRGB,0,RGB_dialog_compl);                                  //  run dialog, parallel
   return;
}


//  dialog completion function

int RGB_dialog_compl(zdialog *zd, int zstat)
{
   zdialog_free(zdRGB);                                                    //  kill dialog
   zdRGB = null;
   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   return 0;
}


//  mouse function

void RGB_mousefunc()                                                       //  mouse function
{
   int         px, py;
   double      red, green, blue;
   double      fbright, fred;
   char        text[60];
   uint8       *ppix24;
   uint16      *ppix48;
   
   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;
      px = Mxclick;                                                        //  click position
      py = Myclick;
      
      if (E3rgb48) {                                                       //  use current image being edited
         if (px < 0 || px > E3ww-1 ||                                      //  outside image area     v.6.4
             py < 0 || py > E3hh-1) return;
         ppix48 = bmpixel(E3rgb48,px,py);                                  //  bugfix: * Mscale removed  v.6.7
         red = ppix48[0] / 256.0;
         green = ppix48[1] / 256.0;
         blue = ppix48[2] / 256.0;
         fbright = brightness(ppix48) / 256.0;
         fred = redness(ppix48);
      }

      else if (Frgb48) {                                                   //  use edited image
         if (px < 0 || px > Fww-1 || 
             py < 0 || py > Fhh-1) return;
         ppix48 = bmpixel(Frgb48,px,py);
         red = ppix48[0] / 256.0;
         green = ppix48[1] / 256.0;
         blue = ppix48[2] / 256.0;
         fbright = brightness(ppix48) / 256.0;
         fred = redness(ppix48);
      }

      else  {                                                              //  use 24 bpp image
         if (px < 0 || px > Fww-1 || 
             py < 0 || py > Fhh-1) return;
         ppix24 = (uint8 *) Frgb24->bmp + (py * Fww + px) * 3;
         red = ppix24[0];
         green = ppix24[1];
         blue = ppix24[2];
         fbright = brightness(ppix24);
         fred = redness(ppix24);
      }
      
      snprintf(text,59,"Pixel: %d %d  RGB: %6.3f %6.3f %6.3f",             //  show pixel and RGB colors
                                  px, py, red, green, blue);
      zdialog_stuff(zdRGB,"labrgb",text);
   }
   
   return;
}


/**************************************************************************/

//  choose or set lens parameters for panoramas

void  m_parms(GtkWidget *, const char *)
{
   zdialog     *zd;
   int         ii, zstat, radb;
   char        text[20];
   
   zd = zdialog_new(ZTX("Lens Parameters"),mWin,Bapply,Bcancel,0);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=5|homog");            //        Lens    mm    bow
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=5|homog");            //   (o)  name1   30    0.33
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"space=5|homog");            //   (x)  name2   40    0.22
   zdialog_add_widget(zd,"vbox","vb4","hb1",0,"space=5|homog");            //   (o)  name3   45    0.28
   zdialog_add_widget(zd,"label","space","vb1");                           //   (o)  name4   50    0.44
   zdialog_add_widget(zd,"radio","radb0","vb1",0);                         //
   zdialog_add_widget(zd,"radio","radb1","vb1",0);                         //         [apply]  [cancel]
   zdialog_add_widget(zd,"radio","radb2","vb1",0);
   zdialog_add_widget(zd,"radio","radb3","vb1",0);
   zdialog_add_widget(zd,"label","lname","vb2",ZTX("lens name"));          //  fix translation   v.8.4.1
   zdialog_add_widget(zd,"entry","name0","vb2","scc=10");
   zdialog_add_widget(zd,"entry","name1","vb2","scc=10");
   zdialog_add_widget(zd,"entry","name2","vb2","scc=10");
   zdialog_add_widget(zd,"entry","name3","vb2","scc=10");
   zdialog_add_widget(zd,"label","lmm","vb3",ZTX("lens mm"));
   zdialog_add_widget(zd,"entry","mm0","vb3","0","scc=5");
   zdialog_add_widget(zd,"entry","mm1","vb3","0","scc=5");
   zdialog_add_widget(zd,"entry","mm2","vb3","0","scc=5");
   zdialog_add_widget(zd,"entry","mm3","vb3","0","scc=5");
   zdialog_add_widget(zd,"label","lbow","vb4",ZTX("lens bow"));
   zdialog_add_widget(zd,"entry","bow0","vb4","0.0","scc=6");
   zdialog_add_widget(zd,"entry","bow1","vb4","0.0","scc=6");
   zdialog_add_widget(zd,"entry","bow2","vb4","0.0","scc=6");
   zdialog_add_widget(zd,"entry","bow3","vb4","0.0","scc=6");
   
   for (ii = 0; ii < 4; ii++)                                              //  stuff lens data into dialog
   {
      snprintf(text,20,"name%d",ii);
      zdialog_stuff(zd,text,lens4_name[ii]);
      snprintf(text,20,"mm%d",ii);
      zdialog_stuff(zd,text,lens4_mm[ii]);
      snprintf(text,20,"bow%d",ii);
      zdialog_stuff(zd,text,lens4_bow[ii]);
   }

   snprintf(text,20,"radb%d",curr_lens);                                   //  current lens = selected
   zdialog_stuff(zd,text,1);

   zstat = zdialog_run(zd,0,0);                                            //  run dialog, get inputs

   if (zstat != 1) {
      zdialog_free(zd);                                                    //  canceled
      return;
   }
   
   for (ii = 0; ii < 4; ii++)                                              //  fetch lens data (revisions)
   {
      snprintf(text,20,"name%d",ii);
      zdialog_fetch(zd,text,lens4_name[ii],lens_cc);
      repl_1str(lens4_name[ii],lens4_name[ii]," ","_");                    //  replace blank with _
      snprintf(text,20,"mm%d",ii);
      zdialog_fetch(zd,text,lens4_mm[ii]);
      snprintf(text,20,"bow%d",ii);
      zdialog_fetch(zd,text,lens4_bow[ii]);
      snprintf(text,20,"radb%d",ii);                                       //  detect which is selected
      zdialog_fetch(zd,text,radb);
      if (radb) curr_lens = ii;
   }
   
   zdialog_free(zd);
   return;
}


/**************************************************************************/

//  set GUI language

void  m_lang(GtkWidget *, const char *)                                    //  v.6.2
{
   zdialog     *zd;
   int         ii, cc, err, zstat;

   char        lang[20], *pp, command[100], locmess[200];

   const char  *langs[10] = { "cz Czech", "de German", "el Greek", 
                              "en English", "es Spanish", "fr French", 
                              "gl Galacian", "it Italian", "zh_CN Chinese", 0 };
   
   strcpy(locmess,ZTX("Available Translations"));
   cc = strlen(locmess);

   for (ii = 0; langs[ii]; ii++)
   {
      strcpy(locmess+cc,"\n ");
      strcpy(locmess+cc+2,langs[ii]);
      cc += strlen(langs[ii]) + 2;
   }
   
   zd = zdialog_new(ZTX("Set Language"),mWin,Bapply,Bcancel,0);
   zdialog_add_widget(zd,"label","lab0","dialog",locmess,"space=5");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"label","lab1","hb1","language code");
   zdialog_add_widget(zd,"combo","combo","hb1",0);

   for (ii = 0; langs[ii]; ii++)                                           //  stuff languages into list
      zdialog_cb_app(zd,"combo",langs[ii]);

   zstat = zdialog_run(zd,0,0);                                            //  run dialog

   if (zstat != 1) {
      zdialog_free(zd);                                                    //  canceled
      return;
   }
   
   zdialog_fetch(zd,"combo",lang,20);                                      //  get user input
   zdialog_free(zd);
   
   pp = strchr(lang,' ');                                                  //  isolate lc_RC part
   *pp = 0;

   sprintf(command,"fotoxx -l %s &",lang);                                 //  restart fotoxx
   err = system(command);
   m_quit(0,0);

   return;
}


/**************************************************************************/

//  create desktop icon / launcher

void  m_launcher(GtkWidget *, const char *)                                //  v.7.0
{
   zmake_launcher("Graphics","Image Editor");
   return;
}


/**************************************************************************/

//  convert multiple RAW files to tiff

void  m_multiraw(GtkWidget *, const char *)                                //  v.7.1
{
   zdialog        *zd;
   char           entraw[200], *pp, *rawspec = 0;
   char           *outfile, command[1000];
   const char     *rawfile;
   int            zstat, ftf, err;
   struct stat    fstat;

   if (mod_keep()) return;
   if (! menulock(1)) return;

   if (! Fufraw) {
      zmessageACK(ZTX("Package ufraw required for this function"));
      goto rawdone;
   }

   zd = zdialog_new(ZTX("Convert multiple RAW files"),mWin,BOK,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"label","labraw","hb1",ZTX("RAW file template"),"space=5");
   zdialog_add_widget(zd,"entry","entraw","hb1","*.RAW","space=5");

   zstat = zdialog_run(zd);                                                //  run dialog (blocking)
   zdialog_fetch(zd,"entraw",entraw,199);                                  //  get RAW file spec
   zdialog_free(zd);
   if (zstat != 1) goto rawdone;
   
   if (*entraw != '/')                                                     //  needs a directory
   {
      if (image_file) {
         rawspec = strdupz(image_file,200);                                //  use same as image file
         pp = (char *) strrchr(rawspec,'/');
         if (pp) pp++;
         else  pp = rawspec;
         strncpy0(pp,entraw,199);
      }
      else {
         pp = getcwd(null,0);                                              //  use curr. directory
         rawspec = strdupz(pp,200);
         free(pp);
         strcat(rawspec,"/");
         pp = (char *) strrchr(rawspec,'/');
         strncpy0(pp+1,entraw,199);
      }
   }
   
   else rawspec = strdupz(entraw);                                         //  absolute path was given

   write_popup_text("open","converting RAW files",500,200);

   gdk_window_set_cursor(drWin->window,busycursor);                        //  set function busy cursor  v.8.4
   zmainloop();
   
   for (ftf = 1; ; )
   {
      rawfile = SearchWild(rawspec,ftf);                                   //  find all *.RAW files
      if (! rawfile) break;
      
      err = stat(rawfile,&fstat);
      if (err) continue;

      outfile = strdupz(rawfile,5);
      pp = (char *) strrchr(rawfile,'.');
      pp = outfile + (pp - rawfile);
      strcpy(pp,".tiff");

      write_popup_text("write",rawfile,0,0);                               //  convert next file
      
      //  try new ufraw command format first, then old if it fails

      snprintf(command,999,"ufraw-batch --out-type=tiff --out-depth=16"
                  " --overwrite --output=\"%s\" \"%s\" ",outfile, rawfile);
      err = system(command);

      if (err) {
         snprintf(command,999,"ufraw-batch --out-type=tiff16"
                  " --overwrite --output=\"%s\" \"%s\" ",outfile, rawfile);
         err = system(command);
      }

      if (err) {
         write_popup_text("write",wstrerror(err),0,0);
         zfree(outfile);
         continue;
      }
      
      menulock(0);
      f_open(outfile);                                                     //  open converted file
      menulock(1);
      mwpaint2();
      zmainloop();

      image_gallery(outfile,"init");                                       //  update image gallery file list
      image_gallery(0,"paint2");                                           //  refresh gallery window if active
      Fsearchlist = 0;
      zfree(outfile);
   }
   
   write_popup_text("close",0,0,0);

   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor

rawdone:
   if (rawspec) zfree(rawspec);
   menulock(0);
   return;
}


/**************************************************************************/

//  burn images to CD/DVD                                                  //  v.7.2

int  burn_showthumb();

GtkWidget      *burn_drawarea = 0;
GtkWidget      *burn_files = 0;
const char     *burn_font = "Monospace 9";
int            burn_fontheight = 14;
int            burn_cursorpos = 0;

void  m_burn(GtkWidget *, const char *)
{
   int  burn_dialog_event(zdialog *zd, const char *event);
   int  burn_dialog_compl(zdialog *zd, int zstat);
   int  burn_mouseclick(GtkWidget *, GdkEventButton *event, void *);

   GdkCursor   *cursor;
   GdkWindow   *gdkwin;
   
   if (! menulock(1)) return;                                              //  lock menus

   m_gallery(0,0);                                                         //  activate image gallery window

   zdburn = zdialog_new(ZTX("Burn Images to CD/DVD"),0,ZTX("Burn"),Bcancel,null);
   zdialog_add_widget(zdburn,"hbox","hb1","dialog",0,"expand|space=5");
   zdialog_add_widget(zdburn,"frame","fr11","hb1",0,"expand");
   zdialog_add_widget(zdburn,"scrwin","scrwin","fr11",0,"expand");
   zdialog_add_widget(zdburn,"edit","files","scrwin");
   zdialog_add_widget(zdburn,"vbox","vb12","hb1");
   zdialog_add_widget(zdburn,"frame","fr12","vb12");
   zdialog_add_widget(zdburn,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdburn,"button","delete","hb2",Bdelete,"space=8");
   zdialog_add_widget(zdburn,"button","insert","hb2",Binsert,"space=8");
   zdialog_add_widget(zdburn,"button","addall","hb2",Baddall,"space=30");

   GtkWidget *frame = zdialog_widget(zdburn,"fr12");                       //  drawing area for thumbnail image
   burn_drawarea = gtk_drawing_area_new();
   gtk_widget_set_size_request(burn_drawarea,128,128);
   gtk_container_add(GTK_CONTAINER(frame),burn_drawarea);
   
   burn_files = zdialog_widget(zdburn,"files");                            //  activate mouse-clicks for
   gtk_widget_add_events(burn_files,GDK_BUTTON_PRESS_MASK);                //    file list widget
   G_SIGNAL(burn_files,"button-press-event",burn_mouseclick,0)

   PangoFontDescription *pfontdesc = pango_font_description_from_string(burn_font);
   gtk_widget_modify_font(burn_files,pfontdesc);
   
   zdialog_resize(zdburn,400,0);
   zdialog_run(zdburn,burn_dialog_event,burn_dialog_compl);

   cursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);                            //  arrow cursor for file list widget
   gdkwin = gtk_text_view_get_window(GTK_TEXT_VIEW(burn_files),textwin);
   gdk_window_set_cursor(gdkwin,cursor);
   
   burn_cursorpos = 0;
   return;
}


//  burn dialog event function

int burn_dialog_event(zdialog *zd, const char *event)
{
   GtkTextBuffer  *textBuff;
   GtkTextIter    iter1, iter2;
   static char    *imagefile = 0;
   const char     *xfile;
   int            line;

   if (strEqu(event,"delete"))                                             //  delete file at cursor position
   {
      if (imagefile) free(imagefile);
      imagefile = 0;
      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(burn_files));
      line = burn_cursorpos;
      gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);              //  iter at line start
      iter2 = iter1;
      gtk_text_iter_forward_to_line_end(&iter2);                           //  iter at line end

      imagefile = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);      //  save selected file
      if (*imagefile != '/') {
         free(imagefile);
         imagefile = 0;
         return 0;
      }

      gtk_text_buffer_delete(textBuff,&iter1,&iter2);                      //  delete file text
      gtk_text_buffer_get_iter_at_line(textBuff,&iter2,line+1);
      gtk_text_buffer_delete(textBuff,&iter1,&iter2);                      //  delete empty line (\n)

      burn_showthumb();                                                    //  thumbnail = next file
   }

   if (strEqu(event,"insert"))                                             //  insert last deleted file
   {
      if (! imagefile) return 0;                                           //    at current cursor position
      burn_insert_file(imagefile);
   }

   if (strEqu(event,"addall"))                                             //  insert all files in image gallery
   {
      if (imagefile) free(imagefile);
      imagefile = 0;

      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(burn_files));
      xfile = "first";

      while (true)
      {
         imagefile = image_gallery(imagefile,xfile,1,0);                   //  next file
         if (! imagefile) break;
         xfile = "next";
         gtk_text_buffer_get_iter_at_line(textBuff,&iter1,burn_cursorpos);
         gtk_text_buffer_insert(textBuff,&iter1,"\n",1);                   //  insert new blank line
         gtk_text_buffer_get_iter_at_line(textBuff,&iter1,burn_cursorpos);
         gtk_text_buffer_insert(textBuff,&iter1,imagefile,-1);             //  insert image file
         burn_cursorpos++;                                                 //  advance cursor position
      }
   }

   return 0;
}


//  burn dialog completion function - send all selected files to brasero

int burn_dialog_compl(zdialog *zd, int zstat)
{
   int            line, nlines, cc1, cc2, err;
   char           *imagefile = 0;
   char           *command;
   GtkTextBuffer  *textBuff;
   GtkTextIter    iter1, iter2;

   if (zstat != 1) {                                                       //  cancelled
      zdialog_free(zdburn);                                                //  kill dialog
      zdburn = null;
      menulock(0);
      return 0;
   }

   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(burn_files));
   nlines = gtk_text_buffer_get_line_count(textBuff);
   cc1 = gtk_text_buffer_get_char_count(textBuff);
   cc1 = cc1 + 5 * nlines + 20;
   command = zmalloc(cc1);
   strcpy(command,"brasero");
   cc2 = strlen(command);

   for (line = 0; line < nlines; line++)
   {
      gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);              //  iter at line start
      iter2 = iter1;
      gtk_text_iter_forward_to_line_end(&iter2);
      imagefile = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);      //  get imagefile at line
      if (imagefile && *imagefile == '/') {
         strcpy(command+cc2," \"");
         cc2 += 2;
         strcpy(command+cc2,imagefile);
         cc2 += strlen(imagefile);
         strcpy(command+cc2,"\"");
         cc2 += 1;
         free(imagefile);
      }
   }
   
   zdialog_free(zdburn);                                                   //  kill dialog
   zdburn = null;
   
   strcat(command," &");                                                   //  do command in background  v.8.4
   err = system(command);                                                  //  start brasero
   zfree(command);
   menulock(0);
   return 0;
}


//  called from image gallery window when a thumbnail is clicked
//  add image file to list at current cursor position, set thumbnail = file

void burn_insert_file(const char *imagefile)
{
   GtkTextIter    iter;
   GtkTextBuffer  *textBuff;

   if (*imagefile == '/') {
      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(burn_files));
      gtk_text_buffer_get_iter_at_line(textBuff,&iter,burn_cursorpos);
      gtk_text_buffer_insert(textBuff,&iter,"\n",1);                       //  insert new blank line
      gtk_text_buffer_get_iter_at_line(textBuff,&iter,burn_cursorpos);
      gtk_text_buffer_insert(textBuff,&iter,imagefile,-1);                 //  insert image file
      burn_showthumb();                                                    //  update thumbnail
      burn_cursorpos++;                                                    //  advance cursor position
   }

   return;
}


//  process mouse click in files window: 
//  set new cursor position and set thumbnail = clicked file

int burn_mouseclick(GtkWidget *, GdkEventButton *event, void *)
{
   int            mpx, mpy;
   GtkWidget      *scrollwin;
   GtkAdjustment  *scrolladj;
   double         scrollpos;

   if (event->type != GDK_BUTTON_PRESS) return 0;
   mpx = int(event->x);                                                    //  mouse position
   mpy = int(event->y);
   scrollwin = zdialog_widget(zdburn,"scrwin");                            //  window scroll position
   scrolladj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrollwin));
   scrollpos = gtk_adjustment_get_value(scrolladj);
   burn_cursorpos = (mpy + scrollpos) / burn_fontheight;                   //  line selected
   burn_showthumb();                                                       //  show thumbnail image
   return 0;
}


//  show thumbnail for file at current cursor position

int burn_showthumb()
{
   int            line;
   char           *imagefile;
   GtkTextBuffer  *textBuff;
   GtkTextIter    iter1, iter2;
   GdkPixbuf      *thumbnail = 0;

   gdk_window_clear(burn_drawarea->window);

   line = burn_cursorpos;
   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(burn_files));
   gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);                 //  iter at line start
   iter2 = iter1;
   gtk_text_iter_forward_to_line_end(&iter2);                              //  iter at line end

   imagefile = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);         //  get selected file
   if (*imagefile != '/') {
      free(imagefile);
      return 0;
   }

   thumbnail = image_thumbnail(imagefile,128);                             //  get thumbnail
   free(imagefile);

   if (thumbnail) {
      gdk_draw_pixbuf(burn_drawarea->window,0,thumbnail,0,0,0,0,-1,-1,nodither,0,0);
      g_object_unref(thumbnail);
   }
   return 0;
}


/**************************************************************************
      Image Tag and EXIF functions
***************************************************************************/

void  edit_tags_fixup(cchar * widgetname);                                 //  fixup tag selection widgets
void  edit_tags_mouse(GtkTextView *, GdkEventButton *, cchar *);           //  select tag via mouse click
int   add_unique_tag(cchar *tag, char *taglist, int maxcc);                //  add tag if unique and enough space
void  add_new_filetag();                                                   //  add tags_atag to tags_filetags
void  delete_filetag();                                                    //  remove tags_atag from tags_filetags
void  add_new_recentag();                                                  //  add tags_atag to tags_recentags

char     tags_imdate[12] = "";                                             //  image date, yyyymmdd
char     tags_limdate[12] = "";                                            //  last image date read or set
int      tags_stars = 0;                                                   //  image rating in "stars"
char     tags_atag[maxtag1] = "";                                          //  one tag
char     tags_filetags[maxtag2] = "";                                      //  tags for one file
char     tags_asstags[maxtag3] = "";                                       //  all assigned tags
char     tags_searchfile[maxtagF] = "";                                    //  image search /path*/file*
char     tags_searchtags[maxtag4] = "";                                    //  image search tags
char     tags_recentags[maxtag5] = "";                                     //  recently added tags
int      tags_changed = 0;                                                 //  tags have been changed


/**************************************************************************/

//  edit tags menu function

void m_edit_tags(GtkWidget *, const char *)
{
   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(Bexiftoolmissing);
      return;
   }
   
   edit_tags_dialog();
   return;
}


//  activate edit tags dialog, stuff data from current file

void edit_tags_dialog()
{
   int edit_tags_dialog_event(zdialog *zd, const char *event);
   int edit_tags_dialog_compl(zdialog *zd, int zstat);

   char     *ppv, pstarsN[12];

   if (! image_file) return;

   if (! zdtags)                                                           //  (re) start tag edit dialog 
   {
      load_asstags();                                                      //  get all assigned tags

      zdtags = zdialog_new(ZTX("Edit Tags"),mWin,Bdone,Bcancel,0);         //  tag edit dialog

      zdialog_add_widget(zdtags,"hbox","hb1","dialog",0,"space=5");
      zdialog_add_widget(zdtags,"label","labfile","hb1",ZTX("file:"),"space=10");
      zdialog_add_widget(zdtags,"label","file","hb1");

      zdialog_add_widget(zdtags,"hbox","hb2","dialog",0,"space=5");
      zdialog_add_widget(zdtags,"label","lab21","hb2",ZTX("image date (yyyymmdd)"),"space=10");
      zdialog_add_widget(zdtags,"entry","imdate","hb2",0,"scc=12");
      zdialog_add_widget(zdtags,"button","limdate","hb2",ZTX("use last"),"space=10");

      zdialog_add_widget(zdtags,"hbox","hb3","dialog",0,"space=5");
      zdialog_add_widget(zdtags,"label","labstars","hb3",ZTX("image stars"),"space=10");
      zdialog_add_widget(zdtags,"vbox","vb3","hb3");
      zdialog_add_widget(zdtags,"hbox","hb31","vb3",0,"homog");
      zdialog_add_widget(zdtags,"hbox","hb32","vb3",0,"homog");
      zdialog_add_widget(zdtags,"label","lab30","hb31","0");
      zdialog_add_widget(zdtags,"label","lab31","hb31","1");
      zdialog_add_widget(zdtags,"label","lab32","hb31","2");
      zdialog_add_widget(zdtags,"label","lab33","hb31","3");
      zdialog_add_widget(zdtags,"label","lab34","hb31","4");
      zdialog_add_widget(zdtags,"label","lab35","hb31","5");
      zdialog_add_widget(zdtags,"radio","pstars0","hb32",0);
      zdialog_add_widget(zdtags,"radio","pstars1","hb32",0);
      zdialog_add_widget(zdtags,"radio","pstars2","hb32",0);
      zdialog_add_widget(zdtags,"radio","pstars3","hb32",0);
      zdialog_add_widget(zdtags,"radio","pstars4","hb32",0);
      zdialog_add_widget(zdtags,"radio","pstars5","hb32",0);

      zdialog_add_widget(zdtags,"hbox","hb4","dialog","space=5");
      zdialog_add_widget(zdtags,"label","lab4","hb4",ZTX("current tags"),"space=10");
      zdialog_add_widget(zdtags,"frame","frame4","hb4",0,"expand");
      zdialog_add_widget(zdtags,"edit","filetags","frame4",0,"expand");

      zdialog_add_widget(zdtags,"hbox","hb5","dialog","space=5");
      zdialog_add_widget(zdtags,"label","recent","hb5",ZTX("recently added"),"space=10");
      zdialog_add_widget(zdtags,"frame","frame5","hb5",0,"expand");
      zdialog_add_widget(zdtags,"edit","recentags","frame5",0,"expand");

      zdialog_add_widget(zdtags,"hbox","hb6","dialog",0,"space=5");
      zdialog_add_widget(zdtags,"button","addtag","hb6",ZTX("create tag"),"space=10");
      zdialog_add_widget(zdtags,"entry","atag","hb6",0);

      zdialog_add_widget(zdtags,"hbox","hb7","dialog",0,"space=5");
      zdialog_add_widget(zdtags,"hbox","hb8","dialog");
      zdialog_add_widget(zdtags,"label","labasstags","hb8",ZTX("assigned tags"),"space=10");
      zdialog_add_widget(zdtags,"frame","frame8","dialog",0,"expand");
      zdialog_add_widget(zdtags,"edit","asstags","frame8",0,"expand");

      zdialog_resize(zdtags,400,300);
      zdialog_run(zdtags,edit_tags_dialog_event,edit_tags_dialog_compl);   //  start dialog
      
      edit_tags_fixup("filetags");                                         //  setup for mouse tag selection
      edit_tags_fixup("asstags");
      edit_tags_fixup("recentags");
   }

   load_filetags(image_file);                                              //  get file tags from EXIF data

   ppv = (char *) strrchr(image_file,'/');
   zdialog_stuff(zdtags,"file",ppv+1);                                     //  stuff dialog file name

   zdialog_stuff(zdtags,"imdate",tags_imdate);                             //  stuff dialog data
   sprintf(pstarsN,"pstars%d",tags_stars);
   zdialog_stuff(zdtags,pstarsN,1);
   zdialog_stuff(zdtags,"filetags",tags_filetags);
   zdialog_stuff(zdtags,"asstags",tags_asstags);
   zdialog_stuff(zdtags,"recentags",tags_recentags);

   tags_changed = 0;
   return;
}


//  setup tag display widget for tag selection using mouse clicks

void edit_tags_fixup(const char * widgetname)
{
   GtkWidget         *widget;
   GdkWindow         *gdkwin;

   widget = zdialog_widget(zdtags,widgetname);                             //  make widget wrap text
   gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(widget),GTK_WRAP_WORD);
   gtk_text_view_set_editable(GTK_TEXT_VIEW(widget),0);                    //  disable widget editing

   gdkwin = gtk_text_view_get_window(GTK_TEXT_VIEW(widget),textwin);       //  cursor for tag selection
   gdk_window_set_cursor(gdkwin,arrowcursor);

   gtk_widget_add_events(widget,GDK_BUTTON_PRESS_MASK);                    //  connect mouse-click event
   G_SIGNAL(widget,"button-press-event",edit_tags_mouse,widgetname)
}


//  edit tags mouse-click event function
//  get clicked tag and add to or remove from tags_filetags

void edit_tags_mouse(GtkTextView *widget, GdkEventButton *event, const char *widgetname)
{
   GtkTextIter    iter;
   int            mpx, mpy, tbx, tby, offset, cc;
   char           *ptext, *pp1, *pp2;

   if (event->type != GDK_BUTTON_PRESS) return;
   mpx = int(event->x);                                                    //  mouse click position
   mpy = int(event->y);

   gtk_text_view_window_to_buffer_coords(widget,GTK_TEXT_WINDOW_TEXT,mpx,mpy,&tbx,&tby);
   gtk_text_view_get_iter_at_location(widget,&iter,tbx,tby);
   offset = gtk_text_iter_get_offset(&iter);                               //  graphic position in widget text

   ptext = 0;   
   if (strEqu(widgetname,"filetags")) ptext = tags_filetags;               //  get corresponding text
   if (strEqu(widgetname,"asstags")) ptext = tags_asstags;
   if (strEqu(widgetname,"recentags")) ptext = tags_recentags;
   if (strEqu(widgetname,"asstags2")) ptext = tags_asstags;
   if (! ptext) return;

   pp1 = ptext + utf8_position(ptext,offset);                              //  graphic position to byte position
   if (! *pp1 || *pp1 == ' ') return;                                      //  reject ambiguity
   while (pp1 > ptext && *pp1 != ' ') pp1--;                               //  find preceeding delimiter
   if (*pp1 == ' ') pp1++;

   pp2 = strchr(pp1,' ');                                                  //  find delimiter following
   if (pp2) cc = pp2 - pp1;
   else cc = strlen(pp1);
   if (cc >= maxtag1) return;                                              //  reject tag too big
   strncpy0(tags_atag,pp1,cc+1);                                           //  tags_atag = selected tag
   
   if (strEqu(widgetname,"filetags")) {
      delete_filetag();                                                    //  remove tag from file tags
      zdialog_stuff(zdtags,"filetags",tags_filetags);                      //  update dialog widgets
   }
   
   if (strEqu(widgetname,"asstags")) {
      add_new_filetag();                                                   //  add assigned tag to file tags
      add_new_recentag();
      zdialog_stuff(zdtags,"filetags",tags_filetags);
   }

   if (strEqu(widgetname,"recentags")) {
      add_new_filetag();                                                   //  add recent tag to file tags
      zdialog_stuff(zdtags,"filetags",tags_filetags);
   }

   if (strEqu(widgetname,"asstags2")) {                                    //  search dialog:
      zdialog_fetch(zdtags,"searchtags",tags_searchtags,maxtag4);          //  add assigned tag to search tags
      strncatv(tags_searchtags,maxtag4," ",tags_atag,0);
      zdialog_stuff(zdtags,"searchtags",tags_searchtags);
   }

   return;
}


//  edit tags dialog event and completion functions

int edit_tags_dialog_event(zdialog *zd, const char *event)
{
   int      err;
   
   if (strEqu(event,"imdate")) {                                           //  image date revised
      err = zdialog_fetch(zd,"imdate",tags_imdate,11);
      if (err) return 1;
      tags_changed++;
   }
   
   if (strEqu(event,"limdate")) {                                          //  repeat last date used  v.8.1
      if (*tags_limdate) {
         zdialog_stuff(zd,"imdate",tags_limdate);
         strcpy(tags_imdate,tags_limdate);
         tags_changed++;
      }
   }

   if (strnEqu(event,"pstars",6)) {                                        //  stars revised
      tags_stars = event[6] - '0';
      tags_changed++;
   }

   if (strEqu(event,"addtag")) {
      err = zdialog_fetch(zd,"atag",tags_atag,maxtag1);                    //  add new tag to file
      if (err) return 1;                                                   //  reject too big tag
      add_new_filetag();
      add_new_recentag();
      zdialog_stuff(zd,"filetags",tags_filetags);                          //  update dialog widgets
      zdialog_stuff(zd,"asstags",tags_asstags);
      zdialog_stuff(zd,"atag","");
   }

   return 0;
}


int edit_tags_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat == 1) update_filetags(image_file);                            //  done, update file EXIF tags
   zdialog_free(zdtags);                                                   //  kill dialog
   zdtags = null;
   return 0;
}


//  add input tag to output tag list if not already there and enough room
//  returns:   0 = added OK     1 = not unique (case ignored)
//             2 = overflow     3 = bad utf8 characters

int add_unique_tag(const char *tag, char *taglist, int maxcc)
{
   char     *pp1, *pp2, *ppv, temptag1[maxtag1], temptag2[maxtag1];
   int      atcc, cc1, cc2;

   strncpy0(temptag1,tag,maxtag1);                                         //  remove leading and trailing blanks
   atcc = strTrim2(temptag2,temptag1);
   if (! atcc) return 0;

   for (ppv = temptag2; *ppv; ppv++)                                       //  replace imbedded blanks with '_'
      if (*ppv == ' ') *ppv = '_';
   
   if (utf8_check(temptag2)) {                                             //  check for valid utf8 encoding
      printf("bad utf8 characters: %s \n",temptag2);
      return 3;
   }
   
   pp1 = taglist;
   cc1 = strlen(temptag2);

   while (true)                                                            //  check if already in tag list
   {
      while (*pp1 == ' ') pp1++;
      if (! *pp1) break;
      pp2 = pp1 + 1;
      while (*pp2 && *pp2 != ' ') pp2++;
      cc2 = pp2 - pp1;
      if (cc2 == cc1 && strncaseEqu(temptag2,pp1,cc1)) return 1;
      pp1 = pp2;
   }
   
   cc2 = strlen(taglist);                                                  //  append to tag list if space enough
   if (cc1 + cc2 + 1 >= maxcc) return 2;
   strcpy(taglist + cc2,temptag2);
   strcpy(taglist + cc2 + cc1," ");
   return 0;
}


/**************************************************************************/

//  image file EXIF data >> tags_imdate, tags_stars, tags_filetags in memory

void load_filetags(const char *file)
{
   int         ii, jj, cc, err;
   const char  *pp;
   const char  *exifkeys[2] = { exif_date_key, exif_tags_key };
   char        **ppv, *imagedate, *imagetags;

   *tags_filetags = *tags_imdate = 0;
   tags_stars = 0;
   
   ppv = exif_get(image_file,exifkeys,2);
   imagedate = ppv[0];
   imagetags = ppv[1];

   if (imagedate) {                       
      if (strlen(imagedate) > 9) {
         strncpy(tags_imdate,imagedate,4);                                 //  reformat yyyy:mm:dd
         strncpy(tags_imdate+4,imagedate+5,2);                             //    to  yyyymmdd
         strncpy(tags_imdate+6,imagedate+8,2);
         tags_imdate[8] = 0;
         strcpy(tags_limdate,tags_imdate);
      }
      zfree(imagedate);
   }

   if (imagetags)
   {
      for (ii = 1; ; ii++)
      {
         pp = strField(imagetags,' ',ii);                                  //  assume blank delimited tags
         if (! pp) break;
         cc = strlen(pp);
         if (cc >= maxtag1) continue;                                      //  reject tags too big
         for (jj = 0; jj < cc; jj++)
            if (pp[jj] > 0 && pp[jj] < ' ') break;                         //  reject tags with control characters
         if (jj < cc) continue;
         
         if (strnEqu(pp,"stars=",6)) {                                     //  "stars=N" tag
            err = convSI(pp+6,tags_stars,0,5);
            if (err > 1) tags_stars = 0;
            continue;
         }

         strcpy(tags_atag,pp);                                             //  add to file tags if unique
         add_new_filetag();
      }

      zfree(imagetags);
   }
   
   return;
}


/**************************************************************************/

//  tags_imdate, tags_stars, tags_filetags in memory >> image file EXIF data

void update_filetags(const char *file)
{
   const char  *exifkeys[2] = { exif_date_key, exif_tags_key };
   const char  *exifdata[2];
   char        imagedate[20];

   if (! tags_changed) return;

   *imagedate = 0;                                                         //  v.7.5

   if (*tags_imdate) {
      if (strlen(tags_imdate) == 4) strcat(tags_imdate,"0101");            //  allow short dates   v.8.1
      if (strlen(tags_imdate) == 6) strcat(tags_imdate,"01");
      strncpy(imagedate,tags_imdate,4);                                    //  yyyymmdd >> yyyy:mm:dd
      strncpy(imagedate+5,tags_imdate+4,2);
      strncpy(imagedate+8,tags_imdate+6,2);
      imagedate[4] = imagedate[7] = ':';
      imagedate[10] = 0;
      strcpy(imagedate+10," 00:00:00");                                    //  hh:mm:ss, new EXIF req.   v.8.7
      strcpy(tags_limdate,tags_imdate);
   }

   if (tags_stars > 0) {                                                   //  add "stars=N" tag
      sprintf(tags_atag,"stars=%d",tags_stars);
      add_new_filetag();
   }
   
   exifdata[0] = imagedate;                                                //  update file EXIF data
   exifdata[1] = tags_filetags;
   exif_set(file,exifkeys,exifdata,2);
   
   update_asstags(file);                                                   //  update assigned tags file
   tags_changed = 0;
   return;
}


//  add new tag to file tags, if not already and enough space.

void add_new_filetag()
{
   int         err;

   err = add_unique_tag(tags_atag,tags_filetags,maxtag2);
   if (err == 2) { 
      zmessageACK(ZTX("File tags exceed %d characters"),maxtag2);
      return;
   }
   
   tags_changed++;
   return;
}


//  add new tag to recent tags, if not already.
//  remove oldest to make space if needed.

void add_new_recentag()
{
   int         err;
   char        *ppv, temp_recentags[maxtag5];

   if (strnEqu(tags_atag,"stars=",6)) return;                              //  omit this tag

   err = add_unique_tag(tags_atag,tags_recentags,maxtag5);                 //  add tag to recent tags

   while (err == 2)
   {
      strncpy0(temp_recentags,tags_recentags,maxtag5);                     //  remove oldest to make room
      ppv = temp_recentags;
      while (*ppv && *ppv == ' ') ppv++;
      while (*ppv && *ppv != ' ') ppv++;
      while (*ppv && *ppv == ' ') ppv++;
      strcpy(tags_recentags,ppv);
      err = add_unique_tag(tags_atag,tags_recentags,maxtag5);
   }

   zdialog_stuff(zdtags,"recentags",tags_recentags);                       //  update dialog
   return;
}


//  delete a tag from file tags, if present

void delete_filetag()
{
   int         ii, ftcc, atcc;
   char        temp_filetags[maxtag2];
   const char  *pp;
   
   strncpy0(temp_filetags,tags_filetags,maxtag2);
   *tags_filetags = 0;
   ftcc = 0;
   
   for (ii = 1; ; ii++)
   {
      pp = strField(temp_filetags,' ',ii);
      if (! pp) break;
      if (strcaseEqu(pp,tags_atag)) continue;
      atcc = strlen(pp);
      strcpy(tags_filetags + ftcc, pp);
      ftcc += atcc;
      tags_filetags[ftcc] = ' ';
      ftcc++;
      tags_filetags[ftcc] = 0;
   }

   tags_changed++;
   return;
}


/**************************************************************************/

//  load assigned tags file >> tags_asstags in memory
//  create list of all assigned tags with no duplicates

void load_asstags()
{
   FILE        *fid;
   int         ntags = 0, ntcc, atcc, ii, err;
   char        *ppv, buff[1000];
   const char  *pp1;
   char        *tags[maxntags];
   
   ntcc = 0;
   *tags_asstags = 0;

   fid = fopen(asstagsfile,"r");
   if (! fid) return;                                                      //  no tags
   
   while (true)                                                            //  read assigned tags file
   {
      ppv = fgets_trim(buff,999,fid);
      if (! ppv) break;
      if (strnNeq(buff,"tags: ",6)) continue;

      for (ii = 1; ; ii++)                                                 //  add file tags to assigned tags
      {                                                                    //    unless already present
         pp1 = strField(buff+6,' ',ii);
         if (! pp1) break;
         if (strnEqu(pp1,"stars=",6)) continue;                            //  omit this tag
         err = add_unique_tag(pp1,tags_asstags,maxtag3);
         if (err == 2) goto overflow;
      }
   }

   err = fclose(fid);
   if (err) goto tagsfileerr;
   
   for (ii = 1; ; ii++)                                                    //  build sort list
   {
      pp1 = strField(tags_asstags,' ',ii);
      if (! pp1) break;
      tags[ntags] = strdupz(pp1);
      ntags++;
      if (ntags == maxntags) goto toomanytags;
   }
   
   HeapSort(tags,ntags);                                                   //  sort alphabetically
   
   ntcc = 0;
   *tags_asstags = 0;

   for (ii = 0; ii < ntags; ii++)                                          //  build sorted assigned tags list
   {
      atcc = strlen(tags[ii]);
      if (ntcc + atcc + 1 > maxtag3) goto overflow;
      strcpy(tags_asstags + ntcc,tags[ii]);
      ntcc += atcc;
      tags_asstags[ntcc] = ' ';
      ntcc++;
      zfree(tags[ii]);
   }

   tags_asstags[ntcc] = 0;
   
   return;

overflow:
   zmessageACK(ZTX("Total tags exceed %d characters"),maxtag3);
   return;

toomanytags:
   zmessageACK(ZTX("Too many tags: %d"),maxntags);
   return;

tagsfileerr:
   zmessLogACK(ZTX("Assigned tags file error: %s"),strerror(errno));
   return;
}


/**************************************************************************/

//  update tags_asstags in memory from tags_filetags
//  update assigned tags file (add or replace changed file and its tags)

void update_asstags(const char *file, int del)
{
   char           *ppv, temp_asstagsfile[1000], imagedate[12], filedate[16];
   char           datebuff[1000], tagsbuff[1000], filebuff[1000];
   const char     *pp1;
   int            ii, ntcc, err;
   FILE           *fidr, *fidw;
   struct stat    statb;
   struct tm      bdt;

   ntcc = strlen(tags_asstags);
   
   if (! del)                                                              //  unless deleted
   {
      for (ii = 1; ; ii++)                                                 //  add file tags to assigned tags
      {                                                                    //    unless already present
         pp1 = strField(tags_filetags,' ',ii);
         if (! pp1) break;
         if (strnEqu(pp1,"stars=",6)) continue;                            //  omit this tag
         
         err = add_unique_tag(pp1,tags_asstags,maxtag3);
         if (err == 2) {
            zmessageACK(ZTX("Total tags exceed %d characters"),maxtag3);
            break;
         }
      }
   }

   strcpy(temp_asstagsfile,asstagsfile);                                   //  temp tag file
   strcat(temp_asstagsfile,"_temp");

   fidr = fopen(asstagsfile,"r");                                          //  read tag file
   
   fidw = fopen(temp_asstagsfile,"w");                                     //  write temp tag file
   if (! fidw) goto tagserror;

   if (fidr) {   
      while (true)                                                         //  copy assigned tags file to temp
      {                                                                    //    file, omitting this image file
         ppv = fgets_trim(datebuff,999,fidr);
         if (! ppv) break;
         if (strnNeq(datebuff,"date: ",6)) continue;

         ppv = fgets_trim(tagsbuff,999,fidr);
         if (! ppv) break;
         if (strnNeq(tagsbuff,"tags: ",6)) continue;

         ppv = fgets_trim(filebuff,999,fidr);
         if (! ppv) break;
         if (strnNeq(filebuff,"file: ",6)) continue;

         if (strEqu(filebuff+6,file)) continue;                            //  if my file, skip copy

         fprintf(fidw,"%s\n",datebuff);                                    //  copy to temp file
         fprintf(fidw,"%s\n",tagsbuff);
         fprintf(fidw,"%s\n\n",filebuff);
      }
   }
   
   if (! del)                                                              //  unless deleted, append 
   {                                                                       //    revised file data to temp file
      if (*tags_imdate) {
         strncpy(imagedate,tags_imdate,4);
         strncpy(imagedate+5,tags_imdate+4,2);                             //  tag date = yyyy:mm:dd
         strncpy(imagedate+8,tags_imdate+6,2);
         imagedate[4] = imagedate[7] = ':';
         imagedate[10] = 0;
      }
      else strcpy(imagedate,"null");

      err = stat(file,&statb);
      gmtime_r(&statb.st_mtime,&bdt);
      sprintf(filedate,"%04d%02d%02d%02d%02d%02d",                         //  file date = yyyymmddhhmmss  v.8.4
               bdt.tm_year + 1900, bdt.tm_mon + 1, bdt.tm_mday,
               bdt.tm_hour, bdt.tm_min, bdt.tm_sec);

      err = fprintf(fidw,"date: %s  %s\n",imagedate,filedate);             //  output tag date and file date
      
      if (*tags_filetags) err = fprintf(fidw,"tags: %s\n",tags_filetags);  //  output image tags
      else  err = fprintf(fidw,"tags: null\n");                            //  "null" if none   v.7.5

      err = fprintf(fidw,"file: %s\n\n",file);                             //  output filespec
      if (err <= 0) goto tagserror;
   }

   if (fidr) {
      err = fclose(fidr);
      if (err) goto tagserror;
   }

   err = fclose(fidw);
   if (err) goto tagserror;
   
   err = rename(temp_asstagsfile,asstagsfile);                             //  replace tag file with temp file
   if (err) goto tagserror;

   return;
   
tagserror:
   zmessLogACK(ZTX("Assigned tags file error: %s"),strerror(errno));
   return;
}


/**************************************************************************/

//  search image tags for matching images

char     searchDateFrom[12] = "";                                          //  image search date range
char     searchDateTo[12] = "";
int      searchStarsFrom = 0;                                              //  image search stars range
int      searchStarsTo = 0;

void m_search_tags(GtkWidget *, const char *)
{
   int search_tags_dialog_event(zdialog*, const char *event);
   int search_tags_dialog_compl(zdialog*, int zstat);

   if (zdtags) {
      zdialog_free(zdtags);
      zdtags = null;
   }
   
   zdtags = zdialog_new(ZTX("Search Tags"),mWin,Bsearch,Bcancel,0);
   
   zdialog_add_widget(zdtags,"hbox","hb1","dialog",0,"space=2");
   zdialog_add_widget(zdtags,"vbox","vb1","hb1",0,"homog|space=5");
   zdialog_add_widget(zdtags,"vbox","vb2","hb1",0,"homog|space=2");

   zdialog_add_widget(zdtags,"label","labDR","vb1",ZTX("date range"));
   zdialog_add_widget(zdtags,"label","labSR","vb1",ZTX("stars range"));
   zdialog_add_widget(zdtags,"label","labF","vb1",ZTX("/path*/file*"));
   zdialog_add_widget(zdtags,"label","labT","vb1",ZTX("search tags"));

   zdialog_add_widget(zdtags,"hbox","hbDR","vb2",0,"space=5");             //  date range    yyyymmdd  yyyymmdd
   zdialog_add_widget(zdtags,"entry","datefrom","hbDR",0,"scc=10");        //  stars range   4  5
   zdialog_add_widget(zdtags,"entry","dateto","hbDR",0,"scc=10");          //  /path*/file*  /dname/dname*/fname*
   zdialog_add_widget(zdtags,"label","labDF","hbDR","(yyyymmdd)");         //  search tags   rosi  alaska

   zdialog_add_widget(zdtags,"hbox","hbSR","vb2",0);                       //  (o) match all tags  (o) match any
   zdialog_add_widget(zdtags,"entry","starsfrom","hbSR",0,"scc=2");
   zdialog_add_widget(zdtags,"entry","starsto","hbSR",0,"scc=2");          //  assigned tags

   zdialog_add_widget(zdtags,"hbox","hbF","vb2",0,"space=5");              //  list of all tags in a box
   zdialog_add_widget(zdtags,"entry","searchfile","hbF",0,"expand");
   zdialog_add_widget(zdtags,"button","fileclear","hbF",Bclear);

   zdialog_add_widget(zdtags,"hbox","hbT","vb2",0,"space=5");
   zdialog_add_widget(zdtags,"entry","searchtags","hbT",0,"expand");
   zdialog_add_widget(zdtags,"button","tagsclear","hbT",Bclear);

   zdialog_add_widget(zdtags,"hbox","hbM","dialog");
   zdialog_add_widget(zdtags,"radio","rmall","hbM",ZTX("match all tags"),"space=5");
   zdialog_add_widget(zdtags,"label","lspace","hbM","","space=10");
   zdialog_add_widget(zdtags,"radio","rmany","hbM",ZTX("match any tag"));

   zdialog_add_widget(zdtags,"hbox","hbsp","dialog",0,"space=1");

   zdialog_add_widget(zdtags,"hbox","hbAT","dialog");
   zdialog_add_widget(zdtags,"label","labasstags","hbAT",ZTX("assigned tags"));
   zdialog_add_widget(zdtags,"frame","frameAT","dialog",0,"expand");
   zdialog_add_widget(zdtags,"edit","asstags2","frameAT",0,"expand");

   zdialog_resize(zdtags,400,0);                                           //  start dialog
   zdialog_run(zdtags,search_tags_dialog_event,search_tags_dialog_compl);

   edit_tags_fixup("asstags2");                                            //  setup tag selection via mouse
   
   zdialog_stuff(zdtags,"datefrom",searchDateFrom);                        //  stuff previous date range
   zdialog_stuff(zdtags,"dateto",searchDateTo);
   if (strNeq(tags_searchtags,"null"))                                     //  stuff previous search tags
      zdialog_stuff(zdtags,"searchtags",tags_searchtags);
   zdialog_stuff(zdtags,"rmall",1);                                        //  default is match all tags
   zdialog_stuff(zdtags,"rmany",0);
   load_asstags();                                                         //  stuff assigned tags
   zdialog_stuff(zdtags,"asstags2",tags_asstags);
   
   return;
}

  
int search_tags_dialog_event(zdialog *zd, const char *event)               //  dialog event function
{
   if (strEqu(event,"fileclear"))
      zdialog_stuff(zd,"searchfile","");

   if (strEqu(event,"tagsclear")) 
      zdialog_stuff(zd,"searchtags","");

   return 0;
}


int search_tags_dialog_compl(zdialog *zd, int zstat)                       //  dialog completion function
{
   const char  *pps, *ppf;
   char        resultsfile[200];
   char        *ppv, *file, *tags, rbuff[1000];
   char        date1[12], date2[12], lcfile[1000];
   int         err, nfiles, iis, iif, stars, nmatch, nfail;
   int         date1cc, date2cc, Fmall, Fdates, Ffiles, Ftags, Fstars;
   FILE        *fidr, *fidw;
   struct stat statbuf;

   if (zstat == 1) {
      zdialog_fetch(zd,"datefrom",searchDateFrom,10);                      //  get search date range
      zdialog_fetch(zd,"dateto",searchDateTo,10);
      zdialog_fetch(zd,"starsfrom",searchStarsFrom);                       //  get search stars range
      zdialog_fetch(zd,"starsto",searchStarsTo);
      zdialog_fetch(zd,"searchfile",tags_searchfile,maxtagF);              //  get search /path*/file*   v.6.6
      zdialog_fetch(zd,"searchtags",tags_searchtags,maxtag4);              //  get search tags
      zdialog_fetch(zd,"rmall",Fmall);                                     //  get match all/any option
   }
   
   zdialog_free(zdtags);                                                   //  kill dialog
   zdtags = null;
   if (zstat != 1) return 0;                                               //  cancelled

   strcpy(date1,"0000");                                                   //  defaults for missing dates  v.6.6
   strcpy(date2,"9999");                                                   //  (year 0000 to year 9999)
   date1cc = date2cc = 4;
   Fdates = 0;

   if (*searchDateFrom) {                                                  //  date from is given
      Fdates++;
      strncpy(date1,searchDateFrom,4);                                     //  convert format
      strncpy(date1+5,searchDateFrom+4,2);                                 //    yyyymmdd >> yyyy:mm:dd
      strncpy(date1+8,searchDateFrom+6,2);
      date1[4] = date1[7] = ':';
      date1cc = strlen(date1);
   }
   if (*searchDateTo) {                                                    //  date to is given
      Fdates++;
      strncpy(date2,searchDateTo,4);
      strncpy(date2+5,searchDateTo+4,2);
      strncpy(date2+8,searchDateTo+6,2);
      date2[4] = date2[7] = ':';
      date2cc = strlen(date2);
   }

   Fstars = 0;
   if (searchStarsFrom || searchStarsTo) Fstars = 1;                       //  stars given
   
   Ffiles = 0;
   if (! blank_null(tags_searchfile)) Ffiles = 1;                          //  search file* given     v.6.6

   Ftags = 0;
   if (! blank_null(tags_searchtags)) Ftags = 1;                           //  search tags given

   if (! Ffiles && ! Ftags && ! Fdates && ! Fstars) {                      //  no search criteria was given,
      strcpy(tags_searchtags,"null");                                      //    find images with no tags   v.6.9.3
      Ftags = 1;
   }
   
   if (Ffiles) strToLower(tags_searchfile);                                //  v.6.6
   if (Ftags) strToLower(tags_searchtags);
   
   snprintf(resultsfile,199,"%s/search_results",get_zuserdir());
   fidw = fopen(resultsfile,"w");                                          //  search results output file
   if (! fidw) goto writerror;

   fidr = fopen(asstagsfile,"r");                                          //  read assigned tags file
   if (! fidr) goto noasstags;
   
   nfiles = 0;                                                             //  count matching files found

   while (true)
   {
      ppv = fgets_trim(rbuff,999,fidr);                                    //  next assigned tags record
      if (! ppv) break;
      if (! strnEqu(ppv,"date: ",6)) continue;                             //  date: yyyy:mm:dd 

      if (Fdates) {      
         if (strncmp(ppv+6,date1,date1cc) < 0) continue;                   //  check search date range
         if (strncmp(ppv+6,date2,date2cc) > 0) continue;
      }

      ppv = fgets_trim(rbuff,999,fidr);                                    //  next record
      if (! ppv) break;
      if (! strnEqu(ppv,"tags: ",6)) continue;                             //  tags: xxxx xxxxx ...

      if (Ftags)
      {                                                                    //  tag search
         tags = ppv + 6;
         strToLower(tags);                                                 //  v.6.6

         nmatch = nfail = 0;

         for (iis = 1; ; iis++)
         {
            pps = strField(tags_searchtags,' ',iis);                       //  step thru search tags
            if (! pps) break;

            for (iif = 1; ; iif++)                                         //  step thru file tags
            {
               ppf = strField(tags,' ',iif);
               if (! ppf) { nfail++; break; }                              //  count matches and fails
               if (MatchWild(pps,ppf) == 0) { nmatch++; break; }           //  wildcard match   v.6.6
            }
         }

         if (nmatch == 0) continue;                                        //  no match to any tag
         if (Fmall && nfail) continue;                                     //  no match to all tags
      }
      
      if (Fstars)
      {                                                                    //  stars search
         nfail = 0;

         for (iif = 1; ; iif++)                                            //  step thru file tags
         {
            ppf = strField(ppv+6,' ',iif);
            if (! ppf) { nfail++; break; }
            if (! strnEqu(ppf,"stars=",6)) continue;
            stars = atoi(ppf+6);
            if (stars < searchStarsFrom) nfail++;
            if (searchStarsTo && stars > searchStarsTo) nfail++;
            break;
         }

         if (nfail) continue;
      }

      ppv = fgets_trim(rbuff,999,fidr,1);                                  //  next record
      if (! ppv) break;
      if (! strnEqu(ppv,"file: ",6)) continue;                             //  file: /dirks.../file.jpg
      file = ppv+6;
      strToLower(lcfile,file);                                             //  v.6.6
      
      if (Ffiles)                                                          //  test for path*/file* match
      {                                                                    //  v.6.6
         nmatch = 0;

         for (iis = 1; ; iis++)
         {
            pps = strField(tags_searchfile,' ',iis);                       //  step thru search files
            if (! pps) break;
            if (MatchWild(pps,lcfile) == 0) { nmatch++; break; }
         }
         
         if (nmatch == 0) continue;
      }

      err=stat(file,&statbuf);                                             //  check file exists
      if (err) continue;
      if (! S_ISREG(statbuf.st_mode)) continue;

      fprintf(fidw,"%s\n",file);                                           //  write matching file
      nfiles++;
   }

   fclose(fidr);
   
   err = fclose(fidw);
   if (err) goto writerror;
   
   if (! nfiles) {
      zmessageACK(ZTX("No matching images found"));
      return 0;
   }
   
   image_gallery(resultsfile,"initF",0,m_gallery2);                        //  generate gallery of matching files
   file = image_gallery(0,"first",0);
   f_open(file);                                                           //  show first matching file
   image_gallery(0,"paint1");                                              //  show new image gallery window
   Fsearchlist = 1;                                                        //  restricted updates     v.6.4

   return 0;

noasstags:
   zmessageACK(ZTX("No assigned tags index file"));
   return 0;

writerror:
   zmessLogACK(ZTX("Search results file error %s"),strerror(errno));
   return 0;
}


/**************************************************************************/

//  list available EXIF data to popup window

void m_exif_list(GtkWidget *, const char *menu)
{
   char        command[1000];
   const char  *basic = "-common -focallengthin35mmformat";

   if (! image_file) return;

   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(Bexiftoolmissing);
      return;
   }
   
   if (strEqu(menu,ZTX("Basic EXIF data"))) {                              //  bugfix   v.7.7
      snprintf(command,999,"exiftool %s \"%s\" ",basic,image_file);
      popup_command(command,500,300);
   }

   if (strEqu(menu,ZTX("All EXIF data"))) {
      snprintf(command,999,"exiftool -e \"%s\" ",image_file);
      popup_command(command,500,500);
   }
   
   return;
}


/**************************************************************************/

//  get EXIF metadata for given image file and EXIF key(s)
//  returns array of pointers to corresponding key values
//  if a key is missing, corresponding pointer is null
//  returned strings belong to caller, are subject for zfree()
//  no more than 9 keynames may be requested per call
//  EXIF command: 
//       exiftool -exif:keyname1 -exif:keyname2 ... "file"
//  command output: 
//       keyname1: keyvalue1
//       keyname2: keyvalue2
//       ...

char ** exif_get(cchar *file, cchar **keys, int nkeys)
{
   char           command[1000], *pp;
   static char    *rettext[10];
   int            contx = 0, err, ii;
   uint           cc;
   
   if (nkeys < 1 || nkeys > 9) appcrash("exif_get nkeys: %d",nkeys);

   strcpy(command,"exiftool -m -q -S -fast");
   
   for (ii = 0; ii < nkeys; ii++)
   {
      rettext[ii] = null;
      strncatv(command,999," -exif:",keys[ii],null);
   }

   strncatv(command,999," \"",file,"\"",null);

   while (true)
   {
      pp = command_output(contx,command);
      if (! pp) break;

      for (ii = 0; ii < nkeys; ii++)
      {
         cc = strlen(keys[ii]);
         if (strnEqu(pp,keys[ii],cc))
            if (strlen(pp) > cc+2) rettext[ii] = strdupz(pp+cc+2);         //  check not empty   bugfix v.7.3
      }

      zfree(pp);
   }
   
   err = command_status(contx);
   if (err) printf("exif_get failed \n");                                  //  v.6.9

   return rettext;
}


/**************************************************************************/

//  create or change EXIF metadata for given image file and key(s)
//  up to 9 keys may be processed
//  EXIF command: 
//    exiftool -overwrite_original -exif:keyname="keyvalue" ... "file"

int exif_set(cchar *file, cchar **keys, cchar **text, int nkeys)
{
   char     command[1000];
   int      ii, err;
   
   if (nkeys < 1 || nkeys > 9) appcrash("exif_set nkeys: %d",nkeys);
   
   strcpy(command,"exiftool -m -q -overwrite_original");
   
   for (ii = 0; ii < nkeys; ii++)
      strncatv(command,999," -exif:",keys[ii],"=\"",text[ii],"\"",null);
   strncatv(command,999," \"",file,"\"",null);

   err = system(command);
   if (err) printf(" exif_set: %s \n",wstrerror(err));
   return err;
}


/**************************************************************************/

//  copy EXIF data from original image file to new (edited) image file
//  if nkeys > 0, up to 9 keys may be replaced with new values
//  EXIF command:
//    exiftool -tagsfromfile "file1" -exif:keyname="keyvalue" ... "file2"

int exif_copy(cchar *file1, cchar *file2, cchar **keys, cchar **text, int nkeys)
{
   char     command[2000];
   int      ii, err;
   
   strcpy(command,"exiftool -m -q -overwrite_original -tagsfromfile");
   strncatv(command,1999," \"",file1,"\"",null);

   for (ii = 0; ii < nkeys; ii++)
      strncatv(command,1999," -exif:",keys[ii],"=\"",text[ii],"\"",null);

   strncatv(command,1999," \"",file2,"\"",null);

   err = system(command);
   if (err) printf(" exif_copy: %s \n",wstrerror(err));
   return err;
}


/**************************************************************************
   Select an area within the current image.
   Subsequent edit functions are carried out within the area.
   Otherwise, edit functions apply to the entire image.
***************************************************************************/

int         sa_type = 0;                                                   //  0=none, 1=mouse, 2=color
int         sa_stat = 0;                                                   //  0=none, 1=active, 2=suspend, 3=compl

uint16      *sa_pixseq = 0;                                                //  mark pixels by select sequence no.
int         sa_currseq = 0;                                                //  current select sequence no.
int         sa_Ncurrseq = 0;                                               //  select sequence pixel count

char        *sa_pixisin = 0;                                               //  mark pixels, edge/inside = 1/2

struct      sa_pixel1 { int16  px, py, dist; };                            //  map pixels in select area
sa_pixel1   *sa_pixel = 0;                                                 //    with distance from edge
int         sa_Npixel = 0;                                                 //  count of select_area pixels

int         sa_calced = 0;                                                 //  edge calculation done
int         sa_blend = 0;                                                  //  edge blend width

char        *sa_stackdir = 0;                                              //  pixel search stack
int         *sa_stackii = 0;
int         sa_maxstack;
int         sa_Nstack;


/**************************************************************************/

//  outline an image area using the mouse                                  //  overhauled   v.8.7
//  may run parallel with edit functions                                   //  freehand draw and follow edge

void   sam_mousefunc();
int    sam_nearpix(int mx, int my, int rad, int &npx, int &npy);
void   sam_drawline(int px1, int py1, int px2, int py2);
void   sam_followedge(int mx1, int my1, int &mx2, int &my2);
void   sam_findedge(int px, int py, int thresh, int &epx, int &epy);
double sam_contrast(int mx, int my);
void   sam_show();
void   sam_finish();

int         sam_begpx, sam_begpy;                                          //  first pixel drawn
uint16      sam_endpx[10000], sam_endpy[10000];                            //  last pixel drawn per seqence no.
int         sam_maxseq = 9999;
int         sam_follow = 0;                                                //  follow edge mode
int         sam_thresh;                                                    //  mouse pixel distance threshold


//  user dialog

void m_select_mouse(GtkWidget *, const char *)                             //  menu function
{
   int   sam_dialog_event(zdialog *, const char *event);                   //  dialog event and completion funcs
   int   sam_dialog_compl(zdialog *, int zstat);

   const char  *title = ZTX("Select Area for Following Edits");
   const char  *helptext = ZTX("Drag and click to enclose an area.\n"
                               "Use right click to undo prior.");

   if (! image_file) return;                                               //  no image
   if (zdsela) return;                                                     //  already active
   
   if (sa_type && sa_type != 1)                                            //  delete other select_area type
      m_select_delete(0,0);
   if (sa_type && sa_type != 1) return;                                    //  refused
   
   if (Fpreview) edit_fullsize();                                          //  use full-size pixmaps
   if (Fimageturned) turn_image(-Fimageturned);                            //  use native orientation
   
   if (! Frgb48) {                                                         //  create Frgb48 if not already
      mutex_lock(&pixmaps_lock);
      Frgb48 = image_load(image_file,48);
      mutex_unlock(&pixmaps_lock);
      if (! Frgb48) return;
      update_statusbar();
   }

   zdsela = zdialog_new(title,mWin,BOK,Bcancel,null);
   zdialog_add_widget(zdsela,"label","labhelp","dialog",helptext,"space=10");
   zdialog_add_widget(zdsela,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdsela,"button","start","hb1",Bstart);
   zdialog_add_widget(zdsela,"button","susp-resm","hb1",Bsuspend);
   zdialog_add_widget(zdsela,"button","show-hide","hb1",Bhide);
   zdialog_add_widget(zdsela,"button","finish","hb1",Bfinish);
   zdialog_add_widget(zdsela,"button","delete","hb1",Bdelete);
   zdialog_add_widget(zdsela,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdsela,"radio","fhdraw","hb2","freehand draw","space=5");
   zdialog_add_widget(zdsela,"radio","follow","hb2","follow edge","space=5");
   zdialog_add_widget(zdsela,"hbox","hb3","dialog",0,"space=10");
   zdialog_add_widget(zdsela,"label","labblend","hb3",Bblendwidth);
   zdialog_add_widget(zdsela,"hscale","blendwidth","hb3","0|300|1|0","expand");
   
   zdialog_stuff(zdsela,"fhdraw",1);                                       //  default freehand draw mode
   zdialog_stuff(zdsela,"follow",0);
   sam_follow = 0;

   zdialog_run(zdsela,sam_dialog_event,sam_dialog_compl);                  //  run dialog - parallel
   m_select_show(0,0);                                                     //  show existing area
   return;
}


//  dialog event and completion functions

int sam_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat != 1) m_select_delete(0,0);                                   //  kill, cancel
   mouseCBfunc = 0;                                                        //  disconnect mouse function
   Mcapture = 0;
   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor
   zdialog_free(zdsela);                                                   //  kill dialog
   zdsela = null;
   return 0;
}

int sam_dialog_event(zdialog *zd, const char *event)
{
   if (strEqu(event,"start")) {                                            //  start or resume after disable
      m_select_disable(0,0);                                               //  disable area 
      sa_type = 1;                                                         //  area type = mouse select
      sa_blend = sa_calced = 0;                                            //  undo blend, edge calculation
      sa_stat = 1;                                                         //  edit active

      if (! sa_pixseq) {
         int cc = Fww * Fhh;                                               //  first edit for area
         sa_pixseq = (uint16 *) zmalloc(2*cc);
         memset(sa_pixseq,0,2*cc);
         sa_currseq = 0;
         sam_begpx = sam_begpy = 0;
      }
      
      m_select_show(0,0);
      zdialog_stuff(zdsela,"susp-resm",Bsuspend);
      mouseCBfunc = sam_mousefunc;                                         //  connect mouse function
      Mcapture++;                                                          //  mouse captured for me
      gdk_window_set_cursor(drWin->window,drawcursor);                     //  set draw cursor
   }

   if (strEqu(event,"show-hide")) {                                        //  toggle show/hide
      if (! Fshowarea) m_select_show(0,0);
      else m_select_hide(0,0);
   }
   
   if (strEqu(event,"susp-resm")) {                                        //  toggle suspend/resume
      if (sa_stat == 1) {
         mouseCBfunc = 0;                                                  //  disconnect mouse function
         Mcapture = 0;
         gdk_window_set_cursor(drWin->window,0);                           //  restore normal cursor
         zdialog_stuff(zdsela,"susp-resm",Bresume);
         sa_stat = 2;
      }
      else if (sa_stat == 2) {
         mouseCBfunc = sam_mousefunc;                                      //  connect mouse function
         Mcapture++;
         gdk_window_set_cursor(drWin->window,drawcursor);                  //  set draw cursor
         zdialog_stuff(zdsela,"susp-resm",Bsuspend);
         sa_stat = 1;
      }
   }

   if (strEqu(event,"finish")) {                                           //  finish area
      m_select_show(0,0);
      m_select_finish(0,0);                                                //  finish area
      if (sa_stat == 3) {
         mouseCBfunc = 0;                                                  //  disconnect mouse function
         Mcapture = 0;
         gdk_window_set_cursor(drWin->window,0);                           //  restore normal cursor
      }
   }

   if (strEqu(event,"delete")) {                                           //  delete area
      mouseCBfunc = 0;                                                     //  disconnect mouse function
      Mcapture = 0;
      gdk_window_set_cursor(drWin->window,0);                              //  restore normal cursor
      m_select_delete(0,0);
   }
   
   if (strEqu(event,"blendwidth")) {                                       //  blend width changed
      if (sa_Npixel && zdedit) {
         if (sa_calced) {
            zdialog_fetch(zd,"blendwidth",sa_blend);                       //  update sa_blend
            sa_blend = (sa_blend * sa_blend + 15) / 300;                   //  0-300 scaled               v.8.5
            zdialog_send_event(zdedit,event);                              //  notify active edit dialog
         }
         else  m_select_edgecalc(0,0);
      }
   }
   
   if (strEqu(event,"fhdraw")) sam_follow = 0;                             //  freehand draw mode
   if (strEqu(event,"follow")) sam_follow = 1;                             //  follow edge mode

   mwpaint2();                                                             //  update window
   return 0;
}


//  select area mouse function - draw lines

void sam_mousefunc()
{
   int         mx1, my1, mx2, my2;
   int         npdist, npx, npy;
   int         ii, click, newseq, thresh;
   static int  drag = 0, mdx0, mdy0, mdx1, mdy1;

   sam_thresh = 4.0 / Mscale + 1;                                          //  mouse pixel distance threshold
   click = newseq = 0;
   
   if (LMclick || Mxdrag || Mydrag)                                        //  left mouse click or mouse drag
   {
      if (LMclick)                                                         //  left mouse click
      {
         LMclick = 0;
         mx1 = mx2 = Mxclick;                                              //  click position
         my1 = my2 = Myclick;
         newseq++;
         click++;
         drag = 0;
      }
      else                                                                 //  drag motion
      {
         if (Mxdown != mdx0 || Mydown != mdy0) {                           //  new drag initiated
            mdx0 = mdx1 = Mxdown;
            mdy0 = mdy1 = Mydown;
            newseq++;
         }
         mx1 = mdx1;                                                       //  drag start
         my1 = mdy1;
         mx2 = Mxdrag;                                                     //  drag position
         my2 = Mydrag;
         mdx1 = mx2;                                                       //  next drag start
         mdy1 = my2;
         drag++;
         click = 0;
      }
      
      if (Mbutton == 3)                                                    //  right mouse >> erase
      {
         while (true) {
            thresh = sam_thresh;
            npdist = sam_nearpix(mx2,my2,thresh,npx,npy);
            if (! npdist) break;
            ii = npy * Fww + npx;
            sa_pixseq[ii] = 0;
         }
         mwpaint2();
         return;
      }

      if (sa_currseq > sam_maxseq-2) {
         zmessageACK(ZTX("exceed %d edits"),sam_maxseq);                   //  cannot continue
         return;
      }
      
      if (sa_currseq == 0 && newseq)                                       //  1st pixel(s) of 1st sequence
      {
         sa_currseq = 1;
         sam_drawline(mx1,my1,mx2,my2);                                    //  draw initial pixel or line
         sam_begpx = mx1;
         sam_begpy = my1;
         sam_endpx[sa_currseq] = mx2;
         sam_endpy[sa_currseq] = my2;
         return;
      }
      
      if (click) {
         mx1 = sam_endpx[sa_currseq];                                      //  prior sequence end pixel
         my1 = sam_endpy[sa_currseq];                                      //  (before this click)
      }
      
      if (drag) {
         if (newseq) thresh = 2 * sam_thresh;                              //  new drag threshold
         else thresh = 5 * sam_thresh;                                     //  continuation drag threshold
         npx = sam_endpx[sa_currseq];                                      //  distance from prior end pixel
         npy = sam_endpy[sa_currseq];                                      //    (before this drag)
         if (abs(mx1-npx) < thresh && abs(my1-npy) < thresh) {
            mx1 = sam_endpx[sa_currseq];                                   //  if < threshold, connect this
            my1 = sam_endpy[sa_currseq];                                   //    drag to prior drag or click
         }
      }

      if (newseq || drag > 50) {
         sa_currseq++;                                                     //  next sequence no.
         drag = 1;                                                         //  drag length within sequence
      }
      
      if (sam_follow) sam_followedge(mx1,my1,mx2,my2);
      else sam_drawline(mx1,my1,mx2,my2);                                  //  draw end pixel to mouse
      
      sam_endpx[sa_currseq] = mx2;                                         //  set end pixel for this sequence
      sam_endpy[sa_currseq] = my2;
   }

   else if (RMclick)                                                       //  right mouse click
   {
      RMclick = 0;
      if (! sa_currseq) return;
      for (int ii = 0; ii < Fww * Fhh; ii++)                               //  undo last draw action
         if (sa_pixseq[ii] == sa_currseq) sa_pixseq[ii] = 0;
      sa_currseq--;
      mwpaint2();
   }
   
   return;
}


//  Find the nearest pixel within a radius of a given pixel.
//  Returns distance to pixel, or zero if nothing found.
//  Returns 1 for adjacent or diagonally adjacent pixel.

int sam_nearpix(int mx, int my, int rad2, int &npx, int &npy)
{
   int      ii, rad, qx, qy, dx, dy;
   int      seq, mindist, dist;

   npx = npy = 0;
   mindist = (rad2+1) * (rad2+1);

   for (rad = 1; rad <= rad2; rad++)                                       //  seek neighbors within range
   {
      if (rad * rad > mindist) break;                                      //  can stop searching now

      for (qx = mx-rad; qx <= mx+rad; qx++)                                //  search within rad
      for (qy = my-rad; qy <= my+rad; qy++)
      {
         if (qx != mx-rad && qx != mx+rad &&                               //  exclude within rad-1
             qy != my-rad && qy != my+rad) continue;                       //  (already searched)
         if (qx < 0 || qx >= Fww) continue;
         if (qy < 0 || qy >= Fhh) continue;
         ii = qy * Fww + qx;
         seq = sa_pixseq[ii];
         if (! seq) continue;
         dx = (mx - qx) * (mx - qx);                                       //  found pixel
         dy = (my - qy) * (my - qy);
         dist = dx + dy;                                                   //  distance**2
         if (dist < mindist) {
            mindist = dist;
            npx = qx;                                                      //  save nearest pixel found
            npy = qy;
         }
      }
   }
   
   if (npx + npy) return sqrt(mindist) + 0.5;
   return 0;
}


//  draw a line between two given pixels
//  add all in-line pixels to sa_pixseq[]
  
void sam_drawline(int px1, int py1, int px2, int py2)
{
   int      pxm, pym;
   double   slope;
   
   if (px1 == px2 && py1 == py2) {                                         //  only one pixel
      select_drawpix1(px1,py1);
      return;
   }
   
   if (abs(py2 - py1) > abs(px2 - px1)) {
      slope = 1.0 * (px2 - px1) / (py2 - py1);
      if (py2 > py1) {
         for (pym = py1; pym <= py2; pym++) {
            pxm = round(px1 + slope * (pym - py1));
            select_drawpix1(pxm,pym);
         }
      }
      else {
         for (pym = py1; pym >= py2; pym--) {
            pxm = round(px1 + slope * (pym - py1));
            select_drawpix1(pxm,pym);
         }
      }
   }
   else {
      slope = 1.0 * (py2 - py1) / (px2 - px1);
      if (px2 > px1) {
         for (pxm = px1; pxm <= px2; pxm++) {
            pym = round(py1 + slope * (pxm - px1));
            select_drawpix1(pxm,pym);
         }
      }
      else {
         for (pxm = px1; pxm >= px2; pxm--) {
            pym = round(py1 + slope * (pxm - px1));
            select_drawpix1(pxm,pym);
         }
      }
   }

   return;
}


//  Find series of edge pixels from mx1/my1 to mx2/my2 and connect them together.
//  Return mx2/my2 = last edge pixel found (closest to input mx2/my2).

void sam_followedge(int mx1, int my1, int &mx2, int &my2)
{
   int         last = 0, epx, epy, pepx, pepy, thresh;
   double      dx, dy, epdist, f1dist;
   double      fx1, fy1, hyp, xdir, ydir;
   
   dx = mx2 - mx1;
   dy = my2 - my1;
   thresh = 0.01 * sqrt(dx*dx + dy*dy) + 0.5 * sam_thresh + 1;             //  more control with short segments

   if (mx1 == mx2 && my1 == my2) last = 1;                                 //  input line is 1 pixel
   
   pepx = mx1;
   pepy = my1;

   fx1 = mx1;
   fy1 = my1;
   
   while (true)
   {
      sam_findedge(round(fx1),round(fy1),thresh,epx,epy);                  //  find edge pixel near fx1/fy1
      
      if (pepx != epx || pepy != epy) {
         sam_drawline(pepx,pepy,epx,epy);                                  //  connect edge pixels
         pepx = epx;
         pepy = epy;
      }
      
      if (last) {
         mx2 = epx;                                                        //  return mx2/my2 =
         my2 = epy;                                                        //    last edge pixel found
         return;
      }
      
      dx = mx2 - epx;
      dy = my2 - epy;
      epdist = dx * dx + dy * dy;                                          //  edge pixel to mx2/my2
      dx = mx2 - fx1;
      dy = my2 - fy1;
      f1dist = dx * dx + dy * dy;                                          //  fx1/fy1 to mx2/my2

      if (epdist < f1dist) {
         fx1 = epx;                                                        //  if edge pixel closer, 
         fy1 = epy;                                                        //    move fx1/fy1 to edge pixel
         dx = mx2 - fx1;
         dy = my2 - fy1;
         f1dist = dx * dx + dy * dy;
         if (! f1dist) {
            last = 1;
            continue;
         }
      }
      
      hyp = sqrt(f1dist);
      xdir = dx / hyp;                                                     //  unit vector in direction to mx2/my2
      ydir = dy / hyp;
      xdir = 0.5 * thresh * xdir;                                          //  vector length = thresh/2
      ydir = 0.5 * thresh * ydir;
      fx1 = fx1 + xdir;                                                    //  move fx1/fy1 toward mx2/my2
      fy1 = fy1 + ydir;
      
      dx = mx2 - fx1;                                                      //  distance remaining to mx2/my2
      dy = my2 - fy1;
      f1dist = dx * dx + dy * dy;
      if (f1dist < thresh*thresh) last = 1;                                //  last iteration if < thresh
   }
}


//  Find highest contrast pixel within threshold distance of px/py.

void sam_findedge(int px, int py, int thresh, int &epx, int &epy)
{
   int      qx, qy;
   double   contrast, maxcontrast = 0;
   
   epx = px;
   epy = py;
   
   for (qx = px-thresh; qx <= px+thresh; qx++)
   for (qy = py-thresh; qy <= py+thresh; qy++)
   {
      contrast = sam_contrast(qx,qy);
      if (contrast > maxcontrast) {
         maxcontrast = contrast;
         epx = qx;
         epy = qy;
      }
   }

   return;
}


//  Find max. contrast between given pixel and its neighbors.

double sam_contrast(int px, int py)
{
   int         map[4][2] = { {1, 0}, {1, 1}, {0, 1}, {-1, 1} };
   int         ii, qx, qy;
   uint16      *pix1, *pix2;
   double      red, green, blue;
   double      contrast, maxcontrast = 0;
   
   if (px < 1 || px > Fww-2) return 0;                                     //  avoid edge pixels
   if (py < 1 || py > Fhh-2) return 0;
   
   for (ii = 0; ii < 4; ii++)                                              //  compare pixels around target
   {                                                                       //  e.g. (px-1,py) to (px+1,py)
      qx = map[ii][0];
      qy = map[ii][1];
      pix1 = bmpixel(Frgb48,px+qx,py+qy);
      pix2 = bmpixel(Frgb48,px-qx,py-qy);
      red = abs(pix1[0] - pix2[0]);                                        //  0 to 65536
      green = abs(pix1[1] - pix2[1]);
      blue = abs(pix1[2] - pix2[2]);
      contrast = (red + green + blue); 
      if (contrast > maxcontrast) maxcontrast = contrast;
   }
   
   return maxcontrast;
}


//  show select area outline

void sam_show()
{
   int      px, py, ii;
   
   if (! sa_pixseq) return;
   
   for (ii = 0; ii < Fww * Fhh; ii++)
   {
      if (sa_pixseq[ii]) {
         py = ii / Fww;
         px = ii - py * Fww;
         select_drawpix2(px,py);
      }
   }
   
   gdk_gc_set_foreground(gdkgc,&black);
   return;
}


//  finish select area - map pixels enclosed by selected edge pixels 
//    into  sa_pixisin[]   1=edge, 2=inside, [ii]=py*Fww+px
//    and   sa_pixel[] .px .py .dist   count=sa_Npixel, dist=0/1=edge/inside

void sam_finish()
{
   void  sam_finish_pushstack(int px, int py, char direc);
   void  sam_finish_showhole();
   void sam_finish_flasharea();

   int         ii, kk, npix, cc, px, py;
   int         pxmin, pxmax, pymin, pymax;
   int         nedge, ninside, stackfail = 0;
   char        direc;
   int64       seed = 0;

   if (! sa_pixseq) return;                                                //  nothing selected
   if (sa_currseq < 1) return;

try_again:

   if (sa_pixisin) zfree(sa_pixisin);                                      //  allocate pixisin[]
   cc = Fww * Fhh;
   sa_pixisin = zmalloc(cc);
   memset(sa_pixisin,0,cc);

   for (ii = nedge = 0; ii < Fww * Fhh; ii++)                              //  count edge pixels
   {
      if (! sa_pixseq[ii]) continue;
      sa_pixisin[ii] = 1;
      nedge++;                                                             //  populate pixisin[]
   }

   if (nedge < 20) return;

   pxmin = Fww;
   pxmax = 0;
   pymin = Fhh;
   pymax = 0;

   for (ii = 0; ii < Fww * Fhh; ii++)                                      //  get enclosing rectangle
   {                                                                       //    for selected area
      if (! sa_pixseq[ii]) continue;
      py = ii / Fww;
      px = ii - Fww * py;
      if (px > pxmax) pxmax = px;
      if (px < pxmin) pxmin = px;
      if (py > pymax) pymax = py;
      if (py < pymin) pymin = py;
   }
   
   if (pxmin > 0) pxmin--;                                                 //  add margins if room
   if (pxmax < Fww-1) pxmax++;
   if (pymin > 0) pymin--;
   if (pymax < Fhh-1) pymax++;
   
   cc = (pxmax-pxmin) * (pymax-pymin);                                     //  allocate stack memory
   if (sa_stackdir) zfree(sa_stackdir);
   sa_stackdir = zmalloc(cc);
   if (sa_stackii) zfree(sa_stackii);
   sa_stackii = (int *) zmalloc(cc * 4);
   sa_maxstack = cc;
   sa_Nstack = 0;
   
   seed = 1000 * CPUtime();                                                //  find pixel likely to be inside
   py = (pymin + pymax) / 2;
   py = py + (drandz(&seed) - 0.5) * 0.8 * (pymax - pymin);                //  random y value in the middle
   px = pxmin;
   ii = py * Fww + px;                                                     //  move x value from left side
   while (ii < Fww*Fhh && sa_pixisin[ii] == 0) ii++;                       //    until edge is passed
   while (ii < Fww*Fhh && sa_pixisin[ii]) ii++;
   if (ii == Fww * Fhh) goto try_again;
   
   sa_stackii[0] = ii;                                                     //  put 1st pixel into stack
   sa_stackdir[0] = 'r';                                                   //  direction = right
   sa_Nstack = 1;                                                          //  stack count

   while (sa_Nstack)
   {
      kk = sa_Nstack - 1;                                                  //  get last pixel in stack
      ii = sa_stackii[kk];
      direc = sa_stackdir[kk];
      
      py = ii / Fww;                                                       //  reconstruct px, py
      px = ii - Fww * py;

      if (px < pxmin || px > pxmax || py < pymin || py > pymax) {          //  ran off edge, initial pixel
         if (stackfail++ < 20) goto try_again;                             //    was not inside the area
         sam_finish_showhole();
         return;
      }

      if (direc == 'x') {                                                  //  no neighbors left to check
         sa_Nstack--;
         continue;
      }
      
      if (direc == 'r') {                                                  //  push next right pixel into stack
         sam_finish_pushstack(px,py,'r');
         sa_stackdir[kk] = 'l';                                            //  this pixel next direction to look
         continue;
      }

      if (direc == 'l') {                                                  //  or next left pixel
         sam_finish_pushstack(px,py,'l');
         sa_stackdir[kk] = 'a';
         continue;
      }

      if (direc == 'a') {                                                  //  or next ahead pixel
         sam_finish_pushstack(px,py,'a');
         sa_stackdir[kk] = 'x';
         continue;
      }
   }
   
   nedge = ninside = 0;
   for (py = pymin; py <= pymax; py++)                                     //  scan each row of pixels
   for (px = pxmin; px <= pxmax; px++)                                     //    within enclosing rectangle
   {
      ii = py * Fww + px;
      if (sa_pixisin[ii] == 1) nedge++;                                    //  count edge and inside pixels
      if (sa_pixisin[ii] == 2) ninside++;
   }

   cc = (pxmax-pxmin) * (pymax-pymin);
   if (ninside < cc/100) goto try_again;                                   //  inadvertent loop on the edge?
   npix = nedge + ninside;

   sam_finish_flasharea();                                                 //  give user some feedback

   if (sa_pixel) zfree(sa_pixel);                                          //  allocate sa_pixel[]
   cc = npix * sizeof(sa_pixel1);                                          //    for exact pixel count
   sa_pixel = (sa_pixel1 *) zmalloc(cc);
   
   npix = 0;
   for (py = pymin; py <= pymax; py++)                                     //  scan each row of pixels
   for (px = pxmin; px <= pxmax; px++)                                     //    within enclosing rectangle
   {
      ii = py * Fww + px;
      if (sa_pixisin[ii]) {
         sa_pixel[npix].px = px;
         sa_pixel[npix].py = py;
         sa_pixel[npix].dist = sa_pixisin[ii] - 1;
         npix++;
      }
   }
   
   sa_stat = 3;                                                            //  area is finished
   sa_calced = 0;                                                          //  edge calculation is missing
   sa_Npixel = npix;
   return;
}


//  push pixel into stack memory if not already mapped

void sam_finish_pushstack(int px, int py, char direc)
{
   int      ii, kk, ppx, ppy, npx, npy;

   if (sa_Nstack > 1) {
      kk = sa_Nstack - 2;                                                  //  get prior pixel in stack
      ii = sa_stackii[kk];
      ppy = ii / Fww;
      ppx = ii - ppy * Fww;
   }
   else {
      ppx = px - 1;                                                        //  if only one, assume prior = left
      ppy = py;
   }
   
   if (direc == 'r') {                                                     //  get pixel in direction right
      npx = px + ppy - py;
      npy = py + px - ppx;
   }
   else if (direc == 'l') {                                                //  or left
      npx = px + py - ppy;
      npy = py + ppx - px;
   }
   else if (direc == 'a') {                                                //  or ahead
      npx = px + px - ppx;
      npy = py + py - ppy;
   }
   else npx = npy = -1;                                                    //  stop warning
   
   if (npx < 0 || npx >= Fww) return;                                      //  pixel off the edge
   if (npy < 0 || npy >= Fhh) return;
   
   ii = npy * Fww + npx;
   if (sa_pixisin[ii]) return;                                             //  pixel already mapped
   if (sa_Nstack == sa_maxstack) return;                                   //  stack is full (impossible)

   sa_pixisin[ii] = 2;                                                     //  map pixel as interior

   kk = sa_Nstack++;                                                       //  put pixel into stack
   sa_stackii[kk] = ii;
   sa_stackdir[kk] = 'r';                                                  //  direction = right
   return;
}


//  highlight where a hole exists in the area outline

void sam_finish_showhole()                                                 //  v.8.7
{
   int      ii, px, py;

   for (ii = 0; ii < Fww * Fhh; ii++)
   {
      if (sa_pixisin[ii] == 2) {
         py = ii / Fww;
         px = ii - Fww * py;
         select_drawpix2(px,py);
      }
   }

   zmainloop();
   zmessageACK(ZTX("area outline has a hole"));
   mwpaint2();
   return;
}


//  highlight finished area briefly for user feedback

void sam_finish_flasharea()                                                //  v.8.7
{
   int      ii, px, py;

   for (ii = 0; ii < Fww * Fhh; ii++)
   {
      if (sa_pixisin[ii] == 2) {
         py = ii / Fww;
         px = ii - Fww * py;
         select_drawpix2(px,py);
      }
   }

   zmainloop();
   zsleep(1);
   mwpaint2();
   return;
}


/**************************************************************************/

//  select an image area by clicking and mapping colors                    //  revised 8.7
//  may run parallel with edit functions

void  sac_mousefunc();
void  sac_select_pixels();
void  sac_unselect_pixels();
void  sac_select_show();
void  sac_select_finish();

int         sac_mousex, sac_mousey;                                        //  mouse position in image
int         sac_targR, sac_targG, sac_targB;                               //  target pixel RGB
double      sac_targmatch;                                                 //  color range to match (0.001 to 1.0)


void m_select_color(GtkWidget *, const char *)                             //  menu function
{
   int   sac_dialog_event(zdialog*, const char *event);                    //  dialog event and completion funcs
   int   sac_dialog_compl(zdialog*, int zstat);

   const char  *title = ZTX("Select Area for Following Edits");
   const char  *helptext = ZTX("Left click/drag: add to selected area. \n"
                               "Right click: remove prior selection(s). \n"
                               "Color range: add more or less at once.");

   if (! image_file) return;                                               //  no image
   if (zdsela) return;                                                     //  already active

   if (sa_type && sa_type != 2)                                            //  delete other select_area type
      m_select_delete(0,0);
   if (sa_type && sa_type != 2) return;                                    //  refused
   
   if (Fpreview) edit_fullsize();                                          //  use full-size pixmaps
   if (Fimageturned) turn_image(-Fimageturned);                            //  use native orientation

   if (! Frgb48) {                                                         //  create Frgb48 if not already
      mutex_lock(&pixmaps_lock);
      Frgb48 = image_load(image_file,48);
      mutex_unlock(&pixmaps_lock);
      if (! Frgb48) return;
      update_statusbar();
   }

   zdsela = zdialog_new(title,mWin,BOK,Bcancel,null);
   zdialog_add_widget(zdsela,"label","labhelp","dialog",helptext,"space=5");
   zdialog_add_widget(zdsela,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdsela,"button","start","hb1",Bstart);
   zdialog_add_widget(zdsela,"button","susp-resm","hb1",Bsuspend);
   zdialog_add_widget(zdsela,"button","show-hide","hb1",Bhide);
   zdialog_add_widget(zdsela,"button","finish","hb1",Bfinish);
   zdialog_add_widget(zdsela,"button","delete","hb1",Bdelete);
   zdialog_add_widget(zdsela,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdsela,"label","labcolor","hb3",ZTX("color range"));
   zdialog_add_widget(zdsela,"hscale","range","hb3","0|99.9|0.1|40","expand");
   zdialog_add_widget(zdsela,"hbox","hb4","dialog",0,"space=5");
   zdialog_add_widget(zdsela,"label","labblend","hb4",Bblendwidth);
   zdialog_add_widget(zdsela,"hscale","blendwidth","hb4","0|300|1|0","expand");
   
   zdialog_run(zdsela,sac_dialog_event,sac_dialog_compl);                  //  run dialog - parallel
   m_select_show(0,0);                                                     //  show area
   return;
}


int sac_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat != 1) m_select_delete(0,0);                                   //  kill, cancel
   mouseCBfunc = 0;                                                        //  disconnect mouse function
   Mcapture = 0;
   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor
   zdialog_free(zdsela);                                                   //  kill dialog
   zdsela = null;
   return 0;
}


int sac_dialog_event(zdialog *zd, const char *event)
{
   int      cc = Fww * Fhh;

   if (strEqu(event,"start"))                                              //  start or resume after disable
   {
      m_select_disable(0,0);
      sa_type = 2;
      sa_stat = 1;
      sa_blend = sa_calced = 0;

      if (! sa_pixseq) {
         sa_pixseq = (uint16 *) zmalloc(2*cc);
         memset(sa_pixseq,0,2*cc);
         sa_currseq = 0;
      }

      if (sa_stackdir) zfree(sa_stackdir);
      if (sa_stackii) zfree(sa_stackii);
      sa_stackdir = zmalloc(cc);
      sa_stackii = (int *) zmalloc(4*cc);
      sa_maxstack = cc;
      sa_Nstack = 0;

      m_select_show(0,0);
      zdialog_stuff(zdsela,"susp-resm",Bsuspend);
      mouseCBfunc = sac_mousefunc;                                         //  connect mouse function
      Mcapture++;                                                          //  mouse captured for me
      gdk_window_set_cursor(drWin->window,dragcursor);                     //  set drag cursor
   }
   
   if (strEqu(event,"show-hide")) {                                        //  toggle show/hide
      if (! Fshowarea) m_select_show(0,0);
      else m_select_hide(0,0);
   }
   
   if (strEqu(event,"susp-resm")) {                                        //  toggle suspend/resume
      if (sa_stat == 1) {
         mouseCBfunc = 0;                                                  //  disconnect mouse function
         Mcapture = 0;
         gdk_window_set_cursor(drWin->window,0);                           //  restore normal cursor
         zdialog_stuff(zdsela,"susp-resm",Bresume);
         sa_stat = 2;
      }
      else if (sa_stat == 2) {
         mouseCBfunc = sac_mousefunc;                                      //  connect mouse function
         Mcapture++;
         gdk_window_set_cursor(drWin->window,dragcursor);                  //  set drag cursor
         zdialog_stuff(zdsela,"susp-resm",Bsuspend);
         sa_stat = 1;
      }
   }

   if (strEqu(event,"finish")) {                                           //  finish area
      mouseCBfunc = 0;                                                     //  disconnect mouse function
      Mcapture = 0;
      gdk_window_set_cursor(drWin->window,0);                              //  restore normal cursor
      m_select_show(0,0);
      m_select_finish(0,0);                                                //  finish area
   }
   
   if (strEqu(event,"delete")) {                                           //  delete area
      mouseCBfunc = 0;                                                     //  disconnect mouse function
      Mcapture = 0;
      gdk_window_set_cursor(drWin->window,0);                              //  restore normal cursor
      m_select_delete(0,0);
   }

   if (strEqu(event,"blendwidth")) {                                       //  blend width changed
      if (sa_Npixel && zdedit) {
         if (sa_calced) {
            zdialog_fetch(zd,"blendwidth",sa_blend);                       //  update sa_blend
            sa_blend = (sa_blend * sa_blend + 15) / 300;                   //  0-300 scaled               v.8.5
            zdialog_send_event(zdedit,event);                              //  notify active edit dialog
         }
         else  m_select_edgecalc(0,0);
      }
   }

   mwpaint2();                                                             //  update window
   return 0;
}


//  select area by color - mouse function

void sac_mousefunc()
{
   static int  mxdown, mydown, drag = 0;
   
   sac_mousex = sac_mousey = 0;

   if (LMclick) {                                                          //  get mouse position at click
      sac_mousex = Mxclick;
      sac_mousey = Myclick;
      LMclick = 0;
      sa_currseq++;                                                        //  new sequence number for undo
      drag = 1;
   }
      
   if (Mxdrag || Mydrag) {                                                 //  get mouse drag position
      sac_mousex = Mxdrag;
      sac_mousey = Mydrag;
      Mxdrag = Mydrag = 0;

      if (Mxdown != mxdown || Mydown != mydown) {                          //  detect if new drag started
         mxdown = Mxdown;
         mydown = Mydown;
         sa_currseq++;                                                     //  yes - new sequence number
         drag = 1;
      }
      else if (++drag > 50) {                                              //  limit work per sequence no.  v.8.7
         sa_currseq++;
         drag = 1;
      }
   }
   
   if (sac_mousex || sac_mousey) {
      sac_select_pixels();                                                 //  accumulate pixels
      mwpaint2();
   }

   if (RMclick) {
      RMclick = 0;
      if (sa_currseq) {                                                    //  remove selected pixels having
         sac_unselect_pixels();                                            //    current sequence number
         sa_currseq--;
      }
      mwpaint2();
   }

   return;
}


//  find all contiguous pixels within the specified color range

void sac_select_pixels()
{
   void  sac_select_pushstack(int px, int py, char direc);

   int      ii, kk, px, py;
   uint16   *targpix;
   char     direc;
   
   if (! sa_pixseq) return;

   px = sac_mousex;                                                        //  mouse position in image
   py = sac_mousey;

   ii = Fww * py + px;
   for (kk = 0; kk <= int(1/Mscale + 2); kk++)                             //  if target pixel already selected,        
   for (px = sac_mousex-kk; px <= sac_mousex+kk; px++)                     //     find nearest unselected pixel
   for (py = sac_mousey-kk; py <= sac_mousey+kk; py++)                     //  (relax need for mouse precision)
   {                                                                       //  (works better for scaled-down image)
      if (px < 0 || px >= Fww) continue;
      if (py < 0 || py >= Fhh) continue;
      ii = Fww * py + px;
      if (sa_pixseq[ii] == 0) goto gotpix;
   }
   gotpix:
   if (sa_pixseq[ii] > 0) return;                                          //  nothing will be selected

   sa_pixseq[ii] = sa_currseq;                                             //  map pixel to current sequence
   sa_Ncurrseq = 1;                                                        //  current sequence pixel count

   targpix = bmpixel(Frgb48,px,py);                                        //  get color at mouse position
   sac_targR = targpix[0];                                                 //    = target color
   sac_targG = targpix[1];
   sac_targB = targpix[2];
   
   zdialog_fetch(zdsela,"range",sac_targmatch);                            //  color range, 0.0 to 99.9
   sac_targmatch = 1.0 - 0.01 * sac_targmatch;                             //  target match level, 0.001 to 1.0

   sa_stackii[0] = ii;                                                     //  put 1st pixel into stack
   sa_stackdir[0] = 'r';                                                   //  direction = right
   sa_Nstack = 1;                                                          //  stack count

   while (sa_Nstack)
   {
      kk = sa_Nstack - 1;                                                  //  get last pixel in stack
      ii = sa_stackii[kk];
      direc = sa_stackdir[kk];
      
      py = ii / Fww;                                                       //  reconstruct px, py
      px = ii - Fww * py;

      if (direc == 'x') {                                                  //  no neighbors left to check
         sa_Nstack--;
         continue;
      }
      
      if (direc == 'r') {                                                  //  push next right pixel into stack
         sac_select_pushstack(px,py,'r');                                  //    if color within range
         sa_stackdir[kk] = 'l';                                            //  this pixel next direction to look
         continue;
      }

      if (direc == 'l') {                                                  //  or next left pixel
         sac_select_pushstack(px,py,'l');
         sa_stackdir[kk] = 'a';
         continue;
      }

      if (direc == 'a') {                                                  //  or next ahead pixel
         sac_select_pushstack(px,py,'a');
         sa_stackdir[kk] = 'x';
         continue;
      }
   }

   return;
}      


//  push pixel into stack memory if its color is within range
//  and not already mapped to a prior sequence number

void sac_select_pushstack(int px, int py, char direc)
{
   int         ii, kk, ppx, ppy, npx, npy;
   uint16      *matchpix;
   double      match, ff = 1.0 / 65536.0;
   double      dred, dgreen, dblue;

   if (sa_Nstack > 1) {
      kk = sa_Nstack - 2;                                                  //  get prior pixel in stack
      ii = sa_stackii[kk];
      ppy = ii / Fww;
      ppx = ii - ppy * Fww;
   }
   else {
      ppx = px - 1;                                                        //  if only one, assume prior = left
      ppy = py;
   }
   
   if (direc == 'r') {                                                     //  get pixel in direction right
      npx = px + ppy - py;
      npy = py + px - ppx;
   }
   else if (direc == 'l') {                                                //  or left
      npx = px + py - ppy;
      npy = py + ppx - px;
   }
   else if (direc == 'a') {                                                //  or ahead
      npx = px + px - ppx;
      npy = py + py - ppy;
   }
   else npx = npy = -1;                                                    //  stop warning
   
   if (npx < 0 || npx >= Fww) return;                                      //  pixel off the edge
   if (npy < 0 || npy >= Fhh) return;
   
   ii = npy * Fww + npx;
   if (sa_pixseq[ii]) return;                                              //  pixel already mapped

   matchpix = bmpixel(Frgb48,npx,npy);                                     //  match pixel RGB colors
   dred = ff * abs(sac_targR - matchpix[0]);                               //    with target pixel colors
   dgreen = ff * abs(sac_targG - matchpix[1]);
   dblue = ff * abs(sac_targB - matchpix[2]);
   match = (1.0 - dred) * (1.0 - dgreen) * (1.0 - dblue);
   if (match < sac_targmatch) return;                                      //  inadequate match

   if (sa_Nstack == sa_maxstack) return;                                   //  stack is full

   sa_pixseq[ii] = sa_currseq;                                             //  map pixel to curr. sequence
   sa_Ncurrseq++;
   
   kk = sa_Nstack++;                                                       //  put pixel into stack
   sa_stackii[kk] = ii;
   sa_stackdir[kk] = 'r';                                                  //  direction = right

   return;
}


//  un-select all pixels mapped to current sequence number

void sac_unselect_pixels()
{
   if (! sa_currseq) return;
   
   for (int ii = 0; ii < Fww * Fhh; ii++)
   {
      if (sa_pixseq[ii] != sa_currseq) continue;
      sa_pixseq[ii] = 0;
   }
   
   sa_Ncurrseq = 0;
   return;
}


//  find pixels at edge of area and paint black/white

void sac_show()
{
   int      px, py, ii, kk;
   
   if (! sa_pixseq) return;

   for (py = 1; py < Fhh-1; py++)                                          //  find pixels in area
   for (px = 1; px < Fww-1; px++)
   {
      ii = py * Fww + px;
      if (! sa_pixseq[ii]) continue;                                       //  outside of area

      if (! sa_pixseq[ii-1] || ! sa_pixseq[ii+1]) goto edgepixel;          //  check 8 neighbor pixels
      kk = ii - Fww;
      if (! sa_pixseq[kk] || ! sa_pixseq[kk-1] || ! sa_pixseq[kk+1]) goto edgepixel;
      kk = ii + Fww;
      if (! sa_pixseq[kk] || ! sa_pixseq[kk-1] || ! sa_pixseq[kk+1]) goto edgepixel;
      continue;

   edgepixel:
      select_drawpix2(px,py);
   }

   return;
}


//  finish select area - map pixels in sa_pixseq[]
//    into  sa_pixisin[]   1=edge, 2=inside, [ii]=py*Fww+px
//    and   sa_pixel[] .px .py .dist   count=sa_Npixel, dist=0/1=edge/inside

void sac_finish()
{
   int      ii, kk, cc, npix, px, py;

   if (sa_Npixel) return;                                                  //  already finished   
   if (! sa_pixseq) return;                                                //  no pixels selected

   for (ii = npix = 0; ii < Fww * Fhh; ii++)                               //  count selected pixels
      if (sa_pixseq[ii]) npix++;
   if (npix < 10) return;
   
   if (sa_pixisin) zfree(sa_pixisin);                                      //  allocate pixisin[]
   cc = Fww * Fhh;
   sa_pixisin = zmalloc(cc);
   memset(sa_pixisin,0,cc);
   
   npix = 0;

   for (px = 0; px < Fww; px += 1)                                         //  image top & bottom edges
   for (py = 0; py < Fhh; py += Fhh-1)
   {
      ii = py * Fww + px;
      if (! sa_pixseq[ii]) continue;
      sa_pixisin[ii] = 2;                                                  //  if selected, force non-edge
      npix++;
   }

   for (px = 0; px < Fww; px += Fww-1)                                     //  image left and right edges
   for (py = 0; py < Fhh; py += 1)
   {
      ii = py * Fww + px;
      if (! sa_pixseq[ii]) continue;
      sa_pixisin[ii] = 2;                                                  //  if selected, force non-edge
      npix++;
   }

   for (py = 1; py < Fhh-1; py++)                                          //  check all other pixels
   for (px = 1; px < Fww-1; px++)
   {
      ii = py * Fww + px;
      if (! sa_pixseq[ii]) continue;                                       //  outside of selected area
      npix++;

      if (! sa_pixseq[ii-1] || ! sa_pixseq[ii+1]) goto edgepixel;          //  check 8 neighbor pixels
      kk = ii - Fww;
      if (! sa_pixseq[kk] || ! sa_pixseq[kk-1] || ! sa_pixseq[kk+1]) goto edgepixel;
      kk = ii + Fww;
      if (! sa_pixseq[kk] || ! sa_pixseq[kk-1] || ! sa_pixseq[kk+1]) goto edgepixel;

      sa_pixisin[ii] = 2;                                                  //  non-edge pixel
      continue;

   edgepixel:
      sa_pixisin[ii] = 1;                                                  //  edge pixel
   }
   
   if (sa_pixel) zfree(sa_pixel);                                          //  allocate sa_pixel[]
   cc = npix * sizeof(sa_pixel1);                                          //    for exact pixel count
   sa_pixel = (sa_pixel1 *) zmalloc(cc);
   
   for (py = kk = 0; py < Fhh; py++)                                       //  sa_pixel[] in py, px order
   for (px = 0; px < Fww; px++)
   {
      ii = py * Fww + px;
      if (sa_pixisin[ii]) {
         sa_pixel[kk].px = px;
         sa_pixel[kk].py = py;
         sa_pixel[kk].dist = sa_pixisin[ii] - 1;                           //  dist = 0/1 = edge/inside
         kk++;
      }
   }
   
   sa_stat = 3;                                                            //  area is finished
   sa_calced = 0;                                                          //  edge calculation missing
   sa_Npixel = npix;
   return;
}


/**************************************************************************
   functions common to both mouse-selected and color-selected areas
***************************************************************************/

//  show outline of selected area - also called when window is repainted

void m_select_show(GtkWidget *, const char *)                              //  v.8.5
{
   Fshowarea = 1;
   if (zdsela) zdialog_stuff(zdsela,"show-hide",Bhide);
   if (sa_type == 1) sam_show();
   if (sa_type == 2) sac_show();
   return;
}


//  hide select area

void m_select_hide(GtkWidget *, const char *)                              //  v.8.5
{
   Fshowarea = 0;
   if (zdsela) zdialog_stuff(zdsela,"show-hide",Bshow);
   mwpaint2();
}


//  finish select area

void m_select_finish(GtkWidget *, const char *)
{
   if (sa_type == 1) sam_finish();
   if (sa_type == 2) sac_finish();
   return;
}


//  compute distance from all pixels in area to nearest edge

int      sa_edgecalc_pixdone;
int      sa_edgecalc_kill;
int      sa_edgecalc_busy;
uint16   *sa_edgedist;

void m_select_edgecalc(GtkWidget *, const char *)                          //  overhauled   v.8.7
{
   int    edgecalc_dialog_compl(zdialog*, int zstat);
   void * edgecalc_wthread(void *);

   zdialog     *zecdialog = 0;
   const char  *zectext = ZTX("Edge calculation in progress");

   m_select_finish(0,0);                                                   //  finish if needed
   if (! sa_Npixel) return;                                                //  no finished area
   
   if (sa_Npixel > 0.1 * mega) {
      zecdialog = zdialog_new(ZTX("Area Edge Calc"),mWin,Bcancel,0);
      zdialog_add_widget(zecdialog,"label","lab1","dialog",zectext,"space=10");
      zdialog_run(zecdialog,0,edgecalc_dialog_compl);
   }
   
   gdk_window_set_cursor(drWin->window,busycursor);                        //  set busy cursor
   zmainloop();
   
   int cc = Fww * Fhh * 2;                                                 //  allocate memory for calculations
   sa_edgedist = (uint16 *) zmalloc(cc);
   memset(sa_edgedist,0,cc);

   sa_edgecalc_pixdone = 0;
   sa_edgecalc_kill = 0;
   sa_edgecalc_busy = 0;

   for (int ii = 0; ii < NWthreads; ii++)                                  //  start worker threads to calculate
      start_detached_thread(edgecalc_wthread,&wtindex[ii]);                //    sa_pixel[].dist  values
   zadd_locked(sa_edgecalc_busy,+NWthreads);

   while (sa_edgecalc_busy)
   {
      edit_progress(sa_edgecalc_pixdone,sa_Npixel);                        //  monitor progress
      zsleep(0.1);
      zmainloop();                                                         //  allow kill
   }
   
   if (zecdialog) zdialog_free(zecdialog);

   edit_progress(0,0);
   sa_calced = 1;                                                          //  edge calculation available
   if (sa_edgecalc_kill) sa_calced = 0;
   zfree(sa_edgedist);

   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor
   m_select_show(0,0);
   return;
}


int edgecalc_dialog_compl(zdialog *zd, int zstat)                          //  respond to user cancel
{
   sa_edgecalc_kill = 1;
   return 0;
}


void * edgecalc_wthread(void *arg)                                         //  worker thread function
{
   void  edgecalc_onepix(int px, int py);

   int      index = *((int *) (arg));
   int      ii, kk, nn, px1, py1;
   int64    seed = index + time(0);
   
   for (nn = 0; nn < 0.1 * sa_Npixel; nn++)                                //  do a random 10% of pixels
   {
      ii = drandz(&seed) * sa_Npixel;
      if (! sa_pixel[ii].dist) continue;                                   //  ignore edge pixel
      px1 = sa_pixel[ii].px;
      py1 = sa_pixel[ii].py;
      kk = py1 * Fww + px1;
      if (sa_edgedist[kk]) continue;                                       //  already calculated
      edgecalc_onepix(px1,py1);
      if (sa_edgecalc_kill) goto byebye;
   }

   for (ii = index; ii < sa_Npixel; ii += NWthreads)                       //  do all pixels
   {
      if (! sa_pixel[ii].dist) continue;
      px1 = sa_pixel[ii].px;
      py1 = sa_pixel[ii].py;
      kk = py1 * Fww + px1;
      if (sa_edgedist[kk]) continue;
      edgecalc_onepix(px1,py1);
      if (sa_edgecalc_kill) goto byebye;
   }

   for (ii = index; ii < sa_Npixel; ii += NWthreads)                       //  copy data from sa_edgedist[]
   {                                                                       //    to sa_pixel[].dist
      px1 = sa_pixel[ii].px;
      py1 = sa_pixel[ii].py;
      kk = py1 * Fww + px1;
      sa_pixel[ii].dist = sa_edgedist[kk];
   }

byebye:
   zadd_locked(sa_edgecalc_busy,-1);
   pthread_exit(0);
}


void edgecalc_onepix(int px1, int py1)                                     //  calculate 1 pixel
{
   int      ii, px2, py2;
   int      dist, mindist;
   int      epx, epy, pxm, pym, dx, dy, inc;
   double   slope;

   mindist = 99999;
   epx = epy = 0;

   for (ii = 0; ii < sa_Npixel; ii++)
   {
      if (sa_pixel[ii].dist) continue;                                     //  find all edge pixels
      px2 = sa_pixel[ii].px;
      py2 = sa_pixel[ii].py;                                               //  calculate distance to edge pixel
      dx = px2 - px1;
      dy = py2 - py1;
      dist = sqrt(dx*dx + dy*dy) + 0.5;
      if (dist < mindist) {
         mindist = dist;                                                   //  remember minimum
         epx = px2;                                                        //  remember nearest edge pixel
         epy = py2;
      }
   }
   
   if (abs(epy - py1) > abs(epx - px1)) {                                  //  find all in-between pixels
      slope = 1.0 * (epx - px1) / (epy - py1);                             //  pixels along same line need not
      if (epy > py1) inc = 1;                                              //    search for closest edge pixel
      else inc = -1;                                                       //      which is a huge speedup
      for (pym = py1; pym != epy; pym += inc) {
         pxm = px1 + slope * (pym - py1);
         ii = pym * Fww + pxm;
         if (sa_edgedist[ii]) continue;
         dx = epx - pxm;                                                   //  calculate distance to edge
         dy = epy - pym;
         dist = sqrt(dx*dx + dy*dy) + 0.5;
         sa_edgedist[ii] = dist;                                           //  save
         sa_edgecalc_pixdone++;
      }
   }

   else {
      slope = 1.0 * (epy - py1) / (epx - px1);
      if (epx > px1) inc = 1;
      else inc = -1;
      for (pxm = px1; pxm != epx; pxm += inc) {
         pym = py1 + slope * (pxm - px1);
         ii = pym * Fww + pxm;
         if (sa_edgedist[ii]) continue;
         dx = epx - pxm;
         dy = epy - pym;
         dist = sqrt(dx*dx + dy*dy) + 0.5;
         sa_edgedist[ii] = dist;                                           //  save
         sa_edgecalc_pixdone++;
      }
   }

   return;
}


//  invert a selected area

void m_select_invert(GtkWidget *, const char *)                            //  v.8.7
{
   int      ii, jj, kk, npix, cc, px, py;

   m_select_finish(0,0);                                                   //  finish if needed
   if (! sa_Npixel) return;                                                //  no finished area

   for (ii = kk = 0; ii < Fww * Fhh; ii++)                                 //  count inside pixels    v.8.7
      if (sa_pixisin[ii] == 2) kk++;

   zfree(sa_pixel);                                                        //  free old select area
   npix = Fww * Fhh - kk;
   cc = npix * sizeof(sa_pixel1);                                          //  allocate new select area
   sa_pixel = (sa_pixel1 *) zmalloc(cc);                                   //    image size - inside pixels   v.8.7

   for (ii = kk = 0; ii < Fww * Fhh; ii++)                                 //  reverse member pixels
   {
      jj = sa_pixisin[ii];
      if (jj == 2) sa_pixisin[ii] = 0;                                     //  inside >> outside
      else {
         sa_pixisin[ii] = 2 - jj;                                          //  outside >> inside, edge >> edge
         py = ii / Fww;
         px = ii - Fww * py;
         sa_pixel[kk].px = px;
         sa_pixel[kk].py = py;
         sa_pixel[kk].dist = 1 - jj;                                       //  dist = 0/1 = edge/inside
         kk++;
      }
   }

   if (npix != kk) zappcrash("invert area bug: %d %d",kk,npix);
   sa_Npixel = npix;
   sa_calced = 0;                                                          //  edge calculation missing
   m_select_show(0,0);
   return;
}


//  disable image select area
//  sa_pixseq[] and sa_currseq are preserved
//  set sa_stat = 1 restores edit capability

void m_select_disable(GtkWidget *, const char *)
{
   if (! sa_pixseq) return;
   sa_Npixel = sa_blend = sa_calced = sa_Ncurrseq = 0;
   sa_stat = 2;
   if (sa_pixel) zfree(sa_pixel);
   if (sa_pixisin) zfree(sa_pixisin);
   sa_pixel = 0;
   sa_pixisin = 0;
   mwpaint2();
   return;
}


//  clear selected image area, free memory

void m_select_delete(GtkWidget *, const char *)                            //  v.8.7
{
   if (! sa_pixseq) return;
   if (sa_currseq) {
      m_select_show(0,0);
      int yn = zmessageYN(Bdeletearea);                                    //  get user confirmation
      if (! yn) return;
   }
   select_delete();
   return;
}


//  callable version without user query

void select_delete()
{
   sa_type = sa_stat = sa_Npixel = sa_blend = sa_calced = 0;
   sa_currseq = sa_Ncurrseq = 0;
   if (sa_pixel) zfree(sa_pixel);
   if (sa_pixisin) zfree(sa_pixisin);
   if (sa_pixseq) zfree(sa_pixseq);
   if (sa_stackii) zfree(sa_stackii);
   if (sa_stackdir) zfree(sa_stackdir);
   sa_pixel = 0;
   sa_pixisin = 0;
   sa_pixseq = 0;
   sa_stackii = 0;
   sa_stackdir = 0;
   mwpaint2();
   return;
}


//  draw one pixel only if not already drawn

void select_drawpix1(int px, int py)
{
   int ii = Fww * py + px;
   if (sa_pixseq[ii]) return;
   sa_pixseq[ii] = sa_currseq;
   select_drawpix2(px,py);
   return;
}


//  draw one pixel using black or white depending on background brightness

void select_drawpix2(int px, int py)
{
   int         qx, qy, bright;
   uint8       *ppix24;

   ppix24 = (uint8 *) Frgb24->bmp + (py * Fww + px) * 3;                   //  chose white or black depending
   bright = brightness(ppix24);                                            //    on brightness of background
   if (bright < 128) gdk_gc_set_foreground(gdkgc,&white);

   qx = Mscale * (px-Iorgx) + 0.5;                                         //  image to window space
   qy = Mscale * (py-Iorgy) + 0.5;
   gdk_draw_point(drWin->window,gdkgc,qx+Dorgx,qy+Dorgy);                  //  draw pixel

   gdk_gc_set_foreground(gdkgc,&black);
   return;
}


/**************************************************************************
   select area copy and paste functions         v.8.7
***************************************************************************/

RGB      *SArgb48 = 0;                                                     //  select area pixmap storage
int      SAww, SAhh;                                                       //  dimensions
int      SAorgx, SAorgy;                                                   //  origin in paste-to image


//  copy selected area, save in memory

void m_select_copy(GtkWidget *, const char *)                              //  menu function
{
   int      ii, px, py;
   int      pxmin, pxmax, pymin, pymax;
   uint16   *pix1, *pix2;
   
   m_select_finish(0,0);                                                   //  finish area if not already
   if (! sa_Npixel) return;
   
   pxmin = Fww;
   pxmax = 0;
   pymin = Fhh;
   pymax = 0;
   
   for (ii = 0; ii < sa_Npixel; ii++)                                      //  find enclosing rectangle
   {
      px = sa_pixel[ii].px;
      py = sa_pixel[ii].py;
      if (px > pxmax) pxmax = px;
      if (px < pxmin) pxmin = px;
      if (py > pymax) pymax = py;
      if (py < pymin) pymin = py;
   }
   
   RGB_free(SArgb48);
   SAww = pxmax - pxmin + 1;                                               //  create new RGB pixmap for area
   SAhh = pymax - pymin + 1;
   SArgb48 = RGB_make(SAww,SAhh,48);
   
   for (py = 0; py < SAhh; py++)                                           //  clear to black (transparent)
   for (px = 0; px < SAww; px++)
   {
      pix1 = bmpixel(SArgb48,px,py);
      pix1[0] = pix1[1] = pix1[2] = 0;
   }
   
   for (ii = 0; ii < sa_Npixel; ii++)                                      //  copy selected area to new pixmap
   {
      px = sa_pixel[ii].px;
      py = sa_pixel[ii].py;
      pix1 = bmpixel(Frgb48,px,py);
      px = px - pxmin;
      py = py - pymin;
      pix2 = bmpixel(SArgb48,px,py);
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
      pix2[2] = pix1[2];
      if (pix2[2] == 0) pix2[2] = 1;                                       //  disallow blue=0 pixel
   }
   
   return;
}


//  paste selected area into current image
//  this is an edit function - select area image is copied into main image

void m_select_paste(GtkWidget *, const char *)                             //  menu function
{
   int      select_paste_dialog_compl(zdialog *, int zstat);
   void     select_paste_mousefunc();
   
   const char  *dragmess = ZTX("position image via mouse drag");
   
   if (! SArgb48) return;                                                  //  nothing to paste

   if (! edit_setup(0,0)) return;                                          //  setup edit
   
   SAorgx = (E3ww - SAww) / 2;
   SAorgy = (E3hh - SAhh) / 2;                                             //  copy area to middle of image
   if (SAorgx < 0) SAorgx = 0;
   if (SAorgy < 0) SAorgy = 0;
   select_paste_copy();
   
   zdedit = zdialog_new(ZTX("Paste Image"),mWin,Bdone,Bcancel,null);       //  start dialog
   zdialog_add_widget(zdedit,"label","lab1","dialog",dragmess,"space=8");
   zdialog_run(zdedit,0,select_paste_dialog_compl);

   mouseCBfunc = select_paste_mousefunc;                                   //  connect mouse function
   Mcapture = 1;
   gdk_window_set_cursor(drWin->window,0);                                 //  set normal cursor
   return;
}


//  dialog completion function
//  Commit edited image (with pasted pixels) and set up a new select area 
//  matching the pasted pixels, which allows further editing of the area.

int select_paste_dialog_compl(zdialog *zd, int zstat)
{
   int      cc, ii;
   int      px1, py1, px2, py2;
   uint16   *pix1;

   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;

   if (zstat != 1) {                                                       //  cancel paste
      edit_cancel();                                                       //  cancel edit, restore image
      return 0;
   }

   edit_done();                                                            //  commit the edit (pasted image)
   
   if (sa_pixseq) zfree(sa_pixseq);                                        //  allocate sa_pixseq[]
   cc = 2 * Fww * Fhh;
   sa_pixseq = (uint16 *) zmalloc(cc);
   memset(sa_pixseq,0,cc);
   
   for (py1 = 0; py1 < SAhh; py1++)                                        //  map non-transparent pixels
   for (px1 = 0; px1 < SAww; px1++)                                        //    into sa_pixseq[]
   {
      pix1 = bmpixel(SArgb48,px1,py1);                                     //  result is equivalent to a
      if (! pix1[2]) continue;                                             //    1-click select-color area
      px2 = px1 + SAorgx;
      py2 = py1 + SAorgy;
      if (px2 < 0 || px2 >= Fww) continue;                                 //  parts may be beyond edges
      if (py2 < 0 || py2 >= Fhh) continue;
      ii = py2 * Fww + px2;
      sa_pixseq[ii] = 1;
   }
   
   sa_type = 2;
   sac_finish();                                                           //  finish the area
   m_select_show(0,0);
   return 0;
}


//  mouse function - follow mouse drags and move pasted area accordingly

void select_paste_mousefunc()
{
   int            mx1, my1, mx2, my2;
   static int     mdx0, mdy0, mdx1, mdy1;
   
   if (Mxposn > SAorgx && Mxposn < SAorgx + SAww &&                        //  mouse within select area
       Myposn > SAorgy && Myposn < SAorgy + SAhh)
   {
      gdk_window_set_cursor(drWin->window,dragcursor);                     //  set drag cursor

      if (Mxdrag + Mydrag == 0) return;                                    //  no drag underway

      if (Mxdown != mdx0 || Mydown != mdy0) {                              //  new drag initiated
         mdx0 = mdx1 = Mxdown;
         mdy0 = mdy1 = Mydown;
      }
      mx1 = mdx1;                                                          //  drag start
      my1 = mdy1;
      mx2 = Mxdrag;                                                        //  drag position
      my2 = Mydrag;
      mdx1 = mx2;                                                          //  next drag start
      mdy1 = my2;
      
      SAorgx += (mx2 - mx1);                                               //  move position of select area
      SAorgy += (my2 - my1);                                               //    by mouse drag amount
      select_paste_copy();                                                 //  re-copy area to new position

      return;      
   }

   gdk_window_set_cursor(drWin->window,0);                                 //  mouse outside select area
   return;
}


//  copy select area into edit image, starting at SAorgx/y

void select_paste_copy()
{
   int      px1, py1, px2, py2;
   uint16   *pix1, *pix2;

   edit_undo();                                                            //  restore original image

   mutex_lock(&pixmaps_lock);                                              //  lock pixmaps

   for (py1 = 0; py1 < SAhh; py1++)                                        //  copy pixels
   for (px1 = 0; px1 < SAww; px1++)
   {
      px2 = px1 + SAorgx;
      py2 = py1 + SAorgy;
      if (px2 < 0 || px2 >= E3ww) continue;                                //  parts may be beyond edges
      if (py2 < 0 || py2 >= E3hh) continue;
      pix1 = bmpixel(SArgb48,px1,py1);
      pix2 = bmpixel(E3rgb48,px2,py2);
      if (! pix1[2]) continue;                                             //  skip transparent pixels
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
      pix2[2] = pix1[2];
   }

   Fmodified = 1;
   mutex_unlock(&pixmaps_lock);
   mwpaint2();
   return;
}


/**************************************************************************
      begin image edit functions
***************************************************************************/


//  adjust white balance

double   whitebal_red, whitebal_green, whitebal_blue;
int      whitebal_busy = 0;


void m_whitebal(GtkWidget *, const char *)                                 //  v.8.6
{
   void  whitebal_mousefunc();
   int   whitebal_dialog_compl(zdialog* zd, int zstat);
   void  *whitebal_thread(void *);

   const char  *wbtitle = ZTX("Adjust White Balance");
   const char  *wbhelp = ZTX("Click white or gray image location");

   if (! edit_setup(1,2)) return;                                          //  setup edit: preview

   zdedit = zdialog_new(wbtitle,mWin,Bdone,Bcancel,null);                  //  white balance dialog
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","labwbh","hb1",wbhelp,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labpix","hb2","pixel:");
   zdialog_add_widget(zdedit,"label","pixel","hb2","0000 0000");
   zdialog_add_widget(zdedit,"label","labrgb","hb2","   RGB:");
   zdialog_add_widget(zdedit,"label","rgb","hb2","000 000 000");

   zdialog_run(zdedit,0,whitebal_dialog_compl);                            //  run dialog - parallel

   whitebal_red = whitebal_green = whitebal_blue = 1.0;
   start_thread(whitebal_thread,0);                                        //  start working thread

   mouseCBfunc = whitebal_mousefunc;                                       //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks
   gdk_window_set_cursor(drWin->window,0);                                 //  set normal cursor
   return;
}


int whitebal_dialog_compl(zdialog *zd, int zstat)                          //  dialog completion function
{
   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   if (zstat == 1) edit_done();                                            //  done
   else edit_cancel();                                                     //  cancel or destroy
   return 0;
}


void whitebal_mousefunc()                                                  //  mouse function
{
   int         px, py, dx, dy;
   double      red, green, blue, rgbmean;
   char        work[40];
   uint16      *ppix48;
   
   if (! LMclick) return;
   
   LMclick = 0;
   px = Mxclick;                                                           //  mouse click position
   py = Myclick;
   
   if (px < 2) px = 2;                                                     //  pull back from edge
   if (px > E3ww-3) px = E3ww-3;
   if (py < 2) py = 2;
   if (py > E3hh-3) py = E3hh-3;
   
   red = green = blue = 0;

   for (dy = -2; dy <= 2; dy++)                                            //  5x5 block around mouse position
   for (dx = -2; dx <= 2; dx++)
   {
      ppix48 = bmpixel(E1rgb48,px+dx,py+dy);                               //  input image
      red += ppix48[0];
      green += ppix48[1];
      blue += ppix48[2];
   }
   
   red = red / 25.0;                                                       //  mean RGB levels 
   green = green / 25.0;
   blue = blue / 25.0;
   rgbmean = (red + green + blue) / 3.0;

   whitebal_red = rgbmean / red;
   whitebal_green = rgbmean / green;
   whitebal_blue = rgbmean / blue;

   signal_thread();                                                        //  trigger image update

   snprintf(work,40,"%d %d",px,py);
   zdialog_stuff(zdedit,"pixel",work);

   snprintf(work,40,"%7.3f %7.3f %7.3f",red/256,green/256,blue/256);
   zdialog_stuff(zdedit,"rgb",work);
   
   return;
}


//  Update image based on neutral pixel that was clicked

void * whitebal_thread(void *)
{
   void * whitebal_wthread(void *arg);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (int ii = 0; ii < NWthreads; ii++)                               //  start worker threads
         start_detached_thread(whitebal_wthread,&wtindex[ii]);
      zadd_locked(whitebal_busy,+NWthreads);
      
      while (whitebal_busy) zsleep(0.01);                                  //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * whitebal_wthread(void *arg)                                         //  worker thread function
{
   void  whitebalpix(int px, int py, int dist);

   int      px, py, ii, dist;
   int      index = *((int *) arg);

   if (sa_Npixel)                                                          //  process selected area
   {
      for (ii = index; ii < sa_Npixel; ii += NWthreads)                    //  process all enclosed pixels
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         dist = sa_pixel[ii].dist;
         whitebalpix(px,py,dist);
      }
   }

   else                                                                    //  process whole image
   {
      dist = sa_blend = 0;
      for (py = index; py < E1hh; py += NWthreads)
      for (px = 0; px < E1ww; px++)
         whitebalpix(px,py,dist);
   }

   zadd_locked(whitebal_busy,-1);
   pthread_exit(0);
}


void whitebalpix(int px, int py, int dist)                                 //  process one pixel
{
   uint16      *pix1, *pix3;
   double      red1, green1, blue1;
   double      red3, green3, blue3;
   double      brmax, dold, dnew;

   pix1 = bmpixel(E1rgb48,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb48,px,py);                                          //  output pixel
   
   red1 = pix1[0];
   green1 = pix1[1];
   blue1 = pix1[2];
   
   red3 = whitebal_red * red1;                                             //  change color ratios
   green3 = whitebal_green * green1;
   blue3 = whitebal_blue * blue1;

   if (dist < sa_blend) {                                                  //  blend select area if req.
      dnew = 1.0 * dist / sa_blend;
      dold = 1.0 - dnew;
      red3 = dnew * red3 + dold * red1;
      green3 = dnew * green3 + dold * green1;
      blue3 = dnew * blue3 + dold * blue1;
   }
   
   brmax = red3;                                                           //  brmax = brightest color
   if (green3 > brmax) brmax = green3;
   if (blue3 > brmax) brmax = blue3;
   
   if (brmax > 65535) {                                                    //  if overflow, reduce
      brmax = 65535 / brmax;
      red3 = red3 * brmax;
      green3 = green3 * brmax;
      blue3 = blue3 * brmax;
   }

   pix3[0] = int(red3);
   pix3[1] = int(green3);
   pix3[2] = int(blue3);
   
   return;
}


/**************************************************************************/

//  flatten brightness distribution

int      flatten_busy = 0;
double   flatten_value = 0;                                                //  flatten value, 0 - 100%
double   flatten_brdist[65536];

void m_flatten(GtkWidget *, const char *)
{
   int    flatten_dialog_event(zdialog* zd, const char *event);
   int    flatten_dialog_compl(zdialog* zd, int zstat);
   void * flatten_thread(void *);

   const char  *title = ZTX("Flatten Brightness Distribution");

   if (! edit_setup(1,2)) return;                                          //  setup edit: preview

   zdedit = zdialog_new(title,mWin,Bundo,Bredo,Bdone,Bcancel,null);        //  flatten distribution dialog
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=15");
   zdialog_add_widget(zdedit,"label","labfd","hb1",ZTX("Flatten"),"space=5");
   zdialog_add_widget(zdedit,"hscale","flatten","hb1","0|100|1|0","expand");

   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,flatten_dialog_event,flatten_dialog_compl);          //  run dialog - parallel
   
   flatten_value = 0;
   start_thread(flatten_thread,0);                                         //  start working thread
   return;
}


//  flatten dialog event and completion functions

int flatten_dialog_event(zdialog *zd, const char *event)                   //  flatten dialog event function
{
   zdialog_fetch(zd,"flatten",flatten_value);                              //  get slider value
   signal_thread();                                                        //  trigger update thread
   return 1;
}


int flatten_dialog_compl(zdialog *zd, int zstat)                           //  flatten dialog completion function
{
   if (zstat == 1) edit_undo();                                            //  undo
   else if (zstat == 2) edit_redo();                                       //  redo
   else if (zstat == 3) edit_done();                                       //  done
   else edit_cancel();                                                     //  cancel or destroy
   return 0;
}


//  thread function - use multiple working threads

void * flatten_thread(void *)
{
   void  * flatten_wthread(void *arg);

   int         px, py, ii;
   double      bright1;
   uint16      *pix1;

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      for (ii = 0; ii < 65536; ii++)                                       //  clear brightness distribution data
         flatten_brdist[ii] = 0;

      if (sa_Npixel)                                                       //  process selected area
      {
         for (ii = 0; ii < sa_Npixel; ii++)                                //  process enclosed pixels
         {
            px = sa_pixel[ii].px;                                          //  compute brightness distribution
            py = sa_pixel[ii].py;
            pix1 = bmpixel(E1rgb48,px,py);
            bright1 = brightness(pix1);
            flatten_brdist[int(bright1)]++;
         }
         
         for (ii = 1; ii < 65536; ii++)                                    //  cumulative brightness distribution
            flatten_brdist[ii] += flatten_brdist[ii-1];                    //   0 ... sa_Npixel

         for (ii = 0; ii < 65536; ii++)
            flatten_brdist[ii] = flatten_brdist[ii]                        //  multiplier per brightness level
                               / sa_Npixel * 65536.0 / (ii + 1);
      }

      else                                                                 //  process whole image
      {
         for (py = 0; py < E1hh; py++)                                     //  compute brightness distribution
         for (px = 0; px < E1ww; px++)
         {
            pix1 = bmpixel(E1rgb48,px,py);
            bright1 = brightness(pix1);
            flatten_brdist[int(bright1)]++;
         }
         
         for (ii = 1; ii < 65536; ii++)                                    //  cumulative brightness distribution
            flatten_brdist[ii] += flatten_brdist[ii-1];                    //   0 ... (ww1 * hh1)

         for (ii = 0; ii < 65536; ii++)
            flatten_brdist[ii] = flatten_brdist[ii]                        //  multiplier per brightness level 
                               / (E1ww * E1hh) * 65536.0 / (ii + 1);
      }
      
      for (ii = 0; ii < NWthreads; ii++)                                   //  start worker threads
         start_detached_thread(flatten_wthread,&wtindex[ii]);
      zadd_locked(flatten_busy,+NWthreads);

      while (flatten_busy) zsleep(0.004);                                  //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * flatten_wthread(void *arg)                                          //  worker thread function
{
   void  flatten_1pix(int px, int py, int dist);
   
   int         index = *((int *) (arg));
   int         px, py, ii, dist;

   if (sa_Npixel)                                                          //  process selected area
   {
      for (ii = index; ii < sa_Npixel; ii += NWthreads)                    //  process all enclosed pixels
      {
         px = sa_pixel[ii].px;                                             //  flatten brightness distribution
         py = sa_pixel[ii].py;
         dist = sa_pixel[ii].dist;
         flatten_1pix(px,py,dist);
      }
   }

   else
   {
      dist = sa_blend = 0;
      for (py = index; py < E1hh; py += NWthreads)                         //  flatten brightness distribution
      for (px = 0; px < E1ww; px++)
         flatten_1pix(px,py,dist);
   }

   zadd_locked(flatten_busy,-1);
   pthread_exit(0);
}


void flatten_1pix(int px, int py, int dist)
{
   uint16      *pix1, *pix3;
   double      fold, fnew, dold, dnew, cmax;
   double      red1, green1, blue1, red3, green3, blue3;
   double      bright1, bright2;

   pix1 = bmpixel(E1rgb48,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb48,px,py);                                          //  output pixel
      
   fnew = 0.01 * flatten_value;                                            //  0.0 - 1.0  how much to flatten
   fold = 1.0 - fnew;                                                      //  1.0 - 0.0  how much to retain

   red1 = pix1[0];
   green1 = pix1[1];
   blue1 = pix1[2];

   bright1 = brightness(pix1);                                             //  input brightness
   bright2 = flatten_brdist[int(bright1)];                                 //  output brightness adjustment

   red3 = bright2 * red1;                                                  //  flattened brightness
   green3 = bright2 * green1;
   blue3 = bright2 * blue1;

   red3 = fnew * red3 + fold * red1;                                       //  blend new and old brightness
   green3 = fnew * green3 + fold * green1;
   blue3 = fnew * blue3 + fold * blue1;

   if (dist < sa_blend) {                                                  //  blend over distance sa_blend
      dnew = 1.0 * dist / sa_blend;
      dold = 1.0 - dnew;
      red3 = dnew * red3 + dold * red1;
      green3 = dnew * green3 + dold * green1;
      blue3 = dnew * blue3 + dold * blue1;
   }

   cmax = red3;                                                            //  stop overflow, keep color balance
   if (green3 > cmax) cmax = green3;
   if (blue3 > cmax) cmax = blue3;
   if (cmax > 65535) {
      cmax = 65535 / cmax;
      red3 = red3 * cmax;
      green3 = green3 * cmax;
      blue3 = blue3 * cmax;
   }

   pix3[0] = int(red3 + 0.5);
   pix3[1] = int(green3 + 0.5);
   pix3[2] = int(blue3 + 0.5);
   return;
}


/**************************************************************************/

//  brightness / color / contrast adjustment

int   tune_curve_adjust(void *,GdkEventButton *);                          //  mouse events in drawing area
int   tune_curve_draw();                                                   //  draw curve in drawing area

GtkWidget   *tune_drawarea;                                                //  drawing area for curve
int         tune_busy;
int         tune_ii;                                                       //  ii = current spline curve
int         tune_nap[7];                                                   //  no. anchor points for 7 curves
double      tune_apx[7][50], tune_apy[7][50];                              //  anchor points for 7 curves
double      tune_dat[7][100];                                              //  data points for 7 curves


void m_tune(GtkWidget *, const char *)                                     //  overhauled  v.6.8
{
   int   tune_dialog_event(zdialog *zd, cchar *event);
   int   tune_dialog_compl(zdialog *zd, int zstat);
   void  *tune_thread(void *);

   const char  *title = ZTX("Adjust Brightness and Color");

   if (! edit_setup(1,2)) return;                                          //  setup edit: preview
   
   for (int ii = 0; ii < 7; ii++)
   {                                                                       //  initz. all curves to flat
      tune_nap[ii] = 3;
      tune_apx[ii][0] = 0;                                                 //  3 anchor points: 
      tune_apy[ii][0] = 50;                                                //    (0,50) (50,50) (100,50)
      tune_apx[ii][1] = 50;
      tune_apy[ii][1] = 50;
      tune_apx[ii][2] = 100;
      tune_apy[ii][2] = 50;
      
      for (int jj = 0; jj < 100; jj++)                                     //  initz. curve data points
         tune_dat[ii][jj] = 50;
   }
   
   tune_ii = 0;                                                            //  default curve = brightness

/***
       --------------------------------------------
      |                                            |
      |           curve drawing area               |
      |                                            |
       --------------------------------------------
       darker areas                   lighter areas

      [+++] [---] [+ -] [- +] [+-+] [-+-]

      (o) brightness
      (o) defog
      (o) color intensity    [reset 1]  [reset all]
      (o) color saturation
      (o) color balance    (o) red  (o) green  (o) blue

***/

   zdedit = zdialog_new(title,mWin,Bundo,Bredo,Bdone,Bcancel,null);        //  create dialog

   zdialog_add_widget(zdedit,"frame","fr1","dialog",0,"expand");
   zdialog_add_widget(zdedit,"hbox","hba","dialog");
   zdialog_add_widget(zdedit,"label","labda","hba",Bdarker,"space=5");
   zdialog_add_widget(zdedit,"label","space","hba",0,"expand");
   zdialog_add_widget(zdedit,"label","labba","hba",Blighter,"space=5");
   zdialog_add_widget(zdedit,"hbox","hbb","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","b +++","hbb","+++");
   zdialog_add_widget(zdedit,"button","b ---","hbb"," - - - ");
   zdialog_add_widget(zdedit,"button","b +-", "hbb"," + -  ");
   zdialog_add_widget(zdedit,"button","b -+", "hbb","  - + ");
   zdialog_add_widget(zdedit,"button","b +-+","hbb","+ - +");
   zdialog_add_widget(zdedit,"button","b -+-","hbb"," - + - ");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog");
   zdialog_add_widget(zdedit,"vbox","vb21","hb2");
   zdialog_add_widget(zdedit,"vbox","vb22","hb2");
   zdialog_add_widget(zdedit,"radio","radbri","vb21",Bbrightness);
   zdialog_add_widget(zdedit,"radio","radfog","vb21",ZTX("defog"));
   zdialog_add_widget(zdedit,"radio","radcol","vb21",ZTX("color intensity"));
   zdialog_add_widget(zdedit,"radio","radsat","vb21",ZTX("color saturation"));
   zdialog_add_widget(zdedit,"radio","radbal","vb21",ZTX("color balance"));
   zdialog_add_widget(zdedit,"hbox","hbrs","vb22",0,"space=5");
   zdialog_add_widget(zdedit,"label","space","hbrs",0,"space=20");
   zdialog_add_widget(zdedit,"button","reset1","hbrs",ZTX(" reset 1 "));
   zdialog_add_widget(zdedit,"button","resetA","hbrs",ZTX("reset all"));
   zdialog_add_widget(zdedit,"label","space","vb22",0,"expand");
   zdialog_add_widget(zdedit,"hbox","hbrgb","vb22");
   zdialog_add_widget(zdedit,"radio","radR","hbrgb",Bred,"space=10");
   zdialog_add_widget(zdedit,"radio","radG","hbrgb",Bgreen,"space=5");
   zdialog_add_widget(zdedit,"radio","radB","hbrgb",Bblue,"space=5");
   
   GtkWidget *frame = zdialog_widget(zdedit,"fr1");                        //  add drawing area to frame
   tune_drawarea = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(frame),tune_drawarea);

   gtk_widget_add_events(tune_drawarea,GDK_BUTTON_PRESS_MASK);             //  connect drawing area events
   gtk_widget_add_events(tune_drawarea,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(tune_drawarea,GDK_BUTTON1_MOTION_MASK); 
   G_SIGNAL(tune_drawarea,"motion-notify-event",tune_curve_adjust,0)
   G_SIGNAL(tune_drawarea,"button-press-event",tune_curve_adjust,0)
   G_SIGNAL(tune_drawarea,"expose-event",tune_curve_draw,0)
   
   zdialog_stuff(zdedit,"radbri",1);                                       //  stuff defaults
   zdialog_stuff(zdedit,"radR",1);
   
   zdialog_resize(zdedit,0,370);
   zdialog_run(zdedit,tune_dialog_event,tune_dialog_compl);                //  run dialog - parallel
   start_thread(tune_thread,0);                                            //  start working thread
   return;
}


//  tune dialog event and completion functions

int tune_dialog_event(zdialog *zd, const char *event)
{
   int      ii, jj, curve = -1;
   double   px, py;
   
   if (strnEqu(event,"rad",3))                                             //  new choice of curve
   {
      ii = strcmpv(event,"radbri","radfog","radcol","radsat","radR","radG","radB",0);
      if (ii > 0) curve = ii - 1;
      
      if (strstr("radR radG radB",event))                                  //  if RGB, set color balance
         zdialog_stuff(zd,"radbal",1);
      
      if (strEqu(event,"radbal")) {                                        //  if color balance,
         zdialog_fetch(zd,"radR",ii);                                      //    get current RGB selection
         if (ii) curve = 4;
         zdialog_fetch(zd,"radG",ii);
         if (ii) curve = 5;
         zdialog_fetch(zd,"radB",ii);
         if (ii) curve = 6;
      }

      if (curve >= 0 && curve != tune_ii) {
         tune_ii = curve;                                                  //  set new curve
         tune_curve_draw();                                                //  redraw curve
      }
   }
   
   if (strnEqu(event,"b ",2))                                              //  button to move entire curve
   {
      ii = tune_ii;
      
      for (jj = 0; jj < tune_nap[ii]; jj++)
      {
         px = tune_apx[ii][jj];
         py = tune_apy[ii][jj];
      
         if (strEqu(event,"b +++")) py += 10;
         if (strEqu(event,"b ---")) py -= 10;
         if (strEqu(event,"b +-"))  py += 10.0 - 0.2 * px;
         if (strEqu(event,"b -+"))  py -= 10.0 - 0.2 * px;
         if (strEqu(event,"b +-+")) py -= 5 - 0.2 * abs(px-50);
         if (strEqu(event,"b -+-")) py += 5 - 0.2 * abs(px-50);

         if (py > 100) py = 100;
         if (py < 0) py = 0;
         tune_apy[ii][jj] = py;
      }

      tune_curve_draw();                                                   //  redraw curve
      signal_thread();                                                     //  trigger image update
   }
   
   if (strEqu(event,"reset1")) 
   {
      ii = tune_ii;                                                        //  current curve
      tune_nap[ii] = 3;
      tune_apx[ii][0] = 0;                                                 //  3 anchor points: 
      tune_apy[ii][0] = 50;                                                //    (0,50) (50,50) (100,50)
      tune_apx[ii][1] = 50;
      tune_apy[ii][1] = 50;
      tune_apx[ii][2] = 100;
      tune_apy[ii][2] = 50;
      
      for (int jj = 0; jj < 100; jj++)                                     //  initz. curve data points
         tune_dat[ii][jj] = 50;
      
      tune_curve_draw();                                                   //  update dialog curve
      signal_thread();                                                     //  update image
   }
   
   if (strEqu(event,"resetA")) 
   {
      for (int ii = 0; ii < 7; ii++)                                       //  do all curves
      {
         tune_nap[ii] = 3;
         tune_apx[ii][0] = 0;
         tune_apy[ii][0] = 50;
         tune_apx[ii][1] = 50;
         tune_apy[ii][1] = 50;
         tune_apx[ii][2] = 100;
         tune_apy[ii][2] = 50;
         
         for (int jj = 0; jj < 100; jj++)
            tune_dat[ii][jj] = 50;
      }
      
      tune_curve_draw();
      signal_thread();
   }
   
   if (strEqu(event,"blendwidth")) signal_thread();                        //  select area blend width change

   return 1;
}


int tune_dialog_compl(zdialog *zd, int zstat)                              //  tune dialog completion function
{
   if (zstat == 1) edit_undo();                                            //  undo
   else if (zstat == 2) edit_redo();                                       //  redo
   else if (zstat == 3) edit_done();                                       //  done
   else edit_cancel();                                                     //  cancel or destroy
   return 0;
}


//  Add, delete, or move anchor points in curve using mouse.

int tune_curve_adjust(void *,GdkEventButton *event)
{
   int         ww, hh, px, py;
   int         kk, ii, jj, newjj, minjj = -1;
   int         mx, my, button, evtype;
   double      dist2, mindist2 = 1000000;
   double      xval, yval;
   
   ii = tune_ii;                                                           //  get current curve

   if (tune_nap[ii] > 49) {
      zmessageACK(ZTX("Exceed 50 anchor points"));
      return 0;
   }
   
   mx = int(event->x);                                                     //  mouse position in drawing area
   my = int(event->y);
   evtype = event->type;
   button = event->button;
   ww = tune_drawarea->allocation.width;                                   //  drawing area size
   hh = tune_drawarea->allocation.height;
   
   for (jj = 0; jj < tune_nap[ii]; jj++)                                   //  find closest anchor point
   {
      xval = tune_apx[ii][jj];
      yval = spline2(xval);
      px = int(0.01 * ww * xval);                                          //  0 - ww
      py = int(hh - 0.01 * hh * yval + 0.5);                               //  0 - hh
      dist2 = (px-mx)*(px-mx) + (py-my)*(py-my);
      if (dist2 < mindist2) {
         mindist2 = dist2;
         minjj = jj;
      }
   }

   if (minjj < 0) return 0;                                                //  huh?
   
   if (evtype == GDK_BUTTON_PRESS && button == 3) {                        //  right click, remove anchor point
      if (mindist2 > 25) return 0;
      if (tune_nap[ii] < 3) return 0;
      for (kk = minjj; kk < tune_nap[ii] -1; kk++) {
         tune_apx[ii][kk] = tune_apx[ii][kk+1];
         tune_apy[ii][kk] = tune_apy[ii][kk+1];
      }
      tune_nap[ii]--;

      tune_curve_draw();                                                   //  regen and redraw curve
      signal_thread();                                                     //  trigger image update
      return 0;
   }

//  drag or left click, move nearby anchor point to mouse position,
//  or add a new anchor point if nothing near enough

   xval = 100.0 * mx / ww;                                                 //  0 - 100
   yval = 100.0 * (hh - my) / hh;                                          //  0 - 100

   if (xval < 0 || xval > 100) return 0;                                   //  v.6.8
   if (yval < 0 || yval > 100) return 0;

   if (mindist2 < 100) {                                                   //  existing point < 10 pixels away
      jj = minjj;
      if (jj < tune_nap[ii] - 1 && tune_apx[ii][jj+1] - xval < 5)          //  disallow < 5 x-pixels
         return 0;                                                         //    to next or prior point
      if (jj > 0 && xval - tune_apx[ii][jj-1] < 5) return 0;
      newjj = minjj;                                                       //  point to be moved
   }
   else                                                                    //  > 10 pixels, add a point
   {
      for (jj = 0; jj < tune_nap[ii]; jj++)
         if (xval <= tune_apx[ii][jj]) break;                              //  find point with next higher x

      if (jj < tune_nap[ii] && tune_apx[ii][jj] - xval < 5) return 0;      //  disallow < 5 pixels
      if (jj > 0 && xval - tune_apx[ii][jj-1] < 5) return 0;               //    to next or prior point

      for (kk = tune_nap[ii]; kk > jj; kk--) {                             //  make hole for new point
         tune_apx[ii][kk] = tune_apx[ii][kk-1];
         tune_apy[ii][kk] = tune_apy[ii][kk-1];
      }

      tune_nap[ii]++;                                                      //  up point count
      newjj = jj;                                                          //  point to be added
   }

   tune_apx[ii][newjj] = xval;                                             //  coordinates of new or moved point
   tune_apy[ii][newjj] = yval;
   
   tune_curve_draw();                                                      //  regen and redraw the curve
   signal_thread();                                                        //  trigger image update
   return 0;
}


//  Draw brightness curve based on defined spline anchor points.

int tune_curve_draw()
{
   int         ww, hh, px, py;
   int         ii, jj, jjx, jjy;
   double      xval, yval;
   
   ii = tune_ii;                                                           //  current curve

   ww = tune_drawarea->allocation.width;                                   //  drawing area size
   hh = tune_drawarea->allocation.height;
   if (ww < 50 || hh < 20) return 0;

   spline1(tune_nap[ii],tune_apx[ii],tune_apy[ii]);                        //  make curve fitting anchor points

   gdk_window_clear(tune_drawarea->window);                                //  clear window

   for (px = 0; px < ww; px++)                                             //  generate all points for curve
   {
      xval = 100.0 * px / ww;
      yval = spline2(xval);
      py = int(hh - 0.01 * hh * yval + 0.5);
      gdk_draw_point(tune_drawarea->window,gdkgc,px,py);
   }
   
   for (jj = 0; jj < tune_nap[ii]; jj++)                                   //  draw boxes at anchor points
   {
      xval = tune_apx[ii][jj];
      yval = spline2(xval);
      px = int(0.01 * ww * xval);
      py = int(hh - 0.01 * hh * yval + 0.5);
      for (jjx = -2; jjx < 3; jjx++)
      for (jjy = -2; jjy < 3; jjy++) {
         if (px+jjx < 0 || px+jjx >= ww) continue;
         if (py+jjy < 0 || py+jjy >= hh) continue;
         gdk_draw_point(tune_drawarea->window,gdkgc,px+jjx,py+jjy);
      }
   }

   for (jj = 0; jj < 100; jj++)                                            //  save 100 curve data points
      tune_dat[ii][jj] = spline2(jj);

   return 0;
}


//  Update image based on latest settings of all dialog controls.

void * tune_thread(void *)
{
   void * tune_wthread(void *arg);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (int ii = 0; ii < NWthreads; ii++)                               //  start worker threads
         start_detached_thread(tune_wthread,&wtindex[ii]);
      zadd_locked(tune_busy,+NWthreads);
      
      while (tune_busy) zsleep(0.004);                                     //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * tune_wthread(void *arg)                                             //  worker thread function
{
   void  tune1pix(int px, int py, int dist);

   int      px, py, ii, dist;
   int      index = *((int *) arg);

   if (sa_Npixel)                                                          //  process selected area
   {
      for (ii = index; ii < sa_Npixel; ii += NWthreads)                    //  process all enclosed pixels
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         dist = sa_pixel[ii].dist;
         tune1pix(px,py,dist);
      }
   }

   else                                                                    //  process whole image
   {
      dist = sa_blend = 0;
      for (py = index; py < E1hh; py += NWthreads)
      for (px = 0; px < E1ww; px++)
         tune1pix(px,py,dist);
   }

   zadd_locked(tune_busy,-1);
   pthread_exit(0);
}


void tune1pix(int px, int py, int dist)                                    //  process one pixel
{
   uint16      *pix1, *pix3;
   double      red1, green1, blue1, red3, green3, blue3;
   double      brmin, brmax, brout;
   int         curveindex;

   pix1 = bmpixel(E1rgb48,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb48,px,py);                                          //  output pixel
   
   red1 = red3 = pix1[0];
   green1 = green3 = pix1[1];
   blue1 = blue3 = pix1[2];

   brmax = red1;                                                           //  brmax = brightest color
   if (green1 > brmax) brmax = green1;
   if (blue1 > brmax) brmax = blue1;
   
   curveindex = int(brmax/656);                                            //  index into curve data, 0-99

/* ------------------------------------------------------------------------

      brightness curve values:
           0 = dark
          50 = normal, unchanged
         100 = 200% brightness, clipped
*/

   brout = tune_dat[0][curveindex];                                        //  brightness factor, 0 - 99
   
   if (brout < 49 || brout > 51)
   {
      brout = brout / 50.0;                                                //  0 - 2.0
      if (brout * brmax > 65535.0) brout = 65535.0 / brmax;                //  reduce if necessary
      
      red3 = red3 * brout;                                                 //  apply to all colors
      green3 = green3 * brout;
      blue3 = blue3 * brout;
   }
      
/* ------------------------------------------------------------------------

      defog (whiteness) curve values:                                      //  v.8.2
           0 = zero whiteness
          50 = normal, unchanged
         100 = double whiteness, clipped
*/

   brout = tune_dat[1][curveindex];                                        //  whiteness factor, 0 - 99
   
   if (brout < 49 || brout > 51)
   {
      brmin = red3;                                                        //  brmin = darkest color
      if (green3 < brmin) brmin = green3;
      if (blue3 < brmin) brmin = blue3;

      brout = brout / 50.0 - 1.0;                                          //  range -1 .. +1
      brmin = brmin * brout;                                               //  range -brmin .. +brmin

      if (brmin > 0) 
         if (brmax + brmin > 65535.0) brmin = 65535.0 - brmax;             //  prevent overflow
      
      red3 = red3 + brmin;                                                 //  reduce or add whiteness
      green3 = green3 + brmin;
      blue3 = blue3 + brmin;
   }
      
/* ------------------------------------------------------------------------

      color intensity curve values:
           0 = no color (grey scale)  
          50 = normal, unchanged
         100 = highest color

      50 >> 0: move all RGB values to their mean: (R+G+B)/3
      50 >> 100: increase all RGB values by same factor
      
      In the 2nd case, the movement is greater for darker pixels
*/

   double   red50, green50, blue50, red100, green100, blue100;
   double   rgb0, max50, min50, color, bright, ramper;
   
   brout = tune_dat[2][curveindex];                                        //  brightness factor, 0 - 99
   
   if (brout < 49 || brout > 51)
   {
      red50 = red3;                                                        //  50%  color values (normal)
      green50 = green3;
      blue50 = blue3;
      
      rgb0 = (red50 + green50 + blue50) / 3;                               //  0%  color values (grey scale)
         
      max50 = min50 = red50;
      if (green50 > max50) max50 = green50;                                //  get max/min normal color values
      else if (green50 < min50) min50 = green50;
      if (blue50 > max50) max50 = blue50;
      else if (blue50 < min50) min50 = blue50;
      
      color = (max50 - min50) * 1.0 / (max50 + 1);                         //  gray .. color       0 .. 1
      color = sqrt(color);                                                 //  accelerated curve   0 .. 1
      bright = max50 / 65535.0;                                            //  dark .. bright      0 .. 1
      bright = sqrt(bright);                                               //  accelerated curve   0 .. 1
      ramper = 1 - color + bright * color;                                 //  1 - color * (1 - bright)
      ramper = 1.0 / ramper;                                               //  large if color high and bright low

      red100 = int(red50 * ramper);                                        //  100%  color values (max)
      green100 = int(green50 * ramper);
      blue100 = int(blue50 * ramper);
      
      if (brout < 50) 
      {
         red3 = rgb0 + (brout) * 0.02 * (red50 - rgb0);                    //  compute new color value
         green3 = rgb0 + (brout) * 0.02 * (green50 - rgb0);
         blue3 = rgb0 + (brout) * 0.02 * (blue50 - rgb0);
      }

      if (brout > 50)
      {
         red3 = red50 + (brout - 50) * 0.02 * (red100 - red50);
         green3 = green50 + (brout - 50) * 0.02 * (green100 - green50);
         blue3 = blue50 + (brout - 50) * 0.02 * (blue100 - blue50);
      }
   }

/* ------------------------------------------------------------------------

      color saturation curve values:
           0 = no color saturation (gray scale)
          50 = normal (initial unmodified RGB)
         100 = max. color saturation

      50 >> 0: move all RGB values to their mean: (R+G+B)/3
      50 >> 100: increase RGB spread until one color is 0 or 65535

      In both cases, the average of RGB is not changed.
*/

   double   rinc, ginc, binc, scale;
   double   spread, spread1, spread2;
   int      againlimit = 10;
   
   brout = tune_dat[3][curveindex];                                        //  saturation factor, 0 - 99
   
   if (brout < 49 || brout > 51)
   {
      spread = brout - 50;                                                 //  -50  to  0  to  50
      spread1 = 0.02 * spread + 1;                                         //    0  to  1  to   2
      spread2 = spread1 - 1.0;                                             //   -1  to  0  to   1

      red50 = red3;                                                        //  50%  color values (normal)
      green50 = green3;
      blue50 = blue3;
      
      rgb0 = (red50 + green50 + blue50 + 1) / 3;

      rinc = red50 - rgb0;
      ginc = green50 - rgb0;
      binc = blue50 - rgb0;
      
      scale = 1.0;

   again:
      rinc = scale * rinc;
      ginc = scale * ginc;
      binc = scale * binc;

      red100 = red50 + rinc;
      green100 = green50 + ginc;
      blue100 = blue50 + binc;
      
      if (--againlimit > 0)                                                //  prevent loops   v.8.5.2
      {
         if (red100 > 65535) { scale = (65535.0 - red50) / rinc;  goto again; }
         if (red100 < 0) { scale = -1.0 * red50 / rinc; goto again; }
         if (green100 > 65535) { scale = (65535.0 - green50) / ginc; goto again; }
         if (green100 < 0) { scale = -1.0 * green50 / ginc; goto again; }
         if (blue100 > 65535) { scale = (65535.0 - blue50) / binc; goto again; }
         if (blue100 < 0) { scale = -1.0 * blue50 / binc; goto again; }
      }
      
      if (spread < 0) {                                                    //  make mid-scale == original RGB
         red3 = int(rgb0 + spread1 * (red50 - rgb0));
         green3 = int(rgb0 + spread1 * (green50 - rgb0));
         blue3 = int(rgb0 + spread1 * (blue50 - rgb0));
      }
      else {
         red3 = int(red50 + spread2 * (red100 - red50));
         green3 = int(green50 + spread2 * (green100 - green50));
         blue3 = int(blue50 + spread2 * (blue100 - blue50));
      }
   }

/* ------------------------------------------------------------------------

      color balance curve values:
           0 = 0.5 * original color
          50 = unmodified
         100 = 1.5 * original color, clipped
*/

   brout = tune_dat[4][curveindex];
   if (brout < 49 || brout > 51) red3 = red3 * 0.01 * (brout + 50);
   brout = tune_dat[5][curveindex];
   if (brout < 49 || brout > 51) green3 = green3 * 0.01 * (brout + 50);
   brout = tune_dat[6][curveindex];
   if (brout < 49 || brout > 51) blue3 = blue3 * 0.01 * (brout + 50);


/* ------------------------------------------------------------------------
   
   if working within a select area, blend changes over distance from edge

*/
      
   double      dold, dnew;

   if (dist < sa_blend) {
      dnew = 1.0 * dist / sa_blend;
      dold = 1.0 - dnew;
      red3 = dnew * red3 + dold * red1;
      green3 = dnew * green3 + dold * green1;
      blue3 = dnew * blue3 + dold * blue1;
   }
   
   if (red3 > 65535) red3 = 65535;                                         //  clip overflows
   if (green3 > 65535) green3 = 65535;
   if (blue3 > 65535) blue3 = 65535;

   pix3[0] = int(red3);
   pix3[1] = int(green3);
   pix3[2] = int(blue3);
   
   return;
}


/**************************************************************************/

//  red eye removal function

struct sredmem {                                                           //  red-eye struct in memory
   char        type, space[3];
   int         cx, cy, ww, hh, rad, clicks;
   double      thresh, tstep;
};
sredmem  redmem[100];                                                      //  store up to 100 red-eyes

int      Nredmem = 0, maxredmem = 100;


void m_redeye(GtkWidget *, const char *)
{
   void     redeye_mousefunc();
   int      redeye_dialog_compl(zdialog *zd, int zstat);

   const char  *redeye_message 
         = ZTX("Method 1:\n"
               "  Left-click on red-eye to darken.\n"
               "Method 2:\n"
               "  Drag down and right to enclose red-eye.\n"
               "  Left-click on red-eye to darken.\n"
               "Undo red-eye:\n"
               "  Right-click on red-eye.");

   if (! edit_setup(0,1)) return;                                          //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Red Eye Reduction"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",redeye_message);
   zdialog_run(zdedit,0,redeye_dialog_compl);                              //  run dialog, parallel mode

   Nredmem = 0;
   mouseCBfunc = redeye_mousefunc;                                         //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks
   gdk_window_set_cursor(drWin->window,0);                                 //  set normal cursor
   return;
}


//  dialog completion callback function

int redeye_dialog_compl(zdialog *zd, int zstat)
{
   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   if (Nredmem > 0) Fmodified = 1;
   if (zstat == 1) edit_done();
   else edit_cancel();
   toparc = ptoparc = 0;
   return 0;
}


//  mouse functions to define, darken, and undo red-eyes

int      redeye_createF(int px, int py);                                   //  create 1-click red-eye (type F)
int      redeye_createR(int px, int py, int ww, int hh);                   //  create robust red-eye (type R)
void     redeye_darken(int ii);                                            //  darken red-eye
void     redeye_distr(int ii);                                             //  build pixel redness distribution
int      redeye_find(int px, int py);                                      //  find red-eye at mouse position
void     redeye_remove(int ii);                                            //  remove red-eye at mouse position
int      redeye_radlim(int cx, int cy);                                    //  compute red-eye radius limit


void redeye_mousefunc()
{
   int         ii, px, py, ww, hh;

   if (Nredmem == maxredmem) {
      zmessageACK("%d red-eye limit reached",maxredmem);                   //  too many red-eyes
      return;
   }

   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;

      px = Mxclick;                                                        //  click position
      py = Myclick;
      if (px < 0 || px > E3ww-1 || py < 0 || py > E3hh-1) return;          //  outside image area

      ii = redeye_find(px,py);                                             //  find existing red-eye
      if (ii < 0) ii = redeye_createF(px,py);                              //  or create new type F
      redeye_darken(ii);                                                   //  darken red-eye
   }
   
   if (RMclick)                                                            //  right mouse click
   {
      RMclick = 0;
      px = Mxclick;                                                        //  click position
      py = Myclick;
      ii = redeye_find(px,py);                                             //  find red-eye
      if (ii >= 0) redeye_remove(ii);                                      //  if found, remove
   }

   if (Mxdrag || Mydrag)                                                   //  mouse drag underway
   {
      px = Mxdown;                                                         //  initial position
      py = Mydown;
      ww = Mxdrag - Mxdown;                                                //  increment
      hh = Mydrag - Mydown;
      if (ww < 2 && hh < 2) return;
      if (ww < 2) ww = 2;
      if (hh < 2) hh = 2;
      if (px < 1) px = 1;                                                  //  keep within image area
      if (py < 1) py = 1;      
      if (px + ww > E3ww-1) ww = E3ww-1 - px;
      if (py + hh > E3hh-1) hh = E3hh-1 - py;
      ii = redeye_find(px,py);                                             //  find existing red-eye
      if (ii >= 0) redeye_remove(ii);                                      //  remove it
      ii = redeye_createR(px,py,ww,hh);                                    //  create new red-eye type R
   }

   mwpaint2();
   return;
}


//  create type F redeye (1-click automatic)

int redeye_createF(int cx, int cy)
{
   int         cx0, cy0, cx1, cy1, px, py, rad, radlim;
   int         loops, ii;
   int         Tnpix, Rnpix, R2npix;
   double      rd, rcx, rcy, redpart;
   double      Tsum, Rsum, R2sum, Tavg, Ravg, R2avg;
   double      sumx, sumy, sumr;
   uint16      *ppix;
   
   cx0 = cx;
   cy0 = cy;
   
   for (loops = 0; loops < 8; loops++)
   {
      cx1 = cx;
      cy1 = cy;

      radlim = redeye_radlim(cx,cy);                                       //  radius limit (image edge)
      Tsum = Tavg = Ravg = Tnpix = 0;

      for (rad = 0; rad < radlim-2; rad++)                                 //  find red-eye radius from (cx,cy)
      {
         Rsum = Rnpix = 0;
         R2sum = R2npix = 0;

         for (py = cy-rad-2; py <= cy+rad+2; py++)
         for (px = cx-rad-2; px <= cx+rad+2; px++)
         {
            rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
            ppix = bmpixel(E3rgb48,px,py);
            redpart = redness(ppix);

            if (rd <= rad + 0.5 && rd > rad - 0.5) {                       //  accum. redness at rad
               Rsum += redpart;
               Rnpix++;
            }
            else if (rd <= rad + 2.5 && rd > rad + 1.5) {                  //  accum. redness at rad+2
               R2sum += redpart;
               R2npix++;
            }
         }
         
         Tsum += Rsum;
         Tnpix += Rnpix;
         Tavg = Tsum / Tnpix;                                              //  avg. redness over 0-rad
         Ravg = Rsum / Rnpix;                                              //  avg. redness at rad
         R2avg = R2sum / R2npix;                                           //  avg. redness at rad+2
         if (R2avg > Ravg || Ravg > Tavg) continue;
         if ((Ravg - R2avg) < 0.2 * (Tavg - Ravg)) break;                  //  0.1 --> 0.2      v.8.6
      }
      
      sumx = sumy = sumr = 0;
      rad = int(1.2 * rad + 1);
      if (rad > radlim) rad = radlim;
      
      for (py = cy-rad; py <= cy+rad; py++)                                //  compute center of gravity for
      for (px = cx-rad; px <= cx+rad; px++)                                //   pixels within rad of (cx,cy)
      {
         rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (rd > rad + 0.5) continue;
         ppix = bmpixel(E3rgb48,px,py);
         redpart = redness(ppix);                                          //  weight by redness    v.8.6
         sumx += redpart * (px - cx);
         sumy += redpart * (py - cy);
         sumr += redpart;
      }

      rcx = cx + 1.0 * sumx / sumr;                                        //  new center of red-eye
      rcy = cy + 1.0 * sumy / sumr;
      if (fabs(cx0 - rcx) > 0.6 * rad) break;                              //  give up if big movement
      if (fabs(cy0 - rcy) > 0.6 * rad) break;
      cx = int(rcx + 0.5);
      cy = int(rcy + 0.5);
      if (cx == cx1 && cy == cy1) break;                                   //  done if no change
   }

   radlim = redeye_radlim(cx,cy);
   if (rad > radlim) rad = radlim;

   ii = Nredmem++;                                                         //  add red-eye to memory
   redmem[ii].type = 'F';
   redmem[ii].cx = cx;
   redmem[ii].cy = cy;
   redmem[ii].rad = rad;
   redmem[ii].clicks = 0;
   redmem[ii].thresh = 0;
   return ii;
}


//  create type R red-eye (drag an ellipse over red-eye area)

int redeye_createR(int cx, int cy, int ww, int hh)
{
   int      rad, radlim;

   toparc = 1;                                                             //  paint ellipse over image
   toparcx = cx - ww;                                                      //  v.8.3
   toparcy = cy - hh;
   toparcw = 2 * ww;
   toparch = 2 * hh;

   if (ww > hh) rad = ww;
   else rad = hh;
   radlim = redeye_radlim(cx,cy);
   if (rad > radlim) rad = radlim;

   int ii = Nredmem++;                                                     //  add red-eye to memory
   redmem[ii].type = 'R';
   redmem[ii].cx = cx;
   redmem[ii].cy = cy;
   redmem[ii].ww = 2 * ww;
   redmem[ii].hh = 2 * hh;
   redmem[ii].rad = rad;
   redmem[ii].clicks = 0;
   redmem[ii].thresh = 0;
   return ii;
}


//  darken a red-eye and increase click count

void redeye_darken(int ii)
{
   int         cx, cy, ww, hh, px, py, rad, clicks;
   double      rd, thresh, tstep;
   char        type;
   uint16      *ppix;

   type = redmem[ii].type;
   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   ww = redmem[ii].ww;
   hh = redmem[ii].hh;
   rad = redmem[ii].rad;
   thresh = redmem[ii].thresh;
   tstep = redmem[ii].tstep;
   clicks = redmem[ii].clicks++;
   
   if (thresh == 0)                                                        //  1st click 
   {
      redeye_distr(ii);                                                    //  get pixel redness distribution
      thresh = redmem[ii].thresh;                                          //  initial redness threshhold
      tstep = redmem[ii].tstep;                                            //  redness step size
      toparc = 0;
   }

   tstep = (thresh - tstep) / thresh;                                      //  convert to reduction factor
   thresh = thresh * pow(tstep,clicks);                                    //  reduce threshhold by total clicks

   for (py = cy-rad; py <= cy+rad; py++)                                   //  darken pixels over threshhold
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = bmpixel(E3rgb48,px,py);                                       //  set redness = threshhold
      if (redness(ppix) > thresh)
         ppix[0] = int(thresh * (0.65 * ppix[1] + 0.10 * ppix[2] + 1) / (25 - 0.25 * thresh));
   }

   return;
}


//  Build a distribution of redness for a red-eye. Use this information 
//  to set initial threshhold and step size for stepwise darkening.

void redeye_distr(int ii)
{
   int         cx, cy, ww, hh, rad, px, py;
   int         bin, npix, dbins[20], bsum, blim;
   double      rd, maxred, minred, redpart, dbase, dstep;
   char        type;
   uint16      *ppix;
   
   type = redmem[ii].type;
   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   ww = redmem[ii].ww;
   hh = redmem[ii].hh;
   rad = redmem[ii].rad;
   
   maxred = 0;
   minred = 100;

   for (py = cy-rad; py <= cy+rad; py++)
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = bmpixel(E3rgb48,px,py);
      redpart = redness(ppix);
      if (redpart > maxred) maxred = redpart;
      if (redpart < minred) minred = redpart;
   }
   
   dbase = minred;
   dstep = (maxred - minred) / 19.99;

   for (bin = 0; bin < 20; bin++) dbins[bin] = 0;
   npix = 0;

   for (py = cy-rad; py <= cy+rad; py++)
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = bmpixel(E3rgb48,px,py);
      redpart = redness(ppix);
      bin = int((redpart - dbase) / dstep);
      ++dbins[bin];
      ++npix;
   }
   
   bsum = 0;
   blim = int(0.5 * npix);

   for (bin = 0; bin < 20; bin++)                                          //  find redness level for 50% of
   {                                                                       //    pixels within red-eye radius
      bsum += dbins[bin];
      if (bsum > blim) break;
   }

   redmem[ii].thresh = dbase + dstep * bin;                                //  initial redness threshhold
   redmem[ii].tstep = dstep;                                               //  redness step (5% of range)   v.6.9

   return;
}


//  find a red-eye (nearly) overlapping the mouse click position

int redeye_find(int cx, int cy)
{
   for (int ii = 0; ii < Nredmem; ii++)
   {
      if (cx > redmem[ii].cx - 2 * redmem[ii].rad && 
          cx < redmem[ii].cx + 2 * redmem[ii].rad &&
          cy > redmem[ii].cy - 2 * redmem[ii].rad && 
          cy < redmem[ii].cy + 2 * redmem[ii].rad) 
            return ii;                                                     //  found
   }
   return -1;                                                              //  not found
}


//  remove a red-eye from memory

void redeye_remove(int ii)
{
   int      cx, cy, rad, px, py;
   uint16   *pix1, *pix3;

   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   rad = redmem[ii].rad;

   for (px = cx-rad; px <= cx+rad; px++)
   for (py = cy-rad; py <= cy+rad; py++)
   {
      pix1 = bmpixel(E1rgb48,px,py);
      pix3 = bmpixel(E3rgb48,px,py);
      pix3[0] = pix1[0];
      pix3[1] = pix1[1];
      pix3[2] = pix1[2];
   }
   
   for (ii++; ii < Nredmem; ii++) 
      redmem[ii-1] = redmem[ii];
   Nredmem--;
   
   toparc = 0;
   return;
}


//  compute red-eye radius limit: smaller of 100 and nearest image edge

int redeye_radlim(int cx, int cy)
{
   int radlim = 100;
   if (cx < 100) radlim = cx;
   if (E3ww-1 - cx < 100) radlim = E3ww-1 - cx;
   if (cy < 100) radlim = cy;
   if (E3hh-1 - cy < 100) radlim = E3hh-1 - cy;
   return radlim;
}


/**************************************************************************/

//  image blur function 

int         blur_busy = 0;
int         blur_radius;
double      blur_weight[100][100];                                         //  up to blur radius = 99   v.6.3
int         blur_Npixels, blur_pixdone;


void m_blur(GtkWidget *, const char *)
{
   int    blur_dialog_event(zdialog *zd, const char *event);
   int    blur_dialog_compl(zdialog *zd, int zstat);
   void * blur_thread(void *);

   if (! edit_setup(0,2)) return;                                          //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Set Blur Radius"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","labrad","hb2",ZTX("blur radius"),"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","0|99|1|1","space=5");
   zdialog_add_widget(zdedit,"button","apply","hb2",Bapply,"space=5");

   zdialog_run(zdedit,blur_dialog_event,blur_dialog_compl);                //  start dialog
   
   blur_radius = 1;
   start_thread(blur_thread,0);                                            //  start working thread
   return;
}


//  blur dialog event and completion callback functions

int blur_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) edit_done();                                            //  done
   else edit_cancel();                                                     //  cancel or destroy
   return 0;
}


int blur_dialog_event(zdialog * zd, const char *event)
{
   if (strNeq(event,"apply")) return 0;

   zdialog_fetch(zd,"radius",blur_radius);                                 //  get blur radius

   if (blur_radius == 0) {
      if (Fmodified) edit_undo();                                          //  restore original image
      Fmodified = 0;
      return 0;
   }

   signal_thread();                                                        //  trigger working thread
   wait_thread_idle();                                                     //  wait for completion
   mwpaint2(); 
   return 1;
}


//  image blur thread function

void * blur_thread(void *)
{
   void * blur_wthread(void *arg);

   int      dx, dy, rad, rad2;
   double   m, d, w, sum;
  
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      rad = blur_radius;
      rad2 = rad * rad;

      for (dx = 0; dx <= rad; dx++)                                        //  clear weights array
      for (dy = 0; dy <= rad; dy++)
         blur_weight[dx][dy] = 0;

      for (dx = -rad; dx <= rad; dx++)                                     //  blur_weight[dx][dy] = no. of pixels
      for (dy = -rad; dy <= rad; dy++)                                     //    at distance (dx,dy) from center
         ++blur_weight[abs(dx)][abs(dy)];

      m = sqrt(rad2 + rad2);                                               //  corner pixel distance from center
      sum = 0;

      for (dx = 0; dx <= rad; dx++)                                        //  compute weight of pixel
      for (dy = 0; dy <= rad; dy++)                                        //    at distance dx, dy
      {
         d = sqrt(dx*dx + dy*dy);
         w = (m + 1 - d) / m;
         w = w * w;
         sum += blur_weight[dx][dy] * w;
         blur_weight[dx][dy] = w;
      }

      for (dx = 0; dx <= rad; dx++)                                        //  make weights add up to 1.0
      for (dy = 0; dy <= rad; dy++)
         blur_weight[dx][dy] = blur_weight[dx][dy] / sum;
         
      if (sa_Npixel) blur_Npixels = sa_Npixel;
      else  blur_Npixels = E3ww * E3hh;
      blur_pixdone = 0;

      for (int ii = 0; ii < NWthreads; ii++)                               //  start worker threads
         start_detached_thread(blur_wthread,&wtindex[ii]);
      zadd_locked(blur_busy,+NWthreads);
 
      while (blur_busy)                                                    //  wait for completion
      {
         zsleep(0.01);
         edit_progress(blur_pixdone,blur_Npixels);                         //  show progress counter
      }

      edit_progress(0,0);
      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * blur_wthread(void *arg)                                             //  worker thread function
{
   void  blur_pixel(int px, int py, int dist);

   int      index = *((int *) arg);
   int      ii, px, py, dist = 0;
   
   if (! sa_Npixel)                                                        //  process entire image
   {
      for (py = index; py < E3hh-1; py += NWthreads)                       //  loop all image pixels
      for (px = 0; px < E3ww-1; px++)
         blur_pixel(px,py,dist);
   }

   if (sa_Npixel)                                                          //  process selected area
   {
      for (ii = index; ii < sa_Npixel; ii += NWthreads)                    //  process all enclosed pixels
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         dist = sa_pixel[ii].dist;
         blur_pixel(px,py,dist);
      }
   }

   zadd_locked(blur_busy,-1);
   pthread_exit(0);
}


void blur_pixel(int px, int py, int dist)
{
   int         jj, dx, dy, adx, ady, rad;
   double      red, green, blue;
   double      weight1, weight2, f1, f2;
   uint16      *pix1, *pix3, *pixN;

   pix1 = bmpixel(E1rgb48,px,py);                                          //  source pixel
   pix3 = bmpixel(E3rgb48,px,py);                                          //  target pixel
   
   rad = blur_radius;
   red = green = blue = 0;
   weight2 = 0.0;
   
   if (! sa_Npixel) 
   {
      for (dy = -rad; dy <= rad; dy++)                                     //  loop neighbor pixels within radius
      for (dx = -rad; dx <= rad; dx++)
      {
         if (px+dx < 0 || px+dx > E3ww-1) continue;                        //  omit pixels off edge   v.6.3
         if (py+dy < 0 || py+dy > E3hh-1) continue;
         adx = abs(dx);
         ady = abs(dy);
         pixN = pix1 + (dy * E3ww + dx) * 3;
         weight1 = blur_weight[adx][ady];                                  //  weight at distance (dx,dy)
         weight2 += weight1;
         red += pixN[0] * weight1;                                         //  accumulate contributions
         green += pixN[1] * weight1;
         blue += pixN[2] * weight1;
      }

      red = red / weight2;                                                 //  weighted average   v.6.3
      green = green / weight2;
      blue = blue / weight2;

      pix3[0] = int(red);
      pix3[1] = int(green);
      pix3[2] = int(blue);
   }
   
   if (sa_Npixel)
   {
      for (dy = -rad; dy <= rad; dy++)                                     //  loop neighbor pixels within radius
      for (dx = -rad; dx <= rad; dx++)
      {
         if (px+dx < 0 || px+dx > E3ww-1) continue;                        //  omit pixels off edge
         if (py+dy < 0 || py+dy > E3hh-1) continue;
         jj = (py+dy) * E3ww + (px+dx);
         if (! sa_pixisin[jj]) continue;                                   //  omit pixels outside area   v.6.3
         adx = abs(dx);
         ady = abs(dy);
         pixN = pix1 + (dy * E3ww + dx) * 3;
         weight1 = blur_weight[adx][ady];                                  //  weight at distance (dx,dy)
         weight2 += weight1;
         red += pixN[0] * weight1;                                         //  accumulate contributions
         green += pixN[1] * weight1;
         blue += pixN[2] * weight1;
      }
      
      red = red / weight2;                                                 //  weighted average
      green = green / weight2;
      blue = blue / weight2;

      if (dist < sa_blend) {                                               //  blend changes over sa_blend
         f1 = 1.0 * dist / sa_blend;
         f2 = 1.0 - f1;
         red = f1 * red + f2 * pix1[0];
         green = f1 * green + f2 * pix1[1];
         blue = f1 * blue + f2 * pix1[2];
      }

      pix3[0] = int(red);
      pix3[1] = int(green);
      pix3[2] = int(blue);
   }

   blur_pixdone++;
   return;
}


/**************************************************************************/

//  image sharpening function

int      sharp_ED_cycles;
int      sharp_ED_reduce;
int      sharp_ED_thresh;
int      sharp_UM_radius;
int      sharp_UM_amount;
int      sharp_UM_thresh;
int      sharp_LP_radius;
int      sharp_LP_amount;
double   sharp_kernel[19][19];
char     sharp_function[4];
int      sharp_busy;
int      sharp_Npixels, sharp_Ndone;


void m_sharpen(GtkWidget *, const char *)
{
   int      sharp_dialog_event(zdialog *zd, const char *event);
   int      sharp_dialog_compl(zdialog *zd, int zstat);
   void *   sharp_thread(void *);

   if (! edit_setup(0,2)) return;                                          //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Sharpen Image"),mWin,Bundo,Bdone,Bcancel,null); 

   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb21","hb2",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb22","hb2",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb23","hb2",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","ED","vb21",ZTX("edge detection"),"space=5");
   zdialog_add_widget(zdedit,"label","lab21","vb22",ZTX("cycles"));
   zdialog_add_widget(zdedit,"label","lab22","vb22",ZTX("reduce"));
   zdialog_add_widget(zdedit,"label","lab23","vb22",ZTX("threshold"));
   zdialog_add_widget(zdedit,"spin","cyclesED","vb23","1|30|1|10");
   zdialog_add_widget(zdedit,"spin","reduceED","vb23","50|95|1|80");
   zdialog_add_widget(zdedit,"spin","threshED","vb23","1|99|1|1");

   zdialog_add_widget(zdedit,"hsep","sep4","dialog");
   zdialog_add_widget(zdedit,"hbox","hb4","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb41","hb4",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb42","hb4",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb43","hb4",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","UM","vb41",ZTX("unsharp mask"),"space=5");
   zdialog_add_widget(zdedit,"label","lab41","vb42",ZTX("radius"));
   zdialog_add_widget(zdedit,"label","lab42","vb42",ZTX("amount"));
   zdialog_add_widget(zdedit,"label","lab43","vb42",ZTX("threshold"));
   zdialog_add_widget(zdedit,"spin","radiusUM","vb43","1|9|1|2");
   zdialog_add_widget(zdedit,"spin","amountUM","vb43","1|200|1|100");
   zdialog_add_widget(zdedit,"spin","threshUM","vb43","1|99|1|1");

   zdialog_add_widget(zdedit,"hsep","sep5","dialog");
   zdialog_add_widget(zdedit,"hbox","hb5","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb51","hb5",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb52","hb5",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb53","hb5",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","LP","vb51","Laplacian","space=5");
   zdialog_add_widget(zdedit,"label","lab51","vb52",ZTX("radius"));
   zdialog_add_widget(zdedit,"label","lab52","vb52",ZTX("amount"));
   zdialog_add_widget(zdedit,"spin","radiusLP","vb53","1|9|1|1");
   zdialog_add_widget(zdedit,"spin","amountLP","vb53","1|100|1|50");

   zdialog_run(zdedit,sharp_dialog_event,sharp_dialog_compl);              //  run dialog, parallel
   *sharp_function = 0;
   start_thread(sharp_thread,0);                                           //  start working thread
   return;
}


//  sharpen dialog event and completion callback functions

int sharp_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat == 1) edit_undo();
   else if (zstat == 2) edit_done();
   else edit_cancel();
   return 0;
}


int sharp_dialog_event(zdialog *zd, const char *event)
{
   edit_undo();                                                            //  restore original image

   if (strcmpv(event,"ED","UM","LP",null))
   {
      zdialog_fetch(zd,"cyclesED",sharp_ED_cycles);                        //  get all input values
      zdialog_fetch(zd,"reduceED",sharp_ED_reduce);
      zdialog_fetch(zd,"threshED",sharp_ED_thresh);
      zdialog_fetch(zd,"radiusUM",sharp_UM_radius);
      zdialog_fetch(zd,"amountUM",sharp_UM_amount);
      zdialog_fetch(zd,"threshUM",sharp_UM_thresh);
      zdialog_fetch(zd,"radiusLP",sharp_LP_radius);
      zdialog_fetch(zd,"amountLP",sharp_LP_amount);

      strcpy(sharp_function,event);                                        //  pass to working thread
      signal_thread();
      wait_thread_idle();
   }

   return 0;
}


//  sharpen image thread function

void * sharp_thread(void *)
{
   int    sharp_ED();
   int    sharp_UM();
   int    sharp_LP();
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      if (strEqu(sharp_function,"ED")) sharp_ED();
      if (strEqu(sharp_function,"UM")) sharp_UM();                         //  do requested function
      if (strEqu(sharp_function,"LP")) sharp_LP();

      Fmodified = 1;
      mwpaint2(); 
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  image sharpen function by edge detection and compression

int sharp_ED()
{
   void  sharp_pixel_ED(int px, int py, int dist, int thresh);

   int      sharp_thresh1 = 100;                                           //  initial threshold
   double   sharp_thresh2 = 0.01 * sharp_ED_reduce;                        //  decline rate
   int      px, py, dist, thresh;
   int      ii, cycles;
   
   thresh = sharp_thresh1;
    
   for (cycles = 0; cycles < sharp_ED_cycles; cycles++)
   {
      if (cycles > 0) thresh = int(thresh * sharp_thresh2);

      if (! sa_Npixel)                                                     //  process entire image
      {
         dist = sa_blend = 0;
         for (py = 2; py < E3hh-2; py++)
         for (px = 2; px < E3ww-2; px++)                                   //  loop all pixels
            sharp_pixel_ED(px,py,dist,thresh);
      }

      if (sa_Npixel)                                                       //  selected area to sharpen
      {
         for (ii = 0; ii < sa_Npixel; ii++)                                //  process all enclosed pixels
         {
            px = sa_pixel[ii].px;
            py = sa_pixel[ii].py;
            dist = sa_pixel[ii].dist;
            if (px < 2 || px >= E3ww-2) continue;                          //  omit pixels on edge
            if (py < 2 || py >= E3hh-2) continue;
            sharp_pixel_ED(px,py,dist,thresh);
         }
      }

      edit_progress(cycles,sharp_ED_cycles);                               //  v.6.3
   }

   edit_progress(0,0);
   return 1;
}


void sharp_pixel_ED(int px, int py, int dist, int thresh)
{
   uint16   *pix1, *pix1u, *pix1d;
   uint16   *pix3, *pix3u, *pix3d, *pix3uu, *pix3dd;
   int      dd, rgb, pthresh;
   int      dx[4] = { -1, 0, 1, 1 };                                       //  4 directions: NW N NE E
   int      dy[4] = { -1, -1, -1, 0 };
   int      pv2, pv2u, pv2d, pv2uu, pv2dd, pvdiff;
   double   f1, f2;
   
   pthresh = sharp_ED_thresh;                                              //  pthresh = larger
   if (thresh > pthresh) pthresh = thresh;

   pix1 = bmpixel(E1rgb48,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb48,px,py);                                          //  output pixel

   for (dd = 0; dd < 4; dd++)                                              //  4 directions
   {
      pix3u = pix3 + (dy[dd] * E3ww + dx[dd]) * 3;                         //  upstream pixel
      pix3d = pix3 - (dy[dd] * E3ww - dx[dd]) * 3;                         //  downstream pixel

      for (rgb = 0; rgb < 3; rgb++)                                        //  loop 3 RGB colors
      {
         pv2 = pix3[rgb];
         pv2u = pix3u[rgb];                                                //  brightness difference
         pv2d = pix3d[rgb];                                                //    across target pixel

         pvdiff = pv2d - pv2u;
         if (pvdiff < 0) pvdiff = -pvdiff;
         if (pvdiff < 256 * pthresh) continue;                             //  brightness slope < threshold

         if (pv2u < pv2 && pv2 < pv2d)                                     //  slope up, monotone
         {
            pix3uu = pix3u + (dy[dd] * E3ww + dx[dd]) * 3;                 //  upstream of upstream pixel
            pix3dd = pix3d - (dy[dd] * E3ww - dx[dd]) * 3;                 //  downstream of downstream
            pv2uu = pix3uu[rgb];
            pv2dd = pix3dd[rgb];

            if (pv2uu >= pv2u) {                                           //  shift focus of changes to
               pix3u = pix3;                                               //    avoid up/down/up jaggies
               pv2u = pv2;
            }
            
            if (pv2dd <= pv2d) {
               pix3d = pix3;
               pv2d = pv2;
            }
               
            if (pv2u > 256) pv2u -= 256;
            if (pv2d < 65279) pv2d += 256;
         }
         
         else if (pv2u > pv2 && pv2 > pv2d)                                //  slope down, monotone
         {
            pix3uu = pix3u + (dy[dd] * E3ww + dx[dd]) * 3;
            pix3dd = pix3d - (dy[dd] * E3ww - dx[dd]) * 3;
            pv2uu = pix3uu[rgb];
            pv2dd = pix3dd[rgb];

            if (pv2uu <= pv2u) {
               pix3u = pix3;
               pv2u = pv2;
            }
            
            if (pv2dd >= pv2d) {
               pix3d = pix3;
               pv2d = pv2;
            }

            if (pv2d > 256) pv2d -= 256;
            if (pv2u < 65279) pv2u += 256;
         }

         else continue;                                                    //  slope too small

         if (sa_Npixel && dist < sa_blend) {                               //  if area selection, blend pixel
            f1 = 1.0 * dist / sa_blend;                                    //    changes over sa_blend
            f2 = 1.0 - f1;
            pix1u = pix1 + (dy[dd] * E1ww + dx[dd]) * 3;                   //  upstream input pixel      bugfix
            pix1d = pix1 - (dy[dd] * E1ww - dx[dd]) * 3;                   //  downstream input pixel     v.7.2.1
            pv2u = int(f1 * pv2u + f2 * pix1u[rgb]);
            pv2d = int(f1 * pv2d + f2 * pix1d[rgb]);
         }

         pix3u[rgb] = pv2u;                                                //  modified brightness values
         pix3d[rgb] = pv2d;                                                //    >> image3 pixel
      }
   }

   return;
}


//  image sharpen function using unsharp mask

int sharp_UM()
{
   void * sharp_UM_wthread(void *arg);
   
   if (sa_Npixel) sharp_Npixels = sa_Npixel;
   else  sharp_Npixels = E3ww * E3hh;
   sharp_Ndone = 0;

   for (int ii = 0; ii < NWthreads; ii++)                                  //  start worker threads
      start_detached_thread(sharp_UM_wthread,&wtindex[ii]);
   zadd_locked(sharp_busy,+NWthreads);

   while (sharp_busy)                                                      //  wait for completion
   {
      zsleep(0.01);
      edit_progress(sharp_Ndone,sharp_Npixels);                            //  show progress counter
   }

   edit_progress(0,0);
   return 1;
}


void * sharp_UM_wthread(void *arg)                                         //  worker thread function   v.7.7
{
   void  sharp_pixel_UM(int px, int py, int dist);

   int      index = *((int *) arg);
   int      ii, px, py, dist, rad;
   
   rad = sharp_UM_radius;

   if (! sa_Npixel)                                                        //  process entire image
   {
      dist = sa_blend = 0;
      for (py = index+rad; py < E3hh-rad; py += NWthreads)
      for (px = rad; px < E3ww-rad; px++)                                  //  loop all image3 pixels
      {
         sharp_pixel_UM(px,py,dist);
         sharp_Ndone++;
      }
   }

   if (sa_Npixel)                                                          //  selected area to sharpen
   {
      for (ii = index; ii < sa_Npixel; ii += NWthreads)                    //  process all enclosed pixels
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         dist = sa_pixel[ii].dist;
         if (px < rad || px >= E3ww-rad) continue;                         //  omit pixels on edge
         if (py < rad || py >= E3hh-rad) continue;
         sharp_pixel_UM(px,py,dist);
         sharp_Ndone++;
      }
   }

   zadd_locked(sharp_busy,-1);
   pthread_exit(0);
}


void sharp_pixel_UM(int px, int py, int dist)                              //  process one pixel
{
   int         rad, dx, dy, rgb;
   int         incr, cval, mean;
   uint16      *pix1, *pix3, *pixN;
   double      rad2, f1, f2;

   pix1 = bmpixel(E1rgb48,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb48,px,py);                                          //  output pixel

   rad = sharp_UM_radius;
   rad2 = 2 * rad + 1;
   rad2 = 1.0 / (rad2 * rad2);                                             //  1 / area of unsharp mask

   for (rgb = 0; rgb < 3; rgb++)                                           //  loop 3 RGB colors
   {
      mean = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop neighbor pixels within radius
      for (dx = -rad; dx <= rad; dx++)                                     //  outer loop y
      {
         pixN = pix1 + (dy * E3ww + dx) * 3;                               //  compute mean brightness
         mean += pixN[rgb];
      }

      mean = int(mean * rad2 + 0.5);
      
      cval = pix3[rgb];                                                    //  pixel brightness
      incr = cval - mean;                                                  //  - mean of neighbors
      if (abs(incr) < sharp_UM_thresh) continue;                           //  if < threshold, skip
      incr = incr * sharp_UM_amount / 100;                                 //  reduce to sharp_amount
      cval = cval + incr;                                                  //  add back to pixel
      if (cval < 0) cval = 0;
      if (cval > 65535) cval = 65535;
      
      if (sa_Npixel && dist < sa_blend) {                                  //  if area selection, blend pixel
         f1 = 1.0 * dist / sa_blend;                                       //    changes over sa_blend
         f2 = 1.0 - f1;
         cval = int(f1 * cval + f2 * pix1[rgb]);
      }
      
      pix3[rgb] = cval;
   }
   
   return;
}


//  image sharpen function using Lapacian method

int sharp_LP()
{
   void * sharp_LP_wthread(void *arg);

   int         rad, ii, jj, dx, dy;
   double      rad2, kern, kernsum;

   rad = sharp_LP_radius;
   rad2 = rad * rad;
   kernsum = 0;

   for (dy = -rad; dy <= rad; dy++)                                        //  kernel, Gaussian distribution
   for (dx = -rad; dx <= rad; dx++)
   {
      ii = dx + rad;
      jj = dy + rad;
      kern = (0.5 / rad2) * exp( -(dx*dx + dy*dy) / (2 * rad2));
      if (dx || dy) {
         sharp_kernel[ii][jj] = -kern;                                     //  surrounding cells < 0
         kernsum += kern;
      }
   }
   
   sharp_kernel[rad][rad] = kernsum + 1.0;                                 //  middle cell, total = +1

   if (sa_Npixel) sharp_Npixels = sa_Npixel;
   else  sharp_Npixels = E3ww * E3hh;
   sharp_Ndone = 0;

   for (int ii = 0; ii < NWthreads; ii++)                                  //  start worker threads
      start_detached_thread(sharp_LP_wthread,&wtindex[ii]);
   zadd_locked(sharp_busy,+NWthreads);

   while (sharp_busy)                                                      //  wait for completion
   {
      zsleep(0.01);
      edit_progress(sharp_Ndone,sharp_Npixels);                            //  show progress counter
   }

   edit_progress(0,0);
   return 1;
}


void * sharp_LP_wthread(void *arg)                                         //  worker thread function   v.7.7
{
   void  sharp_pixel_LP(int px, int py, int dist);

   int         index = *((int *) arg);
   int         ii, rad, px, py, dist;

   rad = sharp_LP_radius;

   if (! sa_Npixel)                                                        //  process entire image
   {
      dist = sa_blend = 0;
      for (py = index+rad; py < E3hh-rad; py += NWthreads)
      for (px = rad; px < E3ww-rad; px++)                                  //  loop all image3 pixels
      {
         sharp_pixel_LP(px,py,dist);
         sharp_Ndone++;
      }
   }

   if (sa_Npixel)                                                          //  selected area to sharpen
   {
      for (ii = index; ii < sa_Npixel; ii += NWthreads)                    //  process all enclosed pixels
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         dist = sa_pixel[ii].dist;
         if (px < rad || px >= E3ww-rad) continue;                         //  omit pixels on edge
         if (py < rad || py >= E3hh-rad) continue;
         sharp_pixel_LP(px,py,dist);
         sharp_Ndone++;
      }
   }

   zadd_locked(sharp_busy,-1);
   pthread_exit(0);
}


void sharp_pixel_LP(int px, int py, int dist)                              //  process one pixel
{
   uint16      *pix1, *pix3, *pixN;

   int         rad, dx, dy, rgb;
   int         sumpix[3], maxrgb;
   double      kern, f1, f2;
   double      scale, scale1, scale2;

   pix1 = bmpixel(E1rgb48,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb48,px,py);                                          //  output pixel

   rad = sharp_LP_radius;
   sumpix[0] = sumpix[1] = sumpix[2] = 0;      

   for (dy = -rad; dy <= rad; dy++)                                        //  loop surrounding block of pixels
   for (dx = -rad; dx <= rad; dx++)                                        //  outer loop y
   {
      pixN = pix1 + (dy * E3ww + dx) * 3;
      kern = sharp_kernel[dx+rad][dy+rad];
      for (rgb = 0; rgb < 3; rgb++)
         sumpix[rgb] += int(kern * pixN[rgb] - 0.5);                       //  round
   }
   
   maxrgb = sumpix[0];
   if (sumpix[1] > maxrgb) maxrgb = sumpix[1];
   if (sumpix[2] > maxrgb) maxrgb = sumpix[2];
   if (sumpix[0] < 0) sumpix[0] = 0;
   if (sumpix[1] < 0) sumpix[1] = 0;
   if (sumpix[2] < 0) sumpix[2] = 0;
   
   if (maxrgb > 65535) scale = 65535.0 / maxrgb;
   else scale = 1.0;
   scale1 = (100 - sharp_LP_amount) / 100.0;
   scale2 = (1.0 - scale1) * scale;

   pix3[0] = int(scale1 * pix1[0] + scale2 * sumpix[0]);
   pix3[1] = int(scale1 * pix1[1] + scale2 * sumpix[1]);
   pix3[2] = int(scale1 * pix1[2] + scale2 * sumpix[2]);
   
   if (sa_Npixel && dist < sa_blend) {                                     //  if area selection, blend pixel
      f1 = 1.0 * dist / sa_blend;                                          //    changes over sa_blend
      f2 = 1.0 - f1;
      pix3[0] = int(f1 * pix3[0] + f2 * pix1[0]);
      pix3[1] = int(f1 * pix3[1] + f2 * pix1[1]);
      pix3[2] = int(f1 * pix3[2] + f2 * pix1[2]);
   }

   return;
}


/**************************************************************************/

//  image noise reduction

int      denoise_method = 5;
int      denoise_radius = 4;
int      denoise_busy;
int      denoise_Npixels, denoise_Ndone;

void m_denoise(GtkWidget *, const char *)
{
   int denoise_dialog_event(zdialog *zd, const char *event);               //  dialog event function
   int denoise_dialog_compl(zdialog *zd, int zstat);                       //  dialog completion function
   void * denoise_thread(void *);

   const char  *denoise_message = ZTX(" Press the reduce button to \n"
                                      " reduce noise in small steps. \n"
                                      " Use undo to start over.");

   if (! edit_setup(0,2)) return;                                          //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Noise Reduction"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",denoise_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labalg","hb1",ZTX("algorithm"),"space=5");
   zdialog_add_widget(zdedit,"combo","method","hb1",0,"space=5|expand");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labrad","hb2",ZTX("radius"),"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","1|9|1|4","space=5");
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","reduce","hb3",Breduce,"space=5");
   zdialog_add_widget(zdedit,"button","Undo","hb3",Bundo,"space=5");
   
   zdialog_cb_app(zdedit,"method",ZTX("flatten outliers by color (1)"));
   zdialog_cb_app(zdedit,"method",ZTX("flatten outliers by color (2)"));
   zdialog_cb_app(zdedit,"method",ZTX("set median brightness by color"));
   zdialog_cb_app(zdedit,"method",ZTX("top hat filter by color"));

   zdialog_stuff(zdedit,"method",ZTX("top hat filter by color"));          //  default

   zdialog_run(zdedit,denoise_dialog_event,denoise_dialog_compl);          //  run dialog
   start_thread(denoise_thread,0);                                         //  start working thread
   return;
}


//  denoise dialog event and completion callback functions

int denoise_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat == 1) edit_done();                                            //  done
   else edit_cancel();                                                     //  cancel or destroy
   return 0;
}


int denoise_dialog_event(zdialog * zd, const char *event)
{
   char     method[40];

   if (strEqu(event,"radius")) 
      zdialog_fetch(zd,"radius",denoise_radius);

   if (strEqu(event,"method")) 
   {
      zdialog_fetch(zd,"method",method,39);

      if (strEqu(method,"flatten outliers by color (1)")) {
         denoise_method = 1;
         denoise_radius = 1;
      }

      if (strEqu(method,"flatten outliers by color (2)")) {
         denoise_method = 2;
         denoise_radius = 3;
      }

      if (strEqu(method,"set median brightness by color")) {
         denoise_method = 4;
         denoise_radius = 2;
      }

      if (strEqu(method,"top hat filter by color")) {
         denoise_method = 5;
         denoise_radius = 4;
      }
      
      zdialog_stuff(zd,"radius",denoise_radius);
   }
   
   if (strEqu(event,"reduce")) {
      signal_thread();                                                     //  trigger update thread
      wait_thread_idle();                                                  //  wait for thread done
   }

   if (strEqu(event,"Undo")) edit_undo();
   return 1;
}


//  image noise reduction thread

void * denoise_thread(void *)
{
   void * denoise_wthread(void *arg);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      E9rgb48 = RGB_copy(E3rgb48);                                         //  image3 is reference source
                                                                           //  image9 will be modified
      if (sa_Npixel) denoise_Npixels = sa_Npixel;
      else  denoise_Npixels = E3ww * E3hh;
      denoise_Ndone = 0;

      for (int ii = 0; ii < NWthreads; ii++)                               //  start worker threads
         start_detached_thread(denoise_wthread,&wtindex[ii]);
      zadd_locked(denoise_busy,+NWthreads);

      while (denoise_busy)                                                 //  wait for completion
      {
         zsleep(0.01);
         edit_progress(denoise_Ndone,denoise_Npixels);                     //  show progress counter
      }

      edit_progress(0,0);

      mutex_lock(&pixmaps_lock);
      RGB_free(E3rgb48);                                                   //  image9 >> image3
      E3rgb48 = E9rgb48;
      E9rgb48 = 0;
      mutex_unlock(&pixmaps_lock);

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * denoise_wthread(void *arg)                                          //  worker thread function   v.7.7
{
   void  denoise_func1(uint16 *pix3, uint16 *pix9);
   void  denoise_func2(uint16 *pix3, uint16 *pix9);
   void  denoise_func4(uint16 *pix3, uint16 *pix9);
   void  denoise_func5(uint16 *pix3, uint16 *pix9);
   
   int         index = *((int *) arg);
   int         ii, px, py, dist, rad;
   double      f1, f2;
   uint16      *pix1, *pix3, *pix9;

   rad = denoise_radius;

   if (sa_Npixel)
   {
      for (ii = index; ii < sa_Npixel; ii += NWthreads)                    //  process pixels in select area
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         dist = sa_pixel[ii].dist;
         if (px < rad || px >= E3ww-rad) continue;
         if (py < rad || py >= E3hh-rad) continue;
         pix1 = bmpixel(E1rgb48,px,py);                                    //  source pixel
         pix3 = bmpixel(E3rgb48,px,py);                                    //  source pixel
         pix9 = bmpixel(E9rgb48,px,py);                                    //  target pixel
         if (denoise_method == 1) denoise_func1(pix3,pix9);
         if (denoise_method == 2) denoise_func2(pix3,pix9);
         if (denoise_method == 4) denoise_func4(pix3,pix9);
         if (denoise_method == 5) denoise_func5(pix3,pix9);

         if (dist < sa_blend) {                                            //  blend changes over sa_blend
            f1 = 1.0 * dist / sa_blend;
            f2 = 1.0 - f1;
            pix9[0] = int(f1 * pix9[0] + f2 * pix1[0]);
            pix9[1] = int(f1 * pix9[1] + f2 * pix1[1]);
            pix9[2] = int(f1 * pix9[2] + f2 * pix1[2]);
         }

         denoise_Ndone++;
      }
   }

   if (! sa_Npixel)
   {
      for (py = index+rad; py < E3hh-rad; py += NWthreads)                 //  loop all image3 pixels
      for (px = rad; px < E3ww-rad; px++)
      {
         pix3 = bmpixel(E3rgb48,px,py);                                    //  source pixel
         pix9 = bmpixel(E9rgb48,px,py);                                    //  target pixel
         if (denoise_method == 1) denoise_func1(pix3,pix9);
         if (denoise_method == 2) denoise_func2(pix3,pix9);
         if (denoise_method == 4) denoise_func4(pix3,pix9);
         if (denoise_method == 5) denoise_func5(pix3,pix9);
         denoise_Ndone++;
      }
   }

   zadd_locked(denoise_busy,-1);
   pthread_exit(0);
}


//  flatten outliers within radius, by color 
//  an outlier is the max or min value within a radius

void denoise_func1(uint16 *pix3, uint16 *pix9)
{
   int         dy, dx, rad;
   int         min0, min1, min2, max0, max1, max2;
   uint16      *pixN;

   min0 = min1 = min2 = 65535;
   max0 = max1 = max2 = 0;
   rad = denoise_radius;

   for (dy = -rad; dy <= rad; dy++)                                        //  loop surrounding pixels
   for (dx = -rad; dx <= rad; dx++)
   {
      if (dy == 0 && dx == 0) continue;                                    //  skip self

      pixN = pix3 + (dy * E3ww + dx) * 3;
      if (pixN[0] < min0) min0 = pixN[0];                                  //  find min and max per color
      if (pixN[0] > max0) max0 = pixN[0];
      if (pixN[1] < min1) min1 = pixN[1];
      if (pixN[1] > max1) max1 = pixN[1];
      if (pixN[2] < min2) min2 = pixN[2];
      if (pixN[2] > max2) max2 = pixN[2];
   }
   
   if (pix3[0] <= min0 && min0 < 65279) pix9[0] = min0 + 256;              //  if outlier, flatten a little
   if (pix3[0] >= max0 && max0 > 256) pix9[0] = max0 - 256;
   if (pix3[1] <= min1 && min1 < 65279) pix9[1] = min1 + 256;
   if (pix3[1] >= max1 && max1 > 256) pix9[1] = max1 - 256;
   if (pix3[2] <= min2 && min2 < 65279) pix9[2] = min2 + 256;
   if (pix3[2] >= max2 && max2 > 256) pix9[2] = max2 - 256;
   
   return;
}


//  flatten outliers
//  An outlier pixel has an RGB value outside one sigma of 
//  the mean for all pixels within a given radius of the pixel.

void denoise_func2(uint16 *pix3, uint16 *pix9)                             //  v.8.5
{
   int         rgb, dy, dx, rad, nn;
   double      nn1, val, sum, sum2, mean, variance, sigma;
   uint16      *pixN;

   rad = denoise_radius;
   nn = (rad * 2 + 1);
   nn = nn * nn - 1;
   nn1 = 1.0 / nn;

   for (rgb = 0; rgb < 3; rgb++)                                           //  loop RGB color
   {
      sum = sum2 = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding pixels
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dy == 0 && dx == 0) continue;                                 //  skip self
         pixN = pix3 + (dy * E3ww + dx) * 3;
         val = pixN[rgb];
         sum += val;
         sum2 += val * val;
      }
      
      mean = nn1 * sum;
      variance = nn1 * (sum2 - 2.0 * mean * sum) + mean * mean;
      sigma = sqrt(variance);

      val = pix3[rgb];      
      if (val > mean + sigma) {                                            //  move value to mean +/- sigma
         val = mean + sigma;                                               //  v.8.6
         pix9[rgb] = val;
      }
      else if (val < mean - sigma) {
         val = mean - sigma;
         pix9[rgb] = val;
      }
   }
   
   return;
}


//  use median brightness for pixels within radius

void denoise_func4(uint16 *pix3, uint16 *pix9)
{
   int         dy, dx, rad;
   int         ns, rgb, bsortN[400];
   uint16      *pixN;

   rad = denoise_radius;

   for (rgb = 0; rgb < 3; rgb++)                                           //  loop all RGB colors
   {
      ns = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding pixels
      for (dx = -rad; dx <= rad; dx++)                                     //  get brightness values
      {
         pixN = pix3 + (dy * E3ww + dx) * 3;
         bsortN[ns] = pixN[rgb];
         ns++;
      }

      HeapSort(bsortN,ns);
      pix9[rgb] = bsortN[ns/2];                                            //  median brightness of ns pixels
   }

   return;
}


//  modified top hat filter: execute with increasing radius from 1 to limit
//  detect outlier by comparing with pixels in outer radius

void denoise_func5(uint16 *pix3, uint16 *pix9)
{
   int         dy, dx, rad;
   int         min0, min1, min2, max0, max1, max2;
   uint16      *pixN;

   for (rad = 1; rad <= denoise_radius; rad++)
   for (int loops = 0; loops < 2; loops++)
   {
      min0 = min1 = min2 = 65535;
      max0 = max1 = max2 = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop all pixels within rad
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dx > -rad && dx < rad) continue;                              //  skip inner pixels
         if (dy > -rad && dy < rad) continue;

         pixN = pix3 + (dy * E3ww + dx) * 3;
         if (pixN[0] < min0) min0 = pixN[0];                               //  find min and max per color
         if (pixN[0] > max0) max0 = pixN[0];                               //    among outermost pixels
         if (pixN[1] < min1) min1 = pixN[1];
         if (pixN[1] > max1) max1 = pixN[1];
         if (pixN[2] < min2) min2 = pixN[2];
         if (pixN[2] > max2) max2 = pixN[2];
      }
      
      if (pix3[0] < min0 && pix9[0] < 65279) pix9[0] += 256;               //  if central pixel is outlier,
      if (pix3[0] > max0 && pix9[0] > 256) pix9[0] -= 256;                 //    moderate its values
      if (pix3[1] < min1 && pix9[1] < 65279) pix9[1] += 256;
      if (pix3[1] > max1 && pix9[1] > 256) pix9[1] -= 256;
      if (pix3[2] < min2 && pix9[2] < 65279) pix9[2] += 256;
      if (pix3[2] > max2 && pix9[2] > 256) pix9[2] -= 256;
   }

   return;
}


/**************************************************************************/

//  trim image - use mouse to select image region to retain

int      trimx1, trimy1, trimx2, trimy2;                                   //  trim rectangle
int      trimww, trimhh;
double   trimR;																				//  trim aspect ratio
int      trim_status = 0;


void m_trim(GtkWidget *, const char *)
{
   void     trim_mousefunc();
   int      trim_dialog_compl(zdialog *zd, int zstat);
   void   * trim_thread(void *);

   const char  *trim_message = ZTX("Drag middle to move \n"
                                   "Drag corners to resize");
   char        text[40];
   
   if (! edit_setup(1,0)) return;                                          //  setup edit: use preview   v.8.4.1

   zdedit = zdialog_new(ZTX("Trim Image"),mWin,ZTX("Trim"),Bcancel,null);  //  dialog to get user inuts
   zdialog_add_widget(zdedit,"label","lab1","dialog",trim_message,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog");
   zdialog_add_widget(zdedit,"label","space","hb1",0,"space=10");
   zdialog_add_widget(zdedit,"label","labwhr","hb1","2345 x 1234  (R=1.90)");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog");
   zdialog_add_widget(zdedit,"label","space","hb2",0,"space=10");
   zdialog_add_widget(zdedit,"check","lockwh","hb2",ZTX("Lock Ratio"));    //  v.8.4

   zdialog_run(zdedit,0,trim_dialog_compl);                                //  run dialog, parallel

   mouseCBfunc = trim_mousefunc;                                           //  connect mouse function
   Mcapture++;
   gdk_window_set_cursor(drWin->window,0);                                 //  set normal cursor
   
   trimx1 = int(0.2 * E3ww);                                               //  start with 20% trim margins
   trimy1 = int(0.2 * E3hh);
   trimx2 = int(0.8 * E3ww);
   trimy2 = int(0.8 * E3hh);

   trimww = (trimx2 - trimx1);
   trimhh = (trimy2 - trimy1);
   trimR = 1.0 * trimww / trimhh;
   
   snprintf(text,39,"%d x %d  (R=%.2f)", trimww * Fww / E3ww,              //  stuff dialog poop 
                                    trimhh * Fhh / E3hh, trimR);           //    (final size)    v.8.4.1
   zdialog_stuff(zdedit,"labwhr",text);

   trim_status = 0;
   start_thread(trim_thread,0);                                            //  start working thread
   signal_thread();
   return;
}


//  trim dialog completion callback function

int trim_dialog_compl(zdialog *zd, int zstat)
{
   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   
   if (zstat != 1) {
      edit_cancel();
      return 0;
   }
   
   trimx1 = trimx1 * Fww / E3ww;                                           //  scale from preview size
   trimy1 = trimy1 * Fhh / E3hh;                                           //    to final size    v.8.4.1
   trimx2 = trimx2 * Fww / E3ww;
   trimy2 = trimy2 * Fhh / E3hh;
   trimww = trimx2 - trimx1;
   trimhh = trimy2 - trimy1;

   trim_status = 1;                                                        //  trim full image
   Fmodified = 1;
   edit_done();
   return 0;
}


//  trim mouse function                                                    //  overhauled  v.8.4.1

void trim_mousefunc()
{
   int         mpx, mpy, rlock;
   int         corner, moveall = 0;
   int         dx, dy, dd, d1, d2, d3, d4;
   char        text[40];
   double      drr;
   
   if (LMclick || Mxdrag || Mydrag)                                        //  mouse click or drag
   {
      if (LMclick) {
         mpx = Mxclick;
         mpy = Myclick;
         LMclick = 0;
      }
      else {
         mpx = Mxdrag;
         mpy = Mydrag;
      }
      
      if (Mxdrag || Mydrag) {
         moveall = 1;
         dd = 0.1 * (trimx2 - trimx1);                                     //  test if mouse is in the broad
         if (mpx < trimx1 + dd) moveall = 0;                               //    middle of the rectangle
         if (mpx > trimx2 - dd) moveall = 0;
         dd = 0.1 * (trimy2 - trimy1);
         if (mpy < trimy1 + dd) moveall = 0;
         if (mpy > trimy2 - dd) moveall = 0;
      }

      if (moveall) {                                                       //  yes, move the whole rectangle
         trimx1 += Mxdrag - Mxdown;
         trimx2 += Mxdrag - Mxdown;
         trimy1 += Mydrag - Mydown;
         trimy2 += Mydrag - Mydown;
         Mxdown = Mxdrag;                                                  //  reset drag origin
         Mydown = Mydrag;
         corner = 0;
      }

      else {                                                               //  no, find closest corner
         dx = mpx - trimx1;
         dy = mpy - trimy1;
         d1 = sqrt(dx*dx + dy*dy);
         
         dx = mpx - trimx2;
         dy = mpy - trimy1;
         d2 = sqrt(dx*dx + dy*dy);
         
         dx = mpx - trimx2;
         dy = mpy - trimy2;
         d3 = sqrt(dx*dx + dy*dy);
         
         dx = mpx - trimx1;
         dy = mpy - trimy2;
         d4 = sqrt(dx*dx + dy*dy);
         
         corner = 1;
         dd = d1;
         if (d2 < dd) { corner = 2; dd = d2; }
         if (d3 < dd) { corner = 3; dd = d3; }
         if (d4 < dd) { corner = 4; dd = d4; }
         
         if (corner == 1) { trimx1 = mpx; trimy1 = mpy; }                  //  move this corner to mouse
         if (corner == 2) { trimx2 = mpx; trimy1 = mpy; }
         if (corner == 3) { trimx2 = mpx; trimy2 = mpy; }
         if (corner == 4) { trimx1 = mpx; trimy2 = mpy; }
      }

      if (trimx1 > trimx2-10) trimx1 = trimx2-10;                          //  sanity limits
      if (trimy1 > trimy2-10) trimy1 = trimy2-10;
      if (trimx1 < 0) trimx1 = 0;
      if (trimy1 < 0) trimy1 = 0;
      if (trimx2 > E3ww) trimx2 = E3ww;
      if (trimy2 > E3hh) trimy2 = E3hh;

      zdialog_fetch(zdedit,"lockwh",rlock);                                //  w/h ratio locked
      if (rlock && corner) {
         if (corner < 3)
            trimy2 = trimy1 + 1.0 * (trimx2 - trimx1) / trimR;
         else
            trimy1 = trimy2 - 1.0 * (trimx2 - trimx1) / trimR;
      }

      if (trimx1 > trimx2-10) trimx1 = trimx2-10;                          //  sanity limits
      if (trimy1 > trimy2-10) trimy1 = trimy2-10;
      if (trimx1 < 0) trimx1 = 0;
      if (trimy1 < 0) trimy1 = 0;
      if (trimx2 > E3ww) trimx2 = E3ww;
      if (trimy2 > E3hh) trimy2 = E3hh;

      trimww = trimx2 - trimx1;                                            //  new rectangle dimensions
      trimhh = trimy2 - trimy1;

      drr = 1.0 * trimww / trimhh;                                         //  new w/h ratio
      if (! rlock) trimR = drr;

      snprintf(text,39,"%d x %d  (R=%.2f)", trimww * Fww / E3ww,           //  stuff dialog poop 
                                    trimhh * Fhh / E3hh, drr);             //    (final size)
      zdialog_stuff(zdedit,"labwhr",text);
      
      signal_thread();                                                     //  trigger update thread
   }

   return;
}


//  trim thread function

void * trim_thread(void *)
{
   int      px1, py1, px2, py2;
   uint16   *pix1, *pix3;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      if (trim_status == 0)                                                //  darken margins   v.8.4
      {
         for (py1 = 0; py1 < E3hh; py1++)                                  //  copy pixels E1 >> E3
         for (px1 = 0; px1 < E3ww; px1++)
         {
            pix1 = bmpixel(E1rgb48,px1,py1);
            pix3 = bmpixel(E3rgb48,px1,py1);

            if (px1 < trimx1 || px1 > trimx2 || py1 < trimy1 || py1 > trimy2)
            {
               pix3[0] = pix1[0] / 2;
               pix3[1] = pix1[1] / 2;
               pix3[2] = pix1[2] / 2;
            }

            else
            {
               pix3[0] = pix1[0];
               pix3[1] = pix1[1];
               pix3[2] = pix1[2];
            }
         }

         mwpaint2();                                                       //  update window
      }
      
      if (trim_status == 1)                                                //  do the trim
      {
         mutex_lock(&pixmaps_lock);
         RGB_free(E3rgb48);
         E3rgb48 = RGB_make(trimww,trimhh,48);                             //  new pixmap with requested size
         E3ww = trimww;
         E3hh = trimhh;
         
         for (py1 = trimy1; py1 < trimy2; py1++)                           //  copy pixels
         for (px1 = trimx1; px1 < trimx2; px1++)
         {
            px2 = px1 - trimx1;
            py2 = py1 - trimy1;
            pix1 = bmpixel(E1rgb48,px1,py1);
            pix3 = bmpixel(E3rgb48,px2,py2);
            pix3[0] = pix1[0];
            pix3[1] = pix1[1];
            pix3[2] = pix1[2];
         }

         Fmodified = 1;
         mutex_unlock(&pixmaps_lock);
         mwpaint2();                                                       //  update window
      }
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  Resize (rescale) image
//
//  Output pixels are composites of input pixels, e.g. 2/3 size means 
//  that 3x3 input pixels are mapped into 2x2 output pixels, and an 
//  image size of 1000 x 600 becomes 667 x 400.


int      resize_ww0, resize_hh0, resize_ww1, resize_hh1;


void m_resize(GtkWidget *, const char *)
{
   int      resize_dialog_event(zdialog *zd, cchar * event);
   int      resize_dialog_compl(zdialog *zd, int zstat);
   void   * resize_thread(void *);

   const char  *lockmess = ZTX("Lock aspect ratio");

   if (! edit_setup(0,0)) return;                                          //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Resize Image"),mWin,Bapply,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb11","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb12","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb13","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"label","placeholder","vb11",0);              //             pixels       percent
   zdialog_add_widget(zdedit,"label","labw","vb11",Bwidth);                //    width    [______]     [______]
   zdialog_add_widget(zdedit,"label","labh","vb11",Bheight);               //    height   [______]     [______]
   zdialog_add_widget(zdedit,"label","labpix","vb12","pixels");            //
   zdialog_add_widget(zdedit,"spin","wpix","vb12","20|9999|1|0");          //    presets  [2/3] [1/2] [1/3] [1/4] 
   zdialog_add_widget(zdedit,"spin","hpix","vb12","20|9999|1|0");          //
   zdialog_add_widget(zdedit,"label","labpct","vb13",Bpercent);            //    [_] lock width/height ratio
   zdialog_add_widget(zdedit,"spin","wpct","vb13","1|500|0.1|100");        //
   zdialog_add_widget(zdedit,"spin","hpct","vb13","1|500|0.1|100");        //       [  done  ]  [ cancel ]  
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","preset","hb2",Bpreset,"space=5");
   zdialog_add_widget(zdedit,"button","b 3/4","hb2"," 3/4 ");
   zdialog_add_widget(zdedit,"button","b 2/3","hb2"," 2/3 ");
   zdialog_add_widget(zdedit,"button","b 1/2","hb2"," 1/2 ");
   zdialog_add_widget(zdedit,"button","b 1/3","hb2"," 1/3 ");
   zdialog_add_widget(zdedit,"button","b 1/4","hb2"," 1/4 ");
   zdialog_add_widget(zdedit,"check","lock","dialog",lockmess);

   resize_ww0 = Frgb48->ww;                                                //  original width, height
   resize_hh0 = Frgb48->hh;
   zdialog_stuff(zdedit,"wpix",resize_ww0);
   zdialog_stuff(zdedit,"hpix",resize_hh0);
   zdialog_stuff(zdedit,"lock",1);
   
   zdialog_run(zdedit,resize_dialog_event,resize_dialog_compl);            //  run dialog - parallel
   start_thread(resize_thread,0);                                          //  start working thread
   return;
}


//  resize dialog event function

int resize_dialog_event(zdialog *zd, const char * event)
{
   int         lock;
   double      wpct1, hpct1;

   zdialog_fetch(zd,"wpix",resize_ww1);                                    //  get all widget values
   zdialog_fetch(zd,"hpix",resize_hh1);
   zdialog_fetch(zd,"wpct",wpct1);
   zdialog_fetch(zd,"hpct",hpct1);
   zdialog_fetch(zd,"lock",lock);
   
   if (strEqu(event,"b 3/4")) {
      resize_ww1 = (3 * resize_ww0 + 3) / 4;
      resize_hh1 = (3 * resize_hh0 + 3) / 4;
   }
   
   if (strEqu(event,"b 2/3")) {
      resize_ww1 = (2 * resize_ww0 + 2) / 3;
      resize_hh1 = (2 * resize_hh0 + 2) / 3;
   }
   
   if (strEqu(event,"b 1/2")) {
      resize_ww1 = (resize_ww0 + 1) / 2;
      resize_hh1 = (resize_hh0 + 1) / 2;
   }
   
   if (strEqu(event,"b 1/3")) {
      resize_ww1 = (resize_ww0 + 2) / 3;
      resize_hh1 = (resize_hh0 + 2) / 3;
   }
   
   if (strEqu(event,"b 1/4")) {
      resize_ww1 = (resize_ww0 + 3) / 4;
      resize_hh1 = (resize_hh0 + 3) / 4;
   }

   if (strEqu(event,"wpct"))                                               //  width % - set pixel width
      resize_ww1 = int(wpct1 / 100.0 * resize_ww0 + 0.5);

   if (strEqu(event,"hpct"))                                               //  height % - set pixel height
      resize_hh1 = int(hpct1 / 100.0 * resize_hh0 + 0.5);
   
   if (lock && event[0] == 'w')                                            //  preserve width/height ratio
      resize_hh1 = int(resize_ww1 * (1.0 * resize_hh0 / resize_ww0) + 0.5);
   if (lock && event[0] == 'h') 
      resize_ww1 = int(resize_hh1 * (1.0 * resize_ww0 / resize_hh0) + 0.5);
   
   hpct1 = 100.0 * resize_hh1 / resize_hh0;                                //  set percents to match pixels
   wpct1 = 100.0 * resize_ww1 / resize_ww0;
   
   zdialog_stuff(zd,"wpix",resize_ww1);                                    //  index all widget values
   zdialog_stuff(zd,"hpix",resize_hh1);
   zdialog_stuff(zd,"wpct",wpct1);
   zdialog_stuff(zd,"hpct",hpct1);
   
   return 1;
}


//  resize dialog completion function

int resize_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat == 1) {
      signal_thread();
      edit_done();
   }
   else edit_cancel();
   return 0;
}


//  resize image based on dialog controls.

void * resize_thread(void *)
{
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      if (resize_ww1 != resize_ww0 || resize_hh1 != resize_hh0) 
      {                                                                    //  rescale to target size
         mutex_lock(&pixmaps_lock);
         RGB_free(E3rgb48);
         E3rgb48 = RGB_rescale(Frgb48,resize_ww1,resize_hh1);
         E3ww = resize_ww1;
         E3hh = resize_hh1;
         Fmodified = 1;
         mutex_unlock(&pixmaps_lock);
      }

      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  rotate image through any arbitrary angle

double      rotate_angle = 0;                                              //  E3 rotatation vs. F
double      rotate_delta = 0;
int         rotate_trim = 0;


void m_rotate(GtkWidget *, const char *menu)                               //  menu function
{
   int    rotate_dialog_event(zdialog *zd, const char * event);
   int    rotate_dialog_compl(zdialog *zd, int zstat);
   void * rotate_thread(void *);
   void   rotate_mousefunc();
   
   const char  *rotmess = ZTX("Use buttons or drag right edge with mouse");

   if (! edit_setup(1,0)) return;                                          //  setup edit: use preview   v.6.2

   zdedit = zdialog_new(ZTX("Rotate Image"),mWin,Bundo,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","labrot","dialog",ZTX(rotmess),"space=5");
   zdialog_add_widget(zdedit,"label","labdeg","dialog",ZTX("degrees"),"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb3","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb4","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"button"," +0.1  ","vb1"," + 0.1 ");          //  button name is increment to use
   zdialog_add_widget(zdedit,"button"," -0.1  ","vb1"," - 0.1 ");
   zdialog_add_widget(zdedit,"button"," +1.0  ","vb2"," + 1   ");
   zdialog_add_widget(zdedit,"button"," -1.0  ","vb2"," - 1   ");
   zdialog_add_widget(zdedit,"button"," +10.0 ","vb3"," + 10  ");
   zdialog_add_widget(zdedit,"button"," -10.0 ","vb3"," - 10  ");
   zdialog_add_widget(zdedit,"button"," +90.0 ","vb4"," + 90  ");
   zdialog_add_widget(zdedit,"button"," -90.0 ","vb4"," - 90  ");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","trim","hb2",ZTX("Trim"),"space=10");

   zdialog_run(zdedit,rotate_dialog_event,rotate_dialog_compl);            //  run dialog - parallel

   mouseCBfunc = rotate_mousefunc;                                         //  connect mouse function
   Mcapture++;
   gdk_window_set_cursor(drWin->window,dragcursor);                        //  set drag cursor

   rotate_angle = rotate_delta = rotate_trim = 0;
   start_thread(rotate_thread,0);                                          //  start working thread
   return;
}


//  rotate dialog event and completion callback functions

int rotate_dialog_event(zdialog *zd, const char * event)
{
   int         err;
   double      incr;
   char        text[20];
   
   if (strEqu(event,"trim")) {
      rotate_trim = 1 - rotate_trim;                                       //  toggle trim button   v.8.3
      if (rotate_trim) zdialog_stuff(zd,"trim",ZTX("Undo Trim"));
      else zdialog_stuff(zd,"trim",ZTX("Trim"));
   }
   
   if (strpbrk(event,"+-")) {
      err = convSD(event,incr);                                            //  button name is increment to use
      if (err) return 0;
      rotate_delta += incr;
   }

   zdialog_stuff(zd,"labdeg","computing");
   signal_thread();
   wait_thread_idle();

   gdk_window_set_cursor(drWin->window,dragcursor);                        //  set drag cursor   v.8.4

   sprintf(text,ZTX("degrees: %.1f"),rotate_angle);                        //  update dialog angle display
   zdialog_stuff(zd,"labdeg",text);
   return 1;
}


int rotate_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat == 1) {                                                       //  undo, dialog stays active
      edit_undo();                                                         //     v.6.1.2
      rotate_angle = rotate_delta = rotate_trim = 0;
      return 0;
   }

   if (zstat == 2) {
      rotate_delta = rotate_angle;                                         //  rotate main image   v.6.2
      rotate_angle = 0;
      edit_done();
   }
 
   else edit_cancel();

   rotate_angle = rotate_delta = rotate_trim = 0;

   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor
   return 0;
}


//  rotate mouse function - drag right edge of image up/down for rotation

void rotate_mousefunc()
{
   static int     mpx0 = 0, mpy0 = 0;
   static int     mpy1, mpy2, dist;

   if (! Mxdrag && ! Mydrag) return;                                       //  no drag underway
   if (Mxdrag < 0.8 * E3ww) return;                                        //  not right edge of image

   if (Mxdown != mpx0 || Mydown != mpy0) {
      mpx0 = Mxdown;                                                       //  new drag started
      mpy0 = mpy1 = Mydown;
   }
   
   mpy2 = Mydrag;
   dist = mpy2 - mpy1;                                                     //  drag distance
   mpy1 = mpy2;                                                            //  reset origin for next time
   if (! dist) return;

   rotate_delta = 30.0 * dist / E3ww;                                      //  convert to angle
   rotate_dialog_event(zdedit,"mouse");
   return;
}


//  rotate thread function

void * rotate_thread(void *)
{
   int         px3, py3, px9, py9;
   int         wwcut, hhcut, ww, hh;
   double      trim_angle, radians;
   uint16      *pix3, *pix9;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      mutex_lock(&pixmaps_lock);

      rotate_angle += rotate_delta;                                        //  accum. net rotation   
      rotate_delta = 0;                                                    //    from dialog widget
      
      if (rotate_angle >= 360) rotate_angle -=360;
      if (rotate_angle <= -360) rotate_angle +=360;
      if (fabs(rotate_angle) < 0.01) rotate_angle = 0;

      if (! rotate_angle) {
         RGB_free(E3rgb48);                                                //  E1 >> E3
         E3rgb48 = RGB_copy(E1rgb48);
         E3ww = E1ww;
         E3hh = E1hh;
         Fmodified = 0;
      }
      
      if (rotate_angle) {
         RGB_free(E3rgb48);
         E3rgb48 = RGB_rotate(E1rgb48,rotate_angle);                       //  E3 is rotated E1
         E3ww = E3rgb48->ww;
         E3hh = E3rgb48->hh;
         Fmodified = 1;
      }

      if (rotate_trim)
      {                                                                    //  auto trim      no reset v.8.3
         trim_angle = fabs(rotate_angle);
         while (trim_angle > 45) trim_angle -= 90;
         radians = fabs(trim_angle / 57.296);
         wwcut = int(E3rgb48->hh * sin(radians) + 1);                      //  amount to trim
         hhcut = int(E3rgb48->ww * sin(radians) + 1);
         ww = E3rgb48->ww - 2 * wwcut;
         hh = E3rgb48->hh - 2 * hhcut;
         if (ww > 0 && hh > 0) {
            E9rgb48 = RGB_make(ww,hh,48);
            
            for (py3 = hhcut; py3 < E3hh-hhcut; py3++)                     //  E9 = trimmed E3
            for (px3 = wwcut; px3 < E3ww-wwcut; px3++)
            {
               px9 = px3 - wwcut;
               py9 = py3 - hhcut;
               pix3 = bmpixel(E3rgb48,px3,py3);
               pix9 = bmpixel(E9rgb48,px9,py9);
               pix9[0] = pix3[0];
               pix9[1] = pix3[1];
               pix9[2] = pix3[2];
            }

            RGB_free(E3rgb48);                                             //  E3 = E9
            E3rgb48 = E9rgb48;
            E9rgb48 = 0;
            E3ww = ww;
            E3hh = hh;
         }
      }
      
      mutex_unlock(&pixmaps_lock);
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  unbend an image
//  straighten curvature added by pano or improve perspective

int      unbend_horz, unbend_vert;                                         //  unbend values from dialog
int      unbend_vert2, unbend_horz2;
double   unbend_x1, unbend_x2, unbend_y1, unbend_y2;                       //  unbend axes scaled 0 to 1
int      unbend_busy;
int      unbend_Npixels, unbend_Ndone;
int      unbend_hx1, unbend_hy1, unbend_hx2, unbend_hy2;
int      unbend_vx1, unbend_vy1, unbend_vx2, unbend_vy2;


//  menu function

void m_unbend(GtkWidget *, const char *)                                   //  overhaul for preview   v.6.2
{
   int      unbend_dialog_event(zdialog* zd, const char *event);
   int      unbend_dialog_compl(zdialog* zd, int zstat);
   void   * unbend_thread(void *);
   void     unbend_mousefunc();

   if (! edit_setup(1,0)) return;                                          //  setup edit: preview

   zdedit = zdialog_new(ZTX("Unbend Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog|space=10");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog|space=10");
   zdialog_add_widget(zdedit,"spin","spvert","vb1","-30|30|1|0");
   zdialog_add_widget(zdedit,"spin","sphorz","vb1","-20|20|1|0");
   zdialog_add_widget(zdedit,"label","labvert","vb2",ZTX("vertical unbend"));
   zdialog_add_widget(zdedit,"label","labhorz","vb2",ZTX("horizontal unbend"));
   
   zdialog_resize(zdedit,260,0);
   zdialog_run(zdedit,unbend_dialog_event,unbend_dialog_compl);            //  run dialog, parallel

   unbend_x1 = unbend_x2 = unbend_y1 = unbend_y2 = 0.5;                    //  initial axes thru image middle
   unbend_horz = unbend_vert = 0;                                          //  v.6.3

   Mcapture = 1;
   mouseCBfunc = unbend_mousefunc;                                         //  connect mouse function
   gdk_window_set_cursor(drWin->window,0);                                 //  set normal cursor
   
   start_thread(unbend_thread,0);                                          //  start working thread
   signal_thread();

   return;
}


//  dialog event and completion functions

int unbend_dialog_event(zdialog *zd, const char *event)
{
   zdialog_fetch(zd,"spvert",unbend_vert);                                 //  get new unbend values
   zdialog_fetch(zd,"sphorz",unbend_horz);
   signal_thread();                                                        //  trigger thread
   return 1;
}


int unbend_dialog_compl(zdialog *zd, int zstat)
{
   paint_toplines(2);                                                      //  erase axes-lines       v.8.4.3

   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;

   if (zstat != 1) {
      edit_cancel();                                                       //  canceled
      return 0;
   }

   if (unbend_vert || unbend_horz) Fmodified = 1;                          //  image3 modified
   else Fmodified = 0;
   edit_done();                                                            //  commit changes to image3
   return 1;
}


//  unbend mouse function                                                  //  adjustable axes

void unbend_mousefunc()   
{
   const char  *close;
   double      dist1, dist2;
   double      mpx = 0, mpy = 0;
   
   if (LMclick) {                                                          //  left mouse click   v.7.5
      LMclick = 0;
      mpx = Mxclick;
      mpy = Myclick;
   }
   
   if (Mxdrag || Mydrag) {                                                 //  mouse dragged
      mpx = Mxdrag;
      mpy = Mydrag;
   }
   
   if (! mpx && ! mpy) return;

   mpx = 1.0 * mpx / E3ww;                                                 //  scale mouse position 0 to 1
   mpy = 1.0 * mpy / E3hh;

   if (mpx < 0.2 || mpx > 0.8 ) {                                          //  check reasonable position
      if (mpy < 0.1 || mpy > 0.9) return;
   }
   else if (mpy < 0.2 || mpy > 0.8) {
      if (mpx < 0.1 || mpx > 0.9) return;
   }
   else return;

   close = "?";                                                            //  find closest axis end-point
   dist1 = 2;

   dist2 = mpx * mpx + (mpy-unbend_y1) * (mpy-unbend_y1);
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "left";
   }

   dist2 = (1-mpx) * (1-mpx) + (mpy-unbend_y2) * (mpy-unbend_y2);
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "right";
   }

   dist2 = (mpx-unbend_x1) * (mpx-unbend_x1) + mpy * mpy;
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "top";
   }

   dist2 = (mpx-unbend_x2) * (mpx-unbend_x2) + (1-mpy) * (1-mpy);
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "bottom";
   }
   
   if (strEqu(close,"left")) unbend_y1 = mpy;                              //  set new axis end-point
   if (strEqu(close,"right")) unbend_y2 = mpy;
   if (strEqu(close,"top")) unbend_x1 = mpx;
   if (strEqu(close,"bottom")) unbend_x2 = mpx;

   signal_thread();                                                        //  trigger thread 

   return ;
}


//  unbend thread function

void * unbend_thread(void *arg)
{
   void * unbend_wthread(void *);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      unbend_vert2 = int(unbend_vert * 0.01 * E3hh);                       //  convert % to pixels
      unbend_horz2 = int(unbend_horz * 0.005 * E3ww);

      unbend_hx1 = 0;                                                      //  scale axes to E3ww/hh
      unbend_hy1 = unbend_y1 * E3hh;
      unbend_hx2 = E3ww;
      unbend_hy2 = unbend_y2 * E3hh;

      unbend_vx1 = unbend_x1 * E3ww;
      unbend_vy1 = 0;
      unbend_vx2 = unbend_x2 * E3ww;
      unbend_vy2 = E3hh;

      if (Fpreview) {                                                      //  omit for final unbend   v.8.4.3
         Ntoplines = 2;
         toplinex1[0] = unbend_hx1;                                        //  lines on window
         topliney1[0] = unbend_hy1;
         toplinex2[0] = unbend_hx2;
         topliney2[0] = unbend_hy2;
         toplinex1[1] = unbend_vx1;
         topliney1[1] = unbend_vy1;
         toplinex2[1] = unbend_vx2;
         topliney2[1] = unbend_vy2;
      }
      
      unbend_Npixels = E3ww * E3hh;
      unbend_Ndone = 0;

      for (int ii = 0; ii < NWthreads; ii++)                               //  start worker threads
         start_detached_thread(unbend_wthread,&wtindex[ii]);
      zadd_locked(unbend_busy,+NWthreads);

      while (unbend_busy)                                                  //  wait for completion
      {
         zsleep(0.01);
         edit_progress(unbend_Ndone,unbend_Npixels);                       //  show progress counter
      }

      edit_progress(0,0);
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}

   
void * unbend_wthread(void *arg)                                           //  worker thread function   v.7.7
{
   int         index = *((int *) arg);
   int         vstat, px3, py3, cx3, cy3;
   double      px1, py1, dispx, dispx2, dispy;
   uint16      vpix[3], *pix3;

   for (py3 = index; py3 < E3hh; py3 += NWthreads)                         //  step through F3 pixels
   for (px3 = 0; px3 < E3ww; px3++)
   {
      pix3 = bmpixel(E3rgb48,px3,py3);                                     //  output pixel

      cx3 = unbend_vx1 + (unbend_vx2 - unbend_vx1) * py3 / E3hh;           //  center of unbend
      cy3 = unbend_hy1 + (unbend_hy2 - unbend_hy1) * px3 / E3ww;
      dispx = 2.0 * (px3 - cx3) / E3ww;                                    //  -1.0 ..  0.0 .. +1.0 (roughly)
      dispy = 2.0 * (py3 - cy3) / E3hh;                                    //  -1.0 ..  0.0 .. +1.0
      dispx2 = dispx * dispx - 0.5;                                        //  +0.5 .. -0.5 .. +0.5  curved

      px1 = px3 + dispx * dispy * unbend_horz2;                            //  input virtual pixel, x
      py1 = py3 - dispy * dispx2 * unbend_vert2;                           //  input virtual pixel, y
      vstat = vpixel(E1rgb48,px1,py1,vpix);                                //  input virtual pixel

      if (vstat) {
         pix3[0] = vpix[0];                                                //  input pixel >> output pixel
         pix3[1] = vpix[1];
         pix3[2] = vpix[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
      
      unbend_Ndone++;
   }
      
   zadd_locked(unbend_busy,-1);
   pthread_exit(0);
}


/**************************************************************************/

//  warp/distort area - select image area and pull with mouse

float       *WarpAx, *WarpAy;                                              //  memory of all displaced pixels
float       WarpAmem[4][100];                                              //  undo memory, last 100 warps
int         NWarpA;                                                        //  WarpA mem count

void  WarpA_warpfunc(float wdx, float wdy, float wdw, float wdh, int acc);


//  menu function

void m_WarpA(GtkWidget *, const char *)
{
   int      WarpA_dialog_event(zdialog *zd, const char *event);
   int      WarpA_dialog_compl(zdialog *zd, int zstat);
   
   const char  *WarpA_message = 
         ZTX(" Select an area to warp using select area function. \n"      //  v.6.3
             " Press [start warp] and pull area with mouse. \n"
             " Make multiple mouse pulls until satisfied. \n"
             " When finished, select another area or press [done]."); 
   
   int         px, py, ii;

   if (! edit_setup(0,2)) return;                                          //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Warp Image in Selected Area"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",WarpA_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","swarp","hb1",ZTX("start warp"),"space=5");
   zdialog_add_widget(zdedit,"button","undlast","hb1",Bundolast,"space=5");
   zdialog_add_widget(zdedit,"button","undall","hb1",Bundoall,"space=5");

   zdialog_run(zdedit,WarpA_dialog_event,WarpA_dialog_compl);              //  run dialog

   WarpAx = (float *) zmalloc(E3ww * E3hh * sizeof(float));                //  get memory for pixel displacements
   WarpAy = (float *) zmalloc(E3ww * E3hh * sizeof(float));
   
   NWarpA = 0;                                                             //  no warp data

   for (py = 0; py < E3hh; py++)                                           //  no pixel displacements
   for (px = 0; px < E3ww; px++)
   {
      ii = py * E3ww + px;
      WarpAx[ii] = WarpAy[ii] = 0.0;
   }
   
   return;
}


//  warp dialog event and completion callback functions

int WarpA_dialog_compl(zdialog * zd, int zstat)
{
   if (NWarpA) Fmodified = 1;
   if (zstat == 1) edit_done();
   else edit_cancel();

   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor

   zfree(WarpAx);                                                          //  release undo memory
   zfree(WarpAy);
   return 0;
}


int WarpA_dialog_event(zdialog * zd, const char *event)
{
   void     WarpA_mousefunc(void);

   int         px, py, ii;
   float       wdx, wdy, wdw, wdh;
   uint16      *pix1, *pix3;

   if (strEqu(event,"swarp"))                                              //  start warp
   {
      if (! sa_Npixel) {
         zmessageACK(ZTX("Select area first"));
         return 0;
      }

      if (! sa_calced) m_select_edgecalc(0,0);                             //  8.7

      mouseCBfunc = WarpA_mousefunc;                                       //  connect mouse function
      Mcapture++;
      gdk_window_set_cursor(drWin->window,dragcursor);                     //  set drag cursor
   }

   if (strEqu(event,"undlast")) {
      if (NWarpA) {                                                        //  undo most recent warp
         ii = --NWarpA;
         wdx = WarpAmem[0][ii];
         wdy = WarpAmem[1][ii];
         wdw = WarpAmem[2][ii];
         wdh = WarpAmem[3][ii];
         WarpA_warpfunc(wdx,wdy,-wdw,-wdh,0);                              //  unwarp image
         WarpA_warpfunc(wdx,wdy,-wdw,-wdh,1);                              //  unwarp memory
      }
   }

   if (strEqu(event,"undall")) {                                           //  undo all warps
      for (ii = 0; ii < sa_Npixel; ii++)                                   //  process all enclosed pixels
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         pix1 = bmpixel(E1rgb48,px,py);                                    //  image1 pixel >> image3
         pix3 = bmpixel(E3rgb48,px,py);
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }

      for (py = 0; py < E3hh; py++)                                        //  reset pixel displacements
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;
         WarpAx[ii] = WarpAy[ii] = 0.0;
      }

      NWarpA = 0;                                                          //  erase undo memory
      mwpaint2(); 
   }

   return 1;
}


//  warp mouse function

void  WarpA_mousefunc(void)
{
   static float   wdx, wdy, wdw, wdh;
   static int     ii, warped = 0;

   if (Mxdrag || Mydrag)                                                   //  mouse drag underway
   {
      wdx = Mxdown;                                                        //  drag origin, image coordinates
      wdy = Mydown;
      wdw = Mxdrag - Mxdown;                                               //  drag increment
      wdh = Mydrag - Mydown;
      WarpA_warpfunc(wdx,wdy,wdw,wdh,0);                                   //  warp image
      warped = 1;
      return;
   }
   
   else if (warped) 
   {
      warped = 0;
      WarpA_warpfunc(wdx,wdy,wdw,wdh,1);                                   //  drag done, add to warp memory

      if (NWarpA == 100)                                                   //  if full, throw away oldest
      {
         NWarpA = 99;
         for (ii = 0; ii < NWarpA; ii++)
         {
            WarpAmem[0][ii] = WarpAmem[0][ii+1];
            WarpAmem[1][ii] = WarpAmem[1][ii+1];
            WarpAmem[2][ii] = WarpAmem[2][ii+1];
            WarpAmem[3][ii] = WarpAmem[3][ii+1];
         }
      }

      ii = NWarpA;
      WarpAmem[0][ii] = wdx;                                               //  save warp for undo
      WarpAmem[1][ii] = wdy;
      WarpAmem[2][ii] = wdw;
      WarpAmem[3][ii] = wdh;
      NWarpA++;
   }
   
   return;
}


//  warp image and accumulate warp memory

void  WarpA_warpfunc(float wdx, float wdy, float wdw, float wdh, int acc)
{
   int            ii, jj, px, py, vstat;
   double         ddx, ddy, dpe, dpm, mag, dispx, dispy;
   uint16         vpix[3], *pix3;

   for (ii = 0; ii < sa_Npixel; ii++)                                      //  process all enclosed pixels
   {
      px = sa_pixel[ii].px;
      py = sa_pixel[ii].py;
      dpe = sa_pixel[ii].dist;                                             //  distance from area edge

      ddx = (px - wdx);                                                    //  distance from drag origin
      ddy = (py - wdy);
      dpm = sqrt(ddx*ddx + ddy*ddy);

      if (dpm < 1) mag = 1;
      else mag = dpe / (dpe + dpm);                                        //  magnification, 0...1
      mag = mag * mag;

      dispx = -wdw * mag;                                                  //  warp = drag * magnification
      dispy = -wdh * mag;
      
      jj = py * E3ww + px;

      if (acc) {                                                           //  mouse drag done,
         WarpAx[jj] += dispx;                                              //    accumulate warp memory
         WarpAy[jj] += dispy;
         continue;
      }

      dispx += WarpAx[jj];                                                 //  add this warp to prior
      dispy += WarpAy[jj];

      vstat = vpixel(E1rgb48,px+dispx,py+dispy,vpix);                      //  input virtual pixel
      if (vstat) {
         pix3 = bmpixel(E3rgb48,px,py);                                    //  output pixel
         pix3[0] = vpix[0];
         pix3[1] = vpix[1];
         pix3[2] = vpix[2];
      }
   }

   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************/

//  warp/distort whole image
//  fix perspective problems (e.g. curved walls, leaning buildings)

float       *WarpIx, *WarpIy;                                              //  memory of all dragged pixels
float       WarpImem[4][100];                                              //  undo memory, last 100 drags
int         NWarpI;                                                        //  WarpImem count
int         WarpIdrag;
int         WarpIww, WarpIhh;

void  WarpI_warpfunc(float wdx, float wdy, float wdw, float wdh, int acc);


//  menu function

void m_WarpI(GtkWidget *, const char *)                                    //  v.6.1
{
   int      WarpI_dialog_event(zdialog *zd, const char *event);
   int      WarpI_dialog_compl(zdialog *zd, int zstat);
   void     WarpI_mousefunc(void);

   const char  *WarpI_message = 
         ZTX(" Pull on an image edge using the mouse. \n"
             " Make multiple mouse pulls until satisfied. \n"
             " When finished, press [done]."); 
   
   int         px, py, ii;

   if (! edit_setup(1,0)) return;                                          //  setup edit: use preview

   zdedit = zdialog_new(ZTX("Fix Image Perspective"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",WarpI_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","undlast","hb1",Bundolast,"space=5");
   zdialog_add_widget(zdedit,"button","undall","hb1",Bundoall,"space=5");

   zdialog_run(zdedit,WarpI_dialog_event,WarpI_dialog_compl);              //  run dialog

   NWarpI = WarpIdrag = 0;                                                 //  no drag data

   WarpIx = (float *) zmalloc(E3ww * E3hh * sizeof(float));                //  get memory for pixel displacements
   WarpIy = (float *) zmalloc(E3ww * E3hh * sizeof(float));
   
   for (py = 0; py < E3hh; py++)                                           //  no pixel displacements
   for (px = 0; px < E3ww; px++)
   {
      ii = py * E3ww + px;
      WarpIx[ii] = WarpIy[ii] = 0.0;
   }
   
   WarpIww = E3ww;                                                         //  preview dimensions
   WarpIhh = E3hh;
   
   mouseCBfunc = WarpI_mousefunc;                                          //  connect mouse function
   Mcapture++;
   gdk_window_set_cursor(drWin->window,dragcursor);                        //  set drag cursor

   return;
}


//  WarpI dialog event and completion callback functions

int WarpI_dialog_compl(zdialog * zd, int zstat)
{
   int         fpx, fpy, epx, epy, ii, vstat;
   double      scale, dispx, dispy;
   uint16      vpix[3], *pix3;

   if (zstat != 1) edit_cancel();
   else if (NWarpI == 0) edit_cancel();
   
   else 
   {
      edit_fullsize();                                                     //  get full-size E1/E3

      scale = 1.0 * (E3ww + E3hh) / (WarpIww + WarpIhh);

      for (fpy = 0; fpy < E3hh; fpy++)                                     //  scale net pixel displacements
      for (fpx = 0; fpx < E3ww; fpx++)                                     //    to full image size
      {
         epx = WarpIww * fpx / E3ww;
         epy = WarpIhh * fpy / E3hh;
         ii = epy * WarpIww + epx;
         dispx = WarpIx[ii] * scale;
         dispy = WarpIy[ii] * scale;

         vstat = vpixel(E1rgb48,fpx+dispx,fpy+dispy,vpix);                 //  input virtual pixel
         pix3 = bmpixel(E3rgb48,fpx,fpy);                                  //  output pixel
         if (vstat) {
            pix3[0] = vpix[0];
            pix3[1] = vpix[1];
            pix3[2] = vpix[2];
         }
         else pix3[0] = pix3[1] = pix3[2] = 0;
      }

      edit_done();
   }

   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor

   zfree(WarpIx);                                                          //  release memory
   zfree(WarpIy);
   return 0;
}


int WarpI_dialog_event(zdialog * zd, const char *event)
{
   int         px, py, ii;
   float       wdx, wdy, wdw, wdh;
   
   if (strEqu(event,"undlast")) 
   {
      if (NWarpI == 1) event = "undall";
      else if (NWarpI) {                                                   //  undo most recent drag
         ii = --NWarpI;
         wdx = WarpImem[0][ii];
         wdy = WarpImem[1][ii];
         wdw = WarpImem[2][ii];
         wdh = WarpImem[3][ii];
         WarpI_warpfunc(wdx,wdy,-wdw,-wdh,0);                              //  undrag image
         WarpI_warpfunc(wdx,wdy,-wdw,-wdh,1);                              //  undrag memory
      }
   }

   if (strEqu(event,"undall"))                                             //  undo all drags
   {
      NWarpI = 0;                                                          //  erase undo memory

      for (py = 0; py < E3hh; py++)                                        //  reset pixel displacements
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;
         WarpIx[ii] = WarpIy[ii] = 0.0;
      }

      edit_undo();                                                         //  restore image1
      Fmodified = 0;                                                       //  v.6.3
   }

   return 1;
}


//  WarpI mouse function

void  WarpI_mousefunc(void)
{
   static float   wdx, wdy, wdw, wdh;
   int            ii;

   if (Mxdrag || Mydrag)                                                   //  mouse drag underway
   {
      wdx = Mxdown * Mscale;                                               //  drag origin, window coordinates
      wdy = Mydown * Mscale;
      wdw = (Mxdrag - Mxdown) * Mscale;                                    //  drag increment
      wdh = (Mydrag - Mydown) * Mscale;
      WarpI_warpfunc(wdx,wdy,wdw,wdh,0);                                   //  drag image
      WarpIdrag = 1;
      return;
   }
   
   else if (WarpIdrag) 
   {
      WarpIdrag = 0;
      WarpI_warpfunc(wdx,wdy,wdw,wdh,1);                                   //  drag done, add to memory

      if (NWarpI == 100)                                                   //  if full, throw away oldest
      {
         NWarpI = 99;
         for (ii = 0; ii < NWarpI; ii++)
         {
            WarpImem[0][ii] = WarpImem[0][ii+1];
            WarpImem[1][ii] = WarpImem[1][ii+1];
            WarpImem[2][ii] = WarpImem[2][ii+1];
            WarpImem[3][ii] = WarpImem[3][ii+1];
         }
      }

      ii = NWarpI;
      WarpImem[0][ii] = wdx;                                               //  save drag for undo
      WarpImem[1][ii] = wdy;
      WarpImem[2][ii] = wdw;
      WarpImem[3][ii] = wdh;
      NWarpI++;
   }
   
   return;
}


//  warp image and accumulate warp memory
//  mouse at (mx,my) is moved (mw,mh) pixels

void  WarpI_warpfunc(float mx, float my, float mw, float mh, int acc)
{
   int         ii, px, py, vstat;
   double      mag, dispx, dispy;
   double      d1, d2;
   uint16      vpix[3], *pix3;
   
   d1 = E3ww * E3ww + E3hh * E3hh;
   
   for (py = 0; py < E3hh; py++)                                           //  process all pixels
   for (px = 0; px < E3ww; px++)
   {
      d2 = (px-mx)*(px-mx) + (py-my)*(py-my);                              //  better algorithm   v.8.0
      mag = (1.0 - d2 / d1);
      mag = mag * mag;                                                     //  faster than pow(mag,16);
      mag = mag * mag;
      mag = mag * mag;

      dispx = -mw * mag;                                                   //  displacement = drag * mag
      dispy = -mh * mag;
      
      ii = py * E3ww + px;

      if (acc) {                                                           //  drag done, accumulate drag sum
         WarpIx[ii] += dispx;
         WarpIy[ii] += dispy;
         continue;
      }

      dispx += WarpIx[ii];                                                 //  add this drag to prior sum
      dispy += WarpIy[ii];

      vstat = vpixel(E1rgb48,px+dispx,py+dispy,vpix);                      //  input virtual pixel
      pix3 = bmpixel(E3rgb48,px,py);                                       //  output pixel
      if (vstat) {
         pix3[0] = vpix[0];
         pix3[1] = vpix[1];
         pix3[2] = vpix[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }

   Fmodified = 1;                                                          //  v.6.3
   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************/

//  image color-depth reduction

int      colordep_depth = 16;                                              //  bits per RGB color


void m_colordep(GtkWidget *, const char *)
{
   int      colordep_dialog_event(zdialog *zd, const char *event);
   int      colordep_dialog_compl(zdialog *zd, int zstat);
   void   * colordep_thread(void *);

   const char  *colmess = ZTX("Set color depth to 1-16 bits");
   
   if (! edit_setup(1,2)) return;                                          //  setup edit: preview

   zdedit = zdialog_new(ZTX("Set Color Depth"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",colmess,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"spin","colors","hb1","1|16|1|16","space=5");
   zdialog_add_widget(zdedit,"button","Undo","hb1",Bundo,"space=5");
   zdialog_add_widget(zdedit,"button","Redo","hb1",Bredo,"space=5");

   zdialog_run(zdedit,colordep_dialog_event,colordep_dialog_compl);        //  run dialog - parallel
   
   colordep_depth = 16;
   start_thread(colordep_thread,0);                                        //  start working thread
   return;
}


//  colors dialog event and completion callback functions

int colordep_dialog_event(zdialog *zd, const char *event)
{
   if (strcmpv(event,"colors","blendwidth",0)) {
      zdialog_fetch(zd,"colors",colordep_depth);
      signal_thread();
   }
   
   if (strEqu(event,"Undo")) edit_undo();
   if (strEqu(event,"Redo")) edit_redo();

   return 0;
}


int colordep_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) edit_done();
   else edit_cancel();
   return 0;
}


//  image color depth thread function

void * colordep_thread(void *)
{
   int         ii, px, py, rgb, dist;
   uint16      m1, m2, val1, val3;
   uint16      *pix1, *pix3;
   double      fmag, f1, f2;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
   
      m1 = 0xFFFF << (16 - colordep_depth);                                //  5 > 1111100000000000
      m2 = 0x8000 >> colordep_depth;                                       //  5 > 0000010000000000
      
      fmag = 65535.0 / m1;                                                 //  full brightness range  v.7.0

      if (! sa_Npixel)                                                     //  process entire image
      {
         for (py = 0; py < E3hh; py++)
         for (px = 0; px < E3ww; px++)
         {
            pix1 = bmpixel(E1rgb48,px,py);                                 //  input pixel
            pix3 = bmpixel(E3rgb48,px,py);                                 //  output pixel
            
            for (rgb = 0; rgb < 3; rgb++)
            {
               val1 = pix1[rgb];
               if (val1 < m1) val3 = (val1 + m2) & m1;                     //  round   v.7.0
               else val3 = m1;
               val3 = uint(val3 * fmag);
               pix3[rgb] = val3;
            }
         }
      }

      if (sa_Npixel)                                                       //  process select area
      {
         for (ii = 0; ii < sa_Npixel; ii++)
         {
            px = sa_pixel[ii].px;
            py = sa_pixel[ii].py;
            dist = sa_pixel[ii].dist;
            pix1 = bmpixel(E1rgb48,px,py);                                 //  input pixel
            pix3 = bmpixel(E3rgb48,px,py);                                 //  output pixel

            for (rgb = 0; rgb < 3; rgb++)
            {
               val1 = pix1[rgb];
               if (val1 < m1) val3 = (val1 + m2) & m1;
               else val3 = m1;
               val3 = uint(val3 * fmag);

               if (dist < sa_blend) {                                      //  if area selection, blend pixel
                  f2 = 1.0 * dist / sa_blend;                              //    changes over distance sa_blend
                  f1 = 1.0 - f2;
                  val3 = int(f1 * val1 + f2 * val3);
               }
               pix3[rgb] = val3;
            }
         }
      }

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  convert image to simulate a drawing

int      draw_busy = 0;
int      draw_contrast;
int      draw_threshold;
int      draw_pixcon;
int      draw_reverse;
double   draw_trfunc[256];
double   draw_pixcon3;
uint8    *draw_pixcon_map = 0;

void m_draw(GtkWidget *, const char *)                                     //  v.6.7
{
   int    draw_dialog_event(zdialog* zd, const char *event);
   int    draw_dialog_compl(zdialog* zd, int zstat);
   void * draw_thread(void *);

   const char  *title = ZTX("Simulate Drawing");
   uint16      *pix1, *pix2;
   int         ii, px, py, qx, qy;
   int         red, green, blue, con, maxcon;

   if (! edit_setup(0,2)) return;                                          //  setup edit: no preview

   draw_pixcon_map = (uint8 *) zmalloc(E1ww*E1hh);                         //  set up pixel contrast map
   memset(draw_pixcon_map,0,E1ww*E1hh);

   for (py = 1; py < E1hh-1; py++)                                         //  scan image pixels
   for (px = 1; px < E1ww-1; px++)
   {
      pix1 = bmpixel(E1rgb48,px,py);                                       //  pixel at (px,py)
      red = pix1[0];                                                       //  pixel RGB levels
      green = pix1[1];
      blue = pix1[2];
      maxcon = 0;

      for (qy = py-1; qy < py+2; qy++)                                     //  loop 3x3 block of neighbor
      for (qx = px-1; qx < px+2; qx++)                                     //    pixels around pix1
      {
         pix2 = bmpixel(E1rgb48,qx,qy);                                    //  find max. contrast with
         con = abs(red-pix2[0]) + abs(green-pix2[1]) + abs(blue-pix2[2]);  //    neighbor pixel
         if (con > maxcon) maxcon = con;
      }

      ii = py * E1ww + px;
      draw_pixcon_map[ii] = (maxcon/3) >> 8;                               //  contrast for (px,py) 0-255
   }

   zdedit = zdialog_new(title,mWin,Bundo,Bredo,Bdone,Bcancel,null);        //  setup drawing dialog
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog|expand");
   zdialog_add_widget(zdedit,"label","lab1","vb1",ZTX("contrast"));
   zdialog_add_widget(zdedit,"label","lab2","vb1",ZTX("threshold"));
   zdialog_add_widget(zdedit,"label","lab3","vb1",ZTX("outlines"));
   zdialog_add_widget(zdedit,"hscale","contrast","vb2","0|100|1|0","expand");
   zdialog_add_widget(zdedit,"hscale","threshold","vb2","0|100|1|0","expand");
   zdialog_add_widget(zdedit,"hscale","pixcon","vb2","0|255|1|0","expand");
   zdialog_add_widget(zdedit,"hbox","hb4","dialog");
   zdialog_add_widget(zdedit,"radio","pencil","hb4",ZTX("pencil"),"space=10");
   zdialog_add_widget(zdedit,"radio","chalk","hb4",ZTX("chalk"),"space=10");

   zdialog_run(zdedit,draw_dialog_event,draw_dialog_compl);                //  run dialog - parallel
   
   start_thread(draw_thread,0);                                            //  start working thread
   return;
}


//  draw dialog event and completion functions

int draw_dialog_event(zdialog *zd, const char *event)                      //  draw dialog event function
{
   zdialog_fetch(zd,"contrast",draw_contrast);                             //  get slider values
   zdialog_fetch(zd,"threshold",draw_threshold);
   zdialog_fetch(zd,"pixcon",draw_pixcon);
   zdialog_fetch(zd,"chalk",draw_reverse);
   signal_thread();                                                        //  trigger update thread
   return 1;
}


int draw_dialog_compl(zdialog *zd, int zstat)                              //  draw dialog completion function
{
   if (zstat == 1) { edit_undo(); return 0; }                              //  undo
   else if (zstat == 2) { edit_redo(); return 0; }                         //  redo
   else if (zstat == 3) edit_done();                                       //  done
   else edit_cancel();                                                     //  cancel or destroy
   zfree(draw_pixcon_map);
   return 0;
}


//  thread function - use multiple working threads

void * draw_thread(void *)
{
   void  * draw_wthread(void *arg);

   int         ii;
   double      threshold, contrast, trf;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      threshold = 0.01 * draw_threshold;                                   //  range 0 to 1
      contrast = 0.01 * draw_contrast;                                     //  range 0 to 1
      
      for (ii = 0; ii < 256; ii++)                                         //  brightness transfer function
      {
         trf = 1.0 - 0.003906 * (256 - ii) * contrast;                     //  ramp-up from 0-1 to 1
         if (ii < 256 * threshold) trf = 0;                                //  0 if below threshold
         draw_trfunc[ii] = trf;
      }
      
      for (ii = 0; ii < NWthreads; ii++)                                   //  start worker threads
         start_detached_thread(draw_wthread,&wtindex[ii]);
      zadd_locked(draw_busy,+NWthreads);

      while (draw_busy) zsleep(0.004);                                     //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * draw_wthread(void *arg)                                             //  worker thread function
{
   void  draw_1pix(int px, int py, int dist);

   int         index = *((int *) (arg));
   int         px, py, ii, dist;
   double      pixcon;

   pixcon = draw_pixcon / 255.0;                                           //  0-1 linear ramp
   draw_pixcon3 = 255 * pixcon * pixcon * pixcon;                          //  0-255 cubic ramp

   if (sa_Npixel)                                                          //  process selected area
   {
      for (ii = index; ii < sa_Npixel; ii += NWthreads)                    //  process all enclosed pixels
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         dist = sa_pixel[ii].dist;
         draw_1pix(px,py,dist);
      }
   }

   else
   {
      dist = sa_blend = 0;
      for (py = index; py < E1hh; py += NWthreads)                         //  process all pixels
      for (px = 0; px < E1ww; px++)
         draw_1pix(px,py,dist);
   }

   zadd_locked(draw_busy,-1);
   pthread_exit(0);
}


void draw_1pix(int px, int py, int dist)                                   //  process one pixel
{
   uint16      *pix1, *pix3;
   int         bright1, bright2;
   int         red1, green1, blue1;
   int         red3, green3, blue3;
   double      dold, dnew;
   double      pixcon = draw_pixcon3;
   
   if (! px || ! py) return;
   if (px > E1ww-2 || py > E1hh-2) return;
   
   pix1 = bmpixel(E1rgb48,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb48,px,py);                                          //  output pixel
      
   red1 = pix1[0];
   green1 = pix1[1];
   blue1 = pix1[2];
   
   bright1 = ((red1 + green1 + blue1) / 3) >> 8;                           //  old brightness  0-255
   bright2 = bright1 * draw_trfunc[bright1];                               //  new brightness  0-255
   
   int ii = py * E1ww + px;
   if (draw_pixcon_map[ii] < pixcon) bright2 = 255;

   if (pixcon > 1 && bright2 > draw_threshold) bright2 = 255;              //  empirical !!!

   if (draw_reverse) bright2 = 255 - bright2;                              //  negate if "chalk"

   red3 = green3 = blue3 = bright2 << 8;                                   //  gray scale, new brightness

   if (dist < sa_blend) {                                                  //  blend over distance sa_blend
      dnew = 1.0 * dist / sa_blend;
      dold = 1.0 - dnew;
      red3 = dnew * red3 + dold * red1;
      green3 = dnew * green3 + dold * green1;
      blue3 = dnew * blue3 + dold * blue1;
   }

   pix3[0] = red3;
   pix3[1] = green3;
   pix3[2] = blue3;
   
   return;
}


/**************************************************************************/

//  convert image to simulate an embossing

int      emboss_busy = 0;
int      emboss_radius, emboss_color;
double   emboss_depth;
double   emboss_kernel[20][20];                                            //  support radius <= 9

void m_emboss(GtkWidget *, const char *)                                   //  v.6.7
{
   int    emboss_dialog_event(zdialog* zd, const char *event);
   int    emboss_dialog_compl(zdialog* zd, int zstat);
   void * emboss_thread(void *);

   const char  *title = ZTX("Simulate Embossing");

   if (! edit_setup(0,2)) return;                                          //  setup edit: no preview

   zdedit = zdialog_new(title,mWin,Bundo,Bredo,Bdone,Bcancel,null);        //  setup embossing dialog
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","lab1","hb1",ZTX("radius"),"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb1","0|9|1|0");
   zdialog_add_widget(zdedit,"label","lab2","hb1",ZTX("depth"),"space=5");
   zdialog_add_widget(zdedit,"spin","depth","hb1","0|99|1|0");
   zdialog_add_widget(zdedit,"check","color","hb1",ZTX("color"),"space=8");

   zdialog_run(zdedit,emboss_dialog_event,emboss_dialog_compl);            //  run dialog - parallel
   
   start_thread(emboss_thread,0);                                          //  start working thread
   return;
}


//  emboss dialog event and completion functions

int emboss_dialog_event(zdialog *zd, const char *event)                    //  emboss dialog event function
{
   zdialog_fetch(zd,"radius",emboss_radius);                               //  get user inputs
   zdialog_fetch(zd,"depth",emboss_depth);
   zdialog_fetch(zd,"color",emboss_color);
   signal_thread();                                                        //  trigger update thread
   return 1;
}


int emboss_dialog_compl(zdialog *zd, int zstat)                            //  emboss dialog completion function
{
   if (zstat == 1) edit_undo();                                            //  undo
   else if (zstat == 2) edit_redo();                                       //  redo
   else if (zstat == 3) edit_done();                                       //  done
   else edit_cancel();                                                     //  cancel or destroy
   return 0;
}


//  thread function - use multiple working threads

void * emboss_thread(void *)
{
   void  * emboss_wthread(void *arg);

   int         ii, dx, dy, rad;
   double      depth, kern, coeff;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      rad = emboss_radius;
      depth = emboss_depth;

      coeff = 0.1 * depth / (rad * rad + 1);

      for (dy = -rad; dy <= rad; dy++)                                     //  build kernel with radius and depth
      for (dx = -rad; dx <= rad; dx++)
      {
         kern = coeff * (dx + dy);
         emboss_kernel[dx+rad][dy+rad] = kern;
      }
      
      emboss_kernel[rad][rad] = 1;                                         //  kernel center cell = 1

      for (ii = 0; ii < NWthreads; ii++)                                   //  start worker threads
         start_detached_thread(emboss_wthread,&wtindex[ii]);
      zadd_locked(emboss_busy,+NWthreads);

      while (emboss_busy) zsleep(0.004);                                   //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * emboss_wthread(void *arg)                                           //  worker thread function
{
   void  emboss_1pix(int px, int py, int dist);

   int         index = *((int *) (arg));
   int         px, py, ii, dist;

   if (sa_Npixel)                                                          //  process selected area
   {
      for (ii = index; ii < sa_Npixel; ii += NWthreads)                    //  process all enclosed pixels
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         dist = sa_pixel[ii].dist;
         emboss_1pix(px,py,dist);
      }
   }

   else
   {
      dist = sa_blend = 0;
      for (py = index; py < E1hh; py += NWthreads)                         //  process all pixels
      for (px = 0; px < E1ww; px++)
         emboss_1pix(px,py,dist);
   }

   zadd_locked(emboss_busy,-1);
   pthread_exit(0);
}


void emboss_1pix(int px, int py, int dist)                                 //  process one pixel
{
   uint16      *pix1, *pix3, *pixN;
   int         bright1, bright3;
   int         rgb, dx, dy, rad;
   double      sumpix, kern, dold, dnew;
   
   rad = emboss_radius;

   if (px < rad || py < rad) return;
   if (px > E3ww-rad-1 || py > E3hh-rad-1) return;
   
   pix1 = bmpixel(E1rgb48,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb48,px,py);                                          //  output pixel
   
   if (emboss_color)                                                       //  keep color   v.6.9
   {
      for (rgb = 0; rgb < 3; rgb++)
      {      
         sumpix = 0;
         
         for (dy = -rad; dy <= rad; dy++)                                  //  loop surrounding block of pixels
         for (dx = -rad; dx <= rad; dx++)
         {
            pixN = pix1 + (dy * E1ww + dx) * 3;
            kern = emboss_kernel[dx+rad][dy+rad];
            sumpix += kern * pixN[rgb];
      
            bright1 = pix1[rgb];
            bright3 = sumpix;
            if (bright3 < 0) bright3 = 0;
            if (bright3 > 65535) bright3 = 65535;

            if (dist < sa_blend) {                                         //  blend over distance sa_blend
               dnew = 1.0 * dist / sa_blend;
               dold = 1.0 - dnew;
               bright3 = dnew * bright3 + dold * bright1;
            }

            pix3[rgb] = bright3;
         }
      }
   }
   
   else                                                                    //  use gray scale
   {
      sumpix = 0;
         
      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding block of pixels
      for (dx = -rad; dx <= rad; dx++)
      {
         pixN = pix1 + (dy * E1ww + dx) * 3;
         kern = emboss_kernel[dx+rad][dy+rad];
         sumpix += kern * (pixN[0] + pixN[1] + pixN[2]);
      }
      
      bright1 = 0.3333 * (pix1[0] + pix1[1] + pix1[2]);
      bright3 = 0.3333 * sumpix;
      if (bright3 < 0) bright3 = 0;
      if (bright3 > 65535) bright3 = 65535;
      
      if (dist < sa_blend) {                                               //  blend over distance sa_blend
         dnew = 1.0 * dist / sa_blend;
         dold = 1.0 - dnew;
         bright3 = dnew * bright3 + dold * bright1;
      }

      pix3[0] = pix3[1] = pix3[2] = bright3;
   }

   return;
}


/**************************************************************************/

//  convert image to simulate square tiles

int         tile_size, tile_gap;
int         tile_Npixels, tile_pixdone;
uint16      *tile_pixmap = 0;


void m_tiles(GtkWidget *, const char *)                                    //  new  v.6.8
{
   int    tile_dialog_event(zdialog *zd, const char *event);
   int    tile_dialog_compl(zdialog *zd, int zstat);
   void * tile_thread(void *);

   if (! edit_setup(0,2)) return;                                          //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Set Tile and Gap Size"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labt","hb1",ZTX("tile size"),"space=5");
   zdialog_add_widget(zdedit,"spin","size","hb1","1|99|1|5","space=5");
   zdialog_add_widget(zdedit,"button","apply","hb1",Bapply,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labg","hb2",ZTX("tile gap"),"space=5");
   zdialog_add_widget(zdedit,"spin","gap","hb2","0|9|1|1","space=5");

   zdialog_run(zdedit,tile_dialog_event,tile_dialog_compl);                //  start dialog

   tile_size = 5;
   tile_gap = 1;

   tile_pixmap = (uint16 *) zmalloc(E1ww*E1hh*6);                          //  set up pixel color map
   memset(tile_pixmap,0,E1ww*E1hh*6);

   start_thread(tile_thread,0);                                            //  start working thread
   return;
}


//  tiles dialog event and completion callback functions

int tile_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) edit_done();                                            //  done
   else edit_cancel();                                                     //  cancel or destroy
   zfree(tile_pixmap);
   return 0;
}


int tile_dialog_event(zdialog * zd, const char *event)
{
   if (strNeq(event,"apply")) return 0;

   zdialog_fetch(zd,"size",tile_size);                                     //  get tile size
   zdialog_fetch(zd,"gap",tile_gap);                                       //  get tile gap 

   if (tile_size < 2) {
      if (Fmodified) edit_undo();                                          //  restore original image
      Fmodified = 0;
      return 0;
   }
   
   signal_thread();                                                        //  trigger working thread
   wait_thread_idle();                                                     //  wait for completion
   mwpaint2(); 
   return 1;
}


//  image tiles thread function

void * tile_thread(void *)
{
   int         sg, gg;
   int         sumpix, red, green, blue;
   int         ii, jj, px, py, qx, qy, dist;
   uint16      *pix1, *pix3;

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      sg = tile_size + tile_gap;
      gg = tile_gap;

      for (py = 0; py < E1hh; py += sg)                                    //  initz. pixel color map for
      for (px = 0; px < E1ww; px += sg)                                    //    given pixel size
      {
         sumpix = red = green = blue = 0;

         for (qy = py + gg; qy < py + sg; qy++)                            //  get mean color for pixel block
         for (qx = px + gg; qx < px + sg; qx++)
         {
            if (qy > E1hh-1 || qx > E1ww-1) continue;

            pix1 = bmpixel(E1rgb48,qx,qy);
            red += pix1[0];
            green += pix1[1];
            blue += pix1[2];
            sumpix++;
         }

         if (sumpix) {
            red = (red / sumpix);
            green = (green / sumpix);
            blue = (blue / sumpix);
         }
         
         for (qy = py; qy < py + sg; qy++)                                 //  set color for pixels in block
         for (qx = px; qx < px + sg; qx++)
         {
            if (qy > E1hh-1 || qx > E1ww-1) continue;
            
            jj = (qy * E1ww + qx) * 3;

            if (qx-px < gg || qy-py < gg) {
               tile_pixmap[jj] = tile_pixmap[jj+1] = tile_pixmap[jj+2] = 0;
               continue;
            }

            tile_pixmap[jj] = red;
            tile_pixmap[jj+1] = green;
            tile_pixmap[jj+2] = blue;
         }
      }

      if (! sa_Npixel)                                                     //  process entire image
      {
         for (py = 0; py < E3hh-1; py++)                                   //  loop all image pixels
         for (px = 0; px < E3ww-1; px++)
         {
            pix3 = bmpixel(E3rgb48,px,py);                                 //  target pixel
            jj = (py * E3ww + px) * 3;                                     //  color map for (px,py)
            pix3[0] = tile_pixmap[jj];
            pix3[1] = tile_pixmap[jj+1];
            pix3[2] = tile_pixmap[jj+2];
         }
      }

      if (sa_Npixel)                                                       //  process selected area
      {
         for (ii = 0; ii < sa_Npixel; ii++)                                //  process all enclosed pixels
         {
            px = sa_pixel[ii].px;
            py = sa_pixel[ii].py;
            dist = sa_pixel[ii].dist;
            pix3 = bmpixel(E3rgb48,px,py);
            jj = (py * E3ww + px) * 3;
            pix3[0] = tile_pixmap[jj];
            pix3[1] = tile_pixmap[jj+1];
            pix3[2] = tile_pixmap[jj+2];
         }
      }

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  convert image to simulate a painting                                   //  v.7.0
//  processing a 10 megapixel image needs 140 MB of main memory            //  v.7.3  select area added

namespace paint_namespace 
{
   int         color_depth;
   int         group_area;
   double      color_match;
   int         borders;

   typedef struct  {
      int16       px, py;
      char        direc;
   }  spixstack;

   int         Nstack;
   spixstack   *pixstack;                                                  //  pixel group search memory
   int         *pixgroup;                                                  //  maps (px,py) to pixel group no.
   int         *groupcount;                                                //  count of pixels in each group

   int         group;
   char        direc;
   uint16      gcolor[3];
}

using namespace paint_namespace;


void m_painting(GtkWidget *, const char *)
{
   int      painting_dialog_event(zdialog *zd, const char *event);
   int      painting_dialog_compl(zdialog *zd, int zstat);
   void   * painting_thread(void *);

   if (! edit_setup(0,2)) return;                                          //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Simulate Painting"),mWin,Bdone,Bcancel,null);

   zdialog_add_widget(zdedit,"hbox","hbcd","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","lab1","hbcd",ZTX("color depth"),"space=5");
   zdialog_add_widget(zdedit,"spin","colordepth","hbcd","1|5|1|3","space=5");

   zdialog_add_widget(zdedit,"hbox","hbts","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labts","hbts",ZTX("target group area"),"space=5");
   zdialog_add_widget(zdedit,"spin","grouparea","hbts","0|999|1|100","space=5");

   zdialog_add_widget(zdedit,"hbox","hbcm","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labcm","hbcm",ZTX("req. color match"),"space=5");
   zdialog_add_widget(zdedit,"spin","colormatch","hbcm","0|99|1|50","space=5");

   zdialog_add_widget(zdedit,"hbox","hbbd","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labbd","hbbd",ZTX("borders"),"space=5");
   zdialog_add_widget(zdedit,"check","borders","hbbd",0,"space=5");

   zdialog_add_widget(zdedit,"hbox","hbbu","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","apply","hbbu",Bapply,"space=5");
   zdialog_add_widget(zdedit,"button","undo","hbbu",Bundo,"space=5");

   zdialog_run(zdedit,painting_dialog_event,painting_dialog_compl);        //  run dialog - parallel

   start_thread(painting_thread,0);                                        //  start working thread
   return;
}


//  painting dialog event and completion callback functions

int painting_dialog_event(zdialog *zd, const char *event)
{
   if (strEqu(event,"apply")) {                                            //  apply user settings
      zdialog_fetch(zd,"colordepth",color_depth);                          //  color depth
      zdialog_fetch(zd,"grouparea",group_area);                            //  target group area (pixels)
      zdialog_fetch(zd,"colormatch",color_match);                          //  req. color match to combine groups
      zdialog_fetch(zd,"borders",borders);                                 //  borders wanted
      color_match = 0.01 * color_match;                                    //  scale 0 to 1
      signal_thread();
   }
   
   if (strEqu(event,"undo")) edit_undo();
   return 0;
}


int painting_dialog_compl(zdialog * zd, int zstat)                         //  done or cancel button
{
   if (zstat == 1) edit_done();
   else edit_cancel();
   return 0;
}


//  painting thread function

void * painting_thread(void *)
{
   void  paint_colordepth();
   void  paint_pixgroups();
   void  paint_mergegroups();
   void  paint_paintborders();
   void  paint_blend();

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      paint_colordepth();                                                  //  set new color depth
      paint_pixgroups();                                                   //  group pixel patches of a color
      paint_mergegroups();                                                 //  merge smaller into larger groups
      paint_paintborders();                                                //  add borders around groups
      paint_blend();                                                       //  blend edges of selected area

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  set the specified color depth, 1-5 bits/color

void paint_colordepth()
{
   int            ii, px, py, rgb;
   double         fmag;
   uint16         m1, m2, val1, val3;
   uint16         *pix1, *pix3;

   m1 = 0xFFFF << (16 - color_depth);                                      //  5 > 1111100000000000
   m2 = 0x8000 >> color_depth;                                             //  5 > 0000010000000000

   fmag = 65535.0 / m1;                                                    //  full brightness range

   if (! sa_Npixel)                                                        //  process entire image
   {
      for (py = 0; py < E3hh; py++)                                        //  loop all pixels
      for (px = 0; px < E3ww; px++)
      {
         pix1 = bmpixel(E1rgb48,px,py);                                    //  input pixel
         pix3 = bmpixel(E3rgb48,px,py);                                    //  output pixel
         
         for (rgb = 0; rgb < 3; rgb++)
         {
            val1 = pix1[rgb];
            if (val1 < m1) val3 = (val1 + m2) & m1;                        //  round   v.7.0
            else val3 = m1;
            val3 = uint(val3 * fmag);
            pix3[rgb] = val3;
         }
      }
   }

   if (sa_Npixel)                                                          //  process select area
   {
      for (ii = 0; ii < sa_Npixel; ii++)                                   //  loop all pixels in area
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         pix1 = bmpixel(E1rgb48,px,py);                                    //  input pixel
         pix3 = bmpixel(E3rgb48,px,py);                                    //  output pixel

         for (rgb = 0; rgb < 3; rgb++)
         {
            val1 = pix1[rgb];
            if (val1 < m1) val3 = (val1 + m2) & m1;
            else val3 = m1;
            val3 = uint(val3 * fmag);
            pix3[rgb] = val3;
         }
      }
   }
   
   return;
}


//  find all groups of contiguous pixels with the same color

void paint_pixgroups()
{
   void  paint_pushpix(int px, int py);

   int            cc1, cc2;
   int            ii, kk, px, py;
   uint16         *pix3;
   
   cc1 = E3ww * E3hh;

   cc2 = cc1 * sizeof(int);
   pixgroup = (int *) zmalloc(cc2);                                        //  maps pixel to assigned group
   memset(pixgroup,0,cc2);
   
   if (sa_Npixel) cc1 = sa_Npixel;

   cc2 = cc1 * sizeof(spixstack);
   pixstack = (spixstack *) zmalloc(cc2);                                  //  memory stack for pixel search
   memset(pixstack,0,cc2);
   
   cc2 = cc1 * sizeof(int);
   groupcount = (int *) zmalloc(cc2);                                      //  counts pixels per group
   memset(groupcount,0,cc2);
   
   group = 0;
   
   if (! sa_Npixel)
   {
      for (py = 0; py < E3hh; py++)                                        //  loop all pixels
      for (px = 0; px < E3ww; px++)
      {
         kk = py * E3ww + px;
         if (pixgroup[kk]) continue;                                       //  already assigned to group

         pixgroup[kk] = ++group;                                           //  assign next group
         ++groupcount[group];

         pix3 = bmpixel(E3rgb48,px,py);
         gcolor[0] = pix3[0];
         gcolor[1] = pix3[1];
         gcolor[2] = pix3[2];

         pixstack[0].px = px;                                              //  put pixel into stack with
         pixstack[0].py = py;                                              //    direction = right
         pixstack[0].direc = 'r';
         Nstack = 1;

         while (Nstack)
         {
            kk = Nstack - 1;                                               //  get last pixel in stack
            px = pixstack[kk].px;
            py = pixstack[kk].py;
            direc = pixstack[kk].direc;
            
            if (direc == 'x') {
               Nstack--;
               continue;
            }
            
            if (direc == 'r') {                                            //  push next right pixel into stack
               paint_pushpix(px,py);                                       //   if no group assigned and if
               pixstack[kk].direc = 'l';                                   //    same color as group color
               continue;
            }

            if (direc == 'l') {                                            //  or next left pixel
               paint_pushpix(px,py);
               pixstack[kk].direc = 'a';
               continue;
            }

            if (direc == 'a') {                                            //  or next ahead pixel
               paint_pushpix(px,py);
               pixstack[kk].direc = 'x';
               continue;
            }
         }
      }
   }
   
   if (sa_Npixel)
   {
      for (ii = 0; ii < sa_Npixel; ii++)                                   //  loop all pixels in area
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;

         kk = py * E3ww + px;
         if (pixgroup[kk]) continue;                                       //  already assigned to group

         pixgroup[kk] = ++group;                                           //  assign next group
         ++groupcount[group];

         pix3 = bmpixel(E3rgb48,px,py);
         gcolor[0] = pix3[0];
         gcolor[1] = pix3[1];
         gcolor[2] = pix3[2];

         pixstack[0].px = px;                                              //  put pixel into stack with
         pixstack[0].py = py;                                              //    direction = right
         pixstack[0].direc = 'r';
         Nstack = 1;

         while (Nstack)
         {
            kk = Nstack - 1;                                               //  get last pixel in stack
            px = pixstack[kk].px;
            py = pixstack[kk].py;
            direc = pixstack[kk].direc;
            
            if (direc == 'x') {
               Nstack--;
               continue;
            }
            
            if (direc == 'r') {                                            //  push next right pixel into stack
               paint_pushpix(px,py);                                       //   if no group assigned and if
               pixstack[kk].direc = 'l';                                   //    same color as group color
               continue;
            }

            if (direc == 'l') {                                            //  or next left pixel
               paint_pushpix(px,py);
               pixstack[kk].direc = 'a';
               continue;
            }

            if (direc == 'a') {                                            //  or next ahead pixel
               paint_pushpix(px,py);
               pixstack[kk].direc = 'x';
               continue;
            }
         }
      }
   }

   return;
}      


//  push a pixel into the stack memory if it is not already assigned
//  to another group and it has the same color as the current group

void paint_pushpix(int px, int py)
{
   int      kk, ppx, ppy, npx, npy;
   uint16   *pix3;

   if (Nstack > 1) {
      kk = Nstack - 2;                                                     //  get prior pixel in stack
      ppx = pixstack[kk].px;
      ppy = pixstack[kk].py;
   }
   else {
      ppx = px - 1;                                                        //  if only one, assume prior = left
      ppy = py;
   }
   
   if (direc == 'r') {                                                     //  get pixel in direction right
      npx = px + ppy - py;
      npy = py + px - ppx;
   }
   else if (direc == 'l') {                                                //  or left
      npx = px + py - ppy;
      npy = py + ppx - px;
   }
   else if (direc == 'a') {                                                //  or ahead
      npx = px + px - ppx;
      npy = py + py - ppy;
   }
   else npx = npy = -1;                                                    //  stop warning
   
   if (npx < 0 || npx >= E3ww) return;                                     //  pixel off the edge
   if (npy < 0 || npy >= E3hh) return;
   
   kk = npy * E3ww + npx;

   if (sa_Npixel)
      if (! sa_pixisin[kk]) return;                                        //  pixel outside area

   if (pixgroup[kk]) return;                                               //  pixel already assigned

   pix3 = bmpixel(E3rgb48,npx,npy);
   if (pix3[0] != gcolor[0] || pix3[1] != gcolor[1]                        //  not same color as group
                            || pix3[2] != gcolor[2]) return;
   
   pixgroup[kk] = group;                                                   //  assign pixel to group
   ++groupcount[group];

   kk = Nstack++;                                                          //  put pixel into stack
   pixstack[kk].px = npx;
   pixstack[kk].py = npy;
   pixstack[kk].direc = 'r';                                               //  direction = right
   
   return;
}


//  merge small pixel groups into adjacent larger groups with best color match

void paint_mergegroups()
{
   int         ii, jj, kk, px, py, npx, npy;
   int         nccc, mcount, group2;
   double      ff = 1.0 / 65536.0;
   double      fred, fgreen, fblue, match;
   int         nnpx[4] = {  0, -1, +1, 0 };
   int         nnpy[4] = { -1, 0,  0, +1 };
   uint16      *pix3, *pixN;

   typedef struct  {
      int         group;
      double      match;
      uint16      pixM[3];
   }  snewgroup;

   snewgroup      *newgroup;
   
   nccc = (group + 1) * sizeof(snewgroup);
   newgroup = (snewgroup *) zmalloc(nccc);
   
   if (! sa_Npixel)
   {
      while (true)
      {
         memset(newgroup,0,nccc);

         for (py = 0; py < E3hh; py++)                                     //  loop all pixels
         for (px = 0; px < E3ww; px++)
         {
            kk = E3ww * py + px;                                           //  get assigned group
            group = pixgroup[kk];
            if (groupcount[group] >= group_area) continue;                 //  group count large enough

            pix3 = bmpixel(E3rgb48,px,py);

            for (jj = 0; jj < 4; jj++)                                     //  get 4 neighbor pixels
            {
               npx = px + nnpx[jj];
               npy = py + nnpy[jj];

               if (npx < 0 || npx >= E3ww) continue;                       //  off the edge
               if (npy < 0 || npy >= E3hh) continue;
               
               kk = E3ww * npy + npx;
               if (pixgroup[kk] == group) continue;                        //  in same group

               pixN = bmpixel(E3rgb48,npx,npy);                            //  match color of group neighbor
               fred = ff * abs(pix3[0] - pixN[0]);                         //    to color of group
               fgreen = ff * abs(pix3[1] - pixN[1]);
               fblue = ff * abs(pix3[2] - pixN[2]);
               match = (1.0 - fred) * (1.0 - fgreen) * (1.0 - fblue);      //  color match, 0 to 1.0
               if (match < color_match) continue;

               if (match > newgroup[group].match) {
                  newgroup[group].match = match;                           //  remember best match
                  newgroup[group].group = pixgroup[kk];                    //  and corresp. group no.
                  newgroup[group].pixM[0] = pixN[0];                       //  and corresp. new color
                  newgroup[group].pixM[1] = pixN[1];
                  newgroup[group].pixM[2] = pixN[2];
               }
            }
         }

         mcount = 0;

         for (py = 0; py < E3hh; py++)                                     //  loop all pixels
         for (px = 0; px < E3ww; px++)
         {
            kk = E3ww * py + px;
            group = pixgroup[kk];                                          //  test for new group assignment
            group2 = newgroup[group].group;
            if (! group2) continue;
            
            if (groupcount[group] > groupcount[group2]) continue;          //  accept only bigger new group

            pixgroup[kk] = group2;                                         //  make new group assignment
            --groupcount[group];
            ++groupcount[group2];

            pix3 = bmpixel(E3rgb48,px,py);                                 //  make new color assignment
            pix3[0] = newgroup[group].pixM[0];
            pix3[1] = newgroup[group].pixM[1];
            pix3[2] = newgroup[group].pixM[2];

            mcount++;
         }
         
         if (mcount == 0) break;
      }
   }

   if (sa_Npixel)
   {
      while (true)
      {
         memset(newgroup,0,nccc);

         for (ii = 0; ii < sa_Npixel; ii++)                                //  loop all pixels in area
         {
            px = sa_pixel[ii].px;
            py = sa_pixel[ii].py;

            kk = E3ww * py + px;                                           //  get assigned group
            group = pixgroup[kk];
            if (groupcount[group] >= group_area) continue;                 //  group count large enough

            pix3 = bmpixel(E3rgb48,px,py);

            for (jj = 0; jj < 4; jj++)                                     //  get 4 neighbor pixels
            {
               npx = px + nnpx[jj];
               npy = py + nnpy[jj];

               if (npx < 0 || npx >= E3ww) continue;                       //  off the edge
               if (npy < 0 || npy >= E3hh) continue;

               kk = E3ww * npy + npx;
               if (! sa_pixisin[kk]) continue;                             //  pixel outside area
               if (pixgroup[kk] == group) continue;                        //  already in same group
               
               pixN = bmpixel(E3rgb48,npx,npy);                            //  match color of group neighbor
               fred = ff * abs(pix3[0] - pixN[0]);                         //    to color of group
               fgreen = ff * abs(pix3[1] - pixN[1]);
               fblue = ff * abs(pix3[2] - pixN[2]);
               match = (1.0 - fred) * (1.0 - fgreen) * (1.0 - fblue);      //  color match, 0 to 1.0
               if (match < color_match) continue;

               if (match > newgroup[group].match) {
                  newgroup[group].match = match;                           //  remember best match
                  newgroup[group].group = pixgroup[kk];                    //  and corresp. group no.
                  newgroup[group].pixM[0] = pixN[0];                       //  and corresp. new color
                  newgroup[group].pixM[1] = pixN[1];
                  newgroup[group].pixM[2] = pixN[2];
               }
            }
         }

         mcount = 0;

         for (ii = 0; ii < sa_Npixel; ii++)                                //  loop all pixels in area
         {
            px = sa_pixel[ii].px;
            py = sa_pixel[ii].py;

            kk = E3ww * py + px;
            group = pixgroup[kk];                                          //  test for new group assignment
            group2 = newgroup[group].group;
            if (! group2) continue;
            
            if (groupcount[group] > groupcount[group2]) continue;          //  accept only bigger new group

            pixgroup[kk] = group2;                                         //  make new group assignment
            --groupcount[group];
            ++groupcount[group2];

            pix3 = bmpixel(E3rgb48,px,py);                                 //  make new color assignment
            pix3[0] = newgroup[group].pixM[0];
            pix3[1] = newgroup[group].pixM[1];
            pix3[2] = newgroup[group].pixM[2];

            mcount++;
         }
         
         if (mcount == 0) break;
      }
   }

   zfree(pixgroup);
   zfree(pixstack);
   zfree(groupcount);
   zfree(newgroup);

   return;
}


//  paint borders between the groups of contiguous pixels

void paint_paintborders()
{
   int            ii, kk, px, py, cc;
   uint16         *pix3, *pixL, *pixA;
   
   if (! borders) return;
   
   cc = E3ww * E3hh;
   char * pixblack = zmalloc(cc);
   memset(pixblack,0,cc);

   if (! sa_Npixel)
   {
      for (py = 1; py < E3hh; py++)                                        //  loop all pixels
      for (px = 1; px < E3ww; px++)                                        //  omit top and left
      {
         pix3 = bmpixel(E3rgb48,px,py);                                    //  output pixel
         pixL = bmpixel(E3rgb48,px-1,py);                                  //  pixel to left
         pixA = bmpixel(E3rgb48,px,py-1);                                  //  pixel above
         
         if (pix3[0] != pixL[0] || pix3[1] != pixL[1] || pix3[2] != pixL[2])
         {
            kk = E3ww * py + px-1;                                         //  have horiz. transition
            if (pixblack[kk]) continue;
            kk += 1;
            pixblack[kk] = 1;
            continue;
         }

         if (pix3[0] != pixA[0] || pix3[1] != pixA[1] || pix3[2] != pixA[2])
         {
            kk = E3ww * (py-1) + px;                                       //  have vertical transition
            if (pixblack[kk]) continue;
            kk += E3ww;
            pixblack[kk] = 1;
         }
      }

      for (py = 1; py < E3hh; py++)
      for (px = 1; px < E3ww; px++)
      {
         kk = E3ww * py + px;
         if (! pixblack[kk]) continue;
         pix3 = bmpixel(E3rgb48,px,py);
         pix3[0] = pix3[1] = pix3[2] = 0;
      }
   }
   
   if (sa_Npixel)
   {
      for (ii = 0; ii < sa_Npixel; ii++)
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         if (px < 1 || py < 1) continue;

         pix3 = bmpixel(E3rgb48,px,py);
         pixL = bmpixel(E3rgb48,px-1,py);
         pixA = bmpixel(E3rgb48,px,py-1);
         
         if (pix3[0] != pixL[0] || pix3[1] != pixL[1] || pix3[2] != pixL[2])
         {
            kk = E3ww * py + px-1;
            if (pixblack[kk]) continue;
            kk += 1;
            pixblack[kk] = 1;
            continue;
         }

         if (pix3[0] != pixA[0] || pix3[1] != pixA[1] || pix3[2] != pixA[2])
         {
            kk = E3ww * (py-1) + px;
            if (pixblack[kk]) continue;
            kk += E3ww;
            pixblack[kk] = 1;
         }
      }

      for (ii = 0; ii < sa_Npixel; ii++)
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         if (px < 1 || py < 1) continue;

         kk = E3ww * py + px;
         if (! pixblack[kk]) continue;
         pix3 = bmpixel(E3rgb48,px,py);
         pix3[0] = pix3[1] = pix3[2] = 0;
      }
   }
         
   zfree(pixblack);
   return;
}


//  blend edges of selected area

void paint_blend()
{
   int         ii, px, py, rgb, dist;
   uint16      *pix1, *pix3;
   double      f1, f2;
   
   if (sa_Npixel && sa_blend > 0)
   {
      for (ii = 0; ii < sa_Npixel; ii++)                                   //  loop all pixels in area
      {
         dist = sa_pixel[ii].dist;
         if (dist >= sa_blend) continue;

         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         pix1 = bmpixel(E1rgb48,px,py);                                    //  input pixel
         pix3 = bmpixel(E3rgb48,px,py);                                    //  output pixel

         f2 = 1.0 * dist / sa_blend;                                       //  changes over distance sa_blend
         f1 = 1.0 - f2;

         for (rgb = 0; rgb < 3; rgb++)
            pix3[rgb] = int(f1 * pix1[rgb] + f2 * pix3[rgb]);
      }
   }

   return;
}


/**************************************************************************/

//  pixel edit function - edit individual pixels

void  pixed_mousefunc();
void  pixed_dopixels(int px, int py);
void  pixed_saveundo(int px, int py);
void  pixed_undo1();
void  pixed_freeundo();

int      pixed_RGB[3];
int      pixed_mode;
int      pixed_suspend;
int      pixed_radius;
double   pixed_kernel[200][200];                                           //  radius <= 99

int      pixed_undototpix = 0;                                             //  total undo pixel blocks
int      pixed_undototmem = 0;                                             //  total undo memory allocated
int      pixed_undoseq = 0;                                                //  undo sequence no.
char     pixed_undomemmessage[100];                                        //  translated undo memory message

typedef struct {                                                           //  pixel block before edit
   int         seq;                                                        //  undo sequence no.
   uint16      npix;                                                       //  no. pixels in this block
   uint16      px, py;                                                     //  center pixel (radius org.)
   uint16      radius;                                                     //  radius of pixel block
   uint16      pixel[][3];                                                 //  array of pixel[npix][3] 
}  pixed_savepix;

pixed_savepix   **pixed_undopixmem = 0;                                    //  array of *pixed_savepix


void m_pixedit(GtkWidget *, const char *)
{
   int   pixed_dialog_event(zdialog* zd, const char *event);
   int   pixed_dialog_compl(zdialog* zd, int zstat);

   char        undomemmessage[100];

   if (! edit_setup(0,1)) return;                                          //  setup edit: no preview

   strncpy0(pixed_undomemmessage,ZTX("Undo Memory %d%c"),99);              //  translate undo memory message

   zdedit = zdialog_new(ZTX("Edit Pixels"),mWin,Bdone,Bcancel,null);       //  setup pixel edit dialog
   zdialog_add_widget(zdedit,"hbox","hbc","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labc","hbc",ZTX("color"),"space=8");
   zdialog_add_widget(zdedit,"colorbutt","color","hbc","100|100|100");
   zdialog_add_widget(zdedit,"label","space","hbc",0,"space=10");
   zdialog_add_widget(zdedit,"radio","radio1","hbc",ZTX("pick"),"space=3");
   zdialog_add_widget(zdedit,"radio","radio2","hbc",ZTX("paint"),"space=3");
   zdialog_add_widget(zdedit,"radio","radio3","hbc",ZTX("erase"),"space=3");
   zdialog_add_widget(zdedit,"hbox","hbbr","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vbbr1","hbbr",0,"homog|space=3");
   zdialog_add_widget(zdedit,"vbox","vbbr2","hbbr",0,"homog|space=3");
   zdialog_add_widget(zdedit,"label","space","hbbr",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vbbr3","hbbr",0,"homog|space=3");
   zdialog_add_widget(zdedit,"hbox","hbrad","vbbr1",0,"space=3");
   zdialog_add_widget(zdedit,"label","space","hbrad",0,"expand");
   zdialog_add_widget(zdedit,"label","labbr","hbrad",ZTX("paintbrush radius"));
   zdialog_add_widget(zdedit,"label","labtc","vbbr1",ZTX("transparency center"));
   zdialog_add_widget(zdedit,"label","labte","vbbr1",ZTX("transparency edge"));
   zdialog_add_widget(zdedit,"spin","radius","vbbr2","1|99|1|2");
   zdialog_add_widget(zdedit,"spin","trcent","vbbr2","0|99|1|60");
   zdialog_add_widget(zdedit,"spin","tredge","vbbr2","0|99|1|99");
   zdialog_add_widget(zdedit,"button","susp-resm","vbbr3",Bsuspend);
   zdialog_add_widget(zdedit,"button","undlast","vbbr3",Bundolast);
   zdialog_add_widget(zdedit,"button","undall","vbbr3",Bundoall);
   zdialog_add_widget(zdedit,"label","labmem","dialog");

   zdialog_run(zdedit,pixed_dialog_event,pixed_dialog_compl);              //  run dialog - parallel

   zdialog_send_event(zdedit,"radius");                                    //  get kernel initialized

   snprintf(undomemmessage,99,pixed_undomemmessage,0,'%');                 //  stuff undo memory status
   zdialog_stuff(zdedit,"labmem",undomemmessage);
   
   pixed_RGB[0] = pixed_RGB[1] = pixed_RGB[2] = 100;                       //  initialize color
   
   pixed_mode = 1;                                                         //  mode = pick color
   pixed_suspend = 0;                                                      //  not suspended

   pixed_undopixmem = 0;                                                   //  no undo data
   pixed_undototpix = 0;
   pixed_undototmem = 0;
   pixed_undoseq = 0;

   mouseCBfunc = pixed_mousefunc;                                          //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks
   gdk_window_set_cursor(drWin->window,drawcursor);                        //  set draw cursor
   return;
}


//  pixedit dialog event and completion functions

int pixed_dialog_compl(zdialog *zd, int zstat)                             //  pixedit dialog completion function
{
   if (zstat == 1) edit_done();                                            //  done
   else edit_cancel();                                                     //  cancel or destroy

   paint_toparc(2);                                                        //  remove brush outline      v.8.3

   Mcapture = 0;                                                           //  disconnect mouse
   mouseCBfunc = 0;
   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor

   pixed_freeundo();                                                       //  free undo memory
   return 0;
}


int pixed_dialog_event(zdialog *zd, const char *event)                     //  pixedit dialog event function
{
   char        color[20];
   const char  *pp;
   int         radius, dx, dy, brad;
   double      rad, kern, trcent, tredge;
   
   zdialog_fetch(zd,"radio1",brad);                                        //  pick       v.6.8
   if (brad) pixed_mode = 1;
   zdialog_fetch(zd,"radio2",brad);                                        //  paint
   if (brad) pixed_mode = 2;
   zdialog_fetch(zd,"radio3",brad);                                        //  erase
   if (brad) pixed_mode = 3;
   
   if (strEqu(event,"color")) 
   {
      zdialog_fetch(zd,"color",color,19);                                  //  get color from color wheel
      pp = strField(color,"|",1);
      if (pp) pixed_RGB[0] = atoi(pp);
      pp = strField(color,"|",2);
      if (pp) pixed_RGB[1] = atoi(pp);
      pp = strField(color,"|",3);
      if (pp) pixed_RGB[2] = atoi(pp);
   }
   
   if (strstr("radius trcent tredge",event))                               //  get new brush attributes
   {
      zdialog_fetch(zd,"radius",radius);                                   //  radius
      zdialog_fetch(zd,"trcent",trcent);                                   //  center transparency       v.7.8
      zdialog_fetch(zd,"tredge",tredge);                                   //  edge transparency

      pixed_radius = radius;
      trcent = 0.01 * trcent;                                              //  scale 0 ... 1
      tredge = 0.01 * tredge;
      tredge = (1 - trcent) * (1 - tredge);
      tredge = 1 - tredge;
      trcent = sqrt(trcent);                                               //  speed up the curve
      tredge = sqrt(tredge);

      for (dy = -radius; dy <= radius; dy++)                               //  build kernel
      for (dx = -radius; dx <= radius; dx++)
      {
         rad = sqrt(dx*dx + dy*dy);
         kern = (radius - rad) / radius;                                   //  1 ... 0 
         kern = kern * (trcent - tredge) + tredge;                         //  trcent ... tredge
         if (rad > radius) kern = 1;
         if (kern < 0) kern = 0;
         if (kern > 1) kern = 1;
         pixed_kernel[dx+radius][dy+radius] = kern;
      }
   }
   
   if (strEqu(event,"undlast"))                                            //  undo last edit (click or drag)
      pixed_undo1();                                                       //  v.7.8

   if (strEqu(event,"undall")) {                                           //  undo all edits      v.7.8
      edit_undo();
      pixed_freeundo();
   }

   if (strEqu(event,"susp-resm"))                                          //  toggle suspend / resume   v.7.8
   {
      if (pixed_suspend) {
         pixed_suspend = 0;
         mouseCBfunc = pixed_mousefunc;                                    //  connect mouse function
         Mcapture++;
         gdk_window_set_cursor(drWin->window,drawcursor);                  //  set draw cursor
         zdialog_stuff(zd,"susp-resm",Bsuspend);
      }
      else  {
         pixed_suspend = 1;
         mouseCBfunc = 0;                                                  //  disconnect mouse function
         Mcapture = 0;
         gdk_window_set_cursor(drWin->window,0);                           //  restore normal cursor
         zdialog_stuff(zd,"susp-resm",Bresume);
      }
   }

   return 1;
}


//  pixel edit mouse function

void pixed_mousefunc()
{
   static int  pmxdown = 0, pmydown = 0;
   int         px, py;
   char        color[20];
   uint16      *ppix3;
   
   toparcx = Mxposn - pixed_radius;                                        //  define brush outline circle
   toparcy = Myposn - pixed_radius;                                        //  v.8.3
   toparcw = toparch = 2 * pixed_radius;
   if (pixed_mode == 1) toparc = 0;
   else toparc = 1;
   if (toparc) paint_toparc(3);
   
   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;
      px = Mxclick;
      py = Myclick;

      if (pixed_mode == 1)                                                 //  pick new color from image
      {
         ppix3 = bmpixel(E3rgb48,px,py);
         pixed_RGB[0] = ppix3[0] / 256;
         pixed_RGB[1] = ppix3[1] / 256;
         pixed_RGB[2] = ppix3[2] / 256;
         snprintf(color,19,"%d|%d|%d",pixed_RGB[0],pixed_RGB[1],pixed_RGB[2]);
         if (zdedit) zdialog_stuff(zdedit,"color",color);
      }
      else {                                                               //  paint or erase
         pixed_undoseq++;                                                  //  new undo seq. no.
         pixed_saveundo(px,py);                                            //  save for poss. undo
         pixed_dopixels(px,py);                                            //  do 1 block of pixels
      }
   }
   
   if (Mxdrag || Mydrag)                                                   //  drag in progress
   {
      px = Mxdrag;
      py = Mydrag;
      Mxdrag = Mydrag = 0;

      if (Mxdown != pmxdown || Mydown != pmydown) {                        //  new drag
         pixed_undoseq++;                                                  //  new undo seq. no.
         pmxdown = Mxdown;
         pmydown = Mydown;
      }
      pixed_saveundo(px,py);                                               //  save for poss. undo
      pixed_dopixels(px,py);                                               //  do 1 block of pixels
   }
   
   return;
}


//  paint or erase 1 block of pixels within radius of px, py

void pixed_dopixels(int px, int py)
{
   uint16      *ppix1, *ppix3;
   int         radius, dx, dy;
   int         red, green, blue;
   double      kern;

   radius = pixed_radius;

   red = 256 * pixed_RGB[0];
   green = 256 * pixed_RGB[1];
   blue = 256 * pixed_RGB[2];

   for (dy = -radius; dy <= radius; dy++)                                  //  loop surrounding block of pixels
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3ww-1) continue;                       //  v.7.5
      if (py + dy < 0 || py + dy > E3hh-1) continue;
      
      kern = pixed_kernel[dx+radius][dy+radius];
      ppix1 = bmpixel(E1rgb48,(px+dx),(py+dy));                            //  original image pixel
      ppix3 = bmpixel(E3rgb48,(px+dx),(py+dy));                            //  edited image pixel

      if (pixed_mode == 2)                                                 //  color pixels transparently
      {
         ppix3[0] = (1.0 - kern) * red   + kern * ppix3[0];
         ppix3[1] = (1.0 - kern) * green + kern * ppix3[1];
         ppix3[2] = (1.0 - kern) * blue  + kern * ppix3[2];
         Fmodified = 1;
      }

      if (pixed_mode == 3)                                                 //  restore org. pixels transparently
      {
         ppix3[0] = (1.0 - kern) * ppix1[0] + kern * ppix3[0];
         ppix3[1] = (1.0 - kern) * ppix1[1] + kern * ppix3[1];
         ppix3[2] = (1.0 - kern) * ppix1[2] + kern * ppix3[2];
      }
   }

   mwpaint2();
   return;
}


//  save 1 block of pixels for possible undo

void pixed_saveundo(int px, int py)
{
   int            npix, radius, dx, dy;
   uint16         *ppix3;
   pixed_savepix  *ppixsave1;
   char           undomemmessage[100];
   int            mempercent;
   static int     ppercent = 0;

   if (! pixed_undopixmem)                                                 //  first call
   {
      pixed_undopixmem = (pixed_savepix **) zmalloc(pixed_undomaxpix * sizeof(void *));
      pixed_undototpix = 0;
      pixed_undototmem = 0;
   }
   
   if (pixed_undototmem > pixed_undomaxmem) 
   {
      zmessageACK(ZTX("Undo memory limit has been reached (100 MB). \n"
                      "Save work with [done], then resume editing."));
      Mdrag = 0;                                                           //  stop mouse drag   v.8.3
      return;
   }

   radius = pixed_radius;
   npix = 0;

   for (dy = -radius; dy <= radius; dy++)                                  //  count pixels in block
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3ww-1) continue;
      if (py + dy < 0 || py + dy > E3hh-1) continue;
      npix++;
   }
   
   ppixsave1 = (pixed_savepix *) zmalloc(npix * 6 + 12);                   //  allocate memory for block
   pixed_undopixmem[pixed_undototpix] = ppixsave1;
   pixed_undototpix += 1;
   pixed_undototmem += npix * 6 + 12;
   
   ppixsave1->seq = pixed_undoseq;                                         //  save pixel block poop
   ppixsave1->npix = npix;
   ppixsave1->px = px;
   ppixsave1->py = py;
   ppixsave1->radius = radius;

   npix = 0;

   for (dy = -radius; dy <= radius; dy++)                                  //  save pixels in block
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3ww-1) continue;
      if (py + dy < 0 || py + dy > E3hh-1) continue;
      ppix3 = bmpixel(E3rgb48,(px+dx),(py+dy));                            //  edited image pixel
      ppixsave1->pixel[npix][0] = ppix3[0];
      ppixsave1->pixel[npix][1] = ppix3[1];
      ppixsave1->pixel[npix][2] = ppix3[2];
      npix++;
   }

   mempercent = int(100.0 * pixed_undototmem / pixed_undomaxmem);          //  update undo memory status
   if (mempercent != ppercent) {
      ppercent = mempercent;
      snprintf(undomemmessage,99,pixed_undomemmessage,mempercent,'%');
      zdialog_stuff(zdedit,"labmem",undomemmessage);
   }

   return;
}


//  undo last undo sequence number

void pixed_undo1()
{
   int            pindex, npix, radius, px, py, dx, dy;
   uint16         *ppix3;
   pixed_savepix  *ppixsave1;
   char           undomemmessage[100];
   int            mempercent;
   
   pindex = pixed_undototpix;
   
   while (pindex > 0)
   {
      --pindex;
      ppixsave1 = pixed_undopixmem[pindex];
      if (ppixsave1->seq != pixed_undoseq) break;
      px = ppixsave1->px;
      py = ppixsave1->py;
      radius = ppixsave1->radius;

      npix = 0;

      for (dy = -radius; dy <= radius; dy++)
      for (dx = -radius; dx <= radius; dx++)
      {
         if (px + dx < 0 || px + dx > E3ww-1) continue;
         if (py + dy < 0 || py + dy > E3hh-1) continue;
         ppix3 = bmpixel(E3rgb48,(px+dx),(py+dy));
         ppix3[0] = ppixsave1->pixel[npix][0];
         ppix3[1] = ppixsave1->pixel[npix][1];
         ppix3[2] = ppixsave1->pixel[npix][2];
         npix++;
      }

      npix = ppixsave1->npix;
      zfree(ppixsave1);
      pixed_undopixmem[pindex] = 0;
      pixed_undototmem -= (npix * 6 + 12);
      --pixed_undototpix;
   }
   
   if (pixed_undoseq > 0) --pixed_undoseq;

   mempercent = int(100.0 * pixed_undototmem / pixed_undomaxmem);          //  update undo memory status
   snprintf(undomemmessage,99,pixed_undomemmessage,mempercent,'%');
   zdialog_stuff(zdedit,"labmem",undomemmessage);

   mwpaint2();
   return;
}


//  free all undo memory

void pixed_freeundo()
{
   int            pindex;
   pixed_savepix  *ppixsave1;
   char           undomemmessage[100];

   pindex = pixed_undototpix;
   
   while (pindex > 0)
   {
      --pindex;
      ppixsave1 = pixed_undopixmem[pindex];
      zfree(ppixsave1);
   }
   
   if (pixed_undopixmem) zfree(pixed_undopixmem);
   pixed_undopixmem = 0;
   
   pixed_undoseq = 0;
   pixed_undototpix = 0;
   pixed_undototmem = 0;

   if (zdedit) {
      snprintf(undomemmessage,99,pixed_undomemmessage,0,'%');              //  undo memory = 0%
      zdialog_stuff(zdedit,"labmem",undomemmessage);
   }

   return;
}


/**************************************************************************/

//    Make an HDR (high dynamic range) image from two images of the same 
//    subject with different exposure levels. The HDR image has expanded 
//    visibility of detail in both the brightest and darkest areas.


void *   HDR_align_thread(void *);
void *   HDR_dialog_thread(void *);
void     HDR_combine(int ww);

int      HDR_brcurve_adjust(GtkWidget *, GdkEventButton *);
int      HDR_brcurve_draw(GtkWidget *);

int      HDR_align_stat = 0;

int      HDR_br_np = 0;                                                    //  up to 50 anchor points
double   HDR_br_px[50], HDR_br_py[50];                                     //    for image weight graph
double   HDR_weights[256];                                                 //  curve, weight per brightness
double   HDR_bratio = 1;

GtkWidget  *HDR_brcurve;


//  menu function

void m_HDR(GtkWidget *, const char *)
{
   char        *file2 = 0;

   if (! edit_setup(0,0)) return;                                          //  setup edit: no preview
   
   file2 = zgetfile(ZTX("Select image to combine"),image_file,"open");     //  get 2nd HDR image
   if (! file2) {
      edit_cancel();
      return;
   }

   Grgb48 = image_load(file2,48);                                          //  load and validate image
   if (! Grgb48) {
      edit_cancel();
      return;
   }

   Gww = Grgb48->ww;
   Ghh = Grgb48->hh;
   
   if (Fww != Gww || Fhh != Ghh) {
      zmessageACK(ZTX("2nd image not same size as 1st image"));
      RGB_free(Grgb48);
      Grgb48 = 0;
      edit_cancel();
      return;
   }

   start_thread(HDR_align_thread,0);                                       //  start thread to align images
   wrapup_thread(0);                                                       //  wait for thread exit

   if (HDR_align_stat != 1) {                                              //  check thread exit status
      edit_cancel();
      Nalign = 0;                                                          //  reset align counter
      RGB_free(A1rgb48);                                                   //  free alignment images
      RGB_free(A2rgb48);
      RGB_free(Grgb48);                                                    //  free 2nd input image pixmap
      if (redpixels) zfree(redpixels);                                     //  free edge-pixel flags
      redpixels = 0;
      return;
   }   
   
   int    HDR_dialog_event(zdialog *zd, const char *name);                 //  dialog for user adjustment
   int    HDR_dialog_compl(zdialog *zd, int zstat);

   const char  *weightmess = ZTX("Image Weights per Brightness Level");

   zdedit = zdialog_new(ZTX("HDR Image Weights"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","labt","dialog",weightmess,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"expand");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"expand|space=10");

   zdialog_add_widget(zdedit,"label","lab11","vb1",ZTX("Input Images"));
   zdialog_add_widget(zdedit,"hbox","hb12","vb1",0,"expand");
   zdialog_add_widget(zdedit,"vbox","vb11","hb12",0,"expand|space=10");
   zdialog_add_widget(zdedit,"vbox","vb12","hb12",0,"expand");
   zdialog_add_widget(zdedit,"label","lab111","vb11","image 1");
   zdialog_add_widget(zdedit,"label","lab112","vb11",0,"expand");
   zdialog_add_widget(zdedit,"label","lab113","vb11","image 2");
   zdialog_add_widget(zdedit,"label","lab121","vb12","100 %");
   zdialog_add_widget(zdedit,"label","lab122","vb12",0,"expand");
   zdialog_add_widget(zdedit,"label","lab123","vb12","50/50","expand");
   zdialog_add_widget(zdedit,"label","lab124","vb12",0,"expand");
   zdialog_add_widget(zdedit,"label","lab125","vb12","100 %");
   zdialog_add_widget(zdedit,"label","lab13","vb1"," ");

   zdialog_add_widget(zdedit,"label","lab21","vb2",ZTX("Output Image"));
   zdialog_add_widget(zdedit,"frame","fr22","vb2",0,"expand");
   zdialog_add_widget(zdedit,"hbox","hb23","vb2",0);
   zdialog_add_widget(zdedit,"label","lab231","hb23",Bdarker);
   zdialog_add_widget(zdedit,"label","lab232","hb23",0,"expand");
   zdialog_add_widget(zdedit,"label","lab233","hb23",Blighter);
   
   zdialog_add_widget(zdedit,"hbox","hbb","dialog",0,"space=10");          //  convenience buttons   v.8.6
   zdialog_add_widget(zdedit,"label","space","hbb",0,"expand");
   zdialog_add_widget(zdedit,"button","b100/0","hbb"," 100/0 ");
   zdialog_add_widget(zdedit,"button","b50/50","hbb"," 50/50 ");
   zdialog_add_widget(zdedit,"button","b0/100","hbb"," 0/100 ");
   zdialog_add_widget(zdedit,"label","space","hbb","   ");

   GtkWidget *brframe = zdialog_widget(zdedit,"fr22");                     //  add drawing area to frame
   GtkWidget *brcurve = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(brframe),brcurve);
   HDR_brcurve = brcurve;

   gtk_widget_add_events(brcurve,GDK_BUTTON_PRESS_MASK);                   //  connect drawing area events
   gtk_widget_add_events(brcurve,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(brcurve,GDK_BUTTON1_MOTION_MASK); 
   G_SIGNAL(brcurve,"motion-notify-event",HDR_brcurve_adjust,0)
   G_SIGNAL(brcurve,"button-press-event",HDR_brcurve_adjust,0)
   G_SIGNAL(brcurve,"expose-event",HDR_brcurve_draw,0)

   HDR_bratio = 0;                                                         //  get mean brightness ratio
   for (int ii = 0; ii < 256; ii++)
      HDR_bratio += Bratios2[0][ii] + Bratios2[1][ii] + Bratios2[2][ii];
   HDR_bratio = HDR_bratio / 256 / 3;
   
   HDR_br_np = 3;                                                          //  initz. image weights curve,
   HDR_br_px[0] = 5;                                                       //    3 spline anchor points
   HDR_br_px[1] = 127;
   HDR_br_px[2] = 250;

   if (HDR_bratio < 1) {                                                   //  image1 < image2, ramp up
      HDR_br_py[0] = 0.02;
      HDR_br_py[1] = 0.50;
      HDR_br_py[2] = 0.98;
   }
   else {                                                                  //  image1 > image2, ramp down
      HDR_br_py[0] = 0.98;
      HDR_br_py[1] = 0.50;
      HDR_br_py[2] = 0.02;
   }
   
   spline1(HDR_br_np,HDR_br_px,HDR_br_py);                                 //  make curve fitting anchor points
   
   zdialog_resize(zdedit,450,350);
   zdialog_run(zdedit,HDR_dialog_event,HDR_dialog_compl);                  //  run dialog, parallel
   start_thread(HDR_dialog_thread,0);                                      //  start compute thread
   signal_thread();
   return;
}


//  dialog event and completion functions for HDR image level adjustment

int HDR_dialog_event(zdialog *zd, const char *event)
{
   if (! event || *event != 'b') return 0;

   HDR_br_np = 3;                                                          //  initz. image weights curve,
   HDR_br_px[0] = 5;                                                       //    3 spline anchor points
   HDR_br_px[1] = 127;
   HDR_br_px[2] = 250;

   if (strEqu(event,"b100/0"))                                             //  100% image 1     v.8.6
      HDR_br_py[0] = HDR_br_py[1] = HDR_br_py[2] = 0.98;

   if (strEqu(event,"b0/100"))                                             //  100% image 2
      HDR_br_py[0] = HDR_br_py[1] = HDR_br_py[2] = 0.02;

   if (strEqu(event,"b50/50")) {                                           //  50/50
      if (HDR_bratio < 1) {
         HDR_br_py[0] = 0.02;                                              //  ramp up
         HDR_br_py[1] = 0.50;
         HDR_br_py[2] = 0.98;
      }
      else {
         HDR_br_py[0] = 0.98;                                              //  ramp down
         HDR_br_py[1] = 0.50;
         HDR_br_py[2] = 0.02;
      }
   }

   HDR_brcurve_draw(HDR_brcurve);                                          //  regen and redraw the curve
   
   return 1;
}


int HDR_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat != 1) edit_cancel();                                          //  user cancel
   else edit_done();
   Nalign = 0;                                                             //  reset align counter
   RGB_free(A1rgb48);                                                      //  free alignment images
   RGB_free(A2rgb48);
   RGB_free(Grgb48);                                                       //  free 2nd input image pixmap
   return 0;
}


//  thread to update image brightness levels
//  runs asynchronously to dialog updates

void * HDR_dialog_thread(void *)
{
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      HDR_combine(1);                                                      //  combine images >> E3rgb48
   }
   
   return 0;                                                               //  not executed, stop g++ warning
}


//  Add, delete, or move anchor points to weights curve using mouse.

int HDR_brcurve_adjust(GtkWidget *brcurve, GdkEventButton *event)
{
   int         ww, hh, px, py;
   int         kk, ii, newii, minii = -1;
   int         mx, my, button, evtype;
   double      dist2, mindist2 = 1000000;
   double      xval, yval;
   
   if (HDR_br_np > 49) {
      zmessageACK(ZTX("Exceed 50 anchor points"));
      return 0;
   }
   
   mx = int(event->x);                                                     //  mouse position in drawing area
   my = int(event->y);
   evtype = event->type;
   button = event->button;
   ww = brcurve->allocation.width;                                         //  drawing area size
   hh = brcurve->allocation.height;
   
   for (ii = 0; ii < HDR_br_np; ii++)                                      //  find closest anchor point
   {
      xval = HDR_br_px[ii];
      yval = spline2(xval);
      px = int(0.00392 * ww * xval);                                       //  0 - ww
      py = int(hh - hh * yval + 0.5);                                      //  0 - hh
      dist2 = (px-mx)*(px-mx) + (py-my)*(py-my);
      if (dist2 < mindist2) {
         mindist2 = dist2;
         minii = ii;
      }
   }

   if (minii < 0) return 0;                                                //  huh?
   
   if (evtype == GDK_BUTTON_PRESS && button == 3) {                        //  right click, remove anchor point
      if (mindist2 > 25) return 0;
      if (HDR_br_np < 3) return 0;
      for (kk = minii; kk < HDR_br_np -1; kk++) {
         HDR_br_px[kk] = HDR_br_px[kk+1];
         HDR_br_py[kk] = HDR_br_py[kk+1];
      }
      HDR_br_np--;

      HDR_brcurve_draw(brcurve);                                           //  regen and redraw the curve
      return 0;
   }

//  drag or left click, move nearby anchor point to mouse position,
//  or add a new anchor point if nothing near enough

   xval = 255.0 * mx / ww;                                                 //  0 - 255
   yval = 1.0 * (hh - my) / hh;                                            //  0 - 1.0

   if (xval < 0 || xval > 255) return 0;                                   //  v.6.8
   if (yval < 0 || yval > 1.0) return 0;

   if (mindist2 < 100) {                                                   //  existing point < 10 pixels away
      ii = minii;
      if (ii < HDR_br_np - 1 && HDR_br_px[ii+1] - xval < 5) return 0;      //  disallow < 5 x-pixels
      if (ii > 0 && xval - HDR_br_px[ii-1] < 5) return 0;                  //    to next or prior point
      newii = minii;                                                       //  this point to be moved
   }
   else                                                                    //  > 10 pixels, add a point
   {
      for (ii = 0; ii < HDR_br_np; ii++)
         if (xval <= HDR_br_px[ii]) break;                                 //  find point with next higher x

      if (ii < HDR_br_np && HDR_br_px[ii] - xval < 5) return 0;            //  disallow < 5 x-pixels
      if (ii > 0 && xval - HDR_br_px[ii-1] < 5) return 0;                  //    to next or prior point

      for (kk = HDR_br_np; kk > ii; kk--) {                                //  make hole for new point
         HDR_br_px[kk] = HDR_br_px[kk-1];
         HDR_br_py[kk] = HDR_br_py[kk-1];
      }

      HDR_br_np++;                                                         //  up point count
      newii = ii;                                                          //  this point to be added
   }

   HDR_br_px[newii] = xval;                                                //  coordinates of new or moved point
   HDR_br_py[newii] = yval;
   
   HDR_brcurve_draw(brcurve);                                              //  regen and redraw the curve
   return 0;
}


//  Draw brightness curve based on defined spline anchor points.

int HDR_brcurve_draw(GtkWidget *brcurve)
{
   int         ww, hh, px, py;
   int         ii, iix, iiy;
   double      xval, yval;

   ww = brcurve->allocation.width;                                         //  drawing area size
   hh = brcurve->allocation.height;
   if (ww < 50 || hh < 20) return 0;

   spline1(HDR_br_np,HDR_br_px,HDR_br_py);                                 //  make curve fitting anchor points

   gdk_window_clear(brcurve->window);                                      //  clear window

   for (px = 0; px < ww; px++)                                             //  generate all points for curve
   {
      xval = 255.0 * px / ww;
      yval = spline2(xval);
      py = int(hh - hh * yval + 0.5);
      gdk_draw_point(brcurve->window,gdkgc,px,py);
   }
   
   for (ii = 0; ii < HDR_br_np; ii++)                                      //  draw boxes at anchor points
   {
      xval = HDR_br_px[ii];
      yval = spline2(xval);
      px = int(0.00392 * ww * xval);
      py = int(hh - hh * yval + 0.5);
      for (iix = -2; iix < 3; iix++)
      for (iiy = -2; iiy < 3; iiy++) {
         if (px+iix < 0 || px+iix >= ww) continue;
         if (py+iiy < 0 || py+iiy >= hh) continue;
         gdk_draw_point(brcurve->window,gdkgc,px+iix,py+iiy);
      }
   }

   for (int ii = 0; ii < 256; ii++)                                        //  compute new weight curve for 
      HDR_weights[ii] = spline2(ii);                                       //    all 256 brightness levels
   
   signal_thread();                                                        //  signal thread to update
   return 0;
}


//  HDR image align thread, combine Frgb48 + Grgb48 >> E3rgb48

void * HDR_align_thread(void *)
{
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   double      xystep, xylim, tstep, tlim;
   int         firstpass, lastpass;

   HDR_align_stat = 0;                                                     //  no status yet
   Radjust = Gadjust = Badjust = 1.0;                                      //  no manual color adjustments
   Nalign = 1;                                                             //  alignment in progress
   aligntype = 1;                                                          //  HDR
   pixsamp = 5000;                                                         //  pixel sample size
   showRedpix = 1;                                                         //  highlight alignment pixels
   Fzoom = 0;                                                              //  fit to window if big
   Fblowup = 1;                                                            //  scale up to window if small
   firstpass = 1;
   lastpass = 0;
   
   fullSize = Fww;                                                         //  full image size
   if (Fhh > Fww) fullSize = Fhh;                                          //  (largest dimension)

   alignSize = 140;                                                        //  initial alignment image size
   if (alignSize > fullSize) alignSize = fullSize;
   A1rgb48 = A2rgb48 = 0;

   xoff = yoff = toff = 0;                                                 //  initial offsets = 0
   xshrink = yshrink = 0;                                                  //  no image shrinkage (pano)
   warpxu = warpyu = warpxl = warpyl = 0;                                  //  no warp factors (pano)
   warpxuB = warpyuB = warpxlB = warpylB = 0;

   while (true)                                                            //  next alignment stage / image size
   {   
      A1ww = Fww * alignSize / fullSize;                                   //  align width, height in same ratio
      A1hh = Fhh * alignSize / fullSize;
      A2ww = A1ww;
      A2hh = A1hh;
      
      if (! lastpass) 
      {
         RGB_free(A1rgb48);                                                //  align images = scaled input images
         RGB_free(A2rgb48);
         A1rgb48 = RGB_rescale(Frgb48,A1ww,A1hh);
         A2rgb48 = RGB_rescale(Grgb48,A2ww,A2hh);
         
         alignWidth = A1ww;                                                //  use full image for alignment
         alignHeight = A1hh;                                               //  v.8.0

         getAlignArea();                                                   //  get image overlap area
         getBrightRatios();                                                //  get color brightness ratios
         setColorfixFactors(1);                                            //  set color matching factors
         flagEdgePixels();                                                 //  flag high-contrast pixels

         mutex_lock(&pixmaps_lock);
         RGB_free(E3rgb48);                                                //  resize output image
         E3rgb48 = RGB_make(A1ww,A1hh,48);
         E3ww = A1ww;
         E3hh = A1hh;
         mutex_unlock(&pixmaps_lock);
      }

      xylim = 2;                                                           //  search range from prior stage:
      xystep = 1;                                                          //    -2 -1 0 +1 +2 pixels

      if (firstpass) xylim = 0.05 * alignSize;                             //  1st stage search range, huge

      if (lastpass) {
         xylim = 1;                                                        //  final stage search range:
         xystep = 0.5;                                                     //    -1.0 -0.5 0.0 +0.5 +1.0
      }

      tlim = xylim / alignSize / 2;                                        //  theta max offset, radians
      tstep = xystep / alignSize / 2;                                      //  theta step size

      xfL = xoff - xylim;
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      xoffB = xoff;
      yoffB = yoff;
      toffB = toff;
      
      matchB = matchImages();                                              //  set base match level
      HDR_combine(0);                                                      //  v.8.4

      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test all offset dimensions
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = matchImages();
         if (sigdiff(matchlev,matchB,0.00001) > 0) {
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
         }

         Nalign++;                                                         //  count alignment tests
         SBupdate++;                                                       //  update status bar
      }

      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;

      HDR_combine(0);                                                      //  combine images >> E3rgb48
     
      firstpass = 0;
      if (lastpass) break;                                                 //  done

      if (alignSize == fullSize) {                                         //  full size image was aligned
         lastpass++;                                                       //  one more pass
         continue;
      }

      double R = alignSize;
      alignSize = 2 * alignSize;                                           //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      R = alignSize / R;                                                   //  ratio of new / old image size
      xoff = R * xoff;                                                     //  adjust offsets for image size
      yoff = R * yoff;
   }

   zfree(redpixels);                                                       //  free edge-pixel flags
   redpixels = 0;
   showRedpix = 0;                                                         //  stop red pixel highlights
   Fblowup = 0;                                                            //  reset forced image scaling
   Fmodified = 1;                                                          //  image is modified
   HDR_align_stat = 1;                                                     //  signal success
   exit_thread();
   return 0;                                                               //  never executed, stop g++ warning
}


//  Combine images A1rgb48 and A2rgb48 using weights in HDR_weights[256].
//  Output is to E3rgb48 (not reallocated). Update window.

void HDR_combine(int weight)
{
   int         px3, py3, ii, vstat1, vstat2;
   double      px1, py1, px2, py2;
   double      sintf = sin(toff), costf = cos(toff);
   double      br1, br2, brm;
   uint16      vpix1[3], vpix2[3], *pix3;
   
   for (py3 = 0; py3 < A1hh; py3++)                                        //  step through A1rgb48 pixels
   for (px3 = 0; px3 < A1ww; px3++)
   {
      px1 = costf * px3 - sintf * (py3 - yoff);                            //  A1rgb48 pixel, after offsets
      py1 = costf * py3 + sintf * (px3 - xoff);
      vstat1 = vpixel(A1rgb48,px1,py1,vpix1);

      px2 = costf * (px3 - xoff) + sintf * (py3 - yoff);                   //  corresponding A2rgb48 pixel
      py2 = costf * (py3 - yoff) - sintf * (px3 - xoff);
      vstat2 = vpixel(A2rgb48,px2,py2,vpix2);

      pix3 = bmpixel(E3rgb48,px3,py3);                                     //  output pixel

      if (! vstat1 || ! vstat2) {                                          //  if non-overlapping pixel,
         pix3[0] = pix3[1] = pix3[2] = 0;                                  //    set output pixel black
         continue;
      }
      
      if (weight) {
         br1 = vpix1[0] + vpix1[1] + vpix1[2];                             //  image1 pixel brightness
         br2 = vpix2[0] + vpix2[1] + vpix2[2];                             //  image2
         brm = (br1 + br2) * 0.000651;                                     //  mean, 0 to 255.98
         br1 = HDR_weights[int(brm)];                                      //  image1 weight
         br2 = 1.0 - br1;                                                  //  image2 weight
         
         pix3[0] = int(vpix1[0] * br1 + vpix2[0] * br2);                   //  build output pixel
         pix3[1] = int(vpix1[1] * br1 + vpix2[1] * br2);
         pix3[2] = int(vpix1[2] * br1 + vpix2[2] * br2);
      }

      else {
         pix3[0] = (vpix1[0] + vpix2[0]) / 2;                              //  output pixel is simple average
         pix3[1] = (vpix1[1] + vpix2[1]) / 2;
         pix3[2] = (vpix1[2] + vpix2[2]) / 2;
      }

      if (showRedpix && vstat2) {                                          //  highlight alignment pixels
         ii = py3 * A1ww + px3;
         if (redpixels[ii]) {
            pix3[0] = 65535;
            pix3[1] = pix3[2] = 0;
         }
      }
   }

   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************
   Make an HDF (high depth of field) image from two images of the same           v.8.0
   subject with different focus settings, near and far. One image has
   the nearer parts of the subject in sharp focus, the other image has
   the farther parts in focus. The output image is constructed from the
   sharpest pixels in each of the two input images. Minor differences 
   in image center, rotation and size are automatically compensated.
**************************************************************************/

void *   HDF_align_thread(void *);
void     HDF_combine(int Fcolor);
void     HDF_distort();
void     HDF_mousefunc();
int      HDF_dialog_event(zdialog *zd, const char *event);
int      HDF_dialog_compl(zdialog *zd, int zstat);

RGB      *A2rgb48cache = 0;
int      HDF_align_stat = 0;
int      HDF_distort_busy = 0;
double   HDF_zoffx[4], HDF_zoffxB[4];
double   HDF_zoffy[4], HDF_zoffyB[4];
int      HDF_image;
int      HDF_brush;
int      HDF_suspend;


//  menu function

void m_HDF(GtkWidget *, const char *)
{
   char     *file2 = 0;

   if (! edit_setup(0,0)) return;                                          //  setup edit: no preview
   
   file2 = zgetfile(ZTX("Select image to combine"),image_file,"open");     //  get 2nd HDF image
   if (! file2) {
      edit_cancel();
      return;
   }

   Grgb48 = image_load(file2,48);                                          //  load and validate image
   if (! Grgb48) {
      edit_cancel();
      return;
   }

   Gww = Grgb48->ww;
   Ghh = Grgb48->hh;

   start_thread(HDF_align_thread,0);                                       //  start thread to align images
   wrapup_thread(0);                                                       //  wait for thread exit

   if (HDF_align_stat != 1) {
      edit_cancel();                                                       //  failure
      return;
   }   
   
   HDF_combine(1);                                                         //  combine with color comp.
   Fmodified = 1;

   zdedit = zdialog_new(ZTX("Retouch Image"),mWin,Bdone,Bcancel,null);     //  dialog for retouch
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"radio","radio1","hb1","image 1");
   zdialog_add_widget(zdedit,"radio","radio2","hb1","image 2","space=10");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=3");
   zdialog_add_widget(zdedit,"label","labr","hb2","brush","space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","1|199|1|20");
   zdialog_add_widget(zdedit,"button","susp-resm","hb2",Bsuspend,"space=10");
   
   zdialog_stuff(zdedit,"radio1",1);
   HDF_image = 1;
   HDF_brush = 10;
   HDF_suspend = 0;

   mouseCBfunc = HDF_mousefunc;                                            //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks
   gdk_window_set_cursor(drWin->window,0);                                 //  set normal cursor

   zdialog_run(zdedit,HDF_dialog_event,HDF_dialog_compl);                  //  run dialog, parallel
   return;
}


//  dialog event function

int HDF_dialog_event(zdialog *zd, const char *event)
{
   int      ii;

   zdialog_fetch(zd,"radio1",ii);
   if (ii) HDF_image = 1;
   else HDF_image = 2;
   
   if (strEqu(event,"radius"))
      zdialog_fetch(zd,"radius",HDF_brush);

   if (strEqu(event,"susp-resm"))                                          //  toggle suspend / resume
   {
      if (HDF_suspend) {
         HDF_suspend = 0;
         paint_toparc(3);                                                  //  start brush outline       v.8.3
         mouseCBfunc = HDF_mousefunc;                                      //  connect mouse function
         Mcapture++;
         gdk_window_set_cursor(drWin->window,0);                           //  set normal cursor
         zdialog_stuff(zd,"susp-resm",Bsuspend);
      }
      else  {
         HDF_suspend = 1;
         paint_toparc(2);                                                  //  stop brush outline        v.8.3
         mouseCBfunc = 0;                                                  //  disconnect mouse function
         Mcapture = 0;
         zdialog_stuff(zd,"susp-resm",Bresume);
      }
   }

   return 1;
}


//  dialog completion function

int HDF_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat != 1) edit_cancel();                                          //  user cancel
   else edit_done();
   RGB_free(A1rgb48);                                                      //  free memory
   RGB_free(A2rgb48);
   paint_toparc(2);                                                        //  stop brush outline        v.8.3
   Mcapture = 0;                                                           //  disconnect mouse
   mouseCBfunc = 0;
   return 0;
}


//  dialog mouse function

void HDF_mousefunc()
{
   uint16      vpixI[3], *ppix3;
   int         radius, radius2, vstat;
   int         mx, my, dx, dy, px3, py3;
   int         red, green, blue, max;
   double      pxI, pyI, f1;
   double      sintf = sin(toff), costf = cos(toff);

   radius = HDF_brush;
   radius2 = radius * radius;

   toparcx = Mxposn - radius;                                              //  paint brush outline circle
   toparcy = Myposn - radius;                                              //  v.8.3
   toparcw = toparch = 2 * radius;
   toparc = 1;
   paint_toparc(3);

   if (LMclick) {                                                          //  mouse click
      mx = Mxclick;
      my = Myclick;
   }

   else if (Mxdrag || Mydrag) {                                            //  drag in progress
      mx = Mxdrag;
      my = Mydrag;
   }
   
   else return;

   LMclick = RMclick = 0;

   if (mx < 0 || mx > E3ww-1 || my < 0 || my > E3hh-1)                     //  outside image area
      return;

   for (dy = -radius; dy <= radius; dy++)                                  //  loop surrounding block of pixels
   for (dx = -radius; dx <= radius; dx++)
   {
      px3 = mx + dx;
      py3 = my + dy;
      
      if (px3 < 0 || px3 > E3ww-1) continue;                               //  outside image
      if (py3 < 0 || py3 > E3hh-1) continue;
      if (dx*dx + dy*dy > radius2) continue;                               //  outside radius

      if (HDF_image == 1) {
         pxI = costf * px3 - sintf * (py3 - yoff);                         //  image1 virtual pixel
         pyI = costf * py3 + sintf * (px3 - xoff);
         vstat = vpixel(A1rgb48,pxI,pyI,vpixI);
         if (! vstat) continue;
         red = int(R12match[vpixI[0]]);                                    //  compensate color
         green = int(G12match[vpixI[1]]);
         blue = int(B12match[vpixI[2]]);
      }

      else {
         pxI = costf * (px3 - xoff) + sintf * (py3 - yoff);                //  image2 virtual pixel
         pyI = costf * (py3 - yoff) - sintf * (px3 - xoff);
         vstat = vpixel(A2rgb48,pxI,pyI,vpixI);
         if (! vstat) continue;
         red = int(R21match[vpixI[0]]);
         green = int(G21match[vpixI[1]]);
         blue = int(B21match[vpixI[2]]);
      }

      if (red > 65535 || green > 65535 || blue > 65535) {                  //  fix overflow
         max = red;
         if (green > max) max = green;
         if (blue > max) max = blue;
         f1 = 65535.0 / max;
         red = int(red * f1);
         green = int(green * f1);
         blue = int(blue * f1);
      }
      
      ppix3 = bmpixel(E3rgb48,px3,py3);                                    //  image3 real pixel
      ppix3[0] = red;
      ppix3[1] = green;
      ppix3[2] = blue;
   }
   
   mwpaint2();
   return;
}


//  image align thread, combine Frgb48 + Grgb48 >> E3rgb48

void * HDF_align_thread(void *)
{
   double      xfL, xfH, yfL, yfH, xystep, xylim;
   double      tfL, tfH, tstep, tlim;
   double      zlim, zoffx0, zoffy0;
   double      xyrange;
   double      eighth = 0.7854;                                            //  1/8 of circle in radians
   int         ii, jj, lastpass;

   HDF_align_stat = 0;                                                     //  no status yet
   Radjust = Gadjust = Badjust = 1.0;                                      //  no manual color adjustments
   Nalign = 1;                                                             //  alignment in progress
   aligntype = 2;                                                          //  HDF
   pixsamp = 10000;                                                        //  pixel sample size
   showRedpix = 1;                                                         //  highlight alignment pixels
   Fzoom = 0;                                                              //  fit to window if big
   Fblowup = 1;                                                            //  scale up to window if small
   xoff = yoff = toff = 0;                                                 //  initial offsets = 0
   for (ii = 0; ii < 4; ii++)                                              //  initial distortions = 0
      HDF_zoffx[ii] = HDF_zoffy[ii] = 0;

   A1rgb48 = A2rgb48 = A2rgb48cache = 0;

   fullSize = Fww;                                                         //  full image size
   if (Fhh > Fww) fullSize = Fhh;                                          //  (largest dimension)

   alignSize = 200;                                                        //  initial alignment image size
   if (alignSize > fullSize) alignSize = fullSize;

   lastpass = 0;
   xyrange = 0.05 * alignSize;                                             //  first pass, huge search range
   xshrink = yshrink = xyrange;                                            //  image shrink from distortion  v.8.1
   warpxu = warpyu = warpxl = warpyl = 0;                                  //  no warp factors (pano)
   warpxuB = warpyuB = warpxlB = warpylB = 0;

   while (true)                                                            //  next alignment stage / image size
   {   
      A1ww = Fww * alignSize / fullSize;                                   //  align width, height in same ratio
      A1hh = Fhh * alignSize / fullSize;
      A2ww = A1ww;
      A2hh = A1hh;
      
      RGB_free(A1rgb48);
      RGB_free(A2rgb48cache);

      if (alignSize < fullSize) {
         A1rgb48 = RGB_rescale(Frgb48,A1ww,A1hh);                          //  alignment images are
         A2rgb48cache = RGB_rescale(Grgb48,A2ww,A2hh);                     //    down-scaled input images
      }
      else {
         A1rgb48 = RGB_copy(Frgb48);                                       //  full size, copy input images
         A2rgb48cache = RGB_copy(Grgb48);
      }

      RGB_free(A2rgb48);                                                   //  distort image2 using current
      HDF_distort();                                                       //    zoffx/y settings

      mutex_lock(&pixmaps_lock);
      RGB_free(E3rgb48);                                                   //  prepare new output RGB pixmap
      E3rgb48 = RGB_make(A1ww,A1hh,48);
      E3ww = A1ww;
      E3hh = A1hh;
      mutex_unlock(&pixmaps_lock);

      alignWidth = A1ww;
      alignHeight = A1hh;
      getAlignArea();                                                      //  get image align area
      getBrightRatios();                                                   //  get image brightness ratios
      setColorfixFactors(1);                                               //  compute color matching factors
      flagEdgePixels();                                                    //  flag high-contrast pixels

      HDF_combine(0);                                                      //  combine and update window

      xylim = xyrange;                                                     //  xy search range, pixels
      xystep = 0.5 * xyrange;                                              //    -1.0 -0.5 0.0 +0.5 +1.0
      tlim = xylim / alignSize;                                            //  theta search range, radians
      tstep = xystep / alignSize;                                          //  step size
      zlim = xylim;                                                        //  distortion search range

//  find best alignment based on xoff, yoff, toff

      matchB = 0;      

      xfL = xoff - xylim;                                                  //  set x, y, theta search ranges
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test all offset dimensions
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = matchImages();
         if (sigdiff(matchlev,matchB,0.00001) > 0) {                       //  remember best match
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
            HDF_combine(0);                                                //  combine and update window
         }

         Nalign++;                                                         //  count alignment tests
         SBupdate++;                                                       //  update status bar
      }

      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;

//  find best distortion settings at the four corners

      for (int mpass = 1; mpass <= 2; mpass++)
      {
         for (ii = 0; ii < 4; ii++)                                        //  corner NW NE SE SW
         {
            HDF_zoffxB[ii] = HDF_zoffx[ii];                                //  save baseline match level
            HDF_zoffyB[ii] = HDF_zoffy[ii];
         
            zoffx0 = HDF_zoffx[ii];                                        //  current setting
            zoffy0 = HDF_zoffy[ii];

            for (jj = 0; jj < 8; jj++)                                     //  8 positions around current
            {                                                              //     distortion setting
               HDF_zoffx[ii] = zoffx0 + zlim * cos(eighth * jj);
               HDF_zoffy[ii] = zoffy0 + zlim * sin(eighth * jj);

               RGB_free(A2rgb48);                                          //  distort image2
               HDF_distort();

               matchlev = matchImages();
               if (sigdiff(matchlev,matchB,0.00001) > 0) {                 //  remember best match
                  matchB = matchlev;
                  HDF_zoffxB[ii] = HDF_zoffx[ii];
                  HDF_zoffyB[ii] = HDF_zoffy[ii];
                  HDF_combine(0);                                          //  combine and update window
               }
               
               Nalign++;                                                   //  count alignment tests
               SBupdate++;                                                 //  update status bar
            }

            HDF_zoffx[ii] = HDF_zoffxB[ii];                                //  recover best offset
            HDF_zoffy[ii] = HDF_zoffyB[ii];
         }
      }

// set up for next pass

      if (lastpass) break;                                                 //  done

      if (alignSize == fullSize) {                                         //  full size image was aligned
         lastpass++;                                                       //  one more pass
         xyrange = 0.5;
         continue;
      }

      double R = alignSize;
      alignSize = 1.3 * alignSize;                                         //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      R = alignSize / R;                                                   //  ratio of new / old image size
      xoff = R * xoff;                                                     //  adjust offsets for image size
      yoff = R * yoff;
      for (ii = 0; ii < 4; ii++) {
         HDF_zoffx[ii] = R * HDF_zoffx[ii];
         HDF_zoffy[ii] = R * HDF_zoffy[ii];
      }

      xyrange = 0.7 * xyrange;                                             //  reduce search range
      if (xyrange < 1) xyrange = 1;
      xshrink = yshrink = xyrange;                                         //  image shrink from distortion  v.8.1
   }
   
   RGB_free(A2rgb48cache);                                                 //  free memory
   RGB_free(Grgb48);                                                       //  A1/A2rgb48 still needed

   Fmodified = 1;                                                          //  image is modified
   showRedpix = 0;                                                         //  stop red pixel highlights
   Fblowup = 0;                                                            //  reset forced image scaling
   Nalign = 0;                                                             //  reset align counter
   zfree(redpixels);                                                       //  free edge-pixel flags
   redpixels = 0;
   HDF_align_stat = 1;                                                     //  signal success
   exit_thread();
   return 0;                                                               //  not executed, stop g++ warning
}


//  Combine images A1rgb48 and A2rgb48 using brightness adjustments.
//  Output is to E3rgb48 (not reallocated). Update main window.

void HDF_combine(int Fcolor)
{
   int         px3, py3, ii, vstat1, vstat2;
   int         red1, green1, blue1, red2, green2, blue2, max;
   double      px1, py1, px2, py2, f1;
   double      sintf = sin(toff), costf = cos(toff);
   uint16      vpix1[3], vpix2[3], *pix3;
   
   Radjust = Gadjust = Badjust = 1.0;

   for (py3 = 1; py3 < E3hh-1; py3++)                                      //  step through output pixels
   for (px3 = 1; px3 < E3ww-1; px3++)
   {
      px1 = costf * px3 - sintf * (py3 - yoff);                            //  A1rgb48 pixel, after offsets
      py1 = costf * py3 + sintf * (px3 - xoff);
      vstat1 = vpixel(A1rgb48,px1,py1,vpix1);

      px2 = costf * (px3 - xoff) + sintf * (py3 - yoff);                   //  corresponding A2rgb48 pixel
      py2 = costf * (py3 - yoff) - sintf * (px3 - xoff);
      vstat2 = vpixel(A2rgb48,px2,py2,vpix2);

      pix3 = bmpixel(E3rgb48,px3,py3);                                     //  output pixel

      if (! vstat1 || ! vstat2) {                                          //  no overlap
         pix3[0] = pix3[1] = pix3[2] = 0;                                  //  output pixel is black
         continue;
      }

      if (showRedpix) {                                                    //  show alignment pixels in red
         ii = py3 * A1ww + px3;
         if (redpixels[ii]) {
            pix3[0] = 65535;
            pix3[1] = pix3[2] = 0;
            continue;
         }
      }

      red1 = vpix1[0];                                                     //  image1 and image2 pixels
      green1 = vpix1[1];
      blue1 = vpix1[2];

      red2 = vpix2[0];
      green2 = vpix2[1];
      blue2 = vpix2[2];

      if (Fcolor)
      {
         red1 = int(R12match[red1]);                                       //  compensate color
         green1 = int(G12match[green1]);
         blue1 = int(B12match[blue1]);

         red2 = int(R21match[red2]);
         green2 = int(G21match[green2]);
         blue2 = int(B21match[blue2]);

         if (red1 > 65535 || green1 > 65535 || blue1 > 65535) {            //  fix overflow
            max = red1;
            if (green1 > max) max = green1;
            if (blue1 > max) max = blue1;
            f1 = 65535.0 / max;
            red1 = int(red1 * f1);
            green1 = int(green1 * f1);
            blue1 = int(blue1 * f1);
         }

         if (red2 > 65535 || green2 > 65535 || blue2 > 65535) {
            max = red2;
            if (green2 > max) max = green2;
            if (blue2 > max) max = blue2;
            f1 = 65535.0 / max;
            red2 = int(red2 * f1);
            green2 = int(green2 * f1);
            blue2 = int(blue2 * f1);
         }
      }
      
      pix3[0] = (red1 + red2) / 2;                                         //  output = combined inputs
      pix3[1] = (green1 + green2) / 2;
      pix3[2] = (blue1 + blue2) / 2;
   }

   mwpaint2();                                                             //  update window
   return;
}


//  Distort A2rgb48cache, returning A2rgb48
//  4 corners move HDF_zoffx[ii], HDF_zoffy[ii] pixels
//  and center does not move, 

void HDF_distort()
{
   void * HDF_distort_wthread(void *arg);

   RGB         *rgbin, *rgbout;
   int         ii, ww, hh;

   rgbin = A2rgb48cache;
   ww = rgbin->ww;                                                         //  create output RGB pixmap
   hh = rgbin->hh;
   rgbout = RGB_make(ww,hh,48);
   A2rgb48 = rgbout;
   
   for (ii = 0; ii < NWthreads; ii++)                                      //  start worker threads
      start_detached_thread(HDF_distort_wthread,&wtindex[ii]);
   zadd_locked(HDF_distort_busy,+NWthreads);

   while (HDF_distort_busy)                                                //  wait for completion
      zsleep(0.01);

   return;
}

void * HDF_distort_wthread(void *arg)                                      //  worker thread function
{
   int         index = *((int *) arg);
   int         pxm, pym, ww, hh, vstat;
   double      diag, px, py, dispx, dispy, dispx0, dispy0;
   double      disp0, disp1, disp2, disp3;
   double      disp0x, disp1x, disp2x, disp3x;
   double      disp0y, disp1y, disp2y, disp3y;
   uint16      vpix[3], *pixm;
   RGB         *rgbin, *rgbout;
   
   rgbin = A2rgb48cache;
   rgbout = A2rgb48;

   ww = rgbin->ww;                                                         //  create output RGB pixmap
   hh = rgbin->hh;
   diag = sqrt(ww*ww + hh*hh);
   
   pxm = ww/2;                                                             //  center pixel
   pym = hh/2;
   
   disp0 = (1 - pxm/diag) * (1 - pym/diag);
   disp0 = disp0 * disp0;
   disp0x = disp0 * HDF_zoffx[0];
   disp0y = disp0 * HDF_zoffy[0];
   
   disp1 = (1 - (ww-pxm)/diag) * (1 - pym/diag);
   disp1 = disp1 * disp1;
   disp1x = disp1 * HDF_zoffx[0];
   disp1y = disp1 * HDF_zoffy[0];

   disp2 = (1 - (ww-pxm)/diag) * (1 - (hh-pym)/diag);
   disp2 = disp2 * disp2;
   disp2x = disp2 * HDF_zoffx[0];
   disp2y = disp2 * HDF_zoffy[0];
   
   disp3 = (1 - pxm/diag) * (1 - (hh-pym)/diag);
   disp3 = disp3 * disp3;
   disp3x = disp3 * HDF_zoffx[0];
   disp3y = disp3 * HDF_zoffy[0];
   
   dispx = +disp0x - disp1x - disp2x + disp3x;                             //  center pixel displacement
   dispy = +disp0y + disp1y - disp2y - disp3y;
   
   dispx0 = -dispx;                                                        //  anti-displacement
   dispy0 = -dispy;

   for (pym = index; pym < hh; pym += NWthreads)                           //  loop all pixels
   for (pxm = 0; pxm < ww; pxm++)
   {
      disp0 = (1 - pxm/diag) * (1 - pym/diag);
      disp0 = disp0 * disp0;
      disp0x = disp0 * HDF_zoffx[0];
      disp0y = disp0 * HDF_zoffy[0];
      
      disp1 = (1 - (ww-pxm)/diag) * (1 - pym/diag);
      disp1 = disp1 * disp1;
      disp1x = disp1 * HDF_zoffx[0];
      disp1y = disp1 * HDF_zoffy[0];

      disp2 = (1 - (ww-pxm)/diag) * (1 - (hh-pym)/diag);
      disp2 = disp2 * disp2;
      disp2x = disp2 * HDF_zoffx[0];
      disp2y = disp2 * HDF_zoffy[0];
      
      disp3 = (1 - pxm/diag) * (1 - (hh-pym)/diag);
      disp3 = disp3 * disp3;
      disp3x = disp3 * HDF_zoffx[0];
      disp3y = disp3 * HDF_zoffy[0];
      
      dispx = +disp0x - disp1x - disp2x + disp3x;                          //  (pxm,pym) displacement
      dispy = +disp0y + disp1y - disp2y - disp3y;
      
      dispx += dispx0;                                                     //  relative to center pixel
      dispy += dispy0;

      px = pxm + dispx;                                                    //  source pixel location
      py = pym + dispy;

      vstat = vpixel(rgbin,px,py,vpix);                                    //  input virtual pixel
      pixm = bmpixel(rgbout,pxm,pym);                                      //  output real pixel

      if (vstat) {  
         pixm[0] = vpix[0];
         pixm[1] = vpix[1];
         pixm[2] = vpix[2];
      }
      else pixm[0] = pixm[1] = pixm[2] = 0;
   }
   
   zadd_locked(HDF_distort_busy,-1);
   pthread_exit(0);
}


/**************************************************************************/

//  panorama function - combine left and right images into a wide image

void     pano_prealign();                                                  //  manual pre-align
void     pano_autolens();                                                  //  auto optimize lens parameters
void *   pano_align_thread(void *);                                        //  auto align images
void     pano_final_adjust();                                              //  manual final adjustment
void     pano_get_align_images(int newf, int strf);                        //  scale and curve images for align
void     pano_combine(int fcolor);                                         //  combine images

int      pano_stat;                                                        //  dialog and thread status
int      pano_automatch;                                                   //  auto color matching on/off
RGB      *pano_A1cache, *pano_A2cache;                                     //  cached alignment images
double   pano_curve, pano_bow;                                             //  converted lens parameters


void m_pano(GtkWidget *, const char *)
{
   char        *file2 = 0;

   Grgb48 = A1rgb48 = A2rgb48 = pano_A1cache = pano_A2cache = 0;

   if (! edit_setup(0,0)) return;                                          //  setup edit: no preview
   
   file2 = zgetfile(ZTX("Select image to combine"),image_file,"open");     //  2nd or next pano file
   if (! file2) goto pano_cancel;

   Grgb48 = image_load(file2,48);                                          //  load and validate image
   if (! Grgb48) goto pano_cancel;
   Gww = Grgb48->ww;
   Ghh = Grgb48->hh;
   
   xoff = yoff = xoffB = yoffB = toff = toffB = 0;                         //  initial offsets
   warpxu = warpyu = warpxl = warpyl = 0;                                  //  initial warp factors
   warpxuB = warpyuB = warpxlB = warpylB = 0;

   pano_prealign();                                                        //  do manual pre-align
   if (pano_stat != 1) goto pano_cancel;

   if (overlapixs < alignSize * 20) {                                      //  need > 20 pixel overlap
      zmessageACK(ZTX("Too little overlap, cannot align"));
      goto pano_cancel;
   }

   start_thread(pano_align_thread,0);                                      //  start thread to align images
   wrapup_thread(0);                                                       //  wait for thread exit
   if (pano_stat != 1) goto pano_cancel;
   
   pano_final_adjust();                                                    //  do manual final adjustments
   if (pano_stat != 1) goto pano_cancel;
   Fmodified = 1;
   edit_done();
   goto pano_cleanup;
   
pano_cancel:
   edit_cancel();

pano_cleanup:
   Nalign = 0;
   if (file2) zfree(file2);
   RGB_free(Grgb48);
   RGB_free(A1rgb48);
   RGB_free(A2rgb48);
   RGB_free(pano_A1cache);
   RGB_free(pano_A2cache);
   return;
}


//  perform manual pre-align of image2 to image1
//  return offsets: xoff, yoff, toff
//  lens_mm and lens_bow may also be altered

void pano_prealign()   
{  
   int      pano_prealign_event(zdialog *zd, const char *event);           //  dialog event function
   int      pano_prealign_compl(zdialog *zd, int zstat);                   //  dialog completion function
   void *   pano_prealign_thread(void *);                                  //  working thread

   const char  *align_mess = ZTX("Drag right image into rough alignment with left \n"
                                      " to rotate, drag right edge up or down");
   const char  *proceed_mess = ZTX("Merge the images together");
   const char  *search_mess = ZTX("Auto-search lens mm and bow");

   zdedit = zdialog_new(ZTX("Pre-align Images"),mWin,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",align_mess,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"spin","spmm","hb1","22|200|0.1|35","space=5");
   zdialog_add_widget(zdedit,"label","labmm","hb1",ZTX("lens mm"));        //  fix translation   v.8.4.1
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"spin","spbow","hb2","-9|9|0.01|0","space=5");
   zdialog_add_widget(zdedit,"label","labbow","hb2",ZTX("lens bow"));
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","proceed","hb3",Bproceed,"space=5");
   zdialog_add_widget(zdedit,"label","labproceed","hb3",proceed_mess);
   zdialog_add_widget(zdedit,"hbox","hb4","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","search","hb4",Bsearch,"space=5");
   zdialog_add_widget(zdedit,"label","labsearch","hb4",search_mess);
   
   lens_mm = lens4_mm[curr_lens];                                          //  initz. curr. lens parameters
   lens_bow = lens4_bow[curr_lens];
   zdialog_stuff(zdedit,"spmm",lens_mm);
   zdialog_stuff(zdedit,"spbow",lens_bow);

   zdialog_run(zdedit,pano_prealign_event,pano_prealign_compl);            //  run dialog, parallel
   
   pano_stat = -1;
   start_thread(pano_prealign_thread,0);                                   //  start working thread
   wrapup_thread(0);                                                       //  wait for completion
   return;
}


int pano_prealign_event(zdialog *zd, const char *event)                    //  dialog event function
{
   if (strstr("spmm spbow",event)) {                                       //  revised lens data   v.7.8
      zdialog_fetch(zd,"spmm",lens_mm);
      zdialog_fetch(zd,"spbow",lens_bow);
   }

   if (strEqu(event,"search")) Fautolens = 1;                              //  trigger auto-lens function

   if (strEqu(event,"proceed")) {                                          //  proceed with pano
      pano_stat = 1;                                                       //  signal align success
      wrapup_thread(0);                                                    //  wait for thread exit
      zdialog_free(zdedit);                                                //  kill dialog
      zdedit = null;
   }
   return 0;
}


int pano_prealign_compl(zdialog *zd, int zstat)                            //  dialog completion function
{                                                                          //  (cancel only)
   pano_stat = 0;                                                          //  signal cancel
   wrapup_thread(0);                                                       //  wait for thread exit
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 0;
}


void * pano_prealign_thread(void *)                                        //  prealign working thread
{
   int         mx0, my0, mx, my;                                           //  mouse drag origin, position
   double      lens_mm0, lens_bow0;
   double      mlever = 0.5;
   double      dtoff;
   int         majupdate = 1;

   Radjust = Gadjust = Badjust = 1.0;                                      //  no manual color adjustments
   Nalign = 1;                                                             //  alignment in progress
   aligntype = 3;                                                          //  pano
   pixsamp = 5000;                                                         //  pixel sample size
   showRedpix = 0;                                                         //  no pixel highlight (yet)
   Fblowup = 1;                                                            //  scale-up small image to window

   fullSize = Ghh;                                                         //  full size = image2 height
   alignSize = int(pano_prealign_size);                                    //  prealign image size
   pano_get_align_images(1,0);                                             //  get prealign images and curve them

   xoff = xoffB = 0.8 * A1ww;                                              //  initial x offset (20% overlap)

   alignWidth = int(A1ww - xoff);                                          //  initial alignment on full overlap
   alignHeight = A1hh;
   getAlignArea();                                                         //  get image overlap area
   pano_combine(0);                                                        //  combine images

   lens_mm0 = lens_mm;                                                     //  to detect changes
   lens_bow0 = lens_bow;

   mx0 = my0 = 0;                                                          //  no drag in progress
   Mcapture = KBcapture = 1;                                               //  capture mouse drag and KB keys

   while (pano_stat == -1)                                                 //  loop and align until done
   {
      zsleep(0.05);                                                        //  logic simplified   v.7.5
      
      if (Fautolens) {
         pano_autolens();                                                  //  get lens parameters
         zdialog_stuff(zdedit,"spmm",lens_mm);                             //  update dialog
         zdialog_stuff(zdedit,"spbow",lens_bow);
         majupdate++;
      }
         
      if (lens_mm != lens_mm0 || lens_bow != lens_bow0) {                  //  change in lens parameters
         lens_mm0 = lens_mm;
         lens_bow0 = lens_bow;
         pano_get_align_images(0,0);
         majupdate++;
      }
      
      if (KBkey) {                                                         //  KB input
         if (KBkey == GDK_Left)  xoff -= 0.2;                              //  tweak alignment offsets
         if (KBkey == GDK_Right) xoff += 0.2;
         if (KBkey == GDK_Up)    yoff -= 0.2;
         if (KBkey == GDK_Down)  yoff += 0.2;
         if (KBkey == GDK_r)     toff += 0.0002;
         if (KBkey == GDK_l)     toff -= 0.0002;
         KBkey = 0;

         alignWidth = int(A1ww - xoff);                                    //  entire overlap
         alignHeight = A1hh;                                               //  v.8.0
         getAlignArea();
         pano_combine(0);                                                  //  show combined images
         majupdate++;
         continue;
      }
      
      if (! Mxdrag && ! Mydrag) mx0 = my0 = 0;                             //  no drag in progress

      if (Mxdrag || Mydrag)                                                //  mouse drag underway
      {
         mx = Mxdrag;                                                      //  mouse position in image
         my = Mydrag;

         if (mx < xoff || mx > xoff + A2ww || my > E3hh) {                 //  if mouse not in image2 area,
            mx0 = my0 = Mxdrag = Mydrag = 0;                               //    no drag in progress
            continue;
         }
         
         if (! mx0 && ! my0) {                                             //  new drag, set drag origin
            mx0 = mx;
            my0 = my;
         }
         
         if (mx != mx0 || my != my0)                                       //  drag is progressing
         {
            if (mx > xoff + 0.8 * A2ww) {                                  //  near right edge, theta drag 
               dtoff = mlever * (my - my0) / A2ww;                         //  delta theta, radians
               toff += dtoff;
               xoff += dtoff * (A1hh + yoff);                              //  change center of rotation
               yoff -= dtoff * (A1ww - xoff);                              //    to middle of overlap area
            }
            else  {                                                        //  x/y drag
               xoff += mlever * (mx - mx0);                                //  image2 offsets / mouse leverage
               yoff += mlever * (my - my0);
            }

            mx0 = mx;                                                      //  next drag origin = current mouse
            my0 = my;

            if (xoff > A1ww) xoff = A1ww;                                  //  limit nonsense
            if (xoff < 0.3 * A1ww) xoff = 0.3 * A1ww;
            if (yoff < -0.5 * A2hh) yoff = -0.5 * A2hh;
            if (yoff > 0.5 * A1hh) yoff = 0.5 * A1hh;
            if (toff < -0.20) toff = -0.20;
            if (toff > 0.20) toff = 0.20;

            alignWidth = int(A1ww - xoff);                                 //  entire overlap
            getAlignArea();
            pano_combine(0);                                               //  show combined images
            majupdate++;
            continue;
         }
      }

      if (majupdate) {                                                     //  do major update
         majupdate = 0;
         alignWidth = int(A1ww - xoff);                                    //  entire overlap
         getAlignArea();                                                   //  get image overlap area
         getBrightRatios();                                                //  get color brightness ratios
         setColorfixFactors(1);                                            //  set color matching factors
         flagEdgePixels();                                                 //  flag edge pixels in overlap 
         matchB = matchImages();                                           //  match images
         xoffB = xoff;                                                     //  new base alignment data
         yoffB = yoff;
         toffB = toff;
         pano_combine(0);                                                  //  show combined images
         SBupdate++;                                                       //  update status bar
      }
   }

   Fzoom = 0;                                                              //  reset in case user zoomed in
   KBcapture = Mcapture = 0;
   exit_thread();
   return 0;                                                               //  never executed, stop g++ warning
}


//  optimize lens parameters
//  assumes a good starting point since search ranges are limited
//  inputs and outputs: lens_mm, lens_bow, xoff, yoff, toff

void pano_autolens()
{
   double   mm_range, bow_range, xoff_range, yoff_range, toff_range;
   double   squeeze, xoff_rfinal, rnum;
   double   lens_mmB, lens_bowB;
   int      counter = 0;
   
   mm_range = 0.2 * lens_mm;                                               //  set initial search ranges
   bow_range = 0.5 * lens_bow;
   if (bow_range < 1) bow_range = 1;
   xoff_range = 7;
   yoff_range = 7;
   toff_range = 0.01;

   xoff_rfinal = 0.2;                                                      //  final xoff range - when to quit
   
   Nalign = 1;
   aligntype = 3;
   showRedpix = 1;
   pano_get_align_images(0,0);
   alignWidth = int(A1ww - xoff);
   if (alignWidth > A2ww/3) alignWidth = A2ww/3;
   alignHeight = A1hh - abs(yoff);
   alignWidth = 0.9 * alignWidth;                                          //  do not change align pixels
   alignHeight = 0.9 * alignHeight;                                        //   when align parms change  v.8.1
   getAlignArea();
   getBrightRatios();
   setColorfixFactors(1);
   flagEdgePixels();
   pano_combine(0);

   lens_mmB = lens_mm;                                                     //  initial best fit = current data
   lens_bowB = lens_bow;
   xoffB = xoff;
   yoffB = yoff;
   toffB = toff;
   matchB = matchImages();

   while (true)
   {
      srand48(time(0) + counter++);
      lens_mm = lens_mmB + mm_range * (drand48() - 0.5);                   //  new random lens factors
      lens_bow = lens_bowB + bow_range * (drand48() - 0.5);                //     within search range
      pano_get_align_images(0,0);                                          //  curve images
      getAlignArea();                                                      //  synch align data   v.8.5
      getBrightRatios();
      setColorfixFactors(1);
      flagEdgePixels();
      squeeze = 0.95;                                                      //  search range reduction
         
      for (int ii = 0; ii < 500; ii++)                                     //  loop random alignments  v.8.5
      {                                                                    
         rnum = drand48();
         if (rnum < 0.33)                                                  //  random change some alignment offset 
            xoff = xoffB + xoff_range * (drand48() - 0.5);                 //    within search range
         else if (rnum < 0.67)
            yoff = yoffB + yoff_range * (drand48() - 0.5);
         else
            toff = toffB + toff_range * (drand48() - 0.5);
      
         matchlev = matchImages();                                         //  test quality of image alignment
         if (sigdiff(matchlev,matchB,0.0001) > 0) {
            lens_mmB = lens_mm;                                            //  better
            lens_bowB = lens_bow;
            xoffB = xoff;                                                  //  (no refresh align pixels v.8.1)
            yoffB = yoff;
            toffB = toff;
            matchB = matchlev;                                             //  save new best fit
            pano_combine(0);
            squeeze = 1;                                                   //  keep same search range as long
            break;                                                         //    as improvements are found
         }

         Nalign++;
         SBupdate++;
         if (pano_stat != -1) goto done;
      }
      
      if (xoff_range < xoff_rfinal) goto done;

      mm_range = squeeze * mm_range;                                       //  reduce search range if no 
      if (mm_range < 0.02 * lens_mmB) mm_range = 0.02 * lens_mmB;          //    improvements were found
      bow_range = squeeze * bow_range;
      if (bow_range < 0.1 * lens_bowB) bow_range = 0.1 * lens_bowB;
      if (bow_range < 0.2) bow_range = 0.2;
      xoff_range = squeeze * xoff_range;
      yoff_range = squeeze * yoff_range;
      toff_range = squeeze * toff_range;
   }

done:
   lens_mm = lens_mmB;                                                     //  set best alignment found
   lens_bow = lens_bowB;
   xoff = xoffB;
   yoff = yoffB;
   toff = toffB;
   Fautolens = 0;
   showRedpix = 0;
   pano_combine(0);
   SBupdate++;
   return;
}


//  Thread function for combining A1 + A2 >> E3

void * pano_align_thread(void *)
{
   int         firstpass, lastpass;
   double      xystep, xylim, tstep, tlim;
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   double      wxL, wxH, wyL, wyH;
   double      alignR;
   
   Nalign = 1;                                                             //  alignment in progress
   aligntype = 3;                                                          //  pano
   pano_stat = 0;
   Fzoom = 0;                                                              //  fit to window if big
   Fblowup = 1;                                                            //  scale up to window if small
   showRedpix = 1;                                                         //  highlight alignment pixels
   firstpass = 1;
   lastpass = 0;

   while (true)
   {
      alignR = alignSize;                                                  //  from pre-align or prior pass
      if (firstpass) alignSize = 140;                                      //  set next align size
      else if (lastpass) alignSize = fullSize;
      else  alignSize = int(pano_image_increase * alignSize);              //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      alignR = alignSize / alignR;                                         //  ratio of new / old image size

      xoff = alignR * xoff;                                                //  adjust offsets for new image size
      yoff = alignR * yoff;
      toff = toff;

      warpxu = alignR * warpxu;                                            //  adjust warp values
      warpyu = alignR * warpyu;
      warpxl = alignR * warpxl;
      warpyl = alignR * warpyl;
      
      if (! lastpass) pano_get_align_images(1,0);                          //  get new alignment images

      if (firstpass) {
         alignWidth = int(A1ww - xoff);                                    //  set new alignment area
         if (alignWidth > A1ww/3) alignWidth = A1ww/3;
      }
      else  {
         alignWidth = int(alignR * alignWidth * pano_blend_decrease);
         if (alignWidth < pano_min_alignwidth * alignSize)                 //  keep within range
            alignWidth = int(pano_min_alignwidth * alignSize);
         if (alignWidth > pano_max_alignwidth * alignSize) 
            alignWidth = int(pano_max_alignwidth * alignSize);
      }

      alignHeight = A1hh;

      getAlignArea();                                                      //  get image overlap area
      getBrightRatios();                                                   //  get color brightness ratios
      setColorfixFactors(1);                                               //  set color matching factors
      flagEdgePixels();                                                    //  flag high-contrast pixels in blend

      xylim = 2;                                                           //  +/- search range, centered on
      xystep = 0.571;                                                      //    results from prior stage
      
      if (firstpass) {
         xylim = alignSize * 0.03;                                         //  3% error tolerance in pre-alignment
         xystep = 0.5; 
      }

      if (lastpass) {
         xylim = 1;                                                        //  final stage pixel search steps
         xystep = 0.5;                                                     //   -1.0 -0.5 0.0 +0.5 +1.0
      }

      tlim = xylim / alignSize / 2;                                        //  theta max offset, radians
      tstep = xystep / alignSize / 2;                                      //  theta step size

      xfL = xoff - xylim;                                                  //  set x/y/t search ranges, step sizes
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      xoffB = xoff;                                                        //  initial offsets = best so far
      yoffB = yoff;
      toffB = toff;
      
      warpxuB = warpxu;                                                    //  initial warp values
      warpyuB = warpyu;
      warpxlB = warpxl;
      warpylB = warpyl;

      matchB = matchImages();                                              //  set base match level
      pano_combine(0);                                                     //  v.8.4

      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test x, y, theta offsets
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all possible combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = matchImages();
         if (sigdiff(matchlev,matchB,0.00001) > 0) {                       //  remember best alignment and offsets
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
         }

         Nalign++;
         SBupdate++;                                                       //  update status bar
      }
      
      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;
      
      if (! firstpass)
      {
         wxL = warpxuB - xylim;                                            //  warp image2 corners    v.8.5
         wxH = warpxuB + xylim + xystep/2;                                 //  double range           v.8.6.1
         wyL = warpyuB - xylim;
         wyH = warpyuB + xylim + xystep/2;
         
         for (warpxu = wxL; warpxu < wxH; warpxu += xystep)                //  search upper warp
         for (warpyu = wyL; warpyu < wyH; warpyu += xystep)
         {
            pano_get_align_images(0,1);                                    //  curve and warp
            matchlev = matchImages();
            if (sigdiff(matchlev,matchB,0.00001) > 0) {                    //  remember best warp
               matchB = matchlev;
               warpxuB = warpxu;
               warpyuB = warpyu;
            }

            Nalign++;
            SBupdate++;
         }

         warpxu = warpxuB;                                                 //  restore best warp
         warpyu = warpyuB;
         pano_get_align_images(0,0);

         wxL = warpxlB - xylim;
         wxH = warpxlB + xylim + xystep/2;
         wyL = warpylB - xylim;
         wyH = warpylB + xylim + xystep/2;
         
         for (warpxl = wxL; warpxl < wxH; warpxl += xystep)                //  search lower warp
         for (warpyl = wyL; warpyl < wyH; warpyl += xystep)
         {
            pano_get_align_images(0,2);
            matchlev = matchImages();
            if (sigdiff(matchlev,matchB,0.00001) > 0) {
               matchB = matchlev;
               warpxlB = warpxl;
               warpylB = warpyl;
            }

            Nalign++;
            SBupdate++;
         }
         
         warpxl = warpxlB;
         warpyl = warpylB;
         pano_get_align_images(0,0);
      }

      pano_combine(0);                                                     //  combine images and update window

      firstpass = 0;
      if (lastpass) break;
      if (alignSize == fullSize) lastpass = 1;                             //  one more pass, reduced step size
   }

   pano_stat = 1;                                                          //  signal success
   showRedpix = 0;                                                         //  pixel highlights off
   Fzoom = Fblowup = 0;                                                    //  reset image scaling
   exit_thread();
   return 0;                                                               //  never executed, stop g++ warning
}


//  do manual adjustment of brightness, color, blend width

void pano_final_adjust()
{
   int      pano_adjust_event(zdialog *zd, const char *event);             //  dialog event function
   int      pano_adjust_compl(zdialog *zd, int zstat);                     //  dialog completion function
   void *   pano_adjust_thread(void *);
   
   int      zstat;

   const char  *adjmessage = ZTX("\n Match Brightness and Color");
   
   pano_automatch = 1;                                                     //  init. auto color match on
   alignWidth = 1;
   
   getAlignArea();
   getBrightRatios();                                                      //  get color brightness ratios
   setColorfixFactors(pano_automatch);                                     //  set color matching factors
   pano_combine(1);                                                        //  show final results

   zdedit = zdialog_new(ZTX("Match Images"),mWin,Bdone,Bcancel,null);      //  color adjustment dialog

   zdialog_add_widget(zdedit,"label","lab0","dialog",adjmessage,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");          //  match brightness and color
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog");                //  red           [ 100 ]
   zdialog_add_widget(zdedit,"label","lab1","vb1",Bred,"space=7");         //  green         [ 100 ]
   zdialog_add_widget(zdedit,"label","lab2","vb1",Bgreen,"space=7");       //  blue          [ 100 ]
   zdialog_add_widget(zdedit,"label","lab3","vb1",Bblue,"space=7");        //  brightness    [ 100 ]
   zdialog_add_widget(zdedit,"label","lab4","vb1",Bbrightness,"space=7");  //  blend width   [  0  ]
   zdialog_add_widget(zdedit,"label","lab5","vb1",Bblendwidth,"space=7");  //
   zdialog_add_widget(zdedit,"spin","spred","vb2","50|200|0.1|100");       //  [ apply ]  [ auto ]  off
   zdialog_add_widget(zdedit,"spin","spgreen","vb2","50|200|0.1|100");
   zdialog_add_widget(zdedit,"spin","spblue","vb2","50|200|0.1|100");
   zdialog_add_widget(zdedit,"spin","spbright","vb2","50|200|0.1|100");
   zdialog_add_widget(zdedit,"spin","spblend","vb2","1|200|1|1");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","apply","hb2",Bapply,"space=5");
   zdialog_add_widget(zdedit,"button","auto","hb2",ZTX("Auto"));
   zdialog_add_widget(zdedit,"label","labauto","hb2","on");
   
   zstat = zdialog_run(zdedit,pano_adjust_event,pano_adjust_compl);        //  run dialog, parallel
   
   start_thread(pano_adjust_thread,0);                                     //  start thread
   wrapup_thread(0);                                                       //  wait for completion
   return;
}


//  thread function - stall return from final_adjust until dialog done

void * pano_adjust_thread(void *)
{
   while (true) thread_idle_loop();                                        //  wait for work or exit request
   return 0;                                                               //  not executed, stop g++ warning
}


//  dialog completion function

int pano_adjust_compl(zdialog *zd, int zstat)
{
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   if (zstat == 1) pano_stat = 1;
   else pano_stat = 0;
   wrapup_thread(8);                                                       //  kill thread
   return 0;
}


//  dialog event function
//  A1 + A2 >> E3 under control of spin buttons

int pano_adjust_event(zdialog *zd, const char *event)
{
   double      red, green, blue, bright, bright2;
   
   if (strEqu(event,"auto")) 
   {
      pano_automatch = 1 - pano_automatch;                                 //  toggle color automatch state
      setColorfixFactors(pano_automatch);                                  //  corresp. color matching factors
      pano_combine(1);                                                     //  combine images and update window
      if (pano_automatch) zdialog_stuff(zd,"labauto","on");
      else zdialog_stuff(zd,"labauto","off");
      return 1;
   }

   if (strNeq(event,"apply")) return 0;                                    //  wait for apply button

   zdialog_fetch(zd,"spred",red);                                          //  get color adjustments
   zdialog_fetch(zd,"spgreen",green);
   zdialog_fetch(zd,"spblue",blue);
   zdialog_fetch(zd,"spbright",bright);                                    //  brightness adjustment
   zdialog_fetch(zd,"spblend",alignWidth);                                 //  align and blend width

   bright2 = (red + green + blue) / 3;                                     //  RGB brightness
   bright = bright / bright2;                                              //  bright setpoint / RGB brightness
   red = red * bright;                                                     //  adjust RGB brightness
   green = green * bright;
   blue = blue * bright;
   
   bright = (red + green + blue) / 3;
   zdialog_stuff(zd,"spred",red);                                          //  force back into consistency
   zdialog_stuff(zd,"spgreen",green);
   zdialog_stuff(zd,"spblue",blue);
   zdialog_stuff(zd,"spbright",bright);

   Radjust = red / 100;                                                    //  normalize 0.5 ... 2.0
   Gadjust = green / 100;
   Badjust = blue / 100;

   getAlignArea();
   getBrightRatios();                                                      //  get color brightness ratios
   setColorfixFactors(pano_automatch);                                     //  set color matching factors
   pano_combine(1);                                                        //  combine and update window

   return 1;
}


//  create scaled and curved alignment images in A1rgb48, A2rgb48
//  newf:  create new alignment images from scratch
//  warp:  0: curve both images, warp both halves of image2
//         1: curve image2 only, warp upper half only
//         2: curve image2 only, warp lower half only
//  global variables warpxu/yu and warpxl/yl determine the amount of warp

void pano_get_align_images(int newf, int warp)
{
   void  pano_curve_image(RGB *rgbin, RGB *rgbout, int curve, int warp);

   double      lens_curve, R;

   double   mm[12] =    { 20, 22, 25,   26,  28,  32,   40,   48,   60,  80,   100, 200 };
   double   curve[12] = { 35, 19, 10.1, 8.7, 6.8, 5.26, 3.42, 2.32, 1.7, 1.35, 1.2, 1.1 };

   if (newf) 
   {
      A1ww = Fww * alignSize / fullSize;                                   //  size of alignment images
      A1hh = Fhh * alignSize / fullSize;
      A2ww = Gww * alignSize / fullSize;
      A2hh = Ghh * alignSize / fullSize;
      
      RGB_free(pano_A1cache);                                              //  align images = scaled input images
      RGB_free(pano_A2cache);
      pano_A1cache = RGB_rescale(Frgb48,A1ww,A1hh);                        //  new scaled align images
      pano_A2cache = RGB_rescale(Grgb48,A2ww,A2hh);

      RGB_free(A1rgb48);                                                   //  A1/A2 will be curved
      RGB_free(A2rgb48);
      A1rgb48 = A2rgb48 = 0;

      spline1(12,mm,curve);                                                //  always initialize   bugfix v.6.2
   }
   
   lens_curve = spline2(lens_mm);
   pano_curve = lens_curve * 0.01 * A2ww;                                  //  curve % to pixels
   pano_bow = 0.01 * A2ww * lens_bow;                                      //  lens_bow % to pixels
   xshrink = 0.5 * pano_curve;                                             //  image shrinkage from curving
   yshrink = xshrink * pano_ycurveF * A2hh / A2ww;

   R = 1.0 * A2hh / A2ww;
   if (R > 1) pano_curve = pano_curve / R / R;                             //  adjust for vertical format
   
   if (warp == 0) {                                                        //  curve image1
      if (A2ww <= 0.8 * A1ww) {
         if (A1rgb48) RGB_free(A1rgb48);                                   //  already curved via prior pano
         A1rgb48 = RGB_copy(pano_A1cache);                                 //  make a copy
      }
      else {
         if (! A1rgb48) A1rgb48 = RGB_make(A1ww,A1hh,48);                  //  curve image1, both halves, no warp
         pano_curve_image(pano_A1cache,A1rgb48,2,0);
      }
   }

   if (! A2rgb48) A2rgb48 = RGB_make(A2ww,A2hh,48);

   if (warp == 0) pano_curve_image(pano_A2cache,A2rgb48,2,3);              //  curve and warp all image2
   if (warp == 1) pano_curve_image(pano_A2cache,A2rgb48,1,1);              //    ""  left-upper quadrant
   if (warp == 2) pano_curve_image(pano_A2cache,A2rgb48,1,2);              //    ""  left-lower quadrant

   return;
}


//  curve and warp alignment image
//    curve:   1: left half only   2: both halves
//    warp:    0: none  1: upper half  2: lower half  3: both
//
//  global variables:
//    warpxu/yu:    upper left corner warp displacement
//    warpxl/yl:    lower left corner warp displacement

void  pano_curve_image(RGB *rgbin, RGB *rgbout, int curve, int warp)       //  overhauled  v.8.5
{
   int         pxc, pyc, ww, hh, vstat;
   int         pycL, pycH, pxcL, pxcH;
   double      px, py, xdisp, ydisp, xpull, ypull;
   uint16      vpix[3], *pixc;
   
   ww = rgbout->ww;
   hh = rgbout->hh;
   
   pxcL = 0;
   pxcH = ww;

   if (curve == 1) {                                                       //  curve left half only
      pxcL = pxmL - xoff;                                                  //  from blend stripe to middle
      pxcH = pxmH - xoff;                                                  //  v.7.7
   }

   pycL = 0;
   pycH = hh;
   if (warp == 1) pycH = hh / 2;                                           //  upper half only
   if (warp == 2) pycL = hh / 2;                                           //  lower half only

   for (pyc = pycL; pyc < pycH; pyc++)
   for (pxc = pxcL; pxc < pxcH; pxc++)
   {
      xdisp = (pxc - ww/2.0) / (ww/2.0);                                   //  -1 ... 0 ... +1
      ydisp = (pyc - hh/2.0) / (ww/2.0);
      xpull = xdisp * xdisp * xdisp;
      ypull = pano_ycurveF * ydisp * xdisp * xdisp;

      px = pxc + pano_curve * xpull;                                       //  apply lens curve factor
      py = pyc + pano_curve * ypull;
      px -= pano_bow * xdisp * ydisp * ydisp;                              //  apply lens bow factor

      if (warp && xdisp < 0) {                                             //  bugfix                    v.8.6.1
         if (pyc < hh / 2) {                                               //  warp upper-left quadrant
            px += warpxu * xdisp;
            py += warpyu * ydisp * xdisp;                                  //  bugfix                    v.8.6.1
         }
         else {
            px += warpxl * xdisp;                                          //  warp lower-left
            py += warpyl * ydisp * xdisp;
         }
      }

      vstat = vpixel(rgbin,px,py,vpix);                                    //  input virtual pixel
      pixc = bmpixel(rgbout,pxc,pyc);                                      //  output real pixel
      if (vstat) {  
         pixc[0] = vpix[0];
         pixc[1] = vpix[1];
         pixc[2] = vpix[2];
      }
      else pixc[0] = pixc[1] = pixc[2] = 0;
   }
   
   return;
}


//  combine and images: A1rgb48 + A2rgb48  >>  E3rgb48
//  update window showing current progress

void pano_combine(int fcolor)
{
   int            px3, py3, ii, max, vstat1, vstat2;
   int            red1, green1, blue1;
   int            red2, green2, blue2;
   int            red3, green3, blue3;
   uint16         vpix1[3], vpix2[3], *pix3;
   double         ww, px1, py1, px2, py2, f1, f2;
   double         costf = cos(toff), sintf = sin(toff);

   mutex_lock(&pixmaps_lock);
   
   ww = xoff + A2ww;                                                       //  combined width
   if (toff < 0) ww -= A2hh * toff;                                        //  adjust for theta
   E3ww = int(ww+1);
   E3hh = A1rgb48->hh;
   
   RGB_free(E3rgb48);                                                      //  allocate output pixmap
   E3rgb48 = RGB_make(E3ww,E3hh,48);
   
   overlapixs = 0;                                                         //  counts overlapping pixels
   red1 = green1 = blue1 = 0;                                              //  suppress compiler warnings
   red2 = green2 = blue2 = 0;
   
   for (py3 = 0; py3 < E3hh; py3++)                                        //  step through E3 rows
   for (px3 = 0; px3 < E3ww; px3++)                                        //  step through E3 pixels in row
   {
      vstat1 = vstat2 = 0;
      red3 = green3 = blue3 = 0;

      if (px3 < pxmH) {
         px1 = costf * px3 - sintf * (py3 - yoff);                         //  A1 pixel, after offsets  
         py1 = costf * py3 + sintf * (px3 - xoff);
         vstat1 = vpixel(A1rgb48,px1,py1,vpix1);
      }

      if (px3 >= pxmL) {
         px2 = costf * (px3 - xoff) + sintf * (py3 - yoff);                //  A2 pixel, after offsets
         py2 = costf * (py3 - yoff) - sintf * (px3 - xoff);
         vstat2 = vpixel(A2rgb48,px2,py2,vpix2);
      }

      if (vstat1) {
         red1 = vpix1[0];
         green1 = vpix1[1];
         blue1 = vpix1[2];
         if (!red1 && !green1 && !blue1) vstat1 = 0;                       //  ignore black pixels
      }

      if (vstat2) {
         red2 = vpix2[0];
         green2 = vpix2[1];
         blue2 = vpix2[2];
         if (!red2 && !green2 && !blue2) vstat2 = 0;
      }

      if (fcolor) {                                                        //  brightness compensation  
         if (vstat1) {                                                     //  (auto + manual adjustments)
            red1 = int(R12match[red1]);
            green1 = int(G12match[green1]);
            blue1 = int(B12match[blue1]);
            if (red1 > 65535 || green1 > 65535 || blue1 > 65535) {
               max = red1;
               if (green1 > max) max = green1;
               if (blue1 > max) max = blue1;
               f1 = 65535.0 / max;
               red1 = int(red1 * f1);
               green1 = int(green1 * f1);
               blue1 = int(blue1 * f1);
            }
         }

         if (vstat2) {                                                     //  adjust both images
            red2 = int(R21match[red2]);                                    //    in opposite directions
            green2 = int(G21match[green2]);
            blue2 = int(B21match[blue2]);
            if (red2 > 65535 || green2 > 65535 || blue2 > 65535) {
               max = red2;
               if (green2 > max) max = green2;
               if (blue2 > max) max = blue2;
               f1 = 65535.0 / max;
               red2 = int(red2 * f1);
               green2 = int(green2 * f1);
               blue2 = int(blue2 * f1);
            }
         }
      }

      if (vstat1) {
         if (! vstat2) {
            red3 = red1;                                                   //  use image1 pixel
            green3 = green1;
            blue3 = blue1; 
         }
         else {
            overlapixs++;                                                  //  count overlapped pixels
            if (fcolor) {
               if (alignWidth == 0) f1 = 1.0;
               else f1 = 1.0 * (pxmH - px3) / alignWidth;                  //  use progressive blend
               f2 = 1.0 - f1;
               red3 = int(f1 * red1 + f2 * red2);
               green3 = int(f1 * green1 + f2 * green2);
               blue3 = int(f1 * blue1 + f2 * blue2);
            }
            else {                                                         //  use 50/50 mix
               red3 = (red1 + red2) / 2;
               green3 = (green1 + green2) / 2;
               blue3 = (blue1 + blue2) / 2;
            }
         }
      }

      else if (vstat2) {
         red3 = red2;                                                      //  use image2 pixel
         green3 = green2;
         blue3 = blue2; 
      }

      pix3 = bmpixel(E3rgb48,px3,py3);                                     //  output pixel
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
      
      if (showRedpix && vstat1 && vstat2) {                                //  highlight alignment pixels
         ii = py3 * A1ww + px3;
         if (redpixels[ii]) {
            pix3[0] = 65535;
            pix3[1] = pix3[2] = 0;
         }
      }
   }

   mutex_unlock(&pixmaps_lock);
   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************
    HDR and pano shared functions (image matching, alignment, overlay)
***************************************************************************/

//  compare two doubles for significant difference
//  return:  0  difference not significant
//          +1  d1 > d2
//          -1  d1 < d2

int sigdiff(double d1, double d2, double signf)
{
   double diff = fabs(d1-d2);
   if (diff == 0.0) return 0;
   diff = diff / (fabs(d1) + fabs(d2));
   if (diff < signf) return 0;
   if (d1 > d2) return 1;
   else return -1;
}


/**************************************************************************/

//  Get the rectangle containing the overlap region of two images
//  outputs: pxL pxH pyL pyH        total image overlap rectangle
//           pxmL pxmH pymL pymH    reduced to overlap align area

void getAlignArea()
{
   int         pxL2, pyL2;

   pxL = 0;
   if (xoff > 0) pxL = int(xoff);

   pxH = A1ww;
   if (pxH > xoff + A2ww) pxH = int(xoff + A2ww);
   
   pyL = 0;
   if (yoff > 0) pyL = int(yoff);
   
   pyH = A1hh;
   if (pyH > yoff + A2hh) pyH = int(yoff + A2hh);

   if (toff > 0) {
      pyL2 = int(yoff + toff * (pxH - pxL));
      if (pyL2 > pyL) pyL = pyL2;
   }

   if (toff < 0) {   
      pxL2 = int(xoff - toff * (pyH - pyL));
      if (pxL2 > pxL) pxL = pxL2;
   }
   
   if (xshrink > 0) {
      pxL = pxL + xshrink;                                                 //  reduce overlap area by amount
      pxH = pxH - xshrink;                                                 //   of image shrink (pano, HDF)
      pyL = pyL + yshrink;
      pyH = pyH - yshrink;
   }

   pxM = (pxL + pxH) / 2;                                                  //  midpoint of overlap
   pyM = (pyL + pyH) / 2;

   if (alignWidth < (pxH - pxL)) {
      pxmL = pxM - alignWidth/2;                                           //  overlap area width for
      pxmH = pxM + alignWidth/2;                                           //    image matching & blending
      if (pxmL < pxL) pxmL = pxL;
      if (pxmH > pxH) pxmH = pxH;
   }
   else {                                                                  //  use whole range
      pxmL = pxL;
      pxmH = pxH;
   }

   if (alignHeight < (pyH - pyL)) {
      pymL = pyM - alignHeight/2;
      pymH = pyM + alignHeight/2;
      if (pymL < pyL) pymL = pyL;
      if (pymH > pyH) pymH = pyH;
   }
   else {
      pymL = pyL;
      pymH = pyH;
   }

   return;
}


/**************************************************************************/

//  Compute brightness ratio by color for overlapping image areas.
//    (image2 is overlayed on image1, offset by xoff, yoff, toff)
//
//  Outputs: 
//    Bratios1[rgb][ii] = image2/image1 brightness ratio for color rgb
//                        and image1 brightness ii
//    Bratios2[rgb][ii] = image1/image2 brightness ratio for color rgb
//                        and image2 brightness ii

void getBrightRatios()
{
   uint16      vpix1[3], vpix2[3];
   int         vstat1, vstat2;
   int         px, py, pxinc, pyinc, ii, jj, rgb;
   int         npix, npix1, npix2, npix3;
   int         brdist1[3][256], brdist2[3][256];
   double      px1, py1, px2, py2;
   double      brlev1[3][256], brlev2[3][256];
   double      costf = cos(toff), sintf = sin(toff);
   double      a1, a2, b1, b2, bratio = 1;
   double      s8 = 1.0 / 256.0;
   
   for (rgb = 0; rgb < 3; rgb++)                                           //  clear distributions
   for (ii = 0; ii < 256; ii++)
      brdist1[rgb][ii] = brdist2[rgb][ii] = 0;

   pxinc = pyinc = 1;                                                      //  bugfix  v.7.7.1
   npix = (pxH - pxL) * (pyH - pyL);
   if (npix > 500000) pxinc = 2;                                           //  reduce excessive sample  v.7.7
   if (npix > 1000000) pyinc = 2;

   npix = 0;
   
   for (py = pyL; py < pyH; py += pyinc)                                   //  scan image1/image2 pixels parallel
   for (px = pxL; px < pxH; px += pxinc)                                   //  use entire overlap area
   {
      px1 = costf * px - sintf * (py - yoff);                              //  image1 pixel, after offsets
      py1 = costf * py + sintf * (px - xoff);
      vstat1 = vpixel(A1rgb48,px1,py1,vpix1);
      if (! vstat1) continue;                                              //  does not exist
      if (!vpix1[0] && !vpix1[1] && !vpix1[2]) continue;                   //  ignore black pixels

      px2 = costf * (px - xoff) + sintf * (py - yoff);                     //  corresponding image2 pixel
      py2 = costf * (py - yoff) - sintf * (px - xoff);
      vstat2 = vpixel(A2rgb48,px2,py2,vpix2);
      if (! vstat2) continue;                                              //  does not exist
      if (!vpix2[0] && !vpix2[1] && !vpix2[2]) continue;                   //  ignore black pixels

      ++npix;                                                              //  count overlapping pixels
      
      for (rgb = 0; rgb < 3; rgb++)                                        //  accumulate distributions
      {                                                                    //    by color in 256 bins
         ++brdist1[rgb][int(s8*vpix1[rgb])];
         ++brdist2[rgb][int(s8*vpix2[rgb])];
      }
   }
   
   npix1 = npix / 256;                                                     //  1/256th of total pixels
   
   for (rgb = 0; rgb < 3; rgb++)                                           //  get brlev1[rgb][N] = mean bright
   for (ii = jj = 0; jj < 256; jj++)                                       //    for Nth group of image1 pixels
   {                                                                       //      for color rgb
      brlev1[rgb][jj] = 0;
      npix2 = npix1;                                                       //  1/256th of total pixels

      while (npix2 > 0 && ii < 256)                                        //  next 1/256th group from distr,
      {
         npix3 = brdist1[rgb][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev1[rgb][jj] += ii * npix3;                                    //  brightness * (pixels with)
         brdist1[rgb][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev1[rgb][jj] = brlev1[rgb][jj] / npix1;                           //  mean brightness for group, 0-255
   }

   for (rgb = 0; rgb < 3; rgb++)                                           //  do same for image2
   for (ii = jj = 0; jj < 256; jj++)
   {
      brlev2[rgb][jj] = 0;
      npix2 = npix1;

      while (npix2 > 0 && ii < 256)
      {
         npix3 = brdist2[rgb][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev2[rgb][jj] += ii * npix3;
         brdist2[rgb][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev2[rgb][jj] = brlev2[rgb][jj] / npix1;
   }

   for (rgb = 0; rgb < 3; rgb++)                                           //  color
   for (ii = jj = 0; ii < 256; ii++)                                       //  brlev1 brightness, 0 to 255
   {                                                                       //  bugfix  v.6.4
      if (ii == 0) bratio = 1;
      while (ii > brlev2[rgb][jj] && jj < 256) ++jj;                       //  find matching brlev2 brightness
      a2 = brlev2[rgb][jj];                                                //  next higher value
      b2 = brlev1[rgb][jj];
      if (a2 > 0 && b2 > 0) {
         if (jj > 0) {
            a1 = brlev2[rgb][jj-1];                                        //  next lower value
            b1 = brlev1[rgb][jj-1];
         }
         else   a1 = b1 = 0;
         if (ii == 0)  bratio = b2 / a2;
         else   bratio = (b1 + (ii-a1)/(a2-a1) * (b2-b1)) / ii;            //  interpolate
      }

      if (bratio < 0.2) bratio = 0.2;                                      //  contain outliers
      if (bratio > 5) bratio = 5;
      Bratios2[rgb][ii] = bratio;
   }

   for (rgb = 0; rgb < 3; rgb++)                                           //  color
   for (ii = jj = 0; ii < 256; ii++)                                       //  brlev2 brightness, 0 to 255
   {                                                                       //  bugfix  v.6.4
      if (ii == 0) bratio = 1;
      while (ii > brlev1[rgb][jj] && jj < 256) ++jj;                       //  find matching brlev1 brightness
      a2 = brlev1[rgb][jj];                                                //  next higher value
      b2 = brlev2[rgb][jj];
      if (a2 > 0 && b2 > 0) {
         if (jj > 0) {
            a1 = brlev1[rgb][jj-1];                                        //  next lower value
            b1 = brlev2[rgb][jj-1];
         }
         else   a1 = b1 = 0;
         if (ii == 0)  bratio = b2 / a2;
         else   bratio = (b1 + (ii-a1)/(a2-a1) * (b2-b1)) / ii;            //  interpolate
      }

      if (bratio < 0.2) bratio = 0.2;                                      //  contain outliers
      if (bratio > 5) bratio = 5;
      Bratios1[rgb][ii] = bratio;
   }

   return;
}


/**************************************************************************/

//  Set color matching factors
//     on:   Bratios are used for color matching the two images
//     off:  Bratios are not used - use 1.0 instead
//  In both cases, manual settings Radjust/Gadjust/Badjust are used

void setColorfixFactors(int state)
{
   unsigned      ii, jj;

   if (state) 
   {
      for (ii = 0; ii < 65536; ii++)
      {
         jj = ii >> 8;

         R12match[ii] = sqrt(Bratios1[0][jj]) / Radjust * ii;              //  use sqrt(ratio) so that adjustment
         G12match[ii] = sqrt(Bratios1[1][jj]) / Gadjust * ii;              //    can be applied to both images
         B12match[ii] = sqrt(Bratios1[2][jj]) / Badjust * ii;              //      in opposite directions

         R21match[ii] = sqrt(Bratios2[0][jj]) * Radjust * ii;
         G21match[ii] = sqrt(Bratios2[1][jj]) * Gadjust * ii;
         B21match[ii] = sqrt(Bratios2[2][jj]) * Badjust * ii;
      }
   }

   else 
   {
      for (ii = 0; ii < 65536; ii++)
      {
         R12match[ii] = 1.0 / Radjust * ii;
         G12match[ii] = 1.0 / Gadjust * ii;
         B12match[ii] = 1.0 / Badjust * ii;

         R21match[ii] = Radjust * ii;
         G21match[ii] = Gadjust * ii;
         B21match[ii] = Badjust * ii;
      }
   }
   
   return;
}


/**************************************************************************/

//  find pixels of greatest contrast within overlap area
//  flag high-contrast pixels to use in each image compare region

void flagEdgePixels()
{
   void  flagEdgePixels2(int pxL, int pxH, int pyL, int pyH, int samp);

   int      samp = pixsamp / 9;
   int      pxm1, pxm2, pym1, pym2;
   
   if (redpixels) zfree(redpixels);                                        //  clear flags for alignment pixels
   redpixels = zmalloc(A1ww*A1hh);
   memset(redpixels,0,A1ww*A1hh);
   
   pxm1 = pxmL + 0.333 * (pxmH - pxmL);
   pxm2 = pxmL + 0.667 * (pxmH - pxmL);
   pym1 = pymL + 0.333 * (pymH - pymL);
   pym2 = pymL + 0.667 * (pymH - pymL);
   
   flagEdgePixels2(pxmL+8, pxm1,    pymL+8, pym1,    samp);                //  9 zones   v.8.0
   flagEdgePixels2(pxmL+8, pxm1,    pym1,   pym2,    samp);
   flagEdgePixels2(pxmL+8, pxm1,    pym2,   pymH-10, samp);
   flagEdgePixels2(pxm1,   pxm2,    pymL+8, pym1,    samp);
   flagEdgePixels2(pxm1,   pxm2,    pym1,   pym2,    samp);
   flagEdgePixels2(pxm1,   pxm2,    pym2,   pymH-10, samp);
   flagEdgePixels2(pxm2,   pxmH-10, pymL+8, pym1,    samp);
   flagEdgePixels2(pxm2,   pxmH-10, pym1,   pym2,    samp);
   flagEdgePixels2(pxm2,   pxmH-10, pym2,   pymH-10, samp);

   return;
}


//  Find the highest contrast pixels meeting sample size
//  within the specified sub-region of image overlap.

void flagEdgePixels2(int pxL, int pxH, int pyL, int pyH, int samp)
{
   int         px, py, ii, jj, npix, vstat1, vstat2, vstat3;
   int         red1, green1, blue1, red2, green2, blue2, tcon;
   int         Hdist[256], Vdist[256], Hmin, Vmin;
   double      costf = cos(toff), sintf = sin(toff);
   double      px1, py1, px2, py2, s8 = 1.0 / 769.0;
   uchar       *Hcon, *Vcon;
   uint16      vpix1[3], vpix2[3], vpix3[3];
   
   npix = (pxH - pxL) * (pyH - pyL);                                       //  overlapping pixels
   if (npix < 100) return;                                                 //  insignificant
   if (samp > npix / 4) samp = npix / 4;                                   //  use max. 1/4 of pixels

   Hcon = (uchar *) zmalloc(npix);                                         //  horizontal pixel contrast 0-255
   Vcon = (uchar *) zmalloc(npix);                                         //  vertical pixel contrast 0-255

   for (py = pyL; py < pyH; py++)                                          //  scan image pixels in sub-region
   for (px = pxL; px < pxH; px++)
   {
      ii = (py-pyL) * (pxH-pxL) + (px-pxL);
      Hcon[ii] = Vcon[ii] = 0;                                             //  horiz. = vert. contrast = 0

      px1 = costf * px - sintf * (py - yoff);                              //  image1 pixel
      py1 = costf * py + sintf * (px - xoff);
      vstat1 = vpixel(A1rgb48,px1,py1,vpix1);
      if (! vstat1) continue;                                              //  does not exist
      if (!vpix1[0] && !vpix1[1] && !vpix1[2]) continue;                   //  ignore black pixels

      px2 = costf * (px - xoff) + sintf * (py - yoff);                     //  corresponding image2 pixel
      py2 = costf * (py - yoff) - sintf * (px - xoff);                     //  v.7.5
      vstat2 = vpixel(A2rgb48,px2,py2,vpix2);
      if (! vstat2) continue;
      if (!vpix2[0] && !vpix2[1] && !vpix2[2]) continue;

      vstat3 = vpixel(A1rgb48,px1+4,py1,vpix3);                            //  4 pixels to right
      if (! vstat3) continue;                                              //  reject if off edge
      if (! vpix3[0] && ! vpix3[1] && ! vpix3[2]) continue;

      vstat3 = vpixel(A1rgb48,px1,py1+4,vpix3);                            //  4 pixels below
      if (! vstat3) continue;
      if (! vpix3[0] && ! vpix3[1] && ! vpix3[2]) continue;

      vstat3 = vpixel(A1rgb48,px1+4,py1+4,vpix3);                          //  verify overlap of +4 pixels
      if (! vstat3) continue;                                              //    in all directions   v.7.5
      if (!vpix3[0] && !vpix3[1] && !vpix3[2]) continue;
      
      vstat3 = vpixel(A2rgb48,px2+4,py2+4,vpix3);
      if (! vstat3) continue;
      if (!vpix3[0] && !vpix3[1] && !vpix3[2]) continue;

      vstat3 = vpixel(A1rgb48,px1-4,py1-4,vpix3);
      if (! vstat3) continue;
      if (!vpix3[0] && !vpix3[1] && !vpix3[2]) continue;
      
      vstat3 = vpixel(A2rgb48,px2-4,py2-4,vpix3);
      if (! vstat3) continue;
      if (!vpix3[0] && !vpix3[1] && !vpix3[2]) continue;

      red1 = vpix1[0];
      green1 = vpix1[1];
      blue1 = vpix1[2];

      vstat3 = vpixel(A1rgb48,px1+2,py1,vpix3);                            //  2 pixels to right
      red2 = vpix3[0];
      green2 = vpix3[1];
      blue2 = vpix3[2];
      tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);       //  horizontal contrast
      Hcon[ii] = int(tcon * s8);                                           //    0 - 255

      vstat3 = vpixel(A1rgb48,px1,py1+2,vpix3);                            //  2 pixels below
      red2 = vpix3[0];
      green2 = vpix3[1];
      blue2 = vpix3[2];
      tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);       //  vertical contrast
      Vcon[ii] = int(tcon * s8);
   }

   for (ii = 0; ii < 256; ii++) Hdist[ii] = Vdist[ii] = 0;                 //  clear contrast distributions

   for (py = pyL; py < pyH; py++)                                          //  scan image pixels      v.7.5
   for (px = pxL; px < pxH; px++)
   {                                                                       //  build contrast distributions
      ii = (py-pyL) * (pxH-pxL) + (px-pxL);
      ++Hdist[Hcon[ii]];
      ++Vdist[Vcon[ii]];
   }
   
   for (npix = 0, ii = 255; ii > 0; ii--)                                  //  find minimum contrast needed to get
   {                                                                       //    enough pixels for sample size
      npix += Hdist[ii];                                                   //      (horizontal contrast pixels)
      if (npix > samp) break; 
   }
   Hmin = ii; 

   for (npix = 0, ii = 255; ii > 0; ii--)                                  //  (verticle contrast pixels)
   {
      npix += Vdist[ii];
      if (npix > samp) break;
   }
   Vmin = ii;
   
   for (py = pyL; py < pyH; py++)                                          //  scan image pixels   v.7.5
   for (px = pxL; px < pxH; px++)
   {
      ii = (py-pyL) * (pxH-pxL) + (px-pxL);
      jj = py * A1ww + px;

      if (Hcon[ii] > Hmin) {
         redpixels[jj] = 1;                                                //  flag horizontal group of 3
         redpixels[jj+1] = 1;
         redpixels[jj+2] = 1;
      }

      if (Vcon[ii] > Vmin) {
         redpixels[jj] = 1;                                                //  flag verticle group of 3
         redpixels[jj+A1ww] = 1;
         redpixels[jj+2*A1ww] = 1;
      }
   }
   
   zfree(Hcon);
   zfree(Vcon);
   return;
}


/**************************************************************************/

//  Compare two images in overlapping areas.
//  (image2 is overlayed on image1, offset by xoff, yoff, toff).
//  Use pixels with contrast > minimum needed to reach sample size.
//  return: 1 = perfect match, 0 = total mismatch (black/white)

double matchImages()                                                       //  weighting removed   v.8.5
{
   uint16      vpix1[3], vpix2[3];
   int         px, py, ii, vstat1, vstat2;
   double      px1, py1, px2, py2;
   double      costf = cos(toff), sintf = sin(toff);
   double      match, cmatch, maxcmatch;
   
   if (pxM > A1ww || pyM > A1hh) return 0;                                 //  overlap runs off image, no match

   cmatch = maxcmatch = 0;

   for (py = pymL; py < pymH; py++)                                        //  step through image1 pixels, rows
   for (px = pxmL; px < pxmH; px++)                                        //  step through image1 pixels, cols
   {
      ii = py * A1ww + px;                                                 //  skip low-contrast pixels
      if (! redpixels[ii]) continue;

      px1 = costf * px - sintf * (py - yoff);                              //  image1 pixel
      py1 = costf * py + sintf * (px - xoff);
      vstat1 = vpixel(A1rgb48,px1,py1,vpix1);
      if (! vstat1) continue;                                              //  does not exist
      if (!vpix1[0] && !vpix1[1] && !vpix1[2]) continue;                   //  ignore black pixels

      px2 = costf * (px - xoff) + sintf * (py - yoff);                     //  corresponding image2 pixel
      py2 = costf * (py - yoff) - sintf * (px - xoff);
      vstat2 = vpixel(A2rgb48,px2,py2,vpix2);
      if (! vstat2) continue;
      if (!vpix2[0] && !vpix2[1] && !vpix2[2]) continue;
      
      match = matchPixels(vpix1,vpix2);                                    //  compare brightness adjusted
      cmatch += match;                                                     //  accumulate total match
      maxcmatch += 1.0;
   }

   return cmatch / maxcmatch;
}


/**************************************************************************/

//  Compare 2 pixels using precalculated brightness ratios
//  1.0 = perfect match   0 = total mismatch (black/white)

double matchPixels(uint16 *pix1, uint16 *pix2)
{
   double      red1, green1, blue1, red2, green2, blue2;
   double      reddiff, greendiff, bluediff, match;
   double      ff = 1.0 / 65536.0;

   red1 = R12match[pix1[0]];
   green1 = G12match[pix1[1]];
   blue1 = B12match[pix1[2]];

   red2 = R21match[pix2[0]];
   green2 = G21match[pix2[1]];
   blue2 = B21match[pix2[2]];

   reddiff = ff * fabs(red1-red2);                                         //  0 = perfect match
   greendiff = ff * fabs(green1-green2);                                   //  1 = total mismatch
   bluediff = ff * fabs(blue1-blue2);
   
   match = (1.0 - reddiff) * (1.0 - greendiff) * (1.0 - bluediff);         //  1 = perfect match
   return match;
}


/**************************************************************************/

//  Get a virtual pixel at location (px,py) (real) in an RGB-48 pixmap.
//  Get the overlapping real pixels and build a composite.

int vpixel(RGB *rgb, double px, double py, uint16 *vpix)                   //  overhauled   v.7.7
{
   int            ww, hh, px0, py0;
   uint16         *ppix, *pix0, *pix1, *pix2, *pix3;
   double         f0, f1, f2, f3;
   double         red, green, blue;
   
   ww = rgb->ww;
   hh = rgb->hh;
   ppix = (uint16 *) rgb->bmp;

   px0 = int(px);                                                          //  pixel containing (px,py)
   py0 = int(py);

   if (px0 < 1 || py0 < 1) return 0;
   if (px0 > ww-3 || py0 > hh-3) return 0;
   
   pix0 = ppix + (py0 * ww + px0) * 3;                                     //  4 pixels based at (px0,py0)
   pix1 = pix0 + ww * 3;
   pix2 = pix0 + 3;
   pix3 = pix0 + ww * 3 + 3;

   f0 = (px0+1 - px) * (py0+1 - py);                                       //  overlap of (px,py)
   f1 = (px0+1 - px) * (py - py0);                                         //   in each of the 4 pixels
   f2 = (px - px0) * (py0+1 - py);
   f3 = (px - px0) * (py - py0);
   
   red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];      //  sum the weighted inputs
   green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
   blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
   
   vpix[0] = int(red);
   vpix[1] = int(green);
   vpix[2] = int(blue);

   return 1;
}


/**************************************************************************

   edit transaction and thread support functions         overhauled  v.6.2

   edit transaction management
      edit_setup()                     start new edit - copy E3 > E1
      edit_cancel()                    cancel edit - E1 > E3, delete E1
      edit_done()                      commit edit - add to undo stack
      edit_undo()                      undo edit - E1 > E3
      edit_redo()                      redo edit - run thread again
      edit_fullsize()                  convert preview to full-size pixmaps

   main level thread management
      start_thread(func,arg)           start thread running
      signal_thread()                  signal thread that work is pending
      wait_thread_idle()               wait for pending work complete
      wrapup_thread(command)           wait for exit or command thread exit
      thread_working()                 return idle/working status

   thread function
      thread_idle_loop()               wait for pending work, exit if commanded
      exit_thread()                    exit thread unconditionally
      
   thread_status (thread ownership
      0     no thread is running
      1     thread is running and idle (no work)
      2     thread is working
      0     thread has exited

   thread_command (main program ownership)
      0     idle, no work pending
      8     exit when pending work is done
      9     exit now, unconditionally

   thread_pend    work requested counter
   thread_done    work done counter
   thread_hwm     high water mark
   edit_action    done/cancel/undo/redo in progress

***************************************************************************/

int      thread_command = 0, thread_status = 0;
int      thread_pend = 0, thread_done = 0, thread_hwm = 0;
int      edit_action = 0;


/**************************************************************************

  Setup for a new edit transaction
  Create E1 (edit input) and E3 (edit output) pixmaps from
  previous edit output (Frgb48) or image file (new Frgb48).

  uprev      0     edit full-size image
             1     edit preview image unless area exists and uarea = 2

  uarea      0     select_area is invalid and will be deleted (e.g. rotate)
             1     select_area not used but remains valid (e.g. red-eye)
             2     select_area is used and remains valid (e.g. flatten)

***************************************************************************/

int edit_setup(int uprev, int uarea)
{
   int      yn;

   if (! image_file) return 0;                                             //  no image file
   if (! menulock(1)) return 0;                                            //  lock menu
   
   if (Pundo > Pundo_max) {
      zmessageACK(ZTX("Too many undo buffers, please save image"));
      menulock(0);
      return 0;
   }
   
   if (Fimageturned) {
      turn_image(-Fimageturned);                                           //  un-turn if needed
      uarea = 0;                                                           //  (select area did already)
   }

   if (uarea == 0 && sa_stat) {
      yn = zmessageYN(ZTX("Select area cannot be kept.\n"
                          "Continue?"));
      if (! yn) {
         menulock(0);
         return 0;
      }
      select_delete();
      zdialog_free(zdsela);
      zdsela = 0;
   }
   
   if (uarea == 2 && sa_stat && ! sa_Npixel) {                             //  v.8.5
      yn = zmessageYN(ZTX("Select area is not finished.\n"
                          "Continue without using it?"));
      if (! yn) {
         menulock(0);
         return 0;
      }
   }
   
   Fpreview = 0;                                                           //  use preview image if supported
   if (uprev && ! (uarea == 2 && sa_Npixel)) Fpreview = 1;                 //    and select area will not be used

   if (Fpreview && Fzoom) {
      Fzoom = 0;
      mwpaint();                                                           //  do immediately    v.8.6
   }

   if (Fpreview && sa_stat) m_select_hide(0,0);                            //  v.8.5

   mutex_lock(&pixmaps_lock);                                              //  lock pixmaps

   if (! Frgb48) Frgb48 = image_load(image_file,48);                       //  create Frgb48 if not already
   if (! Frgb48) {
      mutex_unlock(&pixmaps_lock);
      menulock(0);
      return 0;
   }
   
   RGB_free(E1rgb48);                                                      //  free prior edit pixmaps
   RGB_free(E3rgb48);

   if (Fpreview)                                                           //  edit pixmaps are window-size
      E1rgb48 = RGB_rescale(Frgb48,dww,dhh);                               //  E1rgb48 = Frgb48 scaled to window
   else E1rgb48 = RGB_copy(Frgb48);                                        //  edit pixmaps are full-size
   E3rgb48 = RGB_copy(E1rgb48);                                            //  E1 >> E3

   E1ww = E3ww = E1rgb48->ww;
   E1hh = E3hh = E1rgb48->hh;

   save_undo();                                                            //  save Frgb48 in undo stack
   Fmodified = 0;                                                          //  image not modified yet
   thread_command = thread_status = 0;                                     //  no thread running
   thread_pend = thread_done = thread_hwm = 0;                             //  no work pending or done

   mutex_unlock(&pixmaps_lock);
   mwpaint2();
   return 1;
}


/**************************************************************************/

//  process edit cancel

void edit_cancel()
{
   if (edit_action) return;                                                //  v.6.8
   edit_action++;

   wrapup_thread(9);                                                       //  tell thread to quit now

   if (zdedit) {                                                           //  v.6.6
      zdialog_free(zdedit);                                                //  kill dialog
      zdedit = null;
   }

   mutex_lock(&pixmaps_lock);
   RGB_free(E1rgb48);                                                      //  free edit pixmaps E1, E3
   E1rgb48 = 0;
   RGB_free(E3rgb48);
   E3rgb48 = 0;
   RGB_free(Drgb24);
   Drgb24 = 0;
   E1ww = E3ww = Dww = 0;

   Fmodified = Fpreview = 0;                                               //  reset flags
   Ntoplines = Nptoplines;                                                 //  no overlay lines
   paint_toparc(2);                                                        //  no brush outline
   mutex_unlock(&pixmaps_lock);
   menulock(0);                                                            //  unlock menu
   mwpaint2();                                                             //  refresh window
   edit_action = 0;
   return;
}   


/**************************************************************************/

//  process edit dialog [done]  
//  E3rgb48 >> Frgb48 >> Frgb24

void edit_done()
{
   if (edit_action) return;                                                //  v.6.8
   edit_action++;

   if (Fpreview && Fmodified) {
      Fzoom = 0;                                                           //  v.8.3
      edit_fullsize();                                                     //  update full image
   }

   wrapup_thread(8);                                                       //  wait for thread done

   if (zdedit) {                                                           //  v.6.6
      zdialog_free(zdedit);                                                //  kill dialog
      zdedit = null;
   }
   
   mutex_lock(&pixmaps_lock);

   if (Fmodified) {
      RGB_free(Frgb48);                                                    //  memory leak   v.6.8
      Frgb48 = RGB_copy(E3rgb48);                                          //  E3 >> Frgb48
      RGB_free(Frgb24);
      Frgb24 = RGB_convbpp(Frgb48);                                        //  Frgb48 >> Frgb24
      Fww = Frgb24->ww;
      Fhh = Frgb24->hh;
      RGB_free(Drgb24);
      Drgb24 = 0;
      Dww = 0;
      Pundo++;
      Pumax = Pundo;
      save_undo();                                                         //  save next undo state
   }

   RGB_free(E1rgb48);                                                      //  free edit pixmaps
   RGB_free(E3rgb48);
   E1rgb48 = E3rgb48 = 0;
   E1ww = E3ww = 0;
   
   Fmodified = Fpreview = 0;                                               //  reset flags
   Ntoplines = Nptoplines;                                                 //  no overlay lines
   paint_toparc(2);                                                        //  no brush outline
   mutex_unlock(&pixmaps_lock);
   menulock(0);                                                            //  unlock menu
   mwpaint2();                                                             //  update window
   edit_action = 0;
   return;
}


/**************************************************************************/

//  process edit dialog [undo] and [redo] (not the toolbar functions)

void edit_undo()
{
   uint16   *pix1, *pix3;
   int     px, py;
   
   if (! Fmodified) return;                                                //  v.6.3

   if (edit_action) return;                                                //  v.6.8
   edit_action++;
   
   if (sa_Npixel)                                                          //  select area exists
   {
      for (int ii = 0; ii < sa_Npixel; ii++)                               //  restore enclosed pixels
      {
         px = sa_pixel[ii].px;
         py = sa_pixel[ii].py;
         pix1 = bmpixel(E1rgb48,px,py);
         pix3 = bmpixel(E3rgb48,px,py);
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
   }

   else 
   {                                                                       //  restore entire image
      mutex_lock(&pixmaps_lock);
      RGB_free(E3rgb48);                                                   //  (size may have changed)
      E3rgb48 = RGB_copy(E1rgb48);
      E3ww = E1ww;
      E3hh = E1hh;
      RGB_free(Drgb24);
      Drgb24 = 0;
      Dww = 0;
      Fmodified = 0;                                                       //  reset image modified status
      mutex_unlock(&pixmaps_lock);
   }

   mwpaint2();                                                             //  refresh window
   edit_action = 0;
   return;
}


void edit_redo()
{
   if (edit_action) return;                                                //  v.6.8
   edit_action++;
   signal_thread();                                                        //  start thread working
   Fmodified = 1;
   edit_action = 0;
   return;
}


/**************************************************************************/

//  Convert from preview mode (window-size pixmaps) to full-size pixmaps.

void edit_fullsize()                                                       //  v.6.2
{
   if (! Fpreview) return;
   Fpreview = 0;

   mutex_lock(&pixmaps_lock);
   RGB_free(E1rgb48);                                                      //  free preview pixmaps
   RGB_free(E3rgb48);
   E1rgb48 = RGB_copy(Frgb48);                                             //  make full-size pixmaps
   E3rgb48 = RGB_copy(Frgb48);
   E1ww = E3ww = Fww;
   E1hh = E3hh = Fhh;
   mutex_unlock(&pixmaps_lock);

   signal_thread();                                                        //  reinstate edits
   wait_thread_idle();
   return;
}


/**************************************************************************/

//  update progress for long-running functions on status bar
//  update in steps of 1%
//  call with args = 0 when done

void edit_progress(int done, int goal)
{
   static int     ppercentdone = 0;
   
   if (goal == 0) {
      PercentDone = ppercentdone = 0;
      if (pthread_equal(tid_fmain,pthread_self())) update_statusbar();
      else SBupdate++;
      return;
   }

   PercentDone = 100.0 * done / goal;
   if (PercentDone - ppercentdone < 1) return;

   ppercentdone = PercentDone;
   if (pthread_equal(tid_fmain,pthread_self())) update_statusbar();
   else SBupdate++;
   return;
}


/**************************************************************************/

//  start thread that does the edit work

void start_thread(threadfunc func, void *arg)
{
   thread_status = 1;                                                      //  thread is running
   thread_command = thread_pend = thread_done = thread_hwm = 0;            //  nothing pending
   start_detached_thread(func,arg);
   return;
}


//  signal thread that work is pending

void signal_thread()
{
   if (thread_status > 0) thread_pend++;                                   //  v.6.2
   return;
}


//  wait for edit thread to complete pending work and become idle

void wait_thread_idle()
{
   while (thread_status && thread_pend > thread_done)
   {
      zmainloop();
      zsleep(0.01);
   }
   
   return;
}


//  wait for thread exit or command thread exit
//  command = 0    wait for normal completion
//            8    finish pending work and exit
//            9    quit, exit now

void wrapup_thread(int command)
{
   thread_command = command;                                               //  tell thread to quit or finish

   while (thread_status > 0)                                               //  wait for thread to finish
   {                                                                       //    pending work and exit
      zmainloop();
      zsleep(0.01);
   }

   return;
}


//  return thread idle/working status

int thread_working()                                                       //  v.8.4
{
   if (thread_status == 2) return 1;
   return 0;
}


//  called only from edit threads
//  idle loop - wait for work request or exit command

void thread_idle_loop()
{
   thread_status = 1;                                                      //  status = idle
   thread_done = thread_hwm;                                               //  work done = high-water mark

   while (true)
   {
      if (thread_command == 9) exit_thread();                              //  quit now command
      if (thread_command == 8)                                             //  finish work and exit
         if (thread_pend <= thread_done) exit_thread();
      if (thread_pend > thread_done) break;                                //  wait for work request
      zsleep(0.01);
   }
   
   thread_hwm = thread_pend;                                               //  set high-water mark
   thread_status = 2;                                                      //  thread is working
   return;                                                                 //  loop to thread
}


//  called only from edit threads
//  exit thread unconditionally

void exit_thread()
{
   thread_pend = thread_done = thread_hwm = 0;
   thread_status = 0;
   pthread_exit(0);
}


/**************************************************************************
      undo / redo toolbar buttons
***************************************************************************/

//  [undo] menu function - reinstate previous edit in undo/redo stack

void m_undo(GtkWidget *, const char *)
{
   if (Pundo == 0) return;
   if (! menulock(1)) return;
   Pundo--;
   load_undo();
   menulock(0);
   return;
}


//  [redo] menu function - reinstate next edit in undo/redo stack

void m_redo(GtkWidget *, const char *)
{
   if (Pundo == Pumax) return;
   if (! menulock(1)) return;
   Pundo++;
   load_undo();
   menulock(0);
   return;
}


//  Save Frgb48 to undo/redo file stack
//  stack position = Pundo

void save_undo()
{
   char     *pp, buff[24];
   int      fid, cc, cc2;

   pp = strstr(undo_files,"_undo_");
   if (! pp) zappcrash("undo/redo stack corrupted 1");
   snprintf(pp+6,3,"%02d",Pundo);
   
   fid = open(undo_files,O_WRONLY|O_CREAT|O_TRUNC,0640);
   if (! fid) zappcrash("undo/redo stack corrupted 2");

   snprintf(buff,24," %05d %05d fotoxx ",Fww,Fhh);
   cc = write(fid,buff,20);
   if (cc != 20) zappcrash("undo/redo stack corrupted 3");
   
   cc = Fww * Fhh * 6;
   cc2 = write(fid,Frgb48->bmp,cc);
   if (cc2 != cc) zappcrash("undo/redo stack corrupted 4");

   close(fid);
   return;
}


//  Load Frgb48 from undo/redo file stack
//  stack position = Pundo

void load_undo()
{
   char     *pp, buff[24], fotoxx[8];
   int      fid, ww, hh, cc, cc2;

   pp = strstr(undo_files,"_undo_");
   if (! pp) zappcrash("undo/redo stack corrupted 1");
   snprintf(pp+6,3,"%02d",Pundo);
   
   fid = open(undo_files,O_RDONLY);
   if (! fid) zappcrash("undo/redo stack corrupted 2");
   
   *fotoxx = 0;
   cc = read(fid,buff,20);
   sscanf(buff," %d %d %8s ",&ww, &hh, fotoxx);
   if (! strEqu(fotoxx,"fotoxx")) zappcrash("undo/redo stack corrupted 4");

   mutex_lock(&pixmaps_lock);                                              //  v.6.3

   RGB_free(Frgb48);
   Frgb48 = RGB_make(ww,hh,48);
   cc = ww * hh * 6;
   cc2 = read(fid,Frgb48->bmp,cc);
   if (cc2 != cc) zappcrash("undo/redo stack corrupted 5");
   close(fid);
   
   RGB_free(Frgb24);
   Frgb24 = RGB_convbpp(Frgb48);
   Fww = ww;
   Fhh = hh;
   RGB_free(Drgb24);                                                       //  v.6.8
   Drgb24 = 0;
   Dww = 0;

   mutex_unlock(&pixmaps_lock);
   mwpaint2();
   return;
}


/**************************************************************************
      other support functions
***************************************************************************/

//  help menu function

void m_help(GtkWidget *, const char *menu)
{
   if (strEqu(menu,ZTX("About"))) 
      zmessageACK(" %s \n %s \n %s \n %s \n\n %s \n\n %s",
                   fversion,flicense,fhomepage,fcredits,ftranslators,fcontact);
      
   if (strEqu(menu,"FreeImage"))
      zmessageACK(FreeImage_GetCopyrightMessage());

   if (strEqu(menu,ZTX("User Guide"))) 
      showz_userguide();

   if (strEqu(menu,"README"))
      showz_readme();

   if (strEqu(menu,ZTX("Change Log")))
      showz_changelog();

   if (strEqu(menu,ZTX("Translate")))
      showz_translations();
      
   if (strEqu(menu,ZTX("Home Page")))
      showz_html(fhomepage);

   return;
}


/**************************************************************************/

//  restore state data from prior session

int load_fotoxx_state()
{
   int            ww, hh, ii, np;
   FILE           *fid;
   char           buff[1000], text[100], *pp;
   float          parms[2];
   
   lens4_name[0] = strdupz("lens_1",lens_cc);
   lens4_name[1] = strdupz("lens_2",lens_cc);
   lens4_name[2] = strdupz("lens_3",lens_cc);
   lens4_name[3] = strdupz("lens_4",lens_cc);
   lens4_mm[0] = 30;
   lens4_mm[1] = 40;
   lens4_mm[2] = 50;
   lens4_mm[3] = 60;
   lens4_bow[0] = 0;
   lens4_bow[1] = 0;
   lens4_bow[2] = 0;
   lens4_bow[3] = 0;
   curr_lens = 1;

   snprintf(buff,999,"%s/saved_state",get_zuserdir());                     //  open saved state file
   fid = fopen(buff,"r");
   if (! fid) return 0;
   
   pp = fgets_trim(buff,999,fid,1);                                        //  read last image file

   if (pp && *pp == '/') {
      if (image_file) zfree(image_file);
      image_file = strdupz(pp);
   }

   pp = fgets_trim(buff,999,fid,1);                                        //  top directory
   if (pp && *pp == '/') topdirk = strdupz(pp);

   pp = fgets(buff,999,fid);                                               //  main window size
   if (pp) {
      ww = hh = 0;
      sscanf(buff," %d %d ",&ww,&hh);
      if (ww > 200 && ww < 3000) Dww = ww;
      if (hh > 200 && hh < 2000) Dhh = hh;
   }

   pp = fgets(buff,999,fid);                                               //  image gallery window size
   if (pp) {
      ww = hh = 0;
      sscanf(buff," %d %d ",&ww,&hh);
      if (ww > 200 && ww < 3000) image_navi::xwinW = ww;
      if (hh > 200 && hh < 2000) image_navi::xwinH = hh;
   }

   pp = fgets_trim(buff,999,fid,1);                                        //  thumbnail image size
   if (pp) {
      sscanf(buff," %d ",&ww);
      if (ww > 32 && ww < 256) image_navi::thumbsize = ww;
   }

   for (ii = 0; ii < 4; ii++)                                              //  4 sets of lens parameters
   {   
      pp = fgets_trim(buff,999,fid,1);
      if (! pp) break;
      np = sscanf(buff," %s %f %f ",text,&parms[0],&parms[1]);
      if (np != 3) break;
      strncpy0(lens4_name[ii],text,lens_cc);
      lens4_mm[ii] = parms[0];
      lens4_bow[ii] = parms[1];
   }
   
   pp = fgets_trim(buff,999,fid,1);                                        //  current lens
   if (pp) {
      sscanf(buff," %f ",&parms[0]);
      if (parms[0] >= 0 && parms[0] < 4) curr_lens = int(parms[0]);
   }

   pp = fgets_trim(buff,999,fid,1);                                        //  fotoxx version changed
   if (! pp || strNeq(pp,fversion)) { /* do nothing */ }

   fclose(fid);

   for (ii = 0; ii < Nrecentfiles; ii++)                                   //  recent image file list = empty
      recentfiles[ii] = 0;

   snprintf(buff,999,"%s/recent_files",get_zuserdir());                    //  open recent files file  v.8.2
   fid = fopen(buff,"r");
   if (! fid) return 0;
   
   for (ii = 0; ii < Nrecentfiles; ii++)                                   //  read list of recent files
   {
      pp = fgets_trim(buff,999,fid,1);
      if (! pp) break;
      if (*pp == '/') recentfiles[ii] = strdupz(buff);
   }

   fclose(fid);

   return 1;
}


/**************************************************************************/

//  save state data for next session

int save_fotoxx_state()
{
   FILE           *fid;
   char           buff[1000];
   int            ww, hh, ii;

   snprintf(buff,999,"%s/saved_state",get_zuserdir());                     //  open output file
   fid = fopen(buff,"w");
   if (! fid) return 0;
   
   if (image_file && *image_file == '/')                                   //  current image file 
      fputs(image_file,fid);
   fputs("\n",fid);

   if (topdirk && *topdirk == '/')                                         //  top image directory 
      fputs(topdirk,fid);
   fputs("\n",fid);

   gtk_window_get_size(GTK_WINDOW(mWin),&ww,&hh);
   snprintf(buff,20," %d %d \n",ww,hh);                                    //  window size
   fputs(buff,fid);

   snprintf(buff,20," %d %d \n",image_navi::xwinW,image_navi::xwinH);      //  image gallery window size
   fputs(buff,fid);

   snprintf(buff,20," %d \n",image_navi::thumbsize);                       //  thumbnail size
   fputs(buff,fid);
   
   for (ii = 0; ii < 4; ii++)                                              //  4 sets of lens parameters
   {
      snprintf(buff,100," %s %.1f %.2f \n",lens4_name[ii],lens4_mm[ii],lens4_bow[ii]);
      fputs(buff,fid);
   }
   
   snprintf(buff,100," %d \n",curr_lens);                                  //  current lens
   fputs(buff,fid);

   fputs(fversion,fid);                                                    //  fotoxx version
   fputs("\n",fid);
   fputs("\n",fid);
   
   fclose(fid);
   
   snprintf(buff,999,"%s/recent_files",get_zuserdir());                    //  open output file
   fid = fopen(buff,"w");
   if (! fid) return 0;
   
   for (ii = 0; ii < Nrecentfiles; ii++)                                   //  save list of recent files  v.8.2
      if (recentfiles[ii])
         fprintf(fid,"%s \n",recentfiles[ii]);
   
   fclose(fid);

   return 1;
}


/**************************************************************************/

//  free all resources associated with the current image file

void free_resources()
{
   char        *pp, command[200];
   int         ignore;

   mutex_lock(&pixmaps_lock);                                              //  lock pixmaps
   
   strcpy(command,"rm -f ");                                               //  delete all undo files
   strcat(command,undo_files);
   pp = strstr(command,"_undo_");                                          //  clone edit, bugfix  v.6.5
   strcpy(pp + 6,"*");
   ignore = system(command);

   Fmodified = Pundo = Pumax = Fsaved = 0;                                 //  reset undo/redo stack
   Ntoplines = Nptoplines;                                                 //  no image overlay lines
   paint_toparc(2);                                                        //  no brush outline
   
   if (Fshutdown) {                                                        //  stop here if shutdown mode
      mutex_unlock(&pixmaps_lock);
      return;
   }
   
   if (image_file) {
      select_delete();                                                     //  delete select area
      zdialog_free(zdsela);                                                //  kill dialogs if active   v.8.7
      zdialog_free(zdRGB);
      zdsela = zdRGB = null;
      Mcapture = 0;                                                        //  kill mouse function if active  v.8.7
      mouseCBfunc = 0;
      gdk_window_set_cursor(drWin->window,0);                              //  restore normal cursor          v.8.7
      update_filetags(image_file);                                         //  commit tag changes, if any
      zfree(image_file);                                                   //  free image file
      image_file = 0;
   }
   
   RGB_free(Frgb24);
   Frgb24 = 0;
   RGB_free(Frgb48);
   Frgb48 = 0;
   RGB_free(E1rgb48);
   E1rgb48 = 0;
   RGB_free(E3rgb48);
   E3rgb48 = 0;
   RGB_free(Drgb24);
   Drgb24 = 0;

   Fww = E1ww = E3ww = Dww = 0;                                            //  make unusable (crash)

   mutex_unlock(&pixmaps_lock);
   return;
}


/**************************************************************************/

//  ask user if modified image should be kept or discarded

int mod_keep()
{
   if (Fmodified == 0 && Pundo == 0) return 0;                             //  no mods
   if (Fsaved == Pundo) return 0;                                          //  last mods were saved  v.8.3
   if (zmessageYN(ZTX("Discard modifications?"))) return 0;                //  OK to discard
   return 1;
}


/**************************************************************************/

//  menu lock/unlock - some functions must not run concurrently

int menulock(int lock)
{
   static int     mlock = 0;

   if (lock && mlock) {
      zmessageACK("please wait for prior function to complete");
      return 0;
   }
   
   if (! lock && ! mlock) zappcrash("menu lock error");
   
   if (lock) mlock++;
   else mlock--;
   return 1;
}


/**************************************************************************/

//  Upright a turned image - not like an edit.
//  Rotate Frgb24 without setting the Fmodified flag.

void  turn_image(int angle)
{
   while (angle >= 360) angle -= 360;                                      //  amount to turn now
   while (angle <= -360) angle += 360;
   Fimageturned += angle;                                                  //  total turn  v.8.3
   while (Fimageturned >= 360) Fimageturned -= 360;
   while (Fimageturned <= -360) Fimageturned += 360;
   if (angle == 0) return;
   
   mutex_lock(&pixmaps_lock);                                              //  lock pixmaps  v.8.5
   RGB * temp_bmp = RGB_rotate(Frgb24,angle);
   RGB_free(Frgb24);
   Frgb24 = temp_bmp;
   Fww = Frgb24->ww;
   Fhh = Frgb24->hh;
   Fzoom = 0;
   mutex_unlock(&pixmaps_lock);

   mwpaint();                                                              //  synch Dww etc.  v.6.8
   return;
}


/**************************************************************************/

//  FREEIMAGE error handler

void FI_error(FIF fif, const char *message)
{
   printf("FreeImage error: %s  file: %s \n",message,image_file);
   return;
}


/**************************************************************************
      pixmap conversion and rescale functions      revamped v.6.5
***************************************************************************/

//  initialize an RGB pixmap - allocate memory

RGB * RGB_make(int ww, int hh, int bpp)
{
   if (ww < 1 || hh < 1 || (bpp != 24 && bpp != 48))
      zappcrash("RGB_make() %d %d %d",ww,hh,bpp);
   
   RGB *rgb = (RGB *) zmalloc(sizeof(RGB));
   rgb->ww = ww;
   rgb->hh = hh;
   rgb->bpp = bpp;
   if (bpp == 24) rgb->bmp = zmalloc(ww * hh * 3);
   if (bpp == 48) rgb->bmp = zmalloc(ww * hh * 6);
   strcpy(rgb->wmi,"rgbrgb");
   return rgb;
}


//  free RGB pixmap

void RGB_free(RGB *rgb)
{
   if (! rgb) return;                                                      //  v.6.8
   if (! strEqu(rgb->wmi,"rgbrgb")) 
      zappcrash("RGB_free(), bad RGB");
   strcpy(rgb->wmi,"xxxxxx");
   zfree(rgb->bmp);
   zfree(rgb);
   return;
}


//  create a copy of an RGB pixmap

RGB * RGB_copy(RGB *rgb1)
{
   int      cc;
   RGB      *rgb2;

   rgb2 = RGB_make(rgb1->ww, rgb1->hh, rgb1->bpp);
   cc = rgb1->ww * rgb1->hh * (rgb1->bpp / 8);                             //  fix integer overflow for
   memcpy(rgb2->bmp,rgb1->bmp,cc);                                         //     huge images   v.7.8
   return rgb2;
}


//  create a copy of an RGB area

RGB * RGB_copy_area(RGB *rgb1, int orgx, int orgy, int ww2, int hh2)
{
   uint8          *bmp1, *pix1, *bmp2, *pix2;
   uint16         *bmp3, *pix3, *bmp4, *pix4;
   RGB            *rgb2 = 0;
   int            ww1, hh1, bpp, px1, py1, px2, py2;

   ww1 = rgb1->ww;
   hh1 = rgb1->hh;
   bpp = rgb1->bpp;

   if (bpp == 24)
   {
      rgb2 = RGB_make(ww2,hh2,24);
      bmp1 = (uint8 *) rgb1->bmp;
      bmp2 = (uint8 *) rgb2->bmp;
     
      for (py1 = orgy, py2 = 0; py2 < hh2; py1++, py2++) 
      {
         for (px1 = orgx, px2 = 0; px2 < ww2; px1++, px2++)
         {
            pix1 = bmp1 + (py1 * ww1 + px1) * 3;
            pix2 = bmp2 + (py2 * ww2 + px2) * 3;

            pix2[0] = pix1[0];
            pix2[1] = pix1[1];
            pix2[2] = pix1[2];
            pix1 += 3;
            pix2 += 3;
         }
      }
   }

   if (bpp == 48)
   {
      rgb2 = RGB_make(ww2,hh2,48);
      bmp3 = (uint16 *) rgb1->bmp;
      bmp4 = (uint16 *) rgb2->bmp;
     
      for (py1 = orgy, py2 = 0; py2 < hh2; py1++, py2++) 
      {
         for (px1 = orgx, px2 = 0; px2 < ww2; px1++, px2++)
         {
            pix3 = bmp3 + (py1 * ww1 + px1) * 3;
            pix4 = bmp4 + (py2 * ww2 + px2) * 3;

            pix4[0] = pix3[0];
            pix4[1] = pix3[1];
            pix4[2] = pix3[2];
            pix3 += 3;
            pix4 += 3;
         }
      }
   }
   
   return rgb2;
}


//   convert RGB pixmap from 24/48 to 48/24 bits per pixel

RGB * RGB_convbpp(RGB *rgb1)
{
   uint8          *bmp8, *pix8;
   uint16         *bmp16, *pix16;
   RGB            *rgb2 = 0;
   int            ww, hh, bpp, px, py;

   ww = rgb1->ww;
   hh = rgb1->hh;
   bpp = rgb1->bpp;

   if (bpp == 24)
   {
      rgb2 = RGB_make(ww,hh,48);
      bmp8 = (uint8 *) rgb1->bmp;
      bmp16 = (uint16 *) rgb2->bmp;
     
      for (py = 0; py < hh; py++) 
      {
         pix8 = bmp8 + py * ww * 3;
         pix16 = bmp16 + py * ww * 3;

         for (px = 0; px < ww; px++)
         {
            pix16[0] = pix8[0] << 8;
            pix16[1] = pix8[1] << 8;
            pix16[2] = pix8[2] << 8;
            pix8 += 3;
            pix16 += 3;
         }
      }
   }

   if (bpp == 48)
   {
      rgb2 = RGB_make(ww,hh,24);
      bmp8 = (uint8 *) rgb2->bmp;
      bmp16 = (uint16 *) rgb1->bmp;
     
      for (py = 0; py < hh; py++) 
      {
         pix8 = bmp8 + py * ww * 3;
         pix16 = bmp16 + py * ww * 3;

         for (px = 0; px < ww; px++)
         {
            pix8[0] = pix16[0] >> 8;
            pix8[1] = pix16[1] >> 8;
            pix8[2] = pix16[2] >> 8;
            pix8 += 3;
            pix16 += 3;
         }
      }
   }

   return rgb2;
}


//  convert RGB to FI bitmap, inverting row order

FIB * RGB_FIB(RGB *rgb)
{
   FIB         *fib = 0;
   uint8       *bmpf;
   uint8       *bmp1, *pix1, *pix2;
   uint16      *bmp2, *pix3, *pix4;
   int         ww, hh, bpp, px, py, pitch;
   
   ww = rgb->ww;
   hh = rgb->hh;
   bpp = rgb->bpp;

   if (bpp == 24)
   {   
      fib = FreeImage_Allocate(ww,hh,24);
      if (! fib) zappcrash("FIB allocation failure");

      bmp1 = (uint8 *) rgb->bmp;
      bmpf = FreeImage_GetBits(fib);
      pitch = FreeImage_GetPitch(fib);

      for (py = 0; py < hh; py++)
      {
         pix1 = bmp1 + (hh - py - 1) * ww * 3;
         pix2 = bmpf + py * pitch;
         
         for (px = 0; px < ww; px++)
         {
            pix2[RR] = pix1[0];
            pix2[GG] = pix1[1];
            pix2[BB] = pix1[2];
            pix1 += 3;
            pix2 += 3;
         }
      }
   }

   if (bpp == 48)
   {   
      fib = FreeImage_AllocateT(FIT_RGB16,ww,hh,48);
      if (! fib) zappcrash("FIB allocation failure");

      bmp2 = (uint16 *) rgb->bmp;
      bmpf = FreeImage_GetBits(fib);
      pitch = FreeImage_GetPitch(fib);
      
      for (py = 0; py < hh; py++)
      {
         pix3 = bmp2 + (hh - py - 1) * ww * 3;
         pix4 = (uint16 *) (bmpf + py * pitch);
         
         for (px = 0; px < ww; px++)
         {
            pix4[0] = pix3[0];
            pix4[1] = pix3[1];
            pix4[2] = pix3[2];
            pix3 += 3;
            pix4 += 3;
         }
      }
   }

   return fib;
}


//  convert FI to RGB pixmap, inverting row order

RGB * FIB_RGB(FIB *fib)
{
   RGB         *rgb;
   uint8       *bmpf;
   uint8       *bmp1, *pix1, *pix2;
   uint16      *bmp2, *pix3, *pix4;
   int         ww, hh, bpp, px, py, pitch;

   ww = FreeImage_GetWidth(fib);
   hh = FreeImage_GetHeight(fib);
   bpp = FreeImage_GetBPP(fib);
   pitch = FreeImage_GetPitch(fib);
   bmpf = FreeImage_GetBits(fib);
   
   rgb = RGB_make(ww,hh,bpp);

   if (bpp == 24)
   {   
      bmp1 = (uint8 *) rgb->bmp;

      for (py = 0; py < hh; py++)
      {
         pix1 = bmp1 + (hh - py - 1) * ww * 3;
         pix2 = bmpf + py * pitch;
         
         for (px = 0; px < ww; px++)
         {
            pix1[0] = pix2[RR];
            pix1[1] = pix2[GG];
            pix1[2] = pix2[BB];
            pix1 += 3;
            pix2 += 3;
         }
      }
   }

   if (bpp == 48)
   {   
      bmp2 = (uint16 *) rgb->bmp;

      for (py = 0; py < hh; py++)
      {
         pix3 = bmp2 + (hh - py - 1) * ww * 3;
         pix4 = (uint16 *) (bmpf + py * pitch);
         
         for (px = 0; px < ww; px++)
         {
            pix3[0] = pix4[0];
            pix3[1] = pix4[1];
            pix3[2] = pix4[2];
            pix3 += 3;
            pix4 += 3;
         }
      }
   }

   return rgb;
}


//  convert RGB-24 pixmap to GDK pixbuf (24)

PXB * RGB_PXB(RGB *rgb)                                                    //  new v.6.7.2
{
   int      ww, hh, bpp, px, py, rowst;
   uint8    *bmp1, *bmp2, *pix1, *pix2;
   PXB      *pxb;

   ww = rgb->ww;
   hh = rgb->hh;
   bpp = rgb->bpp;
   if (bpp != 24) zappcrash("RGB_PXB bpp %d",bpp);
   
   pxb = gdk_pixbuf_new(colorspace,0,8,ww,hh);
   if (! pxb) zappcrash("pixbuf allocation failure");

   bmp1 = (uint8 *) rgb->bmp;
   bmp2 = gdk_pixbuf_get_pixels(pxb);
   rowst = gdk_pixbuf_get_rowstride(pxb);

   for (py = 0; py < hh; py++)
   for (px = 0; px < ww; px++)
   {
      pix1 = bmp1 + (py * ww + px) * 3;
      pix2 = bmp2 + rowst * py + 3 * px;
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
      pix2[2] = pix1[2];
   }
   
   return pxb;
}


//  convert GDK pixbuf (24/32) to RGB-24 pixmap

RGB * PXB_RGB(PXB *pxb)
{
   RGB         *rgb;
   int         ww, hh, px, py, nch, rowst;
   uint8       *bmp1, *bmp2, *pix1, *pix2;

   ww = gdk_pixbuf_get_width(pxb);
   hh = gdk_pixbuf_get_height(pxb);
   nch = gdk_pixbuf_get_n_channels(pxb);
   rowst = gdk_pixbuf_get_rowstride(pxb);
   bmp1 = gdk_pixbuf_get_pixels(pxb);
   
   rgb = RGB_make(ww,hh,24);
   bmp2 = (uint8 *) rgb->bmp;

   for (py = 0; py < hh; py++)
   for (px = 0; px < ww; px++)
   {
      pix1 = bmp1 + rowst * py + nch * px;
      pix2 = bmp2 + (py * ww + px) * 3;
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
      pix2[2] = pix1[2];
   }

   return rgb;
}


//  rescale RGB pixmap to size ww2 x hh2

RGB * RGB_rescale(RGB *rgb1, int ww2, int hh2)
{
   void RGB_rescale_24(uint8*, uint8*, int, int, int, int);
   void RGB_rescale_48(uint16*, uint16*, int, int, int, int);

   RGB      *rgb2;
   int      ww1, hh1, bpp;
   uint8    *bmp1, *bmp2;
   uint16   *bmp3, *bmp4;

   ww1 = rgb1->ww;
   hh1 = rgb1->hh;
   bpp = rgb1->bpp;
   
   rgb2 = RGB_make(ww2,hh2,bpp);
   
   if (bpp == 24) {
      bmp1 = (uint8 *) rgb1->bmp;
      bmp2 = (uint8 *) rgb2->bmp;
      RGB_rescale_24(bmp1,bmp2,ww1,hh1,ww2,hh2);
   }

   if (bpp == 48) {
      bmp3 = (uint16 *) rgb1->bmp;
      bmp4 = (uint16 *) rgb2->bmp;
      RGB_rescale_48(bmp3,bmp4,ww1,hh1,ww2,hh2);
   }
   
   return rgb2;
}


//  rotate RGB pixmap through given angle in degrees (+ = clockwise)

RGB * RGB_rotate(RGB *rgb1, double angle)
{
   RGB * RGB_rotate_24(RGB *, double);
   RGB * RGB_rotate_48(RGB *, double);

   RGB      *rgb2 = 0;
   int      bpp;
   
   bpp = rgb1->bpp;
   if (bpp == 24) rgb2 = RGB_rotate_24(rgb1,angle);
   if (bpp == 48) rgb2 = RGB_rotate_48(rgb1,angle);
   return rgb2;
}


//  Load an image file into an RGB pixmap of 24 or 48 bits/pixel.
//  Also sets the following global variables:
//    file_MB = disk file megabytes
//    file_bpp = disk file bits per pixel (24 or 48)
//    file_type = "jpeg" or "tiff" or "other"

RGB * image_load(const char *filespec, int bpp)                            //  revamped   v.6.6.1
{
   FIF         fif;
   FIB         *fib = 0, *fib2 = 0;
   int         err, fibpp = 0;                                             //  gcc 4.4   v.7.1.1
   RGB         *rgb24 = 0, *rgb48 = 0;
   struct stat fstat;
   
   if (bpp != 24 && bpp != 48)                                             //  requested bpp must be 24 or 48
      zappcrash("image_load bpp: %d",bpp);

   err = stat(filespec,&fstat);
   if (err) return 0;                                                      //  file not found
   if (! S_ISREG(fstat.st_mode)) return 0;                                 //  not a regular file

   fif = FreeImage_GetFileType(filespec,0);                                //  unsupported file type
   if (fif == FIF_UNKNOWN) goto ILreturn;

   fib = FreeImage_Load(fif,filespec,0);
   if (! fib) goto ILreturn;                                               //  cannot load file

   fibpp = FreeImage_GetBPP(fib);                                          //  disk file bits/pixel
   
   if (fibpp < 24) {
      fib2 = FreeImage_ConvertTo24Bits(fib);                               //  1 or 8
      if (! fib2) goto ILreturn;
      rgb24 = FIB_RGB(fib2);
      if (bpp == 48) rgb48 = RGB_convbpp(rgb24);
   }

   if (fibpp == 24) {                                                      //  24
      rgb24 = FIB_RGB(fib);
      if (bpp == 48) rgb48 = RGB_convbpp(rgb24);
   }
   
   if (fibpp == 32) {                                                      //  24 + alpha
      fib2 = FreeImage_ConvertTo24Bits(fib);
      if (! fib2) goto ILreturn;
      rgb24 = FIB_RGB(fib2);
      if (bpp == 48) rgb48 = RGB_convbpp(rgb24);
   }

   if (fibpp == 48) {                                                      //  48
      rgb48 = FIB_RGB(fib);
      if (bpp == 24) rgb24 = RGB_convbpp(rgb48);
   }
   
ILreturn:
   if (fib) FreeImage_Unload(fib);                                         //  free FI bitmaps
   if (fib2) FreeImage_Unload(fib2);
   if (! rgb24 && ! rgb48) {
      zmessageACK(Bsavetoedit);                                            //  unsupported file type
      return 0;
   }
   
   file_bpp = fibpp;                                                       //  disk file bits/pixel
   file_MB = 1.0 * fstat.st_size / mega;                                   //  disk file megabytes
   if (fif == FIF_JPEG) strcpy(file_type,"jpeg");                          //  disk file type
   else if (fif == FIF_TIFF) strcpy(file_type,"tiff");
   else strcpy(file_type,"other");

   if (bpp == 24) {
      RGB_free(rgb48);                                                     //  return RGB-24 pixmap
      return rgb24;
   }

   RGB_free(rgb24);                                                        //  return RGB-48 pixmap
   return rgb48;
}


/**************************************************************************

   Rescale 24 bpp image (3 x 8 bits per color) to new width and height.
   The scale ratios may be different for width and height.

   Method: 
   The input and output images are overlayed, stretching or shrinking the
   output pixels as needed. The contribution of each input pixel overlapping
   an output pixel is proportional to the area of the output pixel covered by
   the input pixel. The contributions of all overlaping input pixels are added.
   The work is spread among NWthreads to reduce the elapsed time on modern 
   computers having multiple SMP processors.

   Example: if the output image is 40% of the input image, then:
     outpix[0,0] = 0.16 * inpix[0,0] + 0.16 * inpix[1,0] + 0.08 * inpix[2,0]
                 + 0.16 * inpix[0,1] + 0.16 * inpix[1,1] + 0.08 * inpix[2,1]
                 + 0.08 * inpix[0,2] + 0.08 * inpix[1,2] + 0.04 * inpix[2,2]

*********/

typedef struct {
   uint8    *pixmap1;                                                      //  file scope data for threads
   uint8    *pixmap2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   int      *py1L;
   int      *px1L;
   float    *pymap;
   float    *pxmap;
   int      maxmapx;
   int      maxmapy;
   int      index;
   int      done;
} rs24poop;

void RGB_rescale_24(uint8 *pixmap1, uint8 *pixmap2, int ww1, int hh1, int ww2, int hh2)
{
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   int         maxmapx, maxmapy;
   float       scalex, scaley;
   float       px1a, py1a, px1b, py1b;
   float       fx, fy;

   scalex = 1.0 * ww1 / ww2;                                               //  compute x and y scales
   scaley = 1.0 * hh1 / hh2;
   
   if (scalex <= 1) maxmapx = 2;                                           //  compute max input pixels
   else maxmapx = int(scalex + 2);                                         //    mapping into output pixels
   maxmapx += 1;                                                           //      for both dimensions
   if (scaley <= 1) maxmapy = 2;                                           //  (pixels may not be square)
   else maxmapy = int(scaley + 2);
   maxmapy += 1;
   
   memset(pixmap2, 0, ww2 * hh2 * 3);                                      //  clear output pixmap
   
   int *py1L = (int *) zmalloc(hh2 * sizeof(int));                         //  maps first (lowest) input pixel
   int *px1L = (int *) zmalloc(ww2 * sizeof(int));                         //    per output pixel

   float *pymap = (float *) zmalloc(hh2 * maxmapy * sizeof(float));        //  maps overlap of < maxmap input
   float *pxmap = (float *) zmalloc(ww2 * maxmapx * sizeof(float));        //    pixels per output pixel

   for (py2 = 0; py2 < hh2; py2++)                                         //  loop output y-pixels
   {
      py1a = py2 * scaley;                                                 //  corresponding input y-pixels
      py1b = py1a + scaley;
      if (py1b >= hh1) py1b = hh1 - 0.001;                                 //  fix precision limitation
      pyl = int(py1a);
      py1L[py2] = pyl;                                                     //  1st overlapping input pixel

      for (py1 = pyl, pym = 0; py1 < py1b; py1++, pym++)                   //  loop overlapping input pixels
      {
         if (py1 < py1a) {                                                 //  compute amount of overlap
            if (py1+1 < py1b) fy = py1+1 - py1a;                           //    0.0 to 1.0 
            else fy = scaley;
         }
         else if (py1+1 > py1b) fy = py1b - py1;
         else fy = 1;

         ii = py2 * maxmapy + pym;                                         //  save it
         pymap[ii] = 0.9999 * fy / scaley;
      }
      ii = py2 * maxmapy + pym;                                            //  set an end marker after
      pymap[ii] = -1;                                                      //    last overlapping pixel
   }
   
   for (px2 = 0; px2 < ww2; px2++)                                         //  do same for x-pixels
   {
      px1a = px2 * scalex;
      px1b = px1a + scalex;
      if (px1b >= ww1) px1b = ww1 - 0.001;
      pxl = int(px1a);
      px1L[px2] = pxl;

      for (px1 = pxl, pxm = 0; px1 < px1b; px1++, pxm++)
      {
         if (px1 < px1a) {
            if (px1+1 < px1b) fx = px1+1 - px1a;
            else fx = scalex;
         }
         else if (px1+1 > px1b) fx = px1b - px1;
         else fx = 1;

         ii = px2 * maxmapx + pxm;
         pxmap[ii] = 0.9999 * fx / scalex;
      }
      ii = px2 * maxmapx + pxm;
      pxmap[ii] = -1;
   }
   
   rs24poop threaddata[maxWthreads];                                       //  data for threads
   void  * RGB_rescale_24_thread(void *arg);

   for (ii = 0; ii < NWthreads; ii++)                                      //  start worker threads
   {
      threaddata[ii].pixmap1 = pixmap1;
      threaddata[ii].pixmap2 = pixmap2;
      threaddata[ii].ww1 = ww1;
      threaddata[ii].hh1 = hh1;
      threaddata[ii].ww2 = ww2;
      threaddata[ii].hh2 = hh2;
      threaddata[ii].py1L = py1L;
      threaddata[ii].px1L = px1L;
      threaddata[ii].pymap = pymap;
      threaddata[ii].pxmap = pxmap;
      threaddata[ii].maxmapx = maxmapx;
      threaddata[ii].maxmapy = maxmapy;
      threaddata[ii].index = ii;
      threaddata[ii].done = 0;
      start_detached_thread(RGB_rescale_24_thread,&threaddata[ii]);
   }
   
   for (ii = 0; ii < NWthreads; ii++)                                      //  wait for all done
      while (threaddata[ii].done == 0) zsleep(0.004);

   zfree(py1L);
   zfree(px1L);
   zfree(pymap);
   zfree(pxmap);
   return;
}

void * RGB_rescale_24_thread(void *arg)                                    //  worker thread function
{
   rs24poop * threaddata = (rs24poop *) arg;

   uint8       *pixmap1 = threaddata->pixmap1;
   uint8       *pixmap2 = threaddata->pixmap2;
   int         ww1 = threaddata->ww1;
   int         ww2 = threaddata->ww2;
   int         hh2 = threaddata->hh2;
   int         *py1L = threaddata->py1L;
   int         *px1L = threaddata->px1L;
   float       *pymap = threaddata->pymap;
   float       *pxmap = threaddata->pxmap;
   int         maxmapx = threaddata->maxmapx;
   int         maxmapy = threaddata->maxmapy;
   int         index = threaddata->index;
   
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   uint8       *pixel1, *pixel2;
   float       fx, fy, ftot;
   float       red, green, blue;

   for (py2 = index; py2 < hh2; py2 += NWthreads)                          //  loop output y-pixels
   {
      pyl = py1L[py2];                                                     //  corresp. 1st input y-pixel

      for (px2 = 0; px2 < ww2; px2++)                                      //  loop output x-pixels
      {
         pxl = px1L[px2];                                                  //  corresp. 1st input x-pixel

         red = green = blue = 0;                                           //  initz. output pixel

         for (py1 = pyl, pym = 0; ; py1++, pym++)                          //  loop overlapping input y-pixels
         {
            ii = py2 * maxmapy + pym;                                      //  get y-overlap
            fy = pymap[ii];
            if (fy < 0) break;                                             //  no more pixels

            for (px1 = pxl, pxm = 0; ; px1++, pxm++)                       //  loop overlapping input x-pixels
            {
               ii = px2 * maxmapx + pxm;                                   //  get x-overlap
               fx = pxmap[ii];
               if (fx < 0) break;                                          //  no more pixels

               ftot = fx * fy;                                             //  area overlap = x * y overlap
               pixel1 = pixmap1 + (py1 * ww1 + px1) * 3;
               red += pixel1[0] * ftot;                                    //  add input pixel * overlap
               green += pixel1[1] * ftot;
               blue += pixel1[2] * ftot;
            }

            pixel2 = pixmap2 + (py2 * ww2 + px2) * 3;                      //  save output pixel
            pixel2[0] = int(red);
            pixel2[1] = int(green);
            pixel2[2] = int(blue);
         }
      }
   }

   threaddata->done = 1;
   pthread_exit(0);
}


/**************************************************************************

   Rescale 48 bpp image (3 x 16 bits per color) to new width and height.
   Identical to RGB_rescale_24 except for the following: 
      uint8 >> uint16
      xxx24 >> xxx48
      3 >> 6
      memset 3 >> 6.

*******/

typedef struct {
   uint16   *pixmap1;                                                      //  file scope data for threads
   uint16   *pixmap2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   int      *py1L;
   int      *px1L;
   float    *pymap;
   float    *pxmap;
   int      maxmapx;
   int      maxmapy;
   int      index;
   int      done;
} rs48poop;

void RGB_rescale_48(uint16 *pixmap1, uint16 *pixmap2, int ww1, int hh1, int ww2, int hh2)
{
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   int         maxmapx, maxmapy;
   float       scalex, scaley;
   float       px1a, py1a, px1b, py1b;
   float       fx, fy;

   scalex = 1.0 * ww1 / ww2;                                               //  compute x and y scales
   scaley = 1.0 * hh1 / hh2;
   
   if (scalex <= 1) maxmapx = 2;                                           //  compute max input pixels
   else maxmapx = int(scalex + 2);                                         //    mapping into output pixels
   maxmapx += 1;                                                           //      for both dimensions
   if (scaley <= 1) maxmapy = 2;                                           //  (pixels may not be square)
   else maxmapy = int(scaley + 2);
   maxmapy += 1;
   
   memset(pixmap2, 0, ww2 * hh2 * 6);                                      //  clear output pixmap
   
   int *py1L = (int *) zmalloc(hh2 * sizeof(int));                         //  maps first (lowest) input pixel
   int *px1L = (int *) zmalloc(ww2 * sizeof(int));                         //    per output pixel

   float *pymap = (float *) zmalloc(hh2 * maxmapy * sizeof(float));        //  maps overlap of < maxmap input
   float *pxmap = (float *) zmalloc(ww2 * maxmapx * sizeof(float));        //    pixels per output pixel

   for (py2 = 0; py2 < hh2; py2++)                                         //  loop output y-pixels
   {
      py1a = py2 * scaley;                                                 //  corresponding input y-pixels
      py1b = py1a + scaley;
      if (py1b >= hh1) py1b = hh1 - 0.001;                                 //  fix precision limitation
      pyl = int(py1a);
      py1L[py2] = pyl;                                                     //  1st overlapping input pixel

      for (py1 = pyl, pym = 0; py1 < py1b; py1++, pym++)                   //  loop overlapping input pixels
      {
         if (py1 < py1a) {                                                 //  compute amount of overlap
            if (py1+1 < py1b) fy = py1+1 - py1a;                           //    0.0 to 1.0 
            else fy = scaley;
         }
         else if (py1+1 > py1b) fy = py1b - py1;
         else fy = 1;

         ii = py2 * maxmapy + pym;                                         //  save it
         pymap[ii] = 0.9999 * fy / scaley;
      }
      ii = py2 * maxmapy + pym;                                            //  set an end marker after
      pymap[ii] = -1;                                                      //    last overlapping pixel
   }
   
   for (px2 = 0; px2 < ww2; px2++)                                         //  do same for x-pixels
   {
      px1a = px2 * scalex;
      px1b = px1a + scalex;
      if (px1b >= ww1) px1b = ww1 - 0.001;
      pxl = int(px1a);
      px1L[px2] = pxl;

      for (px1 = pxl, pxm = 0; px1 < px1b; px1++, pxm++)
      {
         if (px1 < px1a) {
            if (px1+1 < px1b) fx = px1+1 - px1a;
            else fx = scalex;
         }
         else if (px1+1 > px1b) fx = px1b - px1;
         else fx = 1;

         ii = px2 * maxmapx + pxm;
         pxmap[ii] = 0.9999 * fx / scalex;
      }
      ii = px2 * maxmapx + pxm;
      pxmap[ii] = -1;
   }
   
   rs48poop threaddata[maxWthreads];                                       //  data for threads
   void  * RGB_rescale_48_thread(void *arg);

   for (ii = 0; ii < NWthreads; ii++)                                      //  start worker threads
   {
      threaddata[ii].pixmap1 = pixmap1;
      threaddata[ii].pixmap2 = pixmap2;
      threaddata[ii].ww1 = ww1;
      threaddata[ii].hh1 = hh1;
      threaddata[ii].ww2 = ww2;
      threaddata[ii].hh2 = hh2;
      threaddata[ii].py1L = py1L;
      threaddata[ii].px1L = px1L;
      threaddata[ii].pymap = pymap;
      threaddata[ii].pxmap = pxmap;
      threaddata[ii].maxmapx = maxmapx;
      threaddata[ii].maxmapy = maxmapy;
      threaddata[ii].index = ii;
      threaddata[ii].done = 0;
      start_detached_thread(RGB_rescale_48_thread,&threaddata[ii]);
   }
   
   for (ii = 0; ii < NWthreads; ii++)                                      //  wait for all done
      while (threaddata[ii].done == 0) zsleep(0.004);

   zfree(py1L);
   zfree(px1L);
   zfree(pymap);
   zfree(pxmap);
   return;
}

void * RGB_rescale_48_thread(void *arg)                                    //  worker thread function
{
   rs48poop * threaddata = (rs48poop *) arg;

   uint16      *pixmap1 = threaddata->pixmap1;
   uint16      *pixmap2 = threaddata->pixmap2;
   int         ww1 = threaddata->ww1;
   int         ww2 = threaddata->ww2;
   int         hh2 = threaddata->hh2;
   int         *py1L = threaddata->py1L;
   int         *px1L = threaddata->px1L;
   float       *pymap = threaddata->pymap;
   float       *pxmap = threaddata->pxmap;
   int         maxmapx = threaddata->maxmapx;
   int         maxmapy = threaddata->maxmapy;
   int         index = threaddata->index;
   
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   uint16      *pixel1, *pixel2;
   float       fx, fy, ftot;
   float       red, green, blue;

   for (py2 = index; py2 < hh2; py2 += NWthreads)                          //  loop output y-pixels
   {
      pyl = py1L[py2];                                                     //  corresp. 1st input y-pixel

      for (px2 = 0; px2 < ww2; px2++)                                      //  loop output x-pixels
      {
         pxl = px1L[px2];                                                  //  corresp. 1st input x-pixel

         red = green = blue = 0;                                           //  initz. output pixel

         for (py1 = pyl, pym = 0; ; py1++, pym++)                          //  loop overlapping input y-pixels
         {
            ii = py2 * maxmapy + pym;                                      //  get y-overlap
            fy = pymap[ii];
            if (fy < 0) break;                                             //  no more pixels

            for (px1 = pxl, pxm = 0; ; px1++, pxm++)                       //  loop overlapping input x-pixels
            {
               ii = px2 * maxmapx + pxm;                                   //  get x-overlap
               fx = pxmap[ii];
               if (fx < 0) break;                                          //  no more pixels

               ftot = fx * fy;                                             //  area overlap = x * y overlap
               pixel1 = pixmap1 + (py1 * ww1 + px1) * 3;
               red += pixel1[0] * ftot;                                    //  add input pixel * overlap
               green += pixel1[1] * ftot;
               blue += pixel1[2] * ftot;
            }

            pixel2 = pixmap2 + (py2 * ww2 + px2) * 3;                      //  save output pixel
            pixel2[0] = int(red);
            pixel2[1] = int(green);
            pixel2[2] = int(blue);
         }
      }
   }

   threaddata->done = 1;
   pthread_exit(0);
}


/**************************************************************************

      RGB *rgb2 = RGB_rotate_24(RGB *rgb1, double angle)

      Rotate RGB-24 pixmap through an arbitrary angle (degrees).

      The returned image has the same size as the original, but the
      pixmap size is increased to accomodate the rotated image.
      (e.g. a 100x100 image rotated 45 deg. needs a 142x142 pixmap).
      The parameters ww and hh are the dimensions of the input
      pixmap, and are updated to the dimensions of the output pixmap.

      The space added around the rotated image is black (RGB 0,0,0).
      Angle is in degrees. Positive direction is clockwise.
      Speed is about 3 million pixels/sec/thread for a 2.4 GHz CPU.
      Loss of resolution is less than 1 pixel.
      
      Work is divided among NWthreads to gain speed.

      Algorithm:
      create output pixmap big enough for rotated input pixmap
      loop all output pixels:
         get next output pixel (px2,py2)
         compute (R,theta) from center of pixbuf
         rotate theta by -angle
         (R,theta) is now within the closest input pixel
         convert to input pixel (px1,py1)
         if outside of pixmap
            output pixel = black
            continue
         for 4 input pixels based at (px0,py0) = (int(px1),int(py1))
            compute overlap (0 to 1) with (px1,py1)
            sum RGB values * overlap
         output aggregate RGB to pixel (px2,py2)

******/

int      rotrgb24_busy = 0;
uint8    *rotrgb24_pixmap1;
uint8    *rotrgb24_pixmap2;
int      rotrgb24_ww1;
int      rotrgb24_hh1;
int      rotrgb24_ww2;
int      rotrgb24_hh2;
double   rotrgb24_angle;


RGB * RGB_rotate_24(RGB *rgb1, double angle)
{
   void     *RGB_rotate_24_thread(void *);

   int      ww1, hh1, ww2, hh2, cc, ii;
   uint8    *pixmap1, *pixmap2;
   RGB      *rgb2;
   
   ww1 = rgb1->ww;
   hh1 = rgb1->hh;
   pixmap1 = (uint8 *) rgb1->bmp;

   while (angle < -180) angle += 360;                                      //  normalize, -180 to +180
   while (angle > 180) angle -= 360;
   angle = angle * pi / 180;                                               //  radians, -pi to +pi
   
   if (fabs(angle) < 0.001) {                                              //  angle = 0 within my precision
      rgb2 = RGB_make(ww1,hh1,24);                                         //  return a copy of the input
      pixmap2 = (uint8 *) rgb2->bmp;
      cc = ww1 * hh1 * 3;
      memcpy(pixmap2,pixmap1,cc);
      return rgb2;
   }

   ww2 = int(ww1*fabs(cos(angle)) + hh1*fabs(sin(angle)));                 //  rectangle containing rotated image
   hh2 = int(ww1*fabs(sin(angle)) + hh1*fabs(cos(angle)));
   
   rgb2 = RGB_make(ww2,hh2,24);
   pixmap2 = (uint8 *) rgb2->bmp;

   rotrgb24_pixmap1 = pixmap1;
   rotrgb24_pixmap2 = pixmap2;
   rotrgb24_ww1 = ww1;
   rotrgb24_hh1 = hh1;
   rotrgb24_ww2 = ww2;
   rotrgb24_hh2 = hh2;
   rotrgb24_angle = angle;

   for (ii = 0; ii < NWthreads; ii++)                                      //  start worker threads
      start_detached_thread(RGB_rotate_24_thread,&wtindex[ii]);
   zadd_locked(rotrgb24_busy,+NWthreads);

   while (rotrgb24_busy) zsleep(0.004);                                    //  wait for completion
   return rgb2;
}


void * RGB_rotate_24_thread(void *arg)
{
   int      index = *((int *) (arg));
   int      ww1, hh1, ww2, hh2;
   int      px2, py2, px0, py0;
   uint8    *ppix1, *ppix2, *pix0, *pix1, *pix2, *pix3;
   double   rx1, ry1, rx2, ry2, R, theta, px1, py1;
   double   f0, f1, f2, f3, red, green, blue, angle;

   ppix1 = rotrgb24_pixmap1;                                               //  input pixel array
   ppix2 = rotrgb24_pixmap2;                                               //  output pixel array
   ww1 = rotrgb24_ww1;
   hh1 = rotrgb24_hh1;
   ww2 = rotrgb24_ww2;
   hh2 = rotrgb24_hh2;
   angle = rotrgb24_angle;
   
   for (py2 = index; py2 < hh2; py2 += NWthreads)                          //  loop through output pixels
   for (px2 = 0; px2 < ww2; px2++)                                         //  outer loop y
   {
      rx2 = px2 - 0.5 * ww2;                                               //  (rx2,ry2) = center of pixel
      ry2 = py2 - 0.5 * hh2;
      R = sqrt(rx2*rx2 + ry2*ry2);                                         //  convert to (R,theta)
      if (R < 0.1) theta = 0;
      else theta = qarcsine(ry2 / R);                                      //  quick arc sine
      if (rx2 < 0) {
         if (theta < 0) theta = - pi - theta;                              //  adjust for quandrant
         else theta = pi - theta;
      }

      theta = theta - angle;                                               //  rotate theta backwards
      if (theta > pi) theta -= 2 * pi;                                     //  range -pi to +pi
      if (theta < -pi) theta += 2 * pi;

      rx1 = R * qcosine(theta);                                            //  quick cosine, sine
      ry1 = R * qsine(theta);
      px1 = rx1 + 0.5 * ww1;                                               //  (px1,py1) = corresponding
      py1 = ry1 + 0.5 * hh1;                                               //    point within input pixels

      px0 = int(px1);                                                      //  pixel containing (px1,py1)
      py0 = int(py1);
      
      if (px1 < 0 || px0 >= ww1-1 || py1 < 0 || py0 >= hh1-1) {            //  if outside input pixel array
         pix2 = ppix2 + (py2 * ww2 + px2) * 3;                             //    output is black
         pix2[0] = pix2[1] = pix2[2] = 0;
         continue;
      }

      pix0 = ppix1 + (py0 * ww1 + px0) * 3;                                //  4 input pixels based at (px0,py0)
      pix1 = pix0 + ww1 * 3;
      pix2 = pix0 + 3;
      pix3 = pix1 + 3;

      f0 = (px0+1 - px1) * (py0+1 - py1);                                  //  overlap of (px1,py1)
      f1 = (px0+1 - px1) * (py1 - py0);                                    //    in each of the 4 pixels
      f2 = (px1 - px0) * (py0+1 - py1);
      f3 = (px1 - px0) * (py1 - py0);
   
      red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];   //  sum the weighted inputs
      green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
      blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
      
      pix2 = ppix2 + (py2 * ww2 + px2) * 3;                                //  output pixel
      pix2[0] = int(red);
      pix2[1] = int(green);
      pix2[2] = int(blue);
   }
   
   zadd_locked(rotrgb24_busy,-1);
   pthread_exit(0);
}


/**************************************************************************

   RGB *rgb2 = RGB_rotate_48(RGB *rgb1, double angle)
   Rotate RGB-48 pixmap through an arbitrary angle (degrees).
   Identical to RGB_rotate_24() except for:
      uint8 >> uint16
      rotrgb24 >> rotrgb48
      24 >> 48   
      3 >> 6 

**********/

int      rotrgb48_busy = 0;
uint16   *rotrgb48_pixmap1;
uint16   *rotrgb48_pixmap2;
int      rotrgb48_ww1;
int      rotrgb48_hh1;
int      rotrgb48_ww2;
int      rotrgb48_hh2;
double   rotrgb48_angle;


RGB * RGB_rotate_48(RGB *rgb1, double angle)
{
   void     *RGB_rotate_48_thread(void *);

   int      ww1, hh1, ww2, hh2, cc, ii;
   uint16   *pixmap1, *pixmap2;
   RGB      *rgb2;
   
   ww1 = rgb1->ww;
   hh1 = rgb1->hh;
   pixmap1 = (uint16 *) rgb1->bmp;

   while (angle < -180) angle += 360;                                      //  normalize, -180 to +180
   while (angle > 180) angle -= 360;
   angle = angle * pi / 180;                                               //  radians, -pi to +pi
   
   if (fabs(angle) < 0.001) {                                              //  angle = 0 within my precision
      rgb2 = RGB_make(ww1,hh1,48);                                         //  return a copy of the input
      pixmap2 = (uint16 *) rgb2->bmp;
      cc = ww1 * hh1 * 6;
      memcpy(pixmap2,pixmap1,cc);
      return rgb2;
   }

   ww2 = int(ww1*fabs(cos(angle)) + hh1*fabs(sin(angle)));                 //  rectangle containing rotated image
   hh2 = int(ww1*fabs(sin(angle)) + hh1*fabs(cos(angle)));
   
   rgb2 = RGB_make(ww2,hh2,48);
   pixmap2 = (uint16 *) rgb2->bmp;

   rotrgb48_pixmap1 = pixmap1;
   rotrgb48_pixmap2 = pixmap2;
   rotrgb48_ww1 = ww1;
   rotrgb48_hh1 = hh1;
   rotrgb48_ww2 = ww2;
   rotrgb48_hh2 = hh2;
   rotrgb48_angle = angle;

   for (ii = 0; ii < NWthreads; ii++)                                      //  start worker threads
      start_detached_thread(RGB_rotate_48_thread,&wtindex[ii]);
   zadd_locked(rotrgb48_busy,+NWthreads);

   while (rotrgb48_busy) zsleep(0.004);                                    //  wait for completion
   return rgb2;
}


void * RGB_rotate_48_thread(void *arg)
{
   int      index = *((int *) (arg));
   int      ww1, hh1, ww2, hh2;
   int      px2, py2, px0, py0;
   uint16   *ppix1, *ppix2, *pix0, *pix1, *pix2, *pix3;
   double   rx1, ry1, rx2, ry2, R, theta, px1, py1;
   double   f0, f1, f2, f3, red, green, blue, angle;

   ppix1 = rotrgb48_pixmap1;                                               //  input pixel array
   ppix2 = rotrgb48_pixmap2;                                               //  output pixel array
   ww1 = rotrgb48_ww1;
   hh1 = rotrgb48_hh1;
   ww2 = rotrgb48_ww2;
   hh2 = rotrgb48_hh2;
   angle = rotrgb48_angle;
   
   for (py2 = index; py2 < hh2; py2 += NWthreads)                          //  loop through output pixels
   for (px2 = 0; px2 < ww2; px2++)                                         //  outer loop y
   {
      rx2 = px2 - 0.5 * ww2;                                               //  (rx2,ry2) = center of pixel
      ry2 = py2 - 0.5 * hh2;
      R = sqrt(rx2*rx2 + ry2*ry2);                                         //  convert to (R,theta)
      if (R < 0.1) theta = 0;
      else theta = qarcsine(ry2 / R);                                      //  quick arc sine
      if (rx2 < 0) {
         if (theta < 0) theta = - pi - theta;                              //  adjust for quandrant
         else theta = pi - theta;
      }

      theta = theta - angle;                                               //  rotate theta backwards
      if (theta > pi) theta -= 2 * pi;                                     //  range -pi to +pi
      if (theta < -pi) theta += 2 * pi;

      rx1 = R * qcosine(theta);                                            //  quick cosine, sine
      ry1 = R * qsine(theta);
      px1 = rx1 + 0.5 * ww1;                                               //  (px1,py1) = corresponding
      py1 = ry1 + 0.5 * hh1;                                               //    point within input pixels

      px0 = int(px1);                                                      //  pixel containing (px1,py1)
      py0 = int(py1);
      
      if (px1 < 0 || px0 >= ww1-1 || py1 < 0 || py0 >= hh1-1) {            //  if outside input pixel array
         pix2 = ppix2 + (py2 * ww2 + px2) * 3;                             //    output is black
         pix2[0] = pix2[1] = pix2[2] = 0;
         continue;
      }

      pix0 = ppix1 + (py0 * ww1 + px0) * 3;                                //  4 input pixels based at (px0,py0)
      pix1 = pix0 + ww1 * 3;
      pix2 = pix0 + 3;
      pix3 = pix1 + 3;

      f0 = (px0+1 - px1) * (py0+1 - py1);                                  //  overlap of (px1,py1)
      f1 = (px0+1 - px1) * (py1 - py0);                                    //    in each of the 4 pixels
      f2 = (px1 - px0) * (py0+1 - py1);
      f3 = (px1 - px0) * (py1 - py0);
   
      red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];   //  sum the weighted inputs
      green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
      blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
      
      pix2 = ppix2 + (py2 * ww2 + px2) * 3;                                //  output pixel
      pix2[0] = int(red);
      pix2[1] = int(green);
      pix2[2] = int(blue);
   }
   
   zadd_locked(rotrgb48_busy,-1);
   pthread_exit(0);
}



