// Copyright 2007 "Gilles Degottex"

// This file is part of "fmit"

// "fmit" is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// "fmit" is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "GLStatistics.h"

static const unsigned char g_icon_statistics[] = {
	0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a, 0x00, 0x00, 0x00, 0x0d,
	0x49, 0x48, 0x44, 0x52, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20,
	0x08, 0x06, 0x00, 0x00, 0x00, 0x73, 0x7a, 0x7a, 0xf4, 0x00, 0x00, 0x06,
	0xc4, 0x49, 0x44, 0x41, 0x54, 0x58, 0x85, 0xb5, 0x97, 0x6d, 0x4c, 0x5b,
	0xd7, 0x19, 0xc7, 0x7f, 0xf7, 0x5e, 0x1b, 0x1b, 0xdb, 0x04, 0x6c, 0x63,
	0xc0, 0xe1, 0xd5, 0x04, 0x92, 0x40, 0x63, 0x9c, 0x89, 0xac, 0x1f, 0x12,
	0xa1, 0x69, 0x5a, 0x84, 0x14, 0x05, 0x91, 0x69, 0x5a, 0xaa, 0x54, 0x51,
	0x42, 0xb6, 0x26, 0x43, 0x8b, 0xca, 0x40, 0xcd, 0x3e, 0xa4, 0xca, 0xa2,
	0x4a, 0x45, 0xcd, 0xb4, 0x2f, 0xec, 0x43, 0x22, 0xa5, 0x48, 0x93, 0x26,
	0x2d, 0x9b, 0x54, 0x6d, 0x51, 0xa3, 0xd2, 0xa8, 0xeb, 0xa6, 0xee, 0x25,
	0xb3, 0x50, 0xa7, 0x89, 0xa5, 0x84, 0xb2, 0x26, 0x59, 0x49, 0x00, 0x63,
	0x9b, 0xf2, 0x6e, 0xe3, 0x6b, 0x83, 0x5f, 0xef, 0x3e, 0xf8, 0x05, 0x1b,
	0x9b, 0x8e, 0x6d, 0xda, 0x23, 0x1d, 0x9d, 0x73, 0x8f, 0xce, 0x39, 0xff,
	0xdf, 0xf3, 0x9c, 0x97, 0x7b, 0x8e, 0xc0, 0xff, 0xc1, 0x3a, 0xba, 0x7b,
	0xb5, 0x40, 0x15, 0x60, 0x06, 0xca, 0x80, 0x35, 0xe0, 0x13, 0xe7, 0xc8,
	0xb0, 0xb2, 0xbd, 0xad, 0xf0, 0x5f, 0x0c, 0x2e, 0x02, 0xd5, 0x80, 0x0d,
	0x68, 0x00, 0xea, 0x80, 0x9a, 0x54, 0x5d, 0x4d, 0x2a, 0x99, 0x0b, 0x8c,
	0x3d, 0x09, 0x9c, 0x72, 0x8e, 0x0c, 0x3f, 0xcf, 0xae, 0x54, 0xed, 0x20,
	0x62, 0x04, 0x9a, 0x53, 0x22, 0xe9, 0xd4, 0x90, 0xca, 0xeb, 0x81, 0xa2,
	0x9d, 0x00, 0x35, 0x45, 0x45, 0x18, 0xf4, 0xc5, 0xe8, 0xf5, 0xc5, 0x18,
	0x74, 0xc5, 0x68, 0x8a, 0xd4, 0x3c, 0x7d, 0xe6, 0x22, 0xb4, 0xb1, 0x79,
	0x08, 0x78, 0x0d, 0x78, 0xb5, 0x20, 0x40, 0x47, 0x77, 0xaf, 0x1e, 0xb8,
	0x02, 0x9c, 0x4d, 0x89, 0x0b, 0x00, 0x92, 0x24, 0x52, 0x69, 0x31, 0x61,
	0x31, 0x19, 0x29, 0x2b, 0x35, 0x50, 0x56, 0x5a, 0x82, 0xb1, 0xb4, 0x24,
	0x93, 0x97, 0x96, 0x96, 0xa0, 0xd7, 0xeb, 0xd1, 0xe9, 0xf4, 0xa8, 0x35,
	0x5a, 0xe2, 0x8a, 0x48, 0x38, 0x9a, 0x20, 0xa1, 0x80, 0x4a, 0x14, 0xa8,
	0xb7, 0x68, 0xf8, 0xe0, 0x0f, 0xa3, 0xfc, 0xe4, 0xe6, 0x1d, 0x52, 0x0e,
	0x90, 0x07, 0x90, 0x12, 0x1f, 0x05, 0x1c, 0xa2, 0x28, 0x72, 0xec, 0xab,
	0x76, 0x8e, 0xbe, 0xd8, 0x46, 0x5b, 0x4b, 0x13, 0x55, 0x95, 0xe5, 0x44,
	0xe3, 0x02, 0xa1, 0x70, 0x9c, 0x60, 0x38, 0x41, 0x28, 0x92, 0xe0, 0xaf,
	0x1f, 0x7f, 0x8c, 0xae, 0xa2, 0x99, 0x88, 0x5a, 0xcf, 0x6c, 0x28, 0x41,
	0x5c, 0x56, 0x50, 0x88, 0xa3, 0x28, 0x41, 0x14, 0x05, 0x14, 0x40, 0x51,
	0x14, 0x14, 0x05, 0x4c, 0x25, 0x2a, 0xf6, 0x37, 0xd6, 0xa5, 0xf5, 0x6a,
	0x0b, 0x02, 0x00, 0x97, 0x00, 0x87, 0x24, 0x8a, 0xdc, 0xfc, 0xf1, 0x0f,
	0xd9, 0xdf, 0x64, 0x63, 0x66, 0x31, 0x8c, 0x6b, 0x3d, 0xca, 0xe3, 0x65,
	0x99, 0x44, 0x42, 0x49, 0x0d, 0x9c, 0xcc, 0xff, 0xfc, 0xe0, 0x01, 0xdd,
	0x67, 0x5a, 0x50, 0xa3, 0x20, 0x8a, 0x02, 0x82, 0x40, 0x96, 0x70, 0x56,
	0xae, 0x80, 0xbc, 0x19, 0xc7, 0x64, 0xdc, 0x93, 0xd6, 0xdb, 0xd7, 0xd1,
	0xdd, 0x2b, 0x64, 0x2f, 0xc6, 0x34, 0x40, 0x23, 0xc0, 0xc1, 0xe6, 0x7a,
	0x0e, 0x34, 0xd9, 0x18, 0x7b, 0x2e, 0xb3, 0x19, 0x49, 0x86, 0x51, 0x12,
	0x41, 0x14, 0x84, 0x9c, 0x81, 0xd5, 0x92, 0x84, 0x28, 0x24, 0x50, 0x49,
	0xe9, 0x3a, 0x21, 0x23, 0x98, 0x1e, 0x39, 0xa1, 0x00, 0x0a, 0x84, 0x23,
	0x0a, 0xd6, 0x8a, 0x3d, 0x68, 0x35, 0x45, 0x6c, 0x86, 0x23, 0x3a, 0x60,
	0x2f, 0xe0, 0x49, 0x03, 0x88, 0xd9, 0x20, 0x55, 0x95, 0xe5, 0x78, 0xd7,
	0x22, 0xc4, 0xe2, 0xc9, 0xf9, 0xcb, 0x24, 0x49, 0x40, 0x25, 0x91, 0x29,
	0x1b, 0xcd, 0xe5, 0xc8, 0xbe, 0x95, 0x64, 0x7d, 0xa1, 0x36, 0xe2, 0x56,
	0xff, 0x70, 0x2c, 0x81, 0x20, 0x08, 0xd4, 0xec, 0xad, 0x48, 0x6b, 0x1e,
	0xc8, 0x9e, 0x02, 0x31, 0xfb, 0x43, 0xa7, 0xd5, 0x10, 0x8b, 0x2b, 0xc9,
	0x81, 0x24, 0x61, 0x2b, 0x89, 0x59, 0xdf, 0xa2, 0x40, 0xd5, 0xde, 0x1a,
	0x56, 0x97, 0x3c, 0x59, 0xc2, 0xdb, 0x20, 0xb2, 0xfa, 0x46, 0x63, 0xc9,
	0x98, 0x34, 0xd4, 0x5a, 0xd3, 0x32, 0x2d, 0x85, 0x00, 0x22, 0x00, 0x5a,
	0x4d, 0x51, 0xae, 0xb0, 0x54, 0x48, 0x40, 0xa0, 0xb6, 0xce, 0xc6, 0x17,
	0x9e, 0x99, 0x2f, 0x6d, 0x93, 0x4e, 0xb1, 0x44, 0x12, 0xa0, 0xb1, 0x7e,
	0x6f, 0x5a, 0xf3, 0x60, 0x21, 0x80, 0x75, 0x80, 0x78, 0x22, 0x81, 0x5a,
	0x25, 0xe4, 0x86, 0xbf, 0x80, 0x97, 0xd5, 0x35, 0xb5, 0x78, 0x66, 0xa7,
	0x08, 0xc9, 0xbe, 0xc2, 0x6d, 0x32, 0x09, 0xd4, 0x52, 0x52, 0xc2, 0x56,
	0x57, 0xfd, 0xef, 0x01, 0x22, 0x91, 0x28, 0x45, 0x2a, 0x31, 0xcb, 0x03,
	0xf2, 0x3c, 0xfb, 0xfc, 0xb3, 0x71, 0x2e, 0xbe, 0xf4, 0x75, 0xde, 0xf9,
	0xf9, 0x2d, 0xbe, 0xfd, 0x0d, 0x3b, 0xaf, 0xbf, 0xfa, 0x32, 0xb2, 0x7f,
	0x25, 0x77, 0x9a, 0x24, 0x81, 0xf2, 0x12, 0x15, 0x0e, 0x9b, 0x9e, 0x43,
	0x75, 0xc5, 0x49, 0x80, 0xdd, 0x44, 0x20, 0x12, 0x8d, 0xa1, 0xfe, 0x92,
	0x29, 0xf0, 0xb8, 0xa6, 0xf8, 0xc1, 0x2b, 0xdf, 0xa2, 0xe7, 0x52, 0x3f,
	0xe7, 0xbf, 0x37, 0xc0, 0x47, 0x7f, 0x9b, 0xc6, 0xb6, 0xaf, 0x99, 0xfe,
	0xef, 0x76, 0x13, 0x0d, 0x87, 0x50, 0x4b, 0x02, 0x95, 0x65, 0x6a, 0x0e,
	0xdb, 0x74, 0xd8, 0xeb, 0x75, 0x98, 0x0c, 0x5b, 0x07, 0xad, 0xb5, 0xc2,
	0x4c, 0xb1, 0x56, 0x03, 0x50, 0xdd, 0xd1, 0xdd, 0x5b, 0xb2, 0x1d, 0x60,
	0x19, 0x20, 0x18, 0xdc, 0x48, 0x02, 0xe4, 0xac, 0xe4, 0x2d, 0xcf, 0xde,
	0xfe, 0xe9, 0x5b, 0x5c, 0xb8, 0xd4, 0x47, 0xd7, 0x37, 0x5f, 0xc2, 0x58,
	0x66, 0x24, 0x16, 0xdd, 0xe4, 0xb5, 0xd7, 0xdf, 0xa2, 0x69, 0x7f, 0x2b,
	0xbf, 0x7c, 0xfb, 0x06, 0x8e, 0x06, 0x3d, 0x4d, 0x55, 0x5a, 0xf4, 0x1a,
	0x29, 0xdb, 0x49, 0x42, 0xa1, 0x10, 0x82, 0x20, 0xd0, 0x50, 0x67, 0x85,
	0xe4, 0x09, 0x7b, 0x70, 0x3b, 0xc0, 0x22, 0x80, 0x3f, 0x10, 0xdc, 0xd1,
	0xfb, 0x68, 0x38, 0xc4, 0x83, 0x3f, 0x7e, 0xc8, 0xcb, 0xe7, 0x2e, 0xa2,
	0x92, 0x04, 0x5a, 0xed, 0x0e, 0x9e, 0x3d, 0xfd, 0x14, 0x4b, 0xa9, 0x8a,
	0x9f, 0x0d, 0xdf, 0xe4, 0xdd, 0x5f, 0xff, 0x8a, 0x79, 0x8f, 0x8b, 0x40,
	0x20, 0x80, 0xdf, 0xef, 0xc7, 0xe7, 0xf3, 0x65, 0x92, 0x28, 0x8a, 0xa9,
	0x85, 0x98, 0x59, 0x07, 0x2f, 0x6c, 0x07, 0x58, 0x02, 0x58, 0x97, 0x83,
	0xa9, 0x79, 0x27, 0x0f, 0xe2, 0x9f, 0x8f, 0x27, 0xd8, 0xd7, 0x7c, 0x00,
	0x93, 0xc9, 0x88, 0x4a, 0x12, 0x38, 0xdc, 0xf6, 0x02, 0x9f, 0x4f, 0x8c,
	0xa2, 0x13, 0x42, 0x14, 0x6b, 0xd4, 0x9c, 0x3e, 0x7d, 0x9a, 0x5b, 0xb7,
	0x6e, 0xa1, 0xd7, 0xeb, 0x29, 0x2d, 0x2d, 0xa5, 0xac, 0xac, 0x2c, 0x93,
	0xb4, 0x5a, 0x2d, 0x00, 0xfb, 0xb6, 0x00, 0x0e, 0x15, 0x8e, 0x80, 0x5f,
	0x46, 0x14, 0x84, 0xad, 0x69, 0xc8, 0xf2, 0xde, 0x3d, 0x3b, 0x8d, 0x41,
	0x6f, 0x20, 0x24, 0xfb, 0xd0, 0x12, 0xa4, 0xc5, 0x56, 0x81, 0x28, 0x90,
	0x11, 0xe9, 0xeb, 0xeb, 0xe3, 0xce, 0x9d, 0x3b, 0x78, 0x3c, 0x99, 0x43,
	0x2e, 0xcf, 0x1a, 0x77, 0x02, 0x70, 0x8e, 0x0c, 0xaf, 0x01, 0x41, 0x7f,
	0x40, 0xc6, 0xbf, 0x1e, 0x60, 0x23, 0xe0, 0x27, 0x28, 0xfb, 0x09, 0x06,
	0x7c, 0xc8, 0xb2, 0x8f, 0xa2, 0x22, 0x35, 0xe6, 0x72, 0x33, 0x92, 0x24,
	0xd0, 0x62, 0xab, 0xa4, 0xa6, 0xca, 0x9c, 0xf1, 0x2a, 0x6d, 0xad, 0xad,
	0xad, 0x5c, 0xbd, 0x7a, 0x95, 0x93, 0x27, 0x4f, 0xb2, 0xba, 0xba, 0x5a,
	0x18, 0xa0, 0x61, 0xe7, 0x08, 0x00, 0xb8, 0x01, 0x02, 0xf2, 0x06, 0x16,
	0x8b, 0x19, 0xb3, 0xd9, 0x84, 0xc5, 0x6c, 0xa2, 0xdc, 0x64, 0x42, 0xaf,
	0xd3, 0xf2, 0xa7, 0x8f, 0x7e, 0x4b, 0xe7, 0xf1, 0xe3, 0xa8, 0xa4, 0xad,
	0x7b, 0x86, 0xc9, 0x64, 0xca, 0x11, 0x1b, 0x18, 0x18, 0xe0, 0xd4, 0xa9,
	0x53, 0x1c, 0x3d, 0x7a, 0x94, 0xd1, 0xd1, 0xd1, 0x3c, 0x80, 0xb2, 0x3d,
	0x06, 0xcc, 0xc6, 0x52, 0x48, 0xee, 0x04, 0x23, 0xe4, 0x5e, 0x48, 0xe6,
	0x80, 0x03, 0x8b, 0x2b, 0x6b, 0x58, 0x2c, 0x96, 0xe4, 0x4f, 0x45, 0x49,
	0xfe, 0x54, 0xbc, 0x5e, 0x0f, 0xef, 0xbd, 0xfb, 0x1b, 0x26, 0x27, 0x27,
	0x73, 0x06, 0x3c, 0x72, 0xe4, 0x08, 0x63, 0x63, 0x63, 0x74, 0x76, 0x76,
	0x66, 0xea, 0x06, 0x07, 0x07, 0xb1, 0xdb, 0xed, 0x9c, 0x39, 0x73, 0x86,
	0xaa, 0xaa, 0x2a, 0x3a, 0x3b, 0x3b, 0xb9, 0x7e, 0xfd, 0x3a, 0x13, 0x13,
	0x13, 0x8c, 0x8f, 0x8f, 0x53, 0x5f, 0x5b, 0xc9, 0xca, 0x9a, 0x3f, 0x1d,
	0x05, 0x67, 0x76, 0x04, 0xe6, 0x00, 0x96, 0x96, 0xd7, 0x90, 0x44, 0x32,
	0xdb, 0x10, 0x25, 0x4e, 0xdf, 0xf7, 0x5f, 0xa1, 0xbf, 0xbf, 0x1f, 0xab,
	0xd5, 0x9a, 0x03, 0xd0, 0xde, 0xde, 0xce, 0xd8, 0xd8, 0x58, 0x4e, 0x9d,
	0x2c, 0xcb, 0x54, 0x57, 0x57, 0x73, 0xe5, 0xca, 0x15, 0xec, 0x76, 0x3b,
	0x5e, 0xaf, 0x97, 0xa1, 0xa1, 0x21, 0x16, 0x17, 0x17, 0xe9, 0xea, 0xea,
	0xa2, 0xae, 0xba, 0x2a, 0xdd, 0xd4, 0xbe, 0x3d, 0x02, 0x6e, 0x80, 0xc5,
	0xe5, 0x35, 0x44, 0x51, 0x40, 0x05, 0x2c, 0x2d, 0x2d, 0x73, 0xe9, 0xe2,
	0x77, 0xd0, 0xeb, 0x74, 0x5c, 0xbb, 0x76, 0x2d, 0x2f, 0xa4, 0xe1, 0x70,
	0x98, 0x27, 0x4f, 0x9e, 0x30, 0x34, 0x34, 0x84, 0xcf, 0xe7, 0x03, 0xc0,
	0x60, 0x30, 0xe0, 0x70, 0x38, 0x38, 0x7b, 0xf6, 0x2c, 0x03, 0x03, 0x03,
	0x79, 0x7d, 0x6a, 0xf6, 0x56, 0xa6, 0x8b, 0x87, 0x0b, 0x4d, 0x41, 0x12,
	0x40, 0x80, 0x37, 0x07, 0xdf, 0xe4, 0xf6, 0xed, 0xdb, 0x9c, 0x3f, 0x7f,
	0x8e, 0x1b, 0x37, 0x6e, 0xb0, 0xba, 0xba, 0xca, 0xc3, 0x87, 0x0f, 0x79,
	0xf4, 0xe8, 0x11, 0xb2, 0x2c, 0x03, 0xc9, 0x1d, 0xa0, 0x28, 0x0a, 0x3d,
	0x3d, 0x3d, 0x98, 0xcd, 0xe6, 0x3c, 0xb1, 0x42, 0x56, 0xbb, 0xf5, 0x5b,
	0xfe, 0x4a, 0x41, 0x80, 0x85, 0xa5, 0x55, 0xbc, 0x5e, 0x2f, 0x82, 0x00,
	0x17, 0x2e, 0xf4, 0xa0, 0xd1, 0x68, 0x18, 0x1c, 0x1c, 0xa4, 0xbc, 0xbc,
	0x9c, 0xf6, 0xf6, 0x76, 0x2e, 0x5f, 0xbe, 0x8c, 0xc1, 0x60, 0xc8, 0x74,
	0xba, 0x7b, 0xf7, 0x2e, 0x6e, 0xb7, 0x7b, 0xd7, 0x00, 0x16, 0xb3, 0x31,
	0x5d, 0x6c, 0xeb, 0xe8, 0xee, 0xd5, 0x65, 0x03, 0xb8, 0x00, 0x66, 0xe6,
	0xe6, 0xb9, 0x7f, 0xff, 0x3e, 0x27, 0x4e, 0x9c, 0xa0, 0xad, 0xad, 0x2d,
	0x6f, 0xbb, 0x6d, 0xb7, 0x63, 0xc7, 0x8e, 0x71, 0xef, 0xde, 0x3d, 0x1c,
	0x0e, 0xc7, 0xae, 0x00, 0x34, 0x1a, 0x75, 0xba, 0x58, 0x04, 0xd8, 0xb2,
	0x01, 0x9e, 0x03, 0x89, 0x85, 0xa5, 0x55, 0xb1, 0xa7, 0xe7, 0x0d, 0x34,
	0x9a, 0x1d, 0x6f, 0xde, 0x39, 0x66, 0xb5, 0x5a, 0x99, 0x9f, 0x9f, 0xdf,
	0x55, 0xdb, 0x8d, 0xcd, 0x30, 0xf7, 0x7f, 0x9f, 0xb3, 0x3d, 0x7d, 0x19,
	0x00, 0xe7, 0xc8, 0xf0, 0x66, 0x47, 0x77, 0xaf, 0x4b, 0x51, 0x94, 0x06,
	0x97, 0x67, 0x81, 0xe6, 0xc6, 0xbc, 0x0b, 0xec, 0x7f, 0x64, 0x4b, 0x2b,
	0x3e, 0xa6, 0xa6, 0xdd, 0x4c, 0x4d, 0xcf, 0x31, 0x35, 0xe3, 0x66, 0x6a,
	0xda, 0x8d, 0x67, 0x7e, 0x91, 0x44, 0x22, 0x73, 0x1f, 0x7d, 0xdf, 0x39,
	0x32, 0xec, 0xd9, 0xfe, 0x30, 0x79, 0x0c, 0x34, 0xcc, 0xce, 0xcd, 0xef,
	0x1a, 0x20, 0x16, 0x8b, 0x63, 0x3f, 0x7c, 0x84, 0xf7, 0x7f, 0xe7, 0xc4,
	0xe5, 0x5d, 0x48, 0x89, 0xba, 0xf1, 0xaf, 0xcb, 0x3b, 0x75, 0x99, 0x05,
	0xee, 0x01, 0x6f, 0x40, 0xfe, 0xcb, 0xe8, 0xef, 0xc0, 0x89, 0xa7, 0xcf,
	0x5c, 0x1c, 0xff, 0xda, 0x8b, 0x79, 0x3d, 0x03, 0x72, 0x28, 0x29, 0x30,
	0x33, 0x97, 0x11, 0x9a, 0x99, 0x9b, 0x27, 0x1a, 0x8d, 0x15, 0x12, 0x4a,
	0x00, 0xcf, 0x80, 0x4f, 0x81, 0x71, 0xe0, 0x13, 0x60, 0xcc, 0x39, 0x32,
	0xfc, 0x45, 0x76, 0xa3, 0xed, 0x00, 0x0f, 0x80, 0x1f, 0xbd, 0xf7, 0xe1,
	0x5f, 0xd0, 0x15, 0x6b, 0xd9, 0x53, 0xa2, 0x67, 0x65, 0xcd, 0xcf, 0xb4,
	0xcb, 0xcb, 0xd4, 0xb4, 0x9b, 0x85, 0xa5, 0xc2, 0x67, 0x3c, 0xb0, 0x42,
	0xf2, 0xed, 0x37, 0x91, 0x12, 0x9c, 0x00, 0x26, 0x9d, 0x23, 0xc3, 0xc1,
	0x9d, 0x3a, 0xa4, 0x2d, 0xef, 0x71, 0xda, 0xd1, 0xdd, 0xfb, 0x0b, 0xe0,
	0xdc, 0x0e, 0xed, 0x03, 0xc0, 0x67, 0x29, 0xb1, 0x7f, 0xa4, 0xf2, 0x49,
	0xe7, 0xc8, 0xf0, 0xee, 0x56, 0x61, 0x01, 0x2b, 0xf4, 0x38, 0xed, 0x01,
	0xde, 0x01, 0xba, 0x00, 0x43, 0x96, 0xe0, 0x24, 0x30, 0x5b, 0xe8, 0x89,
	0xfd, 0xbf, 0xd8, 0xbf, 0x00, 0x33, 0x30, 0x7d, 0x18, 0x07, 0x61, 0x44,
	0x81, 0x00, 0x00, 0x00, 0x00, 0x49, 0x45, 0x4e, 0x44, 0xae, 0x42, 0x60,
	0x82
};

#include <iostream>
#include <limits>
using namespace std;
#include <GL/glut.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qimage.h>
#include <Music/Music.h>

void GLStatistics::AverageNote::init()
{
	err_mean = 0.0;
	err_std = 0.0;
}
GLStatistics::AverageNote::AverageNote()
	: ht(-1000)
{
	init();
}
GLStatistics::AverageNote::AverageNote(int h)
	: ht(h)
{
	init();
}
GLStatistics::AverageNote::AverageNote(int h, int num, int den)
	: ht(h)
{
	init();
	factor = QString::number(num)+"/"+QString::number(den);
}
GLStatistics::AverageNote::AverageNote(int h, float cents)
	: ht(h)
{
	init();
	factor = QString::number(cents);
}
QString GLStatistics::AverageNote::getName() const
{
	return Music::h2n(ht)+factor;
}
void GLStatistics::AverageNote::addErr(float err)
{
	// mean estimation
	if(!errs.empty())
		err_mean *= errs.size();
	errs.push_back(err);
 	err_mean += err;
 	err_mean /= errs.size();

	// standard deviation estimation
	err_std = 0.0;
	for(size_t i=0; i<errs.size(); i++)
	{
		float de = errs[i]-err_mean;
		err_std += de*de;
	}
	if(errs.size()>1)
	{
		err_std /= errs.size()-1;						// unbiased estimation of the variance
		err_std = sqrt(err_std);
	}
}

void GLStatistics::addNote(int ht, float err)
{
	if(isHidden() && !setting_keep_hidden->isOn())
		return;

	if(ht<setting_scale_min->minValue() || ht>setting_scale_max->maxValue())
		return;

// 	cerr << "ht=" << ht << endl;

	if(setting_scale_auto->isOn())
	{
		if(m_avg_notes.empty())
		{
			m_avg_notes.resize(1);
			m_avg_notes[0] = AverageNote(ht);
			setting_scale_min->setValue(ht);
			setting_scale_max->setValue(ht);
		}
		else if(ht<setting_scale_min->value())
		{
// 			cerr << "min: ";
			int min_ht = ht;
			vector<AverageNote> avg_notes(setting_scale_max->value()-min_ht+1);
			for(int i=0; i<setting_scale_min->value()-min_ht; i++)
				avg_notes[i].ht = min_ht+i;
			for(size_t i=0; i<m_avg_notes.size(); i++)
				avg_notes[setting_scale_min->value()-min_ht+i] = m_avg_notes[i];
			m_avg_notes = avg_notes;
			setting_scale_min->setValue(ht);
/*			for(size_t i=0; i<m_avg_notes.size(); i++)
				cerr << m_avg_notes[i].ht << " ";
			cerr << "[" << setting_scale_min->value() << ":" << setting_scale_max->value() << "]" << endl;*/
		}
		else if(ht>setting_scale_max->value())
		{
// 			cerr << "max: ";
			int max_ht = ht;
			vector<AverageNote> avg_notes(max_ht-setting_scale_min->value()+1);
			for(size_t i=0; i<m_avg_notes.size(); i++)
				avg_notes[i] = m_avg_notes[i];
			for(size_t i=m_avg_notes.size(); i<avg_notes.size(); i++)
				avg_notes[i].ht = setting_scale_min->value()+i;
			m_avg_notes = avg_notes;
			setting_scale_max->setValue(ht);
/*			for(size_t i=0; i<m_avg_notes.size(); i++)
				cerr << m_avg_notes[i].ht << " ";
			cerr << "[" << setting_scale_min->value() << ":" << setting_scale_max->value() << "]" << endl;*/
		}

// 		cerr << setting_scale_min->value()-ht << endl;

		m_avg_notes[ht-setting_scale_min->value()].addErr(err);
	}
}
void GLStatistics::addNote(int ht, int num, int den, float err)
{
}
void GLStatistics::addNote(int ht, float cents, float err)
{
}

GLStatistics::GLStatistics(QWidget* parent)
: QGLWidget(parent, "GLStatistics")
, View("Statistics", this)
{
	// settings
	QImage img;
	img.loadFromData(g_icon_statistics, sizeof(g_icon_statistics), "PNG");
	setting_show->setIconSet(QIconSet(QImage(img)));
	setting_show->setOn(false);
	hide();

	setting_reset = new QAction(this);
	setting_reset->setMenuText(tr("Reset statistics"));
	connect(setting_reset, SIGNAL(activated()), this, SLOT(reset()));
	setting_reset->addTo(&m_popup_menu);

	setting_keep_hidden = new QAction(this);
	setting_keep_hidden->setMenuText(tr("Keep notes when hidden"));
	setting_keep_hidden->setToggleAction(true);
	setting_keep_hidden->setOn(false);
	connect(setting_keep_hidden, SIGNAL(toggled(bool)), this, SLOT(update()));
	setting_keep_hidden->addTo(&m_popup_menu);

	setting_show_std = new QAction(this);
	setting_show_std->setMenuText(tr("Show standard deviation"));
	setting_show_std->setToggleAction(true);
	setting_show_std->setOn(true);
	setting_show_std->setEnabled(true);
	connect(setting_show_std, SIGNAL(toggled(bool)), this, SLOT(update()));
	setting_show_std->addTo(&m_popup_menu);

	setting_scale_auto = new QAction(this);
	setting_scale_auto->setMenuText(tr("Scale auto"));
	setting_scale_auto->setToggleAction(true);
	setting_scale_auto->setOn(true);
	setting_scale_auto->setEnabled(false);
	connect(setting_scale_auto, SIGNAL(toggled(bool)), this, SLOT(update()));
	setting_scale_auto->addTo(&m_popup_menu);

	m_popup_menu.insertItem(new Title(tr("Scale min"), &m_popup_menu));
	setting_scale_min = new QSpinBox(-96, 96, 1, &m_popup_menu);
	setting_scale_min->setValue(0);
	QToolTip::add(setting_scale_min, tr("Scale min value (in semi-tones)"));
	connect(setting_scale_min, SIGNAL(valueChanged(int)), this, SLOT(update()));
	m_popup_menu.insertItem(setting_scale_min);
	setting_scale_min->setEnabled(false);

	m_popup_menu.insertItem(new Title(tr("Scale max"), &m_popup_menu));
	setting_scale_max = new QSpinBox(-96, 96, 1, &m_popup_menu);
	setting_scale_max->setValue(0);
	QToolTip::add(setting_scale_max, tr("Scale max value (in semi-tones)"));
	connect(setting_scale_max, SIGNAL(valueChanged(int)), this, SLOT(update()));
	m_popup_menu.insertItem(setting_scale_max);
	setting_scale_max->setEnabled(false);

	setting_showTolerance = new QAction(this);
	setting_showTolerance->setMenuText(tr("Show tolerance"));
	setting_showTolerance->setToggleAction(true);
	setting_showTolerance->setOn(true);
	connect(setting_showTolerance, SIGNAL(toggled(bool)), this, SLOT(update()));
	setting_showTolerance->addTo(&m_popup_menu);

	setting_useCents = new QAction(this);
	setting_useCents->setMenuText(tr("Use cents"));
	setting_useCents->setToggleAction(true);
	setting_useCents->setOn(true);
	connect(setting_useCents, SIGNAL(toggled(bool)), this, SLOT(update()));
	setting_useCents->addTo(&m_popup_menu);

	m_popup_menu.insertItem(new Title(tr("Scale range"), &m_popup_menu));
	setting_spinScale = new QSpinBox(5, 50, 1, &m_popup_menu);
	setting_spinScale->setValue(50);
	QToolTip::add(setting_spinScale, tr("Scale range (in cents)"));
	connect(setting_spinScale, SIGNAL(valueChanged(int)), this, SLOT(update()));
	m_popup_menu.insertItem(setting_spinScale);
}

void GLStatistics::save()
{
	s_settings->writeEntry("setting_keep_hidden", setting_keep_hidden->isOn());
	s_settings->writeEntry("setting_show_std", setting_show_std->isOn());
	s_settings->writeEntry("setting_scale_auto", setting_scale_auto->isOn());
	s_settings->writeEntry("setting_scale_min", setting_scale_min->value());
	s_settings->writeEntry("setting_scale_max", setting_scale_max->value());
}
void GLStatistics::load()
{
	setting_keep_hidden->setOn(s_settings->readBoolEntry("setting_keep_hidden", setting_keep_hidden->isOn()));
	setting_show_std->setOn(s_settings->readBoolEntry("setting_show_std", setting_show_std->isOn()));
	setting_scale_auto->setOn(s_settings->readBoolEntry("setting_scale_auto", setting_scale_auto->isOn()));
	setting_scale_min->setValue(s_settings->readNumEntry("setting_scale_min", setting_scale_min->value()));
	setting_scale_max->setValue(s_settings->readNumEntry("setting_scale_max", setting_scale_max->value()));
}
void GLStatistics::clearSettings()
{
	s_settings->removeEntry("setting_keep_hidden");
	s_settings->removeEntry("setting_show_std");
	s_settings->removeEntry("setting_scale_auto");
	s_settings->removeEntry("setting_scale_min");
	s_settings->removeEntry("setting_scale_max");
}

void GLStatistics::reset()
{
	m_avg_notes.clear();
	setting_scale_min->setValue(0);
	setting_scale_max->setValue(0);
	updateGL();
}

void GLStatistics::initializeGL()
{
	// Set the clear color to black
	glClearColor(1.0, 1.0, 1.0, 0.0);

	// glShadeModel( GL_FLAT );
	glShadeModel( GL_SMOOTH );

	glLoadIdentity();
}

void GLStatistics::drawTicksCent(int r, int dx, int dy)
{
	// only work within range that is a pure multiple of r
	float range = int(setting_spinScale->value()/r)*r;

	float scale = 50.0f/setting_spinScale->value();

	if((height()-dy)*r/100.0f*scale>2)
	{
		for(float i=-range; i<=range; i+=r)
		{
			int y = int((height()-dy)*i/100.0f*scale) + (height()-dy)/2 + dy;
			glVertex2i(dx,  y);
			glVertex2i(width(),  y);
		}
	}
}
void GLStatistics::drawTextTickCent(int r, int dy)
{
	// only work within range that is a pure multiple of r
	int range = int(setting_spinScale->value()/r)*r;

	float scale = 50.0f/setting_spinScale->value();
	QString txt;
	for(int i=-range; i<range; i+=r)
	{
		txt = QString::number(i);
		if(i>=0) txt = QString("  ")+txt;
		if(i==0) txt = QString("  ")+txt;
		glRasterPos2i(2, int((height()-dy)*i/100.0f*scale) + (height()-dy)/2 + dy - 4);
		for(size_t i=0; i<txt.length(); i++)
			glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, txt.latin1()[i]);
	}
}

void GLStatistics::paintGL()
{
// 	cerr << "GLStatistics::paintGL " << m_notes.size() << endl;

	glClear(GL_COLOR_BUFFER_BIT);

	glLineWidth(1.0f);
	float human_tol = Music::f2hf(441.0,440.0);
	float scale;
	if(setting_useCents->isOn())
		scale = 50.0f/setting_spinScale->value();
	else
		scale = int(50/setting_spinScale->value());

	// name
	QString str = tr("Statistics");
	glColor3f(0.75,0.75,0.75);
	glRasterPos2i(2, height()-20);
	for(size_t i = 0; i < str.length(); i++)
		glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18, str.latin1()[i]);

	int char_size = 9;
	int ticks_size = 2+3*char_size;
	int grid_width = width()-ticks_size;
	int dy = char_size/2;
	int legend_height = 40;
	int grid_height = height()-legend_height;

	// draw green rect
	if(setting_showTolerance->isOn())
	{
		int green_ytop = int((scale*human_tol+0.5)*grid_height) + legend_height;
		int green_ybottom = int((-scale*human_tol+0.5)*grid_height) + legend_height;

		glBegin(GL_QUADS);
		glColor3f(168/256.0,221/256.0,182/256.0);
		glVertex2i(ticks_size, green_ytop);
		glVertex2i(ticks_size, green_ybottom);
		glVertex2i(width(), green_ybottom);
		glVertex2i(width(), green_ytop);
		glEnd();

		// draw green lines
		glBegin(GL_LINES);
		glColor3f(0.48,0.80,0.57);
		glVertex2i(ticks_size, green_ytop);
		glVertex2i(width(), green_ytop);
		glVertex2i(ticks_size, green_ybottom);
		glVertex2i(width(), green_ybottom);
		glEnd();
	}

	// std
	if(setting_show_std->isOn())
	{
		glBegin(GL_QUADS);
		glLineWidth(1.0f);
		glColor3f(0.85, 0.85, 0.85);
		for(size_t i=0; i<m_avg_notes.size(); i++)
		{
			if(!m_avg_notes[i].errs.empty())
			{
				int top = int((scale*(m_avg_notes[i].err_mean+m_avg_notes[i].err_std)+0.5)*grid_height) + legend_height;
				top = max(top, legend_height);
				int bottom = int((scale*(m_avg_notes[i].err_mean-m_avg_notes[i].err_std)+0.5)*grid_height) + legend_height;
				bottom = max(bottom, legend_height);
				int left = ticks_size+int(i*float(grid_width)/m_avg_notes.size())+1;
				int right = ticks_size+int((i+1)*float(grid_width)/m_avg_notes.size());

				glVertex2i(left, top);
				glVertex2i(left, bottom);
				glVertex2i(right, bottom);
				glVertex2i(right, top);
			}
		}
		glEnd();
	}

	// horiz lines
	if(setting_useCents->isOn())
	{
		glBegin(GL_LINES);
		float gray = 0.87;
// 		glColor3f(gray, gray, gray);
// 		drawTicksCent(1, ticks_size);
		gray = 0.875;
		glColor3f(gray, gray, gray);
		drawTicksCent(2, ticks_size, legend_height);
		gray = 0.75;
		glColor3f(gray, gray, gray);
		drawTicksCent(10, ticks_size, legend_height);
		glEnd();
	}
	else
	{
		glBegin(GL_LINES);
			float gray = 0.5;
			glColor3f(gray, gray, gray);
			glVertex2i(ticks_size,  grid_height/2+legend_height);
			glVertex2i(width(),  grid_height/2+legend_height);
			gray = 0.75;
			glColor3f(gray, gray, gray);
			glVertex2i(ticks_size,  grid_height/4+legend_height);
			glVertex2i(width(),  grid_height/4+legend_height);
			glVertex2i(ticks_size,  3*grid_height/4+legend_height);
			glVertex2i(width(),  3*grid_height/4+legend_height);
			gray = 0.87;
			glColor3f(gray, gray, gray);
			glVertex2i(ticks_size,  grid_height/8+legend_height);
			glVertex2i(width(),  grid_height/8+legend_height);
			glVertex2i(ticks_size,  7*grid_height/8+legend_height);
			glVertex2i(width(),  7*grid_height/8+legend_height);
			glVertex2i(ticks_size,  3*grid_height/8+legend_height);
			glVertex2i(width(),  3*grid_height/8+legend_height);
			glVertex2i(ticks_size,  5*grid_height/8+legend_height);
			glVertex2i(width(),  5*grid_height/8+legend_height);
		glEnd();
	}

	// center line
// 	glBegin(GL_LINES);
// 	glColor3f(0,0,0);
// 	glVertex2i(ticks_size, grid_height/2+legend_height);
// 	glVertex2i(width(), grid_height/2+legend_height);
// 	glEnd();

	// text marks
	float gray = 0.5;
	glColor3f(gray, gray, gray);
	if(setting_useCents->isOn())
	{
		int grad = 10;
		if(setting_spinScale->value() <= 25) grad=5;
		if(setting_spinScale->value() <= 10) grad=1;
		drawTextTickCent(grad, legend_height);
	}
	else
	{
		string sfraq, sufraq;
		sufraq = string("1/")+QString::number(int(50/setting_spinScale->value())*2).latin1();
		sfraq = string("+")+sufraq;
		glRasterPos2i(2, 3*grid_height/4-dy+legend_height);
		for(size_t i = 0; i < sfraq.size(); i++)
			glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, sfraq[i]);
		sfraq = string("-")+sufraq;
		glRasterPos2i(2, grid_height/4-dy+legend_height);
		for(size_t i = 0; i < sfraq.size(); i++)
			glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, sfraq[i]);

		sufraq = string("1/")+QString::number(int(50/setting_spinScale->value())*4).latin1();
		sfraq = string("+")+sufraq;
		glRasterPos2i(2, 5*grid_height/8-dy+legend_height);
		for(size_t i = 0; i < sfraq.size(); i++)
			glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, sfraq[i]);
		sfraq = string("-")+sufraq;
		glRasterPos2i(2, 3*grid_height/8-dy+legend_height);
		for(size_t i = 0; i < sfraq.size(); i++)
			glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, sfraq[i]);
	}

	// vertical lines
	glBegin(GL_LINES);
	gray = 0.5;
	glColor3f(gray, gray, gray);
	for(size_t i=0; i<m_avg_notes.size(); i++)
	{
		int x = ticks_size+int(i*float(grid_width)/m_avg_notes.size());
		glVertex2i(x, legend_height);
		glVertex2i(x, height());
	}
	glEnd();

	// note names
	glColor3f(0, 0, 1);
	for(size_t i=0; i<m_avg_notes.size(); i++)
	{
		string str = m_avg_notes[i].getName();

		glRasterPos2f(ticks_size+(i+0.5)*float(grid_width)/m_avg_notes.size(), 2);

		for(size_t c=0; c<str.length(); c++)
			glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, str[c]);
	}

	// sizes
	glColor3f(0, 0, 1);
	for(size_t i=0; i<m_avg_notes.size(); i++)
	{
		if(!m_avg_notes[i].errs.empty())
		{

			string str = QString::number(m_avg_notes[i].errs.size());

			glRasterPos2f(ticks_size+(i+0.5)*float(grid_width)/m_avg_notes.size(), 4+legend_height/2);

			for(size_t c=0; c<str.length(); c++)
				glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, str[c]);
		}
	}

	// means
	glLineWidth(2.0f);
	glBegin(GL_LINES);
	glColor3f(0, 0, 0);
	for(size_t i=0; i<m_avg_notes.size(); i++)
	{
		if(!m_avg_notes[i].errs.empty())
		{
			int y = int((scale*(m_avg_notes[i].err_mean)+0.5)*grid_height) + legend_height;
			int left = ticks_size+int(i*float(grid_width)/m_avg_notes.size())+1;
			int right = ticks_size+int((i+1)*float(grid_width)/m_avg_notes.size());

			glVertex2i(left, y);
			glVertex2i(right, y);
		}
	}
	glEnd();

	glFlush();
}

void GLStatistics::resizeGL( int w, int h )
{
	// Set the new viewport size
	glViewport(0, 0, (GLint)w, (GLint)h);

	// Choose the projection matrix to be the matrix
	// manipulated by the following calls
	glMatrixMode(GL_PROJECTION);

	// Set the projection matrix to be the identity matrix
	glLoadIdentity();

	// Define the dimensions of the Orthographic Viewing Volume
	glOrtho(0.0, w, 0.0, h, 0.0, 1.0);

	// Choose the modelview matrix to be the matrix
	// manipulated by further calls
	glMatrixMode(GL_MODELVIEW);
}
