/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * fische-3.0
 * Copyright (C) Marcel Ebmer 2009 <marcel@26elf.at>
 * 
 * fische-3.0 is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * fische-3.0 is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cstdlib>
#include <iostream>
#include <cmath>
#include "vectorfield.h"

using namespace std;

VectorField::VectorField(int x, int y)
{
	xRes = x;
	yRes = y;

	data = NULL;
	
	int n = 0;
	char* f = fillField(n);
	while(f)
	{
		field.push_back(f);
		n++;
		f = fillField(n);
	}

	if(field.empty())
	{
		cerr << "ERROR: unable to calculate any vectorfields. out of memory?" << endl;
		exit(EXIT_FAILURE);
	}

	data = field.at(0);
	cout << "* successfully calculated " << field.size() << " different vectorfields" << endl;
}

VectorField::~VectorField()
{
	for(unsigned int i = 0; i < field.size(); i--)
	{
		free(field[i]);
	}
	field.clear();
}

char* VectorField::get()
{
	return (char*)data;
}

void VectorField::change()
{
	int n = rand() % field.size();
	while(data == field[n]) n = rand() % field.size();
	data = field[n];
}

int VectorField::width()
{
	return xRes;
}

int VectorField::height()
{
	return yRes;
}

char* VectorField::fillField(int n)
{
	char* v;
	double d;
	char* f = (char*)malloc(xRes * yRes * 2);
	if(!f) return NULL;
	switch(n)
	{
		case 0:
			// linear vectors showing away from a horizontal mirror axis
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					*v = 0;
					validateX(x, v);
					*(v + 1) = (y < yRes / 2) ? (yRes / 100) : (-yRes / 100);
					validateY(y, v + 1);
				}
			}
			return f;
		case 1:
			// linear vectors showing away from a vertical mirror axis
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					*v = (x < xRes / 2) ? (xRes / 100) : (-xRes / 100);
					validateX(x, v);
					*(v + 1) = 0;
					validateY(y, v + 1);
				}
			}
			return f;
		case 2:
			// radial vectors showing away from the center
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					*v = -(x - xRes / 2) / 25;
					*(v + 1) = -(y - yRes / 2) / 25;
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 3:
			// tangential vectors (right)
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					*(v + 1) = -(x - xRes / 2) / 25;
					*v = (y - yRes / 2) / 25;
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 4:
			// tangential vectors (left)
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					*(v + 1) = (x - xRes / 2) / 25;
					*v = -(y - yRes / 2) / 25;
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 5:
			// tangential-radial vectors (left)
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					*(v + 1) = (x - xRes / 2) / 25 - (y - yRes / 2) / 25;;
					*v = -(y - yRes / 2) / 25 - (x - xRes / 2) / 25;;
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 6:
			// tangential-radial vectors (right)
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					*(v + 1) = -(x - xRes / 2) / 25 - (y - yRes / 2) / 25;;
					*v = (y - yRes / 2) / 25 - (x - xRes / 2) / 25;;
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 7:
			// radial sinewave vectors
			d = 25 / (double)xRes;
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					double r = sqrt(pow((double)(x - xRes / 2), 2) + pow((double)(y - yRes / 2), 2)) * d;
					*v = (char)(-(x - xRes / 2) / 25 - (double)((y - yRes / 2) / 50) * sin(r));
					*(v + 1) = (char)(-(y - yRes / 2) / 25 + (double)((x - xRes / 2) / 50) * sin(r));
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 8:
			// hyperbolic vectors
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					*v = (y - yRes / 2) / 25;
					*(v + 1) = (x - xRes / 2) / 25;
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 9:
			// hyperbolic vectors
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					*v = -(y - yRes / 2) / 25;
					*(v + 1) = -(x - xRes / 2) / 25;
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 10:
			// purely random
			for(int x = 0; x < xRes; x++)
			{
				for(int y = 0; y < yRes; y++)
				{
					v = f + 2 * x + y * 2 * xRes;
					*v = (rand() % xRes - xRes / 2) / 50;
					*(v + 1) = (rand() % yRes - yRes / 2) / 50;
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		default:
			// index too high. return nothing.
			free(f);
			return NULL;
	}
}
