#!/bin/sh

#    The contents of this file are free software; you can redistribute it 
#    and/or modify it under the terms of the Firebird Public Licence, 
#    available at http://firebird.sourceforge.net/FBPL.html; either 
#    version 1.0 of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    Firebird Public License for more details.
#
#    This file was created by members of the firebird development team.  
#    All individual contributions remain the Copyright (C) of those 
#    individuals.  Contributors to this file are either listed here or
#    can be obtained from a CVS history command.
#
#    All rights reserved.

#    Contributor(s):
#       Tom Coleman TMC Systems <tcoleman@autowares.com>
#       Reed Mideke <rfm@cruzers.com>
#       Mark O'Donohue <mark.odonohue@ludwig.edu.au>
#       Neil McCalden <nm@zizz.org>
#       Erik Kunze <kunze@philosys.de>

#
# $Id: Configure.sh,v 1.14 2002/06/21 09:30:43 paul_reeves Exp $
#
# setup_dirs will create 'jrd/build_no.h' by a call to
# a slightly modified 'builds_win32/original/build_no.ksh'
# that gets IB_BUILD_NO from 'this_build', that hopefully
# will be increased automatically in the near future :-)
#
# Shell script to set up the Firebird directory structure
#
#
# + MasterSourceDirectory
#    + interbase     - kit will be built here
#
# + refDatabases
#    + jrd
#       isc.gdb
#       metadata.gdb
#    + msgs
#       msg.gdb
#    + qli
#       help.gdb
#    + utilities
#       rebuild.gdb


# To run this script without it prompting for conformation of the setup:
# $NOPROMPT_SETUP="Anything"; 
# $export NOPROMPT_SETUP
# before running this script file.
# If you want 64bit IO with noprompt on a platform where it is still optional
# set FIREBIRD_64_BIT_IO or it will default to 32bit.

# Bourne shell (eg: /bin/sh) on Solaris can not do export and
# assignment in one command, do assignment and export as two commands

# EOL suppression is also different
# Bash (BSD?) variations: echo -n "xyz"
# Solaris (SYSV) : echo "xyz\c"

SUPP=`echo -n "x"`

if [ "${SUPP}" = "x" ]; then
   MN="-n"
   SC=""
else
   MN=""
   SC="\c"
fi


#------------------------------------------------------------------------
# Prompt for response, store result in Answer

Answer=""

AskQuestion() {
    Test=$1
    DefaultAns=$2
    Answer="$DefaultAns"

    if [ -z "$NOPROMPT_SETUP" ]
      then
        echo ${MN} "${1}${SC}"
        read Answer
    fi
}


#------------------------------------------------------------------------
# Prompt for yes or no answer - returns non-zero for no

AskYNQuestion() {
    while echo ${MN} "${*} (y/n): ${SC}"
    do
        read answer rest
        case $answer in
        [yY]*)
            return 0
            ;;
        [nN]*)
            return 1
            ;;
        *)
            echo "Please answer y or n"
            ;;
        esac
    done
}



#------------------------------------------------------------------------
#
# In order to build the super directories 
# The old one is deleted (if it exists) and a new one is created
# and the source copied from the classic ones.
#
#    refreshLink ../../$ClassicDirName/[a-z]*[^oa] . 
#    is an attempt not to catch .o and .a files if redoing setup
#

buildSuperDir() {
    ClassicDirName=$1
    SuperDirName="super/$ClassicDirName"

    if [ -d $SuperDirName ] 
     then
      rm -rf $SuperDirName
    fi

    mkdir -p $SuperDirName

    cd $SuperDirName
    case $BuildHostType in
     SOLARIS|SCO_EV|HP11)
      ln -s ../../$ClassicDirName/[a-z0-9]*[!oa] .  
      ;;
     SINIXZ)
      ln -s ../../$ClassicDirName/[a-l]*[!oa] .  
      ln -s ../../$ClassicDirName/[m-z]*[!oa] .  
      ln -s ../../$ClassicDirName/[0-9]*[!oa] .  
      ;;
     *)
      ln -s ../../$ClassicDirName/[a-z0-9]*[^oa] .  
      ;;
    esac
    cd ../..

}


#------------------------------------------------------------------------
#
# Build all the directories needed by super server.

buildSuperDirs() {
    mkdir -p super/client 

# Now populate the directories super/* with symlinks to all the source
# files contained in the corresponding top-level directories.

    for dir in alice burp dsql intl journal jrd lock register remote utilities wal
      do
        buildSuperDir $dir
      done
}


#------------------------------------------------------------------------
#
# Refresh a link, if it already exists delete it then recreate it
# otherwise just create it.
#

refreshLink() {
   RealFile=$1
   LinkFile=$2

   if [ $BuildHostType = "SOLARIS"  -o $BuildHostType = "SINIXZ" ]; then   
      if [ -h $LinkFile ]; then
         rm -f $LinkFile
      fi    
   else 
      if [ -L $LinkFile ]; then
         rm -f $LinkFile
      fi
   fi
   
   ln -s $RealFile $LinkFile

}


#------------------------------------------------------------------------
#
# Copy file if it exists
#

copyIfExists() {
   OldFile=$1
   NewFile=$2
   
   if [ -f $OldFile ]
     then
       cp $OldFile $NewFile
   fi

}


#------------------------------------------------------------------------
#
# remove file if it exists
#

rmIfExists() {
   FileName=$1
   
   if [ -f $FileName ]
     then
       rm -f $FileName
   fi

}



#------------------------------------------------------------------------
# print usage for calling this script

printUsage() {
    echo ""
    echo "usage is : ./Configure.sh DEV|PROD [system]"
    echo ""
    echo "Where system defaults to : `uname -s` "
    echo "or you can manually specify one of the following:"
    echo "               AIX|AP|AX|DELTA|DG|EPSON|HP700|HP800|HP9.0|"
    echo "               HP10|HP11|IMP|MU|SCO|SGI|SOLARIS|SUN4|UNIXWARE|"
    echo "               AIX_PPC|LINUX|FREEBSD|NETBSD|DARWIN|SINIXZ"
    echo ""
    echo ""


}

#------------------------------------------------------------------------
# workout the current OS system type on the machine we are running.

getDefaultSystemType() {
    SystemName=`uname -s | tr [:lower:] [:upper:]`

    case $SystemName in 
    LINUX|FREEBSD|NETBSD|DARWIN) 
       BuildHostType=$SystemName
       ;;
    IRIX*) 
       BuildHostType=SGI
       ;;
    SUNOS) 
        #test it is Solaris not SunOS
        OSversion=`uname -r`
        case $OSversion in
            5.*)
                BuildHostType=SOLARIS
                ;;
            *)
                echo
                echo "The SunOS 4.x port needs dusting off and testing."
                echo
                exit 1
        esac

        Arch=`uname -p`
       ;;
    SINIX-Z)
       BuildHostType=SINIXZ
       ;;
    *)
       BuildHostType=""
       ;;
    esac
}

#------------------------------------------------------------------------
# workout the unix 64 bit io stuff

setUnix64BitIo() {

    case "$SYS_TYPE" in 
    LINUX|FREEBSD|NETBSD|SGI|SOLARIS) 
		if [ -z "$NOPROMPT_SETUP" ]; then
			AskYNQuestion "Build a 64 bit IO version of the engine"
			Answer=$?
		else
			if [ -z "$FIREBIRD_64_BIT_IO" ]; then
				# not defined so default to 32bit
				Answer=1
			else
				Answer=0
			fi
		fi
			
		if [ $Answer -eq 0 ]; then
		 	echo "#ifndef _64_BIT_IO_H" > jrd/64bitio.h
			echo "#define _64_BIT_IO_H" >> jrd/64bitio.h
			echo "#define UNIX_64_BIT_IO" >> jrd/64bitio.h
			echo "#endif" >> jrd/64bitio.h
			BuildIOsize=64
		else
			echo "" > jrd/64bitio.h
		fi
       ;;
    DARWIN)
        echo "#define UNIX_64_BIT_IO" > jrd/64bitio.h
		BuildIOsize=64
        ;;
    *)
           echo "" > jrd/64bitio.h
       ;;
    esac
}
#------------------------------------------------------------------------
# set the INTERBASE environment value correctly for DARWIN systems

setDarwinInterbaseEnv() {
  if [ "$BuildBootFlg" = "Yes" ]
    then 
      INTERBASE="/Library/Frameworks/Firebird.framework/Resources/English.lproj/var" 
      export INTERBASE
    else
      cd firebird/bellardo/darwin
      rm -f varpath
      cc -framework Firebird -DVAR_PATH installpath.c -o varpath
      if [ -f varpath ] 
        then
          INTERBASE="`./varpath`"
          export INTERBASE
        else
          INTERBASE="/Library/Frameworks/Firebird.framework/Resources/English.lproj/var"
          export INTERBASE
      fi
      cd ../../..
  fi
}
#------------------------------------------------------------------------
# Check initial variables required for build of interbase

checkVariables() {

     if [ "$INTERBASE" = "" -o "$INTERBASE" = "`pwd`/interbase" ]
       then
         if [ $SYS_TYPE = 'FREEBSD' -o $SYS_TYPE = 'NETBSD' -o $SYS_TYPE = 'SCO_EV' ]
           then
             INTERBASE="/usr/interbase"
             export INTERBASE
           else
             if [ $SYS_TYPE = 'DARWIN' ]
               then
                 setDarwinInterbaseEnv
               else
                 INTERBASE="/opt/interbase"
                 export INTERBASE

                 ## RITTER - the library path is and modified during the build process
                 # In order to ensure correct behaviour in case of a re-build, it's a
                 # good idea to reset the variable.
                 LD_LIBRARY_PATH="${INTERBASE}/lib"
                 export LD_LIBRARY_PATH

               fi
	   fi
     fi

     if [ "$ISC_PASSWORD" = "" ]
       then
         ISC_USER="sysdba"
         ISC_PASSWORD="masterkey"
         export ISC_USER ISC_PASSWORD
     fi


     echo ""
     echo ""
     echo ""
     echo "- Firebird - Database build setup ----------------------------"
     echo "" 
     echo "From command line :"
     echo "Host  OS Type                            : $BuildHostType"
     echo "Build Type                               : $BuildBuildType"
     echo "Boot Type Build                          : $BuildBootFlg"
     echo ""
     echo "File IO bit size (32/64)                 : $BuildIOsize"
     echo ""
     echo "From env. variables:"
     if [ "$BuildBootFlg" = "No" ]
       then
         echo "INTERBASE    (previous firebird install) : $INTERBASE "
     fi
     echo "ISC_USER     (admin user)                : $ISC_USER"
     echo "ISC_PASSWORD (admin password)            : $ISC_PASSWORD"
     echo "" 
     echo "If you wish to have different values please set them before running" 
     echo "this script"
     echo "--------------------------------------------------------------"
     echo ""
     AskQuestion "Press return to continue"

}


#------------------------------------------------------------------------
# copy the boot .c files from the porting subdirectory into the
# build structure.

copyFilesFromPortingDir() {


    if [ ! -d porting ]
      then
        echo "Unable to find the porting subdirectory for the boot build"
        echo "Have you installed the firebird_boot kit?"
        exit
    fi
    
    for dir in `ls porting`
      do
        if [ -d $dir ]; then
          for portfile in `ls porting/$dir`
            do
#              echo "copying porting/$dir/$portfile"
              cp porting/$dir/$portfile $dir
            done
        fi
      done
}


#------------------------------------------------------------------------
# Build the interbase target dir, and fill it where required from the
# currently installed interbase build.


buildTargetInterbaseDirs() {

    if [ -d interbase ] 
      then
        echo ""
        echo ""
        echo "*** Warning *** "
        echo "    The installation target directory:"
        echo "       `pwd`/interbase"
        echo "    already exists, press return to delete this directory or ^C to abort."
        echo "*** Warning *** "
        echo "" 
        AskQuestion "Press return to continue or ^C to abort"
        rm -rf interbase
        echo "Deleted."
        echo ""
    fi

    mkdir interbase

    cd interbase

    mkdir bin
    mkdir lib
    mkdir help
    mkdir UDF
    mkdir -p examples/v3
    mkdir -p examples/v4
    mkdir -p examples/v5
    mkdir include
    mkdir intl

    if [ $BuildBootFlg = "No" ]
      then
        cp $INTERBASE/interbase.msg .
        cp $INTERBASE/isc4.gdb .
        copyIfExists $INTERBASE/ib_license.dat .
        cp $INTERBASE/bin/gpre bin
        cp $INTERBASE/bin/gbak bin
        copyIfExists $INTERBASE/bin/gds_lock_mgr bin
        cp $INTERBASE/help/help.gdb help
        if [ $SYS_TYPE = 'SCO_EV' ]
          then
            refreshLink $INTERBASE/isc_lock1.`uname -n` .
        fi
    fi

    cd ..


}

#------------------------------------------------------------------------
# Check if the system is setup to build from the boot files.
# (we check to see if the porting subdirectory exists).

checkIfBootBuild() {

    if [ -d porting ]
      then
        echo "- Found 'porting' subdirectory - assuming boot type build"
        echo ""

        BuildBootFlg="Yes"
    fi
}


#== Main Program ========================================================


#


PATH=.:$PATH


if [ $# -eq 0 ]
  then
    printUsage
    exit
fi


BuildBootFlg="No"
BuildHostType=""
BuildBuildType="PROD"
MasterSourceDir="`pwd`/refDatabases"
BuildIOsize=32

export BuildHostType
export BuildBuidType
export MasterSourceDir


# Check for boot arg or help arg
checkIfBootBuild

# Work out the default build parameters
getDefaultSystemType


# Check the command line arguments

if [ $# -ge 1 ]
  then
    BuildBuildType=$1
fi

if [ $# -ge 2 ]
  then
    BuildHostType=$2
fi


SYS_TYPE=$BuildHostType


if [ "$SYS_TYPE" = "" ]
  then
    echo "Unable to automatically determine the OS system type for the build"
    echo "you will need to specify it manually"
    echo ""
    printUsage
    exit
fi

setUnix64BitIo
checkVariables

#build reference Databases
echo "- Setting build version strings in file jrd/build_no.h"
echo ""

if [ "$BuildBootFlg" = "Yes" ]
  then
    ./builds/original/buildRefDatabases boot
  else
    ./builds/original/buildRefDatabases
fi



# Some checking for old systems, 
# Buggered if I know if these are needed but I've copied them across.
if [ "$BuildHostType" = 'SUN4' -o "$BuildHostType" = 'SUN3_4' -o "$BuildHostType" = 'SUN386I' ]; then
    SYS_TYPE=SUNOS4
fi
if [ $BuildHostType = 'AX' -o $BuildHostType = 'AP' ]; then
    SYS_TYPE=APOLLO
fi
if [ $BuildHostType = 'SUN3_3' ]; then
    SYS_TYPE=SUNOS3
fi
#
LOCK_HEADER=lock_header_shmem
if [ $SYS_TYPE = 'SUNOS4' -o $BuildHostType = 'DG' -o $BuildHostType = 'SOLARIS' ]; then
    LOCK_HEADER=lock_header_mmap
fi
#



# Here we ensure that the build_no version strings are automatically set to 
# the correct current values ( thanks to Frank for this).

echo ""
echo "- Setting build version strings in file jrd/build_no.h"
echo ""
sh builds_win32/original/build_no.ksh
if [ $? -ne 0 ]; then
  echo "Check builds_win32/original/build_no.ksh  - somebody has broken it!"
  exit 1
fi


# The boot build copies a number of already precompiled C files into
# the build structure.

if [ $BuildBootFlg = "Yes" ]
  then
    echo "- Copying boot files into build tree"
    echo ""
    copyFilesFromPortingDir
fi


echo "- Setting up the superserver directory structure"
echo ""

buildSuperDirs


# Refresh/create a few links

refreshLink . source

refreshLink builds/make.interbase Makefile

refreshLink builds/original/build_kit build_kit
refreshLink builds/original/set_prot set_prot
refreshLink builds/original/unset_prot unset_prot
refreshLink builds/original/add_template add_template
refreshLink builds/original/mpm_template mpm_template
refreshLink builds/original/mpp_template mpp_template
refreshLink builds/original/put_template put_template
refreshLink builds/original/fetch_comp fetch_comp
refreshLink builds/original/maketar maketar

refreshLink builds/refresh refresh
refreshLink builds/refresh.all refresh.all
refreshLink builds/special_opt special_opt


if [ $BuildHostType = 'SCO' ]
  then
    refreshLink builds/original/sco_ship sco_ship
fi



# Build the empty target interbase directories 

buildTargetInterbaseDirs


# Update the metadata/isc links
cd jrd
refreshLink $MasterSourceDir/jrd/isc.gdb isc.gdb
refreshLink $MasterSourceDir/jrd/metadata.gdb yachts.gdb
cd ..

cd super/jrd                                  # TMC 081100
refreshLink ../../jrd/yachts.gdb yachts.gdb   # TMC 081100
refreshLink ../../jrd/isc.gdb    isc.gdb      # FSG 090800
cd ../..                                      # TMC 081100 



echo "- Creating messages link"
echo ""

cd msgs
refreshLink $MasterSourceDir/msgs/msg.gdb msg.gdb
cd ..
# Link these to the ones we will build, so we can do -lgds and -lgds_pyxis
cd jrd
refreshLink source/interbase/lib/gds.so libgds.so
refreshLink source/interbase/lib/gds_pyxis.a libgds_pyxis.a
cd ..


if [ $BuildBootFlg = 'Yes' ]
  then
#
#     INTERBASE in the level above is the source root directory
#
    touch interbase/interbase.msg

    cd interbase/lib
    refreshLink gds.so libgds.so.1.0
    refreshLink libgds.so.1.0 libgds.so.1
    refreshLink libgds.so.1 libgds.so
    refreshLink gds_pyxis.a libgds_pyxis.a
    refreshLink gds.a libgds.a
    cd ../..
fi


echo "- Setting up makefile and database links"
echo ""

# Alice
refreshLink .. alice/source
refreshLink source/jrd/yachts.gdb alice/yachts.lnk
refreshLink source/builds/make.alice alice/makefile
refreshLink .. super/alice/source
refreshLink source/jrd/yachts.gdb super/alice/yachts.lnk
refreshLink source/builds/make.alice_ss super/alice/makefile

# Burp
refreshLink .. burp/source
refreshLink source/jrd/yachts.gdb burp/yachts.lnk
refreshLink source/builds/make.burp burp/makefile
refreshLink .. super/burp/source
refreshLink source/jrd/yachts.gdb super/burp/yachts.lnk
refreshLink source/builds/make.burp_ss super/burp/makefile

# Client
refreshLink .. super/client/source
refreshLink source/builds/make.client super/client/makefile

# Csv
refreshLink .. csv/source
refreshLink source/builds/make.csv csv/makefile

# Dsql
refreshLink .. dsql/source
refreshLink source/jrd/yachts.gdb dsql/yachts.lnk
refreshLink source/builds/make.dsql dsql/makefile
refreshLink .. super/dsql/source
refreshLink source/jrd/yachts.gdb super/dsql/yachts.lnk
refreshLink source/builds/make.dsql_ss super/dsql/makefile

# Dudley
refreshLink .. dudley/source
refreshLink source/jrd/yachts.gdb dudley/yachts.lnk
refreshLink source/builds/make.dudley dudley/makefile

# V3 Examples
refreshLink .. examples/source
refreshLink $MasterSourceDir/examples/atlas.gdb examples/atlas.gdb
refreshLink $MasterSourceDir/examples/c_guide.gdb examples/c_guide.gdb
refreshLink $MasterSourceDir/examples/emp.gdb examples/emp.gdb
refreshLink $MasterSourceDir/examples/nc_guide.gdb examples/nc_guide.gdb
refreshLink $MasterSourceDir/examples/slides.gdb examples/slides.gdb
refreshLink $MasterSourceDir/examples/stocks.gdb examples/stocks.gdb
refreshLink source/builds/make.examples examples/makefile

# V4 Examples
refreshLink .. example4/source
refreshLink source/builds/make.example4 example4/makefile
refreshLink $MasterSourceDir/example4/empbuild.gdb example4/empbuild.gdb

# V5 Examples
refreshLink .. example5/source
refreshLink source/builds/make.example5 example5/makefile
refreshLink $MasterSourceDir/example5/empbuild.gdb example5/empbuild.gdb

# ExtLib
refreshLink .. extlib/source
refreshLink source/builds/make.extlib extlib/makefile

# Gpre
refreshLink .. gpre/source
refreshLink source/jrd/yachts.gdb gpre/yachts.lnk
refreshLink source/builds/make.gpre gpre/makefile

# Intl
refreshLink .. intl/source
refreshLink source/builds/make.intl intl/makefile
refreshLink .. super/intl/source
refreshLink source/builds/make.intl_ss super/intl/makefile

# Isql
refreshLink .. isql/source
refreshLink source/jrd/yachts.gdb isql/yachts.lnk
refreshLink source/builds/make.isql isql/makefile

# Journal
refreshLink .. journal/source
refreshLink $MasterSourceDir/journal/journal.gdb journal/journal.gdb
refreshLink $MasterSourceDir/journal/journal_vms.gdb journal/journal_vms.gdb
refreshLink source/builds/make.journal journal/makefile
refreshLink .. super/journal/source
refreshLink $MasterSourceDir/journal/journal.gdb super/journal/journal.gdb
refreshLink source/builds/make.journal_ss super/journal/makefile

# Jrd
refreshLink .. jrd/source
refreshLink yachts.gdb jrd/yachts.lnk
refreshLink $MasterSourceDir/msgs/msg.gdb msgs/msg.gdb  # TMC 082100
refreshLink source/msgs/msg.gdb jrd/msg.gdb   # TMC 081100
refreshLink source/builds/make.jrd jrd/makefile
refreshLink source/pipe/allp.c jrd/allp.c
refreshLink source/pipe/foot.c jrd/foot.c
refreshLink source/pipe/head.c jrd/head.c
refreshLink .. super/jrd/source
refreshLink gds_ss.a super/jrd/libgds_ss.a
refreshLink $MasterSourceDir/msgs/msg.gdb super/jrd/msg.gdb
refreshLink source/builds/make.jrd_ss super/jrd/makefile
refreshLink source/builds/original/bind_dsql.apollo dsql/bind_dsql
refreshLink source/builds/original/bind_gdslib.apollo jrd/bind_gdslib
refreshLink source/builds/original/bind_gds5.apollo jrd/bind_bridge
refreshLink source/builds/original/bind_pyxis.apollo pyxis/bind_pyxis
refreshLink source/builds/original/bind_csilib.apollo csv/bind_csilib
refreshLink source/builds/original/bind_remotelib.apollo remote/bind_remotelib

if [ $BuildHostType = 'DELTA' -o $BuildHostType = 'IMP' -o $BuildHostType = 'MU' -o $BuildHostType = 'NXT' -o $BuildHostType = 'SCO' -o $BuildHostType = 'UL' ]
  then
    refreshLink source/interbase/lib/gds_b.a jrd/libgds_b.a
else
    refreshLink gds_b.a jrd/libgds_b.a
fi

if [ $SYS_TYPE = 'SINIXZ' ]
  then
    refreshLink gds.so.0 source/interbase/lib/gds.so
    refreshLink gds.so.0 source/interbase/lib/libgds.so.0
fi

if [ $SYS_TYPE = 'SUNOS4' ]
  then
    refreshLink source/interbase/lib/gdslib.so.1.0 jrd/libgdslib.so.1.0
    refreshLink source/interbase/lib/gdslib.so.0.1 jrd/libgdslib.so.0.1
    refreshLink source/interbase/lib/gdsflib.so.0.0 jrd/libgdsflib.so.0.0
    refreshLink source/interbase/lib/gds_pyxis.a jrd/libgds_pyxis.a
fi

if [ $BuildHostType = 'AIX' -o $BuildHostType = 'DECOSF' -o $BuildHostType = 'SGI' ]
  then
    refreshLink source/builds/original/bind_gds.generic jrd/gds.bind
    refreshLink source/builds/original/bind_gds5.generic jrd/gds5.bind
fi
if [ $BuildHostType = 'AIX_PPC' ]
  then
   refreshLink source/builds/original/bind_gds.aix_ppc jrd/gds.bind
   refreshLink source/builds/original/bind_gdsintl.aix_ppc intl/gdsintl.bind
fi


if [ $BuildHostType = 'AIX' -o $BuildHostType = 'AIX_PPC' ]
  then
    refreshLink source/interbase/lib/gdsshr.a jrd/libgdsshr.a
fi

if [ $BuildHostType = 'DG' ]
  then
    refreshLink source/interbase/lib/gds.so.0 jrd/libgds.so.0
    refreshLink source/jrd/libgds.so.0 jrd/libgds.so
    refreshLink source/interbase/lib/gdsf.so jrd/libgdsf.so
    refreshLink source/interbase/lib/gds_pyxis.a jrd/libgds_pyxis.a
fi
if [ $BuildHostType = 'HP700' -o $BuildHostType = 'HP9.0' -o $BuildHostType = 'HP10' -o $BuildHostType = 'HP11' ]; then
    if [ -d super ]; then
       refreshLink source/builds/original/bind_gds.hp super/remote/gds.bind
       refreshLink source/builds/original/bind_gds.hp super/client/gds.bind
    fi
    refreshLink source/interbase/lib/gds.sl jrd/libgds.sl
    refreshLink source/interbase/lib/gds_pyxis.a jrd/libgds_pyxis.a
    refreshLink source/builds/original/bind_gds.hp jrd/gds.bind
    refreshLink source/builds/original/bind_gds5.hp jrd/gds5.bind
fi
if [ $BuildHostType = 'SOLARIS' -o $BuildHostType = 'SGI' -o $BuildHostType = 'SCO_EV' ]; then
    refreshLink source/interbase/lib/gdsmt.so.1 jrd/libgdsmt.so.1
    refreshLink source/interbase/lib/gdsmt.so.0 jrd/libgdsmt.so.0
    refreshLink source/jrd/libgdsmt.so.0 jrd/libgdsmt.so
    refreshLink source/interbase/lib/gds.so.0 jrd/libgds.so.0
    refreshLink source/jrd/libgds.so.0 jrd/libgds.so
    refreshLink source/interbase/lib/gds_pyxis.a jrd/libgds_pyxis.a
fi
if [ $BuildHostType = 'SCO' -o $BuildHostType = 'IMP' ]; then
    refreshLink source/interbase/lib/gds_s.a jrd/libgds_s.a
    refreshLink source/interbase/lib/gdsf_s.a jrd/libgdsf_s.a
    if [ $BuildHostType = 'SCO' ]; then
        refreshLink source/interbase/lib/gds_pyxis.a jrd/libgds_pyxis.a
    fi
fi
if [ $BuildHostType = 'IMP' ]; then
    refreshLink source/builds/original/bind_gds_s.im jrd/gds_s.bind
    refreshLink source/builds/original/bind_gdsf_s.im jrd/gdsf_s.bind
fi
if [ $BuildHostType = 'SCO' ]; then
    refreshLink source/builds/original/bind_gds_s.sco jrd/gds_s.bind
    refreshLink source/builds/original/bind_gdsf_s.sco jrd/gdsf_s.bind
fi
# Lock
refreshLink .. lock/source
refreshLink source/builds/make.lock lock/makefile
refreshLink .. super/lock/source
refreshLink source/builds/make.lock_ss super/lock/makefile
refreshLink source/lock/$LOCK_HEADER lock/lock_header
# Msgs
refreshLink .. msgs/source
refreshLink source/msgs/msg.gdb msgs/master_msg_db # TMC 081100
#refreshLink master_msg_db msgs/msg.gdb
refreshLink source/builds/make.msgs msgs/makefile
# Pipe
refreshLink .. pipe/source
refreshLink source/builds/make.pipe pipe/makefile
# Pyxis
refreshLink .. pyxis/source
refreshLink source/jrd/yachts.gdb pyxis/yachts.lnk
refreshLink $MasterSourceDir/pyxis/forms.gdb pyxis/forms.gdb
refreshLink source/interbase/lib/gds_pyxis.a pyxis/libgds_pyxis.a
refreshLink source/builds/make.pyxis pyxis/makefile
# Qli
refreshLink .. qli/source
refreshLink source/jrd/yachts.gdb qli/yachts.lnk
refreshLink source/jrd/yachts.gdb qli/yachts.link
refreshLink $MasterSourceDir/qli/help.gdb qli/help.gdb              # TMC 081700
refreshLink $MasterSourceDir/qli/help.gdb qli/master_help_db        # TMC 081700
refreshLink source/builds/make.qli qli/makefile
# Register
# Probably registration code - no source from Inprise
#refreshLink .. register/source
#refreshLink .. super/register/source
# Remote
refreshLink .. remote/source
refreshLink source/builds/make.remote remote/makefile
refreshLink .. super/remote/source
refreshLink source/builds/make.remote_ss super/remote/makefile

# Super
refreshLink . super/source
refreshLink ../builds super/builds
refreshLink ../interbase super/interbase
refreshLink builds/original/fetch_comp super/fetch_comp
refreshLink builds/special_opt super/special_opt

if [ $BuildHostType = "DG" ]; then
    refreshLink source/interbase/examples/v3/gds_blk_data.f testing/gds_blk_data.f
fi

# Utilities
refreshLink .. utilities/source
refreshLink source/jrd/isc.gdb utilities/isc.gdb
refreshLink source/jrd/yachts.gdb utilities/yachts.lnk
refreshLink source/builds/make.utilities utilities/makefile
refreshLink $MasterSourceDir/utilities/rebuild.gdb utilities/rebuild.gdb
refreshLink .. super/utilities/source
refreshLink source/jrd/isc.gdb super/utilities/isc.gdb
refreshLink source/jrd/yachts.gdb super/utilities/yachts.lnk
refreshLink source/builds/make.utilities_ss super/utilities/makefile
refreshLink $MasterSourceDir/utilities/rebuild.gdb super/utilities/rebuild.gdb

# Wal
refreshLink .. wal/source
refreshLink source/builds/make.wal wal/makefile
refreshLink .. super/wal/source
refreshLink source/builds/make.wal_ss super/wal/makefile

# Builds
refreshLink ../.. builds/original/source

# System specific directories
refreshLink .. builds/source
if [ $SYS_TYPE = 'SUNOS4' ]; then
    refreshLink original/install.sun4 builds/install
    refreshLink original/servers.isc builds/servers.isc
else
        refreshLink original/install.unix builds/install
        if [ $BuildHostType = 'SUN3_3' ]; then
            refreshLink original/servers.isc builds/servers.isc
        fi
fi

if [ $BuildHostType = 'MU' -o $BuildHostType = 'UL' ]; then
    refreshLink original/inetd.conf.ultrix builds/inetd.conf.isc
else
        if [ $BuildHostType = 'IMP' -o $BuildHostType = 'SCO' -o $BuildHostType = 'EPSON' ]; then
            refreshLink original/inetd.conf.psx builds/inetd.conf.isc
        else
            refreshLink original/inetd.conf.unx builds/inetd.conf.isc
        fi
fi



echo "- Setting up plaform specific makefiles"
echo ""

cd builds/original


# boot.sfx.interbase has the modes required for a boot build so
# we need to replace the original one there.
# The noboot/boot is applied as a prefix, since when I did it as
# a suffix the builds/original/plaform still works on sfx.* and
# it fails.

if [ $BuildBootFlg = "Yes" ]
  then
    if [ ! -f noboot.sfx.interbase ]
      then
        cp sfx.interbase noboot.sfx.interbase
    fi
    cp boot.sfx.interbase sfx.interbase
fi

# pick correct file based on arch and compiler used
if [ $BuildHostType = "SOLARIS" ]; then 
    if [ "$Arch" = "i386" ]; then   
        cp -p prefix.solaris cvs-$$.prefix.solaris
        cp prefix.solx86 prefix.solaris
    fi
fi
   
chmod +x setup_prots
./setup_prots

# Save the production/development flag

echo "$BuildHostType" > .platform_type    
echo "$BuildBuildType" > .version_flag

./platform $BuildHostType


# Restore the sfx.interbase file to stop problems with CVS 
if [ $BuildBootFlg = "Yes" ]
  then
    if [ -f noboot.sfx.interbase ]
      then
        cp noboot.sfx.interbase sfx.interbase
    fi
fi

if [ $BuildHostType = "SOLARIS" ]; then
    if [ "$Arch" = "i386" ]; then   
        cp -p cvs-$$.prefix.solaris prefix.solaris 
        rm cvs-$$.prefix.solaris
    fi
fi

cd ../..


echo "- Directory setup is now complete!"
echo ""
echo "--------------------------------------------------------------"
echo ""
echo "The next step in building the Firebird database, is to setup"
echo "environment variables required for compilation"
echo "Then compile either the classic firebird or super firebird"
echo "(If unsure try classic - which is target firebird)"
echo ""
echo " \$source Configure_SetupEnv.sh"
echo " \$make (firebird|super_firebird) > make.log 2>&1"
echo ""
