/*
 * Copyright (C) 2000-2002 Chris Ross and various contributors
 * Copyright (C) 1999-2000 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "ferite.h"
#include "aphex.h"

/**
 * !group Script Helpers
 * !description Various functions for creating, loading and deleting script objects
 */

/**
 * !function ferite_new_script
 * !declaration FeriteScript *ferite_new_script()
 * !brief This will allocate a new script and set it up ready for use.
 * !return A pointer to the script
 */
FeriteScript *ferite_new_script()
{
    FeriteScript *ptr = NULL;

    FE_ENTER_FUNCTION;
    ptr = fmalloc( sizeof( FeriteScript ) );
    ptr->filename = NULL;
    ptr->scripttext = NULL;
    ptr->mainns = fmalloc( sizeof( FeriteNamespace ) );
    ptr->mainns->name = NULL;
    ptr->mainns->num = 15;
    ptr->mainns->space = ferite_create_hash( NULL, FE_SCRIPT_TOP_LEVEL_NS_SIZE );

    ptr->include_list = ferite_create_stack( NULL, FE_COMPILER_INTERNAL_STACK_SIZE );

    ptr->current_op_file = NULL;
    ptr->current_op_line = 0;

    ptr->error_state = 0;
    ptr->keep_execution = 0;
    ptr->is_executing = 0;
    ptr->is_being_deleted = FE_FALSE;
    ptr->return_value = 0;
    ptr->error = NULL;
    ptr->warning = NULL;
    ptr->last_regex_count = 0;

    ptr->gc = NULL;

#ifdef THREAD_SAFE
    ptr->thread_group = ferite_create_thread_group();
    ptr->lock = aphex_mutex_create();
    ptr->parent = NULL;
#endif
    
    ferite_init_cache( ptr );
    ptr->odata = NULL;
    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !function ferite_script_load
 * !declaration int ferite_script_load( FeriteScript *script, char *filename )
 * !brief Load the text of a script so that it may be compiled.
 * !param FeriteScript *script The script to load the text into
 * !param char *filename The name of the script to load.
 * !return Returns 0 on fail and 1 on success
 */
int ferite_script_load( FeriteScript *script, char *filename )
{
    AphexFile *scriptfile = NULL;

    FE_ENTER_FUNCTION;
    script->filename = NULL;
    script->scripttext = NULL;
    scriptfile = aphex_open_file( filename, "r", NULL );
    if( scriptfile != NULL )
    {
        script->filename = fstrdup( scriptfile->filename );
        script->scripttext = fmalloc( scriptfile->size + 1 );
        memset( script->scripttext, '\0', scriptfile->size + 1 );
        aphex_read_file( scriptfile, script->scripttext, scriptfile->size );
        aphex_close_file( scriptfile );

        /* remove the #! from the file. */
        if( script->scripttext[0] == '#' )
        {
            int i = 0;
            while( script->scripttext[i] != '\n' )
            {
                script->scripttext[i] = ' ';
                i++;
            }
        }
        FE_LEAVE_FUNCTION( 1 );
    }
    FE_LEAVE_FUNCTION( 0 );
}

int ferite_script_being_deleted( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    FE_LEAVE_FUNCTION( script && script->is_being_deleted );
}

/**
 * !function ferite_script_clean
 * !declaration int ferite_script_clean( FeriteScript *script )
 * !brief Clean a script structure of all non-error/warning reporting structures
 * !param FeriteScript *script A pointer to the script to be cleaned
 * !return Returns 0 on fail and 1 on success
 */
int ferite_script_clean( FeriteScript *script )
{
    int i;

    FE_ENTER_FUNCTION;

    if( script != NULL )
    {
        /*
         * THREAD: Basically here we will go through the list of threads and loop until
         * they have finished, this means that threads can't be deleted until they are
         * actually dead.
         */

#ifdef THREAD_SAFE
        if( script->lock != NULL )
        {
            ferite_thread_group_destroy( script, script->thread_group );
            aphex_mutex_destroy( script->lock );
            script->lock = NULL;
        }
#endif

        script->is_being_deleted = FE_TRUE;

        /* delete the garbage collector */
        if( script->gc != NULL )
        {
            ferite_deinit_gc( script );
        }
  /* delete the include list */
        if( script->include_list != NULL )
        {
            for( i = 0; i <= script->include_list->stack_ptr; i++ )
            {
                if( script->include_list->stack[i] != NULL )
                {
                    ferite_unload_native_module( script->include_list->stack[i], script );
                    ffree( script->include_list->stack[i] );
                }
            }
            ferite_delete_stack( NULL, script->include_list );
            script->include_list = NULL;
        }

        /* delete the body of the script */
        if( script->mainns != NULL )
        {
            ferite_delete_namespace( script, script->mainns );
            script->mainns = NULL;
        }
  /* ... and finally... */
        if( script->filename != NULL )
          ffree( script->filename );
        if( script->scripttext != NULL )
          ffree( script->scripttext );

        script->is_being_deleted = FE_FALSE;

        FE_LEAVE_FUNCTION( 1 );
    }
    FE_LEAVE_FUNCTION(0);
}

/**
 * !function ferite_script_delete
 * !declaration int ferite_script_delete( FeriteScript *script )
 * !brief Delete a script object - it calls ferite_script_clean() first so you dont need to
 * !param FeriteScript *script The script to delete
 * !return 1 on success, 0 on fail
 */
int ferite_script_delete( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    if( script != NULL )
    {
        ferite_script_clean( script );
        ferite_free_cache( script );
        if( script->error != NULL )
          ferite_buffer_delete( script->error );
        if( script->warning != NULL )
          ferite_buffer_delete( script->warning );
        /* aww crap. my wrong. this is the final act! */
        ffree( script );
        FE_LEAVE_FUNCTION( 1 );
    }
    FE_LEAVE_FUNCTION(0);
}

/**
 * !function ferite_duplicate_script
 * !declaration FeriteScript *ferite_duplicate_script( FeriteScript *script )
 * !brief Duplicate a script object
 * !param FeriteScript *script The script to make a copy of
 * !return A brand new script object that can be executed and used as if it had been compiled
 */
FeriteScript *ferite_duplicate_script( FeriteScript *script )
{
    FeriteScript *ptr = NULL;
    FeriteClass *klass = NULL;
    FeriteStack *klass_list = NULL;
    FeriteNamespaceBucket *nsb = NULL;
    char *name = NULL;
    int i = 0;

    FE_ENTER_FUNCTION;
    if( script != NULL )
    {
        ptr = ferite_new_script();

        if( script->mainns != NULL )
        {
            /* we duplicate the include list and also the main namespace */
            ferite_delete_namespace( ptr, ptr->mainns );
            ptr->odata = ferite_create_stack( script, 30 );
            ptr->mainns = ferite_namespace_dup( ptr, script->mainns, NULL );

            /* link up the klass's parents */
            klass_list = ptr->odata;
            for( i = 0; i <= klass_list->stack_ptr; i++ )
            {
                klass = klass_list->stack[i];
                if( klass != NULL && klass->parent != NULL )
                {
                    name = (char *)klass->parent;
                    FUD(( "searching for parent '%s' of '%s' ... ", name, klass->name ));
                    nsb = ferite_find_namespace( ptr, ptr->mainns, name, FENS_CLS );
                    if( nsb != NULL )
                    {
                        klass->parent = nsb->data;
                        FUD(( "found '%s'", klass->parent->name ));
                    }
                    else
                      FUD(( "erk, can't find it" ));
                    FUD(( "\n" ));
                    ffree( name );
                }
            }
            ferite_delete_stack( script, ptr->odata );
        }

        /* copy the include list for the script */
        for( i = 0; i <= script->include_list->stack_ptr; i++ )
        {
            if( script->include_list->stack[i] != NULL )
              ferite_stack_push( ptr->include_list, fstrdup(script->include_list->stack[i]) );
        }

        if( script->gc )
        {
            /* setup the gc */
            ferite_init_gc( ptr );
        }
    }
    FE_LEAVE_FUNCTION( ptr );
}

void ferite_init_cache( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    if( script )
    {
        script->vars = ferite_create_stack( NULL, FE_CACHE_SIZE );
        script->objects = ferite_create_stack( NULL, FE_CACHE_SIZE / 2 );
        script->stacks = ferite_create_stack( NULL, FE_CACHE_SIZE );
    }
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_free_cache( FeriteScript *script )
{
    int i = 0;

    FE_ENTER_FUNCTION;
    FE_ASSERT( script != NULL );

    if( script->vars != NULL )
    {
        /* clean up the variables */
        for( i = 1; i <= script->vars->stack_ptr; i++ )
          ffree( script->vars->stack[i] );
        ferite_delete_stack( NULL, script->vars );
        script->vars = NULL;
    }

    if( script->objects != NULL )
    {
        /* clean up the objects */
        for( i = 1; i <= script->objects->stack_ptr; i++ )
          ffree( script->objects->stack[i] );
        ferite_delete_stack( NULL, script->objects );
        script->objects = NULL;
    }

    if( script->stacks != NULL )
    {
        /* clean up the stacks */
        for( i = 1; i <= script->stacks->stack_ptr; i++ )
          ffree( script->stacks->stack[i] );
        ferite_delete_stack( NULL, script->stacks );
        script->stacks = NULL;
    }

    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !end
 */
