/* -*- mode: c; mode: fold; -*- */
/*
 * Copyright (C) 2000-2002 Chris Ross and various contributors
 * Copyright (C) 1999-2000 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "ferite.h"
#include "aphex.h"

extern FeriteOpTable ferite_op_table[];

/**
 * !group Executor
 * !description These functions allow for the execution of code. The main interface is
 *              ferite_script_execute(), but to execute individual functions there is
 *              ferite_call_function(). There are other convinience functions.
 */

/**
 * !function ferite_stop_execution
 * !declaration void ferite_stop_execution( FeriteScript *script )
 * !brief Stop the script executing
 * !param FeriteScript *script The script to stop
 */
void ferite_stop_execution( FeriteScript *script, int return_value )
{
    FE_ENTER_FUNCTION;
    script->keep_execution = 0;
    script->return_value = return_value;
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_is_executing
 * !declaration int ferite_is_executing( FeriteScript *script )
 * !brief Return whether a script is executing
 * !param FeriteScript *script The script to query
 * !return FE_TRUE if the script is executing, FE_FALSE otherwise
 */
int ferite_is_executing( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    FE_LEAVE_FUNCTION( (script && script->is_executing ? FE_TRUE : FE_FALSE) );
}

/**
 * !function ferite_script_execute
 * !declaration int ferite_script_execute( FeriteScript *script )
 * !brief Run a script
 * !param FeriteScript *script The script to run
 * !return FE_TRUE on successfull execution, FE_FALSE otherwise
 */
int ferite_script_execute( FeriteScript *script )
{
    FeriteNamespaceBucket *nsb = NULL;
    FeriteFunction *func = NULL;
    FeriteVariable *err = NULL, *errstr = NULL, *erno = NULL, *rval = NULL;

    FE_ENTER_FUNCTION;

    if( script->mainns != NULL )
    {
        script->error_state = 0;
        script->is_executing = 1;
        nsb = ferite_namespace_element_exists( script, script->mainns, "!__start__" );

        if( nsb != NULL )
        {
            func = nsb->data;
            rval = ferite_script_function_execute( script, func, NULL );

#ifdef THREAD_SAFE
            ferite_thread_group_wait( script, script->thread_group );
#endif

            /* clean up the system */
            if( rval != NULL )
            {
                if( rval->type == F_VAR_LONG && script->return_value == 0 )
                {
                    script->return_value = VAI(rval);
                }
                ferite_variable_destroy( script, rval );
            }

            nsb = ferite_namespace_element_exists( script, script->mainns, "err" );
            if( script->error_state == ERROR_THROWN ) /* error propogated up the system */
            {
                err = nsb->data;
                errstr = ferite_get_variable_from_hash( script, VAO(err)->variables, "str" );
                erno = ferite_get_variable_from_hash( script, VAO(err)->variables, "num" );
                if( script->error == NULL )
                  script->error = ferite_buffer_new( 0 );
                ferite_buffer_printf( script->error, "\n\n[ferite] Fatal Error: Execution stopped: On line %d, in file '%s':\n%s\n", script->current_op_line, script->current_op_file, VAS(errstr)->data );
                FE_LEAVE_FUNCTION( 0 );
            }
            script->is_executing = 0;
            FE_LEAVE_FUNCTION( FE_TRUE );
        }
    }
    else
      ferite_error( script, 0, "Fatal Error: Unable to execute script - looks like the compile failed.\n" );
    FE_LEAVE_FUNCTION( FE_FALSE );
}

/**
 * !function ferite_script_function_execute
 * !declaration int ferite_script_function_execute( FeriteScript *script,  FeriteFunction *function, FeriteVariable **params )
 * !brief Execute a function within a script
 * !param FeriteScript *script The current script
 * !param FeriteFunction *function The function to execute
 * !param FeriteVariable **params The calling arguments
 * !return The variable returned from the function
 */
FeriteVariable *ferite_script_function_execute( FeriteScript *script, FeriteFunction *function, FeriteVariable **params )
{
    FeriteExecuteRec    exec;
    FeriteStack    exec_stack;
    void *    stack_array[32];
    FeriteVariable      *targetvar = NULL, *rval = NULL;
    int i = 0, stop_execute = 0;
    int arg_count = ferite_get_parameter_count( params );
    int sig_count = function->arg_count;
    int offset = 1;

    FE_ENTER_FUNCTION;

    /*{{{ Checks before we do anything */
    FE_ASSERT( script != NULL && function != NULL );

    if( !script->is_executing )
      stop_execute = 1;

    script->keep_execution = 1;
    /*}}}*/

    /*{{{ Obtain function, and create local variable hash */
    exec.function = function;
    exec.variable_list = (FeriteVariable**)ferite_duplicate_stack_contents( script, function->localvars, (void*(*)(FeriteScript*,void*,void*))ferite_duplicate_variable, NULL );
    exec.stack = &exec_stack;
    exec.stack->stack = stack_array;
    exec.stack->stack_ptr = 0;
    exec.stack->size = 32;
    exec.block_depth = 0;
    /*}}}*/

    /*{{{ Copy parameter values over to the local variable stack */
    if( params != NULL ) /* handle function parameters */
    {
        if( function->klass != NULL )
        {
            sig_count -= 2;
            offset += 2;
        }

        /*
         * Go through at add the all variables to the local variables. We either want to jump out when we hit a '.' or when we hit
         * the super/self variables.
         */
        for( i = 0; i < sig_count && function->signature[i] != NULL && function->signature[i]->variable->name[0] != '.'; i++ )
        {
            /* Assign and use the return to add to the fncArgs array */
            targetvar = ferite_op_assign( script, exec.variable_list[i+offset], params[i] );
            ferite_variable_destroy( script, targetvar );
        }

        /* We are now at the 'dot' if we have one, or at an object - we now pick up the self and super variables */
        if( function->klass != NULL )
        {
            /* sort out super */
            targetvar = exec.variable_list[2];
            if( targetvar != NULL )
              VAO(targetvar) = VAO(params[arg_count-1]);

            /* sort out self */
            targetvar = exec.variable_list[1];
            if( targetvar != NULL )
              VAO(targetvar) = VAO(params[arg_count-1]);

            /* up the reference count */
            VAO(targetvar)->refcount += 2;
        }
    }
    /*}}}*/

    FUD(("EXECUTOR: Executing function %s %p\n", function->name, function->ccode->list[1] ));
    rval = ferite_script_real_function_execute( script, function, script->mainns, &exec, params );

    ferite_clean_up_exec_rec( script, &exec );

    if( stop_execute )
      script->is_executing = 0;

    FE_LEAVE_FUNCTION( rval );
}

/**
 * !function ferite_script_real_function_execute
 * !declaration FeriteVariable *ferite_script_real_function_execute( FeriteScript *script, FeriteFunction *function, FeriteNamespace *ns, FeriteExecuteRec *exec, FeriteVariable **params )
 * !brief Execute a eval operator compiled script
 * !param FeriteScript *script The current script to run
 * !param FeriteFunction *function The function to execute
 * !param FeriteNamespace *ns The scripts main namespace
 * !param FeriteExecuteRec *exec The execute records for the current script
 * !param FeriteVariable **params The parameters passed to the function
 * !return The return value from the function on successfull execution, NULL otherwise
 */
FeriteVariable *ferite_script_real_function_execute( FeriteScript *script, FeriteFunction *function, FeriteNamespace *mainns, FeriteExecuteRec *exec, FeriteVariable **params )
{
    /*{{{ Variables */
    FeriteNamespaceBucket *nsb = NULL;
    FeriteOp              *current_op = NULL;
    FeriteOp             **opcode_list = NULL;
    FeriteVariable        *varone = NULL, *vartwo = NULL, *result = NULL;
    FeriteVariable        *(*unaryop)( FeriteScript *s, FeriteVariable *a );
    FeriteVariable        *(*binaryop)( FeriteScript *s, FeriteVariable *a, FeriteVariable *b );
    FeriteVariable        *(*manyop)(FeriteScript *s, FeriteVariable **vars,int count);
    FeriteVariable        *param_list[FE_FUNCTION_PARAMETER_MAX_SIZE], **pml = NULL, *rval = NULL, *return_val = NULL;
    FeriteString          *str = NULL;
    FeriteFunction        *trgt_function_call = NULL;
    FeriteClass           *sclass;
    int                    current_op_loc = 0, keep_function_running = 1, i = 0, j = 0, error_op_location = 0, error_array[100];
    int                    argcount = 0;
    char                  *var_name = NULL;
    /*}}}*/

    FE_ENTER_FUNCTION;

    opcode_list = function->ccode->list;
    current_op = opcode_list[0];
    current_op_loc++;

    script->current_op_file = function->ccode->filename;

    FUD(("EXECUTION STARTING\n"));
    while( keep_function_running && script->keep_execution )
    {
        varone = NULL;
        vartwo = NULL;
        rval = NULL;
        FUD(("[%p] ", current_op ));

        script->current_op_line = current_op->line;
        exec->block_depth = current_op->block_depth;
        switch( current_op->OP_TYPE )
        {
            /*{{{ F_OP_PUSH      (push something onto the stack)    */
          case F_OP_PUSH:
            FUD(("PUSH\n"));
            ferite_stack_push( exec->stack, current_op->opdata );
            break;
            /*}}}*/
            /*{{{ F_OP_POP       (pop something off the stack)      */
          case F_OP_POP:
            FUD(("POP\n"));
            varone = ferite_stack_pop( exec->stack );
            if( varone && FE_VAR_IS_DISPOSABLE( varone ) )
            {
                FUD(("Deleting Variable: %s\n", varone->name ));
                ferite_variable_destroy( script, varone );
            }
            break;
            /*}}}*/
            /*{{{ F_OP_ARGS      (get the funtions parameters)      */
          case F_OP_ARGS:
            argcount = ferite_get_parameter_count( params );
            varone = ferite_create_uarray_variable( script, "fncArgs", argcount+1, FE_STATIC );

            if( function->klass != NULL )
              argcount -= 2;

            for( i = 0; i < argcount; i++ )
            {
                vartwo = ferite_duplicate_variable( script, params[i], NULL );
                ferite_uarray_add( script, VAUA(varone), vartwo, NULL, FE_ARRAY_ADD_AT_END );
            }
            MARK_VARIABLE_AS_DISPOSABLE( varone );
            ferite_stack_push( exec->stack, varone );
            break;
              /*}}}*/
            /*{{{ F_OP_PUSHVAR   (get variable and put onto stack)  */
          case F_OP_PUSHVAR:
            FUD(("PUSHVAR\n"));
              /* try and get it from the local scoped variable hash */
            varone = NULL;
            nsb = ferite_namespace_element_exists( script, mainns, (char *)(current_op->opdata) );
            if( nsb != NULL )
            {
                switch( nsb->type )
                {
                  case FENS_NS:
                    /* we need to do some trickery here. I just have to work out what that is :)
                     * ok plan of action, create a variable, set it's type to F_VAR_NS, push it onto the stack,
                     * it should *never* get out of the executor. */
                    varone = ferite_variable_alloc(script);
                    varone->type = F_VAR_NS;
                    varone->data.pval = nsb->data;
                    /* Since the comment above says that the variable won't escape the executor,
                     * this should be safe. The opcodes shouldn't disapear under our feet. */
                    varone->name = (char *)(current_op->opdata);
                    MARK_VARIABLENAME_AS_STATIC( varone );
                    MARK_VARIABLE_AS_DISPOSABLE( varone );
                    break;
                  case FENS_VAR:
                    varone = nsb->data;
                    break;
                  case FENS_CLS:
                    /* we need to do some trickery here. I just have to work out what that is :)
                     * ok plan of action, create a variable, set it's type to F_VAR_CLASS, push it onto the stack,
                     * it should *never* get out of the executor. */
                    varone = ferite_variable_alloc(script);
                    varone->type = F_VAR_CLASS;
                    varone->data.pval = nsb->data;
                    /* Since the comment above says that the variable won't escape the executor,
                     *  this should be safe. The opcodes shouldn't disapear under our feet. */
                    varone->name = (char *)(current_op->opdata);
                    MARK_VARIABLENAME_AS_STATIC( varone );
                    FUD(( "pushing class %s\n", (char *)(current_op->opdata) ));
                    MARK_VARIABLE_AS_DISPOSABLE( varone );
                    break;
                  default:
                    ferite_error( script, 0, "PUSHVAR with wrong type\n" );
                }
            }

            if( script->error_state == ERROR_THROWN )
              break;

            if( varone != NULL )
              ferite_stack_push( exec->stack, varone );
            else
              ferite_error( script, 0, "Can't Find Variable '%s'\n",  (char *)(current_op->opdata) );
            break;
            /*}}}*/
            /*{{{ F_OP_PUSHATTR  (get variable from and object,class or namespace and put onto stack)  */
          case F_OP_PUSHATTR:
            var_name = (char *)(current_op->opdata)+1;
            vartwo = ferite_stack_pop( exec->stack );
            switch( vartwo->type )
            {
              case F_VAR_OBJ:
                FUD(( "Executing: Searching for '%s' in '%s'\n", var_name, vartwo->name ));
                if( VAO(vartwo) != NULL )
                {
                    varone = ferite_object_get_var( script, VAO(vartwo), var_name ); 
                    if( varone == NULL )
                    {
			/* we see if the object has a .attribute_missing method - and call it */
			trgt_function_call = ferite_object_get_function( script, VAO(vartwo), "attribute_missing" );
			if( trgt_function_call == NULL )
			{
                            ferite_error( script, 0, "Trying to access non-existant variable '%s' in object '%s' (it could be static)\n", var_name, vartwo->name );
                            break;
			}
			else
			{
                            str = ferite_str_new( var_name, strlen( var_name ), FE_CHARSET_DEFAULT );
                            pml = ferite_create_parameter_list_from_data( script, "soo", str, VAO(vartwo), VAO(vartwo) );
                            varone = ferite_call_function( script, trgt_function_call, pml );
                            ferite_delete_parameter_list( script, pml );
                            ferite_str_destroy( str );
			}
                    }
                    if( FE_VAR_IS_STATIC( varone ) )
                      ferite_error( script, 0, "Trying to access static variable '%s' in object '%s'\n", var_name, vartwo->name );
                }
                else
                  ferite_error( script, 0, "Trying to access variable '%s' in object '%s' which is null\n", var_name, vartwo->name );
                break;
              case F_VAR_NS:
                nsb = ferite_namespace_element_exists( script, VAN(vartwo), var_name );
                if( nsb == NULL )
                {
                    ferite_error( script, 0, "Can't find '%s' in namespace '%s'\n", var_name, vartwo->name );
                    break;
                }
                switch( nsb->type )
                {
                  case FENS_NS:
                                  /* we need to do some trickery here. I just have to work out what that is :)
                                  * ok plan of action, create a variable, set it's type to F_VAR_NS, push it onto the stack,
                                  * it should *never* get out of the executor. */
                    varone = ferite_variable_alloc(script);
                    varone->type = F_VAR_NS;
                    varone->data.pval = nsb->data;
                    varone->name = var_name;
                    MARK_VARIABLENAME_AS_STATIC( varone );
                    MARK_VARIABLE_AS_DISPOSABLE( varone );
                    break;
                  case FENS_VAR:
                    varone = nsb->data;
                    break;
                  case FENS_CLS:
                                  /* we need to do some trickery here. I just have to work out what that is :)
                                  * ok plan of action, create a variable, set it's type to F_VAR_CLASS, push it onto the stack,
                                  * it should *never* get out of the executor. */
                    varone = ferite_variable_alloc(script);
                    varone->type = F_VAR_CLASS;
                    varone->data.pval = nsb->data;
                    varone->name = var_name;
                    MARK_VARIABLENAME_AS_STATIC( varone );
                    FUD(( "EXEC: pushing class %s\n", (char *)(current_op->opdata) ));
                    MARK_VARIABLE_AS_DISPOSABLE( varone );
                    break;
                  default:
                    ferite_error( script, 0, "Expecting variable, class or namespace for '%s' in '%s'\n", var_name, vartwo->name );
                    break;
                }
                break;
              case F_VAR_CLASS:
                varone = ferite_class_get_var( script, VAC(vartwo), var_name );
                if( varone == NULL )
                  ferite_error( script, 0, "Unable to find variable %s in class %s\n", var_name, vartwo->name );
                else
                {
                    if( !FE_VAR_IS_STATIC( varone ) )
                      ferite_error( script, 0, "Trying to access non-static variable %s in class %s\n", var_name, vartwo->name );
                }
                break;
              default:
                ferite_error( script, 0, "Can not get variable '%s' from '%s', expecting object/namespace/class but found a %s.\n",
                              (char *)(current_op->opdata),
                              vartwo->name,
                              ferite_variable_id_to_str( script, vartwo->type ) );
                break;
            }
            if( vartwo && FE_VAR_IS_DISPOSABLE( vartwo ) )
              ferite_variable_destroy( script, vartwo );
            if( script->error_state == ERROR_THROWN )
              break;

            if( varone != NULL )
              ferite_stack_push( exec->stack, varone );
            else
            {
                ferite_error( script, 0, "Can't Find Variable '%s'\n",  (char *)(current_op->opdata) );
                break;
            }
            break;
              /*}}}*/
            /*{{{ F_OP_PUSHINDEX (get a variable from the local vars and push it onto the stack) */
          case F_OP_PUSHINDEX:
            varone = exec->variable_list[current_op->addr];
            ferite_stack_push( exec->stack, varone );
            break;
            /*}}}*/
            /*{{{ F_OP_UNARY     (do a unary operation)             */
          case F_OP_UNARY:
            FUD(("UNARY\n"));
            varone = ferite_stack_pop( exec->stack );
            unaryop = (FeriteVariable *(*)( FeriteScript *, FeriteVariable * ))ferite_op_table[current_op->addr].ptr;
            FUD(("Unary Op Parameters: %s\n", varone->name ));
            if( (result = unaryop( script, varone )) )
              ferite_stack_push( exec->stack, result );
            if( FE_VAR_IS_DISPOSABLE( varone ) )
            {
                FUD(("Deleting Variable: %s\n", varone->name ));
                ferite_variable_destroy( script, varone );
            }
            break;
              /*}}}*/
            /*{{{ F_OP_MANY      (do something requiring many args) */
          case F_OP_MANY:
            {
                FeriteVariable **many;
                int n,*count;
                FUD(("MANY\n"));
                count = (int*)(current_op->opdata);
                many = fmalloc(sizeof(FeriteVariable*) * *count);
                for( n = 0; n < *count; n++ )
                {
                    many[n] = ferite_stack_pop( exec->stack );
                    LOCK_VARIABLE( many[n] );
                }
                manyop = (FeriteVariable *(*)(FeriteScript *, FeriteVariable **,int))ferite_op_table[current_op->addr].ptr;
                if( (result = manyop(script, many, *count)) )
                  ferite_stack_push( exec->stack, result );
                for( n = 0; n < *count; n++ )
                {
                    UNLOCK_VARIABLE( many[n] );
                    if( FE_VAR_IS_DISPOSABLE( many[n] ) )
                    {
                        FUD(("Deleting Variable: %s\n",many[n]->name));
                        ferite_variable_destroy(script, many[n]);
                    }
                }
                ffree(many);
                break;
            }
            /*}}}*/
            /*{{{ F_OP_CASE      (do a case statement)              */
          case F_OP_CASE:
            /*
             * This will simply take the first operand, duplicate it, and then push it back onto
             * the stack. It then lets the switch statement run into the next block.
             */
            vartwo = ferite_stack_pop( exec->stack );
            varone = ferite_stack_pop( exec->stack );
            result = ferite_duplicate_variable( script, varone, NULL );
            MARK_VARIABLE_AS_DISPOSABLE( result );
            ferite_stack_push( exec->stack, result );
            ferite_stack_push( exec->stack, varone );
            ferite_stack_push( exec->stack, vartwo );
            /* NOTE!!!! THERE IS NO BREAK! THIS IS DELIBERATE!!!!! */
            /*}}}*/
            /*{{{ F_OP_BINARY    (do a binary operation)            */
          case F_OP_BINARY:
            FUD(("BINARY\n"));
            vartwo = ferite_stack_pop( exec->stack );
            varone = ferite_stack_pop( exec->stack );
            binaryop = (FeriteVariable *(*)( FeriteScript *, FeriteVariable *, FeriteVariable * ))ferite_op_table[current_op->addr].ptr;
            FUD(("Binary Op Parameters: %s, %s\n", varone->name, vartwo->name ));
            if( (result =  binaryop( script, varone, vartwo )) )
              ferite_stack_push( exec->stack, result );
            if( FE_VAR_IS_DISPOSABLE( vartwo ) )
            {
                FUD(("Deleting Variable: %s\n", vartwo->name ));
                ferite_variable_destroy( script, vartwo );
            }
            if( FE_VAR_IS_DISPOSABLE( varone ) )
            {
                FUD(("Deleting Variable: %s\n", varone->name ));
                ferite_variable_destroy( script, varone );
            }
            break;
            /*}}}*/
            /*{{{ Function stuff (function, method and new)         */
          case F_OP_METHOD:
          case F_OP_FUNCTION:
          case F_OP_NEWOBJ:
            /*{{{ Create Parameter List */
            FUD(("CREATING PARAMETER LIST.\n"));

            exec->stack->stack_ptr -= current_op->opdataf->argument_count;
            for( i = exec->stack->stack_ptr + 1, j = 0; j < current_op->opdataf->argument_count; i++, j++ )
            {
                param_list[j] = exec->stack->stack[i];
                LOCK_VARIABLE( param_list[j] );
            }
            param_list[j] = NULL;
            FUD(("PARAMETER LIST CREATED.\n"));
            /*}}}*/
            switch( current_op->OP_TYPE )
            {
                /*{{{ F_OP_METHOD (object and namespace) */
              case F_OP_METHOD:
                vartwo = ferite_stack_pop( exec->stack );
                if( current_op->opdataf->function == NULL )
                {
                    switch( vartwo->type )
                    {
                      case F_VAR_OBJ:
                        FUD(( "Trying to find '%s' in '%s'\n", (char *)(current_op->opdata), vartwo->name ));
                        /* what we do here is check to see if they are searching the super class, if so, we swap the objects           */
                        /* template with that of it's parents - if it has no parent we don't bother..., we then look for the function  */
                        /* that we are after..... i personally would consider this to be, as we english say, "dash cunning"            */
                        if( VAO(vartwo) == NULL )
                        {
                            ferite_error( script, 0, "Trying to access method '%s' in a null object '%s'\n", (char *)(current_op->opdata), vartwo->name );
                            break;
                        }
                        sclass = VAO(vartwo)->klass;
                        if( strncmp( vartwo->name, "super", 5 ) == 0 )
                        {
                            if( VAO(vartwo)->klass->parent != NULL )
                            {
                                VAO(vartwo)->klass = VAO(vartwo)->klass->parent;
                            }
                        }
                        trgt_function_call = ferite_object_get_function( script, VAO(vartwo), (char *)(current_op->opdata)+1 );
                        VAO(vartwo)->klass = sclass;
                        if( trgt_function_call == NULL )
                        {
                            /* ok this is where we try and be dash cunning, fail miserably and settle for an autoload function. */
                            trgt_function_call = ferite_object_get_function( script, VAO(vartwo), "method_missing" );
                            if( trgt_function_call == NULL )
                            {
                                ferite_error( script, 0, "Unable to find method '%s' in object '%s'\n", (char *)(current_op->opdata)+1, vartwo->name );
                                break;
                            }
                            else
                            {
                                /* we have an autoload function, soooooooo, what do we do? */
                                /* There are two ways in which we can handle the autoload call, we either shift the parameter stack
                                 * and place the name of the function first, this would make life easier, but then again when has
                                 * life ever been easy? */
                                varone = ferite_create_string_variable_from_ptr( script, "function-name", (char *)(current_op->opdata)+1, 0, FE_CHARSET_DEFAULT, FE_STATIC );
                                ferite_add_to_parameter_list( param_list, varone );
                                MARK_VARIABLE_AS_DISPOSABLE( varone );
                            }
                        }
                        if( trgt_function_call->is_static )
                        {
                            ferite_error( script, 0, "Trying to access static member method %s through object %s\n", (char *)(current_op->opdata)+1, vartwo->name );
                            break;
                        }
                        /* den swap it back :)                                                                                         */
                        /*                                                                                                             */
                        /* NB: we dont need to do this for varaibles because each class automatically get a duplicae of it's parents   */
                        /*     varaibles hash -> so all objects within the system upn definition.                                      */
                        break;
                      case F_VAR_NS:
                        nsb = ferite_namespace_element_exists( script, VAN(vartwo), (char *)(current_op->opdata)+1 );
                        if( nsb == NULL || nsb->type != FENS_FNC )
                        {
                            ferite_error( script, 0, "Unable to find method '%s' in namespace '%s'\n", (char *)(current_op->opdata)+1, vartwo->name );
                            break;
                        }
                        trgt_function_call = nsb->data;
                        break;
                      case F_VAR_CLASS:
                        trgt_function_call = ferite_class_get_function( script, VAC(vartwo), (char *)(current_op->opdata)+1 );
                        if( trgt_function_call == NULL )
                        {
                            ferite_error( script, 0, "Unable to access method %s within class %s\n", (char *)(current_op->opdata)+1, vartwo->name );
                            break;
                        }
                        else
                        {
                            if( !(trgt_function_call->is_static) )
                              ferite_error( script, 0, "Trying to access non-static method %s within class %s\n", (char *)(current_op->opdata)+1, vartwo->name );
                        }
                        break;
                      default:
                        ferite_error( script, 0, "Expecting container found %s when trying to call %s.%s()\n", ferite_variable_id_to_str( script, vartwo->type ), vartwo->name, (char *)(current_op->opdata)+1 );
                        if( vartwo && FE_VAR_IS_DISPOSABLE( vartwo ) ) /* the var was created */
                          ferite_variable_destroy( script, vartwo );
                        break;
                    }
                    /* make a quick gettaway */
                    if( script->error_state == ERROR_THROWN )
                    {
                        if( vartwo && FE_VAR_IS_DISPOSABLE( vartwo ) ) /* the var was created */
                          ferite_variable_destroy( script, vartwo );
                        break;
                    }

                    if( trgt_function_call != NULL )
                    {
                        if( vartwo->type == F_VAR_OBJ )
                        {
                            FUD(( "Adding super and self to the objects method call\n" ));
                            ferite_object_add_self_variable_to_params( script, param_list, VAO(vartwo) );
                        }

                        for( ; trgt_function_call != NULL; trgt_function_call = trgt_function_call->next )
                        {
                            if( ferite_check_params( script, param_list, trgt_function_call->signature ) != 0 )
                              break;
                        }

                        if( trgt_function_call == NULL )
                        {
                            ferite_error( script, 0, "Unable to find method %s%s that accepts the parameters passed\n", ( vartwo->type == F_VAR_OBJ ? VAO(vartwo)->klass->name : vartwo->name ), (char *)(current_op->opdata) );
                            current_op->opdataf->function = NULL;
                        }
                        /*
                         * Disable cache for now, we have side-effects.
                         * Cached methods isn't that great since it will fuck up inheritance.
                         * Sys.Stream that uses inheritance will call wrong methods at the wrong time:
                         * If you have used a ProcessStream and opens up a FileStream, The __close__
                         * method will be that of ProcessStream, resulting in less than expected behaivour
                         *
                         else
                         current_op->opdataf->function = trgt_function_call;
                         */
                    }
                    else
                    {
                        ferite_error( script, 0,"Error: Can't find function %s() in object %s %s\n",
                                      (char *)(current_op->opdata),
                                      VAO(vartwo)->name,
                                      (VAO(vartwo)->name == NULL ? "\n       (cause: object has not been created?)" : ""));
                    }
                }
                else
                {
                    if( vartwo->type == F_VAR_OBJ )
                    {
                        FUD(( "Adding super and self to the objects method call\n" ));
                        ferite_object_add_self_variable_to_params( script, param_list, VAO(vartwo) );
                    }

                    trgt_function_call = current_op->opdataf->function;
                }

                if( trgt_function_call != NULL )
                {
                    LOCK_VARIABLE(trgt_function_call); /* lock the method */
                    if( trgt_function_call->type == FNC_IS_EXTRL )
                    {
                        if( trgt_function_call->fncPtr != NULL )
                        {
                            if( trgt_function_call->native_information != NULL )
                            {
                                script->current_op_file = trgt_function_call->native_information->file;
                                script->current_op_line = trgt_function_call->native_information->line;
                            }
                            rval = (trgt_function_call->fncPtr)( script, trgt_function_call, param_list );
                        }
                        else
                          ferite_error( script, 0, "Unable to execute external method %s.%s (does it exist?)\n", ( vartwo->type == F_VAR_OBJ ? VAO(vartwo)->klass->name : vartwo->name ), trgt_function_call->name );
                    }
                    else /* internal function call */
                    {
                        rval = ferite_script_function_execute( script, trgt_function_call, param_list );
                    }
                    UNLOCK_VARIABLE(trgt_function_call); /* unlock the method */
                    if( script->error_state != ERROR_THROWN )
                    {
                        script->current_op_file = function->ccode->filename; /* we do this because the other function might cause it to change */
                        script->current_op_line = current_op->line;
                    }
                }

                if( vartwo && FE_VAR_IS_DISPOSABLE( vartwo ) ) /* the var was created */
                  ferite_variable_destroy( script, vartwo );
                break;
                /*}}}*/
                /*{{{ F_OP_FUNCTION (script) */
              case F_OP_FUNCTION:
                trgt_function_call = NULL;
                if( ((char*)current_op->opdata)[0] == 's' && ((char*)current_op->opdata)[1] == 'u' && memcmp( current_op->opdata, "super\0", 6 ) == 0 && function->klass != NULL )
                {
                    /* handle super() */
                    varone = exec->variable_list[1];
                    if( varone != NULL && varone->type == F_VAR_OBJ && VAO(varone) != NULL )
                    {
                        ferite_object_add_self_variable_to_params( script, param_list, VAO(varone) );
                        rval = ferite_object_call_super( script, VAO(varone), param_list );
                    }
                    else
                    {
                        ferite_error( script, 0, "Trying to call super class's constructor in non-instance object function" );
                    }
                    break;
                }
                else
                {
                    if( current_op->opdataf->function == NULL )
                    {
                        /* handle other functions */
                        nsb = ferite_find_namespace( script, mainns, (char *)(current_op->opdata), FENS_FNC );
                        if( nsb != NULL )
                          trgt_function_call = nsb->data;
                        if( trgt_function_call == NULL ) /* we are in an eval check parent script */
                        {
                            nsb = ferite_find_namespace( script, script->mainns, (char *)(current_op->opdata), FENS_FNC );
                            if( nsb != NULL )
                            {
                                if( (trgt_function_call = nsb->data) == NULL )
                                {
                                    ferite_raise_script_error( script, 0, "Can't find function '%s'.", (char *)(current_op->opdata));
                                    break;
                                }
                            }
                            else
                            {
                                ferite_error( script, 0, "Function '%s' doesn't exist", (char *)(current_op->opdata) );
                                break;
                            }
                        }

                        for( ; trgt_function_call != NULL; trgt_function_call = trgt_function_call->next )
                        {
                            if( ferite_check_params( script, param_list, trgt_function_call->signature ) )
                              break;
                        }

                        if( trgt_function_call == NULL )
                        {
                            ferite_error( script, 0, "Unable to find function %s that accepts the parameters passed\n", (char *)(current_op->opdata) );
                            current_op->opdataf->function = NULL;
                        }
                        /*
                         * Disable cache for now, we have side-effects.
                         * Cached methods isn't that great since it will fuck up inheritance.
                         * Sys.Stream that uses inheritance will call wrong methods at the wrong time:
                         * If you have used a ProcessStream and opens up a FileStream, The __close__
                         * method will be that of ProcessStream, resulting in less than expected behaivour
                         *
                         else
                         current_op->opdataf->function = trgt_function_call;
                         */
                    }
                    else
                      trgt_function_call = current_op->opdataf->function;

                    if( trgt_function_call != NULL )
                    {
                        LOCK_VARIABLE(trgt_function_call); /* lock the method */
                        if( trgt_function_call->type == FNC_IS_EXTRL )
                        {
                            if( trgt_function_call->fncPtr != NULL )
                            {
                                if( trgt_function_call->native_information != NULL )
                                {
                                    script->current_op_file = trgt_function_call->native_information->file;
                                    script->current_op_line = trgt_function_call->native_information->line;
                                }
                                rval = (trgt_function_call->fncPtr)( script, trgt_function_call, param_list );
                            }
                            else
                              ferite_error( script, 0, "Unable to execute external method %s (does it exist?)\n", trgt_function_call->name );
                        }
                        else /* internal function call */
                        {
                            rval = ferite_script_function_execute( script, trgt_function_call, param_list );
                        }
                        UNLOCK_VARIABLE(trgt_function_call); /* lock the method */
                        if( script->error_state != ERROR_THROWN )
                        {
                            script->current_op_file = function->ccode->filename; /* we do this because the other function might cause it to change */
                            script->current_op_line = current_op->line;
                        }
                    }
                }
                break;
                /*}}}*/
                /*{{{ F_OP_NEWOBJ */
              case F_OP_NEWOBJ:
                FUD(("Creating new object\n"));
                vartwo = ferite_stack_pop( exec->stack );
                if( vartwo->type != F_VAR_CLASS )
                {
                    ferite_error( script, 0, "%s is not a class, bad luck, try again :)\n", vartwo->name );
                    break;
                }
                rval = (void *)ferite_new_object( script, (FeriteClass *)(vartwo->data.pval), param_list );
                if( script->error_state != ERROR_THROWN )
                {
                    script->current_op_file = function->ccode->filename; /* we do this because the other function might cause it to change */
                    script->current_op_line = current_op->line;
                }
                if( vartwo && FE_VAR_IS_DISPOSABLE( vartwo ) ) /* the var was created */
                  ferite_variable_destroy( script, vartwo );
                break;
                /*}}}*/
            }
            for( i = 0; i < FE_FUNCTION_PARAMETER_MAX_SIZE; i++ )
            {
                if( param_list[i] != NULL )
                {
                    UNLOCK_VARIABLE(param_list[i]);
                    if( FE_VAR_IS_DISPOSABLE( param_list[i] ) )
                      ferite_variable_destroy( script, PTR2VAR(param_list[i]) );
                    param_list[i] = NULL;
                }
                else
                  break;
            }
            if( rval != NULL )
              ferite_stack_push( exec->stack, rval );
            break;
            /*}}}*/
            /*{{{ F_OP_BNE       (branch if not equal)              */
          case F_OP_BNE:
            FUD(("BNE\n"));
            varone = ferite_stack_pop( exec->stack );
            if( ferite_variable_is_false( script, varone) )
            {
                FUD(("BNE: Branching\n" ));
                current_op_loc = current_op->addr;
            }
            else
            {
                FUD(("BNE: Not Branching\n" ));
            }
            if( FE_VAR_IS_DISPOSABLE( varone ) ) /* the var was created */
              ferite_variable_destroy( script, varone );
            break;
            /*}}}*/
            /*{{{ F_OP_BIE       (branch if equal)                  */
          case F_OP_BIE:
            FUD(("BIE\n"));
            varone = ferite_stack_pop( exec->stack );
            if( !ferite_variable_is_false( script, varone ) )
            {
                FUD(("BIE: Branching\n" ));
                current_op_loc = current_op->addr;
            }
            else
            {
                FUD(("BIE: Not Branching\n" ));
            }
            if( varone != NULL && FE_VAR_IS_DISPOSABLE( varone ) ) /* the var was created */
              ferite_variable_destroy( script, varone );
            break;
            /*}}}*/
            /*{{{ F_OP_JMP       (jump to a address)                */
          case F_OP_JMP:
            FUD(("JMP\n"));
            current_op_loc = current_op->addr;
            break;
            /*}}}*/
            /*{{{ F_OP_NOP       (do nothing)                       */
          case F_OP_NOP:
            FUD(("NOP. Nothing Done :)\n"));
            break;
            /*}}}*/
            /*{{{ F_OP_RGX       (regular expression instruction)   */
          case F_OP_RGX:
            varone = ferite_stack_pop( exec->stack );
            FUD(( "REGEX: Applying Regex '%s' to %s\n", PARGX(current_op->opdata)->compile_buf, varone->name ));
            vartwo = ferite_execute_regex( PARGX(current_op->opdata), varone, script, exec );
            MARK_VARIABLE_AS_DISPOSABLE( vartwo );
            ferite_stack_push( exec->stack, vartwo );
            if( varone && FE_VAR_IS_DISPOSABLE( varone ) )
            {
                FUD(("Deleting Variable: %s\n", varone->name ));
                ferite_variable_destroy( script, varone );
            }
            break;
            /*}}}*/
            /*{{{ F_OP_EXIT      (exit and return from a function)  */
          case F_OP_EXIT:
            FUD(("Exiting\n"));
            keep_function_running = 0; /* quit the function */
            varone = ferite_stack_pop( exec->stack );
            return_val = ferite_duplicate_variable( script, varone, NULL );
            MARK_VARIABLE_AS_DISPOSABLE( return_val );
            if( varone && FE_VAR_IS_DISPOSABLE( varone ) )
            {
                FUD(("Deleting Variable: %s\n", varone->name ));
                ferite_variable_destroy( script, varone );
            }
            break;
            /*}}}*/
            /*{{{ F_OP_ERR       (set an error handler)             */
          case F_OP_ERR:
            if( current_op->addr == -1 )
            {  /* reset the error counter */
                script->error_state = NO_ERROR;
                error_array[--error_op_location] = 0;
            }
            else
            {
                FUD(("ERROR HANDLER: Setting Error location to %ld",  current_op->addr ));
                error_array[error_op_location++] = current_op->addr;
            }
            break;
            /*}}}*/
          default:
            ferite_error( script, 0, "Unknown op type [%d]\n", current_op->OP_TYPE );
        }

        /*{{{ error checking */
        FUD(( "ERROR STATE: %d\n", script->error_state ));
        switch( script->error_state )
        {
          case ERROR_THROWN:
            FUD(( "ERROR STATE: reported...\n" ));
            if( error_op_location < 1 || error_array[error_op_location-1] == 0 )
            { /* there is no error handler propogate upwards */
                FUD(( "ERROR STATE: No error handler found... bombing out.\n" ));
                FUD(( "EXEC: detected error - stoping execution\n" ));
                keep_function_running = 0;
            }
            else
            {
                FUD(( "ERROR STATE: Going to error handler code\n" ));
                current_op_loc = error_array[error_op_location-1];
                current_op = opcode_list[current_op_loc];
                current_op_loc++;
		
		/* We also need to reset the error log otherwise we will get errors we have caught */
		ferite_reset_errors( script );
            }
            break;
          default:
            current_op = opcode_list[current_op_loc];
            current_op_loc++;
        }
      /*}}}*/

#define FE_NEXT_OP { current_op = opcode_list[current_op_loc]; current_op_loc++; continue; }
	
        if( !keep_function_running || !script->keep_execution)
          break;

        /*{{{ GARBAGE COLLECTOR */
#ifndef FE_USE_GENERATIONAL_GC
        {
            FeriteStdGC *gc = script->gc;
            gc->count++;
            if( gc->count > FE_GC_RUN_AFTER_OPS )
            {
                ferite_check_std_gc( script );
                gc->count = 0;
            }
        }
#endif

      /*}}}*/
    }

    FUD(("EXECUTION COMPLETE. Have a nice day :). (%s)\n", function->name));
    FE_LEAVE_FUNCTION( return_val );
}

/**
 * !function ferite_clean_up_exec_rec
 * !declaration void ferite_clean_up_exec_rec( FeriteScript *script, FeriteExecuteRec *exec )
 * !brief Clean up and Execution Record
 * !param FeriteScript *script The current script
 * !param FeriteExecuteRex *exec   Pointer to the execution record
 */
void ferite_clean_up_exec_rec( FeriteScript *script, FeriteExecuteRec *exec )
{
    int counter, i;
    FeriteVariable *targetvar;

    FE_ENTER_FUNCTION;
    /*{{{ Clean up local scope variables */
    FUD(("DELETING LOCAL VARIABLES\n" ));
    for( i = 1; i <= exec->function->localvars->stack_ptr; i++ )
    {
        if( exec->variable_list[i] != NULL )
          ferite_variable_destroy( script, exec->variable_list[i] );
    }
    ffree( exec->variable_list );
    /*}}}*/

    /*{{{ Clean up execution stack */
    counter = 0;
    for( i = 1; i <= exec->stack->stack_ptr; i++ )
    {
        targetvar = exec->stack->stack[i];
        if( targetvar && FE_VAR_IS_DISPOSABLE( targetvar ) )
        {
            FUD(("[%d/%d] DESTROYING STACK VARIABLE %s (%s)\n", i,
                 exec->stack->stack_ptr,
                 targetvar->name,
                 ferite_variable_id_to_str( script, targetvar->type) ));
            ferite_variable_destroy( script, targetvar );
            counter++;
        }
    }
    FUD(( "%d variables lefton stack\n", counter ));
    FUD(("IN TOTAL %d VARIABLES WHERE NOT DEALLOCATED\n", counter));
   /*}}}*/
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_create_parameter_list
 * !declaration FeriteVariable **ferite_create_parameter_list( int size )
 * !brief Create a parameter list, NULLify it and then return it
 * !param int size The number of parameters to hold
 * !return The created list
 */
FeriteVariable **ferite_create_parameter_list( int size )
{
    FeriteVariable **list = NULL;

    FE_ENTER_FUNCTION;
    list = fcalloc( sizeof( FeriteVariable* ) * (size+1), sizeof(char) );
    FE_LEAVE_FUNCTION( list );
}

/**
 * !function ferite_add_to_parameter_list
 * !declaration FeriteVariable **ferite_add_to_parameter_list( FeriteVariable **list, FeriteVariable *var )
 * !brief Place a parameter within the next availible place within the parameter list
 * !param FeriteVariable **list The list to place it in
 * !param FeriteVariable *var  The variable to place within the list
 * !return The list passed to the function
 */
FeriteVariable **ferite_add_to_parameter_list( FeriteVariable **list, FeriteVariable *var )
{
    int size = ferite_get_parameter_count( list );

    FE_ENTER_FUNCTION;
    list[size] = var;
    list[size+1] = NULL;
    FE_LEAVE_FUNCTION( list );
}

/**
 * !function ferite_delete_parameter_list
 * !declaration void ferite_delete_parameter_list( FeriteScript *script, FeriteVariable **list )
 * !brief Delete a parameter list, and destroy any disposable variables
 * !param FeriteScript *script The current script
 * !param FeriteVariable **list   The list to be deleted
 */
void ferite_delete_parameter_list( FeriteScript *script, FeriteVariable **list )
{
    int i = 0;
    int size = ferite_get_parameter_count( list );

    FE_ENTER_FUNCTION;
    while( list[i] != NULL && i < size )
    {
        if( list[i] )
        {
            UNLOCK_VARIABLE(list[i]);
            if( FE_VAR_IS_DISPOSABLE( list[i] ) )
              ferite_variable_destroy( script, PTR2VAR(list[i]) );
        }
        i++;
    }
    ffree( list );
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_create_parameter_list_from_data
 * !declaration FeriteVariable **ferite_create_parameter_list_from_data( FeriteScript *script, char *format, ... )
 * !brief This function is used to make creating a parameter list very easy.
 * !param script The current script
 * !param format The signiture for the parameters
 * !description
 * This function makes the whole process of creating a parameter list very easy. It allows
 * you to create a list from the actual data and not have to worry about the finer details of
 * ferite variables. It returns a parameter list.<nl/>
 * <nl/>
 * e.g.:<nl/>
 * ferite_create_parameter_list_from_data( script, "nns", 2.3, 2, "Jello" );<nl/>
 * <nl/>
 * The formats are as follows:<nl/>
 *   n = number<nl/>
 *   s = string<nl/>
 *   o = object<nl/>
 *   a = array
 * !return The list that is created.
 */
FeriteVariable **ferite_create_parameter_list_from_data( FeriteScript *script, char *format, ... )
{
    FeriteVariable **retval = NULL;
    FeriteVariable *var = NULL;
    va_list          ap;
    int              i = 0;

    retval = fmalloc( sizeof( FeriteVariable * ) * (strlen(format) + 3) );
    memset( retval, '\0', sizeof( FeriteVariable * ) * (strlen(format) + 3) );

    va_start( ap, format );
    for( i = 0; i < (signed)strlen(format); i++ )
    {
        switch( format[i] )
        {
          case 'n':
            var = ferite_create_number_double_variable( script, "list_from_data-number", va_arg( ap, double ), FE_STATIC );
            break;
          case 'l':
            var = ferite_create_number_long_variable( script, "list_from_data-number", va_arg( ap, long ), FE_STATIC );
            break;
          case 's':
            var = ferite_create_string_variable( script, "list_from_data-string", va_arg( ap, FeriteString * ), FE_STATIC );
            break;
          case 'o':
            var = ferite_create_object_variable( script, "list_from_data-object", FE_STATIC );
            VAO(var) = va_arg( ap, FeriteObject * );
            VAO(var)->refcount++;
            break;
          case 'a':
            var = ferite_create_uarray_variable( script, "list_from_data-array", 0, FE_STATIC );
            ferite_uarray_destroy( script, VAUA(var) );
            VAUA(var) = ferite_uarray_dup( script, va_arg( ap, FeriteUnifiedArray *), (void *(*)(FeriteScript*,FeriteVariable*,void*))ferite_duplicate_variable );
            break;
        }
        MARK_VARIABLE_AS_DISPOSABLE( var );
        retval[i] = var;
    }
    va_end( ap );
    return retval;
}

/**
 * !function ferite_call_function
 * !declaration FeriteVariable *ferite_call_function( FeriteScript *script, FeriteFunction *function, FeriteVariable **params )
 * !brief This will call any function from it's function pointer and a parameter list
 * !param FeriteScript *script The current script
 * !param FeriteFunction *function The function to be called
 * !param FeriteVariable **params The parameter list to be passed to the function
 * !description This function will work on either an internal or native function and will happily choose the correct function
 *              if it happens to be overloaded.
 */
FeriteVariable *ferite_call_function( FeriteScript *script, FeriteFunction *function, FeriteVariable **params )
{
    FeriteVariable *retval = NULL;

    FE_ENTER_FUNCTION;
    for( ; function != NULL; function = function->next )
    {
        if( ferite_check_params( script, params, function->signature ) == 1 )
        {
            LOCK_VARIABLE( function );
            if( function->type == FNC_IS_EXTRL )
            {
                retval = (function->fncPtr)( script, function, params );
            }
            else
            {
                retval = ferite_script_function_execute( script, function, params );
                if( script->error_state == ERROR_THROWN )
                  retval = ferite_create_void_variable( script, "error...", FE_STATIC );
            }
            UNLOCK_VARIABLE( function );
            break;
        }
    }
    if( function == NULL )
      ferite_error( script, 0, "Incorrect parameters for function %s.\n", function->name );
    FE_LEAVE_FUNCTION( retval );
}

/**
 * !end
 */
