/*
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * (very basic) SMI handler for FAUmachine BIOS.
 */

#include "build_config.h"

#include "compiler.h"

#include "assert.h"
#include "debug.h"
#include "stdio.h"
#include "acpi-tables.h"

/* ==================== REAL-MODE INIT ==================== */
#ifdef INIT_RM

CODE16;

#include "string.h"
#include "var.h"
#include "bios.lds.h"
#include "pci.h" /* for configspace helper functions */

/* pci common defines */
#define PCICONF_ADDR	0x0cf8
#define PCICONF_DATA	0x0cfc

/* 440BX specific defines */
#define	SMRAM_REG	0x72
#define D_OPEN		(1 << 6)
#define D_CLS		(1 << 5)
#define D_LCK		(1 << 4)
#define G_SMRAME	(1 << 3)
#define C_BASE_SEG_MASK	0x07
#define SMRAM_SIZE	0x20000

/* PIIX specific defines */
#define PIIX_DEV	7
#define PIIX_PM_FN	3

#define PM_BASE		0x4000
#define DEVACTB		0x58
#define APMC_EN		0x2000000
#define APMC		0xb2
#define APMS		0xb3

#define PMBA		0x40
#define PMBA_BA_MASK	0x0000ffc0

#define PMREGMISC	0x80
#define PMIOSE		(1 << 0)

#define PMEN		0x4002
#define PMEN_RES_MASK	0xfade /* reserved bits in PMEN */
#define PWRBTN_EN	(1 << 8)

#define GLBCTL		0x4028
#define EOS		0x10000
#define SMI_EN		(1 << 0)
#define GLBCTL_RES_MASK	0xf8fe00f8 /* reserved bits in GLBCTL */

/*
 * Initialize the chipsets power management functions.
 * Check that we have the right North- and Southbridge, then enable
 * Power Management IO space registers
 */
void
smi_init_chipset(void)
{
	uint32_t ret;

	/* set Power Management Base adress to 0x4000 */
	ret = pci_creadl(0, PIIX_DEV, PIIX_PM_FN, PMBA);
	ret &= ~ PMBA_BA_MASK;
	ret |= PM_BASE;
	outl(ret, PCICONF_DATA);

	/* enable Power Management I/O */
	ret = pci_creadb(0, PIIX_DEV, PIIX_PM_FN, PMREGMISC);
	ret |= PMIOSE;
	pci_cwriteb(0, PIIX_DEV, PIIX_PM_FN, PMREGMISC, ret);
}

/*
 * trigger an 'relocate SMBASE' SMI.
 */
void
smi_redirect(uint32_t base)
{
	uint32_t ret;

	/* enable APMC decoding/SMI generation in the PIIX configspace */
	ret = pci_creadl(0, PIIX_DEV, PIIX_PM_FN, DEVACTB);
	if (ret & APMC_EN) {
		dprintf("BIOS: trigger_smi: APMC_EN was already set!\n");
	}
	ret |= APMC_EN;
	outl(ret, PCICONF_DATA);

	/* enable SMI generation in PIIX I/O space */
	ret = inl(GLBCTL);
	ret &= GLBCTL_RES_MASK;
	ret |= EOS | SMI_EN; /* "arm" SMI and enable it */
	outl(ret, GLBCTL);
	ret = inl(GLBCTL);
	if (! ((ret & EOS) && (ret & SMI_EN))) {
		dprintf("could not set EOS! help!\n");
	}

	/* set APMS */
	outb((base >> 16) & 0xff, APMS);
	/* write 0x02 to APMC to trigger SMI and SMBASE relocation */
	outb(0x02, APMC);
	/* returned from SMI... */
	/* FIXME/PROBLEM:
	 * as SMIs are asynchronous in the qemu CPU, the SMI handler will NOT
	 * trigger exactly at the instruction above, but at the end of the
	 * current code block, wherever that is (most likely the next "ret").
	 * So we can't check for return values of the SMI Handler here,
	 * because there won't be any. */
#if 0
	ret = inb(APMS);
	if (ret != 0x42) {
		dprintf("SMBASE relocation didn't work ");
		dprintf("(expected 0x42, got 0x%02x from APMS)\n", ret & 0xff);
	}
#endif

	/* disable SMI generation for APMC again */
	ret = pci_creadl(0, PIIX_DEV, PIIX_PM_FN, DEVACTB);
	ret &= ~APMC_EN;
	outl(ret, PCICONF_DATA);
}

/*
 * Enable generation of SMIs on power button events
 */
void
smi_enable(void)
{
	uint32_t ret;

	/* enable SMI generation for PWRBTN/APMC in the PIIX configspace */
	/* enable APMC decoding/SMI generation in the PIIX configspace */
	ret = pci_creadl(0, PIIX_DEV, PIIX_PM_FN, DEVACTB);
	if (ret & APMC_EN) {
		dprintf("BIOS: smi_enable: APMC_EN was already set!\n");
	}
	ret |= APMC_EN;
	outl(ret, PCICONF_DATA);

	/* enable power button SMI */
	ret = inw(PMEN);
	if (ret & PWRBTN_EN) {
		dprintf("BIOS: smi_enable: PWRBTN_EN already set!\n");
	}
	ret &= PMEN_RES_MASK; /* don't touch reserved bits */
	ret |= PWRBTN_EN;
	outw(ret, PMEN);
}

#endif /* INIT_RM */
