/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __JIT_H_INCLUDED
#define __JIT_H_INCLUDED

#undef CONFIG_JIT_BC

#include <stddef.h>

struct jit_comp {
	int dummy;
};

struct jit_expr {
	struct jit_expr *prev;
	struct jit_expr *next;

	enum jit_expr_type {
		JIT_CONST,
		JIT_GLOBAL,
		JIT_PARAM,
		JIT_LOCAL,

		JIT_STAR,
		JIT_ADDR,

		JIT_CONV,

		JIT_NOT,
		JIT_INV,
		JIT_NEG,

		JIT_ADD,
		JIT_SUB,
		JIT_MUL,
		JIT_DIV,
		JIT_REM,
		JIT_AND,
		JIT_OR,
		JIT_XOR,
		JIT_LSL,
		JIT_ASR,
		JIT_LSR,

		JIT_EQ,
		JIT_NEQ,
		JIT_B,
		JIT_AE,
	} type;

	int sign;
	unsigned long size;

	/* JIT_CONST */
	unsigned long val;

	/* JIT_GLOBAL */
	void *addr;

	/* JIT_PARAM */
	/* JIT_LOCAL */
	int reg;
	unsigned long offset;

	/* JIT_CPSSP */
	/* Nothing... */

	/* JIT_ADD */
	/* JIT_SUB */
	/* ... */
	struct jit_expr *op0;
	struct jit_expr *op1;
};

struct jit_label {
	struct jit_label *prev;
	struct jit_label *next;

	int id;
	void *jmp[32];
	void *label;
};

struct jit_stmt {
	struct jit_stmt *prev;
	struct jit_stmt *next;

	enum {
		JIT_STMT_ASSIGN,
		JIT_STMT_GOTO,
		JIT_STMT_IFGOTO,
		JIT_STMT_SWITCHGOTO,
		JIT_STMT_LABEL,
		JIT_STMT_CALL,
		JIT_STMT_BRK,
	} type;
	union {
		struct {
			struct jit_expr *dst;
			struct jit_expr *src;
		} assign;
		struct jit_label *goto_label;
		struct {
			struct jit_expr *cond;
			struct jit_label *label;
		} ifgoto;
		struct {
			struct jit_expr *expr;
			struct jit_label *def;
			int val[16];
			struct jit_label *lab[16];
		} switchgoto;
		struct jit_label *label;
		struct {
			struct jit_expr *dst;
			void *func;
			struct jit_expr *arg[6];
		} call;
	};
};

struct jit_func {
	struct jit_expr *param_first;
	struct jit_expr *param_last;

	struct jit_expr *local_first;
	struct jit_expr *local_last;

	struct jit_expr *expr_first;
	struct jit_expr *expr_last;

	struct jit_label *label_first;
	struct jit_label *label_last;

	struct jit_stmt *stmt_first;
	struct jit_stmt *stmt_last;

	void *head;
	void *tail;
};


#if defined(__x86_64__) && ! defined(CONFIG_JIT_BC)
#include "jit-x86_64.h"
#else
#include "jit-bc.h"
#endif


extern struct jit_expr *
jit_param_alloc(struct jit_func *func, int sign, unsigned long size);
extern struct jit_expr *
jit_local_alloc(struct jit_func *func, int sign, unsigned long size);
extern struct jit_label *
jit_label_alloc(struct jit_func *func);

extern struct jit_expr *
jit_sconst(struct jit_func *func, signed long c);
extern struct jit_expr *
jit_uconst(struct jit_func *func, unsigned long c);
extern struct jit_expr *
jit__global(struct jit_func *func, int sign, unsigned long, void *addr);
#define jit_uglobal(v) \
	jit__global(1, sizeof(v), &v)
extern struct jit_expr *
jit_star(struct jit_func *func, int sign, unsigned long size, struct jit_expr *addr);
extern struct jit_expr *
jit_addr(struct jit_func *func, struct jit_expr *expr);
extern struct jit_expr *
jit_attr(struct jit_func *func, int sign, unsigned long size, struct jit_expr *base, unsigned long offset);
extern struct jit_expr *
jit_cpssp(struct jit_func *func);

extern struct jit_expr *
jit_conv(struct jit_func *func, int sign, unsigned long size, struct jit_expr *op);

extern struct jit_expr *
jit_not(struct jit_func *func, struct jit_expr *op);
extern struct jit_expr *
jit_inv(struct jit_func *func, struct jit_expr *op);
extern struct jit_expr *
jit_neg(struct jit_func *func, struct jit_expr *op);

extern struct jit_expr *
jit_add(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_sub(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_mul(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_div(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_rem(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_and(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_or(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_xor(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_lsl(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_asr(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_lsr(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_eq(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_neq(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_b(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);
extern struct jit_expr *
jit_ae(struct jit_func *func, struct jit_expr *op0, struct jit_expr *op1);

extern void
jit_assign(struct jit_func *func, struct jit_expr *var, struct jit_expr *expr);
extern void
jit_goto(struct jit_func *func, struct jit_label *label);
extern void
jit_ifgoto(struct jit_func *func, struct jit_expr *cond, struct jit_label *label);
extern void
jit_switchgoto(struct jit_func *func, struct jit_expr *expr,
		struct jit_label *def, ...);
extern void
jit_label(struct jit_func *func, struct jit_label *label);
extern void
jit_call(struct jit_func *func, struct jit_expr *res, void *ptr, ...);

extern void
jit_brk(struct jit_func *func);

extern struct jit_func *
jit_func_alloc(void);
extern void
jit_dump(struct jit_func *func);
extern void *
jit_compile(struct jit_func *func);

extern void
jit_comp_reset(struct jit_comp *comp);
extern void
jit_comp_create(struct jit_comp *comp);
extern void
jit_comp_destroy(struct jit_comp *comp);

#endif /* __JIT_H_INCLUDED */
