/*
 * Copyright (C) 2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_CONTROL_FLOW		0

#include "config.h"

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/signal.h>
#include <sys/time.h>

#ifdef HAVE_LINUX_KVM_H
#include <sys/user.h>
#define __user
#include <linux/kvm.h>
#ifndef KVM_CAP_COALESCED_MMIO
#define KVM_CAP_COALESCED_MMIO			15
#endif
#ifndef KVM_CAP_SYNC_MMU
#define KVM_CAP_SYNC_MMU			16
#endif
#ifndef KVM_CAP_DESTROY_MEMORY_REGION_WORKS
#define KVM_CAP_DESTROY_MEMORY_REGION_WORKS	21
#endif

#ifndef KVM_REGISTER_COALESCED_MMIO
struct kvm_coalesced_mmio_zone {
	uint64_t addr;
	uint32_t size;
	uint32_t pad;
};
struct kvm_coalesced_mmio {
	uint64_t phys_addr;
	uint32_t len;
	uint32_t pad;
	uint8_t data[8];
};
struct kvm_coalesced_mmio_ring {
	uint32_t first, last;
	struct kvm_coalesced_mmio coalesced_mmio[0];
};
#define KVM_COALESCED_MMIO_MAX \
		((4096 - sizeof(struct kvm_coalesced_mmio_ring)) / \
		sizeof(struct kvm_coalesced_mmio))

#define KVM_REGISTER_COALESCED_MMIO		\
		_IOW(KVMIO,  0x67, struct kvm_coalesced_mmio_zone)
#define KVM_UNREGISTER_COALESCED_MMIO		\
		_IOW(KVMIO,  0x68, struct kvm_coalesced_mmio_zone)
#endif
#endif

#include "glue.h"

#include "chip_intel_80686_klamath_def.h"	/* FIXME */
#include "cpu_host.h"

#define COMP_(x) cpu_host_ ## x

#ifdef HAVE_LINUX_KVM_H

struct cpssp {
	unsigned int state_power;
	unsigned int state_n_reset;
	unsigned int state_n_init;
	unsigned int state_n_ignne;
	struct sig_host_bus *port_main;
	struct sig_icc_bus *port_icc;

	struct kvm_userspace_memory_region memregion[32];
	unsigned int nmemregions;

	unsigned long long a20_mask;
	enum {
		CPU_INTERRUPT_IRQ = 1,
		CPU_INTERRUPT_NMI = 2,
		CPU_INTERRUPT_SMI = 4,
	} interrupt_request;
	enum {
		HF_HALTED_MASK = 1,
	} hflags;

	unsigned int kvm_fd;
	unsigned int vm_fd;
	unsigned int cpu_fd;
	unsigned int coalesced_mmio;
	struct kvm_run *cpu_run;

	struct process process;
};

static void
COMP_(dump)(struct cpssp *cpssp)
{
	struct kvm_regs regs;
	struct kvm_sregs sregs;
	unsigned long addr;
	char *_haddr_mr;
	uint8_t *haddr_mr;
	int ret;

	ret = ioctl(cpssp->cpu_fd, KVM_GET_REGS, &regs);
	assert(0 <= ret);
	ret = ioctl(cpssp->cpu_fd, KVM_GET_SREGS, &sregs);
	assert(0 <= ret);

	fprintf(stderr, "rip=%016llx rflags=%016llx\n",
			regs.rip, regs.rflags);
	fprintf(stderr, "rax=%016llx rbx=%016llx rcx=%016llx rdx=%016llx\n",
			regs.rax, regs.rbx, regs.rcx, regs.rdx);
	fprintf(stderr, "rdi=%016llx rsi=%016llx rbp=%016llx rsp=%016llx\n",
			regs.rdi, regs.rsi, regs.rbp, regs.rsp);
	fprintf(stderr, "r8=%016llx r9=%016llx r10=%016llx r11=%016llx\n",
			regs.r8, regs.r9, regs.r10, regs.r11);
	fprintf(stderr, "r12=%016llx r13=%016llx r14=%016llx r15=%016llx\n",
			regs.r12, regs.r13, regs.r14, regs.r15);
	fprintf(stderr, "cr0=%08lx\n",
			sregs.cr0);
	fprintf(stderr, "cs=%04x %016llx %08lx %d %d %d %d %d %d %d %d %d\n",
			sregs.cs.selector, sregs.cs.base, sregs.cs.limit,
			sregs.cs.type, sregs.cs.present, sregs.cs.dpl,
			sregs.cs.db, sregs.cs.s, sregs.cs.l,
			sregs.cs.g, sregs.cs.avl, sregs.cs.unusable);
	fprintf(stderr, "ss=%04x %016llx %08lx %d %d %d %d %d %d %d %d %d\n",
			sregs.ss.selector, sregs.ss.base, sregs.ss.limit,
			sregs.ss.type, sregs.ss.present, sregs.ss.dpl,
			sregs.ss.db, sregs.ss.s, sregs.ss.l,
			sregs.ss.g, sregs.ss.avl, sregs.ss.unusable);
	fprintf(stderr, "ds=%04x %016llx %08lx %d %d %d %d %d %d %d %d %d\n",
			sregs.ds.selector, sregs.ds.base, sregs.ds.limit,
			sregs.ds.type, sregs.ds.present, sregs.ds.dpl,
			sregs.ds.db, sregs.ds.s, sregs.ds.l,
			sregs.ds.g, sregs.ds.avl, sregs.ds.unusable);
	fprintf(stderr, "es=%04x %016llx %08lx %d %d %d %d %d %d %d %d %d\n",
			sregs.es.selector, sregs.es.base, sregs.es.limit,
			sregs.es.type, sregs.es.present, sregs.es.dpl,
			sregs.es.db, sregs.es.s, sregs.es.l,
			sregs.es.g, sregs.es.avl, sregs.es.unusable);
	fprintf(stderr, "fs=%04x %016llx %08lx %d %d %d %d %d %d %d %d %d\n",
			sregs.fs.selector, sregs.fs.base, sregs.fs.limit,
			sregs.fs.type, sregs.fs.present, sregs.fs.dpl,
			sregs.fs.db, sregs.fs.s, sregs.fs.l,
			sregs.fs.g, sregs.fs.avl, sregs.fs.unusable);
	fprintf(stderr, "gs=%04x %016llx %08lx %d %d %d %d %d %d %d %d %d\n",
			sregs.gs.selector, sregs.gs.base, sregs.gs.limit,
			sregs.gs.type, sregs.gs.present, sregs.gs.dpl,
			sregs.gs.db, sregs.gs.s, sregs.gs.l,
			sregs.gs.g, sregs.gs.avl, sregs.gs.unusable);

	addr = sregs.cs.base + regs.rip;

	sig_host_bus_map_r(cpssp->port_main, cpssp, 0, addr & ~0xfff,
			&_haddr_mr);
	haddr_mr = (uint8_t *) _haddr_mr;

	fprintf(stderr, "At %x: %02x %02x %02x %02x %02x ...\n",
		addr,
		haddr_mr[(addr & 0xfff) + 0],
		haddr_mr[(addr & 0xfff) + 1],
		haddr_mr[(addr & 0xfff) + 2],
		haddr_mr[(addr & 0xfff) + 3],
		haddr_mr[(addr & 0xfff) + 4]);
}

static void
COMP_(map)(
	struct cpssp *cpssp,
	unsigned long long paddr,
	char **haddr_mrp,
	char **haddr_mwp,
	char **haddr_mxp
)
{
	paddr &= cpssp->a20_mask;

	/* Hack for speedup -- FIXME */
	if (512*1024*1024 <= paddr
	 && paddr < 0xffff0000) {
		*haddr_mrp = NULL;
		*haddr_mwp = NULL;
		*haddr_mxp = NULL;
	} else {
		sig_host_bus_map_r(cpssp->port_main, cpssp, 0, paddr, haddr_mrp);
		sig_host_bus_map_w(cpssp->port_main, cpssp, 0, paddr, haddr_mwp);
		sig_host_bus_map_x(cpssp->port_main, cpssp, 0, paddr, haddr_mxp);
	}
}

static void
COMP_(memory)(struct cpssp *cpssp)
{
	unsigned long long paddr;
	unsigned int slot;
	int ret;

	/* Flush old mappings. */
	for (slot = 0; slot < cpssp->nmemregions; slot++) {
		if (cpssp->memregion[slot].userspace_addr) {
			/* Memory */
			/* cpssp->memregion[slot].slot -- Leave as is. */
			cpssp->memregion[slot].flags = 0;
			/* cpssp->memregion[slot].guest_phys_addr -- Leave as is. */
			cpssp->memregion[slot].memory_size = 0;
			cpssp->memregion[slot].userspace_addr = 0;

			ret = ioctl(cpssp->vm_fd, KVM_SET_USER_MEMORY_REGION,
					&cpssp->memregion[slot]);
			assert(0 <= ret);
		} else {
			/* I/O-Memory */
			struct kvm_coalesced_mmio_zone zone;

			zone.addr = cpssp->memregion[slot].guest_phys_addr;
			zone.size = cpssp->memregion[slot].memory_size;

			ret = ioctl(cpssp->vm_fd, KVM_UNREGISTER_COALESCED_MMIO,
					&zone);
			assert(0 <= ret);
		}
	}
	cpssp->nmemregions = 0;

	/* Fill new mappings. */
	slot = 0;
	paddr = 0;
	while (paddr < 0x100000000ULL) {
		char *haddr_mr;
		char *haddr_mw;
		char *haddr_mx;
		char *haddr_mr2;
		char *haddr_mw2;
		char *haddr_mx2;
		unsigned long len;

		COMP_(map)(cpssp, paddr,
				&haddr_mr, &haddr_mw, &haddr_mx);
		len = 4096;
	again:	;
		COMP_(map)(cpssp, paddr + len,
				&haddr_mr2, &haddr_mw2, &haddr_mx2);
		if (! haddr_mr
		 && ! haddr_mr2
		 && ! haddr_mw
		 && ! haddr_mw2) {
			/* Nothing... */
			len += 4096;
			goto again;
		} else if (haddr_mr == haddr_mw
			&& haddr_mr2 == haddr_mw2
			&& haddr_mr + len == haddr_mr2
			&& haddr_mw + len == haddr_mw2) {
			/* Read-write memory region */
			len += 4096;
			goto again;
		} else if (haddr_mr != haddr_mw
			&& haddr_mr2 != haddr_mw2
			&& haddr_mr + len == haddr_mr2) {
			/* Read-only memory region */
			len += 4096;
			goto again;
		}

		if (haddr_mr) assert(! ((long) haddr_mr & 0xfff));
		if (haddr_mw) assert(! ((long) haddr_mw & 0xfff));

		cpssp->memregion[slot].slot = slot;
		cpssp->memregion[slot].flags = 0;
		cpssp->memregion[slot].guest_phys_addr = paddr;
		cpssp->memregion[slot].memory_size = len;
		cpssp->memregion[slot].userspace_addr = (long) haddr_mr;
		slot++;

		paddr += len;
	}

	cpssp->nmemregions = slot;

	for (slot = 0; slot < cpssp->nmemregions; slot++) {
		if (cpssp->memregion[slot].userspace_addr) {
			/* Memory */
			ret = ioctl(cpssp->vm_fd, KVM_SET_USER_MEMORY_REGION,
					&cpssp->memregion[slot]);
			assert(0 <= ret);

		} else {
			/* I/O-Memory */
			struct kvm_coalesced_mmio_zone zone;

			zone.addr = cpssp->memregion[slot].guest_phys_addr;
			zone.size = cpssp->memregion[slot].memory_size;

			ret = ioctl(cpssp->vm_fd, KVM_REGISTER_COALESCED_MMIO,
					&zone);
			assert(0 <= ret);
		}
	}
}

static void
COMP_(do_reset)(struct cpssp *cpssp, int reset_flag)
{
	struct kvm_regs regs;
	struct kvm_sregs sregs;
	int ret;

	ret = ioctl(cpssp->cpu_fd, KVM_GET_REGS, &regs);
	assert(0 <= ret);
	ret = ioctl(cpssp->cpu_fd, KVM_GET_SREGS, &sregs);
	assert(0 <= ret);

	// COMP_(dump)(cpssp);

	cpssp->hflags = 0;
	if (reset_flag) {
		cpssp->interrupt_request = 0;
	}

	/* Code segment registers */
	sregs.cs.base = 0xffff0000;
	sregs.cs.limit = 0xffff;
	sregs.cs.selector = 0xf000;
	sregs.cs.type = 0xb; /* execute/read, accessed */
	sregs.cs.present = 1;
	sregs.cs.dpl = 0;
	sregs.cs.db = 0; /* 16bit segment */
	sregs.cs.s = 1; /* code/data */
	sregs.cs.l = 0; /* ? FIXME */
	sregs.cs.g = 0; /* byte limit */
	sregs.cs.avl = 0;
	sregs.cs.unusable = 0;

	/* Instruction pointer */
	regs.rip = 0x0000fff0;

	/* Other segment registers */
	sregs.ss.base = 0x00000000;
	sregs.ss.limit = 0xffff;
	sregs.ss.selector = 0x0000;
	sregs.ss.type = 0x3; /* read/write, accessed */
	sregs.ss.present = 1;
	sregs.ss.dpl = 0;
	sregs.ss.db = 0; /* 16bit stack pointer */
	sregs.ss.s = 1; /* code/data */
	sregs.ss.l = 0; /* ? FIXME */
	sregs.ss.g = 0; /* byte limit */
	sregs.ss.avl = 0;
	sregs.ss.unusable = 0;

	sregs.ds.base = 0x00000000;
	sregs.ds.limit = 0xffff;
	sregs.ds.selector = 0x0000;
	sregs.ds.type = 0x3; /* read/write, accessed */
	sregs.ds.present = 1;
	sregs.ds.dpl = 0;
	sregs.ds.db = 0;
	sregs.ds.s = 1; /* code/data */
	sregs.ds.l = 0; /* ? FIXME */
	sregs.ds.g = 0; /* byte limit */
	sregs.ds.avl = 0;
	sregs.ds.unusable = 0;

	sregs.es.base = 0x00000000;
	sregs.es.limit = 0xffff;
	sregs.es.selector = 0x0000;
	sregs.es.type = 0x3; /* read/write, accessed */
	sregs.es.present = 1;
	sregs.es.dpl = 0;
	sregs.es.db = 0;
	sregs.es.s = 1; /* code/data */
	sregs.es.l = 0; /* ? FIXME */
	sregs.es.g = 0; /* byte limit */
	sregs.es.avl = 0;
	sregs.es.unusable = 0;

	sregs.fs.base = 0x00000000;
	sregs.fs.limit = 0xffff;
	sregs.fs.selector = 0x0000;
	sregs.fs.type = 0x3; /* read/write, accessed */
	sregs.fs.present = 1;
	sregs.fs.dpl = 0;
	sregs.fs.db = 0;
	sregs.fs.s = 1; /* code/data */
	sregs.fs.l = 0; /* ? FIXME */
	sregs.fs.g = 0; /* byte limit */
	sregs.fs.avl = 0;
	sregs.fs.unusable = 0;

	sregs.gs.base = 0x00000000;
	sregs.gs.limit = 0xffff;
	sregs.gs.selector = 0x0000;
	sregs.gs.type = 0x3; /* read/write, accessed */
	sregs.gs.present = 1;
	sregs.gs.dpl = 0;
	sregs.gs.db = 0;
	sregs.gs.s = 1; /* code/data */
	sregs.gs.l = 0; /* ? FIXME */
	sregs.gs.g = 0; /* byte limit */
	sregs.gs.avl = 0;
	sregs.gs.unusable = 0;

	sregs.cr0 = 0x60000010;
	/* must reset cr3, KVM gets confused otherwise */
	sregs.cr3 = 0x00000000;

	/* edx - depends on processor version - FIXME VOSSI */
	regs.rdx = 0x00000600; /* Indicate P6 processor. */

	/* eax - should contain BIST code - FIXME VOSSI */
	regs.rax = 0x00000000;

	/* Other registers */
	regs.rbx = 0x00000000;
	regs.rcx = 0x00000000;
	regs.rsi = 0x00000000;
	regs.rdi = 0x00000000;
	regs.rbp = 0x00000000;
	regs.rsp = 0x00000000;

	if (reset_flag) {
		/* FPU init */
		/* FIXME */
	}

	/* gdtr, idtr */
	/* FIXME */

	/* ... */
	/* FIXME */

	COMP_(memory)(cpssp);

	ret = ioctl(cpssp->cpu_fd, KVM_SET_SREGS, &sregs);
	assert(0 <= ret);
	ret = ioctl(cpssp->cpu_fd, KVM_SET_REGS, &regs);
	assert(0 <= ret);
}

static void
COMP_(reset)(struct cpssp *cpssp)
{
	COMP_(do_reset)(cpssp, 1);
}

static void
COMP_(init)(struct cpssp *cpssp)
{
	COMP_(do_reset)(cpssp, 0);
}

static void
COMP_(ior)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint32_t *valp)
{
	*valp = -1;
	if (unlikely(sig_host_bus_ior(cpssp->port_main, cpssp, port, bs, valp) != 0)) {
		sig_host_bus_type_addr(cpssp->port_main, cpssp,
				0, SIG_HOST_BUS_IOR, port);
		/* delay... */
		sig_host_bus_read_data(cpssp->port_main, cpssp,
				bs, valp);
	}
}

static void
COMP_(iow)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint32_t val)
{
	if (unlikely(sig_host_bus_iow(cpssp->port_main, cpssp, port, bs, val) != 0)) {
		sig_host_bus_type_addr(cpssp->port_main, cpssp,
				0, SIG_HOST_BUS_IOW, port);
		/* delay... */
		sig_host_bus_write_data(cpssp->port_main, cpssp,
				bs, val);
	}
}

static void
COMP_(mr)(struct cpssp *cpssp, unsigned int state, unsigned long addr, unsigned int bs, uint32_t *valp)
{
	*valp = -1;
	if (unlikely(sig_host_bus_mr(cpssp->port_main, cpssp, state, addr, bs, valp) != 0)) {
		sig_host_bus_type_addr(cpssp->port_main, cpssp,
				0, SIG_HOST_BUS_MR, addr);
		/* delay... */
		sig_host_bus_read_data(cpssp->port_main, cpssp,
				bs, valp);
	}
}

static void
COMP_(mw)(struct cpssp *cpssp, unsigned int state, unsigned long addr, unsigned int bs, uint32_t val)
{
	if (unlikely(sig_host_bus_mw(cpssp->port_main, cpssp, state, addr, bs, val) != 0)) {
		sig_host_bus_type_addr(cpssp->port_main, cpssp,
				0, SIG_HOST_BUS_MW, addr);
		/* delay... */
		sig_host_bus_write_data(cpssp->port_main, cpssp,
				bs, val);
	}
}


static uint8_t
COMP_(inb)(struct cpssp *cpssp, uint16_t port)
{
	uint8_t val8;
	uint32_t val32;

	COMP_(ior)(cpssp, port & ~3, 1 << (port & 3), &val32);
	val8 = (val32 >> ((port & 3) * 8)) & 0xff;

	return val8;
}

static uint16_t
COMP_(inw)(struct cpssp *cpssp, uint16_t port)
{
	uint16_t val16;

	if ((port & 3) == 3) {
		uint32_t val0;
		uint32_t val1;

		COMP_(ior)(cpssp, (port & ~3) + 0, (0x3 << (port & 3)) & 0xf, &val0);
		COMP_(ior)(cpssp, (port & ~3) + 4, (0x3 << (port & 3)) >> 4, &val1);
		val16 = (val0 >> ((port & 3) * 8))
		      | (val1 << ((4 - (port & 3)) * 8));

	} else {
		uint32_t val32;

		COMP_(ior)(cpssp, port & ~3, 3 << (port & 2), &val32);
		val16 = (val32 >> ((port & 2) * 8)) & 0xffff;
	}

	return val16;
}

static uint32_t
COMP_(inl)(struct cpssp *cpssp, uint16_t port)
{
	uint32_t val32;

	if (port & 3) {
		uint32_t val0;
		uint32_t val1;

		COMP_(ior)(cpssp, (port & ~3) + 0, (0xf << (port & 3)) & 0xf, &val0);
		COMP_(ior)(cpssp, (port & ~3) + 4, (0xf << (port & 3)) >> 4, &val1);
		val32 = (val0 >> ((port & 3) * 8))
		      | (val1 << ((4 - (port & 3)) * 8));

	} else {
		COMP_(ior)(cpssp, port, 0xf, &val32);
	}

	return val32;
}

static void
COMP_(outb)(struct cpssp *cpssp, uint16_t port, uint8_t val)
{
	uint32_t val32;

	if (port == 0xfffe
	 && val == 42) {
		/* See HACK_FOR_KVM_INTEL in bios/early_init.S! */
		struct kvm_regs regs;
		struct kvm_sregs sregs;
		int ret;

		fprintf(stderr, "WARNING: Hack activated!\n");

		ret = ioctl(cpssp->cpu_fd, KVM_GET_REGS, &regs);
		assert(0 <= ret);
		ret = ioctl(cpssp->cpu_fd, KVM_GET_SREGS, &sregs);
		assert(0 <= ret);

		/* movl %cr0, %eax; andl $~1, %eax; movl %eax, %cr0 */
		sregs.cr0 &= ~1;

		/* ljmp $OFF_init_rm_bsp, $SEG_init_rm_bsp */
		sregs.cs.base = 0x10000; /* Segment of init_rm_bsp */
		sregs.cs.limit = 0xffff;
		sregs.cs.selector = 0x1000;
		sregs.cs.type = 0xb; /* execute/read, accessed */
		sregs.cs.present = 1;
		sregs.cs.dpl = 0;
		sregs.cs.db = 0; /* 16bit segment */
		sregs.cs.s = 1; /* code/data */
		sregs.cs.l = 0; /* FIXME */
		sregs.cs.g = 0; /* byte limit */
		sregs.cs.avl = 0;
		sregs.cs.unusable = 0;

		sregs.ds.base = 0;
		sregs.ds.limit = 0xffff;
		sregs.ds.selector = 0;

		sregs.es.base = 0;
		sregs.es.limit = 0xffff;
		sregs.es.selector = 0;

		regs.rip = 0x0003; /* Offset of init_rm_bsp */

		ret = ioctl(cpssp->cpu_fd, KVM_SET_SREGS, &sregs);
		assert(0 <= ret);
		ret = ioctl(cpssp->cpu_fd, KVM_SET_REGS, &regs);
		assert(0 <= ret);
		return;
	}
	if (port == 0xffff) {
		/* System BIOS / VGA BIOS output port. */
		fprintf(stderr, "%c", val);
		return;
	}

	val32 = val << ((port & 3) * 8);
	COMP_(iow)(cpssp, port & ~3, 1 << (port & 3), val32);
}

static void
COMP_(outw)(struct cpssp *cpssp, uint16_t port, uint16_t val)
{
	uint32_t val32;

	if ((port & 3) == 3) {
		unsigned char value0 = (val >> 0) & 0xff;
		unsigned char value8 = (val >> 8) & 0xff;

		fprintf(stderr, "%s: WARNING: outw->outb port 0x%04x "
				"val 0x%x\n", __FUNCTION__, port, val);

		COMP_(outb)(cpssp, value0, port + 0);
		COMP_(outb)(cpssp, value8, port + 1);

	} else {
		val32 = val << ((port & 3) * 8);
		COMP_(iow)(cpssp, port & ~3, 3 << (port & 3), val32);
	}
}

static void
COMP_(outl)(struct cpssp *cpssp, uint16_t port, uint32_t val)
{
	if (port & 3) {
		uint32_t val0;
		uint32_t val1;

		fprintf(stderr, "%s: WARNING: outl->outw port 0x%04x "
			"val 0x%08lx\n", __FUNCTION__, port, (long) val);

		val0 = val << ((port & 3) * 8);
		val1 = val << ((4 - (port & 3)) * 8);

		COMP_(iow)(cpssp, port & ~3, (0xf << (port & 3)) & 0xf, val0);
		COMP_(iow)(cpssp, (port & ~3) + 4, 0xf >> (4 - (port & 3)), val1);

	} else {
		COMP_(iow)(cpssp, port & ~3, 0xf, val);
	}
}

static uint8_t
COMP_(readb)(struct cpssp *cpssp, unsigned int state, unsigned long addr)
{
	uint8_t val8;
	uint32_t val32;

	COMP_(mr)(cpssp, state, addr & ~3, 1 << (addr & 3), &val32);
	val8 = (val32 >> ((addr & 3) * 8)) & 0xff;

	return val8;
}

static uint16_t
COMP_(readw)(struct cpssp *cpssp, unsigned int state, unsigned long addr)
{
	uint16_t val16;

	if ((addr & 3) == 3) {
		uint32_t val0;
		uint32_t val1;

		COMP_(mr)(cpssp, state, (addr & ~3) + 0, (0x3 << (addr & 3)) & 0xf, &val0);
		COMP_(mr)(cpssp, state, (addr & ~3) + 4, (0x3 << (addr & 3)) >> 4, &val1);
		val16 = (val0 >> ((addr & 3) * 8))
		      | (val1 << ((4 - (addr & 3)) * 8));

	} else {
		uint32_t val32;

		COMP_(mr)(cpssp, state, addr & ~3, 3 << (addr & 2), &val32);
		val16 = (val32 >> ((addr & 2) * 8)) & 0xffff;
	}

	return val16;
}

static uint32_t
COMP_(readl)(struct cpssp *cpssp, unsigned int state, unsigned long addr)
{
	uint32_t val32;

	if (addr & 3) {
		uint32_t val0;
		uint32_t val1;

		COMP_(mr)(cpssp, state, (addr & ~3) + 0, (0xf << (addr & 3)) & 0xf, &val0);
		COMP_(mr)(cpssp, state, (addr & ~3) + 4, (0xf << (addr & 3)) >> 4, &val1);
		val32 = (val0 >> ((addr & 3) * 8))
		      | (val1 << ((4 - (addr & 3)) * 8));

	} else {
		COMP_(mr)(cpssp, state, addr, 0xf, &val32);
	}

	return val32;
}

static void
COMP_(writeb)(struct cpssp *cpssp, unsigned int state, unsigned long addr, uint8_t val)
{
	uint32_t val32;

	val32 = val << ((addr & 3) * 8);
	COMP_(mw)(cpssp, state, addr & ~3, 1 << (addr & 3), val32);
}

static void
COMP_(writew)(struct cpssp *cpssp, unsigned int state, unsigned long addr, uint16_t val)
{
	uint32_t val32;

	if ((addr & 3) == 3) {
		unsigned char value0 = (val >> 0) & 0xff;
		unsigned char value8 = (val >> 8) & 0xff;

		fprintf(stderr, "%s: WARNING: writew->writeb addr 0x%04x "
				"val 0x%x\n", __FUNCTION__, addr, val);

		COMP_(writeb)(cpssp, state, value0, addr + 0);
		COMP_(writeb)(cpssp, state, value8, addr + 1);

	} else {
		val32 = val << ((addr & 3) * 8);
		COMP_(mw)(cpssp, state, addr & ~3, 3 << (addr & 3), val32);
	}
}

static void
COMP_(writel)(struct cpssp *cpssp, unsigned int state, unsigned long addr, uint32_t val)
{
	if (addr & 3) {
		uint32_t val0;
		uint32_t val1;

		fprintf(stderr, "%s: WARNING: writel->writew addr 0x%04x "
			"val 0x%08lx\n", __FUNCTION__, addr, (long) val);

		val0 = val << ((addr & 3) * 8);
		val1 = val << ((4 - (addr & 3)) * 8);

		COMP_(mw)(cpssp, state, addr & ~3, (0xf << (addr & 3)) & 0xf, val0);
		COMP_(mw)(cpssp, state, (addr & ~3) + 4, 0xf >> (4 - (addr & 3)), val1);

	} else {
		COMP_(mw)(cpssp, state, addr & ~3, 0xf, val);
	}
}

static void
COMP_(handle_io)(struct cpssp *cpssp)
{
	uint8_t direction;
	uint8_t size;
	uint16_t port;
	uint32_t count;
	uint64_t data_offset;
	char *data;
	unsigned int i;

	direction = cpssp->cpu_run->io.direction;
	size = cpssp->cpu_run->io.size;
	port = cpssp->cpu_run->io.port;
	count = cpssp->cpu_run->io.count;
	data_offset = cpssp->cpu_run->io.data_offset;

	data = (char *) cpssp->cpu_run + data_offset;

	for (i = 0; i < count; i++) {
		switch (direction) {
		case KVM_EXIT_IO_IN:
			switch (size) {
			case 1:
				*(uint8_t *) data = COMP_(inb)(cpssp, port);
				break;
			case 2:
				*(uint16_t *) data = COMP_(inw)(cpssp, port);
				break;
			case 4:
				*(uint32_t *) data = COMP_(inl)(cpssp, port);
				break;
			default:
				assert(0); /* Mustn't happen. */
			}
			break;
		case KVM_EXIT_IO_OUT:
			switch (size) {
			case 1:
				COMP_(outb)(cpssp, port, *(uint8_t *) data);
				break;
			case 2:
				COMP_(outw)(cpssp, port, *(uint16_t *) data);
				break;
			case 4:
				COMP_(outl)(cpssp, port, *(uint32_t *) data);
				break;
			default:
				assert(0); /* Mustn't happen. */
			}
			break;
		default:
			assert(0); /* Mustn't happen. */
		}

		data += size;
	}
}

static void
COMP_(handle_mmio_one)(
	struct cpssp *cpssp,
	uint64_t paddr,
	uint8_t *data,
	uint32_t len,
	uint8_t is_write
)
{
	if (is_write) {
		switch (len) {
		case 1:
			COMP_(writeb)(cpssp, 0, paddr, *(uint8_t *) data);
			break;
		case 2:
			COMP_(writew)(cpssp, 0, paddr, *(uint16_t *) data);
			break;
		case 4:
			COMP_(writel)(cpssp, 0, paddr, *(uint32_t *) data);
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
	} else {
		switch (len) {
		case 1:
			*(uint8_t *) data = COMP_(readb)(cpssp, 0, paddr);
			break;
		case 2:
			*(uint16_t *) data = COMP_(readw)(cpssp, 0, paddr);
			break;
		case 4:
			*(uint32_t *) data = COMP_(readl)(cpssp, 0, paddr);
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
	}
}

static void
COMP_(handle_mmio)(struct cpssp *cpssp)
{
	uint64_t paddr;
	uint8_t *data;
	uint32_t len;
	uint8_t is_write;

	paddr = cpssp->cpu_run->mmio.phys_addr;
	data = cpssp->cpu_run->mmio.data;
	len = cpssp->cpu_run->mmio.len;
	is_write = cpssp->cpu_run->mmio.is_write;

	COMP_(handle_mmio_one)(cpssp, paddr, data, len, is_write);
}

static void __attribute__((__noreturn__))
COMP_(step)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
#if DEBUG_CONTROL_FLOW
	struct kvm_regs regs;
	struct kvm_sregs sregs;
#endif
	unsigned int start_low, start_high;
	unsigned long long start;
	unsigned int end_low, end_high;
	unsigned long long end;
	int ret;

again:	;
	sched_to_scheduler();

	while (cpssp->process.inst_cnt + 1 < cpssp->process.inst_limit) {

		if (unlikely(! cpssp->state_power)) {
			/*
			 * Power-off
			 */
#if DEBUG_CONTROL_FLOW
			fprintf(stderr, "Power-off\n");
#endif
			if (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
				cpssp->process.inst_cnt = cpssp->process.inst_limit;
			}

		} else if (unlikely(cpssp->state_n_reset != 3)) {
			/*
			 * Reset
			 */
#if DEBUG_CONTROL_FLOW
			fprintf(stderr, "Reset\n");
#endif
			if (cpssp->state_n_reset == 1) {
				COMP_(reset)(cpssp);
				cpssp->state_n_reset |= 2; /* Reset handled. */
			}
			if (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
				cpssp->process.inst_cnt = cpssp->process.inst_limit;
			}

		} else if (unlikely(cpssp->state_n_init != 3)) {
			/*
			 * Init
			 */
#if DEBUG_CONTROL_FLOW
			fprintf(stderr, "Init\n");
#endif
			if (cpssp->state_n_init == 1) {
				COMP_(init)(cpssp);
				cpssp->state_n_init |= 2; /* Init handled. */
			}
			if (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
				cpssp->process.inst_cnt = cpssp->process.inst_limit;
			}

		/* FIXME */

		} else if (unlikely(cpssp->interrupt_request & CPU_INTERRUPT_SMI)) {
			/*
			 * SMI
			 */
#if DEBUG_CONTROL_FLOW
			fprintf(stderr, "SMM-Interrupt\n");
#endif
			assert(0); /* FIXME */

		} else if (unlikely(cpssp->interrupt_request & CPU_INTERRUPT_IRQ)) {
			/*
			 * IRQ
			 */
			if (cpssp->cpu_run->if_flag) {
				cpssp->hflags &= ~HF_HALTED_MASK;

				if (cpssp->cpu_run->ready_for_interrupt_injection) {
					uint8_t vec;
					struct kvm_interrupt irq;

					cpssp->interrupt_request &= ~CPU_INTERRUPT_IRQ;

					sig_host_bus_inta_addr(cpssp->port_main, cpssp);
					sig_host_bus_inta_data(cpssp->port_main, cpssp,
							&vec);

#if DEBUG_CONTROL_FLOW
					fprintf(stderr, "Interrupt 0x%02x\n",
							vec);
#endif

					irq.irq = vec;

					ret = ioctl(cpssp->cpu_fd, KVM_INTERRUPT, &irq);
					assert(0 <= ret);
				} else {
					cpssp->cpu_run->request_interrupt_window = 1;
				}
			}
			goto run;

		} else if (unlikely(cpssp->hflags & HF_HALTED_MASK)) {
			/*
			 * Halted
			 */
#if DEBUG_CONTROL_FLOW
			ret = ioctl(cpssp->cpu_fd, KVM_GET_REGS, &regs);
			assert(0 <= ret);
			ret = ioctl(cpssp->cpu_fd, KVM_GET_SREGS, &sregs);
			assert(0 <= ret);

			fprintf(stderr, "Halting at %08llx (%08llx:%08llx)\n",
					sregs.cs.base + regs.rip,
					sregs.cs.base, regs.rip);
#endif

			if (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
				cpssp->process.inst_cnt = cpssp->process.inst_limit;
			}

		} else {
			/*
			 * Normal running...
			 */
			struct kvm_coalesced_mmio_ring *ring;

		run:	;
#if DEBUG_CONTROL_FLOW
			ret = ioctl(cpssp->cpu_fd, KVM_GET_REGS, &regs);
			assert(0 <= ret);
			ret = ioctl(cpssp->cpu_fd, KVM_GET_SREGS, &sregs);
			assert(0 <= ret);

			fprintf(stderr, "Executing at %08lx (%08lx:%08lx)\n",
					sregs.cs.base + regs.rip,
					sregs.cs.base, regs.rip);
#endif

			asm volatile (
				"rdtsc\n\t"
				: "=a" (start_low), "=d" (start_high)
			);
			start = ((uint64_t) start_high << 32) | start_low;

			ret = ioctl(cpssp->cpu_fd, KVM_RUN, 0);

			/* Do coalesced memory I/O. */
			ring = (struct kvm_coalesced_mmio_ring *)
					((char *) cpssp->cpu_run
					+ cpssp->coalesced_mmio * 4096);

			while (ring->first != ring->last) {
				COMP_(handle_mmio_one)(cpssp,
					ring->coalesced_mmio[ring->first].phys_addr,
					&ring->coalesced_mmio[ring->first].data[0],
					ring->coalesced_mmio[ring->first].len,
					1);
				ring->first = (ring->first + 1)
					% KVM_COALESCED_MMIO_MAX;
			}

			switch (cpssp->cpu_run->exit_reason) {
			case KVM_EXIT_UNKNOWN:
				COMP_(dump)(cpssp);
				assert(0); /* Mustn't happen. */
				break;
			case KVM_EXIT_EXCEPTION:
				fprintf(stderr, "Exception 0x%x 0x%x\n",
						cpssp->cpu_run->ex.exception,
						cpssp->cpu_run->ex.error_code);
				COMP_(dump)(cpssp);
				assert(0); /* Mustn't happen. */
				break;
			case KVM_EXIT_IO:
				COMP_(handle_io)(cpssp);
				break;
			case KVM_EXIT_HYPERCALL:
				COMP_(dump)(cpssp);
				assert(0); /* Mustn't happen. */
				break;
			case KVM_EXIT_DEBUG:
				COMP_(dump)(cpssp);
				assert(0); /* FIXME */
				break;
			case KVM_EXIT_HLT:
				cpssp->hflags |= HF_HALTED_MASK;
				break;
			case KVM_EXIT_MMIO:
				COMP_(handle_mmio)(cpssp);
				break;
			case KVM_EXIT_IRQ_WINDOW_OPEN:
				/* Just return to scheduler. */
				break;
			case KVM_EXIT_SHUTDOWN:
				COMP_(dump)(cpssp);
				assert(0); /* FIXME */
				break;
			case KVM_EXIT_FAIL_ENTRY:
				COMP_(dump)(cpssp);
				assert(0); /* Mustn't happen. */
				break;
			case KVM_EXIT_INTR:
				/* Just return to scheduler. */
				break;
			default:
				fprintf(stderr, "reason=%d\n",
						cpssp->cpu_run->exit_reason);
				COMP_(dump)(cpssp);
				assert(0); /* FIXME */
			}

			asm volatile (
				"rdtsc\n\t"
				: "=a" (end_low), "=d" (end_high)
			);
			end = ((uint64_t) end_high << 32) | end_low;

			assert(start < end);
			cpssp->process.inst_cnt += end - start;
		}
	}

	goto again;
}

static void
COMP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_power = val;
}

static void
COMP_(n_reset_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = _cpssp;

	if (n_val) {
		/* Reset gone. */
		cpssp->state_n_reset |= 1;
	} else {
		/* New reset. */
		cpssp->state_n_reset = 0;
	}
}

static void
COMP_(n_init_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = _cpssp;

	if (n_val) {
		/* Init gone. */
		cpssp->state_n_init |= 1;
	} else {
		/* New init. */
		cpssp->state_n_init = 0;
	}
}

static void
COMP_(unmap)(void *_cpssp, uint32_t pa, uint32_t len)
{
	struct cpssp *cpssp = _cpssp;

	COMP_(memory)(cpssp);
}

static void
COMP_(lint0_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	if (val) {
		cpssp->interrupt_request |= CPU_INTERRUPT_IRQ;
	} else {
		cpssp->interrupt_request &= ~CPU_INTERRUPT_IRQ;
	}
}

static void
COMP_(lint1_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	if (val) {
		cpssp->interrupt_request |= CPU_INTERRUPT_NMI;
	} else {
		cpssp->interrupt_request &= ~CPU_INTERRUPT_NMI;
	}
}

static void
COMP_(smi_set)(void *_cpssp, unsigned int val)
{
	fprintf(stderr, "%s %d\n", __FUNCTION__, __LINE__);
	fprintf(stderr, "val=%d\n", val);
}

static void
COMP_(a20_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	val = (val != 0);

	if (val != ((cpssp->a20_mask >> 20) & 1)) {
		/*
		 * Set new mask.
		 */
		cpssp->a20_mask = (unsigned long long)
				(~(1ULL << 20) | (val << 20));

		/*
		 * When a20 is changed, all the MMU mappings are
		 * invalid, so we must flush everything.
		 */
		COMP_(memory)(cpssp);
	}
}

static void
COMP_(n_ferr_set)(void *_cpssp, unsigned int val)
{
	fprintf(stderr, "%s %d\n", __FUNCTION__, __LINE__);
	fprintf(stderr, "val=%d\n", val);
}

static void
COMP_(n_ignne_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_n_ignne = n_val;
}

static void
COMP_(sigalrm_handler)(int sig)
{
        assert(sig == SIGALRM);
}

static void
COMP_(start_timer)(void)
{
        static int running = 0;
        struct sigaction sa;
        struct itimerval it;
        int ret;

        if (running) {
                return;
        }
        running = 1;

        sa.sa_handler = COMP_(sigalrm_handler);
        sa.sa_flags = SA_RESTART;
        sigemptyset(&sa.sa_mask);
        ret = sigaction(SIGALRM, &sa, NULL);
        assert(0 <= ret);

        it.it_interval.tv_sec = 0;
        it.it_interval.tv_usec = 100;
        it.it_value.tv_sec = 0;
        it.it_value.tv_usec = 100;
        ret = setitimer(ITIMER_REAL, &it, NULL);
        assert(0 <= ret);
}

#endif /* HAVE_LINUX_KVM_H */

void *
COMP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_slot1_conn *port_conn
)
{
#ifdef HAVE_LINUX_KVM_H
	static const struct sig_boolean_funcs power_funcs = {
		.set = COMP_(power_set),
	};
	static const struct sig_boolean_funcs n_reset_funcs = {
		.set = COMP_(n_reset_set),
	};
	static const struct sig_boolean_funcs n_init_funcs = {
		.set = COMP_(n_init_set),
	};
	static const struct sig_host_bus_funcs main_funcs = {
		.unmap = COMP_(unmap),
	};
	static const struct sig_std_logic_funcs lint0_funcs = {
		.boolean_or_set = COMP_(lint0_set),
	};
	static const struct sig_std_logic_funcs lint1_funcs = {
		.boolean_or_set = COMP_(lint1_set),
	};
	static const struct sig_boolean_funcs smi_funcs = {
		.set = COMP_(smi_set),
	};
	static const struct sig_boolean_funcs a20_funcs = {
		.set = COMP_(a20_set),
	};
	static const struct sig_std_logic_funcs n_ferr_funcs = {
		.boolean_or_set = COMP_(n_ferr_set),
	};
	static const struct sig_boolean_funcs n_ignne_funcs = {
		.set = COMP_(n_ignne_set),
	};
	static const struct sig_icc_bus_funcs icc_funcs = {
		/* FIXME */
	};
	struct cpssp *cpssp;
	int mmap_size;
	struct {
		struct kvm_cpuid2 cpuid;
		struct kvm_cpuid_entry2 entries[100];
	} __attribute__((packed)) cpuid_data;
	int ret;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);
	memset(cpssp, 0, sizeof(*cpssp));

	cpssp->process.inst_hz = 4000*1000*1000ULL;

	/* Open /dev/kvm. */
	cpssp->kvm_fd = open("/dev/kvm", O_RDWR);
	if (cpssp->kvm_fd < 0) {
		fprintf(stderr, "%s: Can't open %s: %s.\n", progname,
				"/dev/kvm", strerror(errno));
		assert(0); /* FIXME */
	}

	/* Check API version. */
	ret = ioctl(cpssp->kvm_fd, KVM_GET_API_VERSION, 0);
	if (ret < 0) {
		fprintf(stderr, "%s: %s: %s.\n", progname,
				"KVM_GET_API_VERSION", strerror(errno));
		assert(0); /* FIXME */
	}
	if (ret != KVM_API_VERSION) {
		fprintf(stderr, "%s: %d: %s.\n", progname,
				ret, "Unknown API version");
		assert(0); /* FIXME */
	}

	/* Check KVM_CAP_SET_TSS_ADDR. */
	ret = ioctl(cpssp->kvm_fd, KVM_CHECK_EXTENSION, KVM_CAP_SET_TSS_ADDR);
	assert(0 <= ret);
	assert(ret == 1);

	/* Check KVM_CAP_USER_MEMORY. */
	ret = ioctl(cpssp->kvm_fd, KVM_CHECK_EXTENSION, KVM_CAP_USER_MEMORY);
	assert(0 <= ret);
	assert(ret == 1);

	/* Check KVM_CAP_DESTROY_MEMORY_REGION_WORKS. */
	ret = ioctl(cpssp->kvm_fd, KVM_CHECK_EXTENSION,
			KVM_CAP_DESTROY_MEMORY_REGION_WORKS);
	assert(0 <= ret);
	assert(ret == 1);

	/* Check KVM_CAP_SYNC_MMU. */
	ret = ioctl(cpssp->kvm_fd, KVM_CHECK_EXTENSION, KVM_CAP_SYNC_MMU);
	assert(0 <= ret);
	assert(ret == 1);

	/* Check KVM_CAP_COALESCED_MMIO. */
	ret = ioctl(cpssp->kvm_fd, KVM_CHECK_EXTENSION, KVM_CAP_COALESCED_MMIO);
	assert(0 <= ret);
	assert(0 < ret);
	cpssp->coalesced_mmio = ret;

	/* Create virtual machine. */
	cpssp->vm_fd = ioctl(cpssp->kvm_fd, KVM_CREATE_VM, 0);
	assert(0 <= cpssp->vm_fd);

	ret = ioctl(cpssp->vm_fd, KVM_SET_TSS_ADDR, 0xff000000);
	assert(0 <= ret);

	/* Create virtual CPU. */
	cpssp->cpu_fd = ioctl(cpssp->vm_fd, KVM_CREATE_VCPU, 0);
	assert(0 <= cpssp->cpu_fd);

	mmap_size = ioctl(cpssp->kvm_fd, KVM_GET_VCPU_MMAP_SIZE, 0);
	assert(0 <= mmap_size);

	cpssp->cpu_run = mmap(NULL, mmap_size, PROT_READ | PROT_WRITE,
			MAP_SHARED, cpssp->cpu_fd, 0);
	assert(cpssp->cpu_run != MAP_FAILED);

	cpuid_data.entries[0].function = 0;
	cpuid_data.entries[0].index = 0;
	cpuid_data.entries[0].flags = 0;
	cpuid_data.entries[0].eax = 2;
	cpuid_data.entries[0].ebx = 0x756e6547;
	cpuid_data.entries[0].ecx = 0x6c65746e;
	cpuid_data.entries[0].edx = 0x49656e69;

	cpuid_data.entries[1].function = 1;
	cpuid_data.entries[1].index = 0;
	cpuid_data.entries[1].flags = 0;
	cpuid_data.entries[1].eax = 0x00000634;
	cpuid_data.entries[1].ebx = 0x00000000;
	cpuid_data.entries[1].ecx = 0x00000000;
	cpuid_data.entries[1].edx = CONFIG_CPU_FPU_SUPPORT << 0
			| CONFIG_CPU_VME_SUPPORT << 1
			| CONFIG_CPU_DE_SUPPORT << 2
			| CONFIG_CPU_PSE_SUPPORT << 3
			| CONFIG_CPU_TSC_SUPPORT << 4
			| CONFIG_CPU_MSR_SUPPORT << 5
			| CONFIG_CPU_PAE_SUPPORT << 6
			| CONFIG_CPU_MCE_SUPPORT << 7
			| CONFIG_CPU_CX8_SUPPORT << 8
			// | CONFIG_CPU_APIC_SUPPORT << 9
			| CONFIG_CPU_SEP_SUPPORT << 11
			| CONFIG_CPU_MTRR_SUPPORT << 12
			| CONFIG_CPU_PGE_SUPPORT << 13
			| CONFIG_CPU_MCA_SUPPORT << 14
			| CONFIG_CPU_CMOV_SUPPORT << 15
			| CONFIG_CPU_PAT_SUPPORT << 16
			| CONFIG_CPU_PSE36_SUPPORT << 17
			| CONFIG_CPU_PSN_SUPPORT << 18
			| CONFIG_CPU_CFLSH_SUPPORT << 19
			| CONFIG_CPU_DS_SUPPORT << 21
			| CONFIG_CPU_ACPI_SUPPORT << 22
			| CONFIG_CPU_MMX_SUPPORT << 23
			| CONFIG_CPU_FXSR_SUPPORT << 24
			| CONFIG_CPU_SSE_SUPPORT << 25
			| CONFIG_CPU_SSE2_SUPPORT << 26
			| CONFIG_CPU_SS_SUPPORT << 27
			| CONFIG_CPU_HTT_SUPPORT << 28
			| CONFIG_CPU_TM_SUPPORT << 29
			| CONFIG_CPU_PBE_SUPPORT << 31;

	cpuid_data.entries[2].function = 2;
	cpuid_data.entries[2].index = 0;
	cpuid_data.entries[2].flags = 0;
	cpuid_data.entries[2].eax = 0x03020101;
	cpuid_data.entries[2].ebx = 0x00000000;
	cpuid_data.entries[2].ecx = 0x00000000;
	cpuid_data.entries[2].edx = 0x0c040843;

	cpuid_data.cpuid.nent = 3;
	ret = ioctl(cpssp->cpu_fd, KVM_SET_CPUID2, &cpuid_data);
	assert(0 <= ret);

	/* Start timer to interrupt running CPU every now and then. */
	COMP_(start_timer)();

	/* Call */
	cpssp->port_main = port_conn->main;
	sig_host_bus_connect(port_conn->main, cpssp, &main_funcs);
	cpssp->port_icc = port_conn->icc;
	sig_icc_bus_connect(port_conn->icc, cpssp, &icc_funcs);

	/* Out */

	/* In */
	sig_boolean_connect_in(port_conn->power, cpssp, &power_funcs);
	sig_boolean_connect_in(port_conn->n_reset, cpssp, &n_reset_funcs);
	sig_boolean_connect_in(port_conn->n_init, cpssp, &n_init_funcs);
	sig_std_logic_connect_in(port_conn->lint0, cpssp, &lint0_funcs);
	sig_std_logic_connect_in(port_conn->lint1, cpssp, &lint1_funcs);
	sig_boolean_connect_in(port_conn->smi, cpssp, &smi_funcs);
	sig_boolean_connect_in(port_conn->a20, cpssp, &a20_funcs);
	sig_std_logic_connect_in(port_conn->n_ferr, cpssp, &n_ferr_funcs);
	sig_boolean_connect_in(port_conn->n_ignne, cpssp, &n_ignne_funcs);

	sched_process_init(&cpssp->process, COMP_(step), cpssp);

	return cpssp;

#else /* HAVE_LINUX_KVM_H */
	fprintf(stderr, "KVM is missing!\n");
	return NULL;
#endif /* HAVE_LINUX_KVM_H */
}

void
COMP_(destroy)(void *_cpssp)
{
#ifdef HAVE_LINUX_KVM_H
	struct cpssp *cpssp = _cpssp;

	/* FIXME */

	shm_free(cpssp);
#endif /* HAVE_LINUX_KVM_H */
}

void
COMP_(suspend)(void *_cpssp, FILE *fComp)
{
	fprintf(stdout, "cpu_host_suspend not yet implemented\n");
}

void
COMP_(resume)(void *_cpssp, FILE *fComp)
{
	fprintf(stdout, "cpu_host_resume not yet implemented\n");
}
