/* $Id: telephone_isp.c,v 1.23 2009-01-28 12:59:22 potyra Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <sched.h>
#include <signal.h>
#include <stdbool.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <unistd.h>

#include "glue-io.h"
#include "glue-log.h"
#include "glue-main.h"
#include "glue-shm.h"

#include "telephone_isp.h"

/* ********************* DEFINITIONS ************************** */

#define COMP	"telephone_isp"

#define SCRIPT_HEADER	    "Faum-ISP at your service\r\n"
#define SCRIPT_PROMPT_LOGIN "Login: "
#define SCRIPT_PROMPT_PASS  "Password: "

enum isp_states {
	STATE_OFFLINE,
	STATE_CONNECTED,
	STATE_AUTH
};

enum auth_states {
	AUTH_UNTRIGGERED,
	AUTH_WAIT_LOGIN,
	AUTH_WAIT_PASS,
	AUTH_SUCCESS,
	AUTH_FAILURE
};

/* ****************** GLOBAL VARIABLES ************************ */

/* FIXME */
struct cpssp {
	struct sig_telephone *port_phone;

	int state_switch;
	enum isp_states isp_state;
	enum auth_states auth_state;
	int auth_buf_pos;
	pid_t child_pid;
	int child_fd;
};


/* ******************** IMPLEMENTAION ************************* */

static int
isp_get_power_state(struct cpssp *cpssp)
{
	return cpssp->state_switch;
}

static void
isp_send_tele(struct cpssp *cpssp, const uint8_t *data, int len) {
	int i;

	if (! isp_get_power_state(cpssp)) {
		return;
	}

	if (cpssp->isp_state != STATE_AUTH) {
		return;
	}

	for (i = 0; i < len; i++) {
		sig_telephone_send_data(cpssp->port_phone, cpssp, data[i]);
	}
}

static void 
auth_handle_login(struct cpssp *cpssp, char *user, int len) 
{
	/* FIXME check user */
	faum_log(FAUM_LOG_DEBUG, "node-isp", "", "sent user %s\n", user);

	cpssp->auth_state=AUTH_WAIT_PASS;
	isp_send_tele(cpssp, SCRIPT_PROMPT_PASS, strlen(SCRIPT_PROMPT_PASS));
}

static void
auth_handle_password(struct cpssp *cpssp, char *pass, int len)
{
	/* FIXME check password */
	faum_log(FAUM_LOG_DEBUG, "node-isp", "", "sent pass %s\n", pass);
	cpssp->auth_state=AUTH_SUCCESS;
}

static void
auth_reset(struct cpssp *cpssp) 
{
	cpssp->auth_state = AUTH_UNTRIGGERED;
	cpssp->auth_buf_pos = 0;
}

static bool
auth_create(struct cpssp *cpssp)
{
	auth_reset(cpssp);
	return true;
}

static bool
auth_push_char(struct cpssp *cpssp, unsigned char c)
{
	static char buf[1024];
	
	switch (cpssp->auth_state) {

	case AUTH_UNTRIGGERED:
	case AUTH_FAILURE:
	case AUTH_SUCCESS:
		faum_log(FAUM_LOG_DEBUG, "node-isp", "", 
				"invalid state of auth\n");
		return false;

	case AUTH_WAIT_PASS:
		if ((c == '\n') || (c == '\r')) {
			/* buffer empty? */
			if (! cpssp->auth_buf_pos) {
				return true;
			}
			auth_handle_password(cpssp, buf, cpssp->auth_buf_pos);
			cpssp->auth_buf_pos=0;
			/* done with it! */
			return false;
		} else {
			buf[cpssp->auth_buf_pos++] = c;
			assert(cpssp->auth_buf_pos < 1024);
			isp_send_tele(cpssp, &c, 1);
		}
		return true;

		break;

	case AUTH_WAIT_LOGIN:
		if ((c == '\n') || (c == '\r')) {
			/* buffer empty? */
			if (! cpssp->auth_buf_pos) {
				return true;
			}
			isp_send_tele(cpssp, "\r\n", 2);
			auth_handle_login(cpssp, buf, cpssp->auth_buf_pos);
			cpssp->auth_buf_pos=0;
			return true;
		} else {
			buf[cpssp->auth_buf_pos++] = c;
			assert(cpssp->auth_buf_pos < 1024);
			isp_send_tele(cpssp, &c, 1);
		}
		return true;
		break;
	}

	return true;
}

static bool 
auth_get_state(struct cpssp *cpssp)
{
	switch (cpssp->auth_state) {
	case AUTH_SUCCESS:
		return true;

	case AUTH_FAILURE:
		return false;

	case AUTH_UNTRIGGERED:
	case AUTH_WAIT_LOGIN:
	case AUTH_WAIT_PASS:
		faum_log(FAUM_LOG_DEBUG, "node-isp", "", 
				"external state UNDEFINED!\n");
	}
	
	return false;
}

static void
auth_trigger(struct cpssp *cpssp) 
{
	unsigned char backbuf[1024];

	sprintf(backbuf, "%s%s", SCRIPT_HEADER, SCRIPT_PROMPT_LOGIN);
	isp_send_tele(cpssp, backbuf, strlen(backbuf));

	cpssp->auth_state=AUTH_WAIT_LOGIN;
}

#if 0 /* sig_child handled by glue-main already. */
static void
sig_sigchild(int signum) 
{
	int ret;
	int status;
	
	ret = waitpid(child_pid, &status, WNOHANG);

	if (ret && (ret == child_pid)) {
		if (isp_state != STATE_OFFLINE) {
			/* child died unexpected... send hangup to peer */

			isp_state=STATE_OFFLINE;
			sig_telephone_send_ctrl(cpssp_port_phone, NULL,
						SIG_TELE_HANGUP);

		}
		
		if (child_fd) {
			io_unregister(child_fd);
			close(child_fd);
			child_fd = 0;
		}
	}
}
#endif


static void
slirp_recv(int fd, void *_cpssp)
{
	static uint8_t buf[2000];
	ssize_t ret, i;
	struct cpssp *cpssp = (struct cpssp*)_cpssp;

	if (! isp_get_power_state(cpssp)) {
		return;
	}

	if (cpssp->isp_state != STATE_CONNECTED) {
		return;
	}

	for (;;) {
		errno = 0;
		ret = read(fd, buf, sizeof(buf));
		if (ret < 0 && errno == EAGAIN) {
			break;
		}

		if (ret == 0) {
			faum_log(FAUM_LOG_WARNING, "node-isp", "slirp", 
						"broken pipe?\n");
			break;
		}

		for (i = 0; i < ret; i++) {
			sig_telephone_send_data(cpssp->port_phone, cpssp,
						buf[i]);
		}
	}
}

static void 
sanitize_locale(void) 
{
	unsetenv("LANG");
	unsetenv("LC_CTYPE");
	unsetenv("LC_NUMERIC");
	unsetenv("LC_TIME");
	unsetenv("LC_COLLATE");
	unsetenv("LC_MONETARY");
	unsetenv("LC_MESSAGES");
	unsetenv("LC_PAPER");
	unsetenv("LC_NAME");
	unsetenv("LC_ADDRESS");
	unsetenv("LC_TELEPHONE");
	unsetenv("LC_MEASUREMENT");
	unsetenv("LC_IDENTIFICATION");
	unsetenv("LC_ALL");
}

static void 
launch_slirp(struct cpssp *cpssp) 
{
	int ret;
	int iopipe[2];

	ret = socketpair(PF_UNIX, SOCK_STREAM, 0, iopipe);
	if (ret < 0) {
		perror("pipe");

		/* send hangup to peer */
		sig_telephone_send_ctrl(cpssp->port_phone, cpssp, 
					SIG_TELE_HANGUP);
		cpssp->isp_state=STATE_OFFLINE;

		auth_reset(cpssp);
		return;
	}

	cpssp->child_pid = fork();
	switch (cpssp->child_pid) {
	case 0:	/* child */
		sanitize_locale(); /* FIXME needed? */

		ret = dup2(iopipe[0], 0);
		assert(0 <= ret);
		ret = close(1);
		assert(0 <= ret);
		ret = open("/dev/null", O_WRONLY);
		assert(ret == 1);
		ret = close(iopipe[0]);
		assert(0 <= ret);
		ret = close(iopipe[1]);
		assert(0 <= ret);

		/* launch slirp */
		ret = execlp("slirp", "slirp", "ppp" ,"mtu 1400", NULL);

		/* we shouldn't get here */
		if (ret == -1) {
			perror("could not execute\n");
			faum_log(FAUM_LOG_CRITICAL, "node-isp", "slirp", 
					"Couldn't find slirp.\n");
		}
		
		exit(1);
		break;
		
	case -1: /* fork not possible */
		faum_log(FAUM_LOG_DEBUG, "node-isp", "", 
				"slirp fork failed\n");
		close(iopipe[0]);
		close(iopipe[1]);
		
		/* send hangup to peer */
		sig_telephone_send_ctrl(cpssp->port_phone, cpssp, 
					SIG_TELE_HANGUP);
		cpssp->isp_state=STATE_OFFLINE;

		auth_reset(cpssp);
		return;

	default: /* parent */
		ret = close(iopipe[0]);
		assert(0 <= ret);

		cpssp->child_fd = iopipe[1];
		io_register(cpssp->child_fd, cpssp, slirp_recv);
		cpssp->isp_state = STATE_CONNECTED;
		break;
	}

	/* now we reach ONLINE state */
}

static void
kill_slirp(struct cpssp *cpssp)
{
	int ret;
	int status;

	if (cpssp->child_pid) {
		ret = kill(cpssp->child_pid, SIGKILL);
		if (ret == -1) {
			perror("could not kill slirp:");
		}

		ret = waitpid(cpssp->child_pid, &status, WNOHANG);
		cpssp->child_pid=0;

		if (cpssp->child_fd != -1) {
			io_unregister(cpssp->child_fd);
			ret = close(cpssp->child_fd);
			assert(ret >= 0);
			cpssp->child_fd = -1;
		}
	}
}

static void
isp_reset(struct cpssp *cpssp) 
{
	if (cpssp->isp_state != STATE_OFFLINE) {
		sig_telephone_send_ctrl(cpssp->port_phone, cpssp, 
					SIG_TELE_HANGUP);
	}
	
	cpssp->isp_state = STATE_OFFLINE;
	kill_slirp(cpssp);
	auth_reset(cpssp);
}

static void
isp_handle_conn_event(struct cpssp *cpssp, enum sig_telephone_protocol event)
{
	switch(event) {
	case SIG_TELE_BUSY:
		/* peer hung up */
		cpssp->isp_state = STATE_OFFLINE;
		isp_reset(cpssp);
		return;
	default:
		faum_log(FAUM_LOG_DEBUG, "node-isp", __func__, 
				"invalid protocol!\n");
	}
}

static void
isp_handle_off_event(struct cpssp *cpssp, enum sig_telephone_protocol event)
{
	if (event != SIG_TELE_RING) {
		faum_log(FAUM_LOG_DEBUG, "node-isp", __func__, 
				"invalid protocol %d!\n", event);
		return;
	} 

	cpssp->isp_state=STATE_AUTH;
	sig_telephone_send_ctrl(cpssp->port_phone, cpssp, SIG_TELE_LIFT);
	auth_trigger(cpssp);
}

static void
isp_handle_auth_event(struct cpssp *cpssp, enum sig_telephone_protocol event)
{
	if (event == SIG_TELE_BUSY) {
		/* peer hung up */
		isp_reset(cpssp);
		return;
	}

	/* invalid */
	faum_log(FAUM_LOG_WARNING, "node-isp", __func__,
			"invalid protocol=%d\n", event);
}

static void
slirp_send(struct cpssp *cpssp, uint8_t data)
{
	int ret;

	if (cpssp->child_fd == -1) {
		faum_log(FAUM_LOG_WARNING, "node-isp", "slirp",
			"slirp fd is closed, dropping data.\n");
		return;
	}

	ret = io_write(cpssp->child_fd, &data, sizeof(data));
	if (ret != sizeof(data)) {
		faum_log(FAUM_LOG_WARNING, COMP, __func__, 
			"dropped one byte to slirp\n");
	}
}

static void
isp_recv_data(void *_cpssp, uint8_t data)
{
	bool more;
	struct cpssp *cpssp = (struct cpssp*)_cpssp;

	if (! isp_get_power_state(cpssp)) {
		return;
	}

	switch (cpssp->isp_state) {
	case STATE_AUTH:
		more = auth_push_char(cpssp, data);
		if (! more) {
			/* authentication complete */
			if (auth_get_state(cpssp)) {
				cpssp->isp_state = STATE_CONNECTED;
				launch_slirp(cpssp);
			} else {
				/* authentication failed. */
				auth_reset(cpssp);
				cpssp->isp_state = STATE_OFFLINE;
				sig_telephone_send_ctrl(cpssp->port_phone, 
							cpssp,
							SIG_TELE_HANGUP);
			}
		}
		break;

	case STATE_CONNECTED:
		/* forward data to slirp */
		slirp_send(cpssp, data);
		break;

	default:
		faum_log(FAUM_LOG_WARNING, "node-isp", "", 
			"received data in wrong state.\n");
		break;
	}
}

static void
isp_recv_ctrl(void *_cpssp, enum sig_telephone_protocol event)
{
	struct cpssp *cpssp = (struct cpssp*)_cpssp;
	if (! isp_get_power_state(cpssp)) {
		return;
	}
	
	switch (cpssp->isp_state) {
	case STATE_CONNECTED:
		isp_handle_conn_event(cpssp, event);
		break;

	case STATE_OFFLINE:
		isp_handle_off_event(cpssp, event);
		break;

	case STATE_AUTH:
		isp_handle_auth_event(cpssp, event);
		break;

	}
}

static void
isp_recv_dial(void *s, uint32_t number)
{
	faum_log(FAUM_LOG_WARNING, "node-isp", "", 
			"received invalid dial event.\n");
}

static void
isp_power_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	cpssp->state_switch = val;

	if (! val) {
		/* on -> off */
		isp_reset(cpssp);
	}
}

void 
telephone_isp_init(
	unsigned int nr,
	struct sig_telephone *port_phone,
	struct sig_boolean *port_switch
)
{
	static const struct sig_telephone_funcs tf = {
		.recv_data = isp_recv_data,
		.recv_ctrl = isp_recv_ctrl,
		.recv_dial = isp_recv_dial
	};
	static struct sig_boolean_funcs tpf = {
		.set = isp_power_set
	};
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	cpssp->port_phone = port_phone;

	sig_telephone_connect(port_phone, cpssp, &tf);
	sig_boolean_connect_in(port_switch, cpssp, &tpf);

	cpssp->isp_state = STATE_OFFLINE;
	cpssp->child_pid = 0;
	cpssp->child_fd = -1;
}

void
telephone_isp_create(unsigned int nr, const char *name)
{
	struct cpssp *cpssp;

	shm_create(COMP, nr, sizeof(*cpssp));
	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	auth_create(cpssp);

	shm_unmap(cpssp, sizeof(*cpssp));
}

void 
telephone_isp_destroy(unsigned int nr)
{
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	kill_slirp(cpssp);

	shm_unmap(cpssp, sizeof(*cpssp));
	shm_destroy(COMP, nr);
}
