/*
 * $Id: serial_modem.c,v 1.20 2009-01-28 12:59:22 potyra Exp $
 *
 *  Implementation of the serial modem, main part.
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdbool.h>
#include <stdlib.h>
#include <unistd.h>

#include "glue-log.h"
#include "glue-main.h"
#include "glue-shm.h"

#include "serial_modem.h"

#define COMP "serial_modem"

#define DEFINITIONS
#include "serial_modem_fsm.c"
#undef DEFINITIONS

/* numerical result codes */
#define RESULT_OK		0
#define RESULT_CONNECT		1
#define RESULT_RING		2
#define RESULT_NO_CARRIER	3
#define RESULT_ERROR		4
#define RESULT_CONNECT_1200	5
#define RESULT_NO_DIALTONE	6
#define RESULT_BUSY		7
#define RESULT_NO_ANSWER	8
#define RESULT_CONNECT_2400	10
#define RESULT_CONNECT_4800	11
#define RESULT_CONNECT_9600	12
#define RESULT_CONNECT_14400	13
#define RESULT_CONNECT_19200	14
#define RESULT_CONNECT_57600	18
#define RESULT_CONNECT_1200_75	22
#define RESULT_CONNECT_75_1200	23
#define RESULT_CONNECT_7200	24
#define RESULT_CONNECT_12000	25
#define RESULT_CONNECT_38400	28
#define RESULT_CONNECT_115200	31
#define RESULT_CONNECT_33333	33
#define RESULT_CONNECT_37333	34
#define RESULT_CONNECT_41333	35
#define RESULT_CONNECT_42666	36
#define RESULT_CONNECT_44000	37
#define RESULT_CONNECT_45333	38
#define RESULT_CONNECT_46666	39
#define RESULT_CONNECT_48000	42
#define RESULT_CONNECT_49333	43
#define RESULT_RINGBACK		45
#define RESULT_CONNECT_50666	53
#define RESULT_CONNECT_52000	54
#define RESULT_CONNECT_53333	55
#define RESULT_CONNECT_54666	56
#define RESULT_CONNECT_56000	57
#define RESULT_CONNECT_57333	58
#define RESULT_CONNECT_16800	59
#define RESULT_CONNECT_21600	61
#define RESULT_CONNECT_24000	62
#define RESULT_CONNECT_26400	63
#define RESULT_CONNECT_28800	64
#define RESULT_CONNECT_31200	65
#define RESULT_CONNECT_33600	66


struct cpssp {
	/* Config */

	/* Ports */
	struct sig_boolean *port_switch;
	struct sig_serial *port_serial;
	struct sig_telephone *port_phone;
	struct sig_boolean *port_opt_online_led;
	struct sig_boolean *port_opt_rxd_led;
	struct sig_boolean *port_opt_txd_led;

	/* Signals */

	/* State */
	unsigned int state_power;

	/* architecture FSM */
	#define NAME 	fsm
	#define STATE
	#include "serial_modem_fsm.c"
	#undef STATE
	#undef NAME
};

static struct cpssp *__cpssp;

static void
modem_opt_online_led_set(unsigned int val)
{
	struct cpssp *cpssp = __cpssp; /* FIXME */

	sig_boolean_set(cpssp->port_opt_online_led, cpssp, val);
}

static void
modem_opt_rxd_led_set(unsigned int val)
{
	struct cpssp *cpssp = __cpssp; /* FIXME */

	sig_boolean_set(cpssp->port_opt_rxd_led, cpssp, val);
}

static void
modem_opt_txd_led_set(unsigned int val)
{
	struct cpssp *cpssp = __cpssp; /* FIXME */

	sig_boolean_set(cpssp->port_opt_txd_led, cpssp, val);
}

static void
modem_phone_send_data(unsigned char data)
{
	struct cpssp *cpssp = __cpssp; /* FIXME */

	sig_telephone_send_data(cpssp->port_phone, cpssp, data);
}

static void
modem_phone_send_ctrl(enum sig_telephone_protocol ctrl)
{
	struct cpssp *cpssp = __cpssp; /* FIXME */

	sig_telephone_send_ctrl(cpssp->port_phone, cpssp, ctrl);
}

static void
modem_phone_dial(uint32_t number)
{
	struct cpssp *cpssp = __cpssp; /* FIXME */

	sig_telephone_dial(cpssp->port_phone, cpssp, number);
}

static unsigned int
serial_modem_power_state(void)
{
	return __cpssp->state_power;
}

static void 
modem_serial_push_data(const char c) 
{
	struct cpssp *cpssp = __cpssp; /* FIXME */

	sig_serial_send(cpssp->port_serial, cpssp, c);
}

/** reset complete modem (forward declaration, see below) 
 * @param cpssp config instance
 */
static void
serial_modem_reset(struct cpssp *cpssp);


#define NAME 		fsm
#define NAME_(x)	fsm_ ## x
#define BEHAVIOR
#include "serial_modem_fsm.c"
#undef BEHAVIOR
#undef NAME_
#undef NAME

#include "serial_modem_dialout.c"

static void
serial_modem_reset(struct cpssp *cpssp)
{
	fsm_reset(cpssp);
	modem_dialout_reset(cpssp);
}


static void
modem_recv(void *_cpssp, uint8_t c)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->state_power) {
		/* put data on bus... always 1st to fsm */
		/* FIXME fsm should be the one to forward to dialout */
		modem_fsm_push_char(cpssp, c);
		modem_dialout_push_char(c);
	}
}

static void
modem_power_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->state_power = val;

	serial_modem_reset(cpssp);
}

void
serial_modem_init(
	unsigned int nr,
	struct sig_boolean *port_switch,
	struct sig_serial *port_serial,
	struct sig_telephone *port_phone,
	struct sig_boolean *port_opt_online_led,
	struct sig_boolean *port_opt_rxd_led,
	struct sig_boolean *port_opt_txd_led
)
{
	static const struct sig_serial_funcs serial_funcs = {
		.recv = modem_recv
	};
	static struct sig_boolean_funcs switch_funcs = {
		.set = modem_power_set
	};
	static struct sig_telephone_funcs phone_funcs = {
		.recv_data = dialout_interrupt_data,
		.recv_ctrl = dialout_interrupt_ctrl,
		.recv_dial = dialout_interrupt_dial
	};
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);
	__cpssp = cpssp;

	/* Call */
	cpssp->port_serial = port_serial;
	sig_serial_connect(port_serial, cpssp, &serial_funcs);

	cpssp->port_phone = port_phone;
	sig_telephone_connect(port_phone, cpssp, &phone_funcs);

	/* Out */
	cpssp->port_opt_online_led = port_opt_online_led;
	sig_boolean_connect_out(port_opt_online_led, cpssp, 0);

	cpssp->port_opt_rxd_led = port_opt_rxd_led;
	sig_boolean_connect_out(port_opt_rxd_led, cpssp, 0);

	cpssp->port_opt_txd_led = port_opt_txd_led;
	sig_boolean_connect_out(port_opt_txd_led, cpssp, 0);

	/* In */
	sig_boolean_connect_in(port_switch, cpssp, &switch_funcs);
}

void
serial_modem_exit(unsigned int nr)
{
	modem_dialout_exit();
}

void
serial_modem_create(unsigned int nr, const char *name)
{
	struct cpssp *cpssp;

	shm_create(COMP, nr, sizeof(*cpssp));
	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	modem_fsm_create(cpssp);

	shm_unmap(cpssp, sizeof(*cpssp));
}

void
serial_modem_destroy(unsigned int nr)
{
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	shm_unmap(cpssp, sizeof(*cpssp));
	shm_destroy(COMP, nr);
}
