/* $Id: isa_gen_idectrl.c,v 1.22 2009-02-24 15:12:26 vrsieh Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include "fixme.h"
#include <stdio.h>
#include <stdlib.h>

#include "glue-shm.h"

#include "isa_gen_idectrl.h"

#define COMP "isa_gen_idectrl"

struct cpssp {
	/*
	 * Config
	 */
	unsigned int irq;
	unsigned int port_0_7;
	unsigned int port_8;

	/*
	 * Signals
	 */
	struct sig_boolean *sig_p5V;
	struct sig_boolean *sig_n_reset;
	struct sig_isa_bus_main *sig_isa_bus;
	struct sig_boolean_or *sig_isa_bus_irq;
	struct sig_ide_bus *sig_ide_bus;

	/*
	 * State
	 */
};

/*
 * IDE Bus Interface
 */
static void
isa_ide_controller_irq_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	sig_boolean_or_set(cpssp->sig_isa_bus_irq, cpssp, val);
}

static __attribute__((__noreturn__)) void
isa_ide_controller_dmarq_set(void *_cpssp, unsigned int val)
{
	assert(0);	/* No DMA capable device. */
}

/*
 * ISA Bus Interface
 */
static int
isa_ide_controller_inb(void *_cpssp, unsigned char *valp, unsigned short port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	uint16_t val16;

	if (cpssp->port_0_7 < port && port < cpssp->port_0_7 + 8) {
		sig_ide_bus_inw(cpssp->sig_ide_bus,
				port - cpssp->port_0_7, &val16);
		*valp = val16;
		return 0;
	} else if (cpssp->port_8 == port) {
		sig_ide_bus_inw(cpssp->sig_ide_bus,
				port - cpssp->port_8 + 8, &val16);
		*valp = val16;
		return 0;
	}

	return -1;
}

static int
isa_ide_controller_inw(void *_cpssp, unsigned short *valp, unsigned short port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->port_0_7 == port) {
		sig_ide_bus_inw(cpssp->sig_ide_bus, 0, valp);
		return 0;
	}

	return -1;
}

static int
isa_ide_controller_outb(void *_cpssp, unsigned char val, unsigned short port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->port_0_7 < port && port < cpssp->port_0_7 + 8) {
		sig_ide_bus_outw(cpssp->sig_ide_bus,
				port - cpssp->port_0_7, val);
		return 0;
	} else if (cpssp->port_8 == port) {
		sig_ide_bus_outw(cpssp->sig_ide_bus,
				port - cpssp->port_8 + 8, val);
		return 0;
	}

	return -1;
}

static int
isa_ide_controller_outw(void *_cpssp, unsigned short val, unsigned short port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->port_0_7 == port) {
		sig_ide_bus_outw(cpssp->sig_ide_bus, 0, val);
		return 0;
	}

	return -1;
}

void
isa_gen_idectrl_init(
	unsigned int nr,
	struct sig_isa_bus *port_isa,
	struct sig_ide_bus *port_ide
)
{
	static const struct sig_isa_bus_main_funcs isa_bus_funcs = {
		.inb = isa_ide_controller_inb,
		.inw = isa_ide_controller_inw,
		.outb = isa_ide_controller_outb,
		.outw = isa_ide_controller_outw,
	};
	static const struct sig_ide_bus_controller_funcs ide_bus_funcs = {
		.irq = isa_ide_controller_irq_set,
		.dmarq_set = isa_ide_controller_dmarq_set,
	};
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	/*
	 * Signals
	 */
	cpssp->sig_p5V = port_isa->p5V;
	/* Connect to sig_p5V - FIXME */
	cpssp->sig_n_reset = port_isa->n_reset;
	/* Connect to sig_n_reset - FIXME */

	cpssp->sig_isa_bus = port_isa->main;
	sig_isa_bus_main_connect(port_isa->main, cpssp, &isa_bus_funcs);
	switch (cpssp->irq) {
	case 0x9: cpssp->sig_isa_bus_irq = port_isa->int9; break;
	case 0xa: cpssp->sig_isa_bus_irq = port_isa->int10; break;
	case 0xb: cpssp->sig_isa_bus_irq = port_isa->int11; break;
	case 0xc: cpssp->sig_isa_bus_irq = port_isa->int12; break;
	default: assert(0); /* Shouldn't happen. */
	}
	sig_boolean_or_connect_out(cpssp->sig_isa_bus_irq, cpssp, 0);

	cpssp->sig_ide_bus = port_ide;
	sig_ide_bus_connect_controller(port_ide, cpssp, &ide_bus_funcs);

	/*
	 * State
	 */
	/* Nothing... */
}

void
isa_gen_idectrl_create(
	unsigned int nr,
	const char *name,
	const char *irq,
	const char *ioaddr_1,
	const char *ioaddr_8
)
{
	struct cpssp *cpssp;

	shm_create(COMP, nr, sizeof(*cpssp));
	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	/* Get Config */
	cpssp->irq = strtoul(irq, NULL, 0);
	cpssp->port_0_7 = strtoul(ioaddr_8, NULL, 0);
	cpssp->port_8 = strtoul(ioaddr_1, NULL, 0);

	shm_unmap(cpssp, sizeof(*cpssp));
}

void
isa_gen_idectrl_destroy(unsigned int nr)
{
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	shm_unmap(cpssp, sizeof(*cpssp));
	shm_destroy(COMP, nr);
}
