/* $Id: cpu_intel_80386_interpreter.c,v 1.10 2009-01-28 12:59:19 potyra Exp $ 
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <inttypes.h>
#include <stdio.h>

#include "glue-main.h"
#include "glue-shm.h"

#include "cpu_intel_80386_interpreter.h"

#define COMP "cpu_intel_80386_interpreter"

struct cpssp {
	/* Config */

	/* Ports */
	struct sig_host_bus_main *host_bus_main;

	/* Signals */

	/* State */
	int state_power;
	int state_n_reset;

	/* Processes */
	struct process process;
};

static struct cpssp *css;

static void
cpu_mr(uint32_t addr, unsigned int bs, uint32_t *valp)
{
	*valp = -1;
	if (sig_host_bus_mr(css->host_bus_main, css, 0, addr, bs, valp) != 0) {
		sig_host_bus_type_addr(css->host_bus_main, css,
				SIG_HOST_BUS_MR, addr);
		/* delay... */
		sig_host_bus_read_data(css->host_bus_main, css,
				bs, valp);
	}
}

static void
cpu_mw(uint32_t addr, unsigned int bs, uint32_t val)
{
	if (sig_host_bus_mw(css->host_bus_main, css, 0, addr, bs, val) != 0) {
		sig_host_bus_type_addr(css->host_bus_main, css,
				SIG_HOST_BUS_MW, addr);
		/* delay... */
		sig_host_bus_write_data(css->host_bus_main, css,
				bs, val);
	}
}

uint8_t
load_uint8_internal(uint32_t addr)
{
	uint32_t val32;
	uint8_t val8;

	cpu_mr(addr & ~3, 1 << (addr & 3), &val32);
	val8 = (val32 >> ((addr & 3) * 8)) & 0xff;

	return val8;
}

uint16_t
load_uint16_internal(uint32_t addr)
{
	uint16_t val16;
	uint32_t val32;

	if ((addr & 3) == 3) {
		val16 = load_uint8_internal(addr)
			| (load_uint8_internal(addr + 1) << 8);
	} else {
		cpu_mr(addr & ~3, 3 << (addr & 3), &val32);
		val16 = (val32 >> ((addr & 3) * 8)) & 0xffff;
	}

	return val16;
}

uint32_t
load_uint32_internal(uint32_t addr)
{
	uint32_t value;

	if (addr & 3) {
		uint32_t val0;
		uint32_t val1;

		cpu_mr(addr & ~3, (0xf << (addr & 3)) & 0xf, &val0);
		cpu_mr((addr & ~3) + 4, 0xf >> (4 - (addr & 3)), &val1);

		value = (val0 >> ((addr & 3) * 8))
			| (val1 << ((4 - (addr & 3)) * 8));

	} else {
		cpu_mr(addr & ~3, 0xf, &value);
	}

	return value;
}

void
store_uint8_internal(uint8_t value, uint32_t addr)
{
	uint32_t val32;

	val32 = value << ((addr & 3) * 8);
	cpu_mw(addr & ~3, 1 << (addr & 3), val32);
}

void
store_uint16_internal(uint16_t value, uint32_t addr)
{
	uint32_t val32;

	if ((addr & 3) == 3) {
                unsigned char value0 = (value >> 0) & 0xff;
                unsigned char value8 = (value >> 8) & 0xff;

                store_uint8_internal(value0, addr + 0);
                store_uint8_internal(value8, addr + 1);

        } else {
                val32 = value << ((addr & 3) * 8);
                cpu_mw(addr & ~3, 3 << (addr & 3), val32);
        }
}

void
store_uint32_internal(uint32_t value, uint32_t addr)
{
        if (addr & 3) {
                uint32_t val0;
                uint32_t val1;

                val0 = value << ((addr & 3) * 8);
                val1 = value >> ((4 - (addr & 3)) * 8);

                cpu_mw(addr & ~3, (0xf << (addr & 3)) & 0xf, val0);
                cpu_mw((addr & ~3) + 4, 0xf >> (4 - (addr & 3)), val1);

        } else {
                cpu_mw(addr & ~3, 0xf, value);
        }
}

static void
cpu_ior(uint32_t port, unsigned int bs, uint32_t *valp)
{
	*valp = -1;
	if (sig_host_bus_ior(css->host_bus_main, css, port, bs, valp) != 0) {
		sig_host_bus_type_addr(css->host_bus_main, css,
				SIG_HOST_BUS_IOR, port);
		/* delay... */
		sig_host_bus_read_data(css->host_bus_main, css,
				bs, valp);
	}
}

static void
cpu_iow(uint32_t port, unsigned int bs, uint32_t val)
{
	if (sig_host_bus_iow(css->host_bus_main, css, port, bs, val) != 0) {
		sig_host_bus_type_addr(css->host_bus_main, css,
				SIG_HOST_BUS_IOW, port);
		/* delay... */
		sig_host_bus_write_data(css->host_bus_main, css,
				bs, val);
	}
}

uint8_t
ior_uint8(uint32_t port)
{
	uint32_t val32;
	uint8_t val8;

	cpu_ior(port & ~3, 1 << (port & 3), &val32);
	val8 = (val32 >> ((port & 3) * 8)) & 0xff;

	return val8;
}

uint16_t
ior_uint16(uint32_t port)
{
	uint16_t val16;
	uint32_t val32;

	if ((port & 3) == 3) {
		val16 = ior_uint8(port)
			| (ior_uint8(port + 1) << 8);
	} else {
		cpu_ior(port & ~3, 3 << (port & 3), &val32);
		val16 = (val32 >> ((port & 3) * 8)) & 0xffff;
	}

	return val16;
}

uint32_t
ior_uint32(uint32_t port)
{
	uint32_t value;

	if (port & 3) {
		uint32_t val0;
		uint32_t val1;

		cpu_ior(port & ~3, (0xf << (port & 3)) & 0xf, &val0);
		cpu_ior((port & ~3) + 4, 0xf >> (4 - (port & 3)), &val1);

		value = (val0 >> ((port & 3) * 8))
			| (val1 << ((4 - (port & 3)) * 8));

	} else {
		cpu_ior(port & ~3, 0xf, &value);
	}

	return value;
}

void
iow_uint8(uint8_t value, uint32_t port)
{
	uint32_t val32;

	if (port == 0xffff) {
		/* System BIOS / VGA BIOS output port. */
		fprintf(stderr, "%c", value);
		return;
	}

	val32 = value << ((port & 3) * 8);
	cpu_iow(port & ~3, 1 << (port & 3), val32);
}

void
iow_uint16(uint16_t value, uint32_t port)
{
	uint32_t val32;

	if ((port & 3) == 3) {
                unsigned char value0 = (value >> 0) & 0xff;
                unsigned char value8 = (value >> 8) & 0xff;

                iow_uint8(value0, port + 0);
                iow_uint8(value8, port + 1);

        } else {
                val32 = value << ((port & 3) * 8);
                cpu_iow(port & ~3, 3 << (port & 3), val32);
        }
}

void
iow_uint32(uint32_t value, uint32_t port)
{
        if (port & 3) {
                uint32_t val0;
                uint32_t val1;

                val0 = value << ((port & 3) * 8);
                val1 = value >> ((4 - (port & 3)) * 8);

                cpu_iow(port & ~3, (0xf << (port & 3)) & 0xf, val0);
                cpu_iow((port & ~3) + 4, 0xf >> (4 - (port & 3)), val1);

        } else {
                cpu_iow(port & ~3, 0xf, value);
        }
}

uint8_t
irq_get(void)
{
	uint8_t vector;
	int ret;

	fprintf(stderr, "%s called...\n", __FUNCTION__);

	ret = sig_host_bus_inta_addr(css->host_bus_main, css);

	assert(!ret);

	ret = sig_host_bus_inta_data(css->host_bus_main, css, &vector);

	assert(!ret);

	fprintf(stderr, "%s returned: %u\n", __FUNCTION__, vector);

	return vector;
}

#define INTERPRETER 1
#include "chip_intel_80386_interpreter.c"
#undef INTERPRETER

void __attribute__((__noreturn__))
step(void *_css)
{
	sched_to_scheduler();
again:	;
	while (css->process.inst_cnt < css->process.inst_limit) {
		if (! css->state_power) {
			if (css->process.inst_cnt < css->process.inst_limit) {
				css->process.inst_cnt = css->process.inst_limit;
			}

		} else if (css->state_n_reset != 3) {
			reset();
			css->state_n_reset |= 2;
			if (css->process.inst_cnt < css->process.inst_limit) {
				css->process.inst_cnt = css->process.inst_limit;
			}

		} else {
			handle_instruction();
			css->process.inst_cnt += 1;
		}
	}

        sched_to_scheduler();

        goto again;
}

static void
power_set(void *_css, unsigned int val)
{
	css->state_power = val;
}

static void
n_reset_set(void *_css, unsigned int n_val)
{
	if (n_val) {
		/* Reset gone. */
		css->state_n_reset |= 1;
	} else {
		/* New reset. */
		css->state_n_reset = 0;
	}
}

static void
_irq_set(void *_css, unsigned int val)
{
	fprintf(stderr, "%s(%u) called...\n", __FUNCTION__, val);
	irq_set(val);
}

static void
_nmi_set(void *_css, unsigned int val)
{
	nmi_set(val);
}

void
cpu_intel_80386_interpreter_init(
	unsigned int nr,
	struct sig_host_bus *port_conn
)
{
	static const struct sig_boolean_or_funcs irq_func = {
		.set = _irq_set,
	};
	static const struct sig_boolean_or_funcs nmi_func = {
		.set = _nmi_set,
	};
	static const struct sig_boolean_funcs power_funcs = {
		.set = power_set,
	};
	static const struct sig_boolean_funcs n_reset_funcs = {
		.set = n_reset_set,
	};

	css = shm_map(COMP, nr, sizeof(*css), 0);

	css->host_bus_main = port_conn->main;

	sig_boolean_or_connect_in(port_conn->lint0, css, &irq_func);
	sig_boolean_or_connect_in(port_conn->lint1, css, &nmi_func);
	sig_boolean_connect_in(port_conn->power, css, &power_funcs);
	sig_boolean_connect_in(port_conn->n_reset, css, &n_reset_funcs);

	sched_process_init(&css->process, step, css);
}

void
cpu_intel_80386_interpreter_create(unsigned int nr, const char *name)
{
	struct cpssp *cpssp;

	assert(nr == 0);

	shm_create(COMP, nr, sizeof(*cpssp));
	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	/* Running a 20MHz CPU - FIXME */
	cpssp->process.inst_hz = 20*1000*1000;

	shm_unmap(cpssp, sizeof(*cpssp));
}

void
cpu_intel_80386_interpreter_destroy(unsigned int nr)
{
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	/* FILL ME */

	shm_unmap(cpssp, sizeof(*cpssp));
	shm_destroy(COMP, nr);
}
