/* $Id: glue-shm.c,v 1.18 2009-01-27 17:06:40 potyra Exp $ 
 *
 * Copyright (C) 2006-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <sys/types.h>
#include <sys/mman.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "glue-main.h"
#include "glue-shm.h"


static void
shm_name(char *tmpname, const char *name, int nr)
{
	if (nr == -1) {
		sprintf(tmpname, "%s/%s.shm", basedir, name);
	} else {
		sprintf(tmpname, "%s/%s-%d.shm", basedir, name, nr);
	}
}

void
shm_fd(const char *name, int nr, int *fdp, off_t *offsetp)
{
	char tmpname[256];

	shm_name(tmpname, name, nr);

	*fdp = open(tmpname, O_RDWR, 0);
	*offsetp = (off_t) 0;
}

void *
shm_map(const char *name, int nr, unsigned int len, int code)
{
	char tmpname[256];
	int fd;
	int prot;
	int map;
	void *retval;
	int ret;

	/* Round to page boundary. */
	len = (len + 0xfff) & 0xfffff000;

	shm_name(tmpname, name, nr);

	fd = open(tmpname, O_RDWR);
	if (fd < 0) {
		fprintf(stderr, "failed to open shm file %s: %s\n",
				tmpname, strerror(errno));
		exit(1);
	}

	if (code) {
		prot = PROT_READ | PROT_WRITE | PROT_EXEC;
	} else {
		prot = PROT_READ | PROT_WRITE;
	}
#ifdef MAP_32BIT
	if (code) {
		map = MAP_SHARED | MAP_32BIT;
	} else {
		map = MAP_SHARED;
	}
#else
	map = MAP_SHARED;
#endif

	if (len) {
		retval = mmap((char *) 0, len, prot, map, fd, 0);
		assert(retval != (void *) -1);
	} else {
		retval = NULL;
	}

	ret = close(fd);
	assert(0 <= ret);

	return retval;
}

int
shm_unmap(void *addr, unsigned int len)
{
	/* Round to page boundary. */
	len = (len + 0xfff) & 0xfffff000;

	return munmap(addr, len);
}

int
shm_create(const char *name, int nr, unsigned long len)
{
	char tmpname[256];
	int fd;
	unsigned long addr;
	int ret;

	time_stop();

	shm_name(tmpname, name, nr);

	fd = open(tmpname, O_RDWR | O_CREAT | O_EXCL, 0700);
	if (fd < 0) {
		fprintf(stderr, "%s: %s: can't create: %s.\n", "FAUmachine",
				tmpname, strerror(errno));
		exit(1);
	}

	for (addr = 0; addr < len; addr += 4096) {
		static char zero[4096];

		ret = write(fd, zero, 4096);
		if (ret < 0) {
			fprintf(stderr,
					"%s: %s: write %ld bytes: %s.\n",
					"FAUmachine",
					tmpname, len, strerror(errno));
			exit(1);
		}
		if (ret < 4096) {
			fprintf(stderr, "%s: %s: write: %s.\n",
					"FAUmachine", tmpname,
					"Short write");
			exit(1);
		}
		assert(ret == 4096);
	}

	ret = fsync(fd);
	assert(0 <= ret);

	ret = close(fd);
	assert(0 <= ret);

	time_cont();

	return 0;
}

void
shm_destroy(const char *name, int nr)
{
	char tmpname[256];

	shm_name(tmpname, name, nr);

	time_stop();

	unlink(tmpname);

	time_cont();
}
