/* $Id: BuiltinSymbolTable.hpp 4552 2009-08-21 10:05:54Z potyra $
 *
 * BuiltinSymbolTable: SymbolTable preloaded with symbols from std.standard.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */


#ifndef __BUILTIN_SYMBOL_TABLE_HPP_INCLUDED
#define __BUILTIN_SYMBOL_TABLE_HPP_INCLUDED

#include "frontend/misc/SymbolTable.hpp"

namespace ast {

/** boolean values as represented in the builtin symbol table
 *  note: true is 0 in vhdl
 */
enum vhdl_bool_vals {
	/** integer value of boolean false in VHDL */
	VHDL_FALSE = 0,
	/** integer value of boolean true in VHDL */
	VHDL_TRUE = 1,
};

//! the SymbolTable with predefined internal symbols.
/** This class predefines all symbols from std.standard.
 *  (or at least to the extent the current implementation for the framework
 *   allows).
 */
class BuiltinSymbolTable : public SymbolTable {
public:
	//! c'tor
	BuiltinSymbolTable();

	/** create a new region and push it onto the stack. should be
	 *  called when entering a nested declarative region.
	 *
	 *  If the regionStack has the size 1, it must be a library unit. 
	 *  Hence the library std and the package standard will be made 
	 *  visible.
	 */

	virtual void pushNewRegion(void);

private: 
	/** register all builtins of std.standard.
	 *  Will call addStandardStd to add the actual
	 *  symbols.
	 *
	 */
	void registerBuiltinSymbols(void);

	/** add all symbols of std.standard. SymbolTable needs to 
	 *  have entered the package standard of library std already.
	 *
	 *  @param standardSyms (empty) list to which all declarations
	 *         should get attached to (to be correctly stored in the
	 *         AST).
	 */
	void addStandardStd(std::list<SymbolDeclaration*> &standardSyms);

	/** add time to std.standard.
	 *  @param standardSyms list of package symbols of standard
	 */
	void addStdStandardTime(std::list<SymbolDeclaration*> &standardSyms);

	/** register a physical type unit in std.standard */
	void addPhysUnit(
		std::list<SymbolDeclaration*> &standardSyms,
		std::list<PhysicalTypeUnit*> &units,
		const char *name,
		universal_integer factor,
		const char *refName
	);

	/** add character enumeration literal c to Enumeration element list. 
	 *  @param elems list of enumeration elements 
	 *  @param c character literal
	 */
	static void 
	addEnumChar(std::list<FunctionDeclaration*> &elems, char c);

	/** add string enumeration literal s to Enumeration element list.
	 *  @param elems list of enumeration elements
	 *  @param c string literal.
	 */
	static void 
	addEnumStr(std::list<FunctionDeclaration*> &elems, const char *s);

	/** register the "foreign" attribute
	 *  @param stringT type declaration of "string"
	 *  @param standardSyms list of packages symbols of standard
	 */
	void
	addForeignAttribute(
		const TypeDeclaration *stringT,
		std::list<SymbolDeclaration*> &standardSyms);

	/** should we import std.standard when registering a LibUnit? */
	bool mayAddStdStandard;
};

}; /* namespace ast */

#endif /* __BUILTIN_SYMBOL_TABLE_HPP_INCLUDED */
