/*
   FALCON - The Falcon Programming Language.
   FILE: vmmaps.hm_currentModule

   Map items used in VM and related stuff
   -------------------------------------------------------------------
   Author: Giancarlo Niccolai
   Begin: mer ott 20 2004

   -------------------------------------------------------------------
   (C) Copyright 2004: the FALCON developers (see list in AUTHORS file)

   See LICENSE file for licensing details.
*/

/** \file
   Map items used in VM and related stuff.
*/

#ifndef flc_vmmaps_H
#define flc_vmmaps_H

#include <falcon/setup.h>
#include <falcon/types.h>
#include <falcon/genericvector.h>
#include <falcon/genericmap.h>
#include <falcon/itemtraits.h>
#include <falcon/symbol.h>
#include <falcon/string.h>
#include <falcon/module.h>
#include <falcon/basealloc.h>

namespace Falcon {

class Symbol;
class Item;


class FALCON_DYN_CLASS ItemVector: public GenericVector
{
   const ItemTraits s_it;

public:
   ItemVector( uint32 prealloc=0 ):
      GenericVector()
  {
      init( &s_it, prealloc );
  }

   Item &itemAt( uint32 pos ) const { return *(Item *) at( pos ); }
   Item *itemPtrAt( uint32 pos ) const { return (Item *) at( pos ); }
   void setItem( const Item &item, uint32 pos ) { set( const_cast<Item *>(&item), pos ); }
   Item &topItem() const { return *(Item *) top(); }
};

class FALCON_DYN_CLASS GlobalsVector: public GenericVector
{
   const VoidpTraits s_vpt;
public:
   GlobalsVector( uint32 prealloc=0 ):
      GenericVector()
   {
      init( &s_vpt, prealloc );
   }

   ItemVector &vat( uint32 pos ) const { return **(ItemVector **) at( pos ); }
   ItemVector *vpat( uint32 pos ) const { return *(ItemVector **) at( pos ); }
   ItemVector &topvp() const { return **(ItemVector **) top(); }
};


/** Instance of a live module entity.

   The VM sees modules as a closed, read-only entity. Mutable data in a module is actually
   held in a per-module map in the VM.

   This class helds a reference to a module known by the VM and to the variable data
   known in the module.

   A module can be entered only once in the VM, and it is uniquely identified by a name.

   This class acts also as a weak reference between live callable items and modules.
   When a live module is unlinked, the contents of this class are zeroed and
   every callable item referencing this module becomes a nil as isCallable()
   gets called.

   This object is garbageable; it gets referenced when it's in the module map and by
   items holding a callable in this module. When a module is unlinked, the LiveModule
   may be destroyed when it is not referenced in garbage anymore, or at VM termination.

   \note Although this is a Garbageable item, and as such, it resides in the memory pool,
   there isn't any need for precise accounting of related memory, as globals are allocated
   and destroyed when the module is linked or unlinked, and not (necessarily) when this
   item is collected. As the memory associated with this object is separately reclaimed
   by detachModule(), and it cannot possibly be reclaimed during a collection loop,
   there is no meaning in accounting it.
*/

class FALCON_DYN_CLASS LiveModule: public Garbageable
{
   Module *m_module;
   ItemVector m_globals;
   ItemVector m_wkitems;

public:
   LiveModule( VMachine *vm, Module *mod );
   ~LiveModule();

   const Module *module() const { return m_module; }
   const ItemVector &globals() const { return m_globals; }
   ItemVector &globals() { return m_globals; }

   const ItemVector &wkitems() const { return m_wkitems; }
   ItemVector &wkitems() { return m_wkitems; }

   /** Just a shortcut to the name of the module held by this LiveModule. */
   const String &name() const { return m_module->name(); }

   /** Just a shortcut to the source of the module held by this LiveModule. */
   const byte *code() const { return m_module->code(); }

   /** Disengage a module after a module unlink. */
   void detachModule();

   /** Is this module still alive?
       Short for this->module() != 0
       \return true if the module is still alive.
   */
   bool isAlive() const { return m_module != 0; }

   /** Return a module item given a global symbol name.
      This is an utility funtion retreiving an global item declared by the
      module that is referenced in this live data.
      \param symName the name of the global symbol for which this item must be found
      \return 0 if not found or a pointer to the item which is indicated by the symbol
   */
   Item *findModuleItem( const String &symName ) const;
};


class LiveModulePtrTraits: public ElementTraits
{
public:
	virtual uint32 memSize() const;
	virtual void init( void *itemZone ) const;
	virtual void copy( void *targetZone, const void *sourceZone ) const;
	virtual int compare( const void *first, const void *second ) const;
	virtual void destroy( void *item ) const;
   virtual bool owning() const;
};

/** Map of active modules in this VM.
   (const String *, LiveModule * )
*/
class FALCON_DYN_CLASS LiveModuleMap: public Map
{
public:
   LiveModuleMap();
};


/** Pair of the symbol and the module it is declared in.
   This is just a commodity class used to store the association between a certain symbol and the module
   it came from given the VM viewpoint (that is, the ID of the source module in the VM module list.

   The class may store either a global item (an item generated by an export request in the
   source module) or a well known item (an item generated via a setWKI request by the module),
   and it can never be both at the same time.
*/
class FALCON_DYN_CLASS SymModule: public BaseAlloc
{
   Symbol *m_symbol;
   LiveModule *m_lmod;

   Item *m_item;
   int32 m_wkiid;

public:
   /** Creates an exported Global Item. */
   SymModule( Item *itm, LiveModule *mod, Symbol *sym ):
      m_item( itm ),
      m_symbol( sym ),
      m_lmod( mod ),
      m_wkiid( -1 )
   {}

   /** Creates an exported Well Known Item. */
   SymModule( int32 wiid, LiveModule *mod, Symbol *sym ):
      m_item( 0 ),
      m_symbol( sym ),
      m_lmod( mod ),
      m_wkiid( wiid )
   {}

   /** Global item pointer.
      This pointers always points to a valid global variable in a vector inside LiveModule structure.
      As the global variable area never chages, the item pointer stays valid as long as the
      LiveModule in which it's stored is alive.
      \note If this SymModule refers to a WKI, the pointer is 0.
      \return a pointer to the referenced item.
   */
   Item *item() const { return m_item; }
   Symbol *symbol() const { return m_symbol; }
   uint32 symbolId() const { return m_symbol->itemId(); }
   LiveModule *liveModule() const { return m_lmod; }

    /** Well known item id.
      To find an item in the well known item array, it is necessary to use a local ID,
      as the array grows as the VM finds wki in the module.
      \note if the SymModule refers to a globally exported item, this id is -1
      \return the id of the item in the WKI array.
   */
   int32 wkiid() const { return m_wkiid; }
};


class SymModuleTraits: public ElementTraits
{
public:
	virtual uint32 memSize() const;
	virtual void init( void *itemZone ) const;
	virtual void copy( void *targetZone, const void *sourceZone ) const;
	virtual int compare( const void *first, const void *second ) const;
	virtual void destroy( void *item ) const;
   virtual bool owning() const;
};


namespace traits
{
   extern SymModuleTraits t_SymModule;
}

/** Map of symbol names and module where they are located.
   (const String *, SymModule )
*/
class FALCON_DYN_CLASS SymModuleMap: public Map
{
public:
   SymModuleMap();
};

}

#endif

/* end of vmmaps.h */
