# This file is part of the Falcon repository manager
# Copyright (C) 2005-2008 Dennis Kaarsemaker
# See the file named COPYING in the root of the source tree for license details
#
# falcontags.py - Specialized tags to be used in djangos template system

import falcon
from django import template
from django.utils.html import escape
import re

register = template.Library()

@register.tag
def api(parser, token):
    """Check template compatibility"""
    try:
        tag_name, api_string = token.split_contents()
    except ValueError:
        raise template.TemplateSyntaxError, _("%r tag requires a single argument") % token.contents.split()[0]
    try:
        api_major, api_minor = [int(x) for x in api_string.split('.')]
    except ValueError:
        raise template.TemplateSyntaxError, _("Template api should be 2 numbers, separated by a fullstop")
    if api_major != falcon.conf.template_api[0] or api_minor > falcon.conf.template_api[1]:
        falcon.util.error(_("The template '%s' was made for an incompatible version of falcon") % falcon.conf.template)
    if api_minor != falcon.conf.template_api[1]:
        falcon.util.warning(_("The template '%s' was made for an older version of falcon but is still compatible") % falcon.conf.template)
    falcon.conf.template_checked = True
    return ApiNode()

class ApiNode(template.Node):
    def render(self, context):
        return ''

@register.filter
def format_changelog(cl):
    """Nice HTML formatting for the debian/changelog format. Will parse the 
       contents of .changes files and extract and format the changelog in them"""
    if not cl:
        return ''
    cl = cl[cl.find('Changes:')+8:cl.find('Files:')]
    out = '<dl>'
    cl = escape(cl.strip())
    # Strip 'empty' lines
    cl = re.sub(' (  )*\.\n','',cl)
    # Split into versions
    versions = [x for x in re.split('(?:^|\n) (?=\S)', cl, re.DOTALL | re.M) if x.strip()]
    for v in versions:
        (ver, log) = v.split("\n",1)
        out += '<b>%s</b>' % ver
        # Join the lines of items
        log = re.sub(r"\n (?:  )*([^*\-+ ])",r"\1", log)
        # Magic :)
        for line in log.split("\n"):
            l = (len(line) - len(line.lstrip()) -1) / 2
            out += '<ul><li>' * l
            out += line.lstrip()[1:]
            out += '</li></ul>' * l
        return out.replace('</ul><ul>','').replace('</ul></li><li><ul>','')

@register.filter
def shortdesc(value):
    """Short version of a debian/control Description: field"""
    return escape(value).split('\n',1)[0]

@register.filter
def longdesc(value):
    """Long version of a debian/control Description: field"""
    try:
        return '<p>%s</p>' % escape(value).split('\n',1)[1].replace('\n .\n','</p><p>')
    except:
        return ''

# Only to be used during build
@register.tag
def highlight(parser, token):
    """Highlight code with pygments"""
    tag_name, syntax = token.split_contents()
    nodelist = parser.parse(('endhighlight',))
    parser.delete_first_token()
    return HighlightNode(syntax, nodelist)

class HighlightNode(template.Node):
    def __init__(self, syntax, code):
        self.syntax = syntax
        self.code = code
    def render(self, context):
        import pygments
        import pygments.lexers
        import pygments.formatters
        formatter=pygments.formatters.html.HtmlFormatter(linenos=False, encoding='utf-8')
        lexer=pygments.lexers.get_lexer_by_name(self.syntax)
        return pygments.highlight(self.code.render(context), lexer, formatter)
