/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <string.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "dialog.h"
#include "selwin.h"
#include "views.h"
#include "enum.h"
#include "replace.h"
#include "logging.h"

/**
 *	replace_object_button_activated - callback to replace the backing store for one object with another
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Replace button in the target selection dialog is activated.
 *	We invoke the evms_replace() API and report the outcome status.
 */
int replace_object_button_activated(struct menu_item *item)
{
	int rc;
	engine_handle_t source;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;

	source = GPOINTER_TO_UINT(dialog->user_data);
	rc = evms_replace(source, get_selected_handle(selwin->clist));
	update_status(_("Replace"), rc);

	if (rc == 0) {
		dialog->status = DLG_STATUS_CLOSING;
		refresh_views();
	}
	return 0;
}

/**
 *	can_replace_object - can a given object be replaced with some other object
 *	@handle: the object that we want to replace
 *
 *	This routine answers the question "Can the given object be replaced with some
 *	candidate object?". Since a call to evms_can_replace(handle, 0) simply answers
 *	the question "Can the given object be replaced?" is insufficient since it does
 *	not look whether there are likely candidates for the replace. Therefore, we
 *	make the additional checks ourselves.
 */
int can_replace_object(object_handle_t handle)
{
	int rc;
	
	rc = evms_can_replace(handle, 0);
	if (rc == 0) {
		handle_array_t *objects;
		
		rc = enum_topmost_objects(NULL, &objects);
		if (rc == 0) {
			rc = ENOENT;

			if (objects->count > 0) {
				int i = 0;
				
				while (i < objects->count && rc != 0) {
					rc = evms_can_replace(handle, objects->handle[i]);
					i++;
				}
			}
			evms_free(objects);
		}
	}
	return rc;
}

/**
 *	filter_replace_source - a clist_filter_func that keeps objects that can be replaced
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be replaced with any object.
 */
int filter_replace_source(object_handle_t handle, void *user_data)
{
	return can_replace_object(handle);
}

/**
 *	filter_replace_target - a clist_filter_func that keeps objects that can be replace targets
 *	@handle: the thing's handle
 *	@user_data: contain's handle of source object
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be the target of an object replace.
 */
int filter_replace_target(object_handle_t handle, void *user_data)
{
	return evms_can_replace(GPOINTER_TO_UINT(user_data), handle);
}

/**
 *	create_replace_target_dialog - create a selection window with targets that can replace source
 *	@source: the object to replace
 *
 *	This routine creates a selection window that is populates with replace targets for
 *	the given source object.
 */
struct dialog_window *create_replace_target_dialog(object_handle_t source)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Select Replace Target Object"),
					NULL, NULL,
					_("_Replace"),
					(menuitem_activate_cb)replace_object_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					GUINT_TO_POINTER(source));

	dialog = (struct dialog_window *)selwin;
	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Storage Object"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	clist_populate(selwin->clist, enum_topmost_objects, filter_replace_target,
			format_standard_item, NULL, GUINT_TO_POINTER(source));

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	return dialog;
}

/**
 *	select_replace_source_button_activated - callback invoked when source object selected to select target
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Next button is activated to display the target objects
 *	that can replace the selected source object.
 */
int select_replace_source_button_activated(struct menu_item *item)
{
	struct selwin *selwin = item->user_data;
	object_handle_t current_source, saved_source;
	struct dialog_window *dialog = item->user_data;

	current_source = get_selected_handle(selwin->clist);
	saved_source = GPOINTER_TO_UINT(dialog->user_data);

	if (current_source != saved_source) {
		GList *next_dialog;
		struct dialog_window *target_dialog;
		/*
		 * Since the selected source item has changed we need to delete any
		 * dialog that we may have built for the replace target selection.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		target_dialog = create_replace_target_dialog(current_source);
		dialog->user_data = GUINT_TO_POINTER(current_source);
		target_dialog->user_data = GUINT_TO_POINTER(current_source);
		append_dialog_to_list(target_dialog, dialog->list);
	}
	dialog->list->current = get_next_dialog(dialog);

	return 0;
}

/**
 *	show_replace_source_dialog - displays a selection window with objects that can be replaced
 *	@source: if non-zero, the source selection list is populated with the given object only
 *
 *	This routine creates a selection window, populates the selection list with objects that
 *	can be replaced.
 */
int show_replace_source_dialog(object_handle_t source)
{
	struct selwin *selwin;
	struct dialog_list dialogs;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Replace Source Object"),
					NULL, NULL, NULL,
					(menuitem_activate_cb)select_replace_source_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;
	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Storage Object"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	if (source == 0)
		clist_populate(selwin->clist, enum_data_objects, filter_replace_source,
				format_standard_item, NULL, NULL);
	else
		clist_populate_handle(selwin->clist, source, format_standard_item, NULL, NULL);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);

	init_dialog_list(&dialogs);
	append_dialog_to_list(dialog, &dialogs);
	process_dialog_list_events(&dialogs);
	delete_dialog_list(dialogs.list);

	return 0;
}

/**
 *	context_replace_menuitem_activated - replace object initiated from context menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to replace the the object selected with one of
 *	the targets we'll present the user.
 */
int context_replace_menuitem_activated(struct menu_item *item)
{
	return show_replace_source_dialog(GPOINTER_TO_UINT(item->user_data));
}

/**
 *	actions_replace_menuitem_activated - replace object initiated from Actions menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Replace Actions pulldown menu item. It
 *	creates a dialog window with a selection list of objects that can be
 *	replaced.
 */
int actions_replace_menuitem_activated(struct menu_item *item)
{
	return show_replace_source_dialog(0);
}
