/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <string.h>
#include <ncurses.h>
#include <panel.h>
#include <glib.h>
#include <frontend.h>

#include "common.h"
#include "dlist.h"
#include "window.h"
#include "menu.h"
#include "dialog.h"

/**
 *	set_dialog_delete_cb - change the delete event notification callback for a dialog
 *	@dialog: the dialog to update
 *	@delete_cb: the delete_cb to register
 *
 *	This routine sets the delete callback for dialog window used to notify the
 *	caller that the dialog is being deleted.
 */
inline void set_dialog_delete_cb(struct dialog_window *dialog, dialog_delete_cb delete_cb)
{
	dialog->delete_cb = delete_cb;
}

/**
 *	ok_button_activated - callback invoke when OK button activated
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the OK button in the message or help
 *	dialog window is activated.
 */
int ok_button_activated(struct menu_item *item)
{
	return 0;
}

/**
 *	get_newline_count - return count of newline characters found in a string
 *	@str: the string to search
 *
 *	This routine simply scans a string to keep track of the count of newline
 *	characters found in it and returns the count.
 */
inline int get_newline_count(char *str)
{
	int i = 0;

	while ((str = strchr(str, '\n')) != NULL) {
		i++;
		str++;
	}
	return i;
}

/**
 *	show_message_dialog - popup a simple message window with an OK button
 *	@title: the window title (can be NULL)
 *	@fmt: a format string. this could be followed by other arguments
 *
 *	This function creates a simple message window that contains a user
 *	supplied message and title. The window is dismissed by activating the
 *	[OK] button. The routine allows for a variable number of arguments
 *	following the format string that may be necessary to produce the final
 *	message.
 **/
void show_message_dialog(char *title, char *fmt, ...)
{
	char *msg, *titulo;
	WINDOW *win, *border_win;
	PANEL *panel, *border_panel;
	struct horizontal_menu *menu;
	int key = 0, width, height, min_menu_size;
	va_list args;

	va_start(args, fmt);
	msg = g_strdup_vprintf(fmt, args);
	va_end(args);

	if (title == NULL)
		titulo = g_strdup("");
	else
		titulo = g_strdup_printf(" %s ", title);
		
	min_menu_size = strlen(_("[OK]")) + 2;

	width = MIN(getmaxx(stdscr) - 6, strlen(msg) + 4);
	width = MAX(width, strlen(titulo) + 4);
	width = MAX(width, min_menu_size);

	height = MIN(getmaxy(stdscr) - 4, get_newline_count(msg) + 6);
	height = MAX(height, 10);
	
	panel = create_centered_popup_window(height, width, WHITE_BKGD);
	border_panel = (PANEL *)panel_userptr(panel);
	win = panel_window(panel);
	border_win = panel_window(border_panel);

	print_centered(border_win, 0, titulo);
	print_centered(win, 1, msg);

	menu = create_horizontal_menu(win, width - 4, getmaxy(win) - 2, 1, 1);
	pack_menu_item_at_end(menu, _("[OK]"), get_accelerator_char(_("[_OK]")),
					(menuitem_activate_cb)ok_button_activated, 
					NULL);

	draw_horizontal_menu(menu);

	g_free(msg);
	g_free(titulo);

	show_popup_window(panel);

	while (key != KEY_ESCAPE && key != KEY_ENTER) {
		key = panel_getch(panel);
		process_horizontal_menu_events(menu, &key);
	}

	delete_horizontal_menu(menu);
	delete_popup_window(panel);
}

/**
 *	prev_button_activated - callback invoked when Previous button activated
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Previous button in the dialog window
 *	is activated.
 */
int prev_button_activated(struct menu_item *item)
{
	struct dialog_window *dialog;

	dialog = item->user_data;
	if (dialog->list != NULL)
		dialog->list->current = get_previous_dialog(dialog);

	return 0;
}

/**
 *	close_window_button_activated - callback invoked to close a dialog
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when a button in the dialog window is
 *	activated that wants the window closed. It sets the window status flag
 *	to DLG_STATUS_CLOSING to indicate that we should stop processing events
 *	and delete the window.
 */
int close_window_button_activated(struct menu_item *item)
{
	struct dialog_window *dialog;

	dialog = item->user_data;
	dialog->status = DLG_STATUS_CLOSING;

	return 0;
}

/*
 *	cancel_button_activated - callback invoked when Cancel button activated
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Cancel button in the dialog window is
 *	activated. It sets the window status flag to DLG_STATUS_CLOSING
 *	to indicate that we should stop processing events and delete the window.
 *	We also display a message indicating the operation was cancelled.
 */
int cancel_button_activated(struct menu_item *item)
{
	close_window_button_activated(item);
	print_statusbar_text(_("The operation was cancelled."));
	return 0;
}

/**
 *	help_button_activated - callback invoked when Help button activated
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Help button in the dialog window
 *	is activated.
 */
static int help_button_activated(struct menu_item *item)
{
	return 0;
}

/**
 *	make_next_button_sensitive - make the "Next" button sensitive if necessary
 *	@dialog: the dialog window containing the next button
 *
 *	This routine checks to see if the "Next" button is sensitive. If it isn't,
 *	it makes it sensitive, gives it the focus and redraws the horizontal menu
 *	that contains it.
 */
void make_next_button_sensitive(struct dialog_window *dialog)
{
	if (is_menu_item_sensitive(dialog->next_button) == FALSE) {
		set_menu_item_sensitivity(dialog->next_button, TRUE);
		set_horizontal_menu_focus(dialog->menu, dialog->next_button);
		draw_horizontal_menu(dialog->menu);
	}
}

/**
 *	make_next_button_insensitive - make the "Next" button insensitive to input
 *	@dialog: the dialog window containing the next button
 *
 *	This routine checks to see if the "Next" button is insensitive. If it isn't,
 *	it makes it insensitive and redraw the horizontal menu that contains it.
 */
void make_next_button_insensitive(struct dialog_window *dialog)
{
	if (is_menu_item_sensitive(dialog->next_button) == TRUE) {
		set_menu_item_sensitivity(dialog->next_button, FALSE);
		draw_horizontal_menu(dialog->menu);
	}
}

/**
 *	get_next_and_prev_button_labels - generate labels and accelerator for "Next" and "Previous"
 *	@next_button_text: text for the "Next" button; includes accelerator; NULL if using defaults
 *	@prev_button_text: text fpr the "Previous" button; includes accel; NULL is using defaults
 *	@next_button_label: address of where to store the returned string for the Next button label
 *	@next_button_accel: address of where to store the accelerator character for the Next button
 *	@prev_button_label: address of where to store the returned string for the Previous button label
 *	@prev_button_accel: address of where to store the accelerator character for the Previous button
 *
 *	This routine generates the label text to use for the previous and next buttons in
 *	the dialog window. If either is NULL then the default text and accelerators are
 *	used. If text is given, then the accelerator is identified before the _ character
 *	that identifies the accelerator is stripped out to produce the label.
 */
void get_next_and_prev_button_labels(char *next_button_text, char *prev_button_text,
				char **next_button_label, int *next_button_accel,
				char **prev_button_label, int *prev_button_accel)
{
	char *tmp;

	if (next_button_text == NULL)
		next_button_text = _("_Next");

	if (prev_button_text == NULL)
		prev_button_text = _("_Previous");

	*next_button_accel = get_accelerator_char(next_button_text);
	*prev_button_accel = get_accelerator_char(prev_button_text);

	tmp = remove_accel_identifier(next_button_text);
	*next_button_label = g_strdup_printf("[%s]", tmp);
	g_free(tmp);

	tmp = remove_accel_identifier(prev_button_text);
	*prev_button_label = g_strdup_printf("[%s]", tmp);
	g_free(tmp);
}

/**
 *	init_dialog_window - initializes the dialog window base class properties
 *	@dialog: the address of the dialog_window structure to initialize
 *	@title: the window title
 *	@help_text: the text for the help window activated by the Help button
 *	@handler: the handler responsible for processing events for this dialog
 *	@show_func: the function responsible for drawing additional window elements
 *	@delete_func: the function responsible for cleaning up after a dialog
 *	@delete_cb: the user-supplied callback function to invoke before calling delete_func
 *	@next_button_text: the label text for the Next button or NULL for default
 *	@next_button_callback: the callback for when the Next button menuitem is activated
 *	@prev_button_text: the label text for the Previous button or NULL for default
 *	@prev_button_callback: the optional callback for when the Previous button is activated
 *	@user_data: misc stuff the caller wants associated with the dialog window
 *
 *	This routine takes the address of a dialog_window structure and creates
 *	the standard dialog window and horizontal/button menu and sets the values
 *	for other structure variables.
 */
void init_dialog_window(struct dialog_window *dialog,
			const char *title,
			const char *help_text,
			dialog_event_handler handler,
			dialog_show_func show_func,
			dialog_delete_func delete_func,
			dialog_delete_cb delete_cb,
			char *next_button_text,
			menuitem_activate_cb next_button_callback,
			char *prev_button_text,
			menuitem_activate_cb prev_button_callback,
			void *user_data)
{
	int next_button_accel, prev_button_accel;
	char *next_button_label, *prev_button_label;

	dialog->panel = create_centered_popup_window(getmaxy(stdscr) - 6,
						getmaxx(stdscr) - 6, WHITE_BKGD);
	dialog->win = panel_window(dialog->panel);

	dialog->menu = create_horizontal_menu(dialog->win, getmaxx(dialog->win) - 4, 
						getmaxy(dialog->win) - 2, 1, 1);
	dialog->handler = handler;
	dialog->show_func = show_func;
	dialog->delete_func = delete_func;
	dialog->delete_cb = delete_cb;
	dialog->user_data = user_data;
	dialog->status = DLG_STATUS_ACTIVE;

	get_next_and_prev_button_labels(next_button_text, prev_button_text, 
					&next_button_label, &next_button_accel,
					&prev_button_label, &prev_button_accel);

	dialog->help_button = pack_menu_item_at_start(dialog->menu, _("[Help]"),
				get_accelerator_char(_("[_Help]")),
				(menuitem_activate_cb)help_button_activated, 
				g_strdup(help_text));
	
	dialog->cancel_button = pack_menu_item_at_end(dialog->menu, _("[Cancel]"),
				get_accelerator_char(_("[_Cancel]")),
				(menuitem_activate_cb)cancel_button_activated, 
				dialog);

	if (prev_button_callback == NULL)
		prev_button_callback = (menuitem_activate_cb)prev_button_activated;

	dialog->prev_button = pack_menu_item_at_end(dialog->menu, prev_button_label,
				prev_button_accel,
				prev_button_callback, 
				dialog);

	dialog->next_button = pack_menu_item_at_end(dialog->menu, next_button_label,
				next_button_accel,
				next_button_callback,
				dialog);

	set_menu_item_sensitivity(dialog->next_button, FALSE);
	set_horizontal_menu_focus(dialog->menu, dialog->prev_button);

	if (title != NULL) {
		char *titulo;
		WINDOW *border_win;
		PANEL *border_panel;

		border_panel = (PANEL *)panel_userptr(dialog->panel);
		border_win = panel_window(border_panel);

		titulo = g_strdup_printf(" %s ", title);
		print_centered(border_win, 0, titulo);
		g_free(titulo);
	}

	g_free(next_button_label);
	g_free(prev_button_label);
}

/**
 *	create_dialog_window - create a dialog window instance
 *	@title: the window title
 *	@help_text: the text for the help window activated by the Help button
 *	@handler: the handler responsible for processing events for this dialog
 *	@show_func: the function responsible for drawing additional window elements
 *	@delete_func: the function responsible for cleaning up after a dialog
 *	@delete_cb: the user-supplied callback function to invoke before calling delete_func
 *	@next_button_text: the label text for the Next button or NULL for default
 *	@next_button_callback: the callback for when the Next button menuitem is activated
 *	@prev_button_text: the label text for the Previous button or NULL for default
 *	@prev_button_callback: the optional callback for when the Previous button is activated
 *	@user_data: misc stuff the caller wants associated with the dialog window
 *
 *	This routine allocates and initializes a dialog window instance.
 */
struct dialog_window *create_dialog_window(
			const char *title,
			const char *help_text,
			dialog_event_handler handler,
			dialog_show_func show_func,
			dialog_delete_func delete_func,
			dialog_delete_cb delete_cb,
			char *next_button_text,
			menuitem_activate_cb next_button_callback,
			char *prev_button_text,
			menuitem_activate_cb prev_button_callback,
			void *user_data)
{
	struct dialog_window *dialog;

	dialog = g_new0(struct dialog_window, 1);

	init_dialog_window(dialog, title, help_text, handler, show_func,
			delete_func, delete_cb, next_button_text, next_button_callback,
			prev_button_text, prev_button_callback, user_data);
	return dialog;
}

/**
 *	show_dialog_window - display/draw a standard dialog window
 *	@dialog: the dialog window
 *
 *	This routine simply calls show_popup_window which besides
 *	drawing the contents of the window, also unhides it if
 *	it was hidden and brings it to the top of the window stack.
 */
void show_dialog_window(struct dialog_window *dialog)
{
	show_popup_window(dialog->panel);

	if (dialog->show_func != NULL)
		dialog->show_func(dialog);

	draw_horizontal_menu(dialog->menu);
}

/**
 *	delete_dialog_window - free the resources for a dialog
 *	@dialog: address of dialog base class
 *	@not_used: reserved for future use
 *
 *	This routine is called to free any dialog subclass
 *	instance. The address given will be freed so be sure
 *	that it coincides with the start address of the dialog
 *	subclass.
 */
void delete_dialog_window(struct dialog_window *dialog, void *not_used)
{
	show_popup_window(dialog->panel);

	if (dialog->delete_cb != NULL)
		dialog->delete_cb(dialog);

	if (dialog->delete_func != NULL)
		dialog->delete_func(dialog);

	delete_popup_window(dialog->panel);
	g_free(dialog);
}

/**
 *	process_dialog_window_events - this is event handler for a dialog window
 *	@dialog: address of dialog window structure
 *	@key: an input value from the keyboard
 *
 *	This routine handles the standard processing for a dialog window instance.
 *	It first invokes the button bar handler and then whatever handler the
 *	dialog registered.
 */
int process_dialog_window_events(struct dialog_window *dialog, int key)
{
	int event_handled;

	event_handled = process_horizontal_menu_events(dialog->menu, &key);

	if (!event_handled && dialog->handler != NULL)
		event_handled = dialog->handler(dialog, key);

	return event_handled;
}

/**
 *	init_dialog_list - initialize dialog list fields
 *	@list: the dialog_list to initialize
 *
 *	This routine initializes the fields in the dialog_list structure to sane values.
 */
void init_dialog_list(struct dialog_list *list)
{
	list->list = NULL;
	list->current = NULL;
}

/**
 *	append_dialog_to_list - append a dialog to a dialog chain
 *	@dialog: the dialog window to add to the list
 *	@list: the dialog list to append to
 *
 *	This routine appends a dialog to the linked list representing
 *	a sequence of dialogs. This list is passed to the
 *	process_dialog_list_events() routine for processing of events
 *	in a series of dialog windows.
 */
void append_dialog_to_list(struct dialog_window *dialog, struct dialog_list *list)
{
	dialog->list = list;
	list->list = g_list_append(list->list, dialog);
	if (list->current == NULL)
		list->current = list->list;
}

inline GList *get_next_dialog(struct dialog_window *dialog)
{
	return g_list_next(dialog->list->current);
}

inline GList *get_previous_dialog(struct dialog_window *dialog)
{
	return g_list_previous(dialog->list->current);
}

/**
 *	process_dialog_list_events - process events for multiple dialog windows
 *	@list: address of structure containing info on multiple dialog windows
 *
 *	This routine processes the events for multiple dialog windows in a chain
 *	or sequence of dialogs. Only one is shown and gets keyboard events processed.
 *	To move through the windows, the current window must change the current
 *	element containing the target window to show. If the current window changes
 *	its dialog->status to DLG_STATUS_CLOSING then we terminate all event
 *	processing and exit.
 */
void process_dialog_list_events(struct dialog_list *list)
{
	struct dialog_window *dialog = list->current->data;

	show_dialog_window(dialog);

	while (dialog->status != DLG_STATUS_CLOSING) {
		if (dialog != list->current->data) {
			hide_popup_window(dialog->panel);
			dialog = list->current->data;
			show_dialog_window(dialog);
		}
		process_dialog_window_events(dialog, panel_getch(dialog->panel));
	}
}

/**
 *	delete_dialog_list - delete multiple dialog windows
 *	@list: address of first link in the list to start delete windows from
 *
 *	This routine is called when we need to delete all the dialog windows
 *	in a linked list.
 */
void delete_dialog_list(GList *list)
{
	GList *next_link;
	GList *link = list;

	while (link != NULL) {
		if (link->data != NULL)
			delete_dialog_window(link->data, NULL);
			
		next_link = link->next;
		g_list_remove(link, link->data);
		link = next_link;
	}
}

/**
 *	process_modal_dialog - handle common steps for a modal type dialog
 *	@dialog: the dialog
 *
 *	This routine handles the display of modal dialog, handling events for
 *	it and when it closes, cleaning up after it.
 */
void process_modal_dialog(struct dialog_window *dialog)
{
	show_dialog_window(dialog);

	while (dialog->status != DLG_STATUS_CLOSING) {
		process_dialog_window_events(dialog, panel_getch(dialog->panel));
	}
	delete_dialog_window(dialog, NULL);
}
