/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <frontend.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "add.h"
#include "enum.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "task.h"
#include "plugin.h"
#include "logging.h"

/**
 *	can_volume_accept_new_feature - see if a feature can be added to a volume
 *	@volume: volume's object handle
 *
 *	This routine checks to see if any feature plugin can
 *	add a feature to the given volume and if so returns TRUE.
 */
gboolean can_volume_accept_new_feature(object_handle_t volume)
{
	gboolean result = FALSE;
	handle_array_t *plugins;

	if (enum_feature_plugins(NULL, &plugins) == 0) {
		gint i;

		for (i = 0; i < plugins->count && result == FALSE; i++) {
			result = evms_can_add_feature_to_volume(volume, plugins->handle[i]) == 0;
		}
		g_free(plugins);
	}
	return result;
}

/**
 *	add_segment_manager_menuitem_activated - assign a segment manager to a storage object
 *	@item: the menu item that initiated this action
 *
 *	This routine is invoked when a menu item is activated to allow assigning a segment
 *	manager to a storage object.
 */
int add_segment_manager_menuitem_activated(struct menu_item *item)
{
	object_handle_t handle = GPOINTER_TO_UINT(item->user_data);

	if (handle == 0) {
		process_plugin_task(_("Add Segment Manager to Storage Object"), EVMS_SEGMENT_MANAGER, FALSE,
					EVMS_Task_Assign_Plugin);
	}

	return 0;
}

/**
 *	create_add_feature_task_dialog - create task, set selected object and create options dialog
 *	@volume: the volume handle
 *	@feature: the feature plugin handle
 *
 *	This routine creates the EVMS_Task_Add_Feature task, sets the given volume as the selected
 *	object and creates the option dialog to configure the feature.
 */
struct dialog_window *create_add_feature_task_dialog(object_handle_t volume, object_handle_t feature)
{
	int rc;
	task_handle_t task;
	struct dialog_window *dialog = NULL;

	rc = evms_create_task(feature, EVMS_Task_Add_Feature, &task);
	if (rc == 0) {
		rc = set_selected_object(task, volume);
		if (rc == 0) {
			GHashTable *hash_table;

			hash_table = g_hash_table_new(g_str_hash, g_str_equal);
			g_hash_table_insert(hash_table, "task", GUINT_TO_POINTER(task));
			g_hash_table_insert(hash_table, "verb", g_strdup(_("Add Feature")));

			dialog = create_task_options_dialog(task, _("Add Feature to Volume"),
					_("Add"), NULL, hash_table);
			set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_task_dialog);
		} else {
			evms_destroy_task(task);
			show_message_dialog(_("Set Selected Object Error"),
					_("Received an error communicating volume to feature plugin: %s"),
					evms_strerror(rc));
			log_error("%s: evms_set_selected_objects() returned error code %d.\n", __FUNCTION__, rc);
		}
	} else {
		show_message_dialog(_("Error starting Add Feature task"),
				_("Received the following error starting Add Feature task: %s"),
				evms_strerror(rc));
		log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__, rc);
	}
	return dialog;
}

/**
 *	feature_for_volume_button_activated - callback invoke when the feature is selected for the volume
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Next button is activated after selecting
 *	a feature to add to a volume. We then create a task now that we know the
 *	volume and the feature to add to it.
 */
int feature_for_volume_button_activated(struct menu_item *item)
{
	GHashTable *hash_table;
	struct selwin *selwin = item->user_data;
	object_handle_t saved_plugin, curr_plugin;
	struct dialog_window *dialog = item->user_data;

	hash_table = dialog->user_data;
	curr_plugin = get_selected_handle(selwin->clist);
	saved_plugin = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "plugin"));
	
	if (curr_plugin != saved_plugin) {
		GList *next_dialog;
		object_handle_t volume;
		struct dialog_window *new_dialog;
		/*
		 * Since the selected feature has changed we need to delete any
		 * dialogs that we may have built for the add feature task.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		volume = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "volume"));
		new_dialog = create_add_feature_task_dialog(volume, curr_plugin);

		if (new_dialog != NULL) {
			g_hash_table_insert(hash_table, "plugin", GUINT_TO_POINTER(curr_plugin));
			append_dialog_to_list(new_dialog, dialog->list);
			dialog->list->current = get_next_dialog(dialog);
		}
	} else {
		dialog->list->current = get_next_dialog(dialog);
	}

	return 0;
}

/**
 *	filter_features_for_volume - keep the plugins that can be added to a given volume
 *	@handle: the plugin handle
 *	@user_data: contains the handle of the volume
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given feature can be added to a certain volume.
 */
int filter_features_for_volume(object_handle_t handle, void *user_data)
{
	return evms_can_add_feature_to_volume(GPOINTER_TO_UINT(user_data), handle);
}

/**
 *	delete_feature_plugin_dialog_cb - callback invoked when feature plugin dialog is being deleted
 *	@dialog: the feature plugin selection dialog
 *
 *	This routine is invoked when the feature plugin dialog is being deleted. We take
 *	this opportunity to destroy the hash table associated with the dialog window.
 */
void delete_feature_plugin_dialog_cb(struct dialog_window *dialog)
{
	g_hash_table_destroy(dialog->user_data);
}

/**
 *	create_feature_plugin_dialog - create a dialog with a list of features that can be added to a volume
 *	@handle: the handle of the volume we want to list features than can be added to it
 *
 *	This routine creates a dialog window with a list of feature plugins that can be added to it.
 */
struct dialog_window *create_feature_plugin_dialog(object_handle_t handle)
{
	struct selwin *selwin;
	GHashTable *hash_table;
	struct dialog_window *dialog;

	hash_table = g_hash_table_new(g_str_hash, g_str_equal);
	g_hash_table_insert(hash_table, "volume", GUINT_TO_POINTER(handle));

	selwin = create_selection_window(_("Add Feature to Volume - Plugin Selection"),
					NULL, NULL, NULL,
					(menuitem_activate_cb)feature_for_volume_button_activated,
					NULL, NULL, hash_table);

	dialog = (struct dialog_window *)selwin;
	set_clist_column_count(selwin->clist, 2);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.95),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, _("Feature"));

	clist_populate(selwin->clist, enum_feature_plugins, filter_features_for_volume,
			format_standard_item, NULL, GUINT_TO_POINTER(handle));

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_dialog_delete_cb(dialog, (dialog_delete_cb)delete_feature_plugin_dialog_cb);

	return dialog;
}

/**
 *	volume_for_feature_button_activated - callback to present feature plugins available for a certain volume
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Next button is activated after selecting
 *	a volume in order to display another dialog containing the feature plugins
 *	that the volume will accept.
 */
int volume_for_feature_button_activated(struct menu_item *item)
{
	object_handle_t current_vol, saved_vol;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;

	current_vol = get_selected_handle(selwin->clist);
	saved_vol = GPOINTER_TO_UINT(dialog->user_data);

	if (current_vol != saved_vol) {
		GList *next_dialog;
		struct dialog_window *new_dialog;
		/*
		 * Since the selected volume has changed we need to delete any
		 * dialogs that we may have built for the add feature task.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		new_dialog = create_feature_plugin_dialog(current_vol);
		dialog->user_data = GUINT_TO_POINTER(current_vol);
		append_dialog_to_list(new_dialog, dialog->list);
	}
	dialog->list->current = get_next_dialog(dialog);

	return 0;
}

/**
 *	filter_feature_volume - keep the volumes that can have an EVMS feature added to it
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given volume can have a feature added to it.
 */
int filter_feature_volume(object_handle_t handle, void *user_data)
{
	return !can_volume_accept_new_feature(handle);
}

/**
 *	show_add_feature_dialog - display dialog to remove/unassign a segment manager
 *	@handle: non-zero when invoked from context popup menu
 *
 *	This routine displays a selection window to allow the user to remove a segment manager
 *	from one of the storage objects listed.
 **/
int show_add_feature_dialog(object_handle_t handle)
{
	struct selwin *selwin;
	struct dialog_list dialogs;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Add Feature to Volume"),
					NULL, NULL,
					NULL,
					(menuitem_activate_cb)volume_for_feature_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;

	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Volume"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	if (handle == 0)
		clist_populate(selwin->clist, enum_volumes,
				filter_feature_volume,
				format_standard_item, NULL, NULL);
	else
		clist_populate_handle(selwin->clist, handle,
				format_standard_item, NULL, NULL);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);

	init_dialog_list(&dialogs);
	append_dialog_to_list(dialog, &dialogs);
	process_dialog_list_events(&dialogs);
	delete_dialog_list(dialogs.list);

	return 0;
}

/**
 *	context_add_feature_menuitem_activated - add a feature to an EVMS volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to add a feature to an existing EVMS volume. The
 *	operation was initiated by selecting the "Add Feature..." menu item on
 *	the popup context menu.
 */
int context_add_feature_menuitem_activated(struct menu_item *item)
{
	return show_add_feature_dialog(GPOINTER_TO_UINT(item->user_data));
}

/**
 *	actions_add_feature_menuitem_activated - add a feature to an EVMS volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Create->Add->Feature to Volume Actions
 *	pulldown menu item.
 */
int actions_add_feature_menuitem_activated(struct menu_item *item)
{
	return show_add_feature_dialog(0);
}
