/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: dload.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <dlfcn.h>
#include <unistd.h>
#include <errno.h>

#include "fullengine.h"
#include "dload.h"
#include "lists.h"
#include "engine.h"
#include "memman.h"
#include "message.h"
#include "internalAPI.h"
#include "commit.h"
#include "cluster.h"
#include "remote.h"
#include <ldm_funcs.h>


static void unload_module(so_record_t * so_record) {

	LOG_PROC_ENTRY();

	LOG_DEBUG("Unload module %s.\n", so_record->name);
	engine_free(so_record->name);
	LOG_DEBUG("Issuing dlclose() on handle %p.\n", so_record->handle);
	dlclose(so_record->handle);
	if (!list_empty(so_record->plugin_list)) {
		LOG_WARNING("Warning: Unloading module %s while plug-ins are still loaded from it.\n", so_record->name);
	}
	destroy_list(so_record->plugin_list);
	engine_free(so_record);

	LOG_PROC_EXIT_VOID();
}


static int release_plugin(plugin_record_t * pPlugRec) {

	LOG_PROC_ENTRY();

	if (pPlugRec != NULL) {

		remove_thing(&plugins_list, pPlugRec);

		remove_thing(pPlugRec->so_record->plugin_list, pPlugRec);

		/*
		 * Unload the module if we have no other plug-ins loaded from
		 * the module.
		 */
		if (list_empty(pPlugRec->so_record->plugin_list)) {
			unload_module(pPlugRec->so_record);
		}
	}

	LOG_PROC_EXIT_INT(0);
	return 0;
}


static int unload_plugin(plugin_record_t * pPlugRec) {

	LOG_PROC_ENTRY();

	if (pPlugRec != NULL) {

		if(pPlugRec == cluster_manager) {
			/*
			 * Daemons and workers do not open the Engine on other
			 * nodes in the cluster.
			 */
			if (!(engine_mode & (ENGINE_DAEMON | ENGINE_WORKER))) {
				remote_close_engine();
			}
			disconnect_from_ece();
			local_focus = TRUE;
		}

		if (pPlugRec->functions.plugin->cleanup_evms_plugin != NULL) {
			pPlugRec->functions.plugin->cleanup_evms_plugin();
		}

		release_plugin(pPlugRec);
	}

	LOG_PROC_EXIT_INT(0);
	return 0;
}


/*
 * Unload the plug-ins in an orderly fashion.  Specifically,
 * disk/segment/region/EVMS region managers may have open file descriptors
 * for objects.  They can close the descriptors on cleanup.
 * The cluster manager plug-in should be left around to handle the
 * closes if they are on a remote node.
 */
int unload_plugins(void) {

	int                rc;
	int                i;
	plugin_record_t  * pPlugRec;
	list_anchor_t            plugin_set;
	evms_plugin_code_t unload_order[] = {EVMS_FILESYSTEM_INTERFACE_MODULE,
		                             EVMS_ASSOCIATIVE_FEATURE,
		                             EVMS_FEATURE,
		                             EVMS_REGION_MANAGER,	
		                             EVMS_SEGMENT_MANAGER,	
		                             EVMS_DEVICE_MANAGER,	
		                             EVMS_CLUSTER_MANAGER_INTERFACE_MODULE,	
		                             EVMS_NO_PLUGIN
	                                    };

	LOG_PROC_ENTRY();

	for (i = 0; unload_order[i] != EVMS_NO_PLUGIN; i++) {
		rc = engine_get_plugin_list(unload_order[i], 0, &plugin_set);

		if (rc == 0) {
			list_element_t iter1;
			list_element_t iter2;

			LIST_FOR_EACH_SAFE(plugin_set, iter1, iter2, pPlugRec) {
				delete_element(iter1);
				unload_plugin(pPlugRec);
			}
		}

		destroy_list(plugin_set);
	}

	LOG_PROC_EXIT_INT(0);
	return 0;
}


/*
 *  Called with a plug-in record to see if it is a duplicate of an existing
 *  plug-in record in the list_anchor_t. Matching them based on the plug-in record ID
 *  field allows for a quick comparison of: OEM id, EVMS plug-in Type and
 *  EVMS plug-in ID.  Plug-ins are duplicates if they have matching ID fields!
 *
 *  Returns: TRUE if a duplicate is found in the list
 *           FALSE if we don't find a duplicate plug-in in the list
 */
static boolean isa_duplicate_plugin( plugin_record_t * pPlugRec, list_anchor_t plugin_list ) {

	boolean           result = FALSE;
	list_element_t    iter1;
	list_element_t    iter2;
	plugin_record_t * pr;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH_SAFE(plugin_list, iter1, iter2, pr) {
		
		if (pr->id == pPlugRec->id) {

			engine_user_message(NULL, NULL,
					    _("Duplicate Plug-ins detected\n"
					      "Already have loaded:\n"
					      "  long name: %-40s\n"
					      "  oem name: %-40s\n"
					      "  ID: %#08X\n"
					      "  version: %d.%d.%d\n"
					      "Currently loading:\n"
					      "  long name: %-40s\n"
					      "  oem name: %-40s\n"
					      "  ID: %#08X\n"
					      "  version: %d.%d.%d\n"),
					    pr->long_name,
					    pr->oem_name,
					    pr->id,
					    pr->version.major, pr->version.minor, pr->version.patchlevel,
					    pPlugRec->long_name,
					    pPlugRec->oem_name,
					    pPlugRec->id,
					    pPlugRec->version.major, pPlugRec->version.minor, pPlugRec->version.patchlevel);

			/*
			 * Check to see if the plug-in we are about to load is
			 * older than the one that is loaded.  If so,  return
			 * TRUE that this is a duplicate plug-in.
			 */
			if (pr->version.major > pPlugRec->version.major) {
				result = TRUE;
			} else {
				if ((pr->version.major == pPlugRec->version.major) &&
				    (pr->version.minor > pPlugRec->version.minor)) {
					result = TRUE;
				} else {
					if ((pr->version.minor == pPlugRec->version.minor) &&
					    (pr->version.patchlevel >= pPlugRec->version.patchlevel)) {
						result = TRUE;
					}
				}
			}

			if (result == FALSE) {
				/*
				 * The plug-in we are about to load is newer than
				 * the one that is currently loaded.  Unload the
				 * current plug-in and allow the new one to be
				 * loaded.
				 */

				engine_user_message(NULL, NULL,
						    _("Unloading version %d.%d.%d of the plug-in.\n"),
						    pr->version.major, pr->version.minor, pr->version.patchlevel);
				delete_element(iter1);
				release_plugin(pr);

			} else {
				engine_user_message(NULL, NULL,
						    _("Keeping version %d.%d.%d of the plug-in.\n"),
						    pr->version.major, pr->version.minor, pr->version.patchlevel);
			}

			break;
		}
	}
	
	LOG_PROC_EXIT_BOOLEAN(result);
	return result;
}


/*
 *  Validates that a plug-in record is useable.  Does simple checking to
 *  see if we could access the plug-in Ok from the EVMS engine.
 */
static boolean isa_valid_plugin_record(plugin_record_t * pPlugRec) {

	boolean result = TRUE;
	int rc;

	LOG_PROC_ENTRY();

	rc = check_version(pPlugRec->required_engine_api_version, engine_services_api_version);
	if (rc != 0) {

		if (rc < 0) {
			engine_user_message(NULL, NULL,
					    _("The plug-in %s in module %s requires Engine services API version (%d.%d.%d) which is greater than this Engine's services API version (%d.%d.%d).\n"),
					    pPlugRec->short_name, pPlugRec->so_record->name,
					    pPlugRec->required_engine_api_version.major, pPlugRec->required_engine_api_version.minor, pPlugRec->required_engine_api_version.patchlevel,
					    engine_services_api_version.major, engine_services_api_version.minor, engine_services_api_version.patchlevel);

		} else {
			engine_user_message(NULL, NULL,
					    _("The plug-in %s in module %s requires Engine services API version (%d.%d.%d) which is less than this Engine's services API version (%d.%d.%d).\n"),
					    pPlugRec->short_name, pPlugRec->so_record->name,
					    pPlugRec->required_engine_api_version.major, pPlugRec->required_engine_api_version.minor, pPlugRec->required_engine_api_version.patchlevel,
					    engine_services_api_version.major, engine_services_api_version.minor, engine_services_api_version.patchlevel);

		}
		/*
		 * The plug-in requires a different version of the Engine.  We can't
		 * trust the rest of the plug-in record to be in the correct format, so
		 * don't do any further checks.
		 */
		result = FALSE;

	} else {
		switch (GetPluginType(pPlugRec->id)) {
		case EVMS_FILESYSTEM_INTERFACE_MODULE:
			rc = check_version(pPlugRec->required_plugin_api_version.fsim, engine_fsim_api_version);
			if (rc != 0) {
				if (rc < 0) {
					engine_user_message(NULL, NULL,
							    _("The plug-in %s in module %s requires Engine FSIM API version (%d.%d.%d) which is greater than this Engine's FSIM API version (%d.%d.%d).\n"),
							    pPlugRec->short_name, pPlugRec->so_record->name,
							    pPlugRec->required_plugin_api_version.fsim.major, pPlugRec->required_plugin_api_version.fsim.minor, pPlugRec->required_plugin_api_version.fsim.patchlevel,
							    engine_fsim_api_version.major, engine_fsim_api_version.minor, engine_fsim_api_version.patchlevel);

				} else {
					engine_user_message(NULL, NULL,
							    _("The plug-in %s in module %s requires Engine FSIM API version (%d.%d.%d) which is less than this Engine's FSIM API version (%d.%d.%d).\n"),
							    pPlugRec->short_name, pPlugRec->so_record->name,
							    pPlugRec->required_plugin_api_version.fsim.major, pPlugRec->required_plugin_api_version.fsim.minor, pPlugRec->required_plugin_api_version.fsim.patchlevel,
							    engine_fsim_api_version.major, engine_fsim_api_version.minor, engine_fsim_api_version.patchlevel);
				}

				result = FALSE;
			}
			break;

		case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
			rc = check_version(pPlugRec->required_plugin_api_version.cluster, engine_cluster_api_version);
			if (rc != 0) {
				if (rc < 0) {
					engine_user_message(NULL, NULL,
							    _("The plug-in %s in module %s requires Engine cluster manager API version (%d.%d.%d) which is greater than this Engine's cluster manager API version (%d.%d.%d).\n"),
							    pPlugRec->short_name, pPlugRec->so_record->name,
							    pPlugRec->required_plugin_api_version.cluster.major, pPlugRec->required_plugin_api_version.cluster.minor, pPlugRec->required_plugin_api_version.cluster.patchlevel,
							    engine_cluster_api_version.major, engine_cluster_api_version.minor, engine_cluster_api_version.patchlevel);

				} else {
					engine_user_message(NULL, NULL,
							    _("The plug-in %s in module %s requires Engine cluster manager API version (%d.%d.%d) which is less than this Engine's cluster manager API version (%d.%d.%d).\n"),
							    pPlugRec->short_name, pPlugRec->so_record->name,
							    pPlugRec->required_plugin_api_version.cluster.major, pPlugRec->required_plugin_api_version.cluster.minor, pPlugRec->required_plugin_api_version.cluster.patchlevel,
							    engine_cluster_api_version.major, engine_cluster_api_version.minor, engine_cluster_api_version.patchlevel);
				}

				result = FALSE;
			}
			break;

		default:
			rc = check_version(pPlugRec->required_plugin_api_version.plugin, engine_plugin_api_version);
			if (rc != 0) {
				if (rc < 0) {
					engine_user_message(NULL, NULL,
							    _("The plug-in %s in module %s requires Engine plug-in API version (%d.%d.%d) which is greater than this Engine's plug-in API version (%d.%d.%d).\n"),
							    pPlugRec->short_name, pPlugRec->so_record->name,
							    pPlugRec->required_plugin_api_version.plugin.major, pPlugRec->required_plugin_api_version.plugin.minor, pPlugRec->required_plugin_api_version.plugin.patchlevel,
							    engine_plugin_api_version.major, engine_plugin_api_version.minor, engine_plugin_api_version.patchlevel);

				} else {
					engine_user_message(NULL, NULL,
							    _("The plug-in %s in module %s requires Engine plug-in API version (%d.%d.%d) which is less than this Engine's plug-in API version (%d.%d.%d).\n"),
							    pPlugRec->short_name, pPlugRec->so_record->name,
							    pPlugRec->required_plugin_api_version.plugin.major, pPlugRec->required_plugin_api_version.plugin.minor, pPlugRec->required_plugin_api_version.plugin.patchlevel,
							    engine_plugin_api_version.major, engine_plugin_api_version.minor, engine_plugin_api_version.patchlevel);

				}

				result = FALSE;
			}
			break;
		}

		if (pPlugRec->container_functions != NULL) {
			rc = check_version(pPlugRec->required_container_api_version, engine_container_api_version);
			if (rc != 0) {
				if (rc < 0) {
					engine_user_message(NULL, NULL,
							    _("The plug-in %s in module %s requires Engine container API version (%d.%d.%d) which is greater than this Engine's container API version (%d.%d.%d).\n"),
							    pPlugRec->short_name, pPlugRec->so_record->name,
							    pPlugRec->required_container_api_version.major, pPlugRec->required_container_api_version.minor, pPlugRec->required_container_api_version.patchlevel,
							    engine_container_api_version.major, engine_container_api_version.minor, engine_container_api_version.patchlevel);

				} else {
					engine_user_message(NULL, NULL,
							    _("The plug-in %s in module %s requires Engine container API version (%d.%d.%d) which is less than this Engine's container API version (%d.%d.%d).\n"),
							    pPlugRec->short_name, pPlugRec->so_record->name,
							    pPlugRec->required_container_api_version.major, pPlugRec->required_container_api_version.minor, pPlugRec->required_container_api_version.patchlevel,
							    engine_container_api_version.major, engine_container_api_version.minor, engine_container_api_version.patchlevel);
				}

				result = FALSE;
			}
		}

		if (pPlugRec->short_name == NULL) {
			if (pPlugRec->long_name == NULL) {
				if (pPlugRec->oem_name == NULL) {
					engine_user_message(NULL, NULL,
							    _("A plug-in in module %s did not provide a short name, long name, nor an oem name.\n"),
							    pPlugRec->so_record->name);
				} else {
					engine_user_message(NULL, NULL,
							    _("A plug-in in module %s did not provide a short name nor a long name.  "
							      "The oem name is \"%s\".\n"),
							    pPlugRec->so_record->name, pPlugRec->oem_name);
				}

			} else {
				if (pPlugRec->oem_name == NULL) {
					engine_user_message(NULL, NULL,
							    _("A plug-in in module %s did not provide a short name nor an oem name.  The long name is \"%s\".\n"),
							    pPlugRec->so_record->name, pPlugRec->long_name);
				} else {
					engine_user_message(NULL, NULL,
							    _("A plug-in in module %s did not provide a short name.  The long name is \"%s\".  "
							      "The oem name is \"%s\".\n"),
							    pPlugRec->so_record->name, pPlugRec->long_name, pPlugRec->oem_name);
				}
			}

			result = FALSE;

		} else {
			if (pPlugRec->long_name == NULL) {
				if (pPlugRec->oem_name == NULL) {
					engine_user_message(NULL, NULL,
							    _("A plug-in in module %s did not provide a long name, nor an oem name.  The short name is \"%s\".\n"),
							    pPlugRec->so_record->name, pPlugRec->short_name);
				} else {
					engine_user_message(NULL, NULL,
							    _("A plug-in in module %s did not provide a long name.  The short name is \"%s\".  The oem name is \"%s\".\n"),
							    pPlugRec->so_record->name, pPlugRec->short_name, pPlugRec->oem_name);
				}

				result = FALSE;

			} else {
				if (pPlugRec->oem_name == NULL) {
					engine_user_message(NULL, NULL,
							    _("A plug-in in module %s did not provide an oem name.  The short name is \"%s\".  The long name is \"%s\".\n"),
							    pPlugRec->so_record->name, pPlugRec->short_name, pPlugRec->long_name);
					result = FALSE;

				} else {
					/* All required names are provided. */
				}
			}
		}

		if (pPlugRec->functions.plugin == NULL) {
			engine_user_message(NULL, NULL,
					    _("The plug-in %s in module %s did not provide a function table.\n"),
					    pPlugRec->short_name, pPlugRec->so_record->name);
			result = FALSE;
		}

	}

	if (!result) {
		engine_user_message(NULL, NULL, _("The plug-in failed to load.\n"));
	}

	LOG_PROC_EXIT_BOOLEAN(result);
	return result;
}


/*
 *  Two routines to load and unload Linux shared objects.
 */

static int load_module(char * soname, so_record_t * * pso_record) {

	int rc = 0;
	char * error = NULL;
	so_record_t * so_record = engine_alloc(sizeof(so_record_t));

	LOG_PROC_ENTRY();

	*pso_record = NULL;

	if (so_record != NULL) {

		status_message(_("Loading module: %s\n"), soname);
		dlerror();
		so_record->handle = dlopen(soname, RTLD_NOW);

		error = dlerror();
		if (error == NULL) {
			so_record->name = engine_strdup(soname);
			so_record->plugin_list = allocate_list();

			if (so_record->plugin_list != NULL) {
				*pso_record = so_record;

			} else {
				LOG_CRITICAL("Error allocating memory for a plug-in list in a .so record.\n");
				engine_free(so_record->name);
				engine_free(so_record);
				rc = ENOMEM;
			}

		} else {
			engine_user_message(NULL, NULL, _("Error loading %s: %s\n"), soname, error);
			engine_free(so_record);
			rc = ELIBBAD;
		}

	} else {
		LOG_CRITICAL("Error allocating memory for a .so record.\n");
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void * get_sym_addr(module_handle_t handle, char * symname) {

	void       * symaddr;
	const char * error;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Get address for symbol %s from module handle %p.\n", symname, handle);

	dlerror();
	symaddr = dlsym(handle, symname);
	if ((error = dlerror()) != NULL) {
		LOG_DEBUG("Error getting address for %s from module %p: %s\n", symname, handle, error);
		symaddr  = NULL;
	}

	LOG_PROC_EXIT_PTR(symaddr);
	return symaddr;
}


static int setup_clustering() {

	int rc = 0;
	int i;

	LOG_PROC_ENTRY();

	rc = connect_to_ece();

	if (rc == 0) {

		if (current_node_name != NULL) {
			if (!(engine_mode & ENGINE_DAEMON)) {
				for (i = 0; i < num_config_nodes; i++) {
					if (strcmp(current_node_name, config_node_names->node_info[i].node_name) == 0) {
						current_nodeid = &config_nodes[i];
						break;
					}
				}

				if (current_nodeid == &no_nodeid) {
					engine_user_message(NULL, NULL,
							    _("%s is not the name of a node in this cluster. "
							      "The Engine cannot be opened on the node named %s.\n"),
							    current_node_name, current_node_name);
					rc = ENOENT;
				}

				if (rc == 0) {
					if (current_nodeid != my_nodeid) {
						local_focus = FALSE;
					}
				}

			} else {
				engine_user_message(NULL, NULL,
						    _("The node_name parameter is not valid when starting the daemon.\n"));
				rc = EINVAL;
			}

		} else {
			current_nodeid = my_nodeid;
		}

		/*
		 * Neither daemons nor workers verify versions with other
		 * daemons and open the Engine on other nodes in the cluster.
		 */
		if ((rc == 0) && (!(engine_mode & (ENGINE_DAEMON | ENGINE_WORKER)))) {
			/*
			 * Make sure the EVMS daemons in the cluster are
			 * at the correct protocol version.
			 */
			status_message(_("Verifying communication protocol versions...\n"));

			rc = remote_verify_version();
			if (rc != 0) {
				engine_user_message(NULL, NULL,
						    _("There was an error when validating the version of the daemon protocol on the other nodes in the cluster.  "
						      "The Engine will run with clustering support disabled.  "
						      "EVMS will only manage local devices on this system.\n"));
				disconnect_from_ece();
				local_focus = TRUE;
				rc = 0;
			}
		}

		if (rc != 0) {
			disconnect_from_ece();
			local_focus = TRUE;
		}

	} else {
		if (current_node_name != NULL) {
			engine_user_message(NULL, NULL,
					    _("There was an error when connecting to %s.  "
					      "The error code was %d: %s  "
					      "This machine is not running in a clustered environment.\n"),
					    cluster_manager->short_name, rc, evms_strerror(rc));

		} else {
			engine_user_message(NULL, NULL,
					    _("There was an error when connecting to %s.  "
					      "The error code was %d: %s  "
					      "EVMS will only manage local devices on this system.\n"),
					    cluster_manager->short_name, rc, evms_strerror(rc));
			rc = 0;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Dummy functions for filling in function table entries that a plug-in
 * does not provide.
 */
static int return_ENOSYS() {
	return ENOSYS;
}


static int return_0() {
	return 0;
}

static void return_void() {
	return;
}


static int validate_plugin_functions(plugin_record_t * pPlugRec) {

	int rc = 0;

	LOG_PROC_ENTRY();

	/* A plug-in must support the following functions. */

	if ((pPlugRec->functions.plugin->setup_evms_plugin        == NULL) ||
	    (pPlugRec->functions.plugin->discover                 == NULL) ||
	    (pPlugRec->functions.plugin->discard                  == NULL) ||
	    (pPlugRec->functions.plugin->add_sectors_to_kill_list == NULL) ||
	    (pPlugRec->functions.plugin->commit_changes           == NULL) ||
	    (pPlugRec->functions.plugin->read                     == NULL) ||
	    (pPlugRec->functions.plugin->write                    == NULL)) {
		rc = ENOSYS;
	}

	/*
	 * The remaining functions are not absolutely necessary for a plug-in to
	 * function.  If the plug-in doesn't specify them, we supply a default.
	 */

	if (pPlugRec->functions.plugin->can_add_feature == NULL) {
		pPlugRec->functions.plugin->can_add_feature = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->can_unassign == NULL) {
		pPlugRec->functions.plugin->can_unassign = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->can_delete == NULL) {
		pPlugRec->functions.plugin->can_delete = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->can_expand == NULL) {
		pPlugRec->functions.plugin->can_expand = (void *) return_0;
	}
	if (pPlugRec->functions.plugin->can_expand_by == NULL) {
		pPlugRec->functions.plugin->can_expand_by = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->can_shrink == NULL) {
		pPlugRec->functions.plugin->can_shrink = (void *) return_0;
	}
	if (pPlugRec->functions.plugin->can_shrink_by == NULL) {
		pPlugRec->functions.plugin->can_shrink_by = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->can_replace_child == NULL) {
		pPlugRec->functions.plugin->can_replace_child = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->can_set_volume == NULL) {
		pPlugRec->functions.plugin->can_set_volume = (void *) return_0;
	}
	if (pPlugRec->functions.plugin->set_volume == NULL) {
		pPlugRec->functions.plugin->set_volume = (void *) return_void;
	}
	if (pPlugRec->functions.plugin->create == NULL) {
		pPlugRec->functions.plugin->create = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->assign == NULL) {
		pPlugRec->functions.plugin->assign = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->delete == NULL) {
		pPlugRec->functions.plugin->delete = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->unassign == NULL) {
		pPlugRec->functions.plugin->unassign = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->expand == NULL) {
		pPlugRec->functions.plugin->expand = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->shrink == NULL) {
		pPlugRec->functions.plugin->shrink = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->replace_child == NULL) {
		pPlugRec->functions.plugin->replace_child = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->can_activate == NULL) {
		pPlugRec->functions.plugin->can_activate = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->activate == NULL) {
		pPlugRec->functions.plugin->activate = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->can_deactivate == NULL) {
		pPlugRec->functions.plugin->can_deactivate = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->deactivate == NULL) {
		pPlugRec->functions.plugin->deactivate = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->get_option_count == NULL) {
		pPlugRec->functions.plugin->get_option_count = (void *) return_0;
	}
	if (pPlugRec->functions.plugin->init_task == NULL) {
		pPlugRec->functions.plugin->init_task = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->set_option == NULL) {
		pPlugRec->functions.plugin->set_option = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->set_objects == NULL) {
		pPlugRec->functions.plugin->set_objects = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->get_info == NULL) {
		pPlugRec->functions.plugin->get_info = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->set_info == NULL) {
		pPlugRec->functions.plugin->set_info = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->get_plugin_info == NULL) {
		pPlugRec->functions.plugin->get_plugin_info = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->get_plugin_functions == NULL) {
		pPlugRec->functions.plugin->get_plugin_functions = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->plugin_function == NULL) {
		pPlugRec->functions.plugin->plugin_function = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.plugin->backup_metadata == NULL) {
		pPlugRec->functions.plugin->backup_metadata = (void *) return_ENOSYS;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int validate_fsim_functions(plugin_record_t * pPlugRec) {

	int rc = 0;

	LOG_PROC_ENTRY();

	/* An FSIM must support the following functions. */

	if ((pPlugRec->functions.fsim->setup_evms_plugin == NULL) ||
	    (pPlugRec->functions.fsim->probe             == NULL) ||
	    (pPlugRec->functions.fsim->discard           == NULL)) {
		rc = ENOSYS;
	}

	/*
	 * The remaining functions are not absolutely necessary for a plug-in to
	 * function.  If the plug-in doesn't specify them, we supply a default.
	 */
	if (pPlugRec->functions.fsim->get_fs_size == NULL) {
		pPlugRec->functions.fsim->get_fs_size = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->get_fs_limits == NULL) {
		pPlugRec->functions.fsim->get_fs_limits = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->can_mkfs == NULL) {
		pPlugRec->functions.fsim->can_mkfs = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->can_unmkfs == NULL) {
		pPlugRec->functions.fsim->can_unmkfs = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->can_fsck == NULL) {
		pPlugRec->functions.fsim->can_fsck = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->can_expand_by == NULL) {
		pPlugRec->functions.fsim->can_expand_by = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->can_shrink_by == NULL) {
		pPlugRec->functions.fsim->can_shrink_by = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->mkfs_setup == NULL) {
		pPlugRec->functions.fsim->mkfs_setup = (void *) return_0;
	}
	if (pPlugRec->functions.fsim->mkfs == NULL) {
		pPlugRec->functions.fsim->mkfs = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->unmkfs_setup == NULL) {
		pPlugRec->functions.fsim->unmkfs_setup = (void *) return_0;
	}
	if (pPlugRec->functions.fsim->unmkfs == NULL) {
		pPlugRec->functions.fsim->unmkfs = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->fsck == NULL) {
		pPlugRec->functions.fsim->fsck = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->expand == NULL) {
		pPlugRec->functions.fsim->expand = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->shrink == NULL) {
		pPlugRec->functions.fsim->shrink = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->commit_changes == NULL) {
		pPlugRec->functions.fsim->commit_changes = (void *) return_0;
	}
	if (pPlugRec->functions.fsim->get_option_count == NULL) {
		pPlugRec->functions.fsim->get_option_count = (void *) return_0;
	}
	if (pPlugRec->functions.fsim->init_task == NULL) {
		pPlugRec->functions.fsim->init_task = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->set_option == NULL) {
		pPlugRec->functions.fsim->set_option = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->set_volumes == NULL) {
		pPlugRec->functions.fsim->set_volumes = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->get_volume_info == NULL) {
		pPlugRec->functions.fsim->get_volume_info = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->get_plugin_info == NULL) {
		pPlugRec->functions.fsim->get_plugin_info = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->get_plugin_functions == NULL) {
		pPlugRec->functions.fsim->get_plugin_functions = (void *) return_ENOSYS;
	}
	if (pPlugRec->functions.fsim->plugin_function == NULL) {
		pPlugRec->functions.fsim->plugin_function = (void *) return_ENOSYS;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int validate_container_functions(plugin_record_t * pPlugRec) {

	int rc = 0;

	LOG_PROC_ENTRY();

	/*
	 * If a plug-in supports containers it must supply the following
	 * functions.
	 */

	if ((pPlugRec->container_functions->can_delete_container     == NULL) ||
	    (pPlugRec->container_functions->create_container         == NULL) ||
	    (pPlugRec->container_functions->discard_container        == NULL) ||
	    (pPlugRec->container_functions->delete_container         == NULL) ||
	    (pPlugRec->container_functions->commit_container_changes == NULL) ||
	    (pPlugRec->container_functions->get_container_info       == NULL)) {

		rc = ENOSYS;
	}

	/*
	 * The remaining functions are not absolutely necessary for a plug-in to
	 * function.  If the plug-in doesn't specify them, we supply a default.
	 */
	if (pPlugRec->container_functions->can_expand_container == NULL) {
		pPlugRec->container_functions->can_expand_container = (void *) return_ENOSYS;
	}
	if (pPlugRec->container_functions->can_shrink_container == NULL) {
		pPlugRec->container_functions->can_shrink_container = (void *) return_ENOSYS;
	}
	if (pPlugRec->container_functions->expand_container == NULL) {
		pPlugRec->container_functions->expand_container = (void *) return_ENOSYS;
	}
	if (pPlugRec->container_functions->shrink_container == NULL) {
		pPlugRec->container_functions->shrink_container = (void *) return_ENOSYS;
	}
	if (pPlugRec->container_functions->get_plugin_functions == NULL) {
		pPlugRec->container_functions->get_plugin_functions = (void *) return_ENOSYS;
	}
	if (pPlugRec->container_functions->plugin_function == NULL) {
		pPlugRec->container_functions->plugin_function = (void *) return_ENOSYS;
	}
	if (pPlugRec->container_functions->backup_container_metadata == NULL) {
		pPlugRec->container_functions->backup_container_metadata = (void *) return_ENOSYS;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int validate_cluster_functions(plugin_record_t * pPlugRec) {

	int rc = 0;

	LOG_PROC_ENTRY();

	/* No cluster functions are optional. */

	if ((pPlugRec->functions.cluster->setup_evms_plugin    == NULL) ||
	    (pPlugRec->functions.cluster->cleanup_evms_plugin  == NULL) ||
	    (pPlugRec->functions.cluster->register_callback    == NULL) ||
	    (pPlugRec->functions.cluster->unregister_callback  == NULL) ||
	    (pPlugRec->functions.cluster->send_msg             == NULL) ||
	    (pPlugRec->functions.cluster->get_my_nodeid        == NULL) ||
	    (pPlugRec->functions.cluster->get_num_config_nodes == NULL) ||
	    (pPlugRec->functions.cluster->get_all_nodes        == NULL) ||
	    (pPlugRec->functions.cluster->get_membership       == NULL)) {

		rc = ENOSYS;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int check_plugin(list_anchor_t plugin_list, plugin_record_t * pPlugRec) {

	int  rc = 0;

	LOG_PROC_ENTRY();

	status_message(_("Validating plug-in: %s\n"),
		       pPlugRec->long_name);

	if (isa_valid_plugin_record( pPlugRec ) == TRUE) {

		/* Don't run with duplicate plug-ins loaded ! */
		if (isa_duplicate_plugin( pPlugRec, plugin_list ) == FALSE) {
			uint plugin_type;

			plugin_type = GetPluginType(pPlugRec->id);

			switch (plugin_type) {
			
			/* Only load Device Managers in daemon mode. */
			case EVMS_ASSOCIATIVE_FEATURE:
			case EVMS_FEATURE:
			case EVMS_REGION_MANAGER:
			case EVMS_SEGMENT_MANAGER:
				if (engine_mode & ENGINE_DAEMON) {
					rc = E_NOLOAD;
					break;
				}

			case EVMS_DEVICE_MANAGER:
				rc = validate_plugin_functions(pPlugRec);
				if (rc == 0) {
					if (pPlugRec->container_functions != NULL) {
						rc = validate_container_functions(pPlugRec);

						if (rc != 0) {
							engine_user_message(NULL, NULL,
									    _("The %s plug-in in module %s failed to load.  It does not have a valid container functions table.\n"),
									    pPlugRec->short_name, pPlugRec->so_record->name);
						}
					}

				} else {
					engine_user_message(NULL, NULL,
							    _("The %s plug-in in module %s failed to load.  It does not have a valid function table.\n"),
							    pPlugRec->short_name, pPlugRec->so_record->name);
				}
				break;

			case EVMS_FILESYSTEM_INTERFACE_MODULE:
				
				/* Don't load FSIMs in daemon mode. */
				if (engine_mode & ENGINE_DAEMON) {
					rc = E_NOLOAD;

				} else {
					rc = validate_fsim_functions(pPlugRec);
					if (rc != 0) {
						engine_user_message(NULL, NULL,
								    _("The %s plug-in in module %s failed to load.  It does not have a valid function table.\n"),
								    pPlugRec->short_name, pPlugRec->so_record->name);
					}

				}
				break;

			case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
				/*
				 * Load a cluster manager only if one has not successfully
				 * loaded already.
				 */
				if (cluster_manager == NULL) {
					rc = validate_cluster_functions(pPlugRec);
					if (rc != 0) {
						engine_user_message(NULL, NULL,
								    _("The %s plug-in in module %s failed to load.  It does not have a valid function table.\n"),
								    pPlugRec->short_name, pPlugRec->so_record->name);
					}

				} else {
					LOG_DETAILS("Skip loading cluster manager %s because cluster manager %s is already loaded.\n", pPlugRec->short_name, cluster_manager->short_name);
				}
				break;

			default:
				engine_user_message(NULL, NULL,
						    _("The %s plug-in in module %s failed to load.  It is of type of %#x which is not valid.\n"),
						    pPlugRec->short_name, pPlugRec->so_record->name, plugin_type);
				rc = ENOSYS;
			}

		} else {
			rc = EEXIST;
		}

	} else {
		LOG_ERROR("Plug-in returned an invalid plug-in record.\n");
		rc = ENOSYS;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int load_module_plugins(so_record_t * so_record, plugin_record_t * * ppPlugRec, list_anchor_t plugin_list) {

	int rc = 0;
	plugin_record_t * pPlugRec = *ppPlugRec;

	LOG_PROC_ENTRY();

	while ((rc == 0) && (pPlugRec != NULL)) {
		pPlugRec->so_record = so_record;

		rc = check_plugin(plugin_list, pPlugRec);

		if (rc == 0) {
			list_element_t el1;

			pPlugRec->type = PLUGIN;

			el1 = insert_thing(plugin_list,
					  pPlugRec,
					  INSERT_AFTER,
					  NULL);

			if (el1 != NULL) {
				list_element_t el2;

				el2 = insert_thing(so_record->plugin_list,
						  pPlugRec,
						  INSERT_AFTER,
						  NULL);

				if (el2 != NULL) {
					LOG_DEFAULT("Loaded from %s.\n", so_record->name);
					LOG_DEFAULT("  short name:  %s\n", pPlugRec->short_name);
					LOG_DEFAULT("  long name:   %s\n", pPlugRec->long_name);
					LOG_DEFAULT("  version:     %d.%d.%d\n", pPlugRec->version.major, pPlugRec->version.minor, pPlugRec->version.patchlevel);

				} else {
					LOG_CRITICAL("Failed to put the plug-in record in .so record.\n");
					delete_element(el1);
				}

			} else {
				LOG_CRITICAL("Failed to put plug-in record in the plugins_list.\n");
			}

		} else {
			/*
			 * Don't let any plug-in load error stop the plug-in
			 * loading process.
			 */
			rc = 0;
		}

		/* Check if the .so has another plug-in record. */
		ppPlugRec++;
		pPlugRec = *ppPlugRec;
	}

	/*
	 * If we didn't load any plug-ins from this module, then unload
	 * the module.
	 */
	if (list_empty(so_record->plugin_list)) {
		unload_module(so_record);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Call the plug-ins' setup_evms_plugin() in a civilized order.
 */
static int setup_plugins() {

	int  rc = 0;
	list_anchor_t            plugin_set;
	list_element_t     iter;
	plugin_record_t  * plug_rec;

	LOG_PROC_ENTRY();

	rc = engine_get_plugin_list(EVMS_CLUSTER_MANAGER_INTERFACE_MODULE, 0, &plugin_set);

	if (rc != 0) {
		LOG_SERIOUS("Failed to get a list of cluster manager plug-ins.  Error code is %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(plugin_set, iter, plug_rec) {
		if (cluster_manager == NULL) {
			status_message(_("Setup plug-in: %s\n"), plug_rec->long_name);
			rc = plug_rec->functions.cluster->setup_evms_plugin(&engine_functions);

			if (rc == 0) {
				/* Save a pointer to the cluster manager plug-in. */
				cluster_manager = plug_rec;

				/* Clustering turns on PIDs in log timestamps. */
				log_pid = TRUE;

				rc = setup_clustering();

				if (rc != 0) {
					destroy_list(plugin_set);
					LOG_PROC_EXIT_INT(rc);
					return rc;
				}

			} else {
				if (rc != E_NOLOAD) {
					engine_user_message(NULL, NULL,
							    _("The plug-in %s in module %s failed to load.  The plug-in's setup_evms_plugin() function failed with error code %d: %s.\n"),
							    plug_rec->short_name, plug_rec->so_record->name, rc, strerror(rc));
				}
			}

		} else {
			LOG_DETAILS("Cluster manger plug-in %s is already loaded.  I'm not loading cluster manager %s.\n",
				    cluster_manager->short_name, plug_rec->short_name);
		}

		if (rc != 0) {
			release_plugin(plug_rec);
		}
	}

	destroy_list(plugin_set);

	rc = engine_get_plugin_list(EVMS_DEVICE_MANAGER, 0, &plugin_set);

	if (rc != 0) {
		LOG_SERIOUS("Failed to get a list of device manager plug-ins.  Error code is %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(plugin_set, iter, plug_rec) {
		status_message(_("Setup plug-in: %s\n"), plug_rec->long_name);
		rc = plug_rec->functions.plugin->setup_evms_plugin(&engine_functions);
		if (rc == 0) {
			/* Handle special plug-ins. */
			switch (plug_rec->id) {
				case EVMS_DISK_PLUGIN_ID:
					{
						/* Start up the Local Disk Manager read cache. */
						plug_rec->functions.plugin->plugin_function(NULL, LDM_Start_Caching, NULL, NULL);
					}

					/*
					 * Save the pointer to the Local Disk Manager Plug-in for
					 * future reference.
					 */
					local_disk_manager = plug_rec;
					break;

				case EVMS_REPLACE_PLUGIN_ID:
					replace_plugin = plug_rec;
					break;

				default:
					break;

			}

		} else {
			if (rc != E_NOLOAD) {
				engine_user_message(NULL, NULL,
						    _("The plug-in %s in module %s failed to load.  The plug-in's setup_evms_plugin() function failed with error code %d: %s.\n"),
						    plug_rec->short_name, plug_rec->so_record->name, rc, strerror(rc));
			}

			release_plugin(plug_rec);
		}
	}

	destroy_list(plugin_set);

	rc = engine_get_plugin_list(EVMS_SEGMENT_MANAGER, 0, &plugin_set);

	if (rc != 0) {
		LOG_SERIOUS("Failed to get a list of segment manager plug-ins.  Error code is %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(plugin_set, iter, plug_rec) {
		status_message(_("Setup plug-in: %s\n"), plug_rec->long_name);
		rc = plug_rec->functions.plugin->setup_evms_plugin(&engine_functions);
		if (rc == 0) {
			/* Handle special plug-ins. */
			switch (plug_rec->id) {
				case EVMS_CSM_PLUGIN_ID:
					cluster_segment_manager = plug_rec;
					break;

				default:
					break;
			}

		} else {
			if (rc != E_NOLOAD) {
				engine_user_message(NULL, NULL,
						    _("The plug-in %s in module %s failed to load.  The plug-in's setup_evms_plugin() function failed with error code %d: %s.\n"),
						    plug_rec->short_name, plug_rec->so_record->name, rc, strerror(rc));
			}

			release_plugin(plug_rec);
		}
	}

	destroy_list(plugin_set);

	rc = engine_get_plugin_list(EVMS_REGION_MANAGER, 0, &plugin_set);

	if (rc != 0) {
		LOG_SERIOUS("Failed to get a list of region manager plug-ins.  Error code is %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(plugin_set, iter, plug_rec) {
		status_message(_("Setup plug-in: %s\n"), plug_rec->long_name);
		rc = plug_rec->functions.plugin->setup_evms_plugin(&engine_functions);
		if (rc != 0) {
			if (rc != E_NOLOAD) {
				engine_user_message(NULL, NULL,
						    _("The plug-in %s in module %s failed to load.  The plug-in's setup_evms_plugin() function failed with error code %d: %s.\n"),
						    plug_rec->short_name, plug_rec->so_record->name, rc, strerror(rc));
			}

			release_plugin(plug_rec);
		}
	}

	destroy_list(plugin_set);

	rc = engine_get_plugin_list(EVMS_FEATURE, 0, &plugin_set);

	if (rc != 0) {
		LOG_SERIOUS("Failed to get a list of EVMS featur plug-ins.  Error code is %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(plugin_set, iter, plug_rec) {
		status_message(_("Setup plug-in: %s\n"), plug_rec->long_name);
		rc = plug_rec->functions.plugin->setup_evms_plugin(&engine_functions);
		if (rc != 0) {
			if (rc != E_NOLOAD) {
				engine_user_message(NULL, NULL,
						    _("The plug-in %s in module %s failed to load.  The plug-in's setup_evms_plugin() function failed with error code %d: %s.\n"),
						    plug_rec->short_name, plug_rec->so_record->name, rc, strerror(rc));
			}

			release_plugin(plug_rec);
		}
	}

	destroy_list(plugin_set);

	rc = engine_get_plugin_list(EVMS_ASSOCIATIVE_FEATURE, 0, &plugin_set);

	if (rc != 0) {
		LOG_SERIOUS("Failed to get a list of EVMS associative feature plug-ins.  Error code is %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(plugin_set, iter, plug_rec) {
		status_message(_("Setup plug-in: %s\n"), plug_rec->long_name);
		rc = plug_rec->functions.plugin->setup_evms_plugin(&engine_functions);
		if (rc != 0) {
			if (rc != E_NOLOAD) {
				engine_user_message(NULL, NULL,
						    _("The plug-in %s in module %s failed to load.  The plug-in's setup_evms_plugin() function failed with error code %d: %s.\n"),
						    plug_rec->short_name, plug_rec->so_record->name, rc, strerror(rc));
			}

			release_plugin(plug_rec);
		}
	}

	destroy_list(plugin_set);

	rc = engine_get_plugin_list(EVMS_FILESYSTEM_INTERFACE_MODULE, 0, &plugin_set);

	if (rc != 0) {
		LOG_SERIOUS("Failed to get a list of file system interface plug-ins.  Error code is %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(plugin_set, iter, plug_rec) {
		status_message(_("Setup plug-in: %s\n"), plug_rec->long_name);
		rc = plug_rec->functions.fsim->setup_evms_plugin(&engine_functions);
		if (rc != 0) {
			if (rc != E_NOLOAD) {
				engine_user_message(NULL, NULL,
						    _("The plug-in %s in module %s failed to load.  The plug-in's setup_evms_plugin() function failed with error code %d: %s.\n"),
						    plug_rec->short_name, plug_rec->so_record->name, rc, strerror(rc));
			}

			release_plugin(plug_rec);
		}
	}

	destroy_list(plugin_set);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 *   This routine is called to load plug-ins for EVMS. Since we have
 *   no idea which plug-ins are available on the system we load them
 *   in the following manner:
 *
 *     (1) Try and open the "well known" EVMS Plugin directory ... /usr/lib/evms/
 *     (2) For each file found in the plug-in directory we'll call load_plugin()
 *         which can recognize plug-ins and load them.
 *     (3) If load_plugin() returns success we'll add the plug-in to the plugin
 *         list_anchor_t by inserting it as an object.
 *
 *   Finally, If we find at least 1 plug-in we'll return success.
 */

int load_plugins() {
	DIR           * pDir;
	struct dirent * pDirent;
	int             rc = 0;
	uint            plugins_loaded = 0;
	plugin_record_t * * ppPlugRec;
	char            so_name[256];

	LOG_PROC_ENTRY();

	/* Check if the plug-ins have been loaded already. */
	if (list_empty(&plugins_list)) {
		pDir = opendir(PluginDirectory);
		if (pDir != NULL) {

			pDirent = readdir(pDir);
			while ((rc == 0) && (pDirent != NULL)) {

				if ((strcmp(pDirent->d_name,".") != 0) &&
				    (strcmp(pDirent->d_name,"..") != 0)) {
					so_record_t * so_record;

					strcpy(so_name, PluginDirectory);
					strcat(so_name, "/");
					strcat(so_name, pDirent->d_name);

					LOG_DETAILS("Module to load is %s\n", so_name);

					rc = load_module(so_name, &so_record);

					if (rc == 0) {
						if (so_record != NULL) {

							/* Check for an array of plug-in pointers. */
							ppPlugRec = (plugin_record_t * *) get_sym_addr(so_record->handle, "evms_plugin_records");

							if (ppPlugRec != NULL) {

								/* Load the plug-in(s) in the module. */
								rc = load_module_plugins(so_record, ppPlugRec, &plugins_list);

							} else {
								/*
								 * The module doesn't export the published name
								 * for a plug-in record table.  It must not be
								 * an Engine plug-in.  Unload the module.
								 */
								engine_user_message(NULL, NULL,
										    _("Failed to load module %s.  It does not export an \"evms_plugin_records\" variable.\n"),
										    so_record->name);
								unload_module(so_record);
							}

						} else {
							LOG_WARNING("load_module() failed.\n");
						}

					} else {
						/*
						 * Don't let module load failures stop the plug-in
						 * loading process.
						 */
						rc = 0;
					}
				}

				pDirent = readdir(pDir);
			}

			closedir(pDir);

		} else {
			LOG_WARNING("Could not open PluginDirectory %s.\n", PluginDirectory);
			rc = ENOENT;
		}

		/* Find out how many plug-ins got loaded. */
		plugins_loaded = list_count(&plugins_list);

		LOG_DEBUG("Loaded %d plug-in(s).\n", plugins_loaded);
		LOG_DEBUG("Return code is %d.\n", rc);
		if (rc == 0) {
			if (plugins_loaded == 0) {
				rc = ENOENT;
			}
		} else {
			if (plugins_loaded != 0) {
				unload_plugins();
			}
		}

		if (rc == 0) {
			rc = setup_plugins();
		}

	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}
