/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-msg-viewer-app.c,v 1.21 2004/11/14 18:22:13 hoa Exp $
 */

#include "etpan-msg-viewer-app.h"

#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <ncurses.h>
#include "etpan-viewer-common.h"
#include "etpan-msg-list-app.h"
#include "etpan-app.h"
#include "etpan-msg-new.h"
#include "etpan-msg-reply.h"
#include "etpan-subapp-thread.h"
#include "etpan-search-input.h"
#include "etpan-imf-helper.h"
#include "etpan-msg-supersedes.h"
#include "etpan-help-viewer.h"
#include <string.h>

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);
static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd);
static void handle_fd(struct etpan_subapp * app, fd_set * fds);

static struct etpan_subapp_driver etpan_msg_viewer_app_driver = {
  .name = "msg-viewer",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = set_fd,
  .handle_fd = handle_fd,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  /* msg */
  mailmessage * msg;
  struct mailfolder * folder;
  
  struct etpan_viewer_common_app_state common_state;
};

enum {
  ETPAN_FLAG_READ,
  ETPAN_FLAG_UNREAD,
  ETPAN_FLAG_DELETED,
  ETPAN_FLAG_FLAGGED,
};

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd)
{
  etpan_subapp_thread_set_fd(app, fds, maxfd);
}

static void handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  etpan_subapp_thread_handle_fd(app, fds);
}

static int reply_message(struct etpan_subapp * app, int reply_type);

static int bounce_message(struct etpan_subapp * app, int do_redirect);

static void modify_messages(struct etpan_subapp * app);

static int change_flags(struct etpan_subapp * app, int flag_type);

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  struct etpan_subapp * parent;
  
  state = app->data;
  
  etpan_viewer_common_handle_key(app, &state->common_state, key);

  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;

  case 'q':
  case KEY_LEFT:
    etpan_app_quit_subapp(app);
    break;
    
  case 'v':
  case KEY_RIGHT:
    parent = etpan_subapp_get_parent(app);
    if (parent != NULL) {
      if (strcmp(parent->driver->name, "msg-list") == 0)
        etpan_view_mime(parent, state->msg);
    }
#if 0
    etpan_msg_list_app_open_mimelist(app, state->folder, state->msg);
#endif
    break;

  case 'n':
    parent = etpan_subapp_get_parent(app);
    if (parent != NULL) {
      if (strcmp(parent->driver->name, "msg-list") == 0) {
        etpan_app_quit_subapp(app);
        etpan_msg_list_app_open_next(parent, 1);
      }
    }
    break;

  case 'p':
    parent = etpan_subapp_get_parent(app);
    if (parent != NULL) {
      if (strcmp(parent->driver->name, "msg-list") == 0) {
        etpan_app_quit_subapp(app);
        etpan_msg_list_app_open_next(parent, 0);
      }
    }
    break;

  case 'm':
    etpan_compose_new(app, state->folder);
    break;
    
  case 'a':
    reply_message(app, ETPAN_THREAD_REPLY);
    break;
    
  case 'r':
    reply_message(app, ETPAN_THREAD_REPLY_FOLLOWUP_TO);
    break;
    
  case 'g':
    reply_message(app, ETPAN_THREAD_REPLY_ALL);
    break;
    
  case 'f':
    reply_message(app, ETPAN_THREAD_REPLY_FORWARD);
    break;
    
  case 'd':
    /* delete */
    change_flags(app, ETPAN_FLAG_DELETED);
    /* go to next message */
    parent = etpan_subapp_get_parent(app);
    if (parent != NULL) {
      if (strcmp(parent->driver->name, "msg-list") == 0) {
        etpan_msg_list_app_open_next(parent, 1);
        etpan_app_quit_subapp(app);
      }
    }
    break;

  case 'F':
    change_flags(app, ETPAN_FLAG_FLAGGED);
    break;
    
  case KEY_CTRL('F'):
    reply_message(app, ETPAN_THREAD_REPLY_FORWARD_AS_ATTACHMENT);
    break;
    
  case KEY_CTRL('Z'):
    if (etpan_is_nntp_protocol(state->folder))
      modify_messages(app);
    break;

  case 'b':
    /* bounce message */
    bounce_message(app, 0);
    break;

  case 'B':
    /* redirect message */
    bounce_message(app, 1);
    break;
  }
}


#define TITLE_NO_MSG_ID "etPan! - message viewer"
#define TITLE_MSG_ID "etPan! - message viewer - "

static void set_title(struct etpan_subapp * app)
{
  char index_str[20];
  int title_set;
  struct app_state * state;
  mailmessage * msg;
  
  state = app->data;
  msg = state->msg;
  
  title_set = 0;
  if (msg != NULL) {
    if (msg->msg_index != 0) {
      char * title;
      size_t len;
      
      snprintf(index_str, sizeof(index_str), "%u", msg->msg_index);
      
      len = sizeof(TITLE_MSG_ID) + strlen(index_str);
      
      title = malloc(len);
      if (title != NULL) {
        snprintf(title, len, "%s%s", TITLE_MSG_ID, index_str);
        etpan_subapp_set_title(app, title);
        free(title);
        title_set = 1;
      }
    }
  }
  
  if (!title_set) {
    etpan_subapp_set_title(app, TITLE_NO_MSG_ID);
  }
}

static int display_init(struct etpan_subapp * app)
{
  set_title(app);
  return etpan_app_subapp_display_init(app);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_viewer_common_set_color(app, &state->common_state);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_viewer_common_display(app, &state->common_state, w, "q: exit");
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->msg = NULL;
  state->folder = NULL;
  
  r = etpan_viewer_common_init(subapp, &state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;
  /* do nothing */
  etpan_viewer_common_done(subapp, &state->common_state);
  free(state);
}

struct etpan_subapp * etpan_msg_viewer_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_msg_viewer_app_driver);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;
  
  state = app->data;

#if 0
  while (1) {
    struct etpan_subapp * subapp;
    
    subapp = etpan_app_find_child_subapp(app, 1);
    if (subapp != NULL)
      etpan_app_leave_subapp(subapp, new_app);
    else
      break;
  }
#endif
  
  etpan_msg_viewer_app_flush(app);
}

int etpan_msg_viewer_app_set_msg(struct etpan_subapp * app,
    struct mailfolder * folder,
    mailmessage * msg,
    char * filename, FILE * f, carray * attr)
{
  struct app_state * state;
  int r;
  int res;
  
  state = app->data;
  
  etpan_msg_viewer_app_flush(app);
  
  r = etpan_viewer_common_set_file(&state->common_state,
      filename, f, attr,
      ETPAN_VIEWER_FREE_FLAGS_CLOSE | ETPAN_VIEWER_FREE_FLAGS_REMOVE);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  r = etpan_queue_ref_msg_mime(app, msg);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "view mime structure - not enough memory"));
    res = r;
    goto flush;
  }
  
  state->folder = folder;
  state->msg = msg;
  set_title(app);
  
  return NO_ERROR;

 flush:
  etpan_viewer_common_flush(&state->common_state);
 err:
  return res;
}

void etpan_msg_viewer_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;

  state = app->data;

  if (state->msg != NULL) {
    etpan_queue_unref_msg_mime(app, state->msg);
  }
  
  state->msg = NULL;
  state->folder = NULL;
  etpan_viewer_common_flush(&state->common_state);
}


mailmessage * etpan_msg_viewer_app_get_msg(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return state->msg;
}

struct mailfolder *
etpan_msg_viewer_app_get_folder(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return state->folder;
}

static int reply_message(struct etpan_subapp * app, int reply_type)
{
  mailmessage * msg;
  struct app_state * state;
  
  state = app->data;
  
  msg = state->msg;
  if (msg == NULL)
    return ERROR_INVAL;
  
  return etpan_reply_message(app, reply_type,
      msg, NULL, NULL, NULL, state->folder);
}

static void mark_one_as_deleted(struct etpan_subapp * app,
    struct mailmessage * msg)
{
  struct mail_flags * flags;
  
  flags = msg->msg_flags;
  if (flags != NULL) {
    carray * msglist;
    
    flags->fl_flags &= ~MAIL_FLAG_NEW;
    flags->fl_flags |= MAIL_FLAG_SEEN;
    flags->fl_flags ^= MAIL_FLAG_DELETED;
  }
}

static void mark_one_as_flagged(struct etpan_subapp * app,
    struct mailmessage * msg)
{
  struct mail_flags * flags;
  
  flags = msg->msg_flags;
  if (flags != NULL) {
    carray * msglist;
    
    flags->fl_flags ^= MAIL_FLAG_FLAGGED;
  }
}

static int check_msg(struct etpan_subapp * app, mailmessage * msg)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_subapp_thread_msg_op_add(app,
      THREAD_ID_MSGLIST_CHECK_MSG, ETPAN_THREAD_MESSAGE_CHECK,
      msg, NULL,
      NULL,
      NULL, NULL, NULL);
}

static int change_flags(struct etpan_subapp * app, int flag_type)
{
  int no_sel;
  struct app_state * state;
  int r;
  
  state = app->data;

  switch (flag_type) {
  case ETPAN_FLAG_DELETED:
    mark_one_as_deleted(app, state->msg);
    break;
  case ETPAN_FLAG_FLAGGED:
    mark_one_as_flagged(app, state->msg);
    break;
  }
  
  r = check_msg(app, state->msg);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "change message flags - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}

/* duplicated from etpan-msg-list-app.c */
/* cancel / supersedes */

static void modify_upcall(struct etpan_subapp * input_app, int valid,
    void * data);

static void modify_messages(struct etpan_subapp * app)
{
  struct etpan_subapp * input_app;
  int r;
  struct app_state * state;

  state = app->data;

  input_app = etpan_app_find_subapp(app->app, "search-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_search_input_new(app->app);
    if (input_app == NULL)
      return;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  r = etpan_search_input_set(input_app,
      "(s)upersedes/c(ancel): ", 1, NULL,
      0, modify_upcall, app);
  if (r != NO_ERROR)
    return;
  
  etpan_app_switch_subapp(input_app, 0);
}

enum {
  MODIFY_TYPE_CANCEL,
  MODIFY_TYPE_SUPERSEDES,
};

static void modify_upcall(struct etpan_subapp * input_app, int valid,
    void * data)
{
  char * val;
  struct etpan_subapp * app;
  struct app_state * state;
  struct mailfolder * folder;
  int r;
  int modify_type;
  mailmessage * msg;
  
  app = data;

  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    ETPAN_APP_LOG((app->app, "supersedes/cancel message - cancel"));
    goto err;
  }
  
  state = app->data;
  
  val = etpan_search_input_get_value(input_app);
  
  switch (val[0]) {
  case 's':
    modify_type = MODIFY_TYPE_SUPERSEDES;
    break;
  case 'c':
    modify_type = MODIFY_TYPE_CANCEL;
    break;
  default:
    ETPAN_APP_LOG((app->app, "supersedes/cancel message - invalid operation"));
    goto err;
  } 
  
  folder = state->folder;
  msg = state->msg;
  if (msg == NULL) {
    ETPAN_APP_LOG((app->app, "supersedes/cancel message - invalid operation"));
    goto err;
  }
  
  switch (modify_type) {
  case MODIFY_TYPE_CANCEL:
    r = etpan_cancel_message(app, msg, NULL);
    break;
  case MODIFY_TYPE_SUPERSEDES:
    r = etpan_supersedes_message(app, msg, NULL);
    break;
  default:
    r = ERROR_INVAL;
    break;
  }
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "supersedes/cancel message - failed"));
    goto err;
  }
  
  etpan_app_quit_subapp(input_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(input_app);
  return;
}

static int bounce_message(struct etpan_subapp * app, int do_redirect)
{
  mailmessage * msg;
  struct app_state * state;
  
  state = app->data;
  
  msg = state->msg;
  if (msg == NULL)
    return ERROR_INVAL;
  
  return etpan_bounce_message(app, msg, NULL, do_redirect);
}


#define HELP_TEXT \
"\
Help for message viewer\n\
-----------------------\n\
\n\
This application will let you view the content of a message.\n\
\n\
- left, right,\n\
  Ctrl-W,\n\
  Ctrl-X      : switch between applications\n\
\n\
- h, l        : scroll horizontally\n\
- up, down    : scroll vertically\n\
\n\
- v           : view MIME structure of the message\n\
\n\
- m           : compose new message\n\
- a           : reply to the author\n\
- r           : reply to the message\n\
- g           : reply to all recipient of the message\n\
- f           : forward the message\n\
- F           : forward the message as attachment\n\
- b           : bounce the message\n\
- B           : redirect (Subject line is modified)\n\
- Ctrl-Z      : cancel or supersedes news articles\n\
\n\
- n,p         : next/previous unread message\n\
\n\
- q           : exit\n\
\n\
- ?           : help\n\
- Ctrl-L      : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
