/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-msg-supersedes.c,v 1.11 2004/03/18 01:32:03 hoa Exp $
 */

#include "etpan-msg-supersedes.h"
#include "etpan-errors.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include <string.h>
#include <stdlib.h>
#include "etpan-msg-new.h"
#include "etpan-subapp-thread.h"
#include "etpan-tools.h"
#include "etpan-imf-helper.h"
#include <unistd.h>
#include "etpan-mime-tools.h"

static void thread_supersedes_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct etpan_message_mime_copy_result * result;
  int r;
  struct mailmime * mime;
  struct mailimf_single_fields single_fields;
  char * name;
  char * value;
  int len;
  struct mailimf_field * field;
  int res;
  clistiter * cur;
  struct mailimf_fields * tmp_fields;
  struct mailmime * edit_part;

  /* handle end of operation */
  
  if (op->err != NO_ERROR) {
    mailmessage * msg;
        
    msg = op->data.mailaccess.msg;
    
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app,
                        "supersedes message - error while retrieving message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app,
                        "supersedes message - error while retrieving message %i",
                        msg->msg_index));
    return;
  }
  
  result = op->result;
  
  mime = result->mime;

  if (mime->mm_type != MAILMIME_MESSAGE) {
    ETPAN_APP_LOG((app->app, "supersedes message - invalid message"));
    res = ERROR_INVAL;
    goto free_mime;
  }
  
  if (mime->mm_data.mm_message.mm_fields == NULL) {
    ETPAN_APP_LOG((app->app, "supersedes message - invalid message"));
    res = ERROR_INVAL;
    goto free_mime;
  }
  
  /* add Supersedes field */
  
  mailimf_single_fields_init(&single_fields,
      mime->mm_data.mm_message.mm_fields);
  
  if (single_fields.fld_message_id == NULL) {
    ETPAN_APP_LOG((app->app, "supersedes message - invalid message"));
    res = ERROR_INVAL;
    goto free_mime;
  }
  
  name = strdup("Supersedes");
  if (name == NULL) {
    ETPAN_APP_LOG((app->app, "supersedes message - not enough memory"));
    res = ERROR_MEMORY;
    goto free_mime;
  }
  
  len = strlen(single_fields.fld_message_id->mid_value);
  value = malloc(len + 3);
  if (value == NULL) {
    ETPAN_APP_LOG((app->app, "supersedes message - not enough memory"));
    free(name);
    res = ERROR_MEMORY;
    goto free_mime;
  }
  strcpy(value + 1, single_fields.fld_message_id->mid_value);
  value[0] = '<';
  value[len + 1] = '>';
  value[len + 2] = 0;
  
  field = mailimf_field_new_custom(name, value);
  if (field == NULL) {
    ETPAN_APP_LOG((app->app, "supersedes message - not enough memory"));
    free(value);
    free(name);
    res = ERROR_MEMORY;
    goto free_mime;
  }
  
  r = mailimf_fields_add(mime->mm_data.mm_message.mm_fields, field);
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_APP_LOG((app->app, "supersedes message - not enough memory"));
    mailimf_field_free(field);
    res = ERROR_MEMORY;
    goto free_mime;
  }
  
  cur = clist_begin(mime->mm_data.mm_message.mm_fields->fld_list);
  while (cur != NULL) {
    struct mailimf_field * field;
    
    field = clist_content(cur);
    
    switch (field->fld_type) {
    case MAILIMF_FIELD_ORIG_DATE:
      cur = clist_delete(mime->mm_data.mm_message.mm_fields->fld_list, cur);
      mailimf_field_free(field);
      break;
      
    case MAILIMF_FIELD_MESSAGE_ID:
      cur = clist_delete(mime->mm_data.mm_message.mm_fields->fld_list, cur);
      mailimf_field_free(field);
      break;
      
    default:
      cur = clist_next(cur);
    }
  }
  
  /* add new field values */
  
  tmp_fields = mailimf_fields_new_with_data(NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL);
  if (tmp_fields == NULL) {
    ETPAN_APP_LOG((app->app, "supersedes message - not enough memory"));
    res = ERROR_MEMORY;
    goto free_mime;
  }
  
  clist_concat(mime->mm_data.mm_message.mm_fields->fld_list,
      tmp_fields->fld_list);
  mailimf_fields_free(tmp_fields);

  edit_part = etpan_message_get_first_text_part(mime);
  
  /* run message editor */
  r = etpan_compose(app, op->data.mailaccess.folder,
      mime, edit_part, 1, NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "supersedes message - not enough memory"));
    etpan_message_mime_clear(mime);
    mailmime_free(result->mime);
  }
  
  free(result);
  op->result = NULL;
  
  return;
  
 free_mime:
  etpan_message_mime_clear(mime);
  mailmime_free(mime);
}



int etpan_supersedes_message(struct etpan_subapp * app,
    mailmessage * msg,
    struct mailmime * mime)
{
  int r;

  ETPAN_APP_LOG((app->app, "supersedes message ..."));
  
  if (msg->msg_uid != NULL)
    ETPAN_APP_LOG((app->app, "retrieving message %i, %s",
                      msg->msg_index, msg->msg_uid));
  else
    ETPAN_APP_LOG((app->app, "retrieving message %i", msg->msg_index));
  
  r = etpan_subapp_thread_msg_op_add(app, THREAD_ID_COMPOSE_SUPERSEDES,
      ETPAN_THREAD_MESSAGE_MIME_COPY,
      msg, mime,
      NULL,
      thread_supersedes_callback, NULL,
      NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "supersedes message - not enough memory"));
    return ERROR_MEMORY;
  }
  
  return NO_ERROR;
}

static int etpan_cancel_get_fields(struct etpan_app * app,
    struct mailfolder * folder,
    struct mailimf_fields * fields,
    struct mailimf_fields ** result)
{
  struct mailimf_fields * new_fields;
  struct mailimf_field * field;
  struct mailimf_mailbox_list * from;
  char * subject;
  int subject_len;
  int r;
  char * msg_id;
  char * value;
  int value_len;
  char * name;
  struct mailimf_date_time * date;
  clistiter * cur;
  struct etpan_account_info * account;
  struct mailimf_single_fields single_fields;
  
  /* build headers */
  
  account = etpan_vfolder_get_account(app->config.vfolder_config,
      app->config.account_config, folder);
  
  if (account == NULL) {
    from = NULL;
  }
  else {
    from = etpan_get_from_field(app, account);
    if (from == NULL)
      goto err;
  }
  
  /* old compatibility */
  
  mailimf_single_fields_init(&single_fields, fields);
  
  msg_id = single_fields.fld_message_id->mid_value;
  subject_len = strlen(msg_id) + strlen("cmsg cancel <>") + 1;
  subject = malloc(subject_len);
  if (subject == NULL)
    goto free_from;
  
  snprintf(subject, subject_len, "cmsg cancel <%s>", msg_id);
  
  /* standard headers */
  
  date = mailimf_get_current_date();
  if (date == NULL)
    goto free_subject;

  new_fields = mailimf_fields_new_with_data_all(date, from, /* from */
      NULL /* sender */, NULL /* reply-to */, NULL /* to */,
      NULL, NULL, NULL, NULL, NULL, subject);
  if (new_fields == NULL) {
    mailimf_date_time_free(date);
    goto free_date;
  }
  
  value_len = strlen(msg_id) + strlen("cancel <>") + 1;
  value = malloc(value_len);
  if (subject == NULL)
    goto free_fields;
  
  snprintf(value, value_len, "cancel <%s>", msg_id);
  
  name = strdup("Control");
  if (name == NULL) {
    free(value);
    goto free_fields;
  }
  
  field = mailimf_field_new_custom(name, value);
  if (field == NULL) {
    free(name);
    free(value);
    goto free_fields;
  }
  
  r = mailimf_fields_add(new_fields, field);
  if (r != MAILIMF_NO_ERROR) {
    mailimf_field_free(field);
    goto free_fields;
  }
  
  value = NULL;
  for(cur = clist_begin(fields->fld_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    field = clist_content(cur);
    if (field->fld_type == MAILIMF_FIELD_OPTIONAL_FIELD) {
      if (strcasecmp(field->fld_data.fld_optional_field->fld_name,
              "newsgroups") == 0) {
        value = field->fld_data.fld_optional_field->fld_value;
      }
    }
  }
  
  if (value == NULL) {
    mailimf_fields_free(fields);
    goto free_fields;
  }
  
  name = strdup("Newsgroups");
  if (name == NULL) {
    mailimf_fields_free(fields);
    free(value);
    goto free_fields;
  }
  
  value = strdup(value);
  if (value == NULL) {
    mailimf_fields_free(fields);
    free(name);
    goto free_fields;
  }
  
  field = mailimf_field_new_custom(name, value);
  if (field == NULL) {
    free(name);
    free(value);
    goto free_fields;
  }
  
  r = mailimf_fields_add(new_fields, field);
  if (r != MAILIMF_NO_ERROR) {
    mailimf_field_free(field);
    goto free_fields;
  }
  
  r = etpan_set_xmailer(new_fields, 1);
  if (r != NO_ERROR)
    goto free_fields;

  * result = new_fields;
  
  return NO_ERROR;

 free_fields:
  mailimf_fields_free(new_fields);
  goto err;
 free_date:
  mailimf_date_time_free(date);
 free_subject:
  free(subject);
 free_from:
  if (from != NULL)
    mailimf_mailbox_list_free(from);
 err:
  return ERROR_MEMORY;
}



/*
  thread_cancel_callback()

  free of message is present twice. This is in two different cases.
*/

#define CANCEL_TEXT "message cancelled by etPan - new generation"

static void thread_cancel_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct etpan_message_fetch_result * result;
  int r;
  char filename[PATH_MAX];
  FILE * f;
  size_t written;
  struct mailimf_fields * new_fields;
  struct mailmime * mime;
  int col;
  size_t cur_token;
  struct mailimf_fields * fields;

  /* handle end of operation */
  
  if (op->err != NO_ERROR) {
    mailmessage * msg;
    
    msg = op->data.mailaccess.msg;
    
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app,
                        "cancel message - error while retrieving message %i, %s %i",
                        msg->msg_index, msg->msg_uid, op->err));
    else
      ETPAN_APP_LOG((app->app,
                        "cancel message - error while retrieving message %i",
                        msg->msg_index));
    return;
  }

  result = op->result;
  
  cur_token = 0;
  r = mailimf_fields_parse(result->content, result->len,
      &cur_token, &fields);
  
  if (r != NO_ERROR)
    r = etpan_new_get_fields(app->app, op->data.mailaccess.folder,
        NULL, &new_fields);
  else
    r = etpan_cancel_get_fields(app->app, op->data.mailaccess.folder,
        fields, &new_fields);
  
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "cancel message - not enough memory"));
    mailimf_fields_free(fields);
    goto exit;
  }
  
  mailimf_fields_free(fields);
  
  f = etpan_get_tmp_file(filename, sizeof(filename));
  if (f == NULL) {
    ETPAN_APP_LOG((app->app, "cancel message - failed"));
    mailimf_fields_free(new_fields);
    goto exit;
  }
  
  written = fwrite(CANCEL_TEXT, 1, sizeof(CANCEL_TEXT) - 1, f);
  if (written != sizeof(CANCEL_TEXT) - 1) {
    ETPAN_APP_LOG((app->app, "cancel message - failed"));
    fclose(f);
    unlink(filename);
    mailimf_fields_free(new_fields);
    goto exit;
  }
  
  fclose(f);
  
  r = etpan_build_simple_mime(app->app, new_fields, filename,
      &mime, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "cancel message - not enough memory"));
    unlink(filename);
    mailimf_fields_free(new_fields);
    goto exit;
  }
  
  f = etpan_get_tmp_file(filename, sizeof(filename));
  if (f == NULL) {
    ETPAN_APP_LOG((app->app, "cancel message - failed"));
    goto free_mime;
  }
  col = 0;
  r = mailmime_write(f, &col, mime);
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_APP_LOG((app->app, "cancel message - failed"));
    fclose(f);
    unlink(filename);
    goto free_mime;
  }
  fclose(f);
  
  r = etpan_message_send(app, op->data.mailaccess.folder,
      mime->mm_data.mm_message.mm_fields, filename);
  if (r != NO_ERROR) {
    /*
      leave file "filename" so that the user can get his file
      when this operation failed
    */
    ETPAN_APP_LOG((app->app, "message send - failed"));
    unlink(filename);
    goto free_mime;
  }
  
  ETPAN_APP_LOG((app->app, "cancel message queued"));
  
  unlink(filename);
  etpan_message_mime_clear(mime);
  mailmime_free(mime);
  
  /* successful finished */
  goto exit;
  
 free_mime:
  etpan_message_mime_clear(mime);
  mailmime_free(mime);
 exit:
  etpan_subapp_thread_op_add(app, THREAD_ID_COMPOSE_CANCEL_FREE,
      ETPAN_THREAD_MESSAGE_FETCH_RESULT_FREE,
      op->data.mailaccess.storage, op->data.mailaccess.folder,
      op->data.mailaccess.msg, NULL,
      result->content,
      NULL, NULL, NULL);
  free(result);
}

#if 0
static void thread_cancel_body_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  int r;
  struct mailmime * mime;
  struct mailstorage * storage;
  struct mailfolder * folder;
  mailmessage * msg;

  folder = op->data.mailaccess.folder;
  storage = op->data.mailaccess.storage;
  msg = op->data.mailaccess.msg;
  
  /* handle end of operation */
  
  if (op->err != NO_ERROR) {
    msg = op->data.mailaccess.msg;
    
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app,
                        "cancel message - error while retrieving message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app,
                        "cancel message - error while retrieving message %i",
                        msg->msg_index));
    return;
  }
  
  mime = msg->mime;
  
  r = etpan_subapp_thread_op_add(app, THREAD_ID_COMPOSE_CANCEL,
      ETPAN_THREAD_MESSAGE_FETCH_SECTION_HEADER,
      storage, folder,
      msg, mime,
      NULL,
      thread_cancel_callback, NULL,
      NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "cancel message - not enough memory"));
    return;
  }
}
#endif

int etpan_cancel_message(struct etpan_subapp * app,
    mailmessage * msg,
    struct mailmime * mime)
{
  int r;
  
  if (msg->msg_uid != NULL)
    ETPAN_APP_LOG((app->app, "retrieving header %i, %s",
                      msg->msg_index, msg->msg_uid));
  else
    ETPAN_APP_LOG((app->app, "retrieving header %i", msg->msg_index));
  
  ETPAN_APP_LOG((app->app, "cancel message ..."));
  
  if (mime != NULL) {
    r = etpan_subapp_thread_msg_op_add(app, THREAD_ID_COMPOSE_CANCEL,
        ETPAN_THREAD_MESSAGE_FETCH_SECTION_HEADER,
        msg, mime,
        NULL,
        thread_cancel_callback, NULL,
        NULL);
  }
  else {
    r = etpan_subapp_thread_msg_op_add(app, THREAD_ID_COMPOSE_CANCEL,
        ETPAN_THREAD_MESSAGE_FETCH_HEADER,
        msg, NULL,
        NULL,
        thread_cancel_callback, NULL,
        NULL);
  }
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "cancel message - not enough memory"));
    return ERROR_MEMORY;
  }
  
  return NO_ERROR;
}



/* extracted/modified from etpan-msg-reply.c - begin */

struct compose_bounce_action_arg {
  struct etpan_subapp * app;
  struct mailfolder * folder;
  mailmessage * msg;
};

/* extracted/modified from etpan-msg-list-app.c - begin */

static int check_message(struct etpan_subapp * app,
    mailmessage * msg)
{
  return etpan_subapp_thread_msg_op_add(app,
      THREAD_ID_COMPOSE_CHECK_MSG, ETPAN_THREAD_MESSAGE_CHECK,
      msg, NULL,
      NULL,
      NULL, NULL, NULL);
}

/* extracted/modified from etpan-msg-list-app.c - end */


static void compose_bounce_action(void * data)
{
  struct compose_bounce_action_arg * arg;
  struct etpan_subapp * app;
  struct mailfolder * folder;
  mailmessage * msg;
  int r;

  arg = data;
  
  app = arg->app;
  folder = arg->folder;
  msg = arg->msg;
  free(arg);
  
  if (msg->msg_flags != NULL) {
    msg->msg_flags->fl_flags |= MAIL_FLAG_FORWARDED;
    
    r = check_message(app, msg);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "flag message - not enough memory"));
    }
  }
  
  etpan_queue_unref_msg(app, msg);
}

static void compose_bounce_action_cancel(void * data)
{
  struct compose_bounce_action_arg * arg;
  struct etpan_subapp * app;
  struct mailfolder * folder;
  mailmessage * msg;
  
  arg = data;
  
  app = arg->app;
  folder = arg->folder;
  msg = arg->msg;
  free(arg);
  
  etpan_queue_unref_msg(app, msg);
}

/* extracted/modified from etpan-msg-reply.c - end */


static void thread_bounce_handle_cancel(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  etpan_queue_unref_msg(app, op->data.mailaccess.msg);
}

static void thread_bounce_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct etpan_message_mime_copy_result * result;
  int r;
  struct mailmime * mime;
  struct mailimf_single_fields single_fields;
  clistiter * cur;
  int do_redirect;
  struct mailimf_fields * tmp_fields;
  struct compose_bounce_action_arg * bounce_action_arg;

  /* handle end of operation */
  
  if (op->err != NO_ERROR) {
    mailmessage * msg;
        
    msg = op->data.mailaccess.msg;
    
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app,
                        "bounce message - error while retrieving message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app,
                        "bounce message - error while retrieving message %i",
                        msg->msg_index));
    goto unref;
  }
  
  do_redirect = 0;
  if (app_op_type == THREAD_ID_COMPOSE_REDIRECT)
    do_redirect = 1;

  result = op->result;
  
  mime = result->mime;
  
  if (mime->mm_type != MAILMIME_MESSAGE) {
    ETPAN_APP_LOG((app->app, "supersedes message - invalid message"));
    goto free_mime;
  }
  
  if (mime->mm_data.mm_message.mm_fields == NULL) {
    ETPAN_APP_LOG((app->app, "bounce message - invalid message"));
    goto free_mime;
  }
  
  /* change subject */
  
  mailimf_single_fields_init(&single_fields,
      mime->mm_data.mm_message.mm_fields);
  
  if (do_redirect) {
    
    /* create a new subject field if it does does exist */
    if (single_fields.fld_subject == NULL) {
      struct mailimf_subject * subject;
      char * value;
      struct mailimf_field * field;
      
      value = strdup("");
      if (value == NULL) {
        goto free_mime;
      }
      subject = mailimf_subject_new(value);
      if (subject == NULL) {
        free(value);
        goto free_mime;
      }
      
      field = mailimf_field_new(MAILIMF_FIELD_SUBJECT,
          NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL,
          NULL, NULL,
          subject,
          NULL, NULL, NULL);
      if (field == NULL) {
        mailimf_subject_free(subject);
        goto free_mime;
      }
      
      r = clist_append(mime->mm_data.mm_message.mm_fields->fld_list, field);
      if (r < 0) {
        mailimf_field_free(field);
        goto free_mime;
      }
      
      mailimf_single_fields_init(&single_fields,
          mime->mm_data.mm_message.mm_fields);
    }
    
    if (single_fields.fld_subject->sbj_value != NULL) {
      char * name;
      char * str;
      struct etpan_account_info * account;
      
      name = NULL;
      account = etpan_vfolder_get_account(app->app->config.vfolder_config,
          app->app->config.account_config, op->data.mailaccess.folder);
      if (account != NULL) {
        if (account->name != NULL)
          name = account->name;
        else
          name = account->addr;
      }
      
      if (name == NULL)
        name = "[unknown]";
      
      str = malloc(strlen(single_fields.fld_subject->sbj_value)
          + strlen(" (by the way of )") + strlen(name));
      if (str == NULL) {
        goto free_mime;
      }
      
      strcpy(str, single_fields.fld_subject->sbj_value);
      strcat(str, " (by the way of ");
      strcat(str, name);
      strcat(str, ")");
      
      free(single_fields.fld_subject->sbj_value);
      single_fields.fld_subject->sbj_value = str;
    }
  }
  
  /* change some fields name */
  
  for(cur = clist_begin(mime->mm_data.mm_message.mm_fields->fld_list) ;
      cur != NULL ; cur = clist_next(cur)) {
    struct mailimf_field * field;
    int add_x;
    
    add_x = 0;
    field = clist_content(cur);
    
    if (strcasecmp(field->fld_data.fld_optional_field->fld_name,
            "Newsgroups") == 0)
      add_x = 1;
    
    if ((strcasecmp(field->fld_data.fld_optional_field->fld_name,
             "Received") == 0)
        || (strcasecmp(field->fld_data.fld_optional_field->fld_name,
                "Return-Path") == 0)
        || (strcasecmp(field->fld_data.fld_optional_field->fld_name,
                "Delivered-To") == 0)
        || (strncasecmp(field->fld_data.fld_optional_field->fld_name,
                "Resent-", 7) == 0))
      add_x = 1;
    
    if (add_x) {
      char * new_name;
      
      new_name = malloc(strlen(field->fld_data.fld_optional_field->fld_name) + 3);
      if (new_name == NULL)
        goto free_mime;
      strcpy(new_name, "X-");
      strcat(new_name, field->fld_data.fld_optional_field->fld_name);
      free(field->fld_data.fld_optional_field->fld_name);
      field->fld_data.fld_optional_field->fld_name = new_name;
    }
  }
  
  /* add new field values */
  
  r = etpan_new_get_resent_fields(app->app,
      op->data.mailaccess.folder, NULL, &tmp_fields);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "bounce message - not enough memory"));
    goto free_mime;
  }
  
  clist_concat(mime->mm_data.mm_message.mm_fields->fld_list,
      tmp_fields->fld_list);
  mailimf_fields_free(tmp_fields);

  /* set bounce action */
  
  bounce_action_arg = malloc(sizeof(* bounce_action_arg));
  if (bounce_action_arg == NULL) {
    ETPAN_APP_LOG((app->app, "bounce message - not enough memory"));
    goto free_mime;
  }
  
  bounce_action_arg->app = app;
  bounce_action_arg->folder = op->data.mailaccess.folder;
  bounce_action_arg->msg = op->data.mailaccess.msg;
  
  /* run message editor */
  r = etpan_compose(app, op->data.mailaccess.folder, mime, NULL, 0,
      compose_bounce_action, bounce_action_arg,
      compose_bounce_action_cancel);
  
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "bounce message - not enough memory"));
    free(bounce_action_arg);
    goto free_mime;
  }
  
  free(result);
  op->result = NULL;
  
  return;
  
 free_mime:
  etpan_message_mime_clear(mime);
  mailmime_free(mime);
  free(op->result);
  op->result = NULL;
 unref:
  etpan_queue_unref_msg(app, op->data.mailaccess.msg);
}



int etpan_bounce_message(struct etpan_subapp * app,
    mailmessage * msg,
    struct mailmime * mime,
    int do_redirect)
{
  int r;
  int app_op_type;
  
  ETPAN_APP_LOG((app->app, "bounce message ..."));
  
  if (msg->msg_uid != NULL)
    ETPAN_APP_LOG((app->app, "retrieving message %i, %s",
                      msg->msg_index, msg->msg_uid));
  else
    ETPAN_APP_LOG((app->app, "retrieving message %i", msg->msg_index));
  
  if (do_redirect)
    app_op_type = THREAD_ID_COMPOSE_REDIRECT;
  else
    app_op_type = THREAD_ID_COMPOSE_BOUNCE;
  
  r = etpan_queue_ref_msg(app, msg);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "bounce message - not enough memory"));
    goto err;
  }
  
  r = etpan_subapp_thread_msg_op_add(app, app_op_type,
      ETPAN_THREAD_MESSAGE_MIME_COPY,
      msg, mime,
      NULL,
      thread_bounce_callback, NULL,
      thread_bounce_handle_cancel);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "bounce message - not enough memory"));
    goto unref;
  }
  
  return NO_ERROR;

 unref:
  etpan_queue_unref_msg(app, msg);
 err:
  return ERROR_MEMORY;
}





static void compose_reedit_action(void * data)
{
  struct compose_bounce_action_arg * arg;
  struct etpan_subapp * app;
  struct mailfolder * folder;
  mailmessage * msg;
  int r;

  arg = data;
  
  app = arg->app;
  folder = arg->folder;
  msg = arg->msg;
  free(arg);
  
  if (msg->msg_flags != NULL) {
    msg->msg_flags->fl_flags &= ~MAIL_FLAG_NEW;
    msg->msg_flags->fl_flags |= MAIL_FLAG_SEEN;
    msg->msg_flags->fl_flags |= MAIL_FLAG_DELETED;
    
    r = check_message(app, msg);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "flag message - not enough memory"));
    }
  }
  
  etpan_queue_unref_msg(app, msg);
}

static void compose_reedit_action_cancel(void * data)
{
  struct compose_bounce_action_arg * arg;
  struct etpan_subapp * app;
  struct mailfolder * folder;
  mailmessage * msg;
  
  arg = data;
  
  app = arg->app;
  folder = arg->folder;
  msg = arg->msg;
  free(arg);
  
  etpan_queue_unref_msg(app, msg);
}

static void thread_reedit_handle_cancel(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  etpan_queue_unref_msg(app, op->data.mailaccess.msg);
}

static void thread_reedit_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct etpan_message_mime_copy_result * result;
  int r;
  struct mailmime * mime;
  clistiter * cur;
  struct mailfolder * post_from_folder;
  struct compose_bounce_action_arg * reedit_action_arg;
  
  /* handle end of operation */
  
  if (op->err != NO_ERROR) {
    mailmessage * msg;
        
    msg = op->data.mailaccess.msg;
    
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app,
                        "reedit message - error while retrieving message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app,
                        "reedit message - error while retrieving message %i",
                        msg->msg_index));
    goto unref;
  }
  
  result = op->result;
  
  mime = result->mime;
  
  if (mime->mm_type != MAILMIME_MESSAGE) {
    ETPAN_APP_LOG((app->app, "reedit message - invalid message"));
    goto free_mime;
  }
  
  post_from_folder = NULL;
  for(cur = clist_begin(mime->mm_data.mm_message.mm_fields->fld_list) ;
      cur != NULL ; cur = clist_next(cur)) {
    struct mailimf_field * field;
    
    field = clist_content(cur);
    if (field->fld_type == MAILIMF_FIELD_OPTIONAL_FIELD) {
      if (strcasecmp(field->fld_data.fld_optional_field->fld_name,
              "X-etPan-Draft-Folder") == 0) {
        struct mailfolder * folder;
        
        folder = etpan_get_folder(app->app,
            field->fld_data.fld_optional_field->fld_value);
        if (folder != NULL)
          post_from_folder = folder;
        
        mailimf_field_free(field);
        clist_delete(mime->mm_data.mm_message.mm_fields->fld_list, cur);
        break;
      }
    }
  }
  
  if (post_from_folder == NULL) {
    ETPAN_APP_LOG((app->app, "reedit message - this is not a draft message"));
    goto free_mime;
  }
  
  /* set bounce action */
  
  reedit_action_arg = malloc(sizeof(* reedit_action_arg));
  if (reedit_action_arg == NULL) {
    ETPAN_APP_LOG((app->app, "bounce message - not enough memory"));
    goto free_mime;
  }
  
  reedit_action_arg->app = app;
  reedit_action_arg->folder = op->data.mailaccess.folder;
  reedit_action_arg->msg = op->data.mailaccess.msg;
  
  /* run message editor */
  r = etpan_compose(app, post_from_folder, mime, NULL, 0,
      compose_reedit_action, reedit_action_arg,
      compose_reedit_action_cancel);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "reedit message - not enough memory"));
    etpan_message_mime_clear(mime);
    mailmime_free(result->mime);
  }
  
  free(result);
  op->result = NULL;
  
  return;
  
 free_mime:
  etpan_message_mime_clear(mime);
  mailmime_free(mime);
  free(op->result);
  op->result = NULL;
 unref:
  etpan_queue_unref_msg(app, op->data.mailaccess.msg);
}



int etpan_reedit_message(struct etpan_subapp * app,
    mailmessage * msg,
    struct mailmime * mime)
{
  int r;
  
  ETPAN_APP_LOG((app->app, "reedit message ..."));
  
  if (msg->msg_uid != NULL)
    ETPAN_APP_LOG((app->app, "retrieving message %i, %s",
                      msg->msg_index, msg->msg_uid));
  else
    ETPAN_APP_LOG((app->app, "retrieving message %i", msg->msg_index));
  
  r = etpan_queue_ref_msg(app, msg);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "reedit message - not enough memory"));
    goto err;
  }
  
  r = etpan_subapp_thread_msg_op_add(app, THREAD_ID_COMPOSE_REEDIT,
      ETPAN_THREAD_MESSAGE_MIME_COPY,
      msg, mime,
      NULL,
      thread_reedit_callback, NULL,
      thread_reedit_handle_cancel);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "reedit message - not enough memory"));
    goto unref;
  }
  
  return NO_ERROR;
  
 unref:
  etpan_queue_unref_msg(app, msg);
 err:
  return ERROR_MEMORY;
}
