/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-msg-list-app.c,v 1.30 2004/12/27 15:27:38 hoa Exp $
 */

#include "etpan-msg-list-app.h"

#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-msg-params.h"
#include "etpan-errors.h"
#include <libetpan/mmapstring.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ncurses.h>
#include "etpan-folder-list-app.h"
#include "etpan-msg-renderer.h"
#include "etpan-msg-viewer-app.h"
#include "etpan-mime-list-app.h"
#include "etpan-cfg-vfolder.h"
#include <sys/types.h>
#include <regex.h>
#include "etpan-search-input.h"
#include "etpan-imf-helper.h"
#if 0
#include "etpan-mime-message-driver.h"
#endif
#include "etpan-msg-new.h"
#include "etpan-subapp-thread.h"
#include "etpan-msg-reply.h"
#include "etpan-folder-sel-app.h"
#include "etpan-msg-supersedes.h"
#include "etpan-mime-tools.h"
#include "etpan-help-viewer.h"
#include "etpan-search.h"

static void idle(struct etpan_subapp * app);
static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd);
static void handle_fd(struct etpan_subapp * app, fd_set * fds);
static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void enter(struct etpan_subapp * app, struct etpan_subapp * old_app);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);
static int get_idle_delay(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_msg_list_app_driver = {
  .name = "msg-list",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = get_idle_delay,
  .idle = idle,
  .set_fd = set_fd,
  .handle_fd = handle_fd,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = enter,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct mailfolder * folder;

  struct mailmessage_tree * env_tree;
  struct etpan_node_msg_params * node_params;
  
  /* colors */
  int main_attr;
  int selection_attr;
  int status_attr;

  /* cursor & selection */
  unsigned int chosen;
  unsigned int first;
  unsigned int selected_count;
  unsigned int selected;

  /* display optimization */
  unsigned int window_size;
  unsigned int * selected_window_backward;
  int selected_window_backward_count;
  unsigned int * selected_window_forward;
  int selected_window_forward_count;
  carray * msg_tree_tab;
  chash * prefix_hash;

  /* search string */
  char * search_str;
  /* folder selection */
  struct mailfolder * last_selected_folder;
  
  /* main title */
  char * folder_name;
  char * path_description;
  
  time_t last_poll_date;
};

static void set_title(struct etpan_subapp * app);

static int get_msglist(struct etpan_subapp * app);

static int get_idle_delay(struct etpan_subapp * app)
{
  struct app_state * state;
  time_t now;
  int diff;
  time_t poll_delay;

  if (!app->enabled)
    return -1;
  
  state = app->data;

  poll_delay = app->app->config.global_config->poll_delay;
  
  if (poll_delay == 0)
    return -1;
  
  if (state->last_poll_date == (time_t) -1)
    return 0;
  
  now = time(NULL);
  diff = (int) difftime(state->last_poll_date + poll_delay, now);
  
  if (diff < 0)
    diff = 0;
  
  return diff;
}

static void idle(struct etpan_subapp * app)
{
  struct app_state * state;
  time_t now;
  time_t poll_delay;
  
  if (!app->enabled)
    return;
  
  state = app->data;

  if (state->folder == NULL)
    return;

  if (state->folder->fld_storage == NULL)
    return;
  
  poll_delay = app->app->config.global_config->poll_delay;
  
  if (poll_delay != 0) {
    now = time(NULL);
    if ((state->last_poll_date == (time_t) -1) ||
        ((int) difftime(now, state->last_poll_date) > poll_delay)) {
      /* get msg list */
      get_msglist(app);
    }
  }
}

static void free_msg_list(struct etpan_subapp * subapp);


static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd)
{
  etpan_subapp_thread_set_fd(app, fds, maxfd);
}

static int etpan_msg_list_display_precalc(struct app_state * state);
static void etpan_msg_list_display_precalc_free(struct app_state * state);

static void etpan_msg_list_auto_state_thread(struct app_state * state,
    struct mailmessage_tree * msg_tree, int close_other);

static int etpan_msg_forward_search(struct app_state * state,
    unsigned int first);

static int etpan_msg_backward_search(struct app_state * state,
    unsigned int last);

static void etpan_msg_list_expand_thread(struct etpan_subapp * app);


static int msgviewer_match_msg(struct etpan_subapp * app, void * data)
{
  return (etpan_msg_viewer_app_get_msg(app) == data);
}


static int open_viewer(struct etpan_subapp * app,
    struct mailfolder * folder,
    mailmessage * msg,
    char * filename, FILE * f, carray * attr)
{
  struct app_state * state;
  struct etpan_subapp * msgviewer_app;
  int r;
  
  state = app->data;
  
  msgviewer_app = etpan_app_find_subapp(app->app, "msg-viewer",
      1, msgviewer_match_msg, msg);
  if (msgviewer_app != NULL) {
    etpan_app_switch_subapp(msgviewer_app, 0);
    return NO_ERROR;
  }
  
  /* reuse a previous interface */
  msgviewer_app = etpan_app_find_subapp(app->app, "msg-viewer",
      0, NULL, NULL);
  if (msgviewer_app == NULL) {
    /* create new */
    msgviewer_app = etpan_msg_viewer_app_new(app->app);
    if (msgviewer_app == NULL)
      return ERROR_MEMORY;
  }

  etpan_subapp_set_parent(msgviewer_app, app);

  r = etpan_msg_viewer_app_set_msg(msgviewer_app, folder, msg,
    filename, f, attr);
  if (r != NO_ERROR)
    return r;
  
  etpan_app_switch_subapp(msgviewer_app, 0);
  
  return NO_ERROR;
}


static int mimelist_match_msg(struct etpan_subapp * mimelist_app, void * data)
{
  return (etpan_mime_list_app_get_msg(mimelist_app) == data);
}

static int etpan_msg_list_app_open_mimelist(struct etpan_subapp * app,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct app_state * state;
  struct etpan_subapp * mimelist_app;
  int r;
  
  state = app->data;
  
  mimelist_app = etpan_app_find_subapp(app->app, "mime-list",
    1, mimelist_match_msg, msg);
  if (mimelist_app != NULL) {
    etpan_app_switch_subapp(mimelist_app, 0);
    return NO_ERROR;
  }
  
  /* reuse a previous interface */
  mimelist_app = etpan_app_find_subapp(app->app, "mime-list",
      0, NULL, NULL);
  if (mimelist_app == NULL) {
    /* create new */
    mimelist_app = etpan_mime_list_app_new(app->app);
    if (mimelist_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(mimelist_app, app);
  
  r = etpan_mime_list_app_set_msg(mimelist_app, folder, msg);
  if (r != NO_ERROR)
    return r;
  
  etpan_app_switch_subapp(mimelist_app, 0);
  
  return NO_ERROR;
}




/* fetch message list */

static int next_unread(struct etpan_subapp * app, int forward,
    int verbose, int across_folder);


static int is_seen(struct mailmessage_tree * node)
{
  mailmessage * msg;
  
  msg = node->node_msg;
  if (msg == NULL)
    return 0;
  
  if (msg->msg_flags == NULL)
    return 0;
  
  if ((msg->msg_flags->fl_flags & MAIL_FLAG_SEEN) == 0)
    return 0;

  return 1;
}

static void thread_get_msg_list_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct app_state * state;
  struct etpan_folder_get_msg_list_result * thread_result;
  struct etpan_folder_get_msg_list_arg * arg;
  int do_update_list;
  int r;
  
  state = app->data;
  
  arg = op->arg;
  
  do_update_list = 0;
  if (arg != NULL) {
    chash_free(arg->node_params);
    free(arg);
    op->arg = NULL;
    do_update_list = 1;
  }
  
  if (op->err != NO_ERROR) {
    ETPAN_APP_LOG((app->app,
                      "get message list - error while getting list of messages - %s",
                      state->folder_name));
    
    return;
  }
  
  if (do_update_list) {
    free_msg_list(app);
  }
  
  thread_result = op->result;
  
  state->env_tree = thread_result->env_tree;
  state->node_params = thread_result->node_params;
  free(thread_result);
  op->result = NULL;
  
  etpan_msg_list_display_precalc(state);
  
  /* set interface */
  if (!do_update_list) {
    etpan_msg_list_auto_state_thread(state, state->env_tree, 1);
  }
  else {
    etpan_msg_list_auto_state_thread(state, state->env_tree, 0);
  }
  etpan_node_msg_set_opened(state->node_params, state->env_tree, TRUE);
  
  if (!do_update_list) {
    
    state->selected = etpan_msg_forward_search(state, 0);
    state->selected_count = 0;
    state->first = 0;
    
    if (state->msg_tree_tab->len > 0) {
      struct mailmessage_tree * node;
      
      node = carray_get(state->msg_tree_tab, state->selected);
      if (is_seen(node)) {
        r = next_unread(app, 1, 0, 0);
            
        if (r != NO_ERROR) {
          /* go to end */
          state->selected = etpan_msg_backward_search(state,
              state->msg_tree_tab->len - 1);
        }
      }
    }
  }
  else {
    if (state->msg_tree_tab->len == 0)
      state->selected = 0;
    else {
      if (state->selected > state->msg_tree_tab->len - 1)
        state->selected = state->msg_tree_tab->len - 1;
      state->selected = etpan_msg_backward_search(state,
          state->selected);
    }
  }
}

static void thread_render_handle_cancel(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  etpan_queue_unref_msg_mime(app, op->data.mailaccess.msg);
}

static int check_message(struct etpan_subapp * app,
    mailmessage * msg);

static void thread_render_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct etpan_message_render_result * result;
  int r;

  if (op->err != NO_ERROR) {
    mailmessage * msg;
    
    msg = op->data.mailaccess.msg;

    etpan_queue_unref_msg_mime(app, msg);
    
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app,
                        "view message - error while retrieving message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app,
                        "view message - error while retrieving message %i",
                        msg->msg_index));
    return;
  }

  result = op->result;
  
  /* give the message to the viewer */
  
  r = open_viewer(app, op->data.mailaccess.folder,
      op->data.mailaccess.msg, result->filename, result->f, result->attr);
  
  if (r != NO_ERROR) {
    fclose(result->f);
    unlink(result->filename);
    if (result->attr != NULL)
      etpan_text_prop_array_free(result->attr);
  }
  free(result->filename);
  free(result);
  
  op->result = NULL;
  
  if (op->data.mailaccess.msg->msg_flags != NULL) {
    op->data.mailaccess.msg->msg_flags->fl_flags &= ~MAIL_FLAG_NEW;
    op->data.mailaccess.msg->msg_flags->fl_flags |= MAIL_FLAG_SEEN;
  }
  
  check_message(app, op->data.mailaccess.msg);
  
  etpan_queue_unref_msg_mime(app, op->data.mailaccess.msg);
}


static void thread_get_bodystructure_handle_cancel(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  etpan_queue_unref_msg_mime(app, op->data.mailaccess.msg);
}

static void thread_get_bodystructure_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct app_state * state;
  int r;
  
  state = app->data;
  
  r = etpan_subapp_thread_op_add(app, THREAD_ID_MSGLIST_FLUSH,
      ETPAN_THREAD_MESSAGE_FLUSH, op->data.mailaccess.storage,
      op->data.mailaccess.folder,
      op->data.mailaccess.msg, NULL, NULL, NULL, NULL, NULL);
  if (r != NO_ERROR)
    ETPAN_APP_LOG((app->app, "mime viewer - not enough memory"));
  
  if (op->err != NO_ERROR) {
    mailmessage * msg;
    
    msg = op->data.mailaccess.msg;
    
    etpan_queue_unref_msg_mime(app, msg);
    
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app,
                        "view mime structure - error while retrieving message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app,
                        "view mime structure - error while retrieving message %i",
                        msg->msg_index));
    return;
  }
  
  /* give the message to mimelist app */

  r = etpan_msg_list_app_open_mimelist(app, op->data.mailaccess.folder,
      op->data.mailaccess.msg);
  
  etpan_queue_unref_msg_mime(app, op->data.mailaccess.msg);
}


static void handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  etpan_subapp_thread_handle_fd(app, fds);
}

static int search_messages(struct etpan_subapp * app);

static int quick_search(struct etpan_subapp * app);

static int view_message(struct etpan_subapp * app);

static int view_mime(struct etpan_subapp * app);

static int reply_message(struct etpan_subapp * app, int reply_type);

static void select_current(struct etpan_subapp * app);

static void expunge_folder(struct etpan_subapp * app);

static int catchup_messages(struct etpan_subapp * app);

static void select_thread(struct etpan_subapp * app);

static void select_all(struct etpan_subapp * app, int sel);

static void filter_select_messages(struct etpan_subapp * app);

static int change_flags(struct etpan_subapp * app, int flag_type);

static int refresh_folder(struct etpan_subapp * app);

static int reedit_message(struct etpan_subapp * app);


enum {
  FOLDER_SELECT_FOR_COPY,
  FOLDER_SELECT_FOR_MOVE,
};

static void choose_folder(struct etpan_subapp * app, int type);

static void spam_messages(struct etpan_subapp * app);

enum {
  ETPAN_FLAG_READ,
  ETPAN_FLAG_UNREAD,
  ETPAN_FLAG_DELETED,
  ETPAN_FLAG_FLAGGED,
};

static void modify_messages(struct etpan_subapp * app);

static int bounce_message(struct etpan_subapp * app, int do_redirect);

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;

  switch (key) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
  case 'G':
    break;
  default:
    state->chosen = 0;
    break;
  }

  switch (key) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
    state->chosen *= 10;
    state->chosen += key - '0';
    break;
  case 'G':
    if (state->chosen == 0) {
      if (state->msg_tree_tab->len > 0) {
        state->selected = etpan_msg_backward_search(state,
            state->msg_tree_tab->len - 1);
      }
    }
    else
      state->selected = etpan_msg_forward_search(state, state->chosen - 1);
    
    state->chosen = 0;
    break;
    
  case 'j':
  case KEY_DOWN:
    if (state->selected_window_forward_count > 1)
      state->selected = state->selected_window_forward[1];
    break;
          
  case 'k':
  case KEY_UP:
    if (state->selected_window_backward_count > 1)
      state->selected = state->selected_window_backward[1];
    break;
    
  case KEY_NPAGE:
    if (state->selected_window_forward_count > 0)
      state->selected = state->selected_window_forward[state->selected_window_forward_count - 1];
    break;
    
  case KEY_PPAGE:
    if (state->selected_window_backward_count > 0)
      state->selected = state->selected_window_backward[state->selected_window_backward_count - 1];
    break;
    
  case KEY_HOME:
    state->selected = etpan_msg_forward_search(state, 0);
    break;
    
  case KEY_END:
    if (state->msg_tree_tab->len > 0) {
      state->selected = etpan_msg_backward_search(state,
          state->msg_tree_tab->len - 1);
    }
    break;
    
  case '=':
    if (state->msg_tree_tab->len > 0) {
      etpan_msg_list_auto_state_thread(state, state->env_tree, 1);
    }
    break;
    
  case '+': /* open thread */
    if (state->env_tree != NULL) {
      struct mailmessage_tree * tree;
      
      tree = carray_get(state->msg_tree_tab, state->selected);
      etpan_node_msg_change_opened(state->node_params, tree);
    }
    break;
    
  case '*':
    etpan_msg_list_expand_thread(app);
    break;
    
  case 'q':
  case KEY_LEFT:
    etpan_app_quit_subapp(app);
    break;
    
  case '\n':
  case KEY_RIGHT:
    view_message(app);
    break;

  case 'v':
    view_mime(app);
    break;

  case 'E':
    reedit_message(app);
    break;
    
  case 'm':
    etpan_compose_new(app, state->folder);
    break;
    
  case 'a':
    reply_message(app, ETPAN_THREAD_REPLY);
    break;
    
  case 'r':
    reply_message(app, ETPAN_THREAD_REPLY_FOLLOWUP_TO);
    break;
    
  case 'g':
    reply_message(app, ETPAN_THREAD_REPLY_ALL);
    break;
    
  case 'f':
    reply_message(app, ETPAN_THREAD_REPLY_FORWARD);
    break;
    
  case KEY_CTRL('F'):
    reply_message(app, ETPAN_THREAD_REPLY_FORWARD_AS_ATTACHMENT);
    break;

  case '/':
    quick_search(app);
    break;
    
  case ':':
    search_messages(app);
    break;
    
  case 'n':
    next_unread(app, 1, 1, 1);
    break;
    
  case 'p':
    next_unread(app, 0, 1, 1);
    break;
    
  case 'R':
    change_flags(app, ETPAN_FLAG_READ);
    break;

  case 'U':
    change_flags(app, ETPAN_FLAG_UNREAD);
    break;
    
  case 'F':
    change_flags(app, ETPAN_FLAG_FLAGGED);
    break;
  
  case ' ':
    select_current(app);
    if (state->selected_window_forward_count > 1)
      state->selected = state->selected_window_forward[1];
    break;

  case 'd':
    change_flags(app, ETPAN_FLAG_DELETED);
    break;

  case '$':
    expunge_folder(app);
    break;
    
  case KEY_CTRL('C'):
    catchup_messages(app);
    break;

  case KEY_CTRL('T'):
    select_thread(app);
    break;
    
  case KEY_CTRL('A'):
    select_all(app, 1);
    break;
    
  case KEY_CTRL('I'):
    select_all(app, 0);
    break;
    
  case KEY_CTRL('U'):
    select_all(app, -1);
    break;
    
  case 'T':
    filter_select_messages(app);
    break;
    
  case KEY_CTRL('R'):
    refresh_folder(app);
    break;
    
  case 'C':
    choose_folder(app, FOLDER_SELECT_FOR_COPY);
    break;
    
  case 'M':
    choose_folder(app, FOLDER_SELECT_FOR_MOVE);
    break;
    
  case 'S':
    spam_messages(app);
    break;
    
  case KEY_CTRL('Z'):
    if (etpan_is_nntp_protocol(state->folder))
      modify_messages(app);
    break;
    
  case 'b':
    /* bounce message */
    bounce_message(app, 0);
    break;

  case 'B':
    /* redirect message */
    bounce_message(app, 1);
    break;

  case KEY_F(1):
  case '?':
    show_help(app);
    break;
  }
}

static int display_init(struct etpan_subapp * app)
{
  set_title(app);
  return etpan_app_subapp_display_init(app);
}

static int etpan_msg_list_display(struct etpan_subapp * subapp, WINDOW * w);

static void display(struct etpan_subapp * app, WINDOW * w)
{
  etpan_msg_list_display(app, w);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
}

static void enter(struct etpan_subapp * app, struct etpan_subapp * old_app)
{
  /* do nothing */
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;
  struct mailfolder * folder;
  
  state = app->data;
  
  /* backup folder reference ... */
  folder = state->folder;
  
  etpan_msg_list_app_set_folder(app, NULL);
  
  /* in order to inform parent we are leaving it */
  if (folder != NULL) {
    struct etpan_subapp * parent;
    
    parent = etpan_subapp_get_parent(app);
    if (parent != NULL) {
      if (strcasecmp(parent->driver->name, "folder-list") == 0)
        etpan_folder_list_leave_folder(parent, folder);
    }
  }
}

/* ***************************************** */
/* implementation */

static int
etpan_msg_list_display_precalc_sub(struct app_state * state,
    MMAPString * prefix,
    struct mailmessage_tree * msg_tree,
    int level, int has_next)
{
  carray * list;
  unsigned int cur;
  int r;
  
  if (msg_tree->node_msg != NULL) {
    chashdatum key;
    chashdatum value;

    r = carray_add(state->msg_tree_tab, msg_tree, NULL);
    if (r < 0)
      return ERROR_MEMORY;

    key.data = &msg_tree;
    key.len = sizeof(msg_tree);

    value.data = prefix->str;
    value.len = prefix->len + 1;

    r = chash_set(state->prefix_hash, &key, &value, NULL);
    if (r < 0)
      return ERROR_MEMORY;
  }

  list = msg_tree->node_children;
  
  if (list->len != 0) {
    char old_prefix[2];
	    
    if (level > 1) {
      memcpy(old_prefix, prefix->str + prefix->len - 2, 2);
      if (has_next)
        memcpy(prefix->str + prefix->len - 2, "| ", 2);
      else
        memcpy(prefix->str + prefix->len - 2, "  ", 2);
    }
    for(cur = 0 ; cur < carray_count(list) ; cur ++) {
      int sub_has_next;
      
      if (cur != list->len - 1) {
	if (level > 0) {
	  if (mmap_string_append(prefix, "+-") == NULL)
            return ERROR_MEMORY;
        }
	sub_has_next = TRUE;
      }
      else {
	if (level > 0) {
	  if (mmap_string_append(prefix, "\\-") == NULL)
            return ERROR_MEMORY;
        }
	sub_has_next = FALSE;
      }

      r = etpan_msg_list_display_precalc_sub(state,
          prefix, carray_get(list, cur), level + 1, sub_has_next);
      if (r != NO_ERROR)
        return r;

      if (mmap_string_truncate(prefix, prefix->len - 2) == NULL) {
        return ERROR_MEMORY;
      }
    }
    if (level > 1) {
      memcpy(prefix->str + prefix->len - 2, old_prefix, 2);
    }
  }

  return NO_ERROR;
}

static int etpan_msg_list_display_precalc(struct app_state * state)
{
  MMAPString * prefix;
  
  if (state->env_tree == NULL)
    return NO_ERROR;
  
  prefix = mmap_string_new("");
  if (prefix == NULL)
    return ERROR_MEMORY;
  
  etpan_msg_list_display_precalc_sub(state, prefix, state->env_tree, 0, FALSE);

  mmap_string_free(prefix);

  return NO_ERROR;
}

static void etpan_msg_list_display_precalc_free(struct app_state * state)
{
  chash_clear(state->prefix_hash);
  
  carray_set_size(state->msg_tree_tab, 0);
}

static void snprint_mail_info(struct etpan_node_msg_params * node_params,
    char * output,
    struct etpan_app * app,
    char * fill, int width,
    int has_child, int opened_thread, char * prefix,
    int sel, struct mailmessage_tree * node)
{
  char * from;
  char * subject;
  char * decoded_from;
  char * decoded_subject;
  int is_addr;
  char * dsp_from;
  char * dsp_subject;
  char * thread_prefix;
  char * state_prefix;
  char * sel_prefix;
  uint32_t index;
  int32_t score;
  mailmessage * msg;
  
  is_addr = 0;
  from = NULL;
  subject = NULL;
  score = 0;
  index = 0;

  * output = 0;

  decoded_subject = NULL;
  decoded_from = NULL;
 
  msg = node->node_msg;
  
  if (msg != NULL) {

    score = etpan_node_msg_get_score(node_params, node);
    
    etpan_get_from_value(&msg->msg_single_fields, &from, &is_addr);
    if (msg->msg_single_fields.fld_subject != NULL)
      subject = msg->msg_single_fields.fld_subject->sbj_value;

    if (from == NULL)
      decoded_from = NULL;
    else {
      if (!is_addr) {
        decoded_from = etpan_decode_mime_header(app, from);
        if (decoded_from == NULL) {
         decoded_from = strdup(from);
         if (decoded_from == NULL)
           goto err;
        }
      }
      else {
        decoded_from = strdup(from);
        if (decoded_from == NULL) {
           goto err;
        }
      }
    }
    
    if (subject == NULL)
      decoded_subject = NULL;
    else {
      decoded_subject = etpan_decode_mime_header(app, subject);
      if (decoded_subject == NULL) {
        decoded_subject = strdup(subject);
        if (decoded_subject == NULL)
          goto free_from;
      }
    }

    index = msg->msg_index;
  }
  else {
    decoded_subject = strdup("{dummy msg}");
    if (decoded_subject == NULL)
      goto err;
  }
  
  if (decoded_from == NULL)
    dsp_from = "";
  else {
    char * p;

    dsp_from = decoded_from;
    for(p = decoded_from ; * p != '\0' ; p ++) {
      if ((* p == '\n') || (* p == '\r'))
        * p = ' ';
      if ((* p < 32) && (* p >= 0))
        * p = ' ';
    }
  }

  if (decoded_subject == NULL)
    dsp_subject = "";
  else {
    char * p;

    dsp_subject = decoded_subject;
    for(p = decoded_subject ; * p != '\0' ; p ++) {
      if ((* p == '\n') || (* p == '\r'))
        * p = ' ';
      if ((* p < 32) && (* p >= 0))
        * p = ' ';
    }
  }

  if (has_child) {
    if (opened_thread)
      thread_prefix = "(-)";
    else
      thread_prefix = "(+)";
  }
  else
    thread_prefix = "   ";

  if (msg->msg_flags != NULL) {
    if ((msg->msg_flags->fl_flags & MAIL_FLAG_DELETED) != 0)
      state_prefix = "D";
    else if ((msg->msg_flags->fl_flags & MAIL_FLAG_FLAGGED) != 0)
      state_prefix = "!";
    else if ((msg->msg_flags->fl_flags & MAIL_FLAG_SEEN) == 0) {
      if ((msg->msg_flags->fl_flags & MAIL_FLAG_NEW) != 0)
        state_prefix = "N";
      else
        state_prefix = "U";
    }
    else if ((msg->msg_flags->fl_flags & MAIL_FLAG_ANSWERED) != 0)
      state_prefix = "A";
    else if ((msg->msg_flags->fl_flags & MAIL_FLAG_FORWARDED) != 0)
      state_prefix = "F";
    else
      state_prefix = " ";
  }
  else
    state_prefix = "X";

  if (sel)
    sel_prefix = "*";
  else
    sel_prefix = " ";

  snprintf(output, width, "%3i: %-21.21s %s %s %s %s%s%s",
	   index % 1000, dsp_from, sel_prefix,
	   state_prefix, thread_prefix, prefix, dsp_subject, fill);

  if (decoded_subject != NULL)
    free(decoded_subject);
  if (decoded_from != NULL)
    free(decoded_from);

  return;

 free_from:
  if (decoded_from)
    free(decoded_from);
 err:
  return;
}

static char * get_folder_name(struct mailfolder * folder)
{
  char * folder_name;
  char * driver_name;
  
  if (folder == NULL)
    return NULL;
  
  if (folder->fld_storage == NULL)
    return NULL;

  if (folder->fld_storage->sto_driver == NULL)
    return NULL;
  
  driver_name = folder->fld_storage->sto_driver->sto_name;
  
  if (strcasecmp(driver_name, "mbox") == 0) {
    struct mbox_mailstorage * mbox;

    mbox = folder->fld_storage->sto_data;

    folder_name = malloc(strlen("[mbox]:") + strlen(mbox->mbox_pathname) + 1);
    if (folder_name == NULL)
      return NULL;

    strcpy(folder_name, "[mbox]:");
    strcat(folder_name, mbox->mbox_pathname);
    
    return folder_name;
  }
  else if (strcasecmp(driver_name, "maildir") == 0) {
    struct maildir_mailstorage * maildir;

    maildir = folder->fld_storage->sto_data;

    folder_name = malloc(strlen("[maildir]:") + strlen(maildir->md_pathname) + 1);
    if (folder_name == NULL)
      return NULL;

    strcpy(folder_name, "[maildir]:");
    strcat(folder_name, maildir->md_pathname);
    
    return folder_name;
  }
  else if (strcasecmp(driver_name, "db") == 0) {
    struct db_mailstorage * db;

    db = folder->fld_storage->sto_data;

    folder_name = malloc(strlen("[db]:") + strlen(db->db_pathname) + 1);
    if (folder_name == NULL)
      return NULL;
    
    strcpy(folder_name, "[db]:");
    strcat(folder_name, db->db_pathname);
    
    return folder_name;
  }
  else if (strcasecmp(driver_name, "mh") == 0) {
    folder_name = malloc(strlen("[mh]:") + strlen(folder->fld_pathname) + 1);
    if (folder_name == NULL)
      return NULL;

    strcpy(folder_name, "[mh]:");
    strcat(folder_name, folder->fld_pathname);
    
    return folder_name;
  }
  else if (strcasecmp(driver_name, "imap") == 0) {
    struct imap_mailstorage * imap;

    imap = folder->fld_storage->sto_data;

    folder_name = malloc(strlen("[imap@]:") + strlen(imap->imap_servername) +
        strlen(folder->fld_pathname) + 1);
    if (folder_name == NULL)
      return NULL;

    strcpy(folder_name, "[imap@");
    strcat(folder_name, imap->imap_servername);
    strcat(folder_name, "]:");
    strcat(folder_name, folder->fld_pathname);

    return folder_name;
  }
  else if (strcasecmp(driver_name, "pop3") == 0) {
    struct pop3_mailstorage * pop;

    pop = folder->fld_storage->sto_data;

    folder_name = malloc(strlen("[pop3@]") + strlen(pop->pop3_servername) + 1);
    if (folder_name == NULL)
      return NULL;

    strcpy(folder_name, "[pop3@");
    strcat(folder_name, pop->pop3_servername);
    strcat(folder_name, "]");

    return folder_name;
  }
  else if (strcasecmp(driver_name, "nntp") == 0) {
    struct nntp_mailstorage * nntp;

    nntp = folder->fld_storage->sto_data;

    folder_name = malloc(strlen("[news@]:") + strlen(nntp->nntp_servername) +
        strlen(folder->fld_pathname) + 1);
    if (folder_name == NULL)
      return NULL;

    strcpy(folder_name, "[news@");
    strcat(folder_name, nntp->nntp_servername);
    strcat(folder_name, "]:");
    strcat(folder_name, folder->fld_pathname);

    return folder_name;
  }

  return NULL;
}

/* this function will display the list of messages */

static void draw_line(struct etpan_subapp * subapp,
    WINDOW * w, char * output, char * fill)
{
  unsigned int i;
  unsigned int count;
  int r;
  unsigned int msg_count;
  unsigned int percent;
  struct app_state * state;
  unsigned int list_lines;
  
  list_lines = subapp->display_height - 1;
  
  state = subapp->data;
  
  /* list */
  
  wattron(w, state->main_attr);
  count = 0;
  if (state->env_tree != NULL) {
    for(i = state->first ; count < list_lines ; i++) {
      chashdatum key;
      chashdatum value;
      struct mailmessage_tree * msg_tree;
      char * prefix;
    
      if (i >= state->msg_tree_tab->len)
        break;

      msg_tree = carray_get(state->msg_tree_tab, i);

      if (etpan_node_msg_is_visible(state->node_params, msg_tree)) {
        key.data = &msg_tree;
        key.len = sizeof(msg_tree);
        r = chash_get(state->prefix_hash, &key, &value);
        if (r < 0)
          prefix = "";
        else
          prefix = value.data;
        
        snprint_mail_info(state->node_params, output,
            subapp->app,
            fill, subapp->display_width + 1,
            carray_count(msg_tree->node_children) != 0,
            etpan_node_msg_is_opened(state->node_params, msg_tree), prefix,
            etpan_node_msg_is_selected(state->node_params, msg_tree),
            msg_tree);
        
        if (i == state->selected) {
          wattroff(w, state->main_attr);
          wattron(w, state->selection_attr);
        }
        mvwaddstr(w, count, 0, output);
        if (i == state->selected) {
          wattroff(w, state->selection_attr);
          wattron(w, state->main_attr);
        }
        count ++;
      }
    }
  }

  while (count < list_lines) {
    mvwaddstr(w, count, 0, fill);
    count ++;
  }
  
  wattroff(w, state->main_attr);
  
  /* status bar */
  
  wattron(w, state->status_attr);

  msg_count = carray_count(state->msg_tree_tab);
  
  if (msg_count == 0)
    percent = 0;
  else if (msg_count ==1)
	percent = 100;
  else
    percent = state->selected * 100 / (msg_count-1);

  if (state->path_description != NULL) {
    snprintf(output, subapp->display_width + 1,
        " %3i %% | %s - %i messages%s",
        percent, state->path_description, msg_count, fill);
  }
  else
    snprintf(output, subapp->display_width + 1,
        " %3i %% | (none) - %i messages%s",
        percent, msg_count, fill);
  
  mvwaddstr(w, subapp->display_height - 1, 0, output);
  wattroff(w, state->status_attr);
}


static void set_node_visible(struct etpan_subapp * app,
    struct mailmessage_tree * node)
{
  struct app_state * state;
  
  state = app->data;
  
  do {
    if (etpan_node_msg_is_visible(state->node_params, node))
      break;
    
    node = node->node_parent;
    if (node != NULL)
      etpan_node_msg_set_opened(state->node_params, node, TRUE);
    
  } while (node != NULL);
}

/*
  This function will prepare and display the message list
*/

static int etpan_msg_list_update_view(struct etpan_subapp * subapp)
{
  unsigned int count;
  unsigned int i;
  struct mailmessage_tree * msg_tree;
  unsigned int list_lines;
  struct app_state * state;
  
  state = subapp->data;
  
  list_lines = subapp->display_height - 1;
  
  if (state->msg_tree_tab->len == 0)
    return NO_ERROR;
  
  if (state->selected >= state->msg_tree_tab->len)
    state->selected = state->msg_tree_tab->len - 1;
  
  msg_tree = carray_get(state->msg_tree_tab, state->selected);
  set_node_visible(subapp, msg_tree);
  
  if (state->window_size != list_lines) {
    unsigned int * buf;
    
    buf = realloc(state->selected_window_backward,
        list_lines * sizeof(* state->selected_window_backward));
    if (buf == NULL)
      return ERROR_MEMORY;
    state->selected_window_backward = buf;
    
    buf = realloc(state->selected_window_forward,
        list_lines * sizeof(* state->selected_window_backward));
    if (buf == NULL)
      return ERROR_MEMORY;
    state->selected_window_forward = buf;
    
    state->window_size = list_lines;
  }
  
  count = 0;
  for(i = state->selected ; count < list_lines ; i++) {
    if (i >= state->msg_tree_tab->len)
      break;

    msg_tree = carray_get(state->msg_tree_tab, i);

    if (etpan_node_msg_is_visible(state->node_params, msg_tree)) {
      state->selected_window_forward[count] = i;
      count ++;
    }
  }
  state->selected_window_forward_count = count;

  count = 0;
  i = state->selected;
  while (count < list_lines) {
    if (i >= state->msg_tree_tab->len)
      break;
    
    msg_tree = carray_get(state->msg_tree_tab, i);

    if (etpan_node_msg_is_visible(state->node_params, msg_tree)) {
      state->selected_window_backward[count] = i;
      count ++;
    }

    if (i == 0)
      break;

    i --;
  }
  state->selected_window_backward_count = count;

  if (state->first <
      state->selected_window_backward[state->selected_window_backward_count - 1])
    state->first =
      state->selected_window_backward[state->selected_window_backward_count - 1];
  
  if (state->selected < state->first)
    state->first = state->selected;
  
  return NO_ERROR;
}

static int etpan_msg_list_display(struct etpan_subapp * app, WINDOW * w)
{
  char * output;
  char * fill;
  struct etpan_app * state;
  
  state = app->data;
  
  output = app->app->output;
  fill = app->app->fill;
  
  etpan_msg_list_update_view(app);
  
  draw_line(app, w, output, fill);
  
  return NO_ERROR;
}

static void etpan_msg_list_auto_state_thread(struct app_state * state,
    struct mailmessage_tree * msg_tree, int close_other)
{
  carray * list;
  unsigned int cur;
  struct mail_flags * flags;
  int one_opened;
  
  if (msg_tree == NULL)
    return;
  
  list = msg_tree->node_children;
  
  one_opened = FALSE;
  for(cur = 0 ; cur < list->len ; cur ++) {
    struct mailmessage_tree * tree;
    
    tree = carray_get(list, cur);
    
    etpan_msg_list_auto_state_thread(state, tree, close_other);
    
    if (etpan_node_msg_is_opened(state->node_params, tree)) {
      one_opened = TRUE;
    }
  }

  if (one_opened) {
    etpan_node_msg_set_opened(state->node_params, msg_tree, TRUE);
    return;
  }
  
  if (msg_tree->node_msg != NULL) {
    flags = msg_tree->node_msg->msg_flags;
    if (flags != NULL) {
      if ((flags->fl_flags & MAIL_FLAG_SEEN) == 0) {
	etpan_node_msg_set_opened(state->node_params, msg_tree, TRUE);
	return;
      }
      if ((flags->fl_flags & MAIL_FLAG_FLAGGED) != 0) {
	etpan_node_msg_set_opened(state->node_params, msg_tree, TRUE);
	return;
      }
    }
    if (etpan_node_msg_is_selected(state->node_params, msg_tree)) {
      etpan_node_msg_set_opened(state->node_params, msg_tree, TRUE);
      return;
    }
  }
  else {
    etpan_node_msg_set_opened(state->node_params, msg_tree, TRUE);
    return;
  }
  
  if (close_other)
    etpan_node_msg_set_opened(state->node_params, msg_tree, FALSE);
}

static int etpan_msg_forward_search(struct app_state * state,
    unsigned int first)
{
  unsigned int i;

  for(i = first ; i < carray_count(state->msg_tree_tab) ; i++) {
    struct mailmessage_tree * msg_tree;

    msg_tree = carray_get(state->msg_tree_tab, i);
    
    if (etpan_node_msg_is_visible(state->node_params, msg_tree)) {
      return i;
    }
  }

  return first;
}

static int etpan_msg_backward_search(struct app_state * state,
    unsigned int last)
{
  unsigned int i;

  i = last;
  do {
    struct mailmessage_tree * msg_tree;

    msg_tree = carray_get(state->msg_tree_tab, i);
    
    if (etpan_node_msg_is_visible(state->node_params, msg_tree))
      return i;
    

    if (i == 0)
      break;

    i --;

  } while (1);

  return last;
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  subapp->data = state;
  
  state->folder = NULL;
  
  /* message list */
  
  state->env_tree = NULL;
  state->node_params = NULL;
  
  /* colors */
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;
  
  /* cursor & selection */
  state->chosen = 0;
  state->first = 0;
  state->selected_count = 0;
  state->selected = 0;

  /* display optimization */
  state->window_size = 0;
  state->selected_window_backward = NULL;
  state->selected_window_backward_count = 0;
  state->selected_window_forward = NULL;
  state->selected_window_forward_count = 0;
  
  state->msg_tree_tab = carray_new(128);
  if (state->msg_tree_tab == NULL)
    goto free_state;
  
  state->prefix_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (state->prefix_hash == NULL)
    goto free_msg_tree_tab;

  state->search_str = NULL;

  state->folder_name = NULL;
  state->path_description = NULL;
  state->last_poll_date = (time_t) -1;
  
  return NO_ERROR;

 free_msg_tree_tab:
  carray_free(state->msg_tree_tab);
 free_state:
  free(subapp->data);
  subapp->data = NULL;
 err:
  return ERROR_MEMORY;
}

static void free_msg_list(struct etpan_subapp * subapp)
{
  struct etpan_folder_free_msg_list_arg * arg;
  struct app_state * state;
  int r;

  state = subapp->data;

  etpan_msg_list_display_precalc_free(state);

  if ((state->folder != NULL) && (state->env_tree != NULL)) {
    arg = malloc(sizeof(* arg));
    if (arg == NULL)
      return;

    arg->env_tree = state->env_tree;
    arg->node_params = state->node_params;
    state->env_tree = NULL;
    state->node_params = NULL;
    
    r = etpan_subapp_thread_folder_op_add(subapp,
        THREAD_ID_MSGLIST_FREE_MSG_LIST,
        ETPAN_THREAD_FOLDER_FREE_MSG_LIST,
        state->folder,
        NULL, NULL,
        arg,
        NULL, NULL, NULL);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((subapp->app, "free msg list - not enough memory"));
      /*
        weird message, isn't it ?
        we have to create an asynchronous operation to free the message
        list, that's explain the thing.
      
        there are still leaks but anyway, we're in case no memory is
        available
      */
      free(arg);
    }
  }
  else if (state->env_tree != NULL) {
    /* free message incoming from different folders */
    
    etpan_free_msg_list(subapp, state->env_tree, state->node_params);
    state->env_tree = NULL;
    state->node_params = NULL;
  }
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;

  if (state->selected_window_backward != NULL)
    free(state->selected_window_backward);
  if (state->selected_window_forward != NULL)
    free(state->selected_window_forward);
  chash_free(state->prefix_hash);
  carray_free(state->msg_tree_tab);
  free(subapp->data);
}

struct etpan_subapp * etpan_msg_list_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_msg_list_app_driver);
}

#define TITLE_NO_PATH "etPan! - message list"
#define TITLE_PATH "etPan! - message list - "

static void set_title(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailfolder * folder;
  int title_set;
  
  state = app->data;
  folder = state->folder;
  
  title_set = 0;
  if (folder != NULL) {
    if (state->folder_name != NULL) {
      char * title;
      size_t len;
      
      len = sizeof(TITLE_PATH) + strlen(state->folder_name);
      
      title = malloc(len);
      if (title != NULL) {
        snprintf(title, len, "%s%s", TITLE_PATH, state->folder_name);
        etpan_subapp_set_title(app, title);
        free(title);
        title_set = 1;
      }
    }
  }
  
  if (!title_set) {
    etpan_subapp_set_title(app, TITLE_NO_PATH);
  }
}

static int recursive_add_node_params_hash(chash * uid_node_params,
    struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * node)
{
  mailmessage * msg;
  int r;
  unsigned int cur;
  
  msg = node->node_msg;
  if (msg != NULL) {
    if (msg->msg_uid != NULL) {
      chashdatum key;
      chashdatum value;
      struct etpan_node_msg_param * node_param;

      node_param = etpan_node_msg_get_param(tree_params, node);

      key.data = msg->msg_uid;
      key.len = strlen(msg->msg_uid);
      value.data = node_param;
      value.len = 0;
      
      r = chash_set(uid_node_params, &key, &value, NULL);
      if (r < 0)
        goto err;
    }
  }
  
  for(cur = 0 ; cur < carray_count(node->node_children) ; cur ++) {
    r = recursive_add_node_params_hash(uid_node_params, tree_params,
        carray_get(node->node_children, cur));
    if (r != NO_ERROR)
      goto err;
  }
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static int get_msglist(struct etpan_subapp * app)
{
  struct etpan_folder_get_msg_list_arg * arg;
  struct mailfolder * folder;
  struct app_state * state;
  int r;
  time_t now;
  
  state = app->data;
  
  folder = state->folder;
  
  now = time(NULL);
  state->last_poll_date = now;
  
  if (folder->fld_storage == NULL)
    return ERROR_INVAL;
  
  if (state->env_tree == NULL) {
    arg = NULL;
  }
  else {
    arg = malloc(sizeof(* arg));
    if (arg == NULL)
      goto err;
        
    arg->node_params = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
    if (arg->node_params == NULL) {
      free(arg);
      goto err;
    }
    
    r = recursive_add_node_params_hash(arg->node_params,
        state->node_params, state->env_tree);
    if (arg->node_params == NULL) {
      chash_free(arg->node_params);
      free(arg);
      goto err;
    }
  }
  
  r = etpan_subapp_thread_folder_op_add(app, THREAD_ID_MSGLIST_GET_MSG_LIST,
      ETPAN_THREAD_FOLDER_GET_MSG_LIST,
      state->folder,
      NULL, NULL, arg, thread_get_msg_list_callback, NULL, NULL);
  if (r != NO_ERROR)
    goto err;
  
  return NO_ERROR;
  
 err:
  ETPAN_APP_LOG((app->app, "get message list - not enough memory"));
  return ERROR_MEMORY;
}


int etpan_msg_list_app_set_folder(struct etpan_subapp * app,
    struct mailfolder * folder)
{
  struct app_state * state;
  struct mailfolder * old_folder;
  int r;
  
  state = app->data;
  
  old_folder = state->folder;
  if ((folder != old_folder) || (folder == NULL)) {
    etpan_subapp_thread_cancel_all(app);
    
    free_msg_list(app);
    
    state->folder = folder;
    state->first = 0;
    state->selected = 0;
    
    if (state->path_description != NULL) {
      free(state->path_description);
      state->path_description = NULL;
    }
    if (state->folder_name != NULL) {
      free(state->folder_name);
      state->folder_name = NULL;
    }
    if (state->search_str != NULL) {
      free(state->search_str);
      state->search_str = NULL;
    }
    
    if (state->folder != NULL) {
      state->path_description = get_folder_name(state->folder);
      state->folder_name = etpan_folder_get_virtual_path(state->folder);
    }
  }
  
  if (folder != NULL) {
    r = get_msglist(app);
    if (r != NO_ERROR)
      return r;
  }
  
  set_title(app);
  
  return NO_ERROR;
}

void etpan_msg_list_app_set_msglist(struct etpan_subapp * app,
    struct mailmessage_tree * env_tree,
    struct etpan_node_msg_params * node_params)
{
  struct app_state * state;
  
  etpan_msg_list_app_set_folder(app, NULL);
  
  state = app->data;
  
  state->env_tree = env_tree;
  state->node_params = node_params;
  
  etpan_msg_list_display_precalc(state);
  
  etpan_msg_list_auto_state_thread(state, state->env_tree, 1);
  etpan_node_msg_set_opened(state->node_params, state->env_tree, TRUE);
  
  state->selected = etpan_msg_forward_search(state, 0);
  state->selected_count = 0;
  state->first = 0;
}

struct mailfolder * etpan_msg_list_app_get_folder(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return state->folder;
}

static int search_messages(struct etpan_subapp * app)
{
  struct etpan_subapp * search_app;
  struct app_state * state;

  state = app->data;
  
  if (state->folder == NULL)
    return ERROR_INVAL;
  
  search_app = etpan_app_find_subapp(app->app, "search-app",
    0, NULL, NULL);
  if (search_app == NULL) {
    search_app = etpan_search_app_new(app->app);
    if (search_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(search_app, app);
  
  etpan_search_app_set(search_app, state->folder);
  
  etpan_app_switch_subapp(search_app, 0);
  
  return NO_ERROR;
}

static void search_upcall(struct etpan_subapp * search_app, int valid,
    void * data);

static int quick_search(struct etpan_subapp * app)
{
  struct etpan_subapp * search_app;
  int r;
  struct app_state * state;

  state = app->data;
  
  if (state->folder == NULL)
    return ERROR_INVAL;
  
  search_app = etpan_app_find_subapp(app->app, "search-input",
      0, NULL, NULL);
  if (search_app == NULL) {
    search_app = etpan_search_input_new(app->app);
    if (search_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(search_app, app);
  
  r = etpan_search_input_set(search_app, "search: ", 256, state->search_str,
      0, search_upcall, app);
  if (r != NO_ERROR)
    return r;
  
  etpan_app_switch_subapp(search_app, 0);

  return NO_ERROR;
}

static int match_msg(struct etpan_subapp * app,
    mailmessage * msg, regex_t * reg)
{
  struct mailimf_mailbox_list * mblist;
  clistiter * cur;
  int r;
  char * subject;
  
  subject = NULL;
  if (msg->msg_single_fields.fld_subject != NULL)
    subject = msg->msg_single_fields.fld_subject->sbj_value;
  
  if (subject != NULL) {
    char * decoded_subject;
    
    decoded_subject = etpan_decode_mime_header(app->app, subject);
    if (decoded_subject == NULL) {
      decoded_subject = strdup(subject);
      if (decoded_subject == NULL)
        return 0;
    }
    
    r = regexec(reg, decoded_subject, 0, NULL, 0);
    free(decoded_subject);
    
    if (r == 0)
      return 1;
  }


  mblist = NULL;
  if (msg->msg_single_fields.fld_from != NULL)
    mblist = msg->msg_single_fields.fld_from->frm_mb_list;
  if (mblist != NULL) {
    for(cur = clist_begin(mblist->mb_list) ; cur != NULL ;
        cur = clist_next(cur)) {
      struct mailimf_mailbox * mb;
      
      mb = clist_content(cur);
      
      if (mb->mb_display_name != NULL) {
        char * decoded_name;

        decoded_name = etpan_decode_mime_header(app->app, mb->mb_display_name);
        if (decoded_name == NULL) {
          decoded_name = strdup(mb->mb_display_name);
          if (decoded_name == NULL)
            return 0;
        }
        
        r = regexec(reg, decoded_name, 0, NULL, 0);
        free(decoded_name);
        
        if (r == 0)
          return 1;
      }

      if (mb->mb_addr_spec != NULL) {
        char * decoded_name;

        decoded_name = etpan_decode_mime_header(app->app, mb->mb_addr_spec);
        if (decoded_name == NULL) {
          decoded_name = strdup(mb->mb_addr_spec);
          if (decoded_name == NULL)
            return 0;
        }
        
        r = regexec(reg, mb->mb_addr_spec, 0, NULL, 0);
        free(decoded_name);
        
        if (r == 0)
          return 1;
      }
    }
  }

  return 0;
}

static void search_upcall(struct etpan_subapp * search_app, int valid,
    void * data)
{
  char * val;
  struct etpan_subapp * app;
  struct app_state * state;
  regex_t reg;
  unsigned int i;
  int found;

  app = data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    ETPAN_APP_LOG((app->app, "cancel search"));
    goto err;
  }

  state = app->data;
  
  val = etpan_search_input_get_value(search_app);
  
  /* keep search string */
  if (state->search_str != NULL)
    free(state->search_str);
  state->search_str = strdup(val);
  
  if (regcomp(&reg, val, REG_EXTENDED | REG_ICASE) != 0) {
    ETPAN_APP_LOG((app->app, "error search - error in regular expression"));
    goto err;
  }
  
  found = 0;
  for(i = state->selected + 1 ; i < state->msg_tree_tab->len ; i ++) {
    struct mailmessage_tree * tree;
    mailmessage * msg;
    
    tree = carray_get(state->msg_tree_tab, i);
    msg = tree->node_msg;
    
    if (msg != NULL) {
      if (match_msg(app, msg, &reg)) {
        found = 1;
        state->selected = i;
        break;
      }
    }
  }
  
  regfree(&reg);

  if (!found) {
    ETPAN_APP_LOG((app->app, "error search - could not find"));
  }

  etpan_app_quit_subapp(search_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(search_app);
  return;
}

mailmessage * etpan_msg_list_app_get_selected_msg(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailmessage_tree * tree;
  
  state = app->data;
  
  if (state->env_tree == NULL)
    return NULL;
  
  if (state->selected >= carray_count(state->msg_tree_tab))
    return NULL;
  
  tree = carray_get(state->msg_tree_tab, state->selected);
  return tree->node_msg;
}

static int view_message(struct etpan_subapp * app)
{
  struct app_state * state;
  mailmessage * msg;
  struct etpan_subapp * msgviewer_app;
  int r;
  int res;
  
  state = app->data;
  
  msg = etpan_msg_list_app_get_selected_msg(app);
  if (msg == NULL)
    return ERROR_INVAL;
  
  if (etpan_subapp_thread_has_match_op(app, THREAD_ID_MSGLIST_RENDER_MSG,
          NULL, NULL, msg, NULL))
    return ERROR_BUSY;
  
  /* is it already opened ? */
  msgviewer_app = etpan_app_find_subapp(app->app, "msg-viewer",
      1, msgviewer_match_msg, msg);
  if (msgviewer_app != NULL) {
    etpan_app_switch_subapp(msgviewer_app, 0);
  }
  else {
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app, "retrieving message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app, "retrieving message %i", msg->msg_index));

    r = etpan_queue_ref_msg_mime(app, msg);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "view message - not enough memory"));
      res = r;
      goto err;
    }
    r = etpan_subapp_thread_msg_op_add(app,
        THREAD_ID_MSGLIST_RENDER_MSG, ETPAN_THREAD_MESSAGE_RENDER_MIME,
        msg, NULL,
        NULL,
        thread_render_callback, NULL, thread_render_handle_cancel);
    if (r != NO_ERROR) {
      etpan_queue_unref_msg_mime(app, msg);
      ETPAN_APP_LOG((app->app, "view message - not enough memory"));
      res = r;
      goto err;
    }
  }

  return NO_ERROR;
  
 err:
  return res;
}


int etpan_view_mime(struct etpan_subapp * app, mailmessage * msg)
{
  struct app_state * state;
  struct etpan_subapp * mimelist_app;
  int r;
  
  state = app->data;
  
  if (etpan_subapp_thread_has_match_op(app,
          THREAD_ID_MSGLIST_GET_BODYSTRUCTURE,
          NULL, NULL, msg, NULL))
    return ERROR_BUSY;
  
  /* is it already opened ? */
  mimelist_app = etpan_app_find_subapp(app->app, "mime-list",
      1, mimelist_match_msg, msg);
  if (mimelist_app != NULL) {
    etpan_app_switch_subapp(mimelist_app, 0);
  }
  else {
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app, "retrieving message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app, "retrieving message %i", msg->msg_index));

    r = etpan_queue_ref_msg_mime(app, msg);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "view mime structure - not enough memory"));
      return ERROR_MEMORY;
    }
    
    r = etpan_subapp_thread_msg_op_add(app,
        THREAD_ID_MSGLIST_GET_BODYSTRUCTURE,
        ETPAN_THREAD_MESSAGE_GET_BODYSTRUCTURE,
        msg, NULL,
        NULL,
        thread_get_bodystructure_callback, NULL,
        thread_get_bodystructure_handle_cancel);
    if (r != NO_ERROR) {
      etpan_queue_unref_msg_mime(app, msg);
      ETPAN_APP_LOG((app->app, "view mime structure - not enough memory"));
    }
  }
  
  return NO_ERROR;
}


static int view_mime(struct etpan_subapp * app)
{
  mailmessage * msg;
  
  msg = etpan_msg_list_app_get_selected_msg(app);
  if (msg == NULL)
    return ERROR_INVAL;
  
  return etpan_view_mime(app, msg);
}


static int reply_message(struct etpan_subapp * app, int reply_type)
{
  mailmessage * msg;
  struct app_state * state;
  
  state = app->data;
  
  msg = etpan_msg_list_app_get_selected_msg(app);
  if (msg == NULL)
    return ERROR_INVAL;
  
  return etpan_reply_message(app, reply_type,
      msg, NULL, NULL, NULL, state->folder);
}

static int next_unread(struct etpan_subapp * app, int forward,
    int verbose, int across_folder)
{
  unsigned int i;
  struct app_state * state;
  struct etpan_subapp * parent;
  
  state = app->data;
  
  if (state->env_tree == NULL)
    return ERROR_INVAL;
  
  i = state->selected;
  while (1) {
    struct mailmessage_tree * tree;
    mailmessage * msg;
    
    if (forward > 0) {
      if (i == carray_count(state->msg_tree_tab) - 1)
        break;
      i ++;
    }
    else {
      if (i == 0)
        break;
      i --;
    }
    
    tree = carray_get(state->msg_tree_tab, i);
    msg = tree->node_msg;
    
    if (msg == NULL)
      continue;
    
    if (msg->msg_flags == NULL) {
      state->selected = i;
      return NO_ERROR;
    }
    
    if ((msg->msg_flags->fl_flags & MAIL_FLAG_SEEN) == 0) {
      state->selected = i;
      return NO_ERROR;
    }
  }
  
  if (app->app->config.global_config->across_folder) {
    if (across_folder) {
      parent = etpan_subapp_get_parent(app);
      if (parent != NULL) {
        int r;
        
        if (strcasecmp(parent->driver->name, "folder-list") == 0) {
          r = etpan_folder_list_next_unread_message(parent, 0);
          if (r == NO_ERROR)
            return ERROR_INVAL;
        }
      }
    }
  }
  
  if (verbose)
    ETPAN_APP_LOG((app->app, "message list - no more unread messages"));
  return ERROR_INVAL;
}


/*
  select_node()
  
  if sel is 0, change the selection
  if it is 1, set to selected
  if it is -1, set to unselected
 */

static void select_node(struct etpan_subapp * app,
    struct mailmessage_tree * node, int sel)
{
  int cur_value;
  int new_value;
  struct app_state * state;
  
  if (node->node_msg == NULL)
    return;
  
  state = app->data;
  
  cur_value = etpan_node_msg_is_selected(state->node_params, node);
  
  switch (sel) {
  case 1:
    new_value = 1;
    break;
  case -1:
    new_value = 0;
    break;
  default:
    if (cur_value == 0)
      new_value = 1;
    else
      new_value = 0;
    break;
  }
  
  if (cur_value == new_value)
    return;
  
  etpan_node_msg_set_selected(state->node_params, node, new_value);
  if (new_value)
    state->selected_count --;
  else
    state->selected_count ++;
}


static struct mailmessage_tree *
get_selected_node(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailmessage_tree * node;
  
  state = app->data;
  
  if (state->env_tree == NULL)
    return NULL;
  
  node = carray_get(state->msg_tree_tab, state->selected);
  
  return node;
}

static void select_current(struct etpan_subapp * app)
{
  struct mailmessage_tree * node;
  
  node = get_selected_node(app);
  if (node == NULL)
    return;
  
  select_node(app, node, 0);
}

static void select_at_least_one(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  if (state->selected_count != 0)
    return;
  
  select_current(app);
}

static void foreach_selected(struct etpan_subapp * app,
    void (* action)(struct etpan_subapp *,
        struct mailmessage_tree *, void *), void * data)
{
  struct app_state * state;
  unsigned int i;
  
  state = app->data;
  
  for(i = 0 ; i < carray_count(state->msg_tree_tab) ; i ++) {
    struct mailmessage_tree * node;
    
    node = carray_get(state->msg_tree_tab, i);
    
    if (!etpan_node_msg_is_selected(state->node_params, node))
      continue;
    
    action(app, node, data);
    select_node(app, node, 0);
  }
}

static void foreach_node(struct etpan_subapp * app,
    void (* action)(struct etpan_subapp *,
        struct mailmessage_tree *, void *), void * data)
{
  struct app_state * state;
  unsigned int i;
  
  state = app->data;
  
  for(i = 0 ; i < carray_count(state->msg_tree_tab) ; i ++) {
    struct mailmessage_tree * node;
    
    node = carray_get(state->msg_tree_tab, i);
    
    action(app, node, data);
  }
}

static void recursive_foreach_node(struct etpan_subapp * app,
    struct mailmessage_tree * node,
    void (* action)(struct etpan_subapp *,
        struct mailmessage_tree *, void *), void * data)
{
  struct app_state * state;
  unsigned int i;
  
  state = app->data;
  
  action(app, node, data);
  
  for(i = 0 ; i < carray_count(node->node_children) ; i ++) {
    struct mailmessage_tree * child;
    
    child = carray_get(node->node_children, i);
    recursive_foreach_node(app, child, action, data);
  }
}

static void expand_node(struct etpan_subapp * app,
    struct mailmessage_tree * node, void * data)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_node_msg_set_opened(state->node_params, node, TRUE);
}

static void etpan_msg_list_expand_thread(struct etpan_subapp * app)
{
  foreach_node(app, expand_node, NULL);
}

static int check_msglist(struct etpan_subapp * app,
    carray * msglist);

static void mark_one_as_read(struct etpan_subapp * app,
    struct mailmessage_tree * node, void * data)
{
  struct mail_flags * flags;
  carray * msglist;
  
  if (node->node_msg == NULL)
    return;
  
  flags = node->node_msg->msg_flags;
  if (flags != NULL) {
    uint32_t old_flags;
    
    old_flags = flags->fl_flags;
    flags->fl_flags &= ~MAIL_FLAG_NEW;
    flags->fl_flags |= MAIL_FLAG_SEEN;
    
    /* optimization - don't check if no change */
    if (old_flags != flags->fl_flags) {
      msglist = data;
      carray_add(msglist, node->node_msg, NULL);
      /* ignore errors */
    }
  }
}

static void mark_one_as_unread(struct etpan_subapp * app,
    struct mailmessage_tree * node, void * data)
{
  struct mail_flags * flags;
  
  if (node->node_msg == NULL)
    return;
  
  flags = node->node_msg->msg_flags;
  if (flags != NULL) {
    uint32_t old_flags;
    
    old_flags = flags->fl_flags;
    flags->fl_flags &= ~(MAIL_FLAG_NEW | MAIL_FLAG_SEEN);
    
    /* optimization - don't check if no change */
    if (old_flags != flags->fl_flags) {
      carray * msglist;
      
      msglist = data;
      carray_add(msglist, node->node_msg, NULL);
      /* ignore errors */
    }
  }
}

static void mark_one_as_deleted(struct etpan_subapp * app,
    struct mailmessage_tree * node, void * data)
{
  struct mail_flags * flags;
  
  if (node->node_msg == NULL)
    return;
  
  flags = node->node_msg->msg_flags;
  if (flags != NULL) {
    carray * msglist;
    
    flags->fl_flags &= ~MAIL_FLAG_NEW;
    flags->fl_flags |= MAIL_FLAG_SEEN;
    flags->fl_flags ^= MAIL_FLAG_DELETED;
    msglist = data;
    carray_add(msglist, node->node_msg, NULL);
    /* ignore errors */
  }
}

static void mark_one_as_flagged(struct etpan_subapp * app,
    struct mailmessage_tree * node, void * data)
{
  struct mail_flags * flags;
  
  if (node->node_msg == NULL)
    return;
  
  flags = node->node_msg->msg_flags;
  if (flags != NULL) {
    carray * msglist;
    
    flags->fl_flags ^= MAIL_FLAG_FLAGGED;
    
    msglist = data;
    carray_add(msglist, node->node_msg, NULL);
    /* ignore errors */
  }
}


static int change_flags(struct etpan_subapp * app, int flag_type)
{
  int no_sel;
  struct app_state * state;
  carray * msglist;
  int r;
  
  state = app->data;

  msglist = carray_new(carray_count(state->msg_tree_tab));
  if (msglist == NULL) {
    ETPAN_APP_LOG((app->app, "change message flags - not enough memory"));
    return ERROR_MEMORY;
  }
  
  no_sel = (state->selected_count == 0);
  
  select_at_least_one(app);
  
  switch (flag_type) {
  case ETPAN_FLAG_READ:
    foreach_selected(app, mark_one_as_read, msglist);
    break;
  case ETPAN_FLAG_UNREAD:
    foreach_selected(app, mark_one_as_unread, msglist);
    break;
  case ETPAN_FLAG_DELETED:
    foreach_selected(app, mark_one_as_deleted, msglist);
    break;
  case ETPAN_FLAG_FLAGGED:
    foreach_selected(app, mark_one_as_flagged, msglist);
    break;
  }
  
  if (no_sel) {
    if (state->selected_window_forward_count > 1)
      state->selected = state->selected_window_forward[1];
  }
  
  r = check_msglist(app, msglist);
  if (r != NO_ERROR) {
    carray_free(msglist);
    ETPAN_APP_LOG((app->app, "change message flags - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}

static void expunge_folder(struct etpan_subapp * app)
{
  struct mailfolder * folder;
  struct app_state * state;
  int r;
  
  state = app->data;
  
  folder = state->folder;
  
  r = etpan_subapp_thread_folder_op_add(app,
      THREAD_ID_MSGLIST_EXPUNGE, ETPAN_THREAD_FOLDER_EXPUNGE,
      folder,
      NULL, NULL,
      NULL,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "expunge folder - not enough memory"));
  }
  
  etpan_msg_list_app_set_folder(app, folder);
}

static int catchup_messages(struct etpan_subapp * app)
{
  carray * msglist;
  int r;
  struct app_state * state;
  
  state = app->data;
  
  msglist = carray_new(carray_count(state->msg_tree_tab));
  if (msglist == NULL) {
    ETPAN_APP_LOG((app->app, "change message flags - not enough memory"));
    return ERROR_MEMORY;
  }
  
  foreach_node(app, mark_one_as_read, msglist);
  
  r = check_msglist(app, msglist);
  if (r != NO_ERROR) {
    carray_free(msglist);
    ETPAN_APP_LOG((app->app, "change message flags - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}

struct selection_rule {
  int do_sel;
};

static void select_one_node(struct etpan_subapp * app,
    struct mailmessage_tree * node, void * data)
{
  struct selection_rule * rule;
  
  rule = data;
  select_node(app, node, rule->do_sel);
}

static void select_thread(struct etpan_subapp * app)
{
  struct mailmessage_tree * node;
  struct selection_rule rule;
  
  node = get_selected_node(app);
  if (node == NULL)
    return;
  
  rule.do_sel = 1;
  recursive_foreach_node(app, node, select_one_node, &rule);
}

static void select_all(struct etpan_subapp * app, int sel)
{
  struct app_state * state;
  struct selection_rule rule;
  
  state = app->data;
  
  rule.do_sel = sel;
  if (state->env_tree != NULL)
    recursive_foreach_node(app, state->env_tree, select_one_node, &rule);
}




static void filter_select_upcall(struct etpan_subapp * search_app, int valid,
    void * data);

static void filter_select_messages(struct etpan_subapp * app)
{
  struct etpan_subapp * search_app;
  int r;
  struct app_state * state;

  state = app->data;
  
  search_app = etpan_app_find_subapp(app->app, "search-input",
    0, NULL, NULL);
  if (search_app == NULL) {
    search_app = etpan_search_input_new(app->app);
    if (search_app == NULL)
      return;
  }
  
  etpan_subapp_set_parent(search_app, app);
  
  r = etpan_search_input_set(search_app, "select: ", 256, state->search_str,
      0, filter_select_upcall, app);
  if (r != NO_ERROR)
    return;
  
  etpan_app_switch_subapp(search_app, 0);
}

static void filter_select_upcall(struct etpan_subapp * search_app,
    int valid, void * data)
{
  char * val;
  struct etpan_subapp * app;
  struct app_state * state;
  regex_t reg;
  unsigned int i;
  int found;
  
  app = data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    ETPAN_APP_LOG((app->app, "cancel filter select"));
    goto err;
  }
  
  state = app->data;
  
  val = etpan_search_input_get_value(search_app);
  
  /* keep search string */
  state->search_str = strdup(val);
  
  if (regcomp(&reg, val, REG_EXTENDED | REG_ICASE) != 0) {
    ETPAN_APP_LOG((app->app,
                      "error filter select - error in regular expression"));
    goto err;
  }
  
  found = 0;
  for(i = 0 ; i < carray_count(state->msg_tree_tab) ; i ++) {
    struct mailmessage_tree * tree;
    mailmessage * msg;
    
    tree = carray_get(state->msg_tree_tab, i);
    msg = tree->node_msg;
    
    if (msg != NULL) {
      if (match_msg(app, msg, &reg)) {
        select_node(app, tree, 1);
        set_node_visible(app, tree);
        found = 1;
      }
    }
  }
  
  regfree(&reg);

  if (!found) {
    ETPAN_APP_LOG((app->app, "error filter select - could not find"));
  }

  etpan_app_quit_subapp(search_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(search_app);
  return;
}

int etpan_msg_list_app_open_next(struct etpan_subapp * app,
    int forward)
{
  struct app_state * state;
  int r;
  
  state = app->data;
  
  r = next_unread(app, forward, 1, 1);
  if (r == NO_ERROR) {
    etpan_msg_list_update_view(app);
    r = view_message(app);
    
    return r;
  }
  
  return r;
}

static int check_message(struct etpan_subapp * app,
    mailmessage * msg)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_subapp_thread_msg_op_add(app,
      THREAD_ID_MSGLIST_CHECK_MSG, ETPAN_THREAD_MESSAGE_CHECK,
      msg, NULL,
      NULL,
      NULL, NULL, NULL);
}

static int check_msglist(struct etpan_subapp * app,
    carray * msglist)
{
  struct mailfolder * folder;
  struct app_state * state;
  struct etpan_msglist_check_arg * arg;
  
  state = app->data;
  folder = state->folder;
  
  arg = malloc(sizeof(* arg));
  if (arg == NULL)
    return ERROR_MEMORY;
  
  arg->msglist = msglist;
  
  return etpan_subapp_thread_folder_op_add(app,
      THREAD_ID_MSGLIST_CHECK_MSG, ETPAN_THREAD_MSGLIST_CHECK,
      folder,
      NULL, NULL,
      arg,
      NULL, NULL, NULL);
}

/* refresh folder */

static int refresh_folder(struct etpan_subapp * app)
{
  struct mailfolder * folder;
  struct app_state * state;
  
  state = app->data;
  
  folder = state->folder;
  
  etpan_msg_list_app_set_folder(app, folder);
  
  return NO_ERROR;
}

/* copy and move messages */

static void folder_sel_upcall(struct etpan_subapp * folder_sel_app, int valid,
    void * data);

struct folder_select_data {
  struct etpan_subapp * msglist_app;
  int type;
};

static void choose_folder(struct etpan_subapp * app, int type)
{
  struct etpan_subapp * folder_sel_app;
  int r;
  struct app_state * state;
  struct folder_select_data * data;
  
  state = app->data;
  
  folder_sel_app = etpan_app_find_subapp(app->app, "folder-sel",
    0, NULL, NULL);
  if (folder_sel_app == NULL) {
    folder_sel_app = etpan_folder_sel_app_new(app->app);
    if (folder_sel_app == NULL) {
      ETPAN_APP_LOG((app->app, "select folder - not enough memory"));
      return;
    }
  }
  
  etpan_subapp_set_parent(folder_sel_app, app);
  
  data = malloc(sizeof(* data));
  if (data == NULL) {
    ETPAN_APP_LOG((app->app, "select folder - not enough memory"));
    return;
  }
  data->msglist_app = app;
  data->type = type;
  
  r = etpan_folder_sel_app_set(folder_sel_app,
      app->app->config.vfolder_config->root, state->last_selected_folder,
      folder_sel_upcall, data);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "select folder - not enough memory"));
    return;
  }
  
  etpan_app_switch_subapp(folder_sel_app, 0);
}

int do_copy_messages(struct etpan_subapp * app,
    struct mailfolder * dest_folder, int remove_source);

static void folder_sel_upcall(struct etpan_subapp * folder_sel_app, int valid,
    void * data)
{
  struct etpan_subapp * app;
  struct app_state * state;
  struct mailfolder * dest_folder;
  struct folder_select_data * select_data;
  int type;
  int r;
  
  select_data = data;
  app = select_data->msglist_app;
  type = select_data->type;
  free(select_data);
  
  if (valid == ETPAN_FOLDER_SEL_CANCEL) {
    ETPAN_APP_LOG((app->app, "select folder - cancel"));
    goto err;
  }
  
  state = app->data;
  
  dest_folder = etpan_folder_sel_app_get_selected_folder(folder_sel_app);

  state->last_selected_folder = dest_folder;

  switch (type) {
  case FOLDER_SELECT_FOR_COPY:
    r = do_copy_messages(app, dest_folder, 0);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "copy messages - failed"));
    }
    break;
    
  case FOLDER_SELECT_FOR_MOVE:
    r = do_copy_messages(app, dest_folder, 1);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "move messages - failed"));
    }
    break;
  }
  
  etpan_app_quit_subapp(folder_sel_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(folder_sel_app);
  return;
}

static void add_to_array(struct etpan_subapp * app,
    struct mailmessage_tree * node, void * data)
{
  carray * msglist;
  
  msglist = data;

  carray_add(msglist, node->node_msg, NULL);
}

static int run_copy_msglist(struct etpan_subapp * app,
    carray * msglist, struct mailfolder * dest_folder);

static int run_move_msglist(struct etpan_subapp * app,
    carray * msglist, struct mailfolder * dest_folder);

int do_copy_messages(struct etpan_subapp * app,
    struct mailfolder * dest_folder, int remove_source)
{
  int no_sel;
  struct app_state * state;
  carray * msglist;
  int r;
  
  state = app->data;

  msglist = carray_new(carray_count(state->msg_tree_tab));
  if (msglist == NULL) {
    ETPAN_APP_LOG((app->app, "copy message - not enough memory"));
    return ERROR_MEMORY;
  }
  
  no_sel = (state->selected_count == 0);
  
  select_at_least_one(app);
  
  foreach_selected(app, add_to_array, msglist);
  
  if (remove_source)
    r = run_move_msglist(app, msglist, dest_folder);
  else
    r = run_copy_msglist(app, msglist, dest_folder);
  
  return r;
}

static int run_copy_msglist(struct etpan_subapp * app,
    carray * msglist, struct mailfolder * dest_folder)
{
  struct mailfolder * folder;
  struct app_state * state;
  struct etpan_msglist_copy_arg * arg;
  int r;
  
  state = app->data;
  folder = state->folder;
  
  arg = malloc(sizeof(* arg));
  if (arg == NULL)
    return ERROR_MEMORY;
  
  arg->msglist = msglist;
  arg->folder = dest_folder;
  
  r = etpan_subapp_thread_folder_op_add(app,
      THREAD_ID_MSGLIST_COPY_MSG, ETPAN_THREAD_MSGLIST_COPY,
      folder,
      NULL, NULL,
      arg,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    carray_free(msglist);
    free(arg);
  }
  
  return r;
}

static int run_move_msglist(struct etpan_subapp * app,
    carray * msglist, struct mailfolder * dest_folder)
{
  struct mailfolder * folder;
  struct app_state * state;
  struct etpan_msglist_move_arg * arg;
  int r;
  
  state = app->data;
  folder = state->folder;
  
  arg = malloc(sizeof(* arg));
  if (arg == NULL)
    return ERROR_MEMORY;
  
  arg->msglist = msglist;
  arg->folder = dest_folder;
  
  r = etpan_subapp_thread_folder_op_add(app,
      THREAD_ID_MSGLIST_MOVE_MSG, ETPAN_THREAD_MSGLIST_MOVE,
      folder,
      NULL, NULL,
      arg,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    carray_free(msglist);
    free(arg);
  }

  return r;
}



/* spam */

static void spam_upcall(struct etpan_subapp * input_app, int valid,
    void * data);

static void spam_messages(struct etpan_subapp * app)
{
  struct etpan_subapp * input_app;
  int r;
  struct app_state * state;

  state = app->data;
  
  input_app = etpan_app_find_subapp(app->app, "search-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_search_input_new(app->app);
    if (input_app == NULL)
      return;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  r = etpan_search_input_set(input_app,
      "(s)pam/(n)on-spam/fix as (S)pam/fix as (N)on-spam: ", 1, NULL,
      0, spam_upcall, app);
  if (r != NO_ERROR)
    return;
  
  etpan_app_switch_subapp(input_app, 0);
}

static void spam_upcall(struct etpan_subapp * input_app, int valid,
    void * data)
{
  char * val;
  struct etpan_subapp * app;
  struct app_state * state;
  int spam_type;
  carray * msglist;
  struct mailfolder * folder;
  int no_sel;
  struct etpan_msglist_spam_arg * arg;
  int r;
  
  app = data;

  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    ETPAN_APP_LOG((app->app, "spam operation - cancel"));
    goto err;
  }
  
  state = app->data;
  
  val = etpan_search_input_get_value(input_app);
  
  switch (val[0]) {
  case 's':
    spam_type = ETPAN_THREAD_SPAM;
    break;
  case 'n':
    spam_type = ETPAN_THREAD_NON_SPAM;
    break;
  case 'S':
    spam_type = ETPAN_THREAD_FIX_SPAM;
    break;
  case 'N':
    spam_type = ETPAN_THREAD_FIX_NON_SPAM;
    break;
  default:
    ETPAN_APP_LOG((app->app, "spam operation - invalid operation"));
    goto err;
  } 
  
  msglist = carray_new(carray_count(state->msg_tree_tab));
  if (msglist == NULL) {
     ETPAN_APP_LOG((app->app, "spam operation - not enough memory"));
     goto err;
  }
  
  /* prepare to run thread */
  no_sel = (state->selected_count == 0);
  
  select_at_least_one(app);
  
  foreach_selected(app, add_to_array, msglist);
  
  folder = state->folder;
  
  arg = malloc(sizeof(* arg));
  if (arg == NULL) {
     ETPAN_APP_LOG((app->app, "spam operation - not enough memory"));
     goto err;
  }
  
  arg->msglist = msglist;
  arg->spam_type = spam_type;
  
  /* run thread */
  r = etpan_subapp_thread_folder_op_add(app,
      THREAD_ID_MSGLIST_SPAM_MSG, ETPAN_THREAD_MSGLIST_SPAM,
      folder,
      NULL, NULL,
      arg,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    carray_free(msglist);
    free(arg);
  }
  
  etpan_app_quit_subapp(input_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(input_app);
  return;
}


/* cancel / supersedes */

static void modify_upcall(struct etpan_subapp * input_app, int valid,
    void * data);

static void modify_messages(struct etpan_subapp * app)
{
  struct etpan_subapp * input_app;
  int r;
  struct app_state * state;

  state = app->data;

  input_app = etpan_app_find_subapp(app->app, "search-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_search_input_new(app->app);
    if (input_app == NULL)
      return;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  r = etpan_search_input_set(input_app,
      "(s)upersedes/c(ancel): ", 1, NULL,
      0, modify_upcall, app);
  if (r != NO_ERROR)
    return;
  
  etpan_app_switch_subapp(input_app, 0);
}

enum {
  MODIFY_TYPE_CANCEL,
  MODIFY_TYPE_SUPERSEDES,
};

static void modify_upcall(struct etpan_subapp * input_app, int valid,
    void * data)
{
  char * val;
  struct etpan_subapp * app;
  struct app_state * state;
  struct mailfolder * folder;
  int r;
  int modify_type;
  mailmessage * msg;
  
  app = data;

  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    ETPAN_APP_LOG((app->app, "supersedes/cancel message - cancel"));
    goto err;
  }
  
  state = app->data;
  
  val = etpan_search_input_get_value(input_app);
  
  switch (val[0]) {
  case 's':
    modify_type = MODIFY_TYPE_SUPERSEDES;
    break;
  case 'c':
    modify_type = MODIFY_TYPE_CANCEL;
    break;
  default:
    ETPAN_APP_LOG((app->app, "supersedes/cancel message - invalid operation"));
    goto err;
  } 
  
  folder = state->folder;
  
  msg = etpan_msg_list_app_get_selected_msg(app);
  if (msg == NULL) {
    ETPAN_APP_LOG((app->app, "supersedes/cancel message - failed"));
    goto err;
  }
  
  switch (modify_type) {
  case MODIFY_TYPE_CANCEL:
    r = etpan_cancel_message(app, msg, NULL);
    break;
  case MODIFY_TYPE_SUPERSEDES:
    r = etpan_supersedes_message(app, msg, NULL);
    break;
  default:
    r = ERROR_INVAL;
    break;
  }
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "supersedes/cancel message - failed"));
    goto err;
  }
  
  etpan_app_quit_subapp(input_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(input_app);
  return;
}

static int bounce_message(struct etpan_subapp * app, int do_redirect)
{
  mailmessage * msg;
  struct app_state * state;
  
  state = app->data;
  
  msg = etpan_msg_list_app_get_selected_msg(app);
  if (msg == NULL)
    return ERROR_INVAL;
  
  return etpan_bounce_message(app, msg, NULL, do_redirect);
}

static int reedit_message(struct etpan_subapp * app)
{
  mailmessage * msg;
  struct app_state * state;
  
  state = app->data;
  
  msg = etpan_msg_list_app_get_selected_msg(app);
  if (msg == NULL)
    return ERROR_INVAL;
  
  return etpan_reedit_message(app, msg, NULL);
}



int etpan_msg_list_first_unread(struct etpan_subapp * app,
    int verbose)
{
  unsigned int i;
  struct app_state * state;
  
  state = app->data;
  
  if (state->env_tree == NULL)
    return ERROR_INVAL;
  
  i = 0;
  while (1) {
    struct mailmessage_tree * tree;
    mailmessage * msg;
    
    if (i == carray_count(state->msg_tree_tab) - 1)
      break;
    i ++;
    
    tree = carray_get(state->msg_tree_tab, i);
    msg = tree->node_msg;
    
    if (msg == NULL)
      continue;
    
    if (msg->msg_flags == NULL) {
      state->selected = i;
      return NO_ERROR;
    }
    
    if ((msg->msg_flags->fl_flags & MAIL_FLAG_SEEN) == 0) {
      state->selected = i;
      return NO_ERROR;
    }
  }
  
  if (verbose)
    ETPAN_APP_LOG((app->app, "message list - no more unread messages"));
  return ERROR_INVAL;
}


#define HELP_TEXT \
"\
Help for messages list viewer\n\
-----------------------------\n\
\n\
This application will let you view the content of a folder.\n\
\n\
- left, right,\n\
  Ctrl-W,\n\
  Ctrl-X      : switch between applications\n\
\n\
- up,down     : move cursor\n\
\n\
- +           : expand/close node\n\
- *           : expand all nodes\n\
- =           : expand only nodes with unread messages or flagged messages\n\
\n\
- [Enter]     : view selected message\n\
- v           : view MIME structure of the selected message\n\
- E           : reedit message\n\
\n\
- m           : compose new message\n\
- a           : reply to the author of the selected message\n\
- r           : reply to the selected messge\n\
- g           : reply to all recipient of the selected message\n\
- f           : forward selected message\n\
- F           : forward selected message as attachment\n\
- b           : bounce the selected message\n\
- B           : redirect (Subject line is modified)\n\
- Ctrl-Z      : cancel or supersedes news articles\n\
\n\
- n,p         : next/previous unread message\n\
- R           : mark messages as read\n\
- U           : mark messages as unread\n\
- F           : flag/unflag messages\n\
- d           : mark/unmark messages as deleted\n\
- Ctrl-C      : mark all messages as read\n\
\n\
- /           : search message by subject and author\n\
- :           : search messages\n\
\n\
- $           : expunge messages\n\
- Ctrl-R      : refresh message list\n\
\n\
- [space]     : select message\n\
- Ctrl-T      : select thread\n\
- Ctrl-A      : select all messages\n\
- Ctrl-I      : invert selection\n\
- Ctrl-U      : unselect all messages\n\
- Shift-T     : select messages by subject and author\n\
- Shift-C     : copy messages\n\
- Shift-M     : move messages\n\
- Shift-S     : bogofilter functions\n\
\n\
- q           : exit\n\
\n\
- ?           : help\n\
- Ctrl-L      : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
