/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-mime-list-app.c,v 1.14 2004/08/24 01:11:59 hoa Exp $
 */

#include "etpan-mime-list-app.h"

#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-thread-manager.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ncurses.h>
#include <libetpan/libetpan.h>
#include "etpan-msg-renderer.h"
#include "etpan-part-viewer-app.h"
#include "etpan-mime-params.h"
#include "etpan-tools.h"
#include "etpan-imf-helper.h"
#include "etpan-header-editor-app.h"
#include "etpan-mime-common.h"
#include "etpan-help-viewer.h"

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd);
static void handle_fd(struct etpan_subapp * app, fd_set * fds);
static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_mime_list_app_driver = {
  .name = "mime-list",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = set_fd,
  .handle_fd = handle_fd,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct etpan_mime_common_app_state common_state;  
};

static void set_fd(struct etpan_subapp * app,
    fd_set * fds, int * maxfd)
{
  struct app_state * state;
  
  state = app->data;

  etpan_mime_common_set_fd(app, &state->common_state, fds, maxfd);
}

static void handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  struct app_state * state;
  
  state = app->data;

  etpan_mime_common_handle_fd(app, &state->common_state, fds);
}

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app,
    int key)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_mime_common_handle_key(app, &state->common_state, key);
  
  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;
    
  case 'q':
  case KEY_LEFT:
    etpan_app_quit_subapp(app);
    break;
  }
}

#define TITLE_NO_MSG_ID "etPan! - message part list"
#define TITLE_MSG_ID "etPan! - message part list - "


static void set_title(struct etpan_subapp * app)
{
  char index_str[20];
  int title_set;
  struct app_state * state;
  mailmessage * msg;
  
  state = app->data;
  msg = state->common_state.msg;

  title_set = 0;
  if (msg != NULL) {
    if (msg->msg_index != 0) {
      char * title;
      size_t len;
      
      snprintf(index_str, sizeof(index_str), "%u", msg->msg_index);
      
      len = sizeof(TITLE_MSG_ID) + strlen(index_str);
      
      title = malloc(len);
      if (title != NULL) {
        snprintf(title, len, "%s%s", TITLE_MSG_ID, index_str);
        etpan_subapp_set_title(app, title);
        free(title);
        title_set = 1;
      }
    }
  }
  
  if (!title_set) {
    etpan_subapp_set_title(app, TITLE_NO_MSG_ID);
  }
}


static int display_init(struct etpan_subapp * app)
{
  set_title(app);
  return etpan_app_subapp_display_init(app);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_mime_common_set_color(app, &state->common_state);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_mime_common_display(app, &state->common_state, w, NULL);
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  r = etpan_mime_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;
  
  etpan_mime_common_done(subapp, &state->common_state);
  
  free(state);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_mime_common_leave(app, &state->common_state, new_app);
}

struct etpan_subapp *
etpan_mime_list_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_mime_list_app_driver);
}

int etpan_mime_list_app_set_msg(struct etpan_subapp * app,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct app_state * state;
  int r;

  state = app->data;
  
  r = etpan_mime_common_set_msg(app, &state->common_state, folder, msg);
  set_title(app);
  
  return r;
}

mailmessage * etpan_mime_list_app_get_msg(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_mime_common_get_msg(app, &state->common_state);
}

struct mailfolder * etpan_mime_list_app_get_folder(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_mime_common_get_folder(app, &state->common_state);
}

#define HELP_TEXT \
"\
Help for MIME message viewer\n\
----------------------------\n\
\n\
This application will let you view specific parts of the \n\
MIME message.\n\
\n\
- Ctrl-W,\n\
  Ctrl-X     : switch between applications\n\
\n\
- arrow keys : move cursor\n\
\n\
- +          : expand/close node\n\
- *          : expand all nodes\n\
\n\
- [Enter]    : view selected part\n\
- s          : save selected MIME part\n\
- h          : view header of selected MIME part\n\
- Ctrl-H     : view MIME headers of selected MIME part\n\
- t          : view source of selected MIME part\n\
\n\
- m          : compose new message\n\
- a          : reply to the author of the selected MIME part\n\
- r          : reply to the selected MIME part\n\
- g          : reply to all recipient of the selected MIME part\n\
- f          : forward selected MIME part\n\
- F          : forward selected MIME part as attachment\n\
- b          : bounce the selected MIME part\n\
- B          : redirect (Subject line is modified)\n\
\n\
- q          : exit\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
