/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-folder-sel-app.c,v 1.10 2003/12/15 16:08:53 hoa Exp $
 */

#include "etpan-folder-sel-app.h"
#include "etpan-subapp.h"
#include "etpan-subapp-thread.h"
#include "etpan-errors.h"
#include "etpan-folder-params.h"
#include "etpan-app-subapp.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ncurses.h>
#include <libetpan/libetpan.h>
#include <libetpan/charconv.h>
#include "etpan-msg-list-app.h"
#include "etpan-cfg-vfolder.h"
#include "etpan-app.h"
#include "etpan-msg-new.h"
#include "etpan-folder-common.h"
#include "etpan-help-viewer.h"

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void enter(struct etpan_subapp * app, struct etpan_subapp * old_app);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_folder_sel_app_driver = {
  .name = "folder-sel",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = enter,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct etpan_folder_common_app_state common_state;
  void (* upcall)(struct etpan_subapp *, int, void *);
  void * upcall_data;
};

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  struct mailfolder * folder;
  
  state = app->data;

  etpan_folder_common_handle_key(app, &state->common_state, key);

  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;
    
  case '\n':
    folder = etpan_folder_common_get_selected_folder(&state->common_state);
    
    if (folder->fld_storage != NULL) {
      if (state->upcall != NULL)
        state->upcall(app, ETPAN_FOLDER_SEL_VALID, state->upcall_data);
    }
    break;

  case KEY_CTRL('G'):
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_FOLDER_SEL_CANCEL, state->upcall_data);
    break;
  }
}

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - select a folder");
  return etpan_app_subapp_display_init(app);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_folder_common_display(app, &state->common_state, w,
      "[enter]: select  ^G: cancel");
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;

  etpan_folder_common_set_color(app, &state->common_state);
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;

  subapp->data = state;
  
  r = etpan_folder_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free_state;
  
  state->upcall = NULL;
  state->upcall_data = NULL;
  
  return NO_ERROR;
  
 free_state:
  free(subapp->data);
  subapp->data = NULL;
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;
  
  etpan_folder_common_done(&state->common_state);

  free(subapp->data);
}


int etpan_folder_sel_app_set(struct etpan_subapp * app,
    struct mailfolder * root,
    struct mailfolder * selected,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * upcall_data)
{
  struct app_state * state;
  int r;
  unsigned int i;

  state = app->data;
  state->upcall = upcall;
  state->upcall_data = upcall_data;
  
  r = etpan_folder_common_set_root_folder(&state->common_state,
      root, 0);
  if (r != NO_ERROR)
    return r;
  
  for(i = 0 ; i < carray_count(state->common_state.folder_tab) ; i ++) {
    if (carray_get(state->common_state.folder_tab, i) == selected)
      state->common_state.selected = i;
  }
  
  return NO_ERROR;
}

struct etpan_subapp * etpan_folder_sel_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_folder_sel_app_driver);
}

static void enter(struct etpan_subapp * app, struct etpan_subapp * old_app)
{
  /* do nothing */
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;

  state = app->data;
  
  state->upcall = NULL;
  state->upcall_data = NULL;
  etpan_folder_common_flush(app, &state->common_state);
}

struct mailfolder *
etpan_folder_sel_app_get_selected_folder(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;

  return etpan_folder_common_get_selected_folder(&state->common_state);
}

#define HELP_TEXT \
"\
Help for folder selection\n\
-------------------------\n\
\n\
This application will let you select a folder.\n\
\n\
- arrow keys : move cursor\n\
\n\
- [Enter]    : select folder\n\
- Ctrl-G     : cancel\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
