/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-folder-edit.c,v 1.15 2004/12/17 10:20:51 hoa Exp $
 */

#include "etpan-folder-edit.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include "etpan-tools.h"
#include "etpan-cfg-storage.h"
#include "etpan-cfg-common.h"
#include <stdlib.h>
#include "etpan-thread-manager.h"
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "etpan-cfg-edit-common.h"
#include "etpan-imf-helper.h"
#include "etpan-mime-tools.h"
#include "etpan-storage-sel.h"
#include "etpan-folder-sel-app.h"
#include "etpan-account-sel.h"
#include "etpan-help-viewer.h"
#include "etpan-search-input.h"

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * app);
static void done(struct etpan_subapp * app);
static int display_init(struct etpan_subapp * app);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);

static struct etpan_subapp_driver etpan_folder_editor_app_driver = {
  .name = "folder-edit",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

#define INFO_TYPE_MAX 20

enum {
  INFO_TYPE_NAME,
  INFO_TYPE_STORAGE,
  INFO_TYPE_LOCATION,
  INFO_TYPE_ACCOUNT,
  INFO_TYPE_SENT_FOLDER,
  INFO_TYPE_DRAFT_FOLDER,
  INFO_TYPE_MAIL_TO,
  INFO_TYPE_POLL,
  INFO_TYPE_MAX_ARTICLES,
};

static int none_info_type[] = {
  INFO_TYPE_NAME,
  INFO_TYPE_STORAGE,
  INFO_TYPE_ACCOUNT,
  INFO_TYPE_SENT_FOLDER,
  INFO_TYPE_DRAFT_FOLDER,
  INFO_TYPE_MAIL_TO,
};

static int mbox_info_type[] = {
  INFO_TYPE_NAME,
  INFO_TYPE_STORAGE,
  INFO_TYPE_ACCOUNT,
  INFO_TYPE_SENT_FOLDER,
  INFO_TYPE_DRAFT_FOLDER,
  INFO_TYPE_MAIL_TO,
  INFO_TYPE_POLL,
};

static int mh_info_type[] = {
  INFO_TYPE_NAME,
  INFO_TYPE_STORAGE,
  INFO_TYPE_LOCATION,
  INFO_TYPE_ACCOUNT,
  INFO_TYPE_SENT_FOLDER,
  INFO_TYPE_DRAFT_FOLDER,
  INFO_TYPE_MAIL_TO,
  INFO_TYPE_POLL,
};

static int maildir_info_type[] = {
  INFO_TYPE_NAME,
  INFO_TYPE_STORAGE,
  INFO_TYPE_ACCOUNT,
  INFO_TYPE_SENT_FOLDER,
  INFO_TYPE_DRAFT_FOLDER,
  INFO_TYPE_MAIL_TO,
  INFO_TYPE_POLL,
};

static int db_info_type[] = {
  INFO_TYPE_NAME,
  INFO_TYPE_STORAGE,
  INFO_TYPE_ACCOUNT,
  INFO_TYPE_SENT_FOLDER,
  INFO_TYPE_DRAFT_FOLDER,
  INFO_TYPE_MAIL_TO,
  INFO_TYPE_POLL,
};

static int imap_info_type[] = {
  INFO_TYPE_NAME,
  INFO_TYPE_STORAGE,
  INFO_TYPE_LOCATION,
  INFO_TYPE_ACCOUNT,
  INFO_TYPE_SENT_FOLDER,
  INFO_TYPE_DRAFT_FOLDER,
  INFO_TYPE_MAIL_TO,
  INFO_TYPE_POLL,
};

static int pop3_info_type[] = {
  INFO_TYPE_NAME,
  INFO_TYPE_STORAGE,
  INFO_TYPE_ACCOUNT,
  INFO_TYPE_SENT_FOLDER,
  INFO_TYPE_DRAFT_FOLDER,
  INFO_TYPE_MAIL_TO,
  INFO_TYPE_POLL,
};

static int nntp_info_type[] = {
  INFO_TYPE_NAME,
  INFO_TYPE_STORAGE,
  INFO_TYPE_LOCATION,
  INFO_TYPE_ACCOUNT,
  INFO_TYPE_SENT_FOLDER,
  INFO_TYPE_DRAFT_FOLDER,
  INFO_TYPE_POLL,
  INFO_TYPE_MAX_ARTICLES,
};

#define TABLE_SIZE(a) (sizeof(a) / sizeof(a[0]))

static struct {
  int type;
  int * info_type_tab;
  unsigned int info_type_count;
} storage_type_tab[] = {
  {
    .type = STORAGE_TYPE_MBOX,
    .info_type_tab = mbox_info_type,
    .info_type_count = TABLE_SIZE(mbox_info_type),
  },
  {
    .type = STORAGE_TYPE_MH,
    .info_type_tab = mh_info_type,
    .info_type_count = TABLE_SIZE(mh_info_type),
  },
  {
    .type = STORAGE_TYPE_MAILDIR,
    .info_type_tab = maildir_info_type,
    .info_type_count = TABLE_SIZE(maildir_info_type),
  },
  {
    .type = STORAGE_TYPE_IMAP,
    .info_type_tab = imap_info_type,
    .info_type_count = TABLE_SIZE(imap_info_type),
  },
  {
    .type = STORAGE_TYPE_POP3,
    .info_type_tab = pop3_info_type,
    .info_type_count = TABLE_SIZE(pop3_info_type),
  },
  {
    .type = STORAGE_TYPE_NNTP,
    .info_type_tab = nntp_info_type,
    .info_type_count = TABLE_SIZE(nntp_info_type),
  },
};

struct app_state {
  char name[256];
  struct mailstorage * storage;
  char location[PATH_MAX];
  struct etpan_account_info * account;
  struct mailfolder * sent_folder;
  struct mailfolder * draft_folder;
  struct mailimf_address_list * mail_to;
  int poll;
  int max;
  
  unsigned int index;
  unsigned int first;
  unsigned int count;
  int * info_type_tab;
  
  int main_attr;
  int selection_attr;
  int status_attr;
  
  char * account_name;
  char * storage_name;
  struct etpan_app_config * config;
  
  /* upcall */
  void (* upcall)(struct etpan_subapp *, int, void *);
  void * upcall_data;
};

static int sel_account(struct etpan_subapp * app);

static int sel_storage(struct etpan_subapp * app);

static int edit_sent_folder(struct etpan_subapp * app);

static int edit_draft_folder(struct etpan_subapp * app);

static int edit_location(struct etpan_subapp * app);

static void ask_quit(struct etpan_subapp * app);

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  unsigned int list_lines;
  int current_type;
  struct app_state * state;

  list_lines = app->display_height - 1;
  
  state = app->data;
  
  if (state->info_type_tab == NULL)
    current_type = -1;
  else
    current_type = state->info_type_tab[state->index];
  
  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;
    
  case 'h':
  case KEY_LEFT:
    switch (current_type) {
    case INFO_TYPE_POLL:
      if (state->poll > 0)
        state->poll = 0;
      break;
    }

    break;
    
  case 'l':
  case KEY_RIGHT:
    switch (current_type) {
    case INFO_TYPE_POLL:
      if (state->poll == 0)
        state->poll = 1;
      break;
    }
    break;

  case 'k':
  case KEY_UP:
    if (state->index > 0)
      state->index --;
    break;
    
  case 'j':
  case KEY_DOWN:
    if (state->index < state->count - 1)
      state->index ++;
    break;
    
  case KEY_NPAGE:
    if (state->index + list_lines - 1 < state->count)
      state->index += list_lines - 1;
    else
      state->index = state->count - 1;
    break;
    
  case KEY_PPAGE:
    if (state->index >= list_lines - 1)
      state->index -= list_lines - 1;
    else
      state->index = 0;
    break;
    
  case KEY_HOME:
    state->index = 0;
    break;
    
  case KEY_END:
    if (state->count > 0)
      state->index = state->count - 1;
    break;

  case KEY_CTRL('G'):
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_FOLDER_EDIT_CANCEL, state->upcall_data);
    break;

  case 'y':
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_FOLDER_EDIT_VALID, state->upcall_data);
    break;

  case 'q':
    ask_quit(app);
    break;
    
  case 'd':
    switch (current_type) {
    case INFO_TYPE_STORAGE:
      state->storage = NULL;
      break;
    case INFO_TYPE_ACCOUNT:
      state->account = NULL;
      break;
    case INFO_TYPE_SENT_FOLDER:
      state->sent_folder = NULL;
      break;
    case INFO_TYPE_DRAFT_FOLDER:
      state->draft_folder = NULL;
      break;
    case INFO_TYPE_MAIL_TO:
      if (state->mail_to != NULL)
        mailimf_address_list_free(state->mail_to);
      state->mail_to = NULL;
      break;
    case INFO_TYPE_MAX_ARTICLES:
      state->max = 0;
      break;
    }
    break;
    
  case '\n':
    switch (current_type) {
    case INFO_TYPE_NAME:
      etpan_cfg_edit_string(app, "Enter folder name: ",
          state->name, sizeof(state->name), 0);
      break;
    case INFO_TYPE_STORAGE:
      sel_storage(app);
      break;
    case INFO_TYPE_ACCOUNT:
      sel_account(app);
      break;
    case INFO_TYPE_LOCATION:
      edit_location(app);
      break;
    case INFO_TYPE_SENT_FOLDER:
      edit_sent_folder(app);
      break;
    case INFO_TYPE_DRAFT_FOLDER:
      edit_draft_folder(app);
      break;
    case INFO_TYPE_MAIL_TO:
      etpan_cfg_edit_addr(app, "Enter default recipient: ",
          &state->mail_to);
      break;
    case INFO_TYPE_MAX_ARTICLES:
      etpan_cfg_edit_int(app,
          "Maximum articles number (0 for all): ", &state->max);
      break;
    }
    break;
  }
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  int * info_type_tab;
  unsigned int info_type_count;
  unsigned int i;
  unsigned int y;
  unsigned int list_lines;
  char * buffer;
  char * output;
  char * fill;
  unsigned int percent;
  char * p;

  buffer = app->app->buffer;
  output = app->app->output;
  fill = app->app->fill;
  
  list_lines = app->display_height - 1;
  
  state = app->data;
  
  /* update view */
  
  info_type_tab = none_info_type;
  info_type_count = TABLE_SIZE(none_info_type);
  if (state->storage != NULL) {
    int storage_type;
    
    storage_type =
      etpan_cfg_storage_get_type(state->storage->sto_driver->sto_name);
    
    for(i = 0 ; i < TABLE_SIZE(storage_type_tab) ; i ++) {
      if (storage_type_tab[i].type == storage_type) {
        info_type_tab = storage_type_tab[i].info_type_tab;
        info_type_count = storage_type_tab[i].info_type_count;
      }
    }
  }
  
  state->info_type_tab = info_type_tab;
  state->count = info_type_count;
  
  if (state->index > state->count - 1)
    state->index = state->count - 1;
  
  if (state->index < state->first)
    state->first = state->index;
  if (state->index - state->first + 1 > (unsigned int) list_lines)
    state->first = state->index - list_lines + 1;
  
  for(p = state->name ; * p != '\0' ; p ++) {
    if (* p == '/')
      * p = '-';
  }
  
  /* display */
  
  wattron(w, state->main_attr);
  y = 0;
  i = state->first;
  while (y < list_lines) {
    
    if (i > state->count - 1)
      break;
    
    switch (info_type_tab[i]) {
    case INFO_TYPE_NAME:
      snprintf(buffer, app->display_width, "name: %s", state->name);
      break;
      
    case INFO_TYPE_STORAGE:
      if (state->storage == NULL)
        snprintf(buffer, app->display_width, "storage: (no storage)");
      else if (state->storage->sto_id == NULL)
        snprintf(buffer, app->display_width, "storage: (no name)");
      else
        snprintf(buffer, app->display_width, "storage: %s",
            state->storage->sto_id);
      break;
      
    case INFO_TYPE_LOCATION:
        snprintf(buffer, app->display_width,
            "location: %s", state->location);
      break;
      
    case INFO_TYPE_ACCOUNT:
      if (state->account == NULL)
        snprintf(buffer, app->display_width, "account: (default account)");
      else if (state->account->id == NULL)
        snprintf(buffer, app->display_width, "account: (no name)");
      else
        snprintf(buffer, app->display_width, "account: %s",
            state->account->id);
      break;
    case INFO_TYPE_SENT_FOLDER:
      if (state->sent_folder != NULL) {
        char * sent_name;
        
        sent_name = etpan_folder_get_virtual_path(state->sent_folder);
        if (sent_name != NULL) {
          snprintf(buffer, app->display_width, "sent folder: %s", sent_name);
          free(sent_name);
        }
        else {
          snprintf(buffer, app->display_width, "sent folder: (no name)");
        }
      }
      else {
        snprintf(buffer, app->display_width,
            "sent folder: (default sent folder)");
      }
      break;
    case INFO_TYPE_DRAFT_FOLDER:
      if (state->draft_folder != NULL) {
        char * draft_name;
        
        draft_name = etpan_folder_get_virtual_path(state->draft_folder);
        if (draft_name != NULL) {
          snprintf(buffer, app->display_width, "draft folder: %s", draft_name);
          free(draft_name);
        }
        else {
          snprintf(buffer, app->display_width, "draft folder: (no name)");
        }
      }
      else {
        snprintf(buffer, app->display_width,
            "draft folder: (default draft folder)");
      }
      break;
    case INFO_TYPE_MAIL_TO:
      if (state->mail_to != NULL) {
        int done;
        struct mailimf_address_list * dup_mail_to;
        
        done = 0;
        dup_mail_to = etpan_dup_address_list(state->mail_to);
        if (dup_mail_to != NULL) {
          char * mail_to_str;
          
          etpan_addr_list_decode(app->app, dup_mail_to);
          mail_to_str = mailimf_address_list_to_string(dup_mail_to);
          mailimf_address_list_free(dup_mail_to);
          if (mail_to_str != NULL) {
            snprintf(buffer, app->display_width,
                "default recipient: %s", mail_to_str);
            free(mail_to_str);
            done = 1;
          }
        }
        if (!done)
          snprintf(buffer, app->display_width,
              "default recipient: (could not display)");
      }
      else {
        snprintf(buffer, app->display_width,
            "default recipient: (no default recipient)");
      }
      break;
    case INFO_TYPE_POLL:
      if (state->poll)
        snprintf(buffer, app->display_width, "poll: yes");
      else
        snprintf(buffer, app->display_width, "poll: no");
      break;
    case INFO_TYPE_MAX_ARTICLES:
      if (state->max == 0)
        snprintf(buffer, app->display_width,
            "Maximum articles number: (all articles)");
      else
        snprintf(buffer, app->display_width,
            "Maximum articles number: %i", state->max);
      break;
    }
    if (i == state->index) {
      wattroff(w, state->main_attr);
      wattron(w, state->selection_attr);
    }
    snprintf(output, app->display_width, "%s%s", buffer, fill);
    mvwprintw(w, y, 0, "%s", output);
    if (i == state->index) {
      wattroff(w, state->selection_attr);
      wattron(w, state->main_attr);
    }
    
    i ++;
    y ++;
  }

  while (y < list_lines) {
    mvwprintw(w, y, 0, "%s", fill);
    y ++;
  }

  wattroff(w, state->main_attr);
  
  /* status bar */

  wattron(w, state->status_attr);
  
  if (state->count == 0)
    percent = 0;
  else if (state->count ==1)
	percent = 100;
  else
    percent = state->index * 100 / (state->count-1);
  
  snprintf(output, app->display_width + 1,
      " %3i %% | y: ok  ^G: cancel%s", percent, fill);
  
  mvwprintw(w, app->display_height - 1, 0, "%s", output);
  
  wattroff(w, state->status_attr);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
}

static int init(struct etpan_subapp * app)
{
  struct app_state * state;

  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->name[0] = '\0';
  
  state->storage = NULL;
  state->location[0] = '\0';
  state->account = NULL;
  state->sent_folder = NULL;
  state->mail_to = NULL;
  state->poll = 1;
  state->max = 0;
  
  state->index = 0;
  state->first = 0;
  state->count = 0;
  state->info_type_tab = NULL;
  
  /* colors */
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;
  
  state->upcall = NULL;
  state->upcall_data = NULL;
  
  app->data = state;
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;

  state = app->data;
  
  free(state);
}

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - edit folder");
  return etpan_app_subapp_display_init(app);
}


struct etpan_subapp * etpan_folder_editor_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_folder_editor_app_driver);
}

void etpan_folder_editor_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  state->name[0] = '\0';
  
  state->storage = NULL;
  state->location[0] = '\0';
  state->account = NULL;
  state->sent_folder = NULL;
  state->draft_folder = NULL;
  if (state->mail_to != NULL)
    mailimf_address_list_free(state->mail_to);
  state->mail_to = NULL;
  state->poll = 1;
  state->max = 0;
  
  state->index = 0;
  state->first = 0;
  state->count = 0;
  state->info_type_tab = NULL;
  
  state->account_name = NULL;
  state->config = NULL;
  state->upcall = NULL;
  state->upcall_data = NULL;
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  etpan_folder_editor_app_flush(app);
}

void etpan_folder_editor_app_set(struct etpan_subapp * app,
    struct etpan_app_config * config,
    struct mailfolder * folder,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * upcall_data)
{
  struct app_state * state;
  struct mailimf_address_list * mail_to;
  
  state = app->data;
  
  etpan_folder_editor_app_flush(app);
  
  state->account_name = NULL;
  state->config = config;
  state->upcall = upcall;
  state->upcall_data = upcall_data;
  
  if (folder == NULL)
    return;

  if (folder->fld_virtual_name != NULL) {
    strncpy(state->name, folder->fld_virtual_name, sizeof(state->name));
    state->name[sizeof(state->name) - 1] = '\0';
  }
  
  state->storage = folder->fld_storage;
  if (folder->fld_pathname != NULL) {
    strncpy(state->location, folder->fld_pathname, sizeof(state->location));
    state->location[sizeof(state->location) - 1] = '\0';
  }
  
  state->account =
    etpan_vfolder_get_account(state->config->vfolder_config,
        state->config->account_config, folder);
  if (state->account == state->config->account_config->default_account)
    state->account = NULL;
  
  state->sent_folder =
    etpan_vfolder_get_sent_folder(state->config->vfolder_config, folder);
  if (state->sent_folder == state->config->vfolder_config->sent_folder)
    state->sent_folder = NULL;

  state->draft_folder =
    etpan_vfolder_get_draft_folder(state->config->vfolder_config, folder);
  if (state->draft_folder == state->config->vfolder_config->draft_folder)
    state->draft_folder = NULL;
  
  mail_to = etpan_vfolder_get_to(state->config->vfolder_config, folder);
  if (mail_to != NULL)
    mail_to = etpan_dup_address_list(mail_to);
  
  state->mail_to = mail_to;
  
  state->poll = etpan_vfolder_do_poll(state->config->vfolder_config, folder);
  
  state->max = etpan_vfolder_get_max(state->config->vfolder_config, folder);
}




static void
storage_sel_upcall(struct etpan_subapp * storage_sel_app,
    int valid, void * data)
{
  struct etpan_subapp * app;
  struct app_state * state;
  
  app = data;
  state = app->data;

  if (state->storage_name != NULL) {
    state->storage = etpan_storage_get(state->config->storage_config,
        state->storage_name);
    free(state->storage_name);
    state->storage_name = NULL;
  }

  if (valid == ETPAN_STORAGE_SEL_CANCEL) {
    etpan_app_quit_subapp(storage_sel_app);
    return;
  }
  
  state->storage = etpan_storage_sel_app_get_storage(storage_sel_app);
  etpan_app_quit_subapp(storage_sel_app);
  return;
}

static int sel_storage(struct etpan_subapp * app)
{
  struct etpan_subapp * storage_app;
  int r;
  struct app_state * state;
  struct mailstorage * storage;
  
  state = app->data;
  
  storage_app = etpan_app_find_subapp(app->app, "storage-sel",
      0, NULL, NULL);
  if (storage_app == NULL)
    storage_app = etpan_storage_sel_app_new(app->app);
  if (storage_app == NULL)
    return ERROR_MEMORY;

  state->storage_name = NULL;
  if (state->storage != NULL) {
    state->storage_name = strdup(state->storage->sto_id);
    if (state->storage_name == NULL)
      return ERROR_MEMORY;
  }
  
  storage = state->storage;
  
  state->storage = NULL;
  
  r = etpan_storage_sel_app_set(storage_app,
      state->config, storage,
      storage_sel_upcall, app);
  if (r != NO_ERROR) {
    if (state->storage_name != NULL) {
      free(state->storage_name);
      state->storage_name = NULL;
    }
    state->storage = storage;
    return r;
  }
  
  etpan_subapp_set_parent(storage_app, app);
  etpan_app_switch_subapp(storage_app, 0);

  return NO_ERROR;
}



static void
draft_folder_sel_upcall(struct etpan_subapp * folder_sel_app,
    int valid, void * data)
{
  struct etpan_subapp * app;
  struct app_state * state;
  
  app = data;
  state = app->data;
  
  if (valid == ETPAN_FOLDER_SEL_CANCEL) {
    etpan_app_quit_subapp(folder_sel_app);
    return;
  }
  
  state->draft_folder =
    etpan_folder_sel_app_get_selected_folder(folder_sel_app);
  etpan_app_quit_subapp(folder_sel_app);
  return;
}

/* TODO - ability to set SELF directory for draft folder ! */

static int edit_draft_folder(struct etpan_subapp * app)
{
  struct etpan_subapp * folder_app;
  int r;
  struct mailfolder * folder;
  struct app_state * state;
  
  state = app->data;
  
  folder_app = etpan_app_find_subapp(app->app, "folder-sel",
      0, NULL, NULL);
  if (folder_app == NULL)
    folder_app = etpan_folder_sel_app_new(app->app);
  if (folder_app == NULL)
    return ERROR_MEMORY;

  if (state->draft_folder != NULL)
    folder = state->draft_folder;
  else
    folder = state->config->vfolder_config->draft_folder;
  
  r = etpan_folder_sel_app_set(folder_app,
      state->config->vfolder_config->root, folder,
      draft_folder_sel_upcall, app);
  if (r != NO_ERROR)
    return r;
  
  etpan_subapp_set_parent(folder_app, app);
  etpan_app_switch_subapp(folder_app, 0);

  return NO_ERROR;
}


static void
sent_folder_sel_upcall(struct etpan_subapp * folder_sel_app,
    int valid, void * data)
{
  struct etpan_subapp * app;
  struct app_state * state;
  
  app = data;
  state = app->data;
  
  if (valid == ETPAN_FOLDER_SEL_CANCEL) {
    etpan_app_quit_subapp(folder_sel_app);
    return;
  }
  
  state->sent_folder =
    etpan_folder_sel_app_get_selected_folder(folder_sel_app);
  etpan_app_quit_subapp(folder_sel_app);
  return;
}

/* TODO - ability to set SELF directory for sent ! */

static int edit_sent_folder(struct etpan_subapp * app)
{
  struct etpan_subapp * folder_app;
  int r;
  struct mailfolder * folder;
  struct app_state * state;
  
  state = app->data;
  
  folder_app = etpan_app_find_subapp(app->app, "folder-sel",
      0, NULL, NULL);
  if (folder_app == NULL)
    folder_app = etpan_folder_sel_app_new(app->app);
  if (folder_app == NULL)
    return ERROR_MEMORY;

  if (state->sent_folder != NULL)
    folder = state->sent_folder;
  else
    folder = state->config->vfolder_config->sent_folder;
  
  r = etpan_folder_sel_app_set(folder_app,
      state->config->vfolder_config->root, folder,
      sent_folder_sel_upcall, app);
  if (r != NO_ERROR)
    return r;
  
  etpan_subapp_set_parent(folder_app, app);
  etpan_app_switch_subapp(folder_app, 0);

  return NO_ERROR;
}


int etpan_folder_editor_app_get_folder(struct etpan_subapp * app,
    struct mailfolder ** pfolder, struct etpan_vfolder_property * prop)
{
  struct app_state * state;
  struct mailfolder * folder;
  
  state = app->data;
  
  /* store properties */
  
  folder = mailfolder_new(state->storage, state->location, state->name);
  if (folder == NULL)
    goto err;

  if (prop != NULL) {
    prop->account = state->account;
    prop->sent_folder = state->sent_folder;
    prop->draft_folder = state->draft_folder;
    prop->poll = state->poll;
    prop->max = state->max;
    prop->mail_to = state->mail_to;
  }
  
  * pfolder = folder;
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static int edit_location(struct etpan_subapp * app)
{
  struct app_state * state;
  int r;
  int storage_type;

  state = app->data;
  
  if (state->storage == NULL)
    return ERROR_INVAL;
  
  storage_type = etpan_cfg_storage_get_type(state->storage->sto_driver->sto_name);
  
  switch (storage_type) {
  case STORAGE_TYPE_IMAP:
    r = etpan_cfg_edit_imap_mailbox(app, state->storage, "Mailbox: ",
        state->location, sizeof(state->location));
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "edit location - not enough memory"));
      return r;
    }
    break;
    
  case STORAGE_TYPE_NNTP:
    r = etpan_cfg_edit_newsgroup(app, state->storage, "Newsgroup: ",
        state->location, sizeof(state->location));
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "edit location - not enough memory"));
      return r;
    }
    break;
    
  case STORAGE_TYPE_MH:
    if (state->location[0] == '\0') {
      struct mh_mailstorage * mh_data;
      
      mh_data = state->storage->sto_data;
      
      strncpy(state->location, mh_data->mh_pathname, sizeof(state->location));
    }
    
    r = etpan_cfg_edit_filename_with_type(app, "Location of MH mailbox: ",
        state->location, sizeof(state->location), FILENAME_TYPE_MH);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "edit location - not enough memory"));
      return r;
    }
    
    break;
  }
  
  return NO_ERROR;
}



static void
account_sel_upcall(struct etpan_subapp * account_sel_app,
    int valid, void * data)
{
  struct etpan_subapp * app;
  struct app_state * state;
  
  app = data;
  state = app->data;

  if (state->account_name != NULL) {
    state->account = etpan_account_get(state->config->account_config,
        state->account_name);
    free(state->account_name);
    state->account_name = NULL;
  }
  
  if (valid == ETPAN_ACCOUNT_SEL_CANCEL) {
    etpan_app_quit_subapp(account_sel_app);
    return;
  }
  
  state->account = etpan_account_sel_app_get_account(account_sel_app);
  etpan_app_quit_subapp(account_sel_app);
  return;
}

static int sel_account(struct etpan_subapp * app)
{
  struct etpan_subapp * account_app;
  int r;
  struct app_state * state;
  struct etpan_account_info * account;
  
  state = app->data;
  
  account_app = etpan_app_find_subapp(app->app, "account-sel",
      0, NULL, NULL);
  if (account_app == NULL)
    account_app = etpan_account_sel_app_new(app->app);
  if (account_app == NULL)
    return ERROR_MEMORY;
  
  state->account_name = NULL;
  if (state->account != NULL) {
    state->account_name = strdup(state->account->id);
    if (state->account_name == NULL)
      return ERROR_MEMORY;
  }
  
  account = state->account;
  
  state->account = NULL;
  
  if (account == NULL)
    account = state->config->account_config->default_account;
  
  r = etpan_account_sel_app_set(account_app,
      state->config, account,
      account_sel_upcall, app);
  if (r != NO_ERROR) {
    if (state->account_name != NULL) {
      free(state->account_name);
      state->account_name = NULL;
    }
    if (account != state->config->account_config->default_account)
      state->account = account;
    return r;
  }
  
  etpan_subapp_set_parent(account_app, app);
  etpan_app_switch_subapp(account_app, 0);

  return NO_ERROR;
}


static void ask_quit_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  struct app_state * state;
  char * name;
  char * dup_name;
  struct etpan_subapp * app;
  int r;
  char * result;
  int do_quit;

  app = data;
  state = app->data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  result = etpan_search_input_get_value(input_app);
  if ((* result != 'y') && (* result != 'n')) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  do_quit = (* result == 'y');
  etpan_app_quit_subapp(input_app);
  
  if (do_quit) {
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_FOLDER_EDIT_VALID, state->upcall_data);
  }
  else {
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_FOLDER_EDIT_CANCEL, state->upcall_data);
  }
}


static void ask_quit(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_subapp * input;
  int r;
  
  input = etpan_app_find_subapp(app->app, "search-input",
      0, NULL, NULL);
  if (input == NULL) {
    input = etpan_search_input_new(app->app);
    if (input == NULL)
      goto err;
  }
  
  r = etpan_search_input_set(input,
      "save the folder (y/n) ? ", 1,
      NULL, 0,
      ask_quit_upcall, app);
  if (input == NULL)
    goto err;
  
  etpan_subapp_set_parent(input, app);
  etpan_app_switch_subapp(input, 0);
  
  return;
  
 err:
  ETPAN_APP_LOG((app->app, "configuration - not enough memory"));
}


#define HELP_TEXT \
"\
Help for folder properties\n\
--------------------------\n\
\n\
This application will let you edit the properties of a folder.\n\
\n\
- up, down\n\
  arrow keys : move cursor\n\
\n\
- left, right\n\
  arrow keys : change value\n\
- d          : clear selected value \n\
- [Enter]    : edit value\n\
\n\
- y          : finished edition of properties\n\
- Ctrl-G     : cancel\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
