/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001 - 2003 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-storage.c,v 1.13 2004/11/12 13:57:02 hoa Exp $
 */

#include "etpan-cfg-storage.h"

#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <libetpan/libetpan.h>

#include "etpan-db-read.h"
#include "etpan-errors.h"
#include "etpan-cfg-common.h"
#include "etpan-tools.h"

static struct etpan_storage_property * storage_property_new(void)
{
  struct etpan_storage_property * prop;
  
  prop = malloc(sizeof(* prop));
  if (prop == NULL)
    return NULL;
  
  prop->generated = 0;
  
  return prop;
}

static void storage_property_free(struct etpan_storage_property * prop)
{
  free(prop);
}

static struct etpan_storage_property *
get_prop(struct etpan_storage_config * config,
    struct mailstorage * storage)
{
  chashdatum key;
  chashdatum value;
  
  key.data = &storage;
  key.len = sizeof(storage);
  chash_get(config->storage_prop, &key, &value);
  
  return value.data;
}

void etpan_storage_config_set_generated(struct etpan_storage_config * config,
    struct mailstorage * storage, int generated)
{
  struct etpan_storage_property * prop;
  
  prop = get_prop(config, storage);;
  
  prop->generated = generated;
}

int etpan_storage_config_is_generated(struct etpan_storage_config * config,
    struct mailstorage * storage)
{
  struct etpan_storage_property * prop;
  
  prop = get_prop(config, storage);;
  
  return prop->generated;
}

int etpan_storage_config_add(struct etpan_storage_config * config,
    struct mailstorage * storage, int generated)
{
  unsigned int index;
  struct etpan_storage_property * prop;
  int r;
  chashdatum key;
  chashdatum value;
  
  r = carray_add(config->storage_tab, storage, &index);
  if (r < 0) {
    goto err;
  }
  
  prop = storage_property_new();
  if (prop == NULL) {
    goto remove;
  }
  prop->generated = generated;
  
  key.data = &storage;
  key.len = sizeof(storage);
  value.data = prop;
  value.len = 0;
  
  r = chash_set(config->storage_prop, &key, &value, NULL);
  if (r < 0) {
    goto free_prop;
  }
  
  return NO_ERROR;
  
 free_prop:
  storage_property_free(prop);
 remove:
  carray_delete(config->storage_tab, index);
 err:
  return ERROR_MEMORY;
}


int etpan_storage_config_replace(struct etpan_storage_config * config,
    unsigned int index,
    struct mailstorage * storage, int generated)
{
  struct etpan_storage_property * prop;
  int r;
  chashdatum key;
  chashdatum value;
  struct mailstorage * old_storage;

  old_storage = carray_get(config->storage_tab, index);
  
  key.data = &storage;
  key.len = sizeof(storage);
  chash_get(config->storage_prop, &key, &value);
  prop = value.data;
  
  storage_property_free(prop);
  mailstorage_free(old_storage);
  
  carray_set(config->storage_tab, index, storage);
  
  prop = storage_property_new();
  if (prop == NULL) {
    goto remove;
  }
  prop->generated = generated;
  
  key.data = &storage;
  key.len = sizeof(storage);
  value.data = prop;
  value.len = 0;
  
  r = chash_set(config->storage_prop, &key, &value, NULL);
  if (r < 0) {
    goto free_prop;
  }
  
  return NO_ERROR;
  
 free_prop:
  storage_property_free(prop);
 remove:
  carray_delete(config->storage_tab, index);
 err:
  return ERROR_MEMORY;
}


/* TODO: find something so that we don't have to run through all table */

void etpan_storage_config_delete(struct etpan_storage_config * config,
    struct mailstorage * storage)
{
  chashdatum key;
  chashdatum value;
  struct etpan_storage_property * prop;
  unsigned int i;
  
  for(i = 0 ; i < carray_count(config->storage_tab) ; i ++) {
    struct mailstorage * cur_storage;
    
    cur_storage = carray_get(config->storage_tab, i);
    if (cur_storage == storage)
      carray_delete_slow(config->storage_tab, i);
  }
  
  key.data = &storage;
  key.len = sizeof(storage);
  chash_get(config->storage_prop, &key, &value);
  prop = value.data;
  
  storage_property_free(prop);
  mailstorage_free(storage);
}


#define TABLE_SIZE(a) (sizeof(a) / sizeof(a[0]))

static struct etpan_type_name connection_types_name[] = {
  {CONNECTION_TYPE_PLAIN, "plain"},
  {CONNECTION_TYPE_STARTTLS, "starttls"},
  {CONNECTION_TYPE_TRY_STARTTLS, "try-starttls"},
  {CONNECTION_TYPE_TLS, "tls"},
  {CONNECTION_TYPE_COMMAND, "command"},
  {CONNECTION_TYPE_COMMAND_STARTTLS, "command-starttls"},
  {CONNECTION_TYPE_COMMAND_TRY_STARTTLS, "command-try-starttls"},
  {CONNECTION_TYPE_COMMAND_TLS, "command-tls"},
};

const char * etpan_cfg_storage_get_connection_type_name(int type)
{
  return etpan_cfg_get_name_from_type(connection_types_name,
      TABLE_SIZE(connection_types_name), type);
}

static char * translate_password(char * passwd)
{
  return etpan_decode_base64(passwd, strlen(passwd));
}

/* imap */

static struct etpan_type_name imap_auth_types_name[] = {
  {IMAP_AUTH_TYPE_PLAIN, "plain"},
  {IMAP_AUTH_TYPE_SASL_ANONYMOUS, "sasl-anonymous"},
  {IMAP_AUTH_TYPE_SASL_CRAM_MD5, "sasl-cram-md5"},
  {IMAP_AUTH_TYPE_SASL_KERBEROS_V4, "sasl-kerberos-v4"},
  {IMAP_AUTH_TYPE_SASL_PLAIN, "sasl-plain"},
  {IMAP_AUTH_TYPE_SASL_SCRAM_MD5, "scram-md5"},
  {IMAP_AUTH_TYPE_SASL_GSSAPI, "gssapi"},
  {IMAP_AUTH_TYPE_SASL_DIGEST_MD5, "digest-md5"},
};

const char * etpan_cfg_storage_get_imap_auth_name(int type)
{
  return etpan_cfg_get_name_from_type(imap_auth_types_name,
      TABLE_SIZE(imap_auth_types_name), type);
}

/* pop3 */

static struct etpan_type_name pop3_auth_types_name[] = {
  {POP3_AUTH_TYPE_PLAIN, "plain"},
  {POP3_AUTH_TYPE_APOP, "apop"},
  {POP3_AUTH_TYPE_TRY_APOP, "try-apop"},
  {POP3_AUTH_TYPE_SASL_ANONYMOUS, "sasl-anonymous"},
  {POP3_AUTH_TYPE_SASL_CRAM_MD5, "sasl-cram-md5"},
  {POP3_AUTH_TYPE_SASL_KERBEROS_V4, "sasl-kerberos-v4"},
  {POP3_AUTH_TYPE_SASL_PLAIN, "sasl-plain"},
  {POP3_AUTH_TYPE_SASL_SCRAM_MD5, "scram-md5"},
  {POP3_AUTH_TYPE_SASL_GSSAPI, "gssapi"},
  {POP3_AUTH_TYPE_SASL_DIGEST_MD5, "digest-md5"},
};

const char * etpan_cfg_storage_get_pop3_auth_name(int type)
{
  return etpan_cfg_get_name_from_type(pop3_auth_types_name,
      TABLE_SIZE(pop3_auth_types_name), type);
}

/* nntp */

static struct etpan_type_name nntp_auth_types_name[] = {
  {NNTP_AUTH_TYPE_PLAIN, "plain"},
};

const char * etpan_cfg_storage_get_nntp_auth_name(int type)
{
  return etpan_cfg_get_name_from_type(nntp_auth_types_name,
      TABLE_SIZE(nntp_auth_types_name), type);
}

static struct etpan_type_name storage_types_name[] = {
  {STORAGE_TYPE_IMAP, "imap"},
  {STORAGE_TYPE_POP3, "pop3"},
  {STORAGE_TYPE_NNTP, "nntp"},
  {STORAGE_TYPE_MH, "mh"},
  {STORAGE_TYPE_MBOX, "mbox"},
  {STORAGE_TYPE_MAILDIR, "maildir"},
  {STORAGE_TYPE_DB, "db"},
};

int etpan_cfg_storage_get_type(char * name)
{
  return etpan_cfg_get_type_from_name(storage_types_name,
      TABLE_SIZE(storage_types_name), name);
}

struct etpan_storage_config * etpan_storage_config_new(void)
{
  struct etpan_storage_config * config;
  carray * storage_tab;
  chash * storage_prop;
  
  storage_tab = carray_new(16);
  if (storage_tab == NULL)
    goto err;
  
  storage_prop = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (storage_prop == NULL)
    goto free_list;
  
  config = malloc(sizeof(* config));
  if (config == NULL)
    goto free_hash;
  
  config->storage_tab = storage_tab;
  config->storage_prop = storage_prop;

  return config;

 free_hash:
  chash_free(storage_prop);
 free_list:
  carray_free(storage_tab);
 err:
  return NULL;
}

void etpan_storage_config_free(struct etpan_storage_config * config)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(config->storage_tab) ; i ++) {
    struct mailstorage * storage;
    chashdatum key;
    chashdatum value;
    struct etpan_storage_property * prop;
    
    storage = carray_get(config->storage_tab, i);
    mailstorage_free(storage);
    
    key.data = &storage;
    key.len = sizeof(storage);
    chash_get(config->storage_prop, &key, &value);
    prop = value.data;
    
    storage_property_free(prop);
  }
  carray_free(config->storage_tab);
  chash_free(config->storage_prop);
  
  free(config);
}



static void server_read(chash * entry,
    char ** pname, uint16_t * pport,
    char ** pcommand,
    int * pconnection_type)
{
  char * hostname;
  char * port_str;
  uint16_t port;
  char * command;
  char * connection_str;
  int connection_type;

  /* server */
  hostname = etpan_db_entry_get_value(entry, "hostname");
  port_str = etpan_db_entry_get_value(entry, "port");
  if (port_str != NULL)
    port = strtoul(port_str, NULL, 10);
  else
    port = 0;
  command = etpan_db_entry_get_value(entry, "command");
  connection_str = etpan_db_entry_get_value(entry, "connection");
  connection_type = etpan_cfg_get_type_from_name(connection_types_name,
      TABLE_SIZE(connection_types_name), connection_str);

  if (hostname == NULL)
    hostname = "localhost";

  if (connection_type == -1)
    connection_type = CONNECTION_TYPE_PLAIN;
  
  * pname = hostname;
  * pport = port;
  * pcommand = command;
  * pconnection_type = connection_type;
}


/* mh */

static int storage_read_mh(chash * entry, char * id,
			   struct mailstorage ** result)
{
  int cached;
  int r;
  struct mailstorage * storage;
  char * location;
  int res;
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  char * cached_str;

  location = etpan_db_entry_get_value(entry, "location");
  if (location == NULL) {
    res = ERROR_INVAL;
    goto err;
  }
  
  cached_str = etpan_db_entry_get_value(entry, "cached");
  if (cached_str != NULL)
    cached = strtol(cached_str, NULL, 10);
  else
    cached = 0;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  snprintf(flags_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);

  r = mh_mailstorage_init(storage, location, cached,
      cache_directory, flags_directory);
  if (r != NO_ERROR) {
    res = ERROR_MEMORY;
    goto free;
  }
    
  * result = storage;

  return NO_ERROR;

 free:
  mailstorage_free(storage);
 err:
  return res;
}

/* mbox */

static int storage_read_mbox(chash * entry, char * id,
			     struct mailstorage ** result)
{
  int cached;
  int r;
  struct mailstorage * storage;
  char * location;
  int res;
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  char * cached_str;
  
  location = etpan_db_entry_get_value(entry, "location");
  if (location == NULL) {
    res = ERROR_INVAL;
    goto err;
  }

  cached_str = etpan_db_entry_get_value(entry, "cached");
  if (cached_str != NULL)
    cached = strtol(cached_str, NULL, 10);
  else
    cached = 0;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  snprintf(flags_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);

  r = mbox_mailstorage_init(storage, location, cached,
      cache_directory, flags_directory);
  if (r != NO_ERROR) {
    res = ERROR_MEMORY;
    goto free;
  }

  * result = storage;

  return NO_ERROR;

 free:
  mailstorage_free(storage);
 err:
  return res;
}


static int storage_read_nntp(chash * entry, char * id,
			     struct mailstorage ** result)
{
  char * hostname;
  uint16_t port;
  char * command;
  int connection_type;
  int cached;
  int r;
  struct mailstorage * storage;
  char * login;
  char * password;
  int auth_type;
  int res;
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  char * auth_str;
  char * cached_str;
  
  /* server */
  server_read(entry, &hostname, &port, &command, &connection_type);

  /* auth */
  login = etpan_db_entry_get_value(entry, "login");
  password = etpan_db_entry_get_value(entry, "password");
  auth_str = etpan_db_entry_get_value(entry, "auth");
  if (auth_str != NULL)
    auth_type = etpan_cfg_get_type_from_name(nntp_auth_types_name,
        TABLE_SIZE(nntp_auth_types_name), auth_str);
  else
    auth_type = -1;
  if (auth_type == -1)
    auth_type = NNTP_AUTH_TYPE_PLAIN;
  cached_str = etpan_db_entry_get_value(entry, "cached");
  if (cached_str != NULL)
    cached = strtol(cached_str, NULL, 10);
  else
    cached = 0;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  snprintf(flags_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);
  
  if (password != NULL) {
    password = translate_password(password);
    if (password == NULL) {
      res = ERROR_MEMORY;
      goto free;
    }
  }

  r = nntp_mailstorage_init(storage, hostname, port,
      command,
      connection_type,
      auth_type, login, password,
      cached, cache_directory, flags_directory);
  free(password);
  if (r != NO_ERROR) {
    res = ERROR_MEMORY;
    goto free;
  }

  * result = storage;

  return NO_ERROR;

 free:
  mailstorage_free(storage);
 err:
  return res;
}


static int storage_read_imap(chash * entry, char * id,
    struct mailstorage ** result)
{
  char * hostname;
  uint16_t port;
  char * command;
  int connection_type;
  int cached;
  int r;
  struct mailstorage * storage;
  char * login;
  char * password;
  int auth_type;
  int res;
  char cache_directory[PATH_MAX];
  char * auth_str;
  char * cached_str;
  
  /* server */
  server_read(entry, &hostname, &port, &command, &connection_type);

  /* auth */
  login = etpan_db_entry_get_value(entry, "login");
  password = etpan_db_entry_get_value(entry, "password");
  auth_str = etpan_db_entry_get_value(entry, "auth");
  if (auth_str != NULL)
    auth_type = etpan_cfg_get_type_from_name(imap_auth_types_name,
        TABLE_SIZE(imap_auth_types_name), auth_str);
  else
    auth_type = -1;
  if (auth_type == -1)
    auth_type = IMAP_AUTH_TYPE_PLAIN;
  cached_str = etpan_db_entry_get_value(entry, "cached");
  if (cached_str != NULL)
    cached = strtol(cached_str, NULL, 10);
  else
    cached = 0;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);

  if (password != NULL) {
    password = translate_password(password);
    if (password == NULL) {
      res = ERROR_MEMORY;
      goto free;
    }
  }
  r = imap_mailstorage_init(storage, hostname, port, command,
      connection_type, auth_type, login, password,
      cached, cache_directory);
  free(password);

  if (r != MAIL_NO_ERROR) {
    res = ERROR_MEMORY;
    goto free;
  }

  * result = storage;

  return MAIL_NO_ERROR;

 free:
  mailstorage_free(storage);
 err:
  return res;
}



static int storage_read_pop3(chash * entry, char * id,
    struct mailstorage ** result)
{
  char * hostname;
  uint16_t port;
  char * command;
  int connection_type;
  int cached;
  int r;
  struct mailstorage * storage;
  char * login;
  char * password;
  int auth_type;
  int res;
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  char * auth_str;
  char * cached_str;

  /* server */
  server_read(entry, &hostname, &port, &command, &connection_type);
 
  /* authentication */
  login = etpan_db_entry_get_value(entry, "login");
  password = etpan_db_entry_get_value(entry, "password");
  auth_str = etpan_db_entry_get_value(entry, "auth");
  if (auth_str != NULL)
    auth_type = etpan_cfg_get_type_from_name(pop3_auth_types_name,
        TABLE_SIZE(pop3_auth_types_name), auth_str);
  else
    auth_type = -1;
  if (auth_type == -1)
    auth_type = POP3_AUTH_TYPE_PLAIN;
  cached_str = etpan_db_entry_get_value(entry, "cached");
  if (cached_str != NULL)
    cached = strtol(cached_str, NULL, 10);
  else
    cached = 0;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  snprintf(flags_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);

  if (password != NULL) {
    password = translate_password(password);
    if (password == NULL) {
      res = ERROR_MEMORY;
      goto free;
    }
  }

  r = pop3_mailstorage_init(storage, hostname, port, command, connection_type,
			     auth_type, login, password,
			     cached, cache_directory, flags_directory);
  free(password);
  if (r != NO_ERROR) {
    res = ERROR_MEMORY;
    goto free;
  }

  * result = storage;

  return NO_ERROR;

 free:
  mailstorage_free(storage);
 err:
  return res;
}


/* maildir */

static int storage_read_maildir(chash * entry, char * id,
    struct mailstorage ** result)
{
  int cached;
  int r;
  struct mailstorage * storage;
  char * location;
  int res;
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  char * cached_str;
  
  location = etpan_db_entry_get_value(entry, "location");
  if (location == NULL) {
    res = ERROR_INVAL;
    goto err;
  }

  cached_str = etpan_db_entry_get_value(entry, "cached");
  if (cached_str != NULL)
    cached = strtol(cached_str, NULL, 10);
  else
    cached = 0;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  snprintf(flags_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);

  r = maildir_mailstorage_init(storage, location, cached,
      cache_directory, flags_directory);
  if (r != NO_ERROR) {
    res = r;
    goto free;
  }

  * result = storage;

  return NO_ERROR;

 free:
  mailstorage_free(storage);
 err:
  return res;
}


/* DB */

static int storage_read_db(chash * entry, char * id,
    struct mailstorage ** result)
{
  int r;
  struct mailstorage * storage;
  char * location;
  int res;
  
  location = etpan_db_entry_get_value(entry, "location");
  if (location == NULL) {
    res = ERROR_INVAL;
    goto err;
  }

  storage =  mailstorage_new(id);
  if (storage == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  r = db_mailstorage_init(storage, location);
  if (r != NO_ERROR) {
    res = r;
    goto free;
  }

  * result = storage;

  return NO_ERROR;

 free:
  mailstorage_free(storage);
 err:
  return res;
}


static int storage_read(chash * entry, struct mailstorage ** result)
{
  int type;
  char * id;
  char * typename;

  typename = etpan_db_entry_get_value(entry, "type");
  id = etpan_db_entry_get_value(entry, "name");

  type = etpan_cfg_get_type_from_name(storage_types_name,
      TABLE_SIZE(storage_types_name), typename);

  if (type == -1)
    return ERROR_PARSE;
  
  if (id == NULL)
    return ERROR_PARSE;

  switch (type) {
  case STORAGE_TYPE_IMAP:
    return storage_read_imap(entry, id, result);

  case STORAGE_TYPE_POP3:
    return storage_read_pop3(entry, id, result);

  case STORAGE_TYPE_NNTP:
    return storage_read_nntp(entry, id, result);

  case STORAGE_TYPE_MH:
    return storage_read_mh(entry, id, result);

  case STORAGE_TYPE_MBOX:
    return storage_read_mbox(entry, id, result);

  case STORAGE_TYPE_MAILDIR:
    return storage_read_maildir(entry, id, result);

  case STORAGE_TYPE_DB:
    return storage_read_db(entry, id, result);
    
  default:
    return ERROR_PARSE;
  }
}


int etpan_storage_config_read(char * filename,
    struct etpan_storage_config ** result)
{
  struct etpan_db * db;
  int r;
  int res;
  struct etpan_storage_config * config;
  unsigned int i;

  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  config = etpan_storage_config_new();
  if (config == NULL) {
    res = ERROR_MEMORY;
    goto free_db;
  }
  
  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    chash * entry;
    struct mailstorage * storage;

    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    r = storage_read(entry, &storage);
    if (r != NO_ERROR) {
      char * name;
      char * id;

      id = etpan_db_entry_get_value(entry, "id");
      name = etpan_db_entry_get_value(entry, "name");
      printf("storage error - %s - %s\n", id, name);
      continue;
    }
    
    r = etpan_storage_config_add(config, storage, 0);
    if (r != NO_ERROR) {
      mailstorage_free(storage);
      res = r;
      goto free_config;
    }
  }
  
  * result = config;
  
  etpan_db_free(db);
  
  return MAIL_NO_ERROR;

 free_config:
  etpan_storage_config_free(config);
 free_db:
  etpan_db_free(db);
 err:
  return res;
}

struct mailstorage *
etpan_storage_get(struct etpan_storage_config * config, char * name)
{
  unsigned int i;
  
  /* TODO : use hash table */
  for(i = 0 ; i < carray_count(config->storage_tab) ; i ++) {
    struct mailstorage * storage;
    
    storage = carray_get(config->storage_tab, i);
    
    if (storage->sto_id != NULL) {
      if (strcasecmp(storage->sto_id, name) == 0)
        return storage;
    }
  }
  
  return NULL;
}






/* write conf */

static void write_server_info(FILE * f, char * hostname, int port,
    char * command, int connection_type)
{
  const char * connection_type_name;
  
  if (hostname != NULL) {
    fprintf(f, "hostname = %s\n", hostname);
    if (port != 0)
      fprintf(f, "port = %i\n", port);
  }
  
  if (command != NULL)
    fprintf(f, "command = %s\n", command);

  connection_type_name =
    etpan_cfg_storage_get_connection_type_name(connection_type);
  fprintf(f, "connection = %s\n", connection_type_name);
}

static void write_login_info(FILE * f, char * login, char * password)
{
  if (login != NULL) {
    fprintf(f, "login = %s\n", login);
  
    if (password != NULL) {
      char * encoded_pass;
      
      encoded_pass = etpan_encode_base64(password, strlen(password));
      fprintf(f, "password = %s\n", encoded_pass);
      free(encoded_pass);
    }
  }
}

/* write imap storage conf */

static void write_imap_storage_conf(FILE * f, unsigned int id,
    struct mailstorage * storage)
{
  struct imap_mailstorage * imap_data;
  const char * auth_type_name;

  if (storage->sto_id == NULL)
    return;
  
  imap_data = storage->sto_data;
  fprintf(f, "id = %i\n", id);
  fprintf(f, "name = %s\n", storage->sto_id);
  fprintf(f, "type = imap\n");
  
  write_server_info(f, imap_data->imap_servername, imap_data->imap_port,
      imap_data->imap_command, imap_data->imap_connection_type);
  
  auth_type_name =
    etpan_cfg_storage_get_imap_auth_name(imap_data->imap_auth_type);
  fprintf(f, "auth = %s\n", auth_type_name);
  
  write_login_info(f, imap_data->imap_login, imap_data->imap_password);
  
  fprintf(f, "cached = %i\n", imap_data->imap_cached);
  fprintf(f, "\n");
}

/* pop3 */

static void write_pop3_storage_conf(FILE * f, unsigned int id,
    struct mailstorage * storage)
{
  struct pop3_mailstorage * pop3_data;
  const char * auth_type_name;

  if (storage->sto_id == NULL)
    return;
  
  pop3_data = storage->sto_data;
  fprintf(f, "id = %i\n", id);
  fprintf(f, "name = %s\n", storage->sto_id);
  fprintf(f, "type = pop3\n");
  
  write_server_info(f, pop3_data->pop3_servername, pop3_data->pop3_port,
      pop3_data->pop3_command, pop3_data->pop3_connection_type);
  
  auth_type_name =
    etpan_cfg_storage_get_pop3_auth_name(pop3_data->pop3_auth_type);
  fprintf(f, "auth = %s\n", auth_type_name);
  
  write_login_info(f, pop3_data->pop3_login, pop3_data->pop3_password);
  
  fprintf(f, "cached = %i\n", pop3_data->pop3_cached);
  fprintf(f, "\n");
}

/* nntp */

static void write_nntp_storage_conf(FILE * f, unsigned int id,
    struct mailstorage * storage)
{
  struct nntp_mailstorage * nntp_data;
  const char * auth_type_name;

  if (storage->sto_id == NULL)
    return;
  
  nntp_data = storage->sto_data;
  fprintf(f, "id = %i\n", id);
  fprintf(f, "name = %s\n", storage->sto_id);
  fprintf(f, "type = nntp\n");
  
  write_server_info(f, nntp_data->nntp_servername, nntp_data->nntp_port,
      nntp_data->nntp_command, nntp_data->nntp_connection_type);
  
  auth_type_name =
    etpan_cfg_storage_get_pop3_auth_name(nntp_data->nntp_auth_type);
  fprintf(f, "auth = %s\n", auth_type_name);
  
  write_login_info(f, nntp_data->nntp_login, nntp_data->nntp_password);
  
  fprintf(f, "cached = %i\n", nntp_data->nntp_cached);
  fprintf(f, "\n");
}

/* mbox */

static void write_mbox_storage_conf(FILE * f, unsigned int id,
    struct mailstorage * storage)
{
  struct mbox_mailstorage * mbox_data;

  if (storage->sto_id == NULL)
    return;
  
  mbox_data = storage->sto_data;
  fprintf(f, "id = %i\n", id);
  fprintf(f, "name = %s\n", storage->sto_id);
  fprintf(f, "type = mbox\n");
  fprintf(f, "location = %s\n", mbox_data->mbox_pathname);
  fprintf(f, "cached = %i\n", mbox_data->mbox_cached);
  fprintf(f, "\n");
}

/* mh */

static void write_mh_storage_conf(FILE * f, unsigned int id,
    struct mailstorage * storage)
{
  struct mh_mailstorage * mh_data;

  if (storage->sto_id == NULL)
    return;
  
  mh_data = storage->sto_data;
  fprintf(f, "id = %i\n", id);
  fprintf(f, "name = %s\n", storage->sto_id);
  fprintf(f, "type = mh\n");
  fprintf(f, "location = %s\n", mh_data->mh_pathname);
  fprintf(f, "cached = %i\n", mh_data->mh_cached);
  fprintf(f, "\n");
}

/* maildir */

static void write_maildir_storage_conf(FILE * f, unsigned int id,
    struct mailstorage * storage)
{
  struct maildir_mailstorage * maildir_data;

  if (storage->sto_id == NULL)
    return;
  
  maildir_data = storage->sto_data;
  fprintf(f, "id = %i\n", id);
  fprintf(f, "name = %s\n", storage->sto_id);
  fprintf(f, "type = maildir\n");
  fprintf(f, "location = %s\n", maildir_data->md_pathname);
  fprintf(f, "cached = %i\n", maildir_data->md_cached);
  fprintf(f, "\n");
}


/* db */

static void write_db_storage_conf(FILE * f, unsigned int id,
    struct mailstorage * storage)
{
  struct db_mailstorage * db_data;

  if (storage->sto_id == NULL)
    return;
  
  db_data = storage->sto_data;
  fprintf(f, "id = %i\n", id);
  fprintf(f, "name = %s\n", storage->sto_id);
  fprintf(f, "type = db\n");
  fprintf(f, "location = %s\n", db_data->db_pathname);
  fprintf(f, "\n");
}


static int write_storage_conf(FILE * f, unsigned int id,
    struct mailstorage * storage)
{
  int type;
  
  type = etpan_cfg_storage_get_type(storage->sto_driver->sto_name);
  
  if (type == -1)
    return ERROR_INVAL;
  
  switch (type) {
  case STORAGE_TYPE_IMAP:
    write_imap_storage_conf(f, id, storage);
    break;

  case STORAGE_TYPE_POP3:
    write_pop3_storage_conf(f, id, storage);
    break;

  case STORAGE_TYPE_NNTP:
    write_nntp_storage_conf(f, id, storage);
    break;

  case STORAGE_TYPE_MH:
    write_mh_storage_conf(f, id, storage);
    break;

  case STORAGE_TYPE_MBOX:
    write_mbox_storage_conf(f, id, storage);
    break;

  case STORAGE_TYPE_MAILDIR:
    write_maildir_storage_conf(f, id, storage);
    break;

  case STORAGE_TYPE_DB:
    write_db_storage_conf(f, id, storage);
    break;
    
  default:
    return ERROR_INVAL;
  }
  
  return NO_ERROR;
}

int etpan_cfg_storage_write(char * filename,
    struct etpan_storage_config * config)
{
  FILE * f;
  unsigned int i;
  mode_t old_mask;

  old_mask = umask(0077);
  f = fopen(filename, "w");
  umask(old_mask);
  if (f == NULL)
    return ERROR_FILE;
  
  for(i = 0 ; i < carray_count(config->storage_tab) ; i ++) {
    struct mailstorage * storage;
    
    storage = carray_get(config->storage_tab, i);
    write_storage_conf(f, i, storage);
  }
  
  fclose(f);
  
  return NO_ERROR;
}

