# -*- coding: utf-8 -*-

# Copyright (c) 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the handler class for reading an XML project session file.
"""

import codecs
import sys
import os
from types import UnicodeType

from ThirdParty.XMLFilter import XMLSAXHandler

from qt import qApp, QTranslator

class WizardHandler(XMLSAXHandler):
    """
    Class implementing a sax handler to read an XML project session file.
    
    The wizards configuration file must have a &lt;Wizards&gt;&lt;/Wizards&gt;
    entry per wizard. Each of these entries must have the following sections.
    
    &lt;Import&gt;: This names the module to be imported.<br />
    &lt;Class&gt;: This names the class.<br />
    &lt;Translations&gt;: This names the basename of the translation files.
    
    Import and Class are mandatory, Translations is optional.

    The value of the Translations section will be combined with the
    location string determined at startup time and be given
    an extension of '.qm' to yield the filename to be loaded
    into a QTranslator object.
    
    An entry of &lt;Separator /&gt; will yield a separator line in the
    Wizards menu.
    """
    def __init__(self, wizards, wizdir, locale):
        """
        Constructor
        
        @param wizards reference to the Wizard manager
        @param wizdir directory containing the wizards
        @param locale locale string used for translations
        """
        self.elements = {
            'Wizard' : (self.startWizard, self.endWizard),
            'Import' : (self.defaultStartElement, self.endImport),
            'Class' : (self.defaultStartElement, self.endClass),
            'Translations' : (self.defaultStartElement, self.endTranslations),
            'Separator' : (self.startSeparator, self.defaultEndElement),
        }
    
        self.wizards = wizards
        self.wizDir = wizdir
        self.locale = locale
        
        self.encoder = codecs.lookup(sys.getdefaultencoding())[0] # encode,decode,reader,writer
        
    def utf8_to_code(self, text):
        """
        Private method to convert a string to unicode and encode it for XML.
        
        @param text the text to encode (string)
        """
        if type(text) is not UnicodeType:
            text = unicode(text, "utf-8")
        return self.encoder(text)[0] # result,size
        
    def unescape(self, text):
        """
        Private method used to unescape certain characters.
        
        @param text the text to unescape (string)
        """
        return text.replace("&gt;",">").replace("&lt;","<").replace("&amp;","&")
        
    def startDocument(self):
        """
        Handler called, when the document parsing is started.
        """
        self.buffer = ""
        
    def startElement(self, name, attrs):
        """
        Handler called, when a starting tag is found.
        
        @param name name of the tag (string)
        @param attrs list of tag attributes
        """
        try:
            self.elements[name][0](attrs)
        except KeyError:
            pass
            
    def endElement(self, name):
        """
        Handler called, when an ending tag is found.
        
        @param name name of the tag (string)
        """
        try:
            self.elements[name][1]()
        except KeyError:
            pass
            
    def characters(self, chars):
        """
        Handler called for ordinary text.
        
        @param chars the scanned text (string)
        """
        self.buffer += chars
        
    ###################################################
    ## below follow the individual handler functions
    ###################################################
    
    def defaultStartElement(self, attrs):
        """
        Handler method for common start tags.
        
        @param attrs list of tag attributes
        """
        self.buffer = ""
        
    def defaultEndElement(self):
        """
        Handler method for the common end tags.
        """
        pass
        
    def startWizard(self, attrs):
        """
        Handler method for the "Wizard" start tag.
        
        @param attrs list of tag attributes
        """
        self.imp = None
        self.cls = None
        self.trans = None
        
    def endWizard(self):
        """
        Handler method for the "Wizard" end tag.
        """
        try:
            exec 'import %s' % self.imp
            exec 'wizard = %s.%s(self.wizards)' % (self.imp, self.cls)
            self.wizards.registeredWizards.append(wizard)
            if self.trans is not None:
                qtrans = QTranslator(None)
                dir, dummy = os.path.splitext('%s.%s' % (self.wizDir, self.imp))
                dir = dir.replace('.', os.sep)
                if qtrans.load(self.trans, dir):
                    self.wizards.registeredTranslators.append(qtrans)
                    qApp.installTranslator(qtrans)
        except:
            print 'Could not bind wizard %s.%s!' % (self.imp, self.cls)
            
    def endImport(self):
        """
        Handler method for the "Import" end tag.
        """
        self.imp = self.utf8_to_code(self.buffer)
        
    def endClass(self):
        """
        Handler method for the "Class" end tag.
        """
        self.cls = self.utf8_to_code(self.buffer)
        
    def endTranslations(self):
        """
        Handler method for the "Translations" end tag.
        """
        self.trans = "%s%s" % (self.utf8_to_code(self.buffer), self.locale)
        
    def startSeparator(self, attrs):
        """
        Handler method for the "Separator" start tag.
        
        @param attrs list of tag attributes
        """
        self.wizards.registeredWizards.append(None)
