# -*- coding: utf-8 -*-

# Copyright (c) 2002, 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#
#
# It is inspired by the cvs log form code found in kdevelop 3.0
#

"""
Module implementing a dialog to show the output of the cvs log command process.
"""

from qt import *

from LogForm import LogForm
from DiffDialog import CvsDiffDialog

class CvsLogDialog(LogForm):
    """
    Module implementing a dialog to show the output of the cvs log command process.
    
    The dialog is nonmodal. Clicking a link in the upper text pane shows 
    a diff of the versions.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs reference to the vcs object
        @param parent parent widget (QWidget)
        """
        LogForm.__init__(self, parent)
        
        self.saveButton.hide()
        
        QWhatsThis.add(self.contents,self.tr("<b>CVS Log</b>\n"
            "<p>This shows the output of the cvs log command."
            " By clicking on the links you may show the difference"
            " between versions.</p>"
        ))
        QWhatsThis.add(self.errors,self.tr("<b>CVS log errors</b>\n"
            "<p>This shows possible error messages of the cvs log"
            " command.</p>"
        ))
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.process = QProcess(self)
        self.vcs = vcs
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.handleProcessExited)
        self.disconnect(self.contents, SIGNAL('linkClicked(const QString&)'),
            self.contents, SLOT('setSource(const QString&)'))
        self.connect(self.contents, SIGNAL('linkClicked(const QString&)'),
            self.handleLinkClicked)
            
        self.rx_sep = QRegExp('\\-+')
        self.rx_sep2 = QRegExp('=+')
        self.rx_date = QRegExp('date: .* author: .* state: .* lines: .*')
        self.rx_rev = QRegExp('revision ((\\d+\\.?)+)')
        # "revision" followed by one or more decimals followed by an optional dot
        self.rx_file = QRegExp('Working file: (.*)')
        # "Working file: " followed by the file name
            
    def closeEvent(self, e):
        """
        Private slot implementing a close event handler.
        
        @param e close event (QCloseEvent)
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        e.accept()
        
    def start(self, fn):
        """
        Public slot to start the cvs log command.
        
        @param fn filename to be diffed (string)
        """
        self.filename = fn
        self.dname, self.fname = self.vcs.splitPath(fn)
        
        self.process.kill()
        self.contents.clear()
        
        self.process.clearArguments()
        self.process.addArgument('cvs')
        self.vcs.addArguments(self.process, self.vcs.options['global'])
        self.process.addArgument('log')
        self.vcs.addArguments(self.process, self.vcs.options['log'])
        self.process.addArgument(self.fname)
        self.process.setWorkingDirectory(QDir(self.dname))
        
        self.process.start()
        self.setCaption(self.trUtf8('CVS Log %1').arg(self.filename))
        
    def handleProcessExited(self):
        """
        Private slot to handle the processExited signal.
        
        Doesn't do anything right now.
        """
        pass
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal.
        
        It reads the output of the process, formats it and inserts it into
        the contents pane.
        """
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            if self.rx_file.exactMatch(s):
                self.fname = self.rx_file.cap(1)
                self.contents.append(s)
            elif self.rx_rev.exactMatch(s):
                ver = self.rx_rev.cap(1)
                dstr = QString('<b>%s</b' % str(s))
                lastVer = int(str(ver.section('.', -1, -1))) - 1
                if lastVer > 0:
                    lv = ver.left(ver.findRev('.') + 1)
                    lv.append(str(lastVer))
                    url = QUrl()
                    url.setProtocol("file")
                    url.setPath(self.dname)
                    url.setFileName(self.fname)
                    query = QString()
                    query.append(lv).append('_').append(ver)
                    url.setQuery(query)
                    dstr.append(' [<a href="')\
                        .append(url.toString())\
                        .append('">')\
                        .append(self.trUtf8('diff to %1').arg(lv))\
                        .append('</a>]')
                    
                self.contents.append(dstr)
                
            elif self.rx_date.exactMatch(s):
                self.contents.append('<i>%s</i>' % str(s))
                
            elif self.rx_sep.exactMatch(s) or self.rx_sep2.exactMatch(s):
                self.contents.append('<hr>')
                
            else:
                self.contents.append(s)
                
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal. 
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        while self.process.canReadLineStderr():
            s = self.process.readLineStderr()
            self.errors.append(s)
            
    def handleLinkClicked(self, link):
        """
        Private slot to handle the linkClicked signal of the contents pane.
        
        @param link the link that was clicked (QString)
        """
        self.contents.setSource('')
        link = QUrl(link)
        filename = link.path()
        ver = link.query()
        v1 = ver.section('_', 0, 0)
        v2 = ver.section('_', 1, 1)
        if v1.isEmpty() or v2.isEmpty():
            return
            
        self.diff = CvsDiffDialog(self.vcs)
        self.diff.show()
        self.diff.start(filename, [v1, v2])
