# -*- coding: utf-8 -*-

# Copyright (c) 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the handler class for reading an XML project session file.
"""

import codecs
import sys
import os
from types import UnicodeType

from ThirdParty.XMLFilter import XMLSAXHandler

from qt import qApp

class SessionHandler(XMLSAXHandler):
    """
    Class implementing a sax handler to read an XML project session file.
    """
    def __init__(self, project):
        """
        Constructor
        
        @param project Reference to the project object to store the
                information into.
        """
        self.elements = {
            'Filename' : (self.startFilename, self.endFilename),
            'ActiveWindow' : (self.startFilename, self.endFilename),
            'Breakpoint' : (self.startBreakpoint, self.endBreakpoint),
            'BpFilename' : (self.defaultStartElement, self.endBFilename),
            'Linenumber' : (self.startLinenumber, self.defaultEndElement),
            'Condition' : (self.defaultStartElement, self.endCondition),
            'Enabled' : (self.startEnabled, self.defaultEndElement),
            'Count' : (self.startCount, self.defaultEndElement),
            'CommandLine' : (self.defaultStartElement, self.endCommandLine),
            'WorkingDirectory' : (self.defaultStartElement, self.endWorkingDirectory),
            'ReportExceptions' : (self.startReportExceptions, self.defaultEndElement),
            'Exceptions' : (self.startExceptions, self.endExceptions),
            'Exception' : (self.defaultStartElement, self.endException),
            'TracePython' : (self.startTracePython, self.defaultEndElement),
            'Bookmark' : (self.startBookmark, self.endBookmark),
            'BmFilename' : (self.defaultStartElement, self.endBFilename),
        }
    
        self.project = project
        self.vm = qApp.mainWidget().getViewManager()
        self.dbg = qApp.mainWidget().getDebugger()
        
        self.encoder = codecs.lookup(sys.getdefaultencoding())[0] # encode,decode,reader,writer
        
    def utf8_to_code(self, text):
        """
        Private method to convert a string to unicode and encode it for XML.
        
        @param text the text to encode (string)
        """
        if type(text) is not UnicodeType:
            text = unicode(text, "utf-8")
        return self.encoder(text)[0] # result,size
        
    def unescape(self, text):
        """
        Private method used to unescape certain characters.
        
        @param text the text to unescape (string)
        """
        return text.replace("&gt;",">").replace("&lt;","<").replace("&amp;","&")
        
    def startDocument(self):
        """
        Handler called, when the document parsing is started.
        """
        self.buffer = ""
        
        # clear all breakpoints and bookmarks first (in case we are rereading a session file)
        files = self.project.getSources(1)
        for file in files:
            editor = self.vm.getOpenEditor(file)
            if editor is not None:
                editor.handleClearBreakpoints()
                editor.handleClearBookmarks()
        
    def startElement(self, name, attrs):
        """
        Handler called, when a starting tag is found.
        
        @param name name of the tag (string)
        @param attrs list of tag attributes
        """
        try:
            self.elements[name][0](attrs)
        except KeyError:
            pass
            
    def endElement(self, name):
        """
        Handler called, when an ending tag is found.
        
        @param name name of the tag (string)
        """
        try:
            self.elements[name][1]()
        except KeyError:
            pass
            
    def characters(self, chars):
        """
        Handler called for ordinary text.
        
        @param chars the scanned text (string)
        """
        self.buffer += chars
        
    ###################################################
    ## below follow the individual handler functions
    ###################################################
    
    def defaultStartElement(self, attrs):
        """
        Handler method for common start tags.
        
        @param attrs list of tag attributes
        """
        self.buffer = ""
        
    def defaultEndElement(self):
        """
        Handler method for the common end tags.
        """
        pass
        
    def startFilename(self, attrs):
        """
        Handler method for the "Filename" start tag.
        """
        self.buffer = ""
        self.cline = int(attrs.get("cline", "0"))
        self.cindex = int(attrs.get("cindex", "0"))
        
    def endFilename(self):
        """
        Handler method for the "Filename" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.vm.handleOpen(self.buffer)
        ed = self.vm.getOpenEditor(self.buffer)
        if ed is not None:
            ed.setCursorPosition(self.cline, self.cindex)
            ed.ensureCursorVisible()
        
    def startBreakpoint(self, attrs):
        """
        Handler method for the "Breakpoint" start tag.
        
        @param attrs list of tag attributes
        """
        self.filename = ""
        self.lineno = 0
        self.bpCond = ""
        self.bpTemp = 0
        self.bpEnabled = 1
        self.bpCount = 0
        
    def endBreakpoint(self):
        """
        Handler method for the "Breakpoint" end tag.
        """
        self.dbg.newBreakpointWithProperties(self.filename, self.lineno,
            (self.bpCond, self.bpTemp, self.bpEnabled, self.bpCount))
            
    def endBFilename(self):
        """
        Handler method for the "BFilename" end tag.
        """
        self.filename = self.utf8_to_code(self.buffer)
        
    def startLinenumber(self, attrs):
        """
        Handler method for the "Linenumber" start tag.
        
        @param attrs list of tag attributes
        """
        self.lineno = int(attrs["value"])
        
    def endCondition(self):
        """
        Handler method for the "Condition" end tag.
        """
        cond = self.utf8_to_code(self.buffer)
        cond = self.unescape(cond)
        if cond == 'None' or cond == '':
            self.bpCond = None
        else:
            self.bpCond = cond
            
    def startEnabled(self, attrs):
        """
        Handler method for the "Enabled" start tag.
        
        @param attrs list of tag attributes
        """
        self.bpEnabled = int(attrs["value"])
        
    def startCount(self, attrs):
        """
        Handler method for the "Count" start tag.
        
        @param attrs list of tag attributes
        """
        self.bpCount = int(attrs["value"])
        
    def endCommandLine(self):
        """
        Handler method for the "CommandLine" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.buffer = self.unescape(self.buffer)
        self.dbg.setArgvHistory(self.buffer)
        self.project.dbgCmdline = self.buffer
        
    def endWorkingDirectory(self):
        """
        Handler method for the "WorkinDirectory" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.dbg.setWdHistory(self.buffer)
        self.project.dbgWd = self.buffer
        
    def startReportExceptions(self, attrs):
        """
        Handler method for the "ReportExceptions" start tag.
        
        @param attrs list of tag attributes
        """
        exc = int(attrs.get("value", "0"))
        self.dbg.setExceptionReporting(exc)
        self.project.dbgReportExceptions = exc
        
    def startExceptions(self, attrs):
        """
        Handler method for the "Exceptions" start tag.
        
        @param attrs list of tag attributes
        """
        self.dbgExcList = []
        
    def endExceptions(self):
        """
        Handler method for the "Exceptions" end tag.
        """
        self.dbg.setExcList(self.dbgExcList)
        
    def endException(self):
        """
        Handler method for the "Exception" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.dbgExcList.append(self.buffer)
        
    def startTracePython(self, attrs):
        """
        Handler method for the "TracePython" start tag.
        
        @param attrs list of tag attributes
        """
        tracePython = int(attrs.get("value", "0"))
        self.dbg.setTracePython(tracePython)
        self.project.dbgTracePython = tracePython
        
    def startBookmark(self, attrs):
        """
        Handler method for the "Bookmark" start tag.
        
        @param attrs list of tag attributes
        """
        self.filename = ""
        self.lineno = 0
        
    def endBookmark(self):
        """
        Handler method for the "Bookmark" end tag.
        """
        editor = self.vm.getOpenEditor(self.filename)
        if editor is not None:
            editor.handleBookmark(self.lineno)
