# -*- coding: utf-8 -*-

# Copyright (c) 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the handler class for reading a keyboard shortcuts file.
"""

import codecs
import sys
from types import UnicodeType

from ThirdParty.XMLFilter import XMLSAXHandler

class ShortcutsHandler(XMLSAXHandler):
    """
    Class implementing a sax handler to read a keyboard shortcuts file.
    """
    def __init__(self):
        """
        Constructor
        """
        self.elements = {
            'Shortcut' : (self.startShortcut, self.endShortcut),
            'Name' : (self.defaultStartElement, self.endName),
            'Accel' : (self.defaultStartElement, self.endAccel),
        }
    
        self.encoder = codecs.lookup(sys.getdefaultencoding())[0] # encode,decode,reader,writer
        
    def utf8_to_code(self, text):
        """
        Private method to convert a string to unicode and encode it for XML.
        
        @param text the text to encode (string)
        """
        if type(text) is not UnicodeType:
            text = unicode(text, "utf-8")
        return self.encoder(text)[0] # result,size
        
    def startDocument(self):
        """
        Handler called, when the document parsing is started.
        """
        self.buffer = ""
        self.shortcuts = {}     # dictionary for storing the shortcuts
        
    def startElement(self, name, attrs):
        """
        Handler called, when a starting tag is found.
        
        @param name name of the tag (string)
        @param attrs list of tag attributes
        """
        try:
            self.elements[name][0](attrs)
        except KeyError:
            pass
            
    def endElement(self, name):
        """
        Handler called, when an ending tag is found.
        
        @param name name of the tag (string)
        """
        try:
            self.elements[name][1]()
        except KeyError:
            pass
            
    def characters(self, chars):
        """
        Handler called for ordinary text.
        
        @param chars the scanned text (string)
        """
        self.buffer += chars
        
    ###################################################
    ## below follow the individual handler functions
    ###################################################
    
    def defaultStartElement(self, attrs):
        """
        Handler method for common start tags.
        
        @param attrs list of tag attributes
        """
        self.buffer = ""
        
    def endName(self):
        """
        Handler method for the "Name" end tag.
        """
        self.name = self.utf8_to_code(self.buffer)
        
    def endAccel(self):
        """
        Handler method for the "Accel" end tag.
        """
        self.accel = self.utf8_to_code(self.buffer)
        
    def startShortcut(self, attrs):
        """
        Handler method for the "Shortcut" start tag.
        
        @param attrs list of tag attributes
        """
        self.name = ''
        self.accel = ''
        self.category = attrs.get('category', '')
        
    def endShortcut(self):
        """
        Handler method for the "Shortcut" end tag.
        """
        if self.category:
            if not self.shortcuts.has_key(self.category):
                self.shortcuts[self.category] = {}
            self.shortcuts[self.category][self.name] = self.accel
        
    def getShortcuts(self):
        """
        Public method to retrieve the shortcuts.
        
        @return Dictionary of dictionaries of shortcuts. The keys of the
            dictionary are the categories, the values are dictionaries.
            These dictionaries have the shortcut name as their key and
            the accelerator as their value.
        """
        return self.shortcuts
