# -*- coding: utf-8 -*-

# Copyright (c) 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Package implementing interfaces to source code documentation tools.

It implements the high level interface and all the supporting dialogs
for the supported source code documentors.
"""

import sys
import os.path
import shutil

from qt import *

import Utilities
from EricdocConfigDialog import EricdocConfigDialog
from HappydocConfigDialog import HappydocConfigDialog
from DocGeneratorExecDialog import DocGeneratorExecDialog

class DocumentationTools(QObject):
    """ 
    Class implementing the high level interface.
    """
    def __init__(self, project, parent):
        """
        Constructor
        
        @param project project object of the current project
        @param parent parent object of this class (QObject)
        """
        QObject.__init__(self, parent)
        
        self.project = project
        
    def initActions(self):
        """
        Private method to initialize the API documentation actions.
        """
        self.ericdocAct = QAction(self.trUtf8('Use eric3-doc'),
            self.trUtf8('Use &eric3-doc'),0, self)
        self.ericdocAct.setStatusTip(\
            self.trUtf8('Generate API documentation using eric3-doc'))
        self.ericdocAct.setWhatsThis(self.trUtf8(
            """<b>Use eric3-doc</b>"""
            """<p>Generate API documentation using eric3-doc.</p>"""
                        ))
        self.connect(self.ericdocAct,SIGNAL('activated()'),self.handleEricdoc)
        
        self.happydocAct = QAction(self.trUtf8('Use Happydoc'),
            self.trUtf8('Use &Happydoc'),0, self)
        self.happydocAct.setStatusTip(\
            self.trUtf8('Generate API documentation using Happydoc'))
        self.happydocAct.setWhatsThis(self.trUtf8(
            """<b>Use Happydoc</b>"""
            """<p>Generate API documentation using Happydoc.</p>"""
                        ))
        self.connect(self.happydocAct,SIGNAL('activated()'),self.handleHappydoc)
        
    def initMenu(self):
        """
        Private method called to build the project API documentation submenu.
        
        @return the menu or None
        """
        self.initActions()
        
        menu = QPopupMenu(self.parent())
        
        self.ericdocAct.addTo(menu)
        
        # entry for happydoc is only included if it is accessible
        if (sys.platform == "win32" and \
            os.path.exists(Utilities.normjoinpath(\
                sys.prefix, 'Scripts', 'happydocwin.py'))) or \
            Utilities.isinpath('happydoc'):
            self.happydocAct.addTo(menu)
            nothingfound = 0
        
        return menu
        
    def handleEricdoc(self):
        """
        Private slot to perform the eric3-doc api documentation generation.
        """
        if len(self.project.pdata['ERIC3DOCPARMS']):
            parms = eval(self.project.pdata['ERIC3DOCPARMS'][0])
        else:
            parms = None
        dlg = EricdocConfigDialog(self.project.ppath, parms)
        if dlg.exec_loop() == QDialog.Accepted:
            args, parms = dlg.generateParameters()
            
            # test for changes of selected parameters and save the in the project
            # 1. there were none, now there are
            if len(self.project.pdata['ERIC3DOCPARMS']) == 0 and len(parms) > 0:
                self.project.pdata['ERIC3DOCPARMS'] = [str(parms)]
                self.project.setDirty(1)
            # 2. there were some, now there aren't
            elif len(self.project.pdata['ERIC3DOCPARMS']) > 0 and len(parms) == 0:
                self.project.pdata['ERIC3DOCPARMS'] = []
                self.project.setDirty(1)
            # 3. there were some and still are
            else:
                sparms = str(parms)
                if sparms != self.project.pdata['ERIC3DOCPARMS'][0]:
                    self.project.pdata['ERIC3DOCPARMS'] = [sparms]
                    self.project.setDirty(1)
                    
            # now do the call
            dia = DocGeneratorExecDialog("Ericdoc")
            res = dia.start(args, self.project.ppath)
            if res:
                dia.exec_loop()
                
            outdir = parms['outputDirectory']
            if outdir == '':
                outdir = 'doc'      # that is eric3-docs default output dir
                
            # add it to the project data, if it isn't in already
            outdir = outdir.replace(self.project.ppath+os.sep, '')
            if outdir not in self.project.pdata['OTHERS']:
                self.project.pdata['OTHERS'].append(outdir)
                self.project.setDirty(1)
                self.project.othersAdded(outdir)
        
    def handleHappydoc(self):
        """
        Private slot to perform the happydoc api documentation generation.
        """
        if len(self.project.pdata['HAPPYDOCPARMS']):
            parms = eval(self.project.pdata['HAPPYDOCPARMS'][0])
        else:
            parms = None
        dlg = HappydocConfigDialog(self.project.ppath, parms)
        if dlg.exec_loop() == QDialog.Accepted:
            args, parms = dlg.generateParameters()
            
            # test for changes of selected parameters and save the in the project
            # 1. there were none, now there are
            if len(self.project.pdata['HAPPYDOCPARMS']) == 0 and len(parms) > 0:
                self.project.pdata['HAPPYDOCPARMS'] = [str(parms)]
                self.project.setDirty(1)
            # 2. there were some, now there aren't
            elif len(self.project.pdata['HAPPYDOCPARMS']) > 0 and len(parms) == 0:
                self.project.pdata['HAPPYDOCPARMS'] = []
                self.project.setDirty(1)
            # 3. there were some and still are
            else:
                sparms = str(parms)
                if sparms != self.project.pdata['HAPPYDOCPARMS'][0]:
                    self.project.pdata['HAPPYDOCPARMS'] = [sparms]
                    self.project.setDirty(1)
                    
            # now do the call
            dia = DocGeneratorExecDialog("Happydoc")
            res = dia.start(args, self.project.ppath)
            if res:
                dia.exec_loop()
                
            outdir = parms['outputDirectory']
            if outdir == '':
                outdir = 'doc'      # that is happydocs default output dir
                
            # add it to the project data, if it isn't in already
            outdir = outdir.replace(self.project.ppath+os.sep, '')
            if outdir not in self.project.pdata['OTHERS']:
                self.project.pdata['OTHERS'].append(outdir)
                self.project.setDirty(1)
                self.project.othersAdded(outdir)
