# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a simple Python syntax checker.
"""

import sys
import os
import types

from qt import *

from SyntaxCheckerForm import SyntaxCheckerForm
import Utilities

class SyntaxCheckerDialog(SyntaxCheckerForm):
    """
    Class implementing a dialog to display the results of a syntax check run.
    """
    def __init__(self, viewmanager, parent = None):
        """
        Constructor
        
        @param viewmanager Reference to the viewmanager object.
        @param parent The parent widget. (QWidget)
        """
        SyntaxCheckerForm.__init__(self, parent)
        
        self.vm = viewmanager
        self.noResults = 1
        self.cancelled = 0
        
    def start(self, fn):
        """
        Public slot to start the syntax check.
        
        @param fn File or list of files or directory to be checked
                (string or list of strings)
        """
        if type(fn) is types.ListType:
            files = fn
        elif os.path.isdir(fn):
            files = Utilities.direntries(fn, 1, '*.py', 0)
        else:
            files = [fn]
        
        self.checkProgress.setTotalSteps(len(files))
        qApp.processEvents()
        
        # now go through all the files
        progress = 0
        for file in files:
            if self.cancelled:
                return
            
            nok, fname, line, code, error = Utilities.compile(file)
            if nok:
                self.noResults = 0
                itm = QListViewItem(self.resultList,
                    fname, line, error, code)
            progress += 1
            self.checkProgress.setProgress(progress)
            qApp.processEvents()
                    
        self.finish()
        
    def finish(self):
        """
        Private slot called when the syntax check finished or the user pressed the button.
        """
        self.cancelled = 1
        self.cancelButton.setText(self.trUtf8('OK'))
        self.cancelButton.setDefault(1)
        
        if self.noResults:
            itm = QListViewItem(self.resultList,
                self.trUtf8('No syntax errors found.'))
            self.showButton.setEnabled(0)
            self.clearErrors()
        else:
            self.showButton.setEnabled(1)
        
    def buttonPressed(self):
        """
        Private slot connected to the button clicked signal.
        """
        if self.cancelled:
            self.close()
        else:
            self.finish()
            
    def openFile(self, itm):
        """
        Private slot to handle the doubleClicked signal of the result list.
        
        @param itm Reference to the item, that was double clicked.
        """
        if self.noResults:
            return
            
        fn = Utilities.normabspath(unicode(itm.text(0)))
        lineno = int(str(itm.text(1)))
        error = unicode(itm.text(2))
        
        self.vm.handlePythonFile(fn, lineno)
        editor = self.vm.getOpenEditor(fn)
        editor.handleSyntaxError(lineno, 1, error)
        
    def handleShow(self):
        """
        Private slot to handle the "Show" button press.
        """
        itm = self.resultList.firstChild()
        while itm:
            self.openFile(itm)
            itm = itm.itemBelow()
            
        # go through the list again to clear syntax error markers
        # for files, that are ok
        openFiles = self.vm.getOpenFilenames()
        errorFiles = []
        itm = self.resultList.firstChild()
        while itm:
            errorFiles.append(Utilities.normabspath(unicode(itm.text(0))))
            itm = itm.itemBelow()
        for file in openFiles:
            if not file in errorFiles:
                editor = self.vm.getOpenEditor(file)
                editor.handleClearSyntaxError()
        
    def clearErrors(self):
        """
        Private method to clear all error markers of open editors.
        """
        openFiles = self.vm.getOpenFilenames()
        for file in openFiles:
            editor = self.vm.getOpenEditor(file)
            editor.handleClearSyntaxError()
