# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to enter the parameters for eric3-api.
"""

import sys
import os
import copy

from qt import *

from KdeQt import KQFileDialog

from EricapiConfigForm import EricapiConfigForm
import Utilities

from eric3config import getConfig

class EricapiConfigDialog(EricapiConfigForm):
    """
    Class implementing a dialog to enter the parameters for eric3-api.
    """
    def __init__(self,ppath,parms = None,parent = None):
        """
        Constructor
        
        @param ppath project path of the current project (string)
        @param parms parameters to set in the dialog
        @param parent parent widget of this dialog
        """
        EricapiConfigForm.__init__(self,parent)
        
        self.initializeDefaults()
        
        # get a copy of the defaults to store the user settings
        self.parameters = copy.deepcopy(self.defaults)
        
        # combine it with the values of parms
        if parms is not None:
            for key, value in parms.items():
                self.parameters[key] = parms[key]
        
        self.ppath = ppath
        
        self.recursionCheckBox.setChecked(self.parameters['useRecursion'])
        self.outputFileEdit.setText(self.parameters['outputFile'])
        self.ignoreDirsListBox.clear()
        for d in self.parameters['ignoreDirectories']:
            self.ignoreDirsListBox.insertItem(d)

    def initializeDefaults(self):
        """
        Private method to set the default values. 
        
        These are needed later on to generate the commandline
        parameters.
        """
        self.defaults = {
            'useRecursion' : 0,
            'outputFile' : '',
            'ignoreDirectories' : []
        }
        
    def generateParameters(self):
        """
        Public method that generates the commandline parameters.
        
        It generates a QStringList to be used
        to set the QProcess arguments for the happydoc call and
        a list containing the non default parameters. The second
        list can be passed back upon object generation to overwrite
        the default settings.
        
        @return a tuple of the commandline parameters and non default parameters
            (QStringList, dictionary)
        """
        parms = {}
        args = QStringList()
        
        # 1. the program name
        args.append(sys.executable)
        args.append(Utilities.normabsjoinpath(getConfig('ericDir'), "eric3-api.py"))
        
        # 2. the commandline options
        if self.parameters['outputFile'] != self.defaults['outputFile']:
            parms['outputFile'] = self.parameters['outputFile']
            args.append('-o')
            if os.path.isabs(self.parameters['outputFile']):
                args.append(self.parameters['outputFile'])
            else:
                args.append(os.path.join(self.ppath, self.parameters['outputFile']))
        if self.parameters['ignoreDirectories'] != self.defaults['ignoreDirectories']:
            parms['ignoreDirectories'] = self.parameters['ignoreDirectories'][:]
            for d in self.parameters['ignoreDirectories']:
                args.append('-x')
                args.append(d)
        if self.parameters['useRecursion'] != self.defaults['useRecursion']:
            parms['useRecursion'] = self.parameters['useRecursion']
            args.append('-r')
            
        return (args, parms)

    def handleOutputFile(self):
        """
        Private slot to select the output file.
        
        It displays a file selection dialog to
        select the file the api is written to.
        """
        filename = KQFileDialog.getSaveFileName(self.outputFileEdit.text(),
            None, self, None, self.trUtf8("Select output file"))
            
        if not filename.isNull():
            # make it relative, if it is in a subdirectory of the project path 
            fn = unicode(QDir.convertSeparators(filename))
            fn = fn.replace(self.ppath+os.sep, '')
            self.outputFileEdit.setText(fn)

    def handleFileChanged(self, filename):
        """
        Private slot to enable/disable the "OK" button.
        
        @param filename name of the file (QString)
        """
        self.okButton.setEnabled(not filename.isEmpty())

    def handleIgnoreDir(self):
        """
        Private slot to select a directory to be ignored.
        
        It displays a directory selection dialog to
        select a directory to be ignored.
        """
        directory = KQFileDialog.getExistingDirectory(self.ignoreDirEdit.text(),
            self, None, self.trUtf8("Select directory to exclude"), 1, 0)
            
        if not directory.isNull():
            # make it relative, if it is a subdirectory of the project path 
            dn = unicode(QDir.convertSeparators(directory))
            dn = dn.replace(self.ppath+os.sep, '')
            while dn.endswith(os.sep):
                dn = dn[:-1]
            self.ignoreDirEdit.setText(dn)
            
    def handleIgnoreDirAdd(self):
        """
        Private slot to add the directory displayed to the listview.
        
        The directory in the ignore directories
        line edit is moved to the listbox above and the edit is cleared.
        """
        self.ignoreDirsListBox.insertItem(QDir.convertSeparators(self.ignoreDirEdit.text()))
        self.ignoreDirEdit.clear()

    def handleIgnoreDirDelete(self):
        """
        Private slot to delete the currently selected directory of the listbox.
        """
        self.ignoreDirsListBox.removeItem(self.ignoreDirsListBox.currentItem())

    def accept(self):
        """
        Protected slot called by the Ok button. 
        
        It saves the values in the parameters dictionary.
        """
        self.parameters['useRecursion'] = self.recursionCheckBox.isChecked()
        outfile = unicode(self.outputFileEdit.text())
        if outfile != '':
            outfile = Utilities.normcasepath(outfile)
        self.parameters['outputFile'] = outfile
        itm = self.ignoreDirsListBox.firstItem()
        self.parameters['ignoreDirectories'] = []
        while itm is not None:
            self.parameters['ignoreDirectories'].append(\
                Utilities.normcasepath(unicode(itm.text())))
            itm = itm.next()
        
        # call the accept slot of the base class
        EricapiConfigForm.accept(self)
