/* Ergo, version 3.2, a program for linear scaling electronic structure
 * calculations.
 * Copyright (C) 2012 Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Primary academic reference:
 * Kohn−Sham Density Functional Theory Electronic Structure Calculations 
 * with Linearly Scaling Computational Time and Memory Usage,
 * Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek,
 * J. Chem. Theory Comput. 7, 340 (2011),
 * <http://dx.doi.org/10.1021/ct100611z>
 * 
 * For further information about Ergo, see <http://www.ergoscf.org>.
 */

#include <stdlib.h>
#include <math.h>
#include <stdio.h>
#include <errno.h>
#include <memory.h>
#include <time.h>
#include <stdarg.h>
#include "integrals_1el_single.h"
#include "pi.h"
#include "boysfunction.h"

#include "integrals_hermite.h"


static ergo_real 
do_1e_repulsion_integral_using_symb_info_h(DistributionSpecStruct* psi,
					   ergo_real pointCharge,
					   const ergo_real* pointChargeCoords,
					   const IntegralInfo* b)
{
  // Let the distr be 1 and the pointcharge 2
  ergo_real alpha1 = psi->exponent;
  // alpha2 is ~ infinity
  ergo_real alpha0 = alpha1;
  // alpham is ~ infinity
  // g is 1 (not needed)
  
  int n1 = 0;
  int n2 = 0;
  for(int i = 0; i < 3; i++)
    n1 += psi->monomialInts[i];
  int n1x = psi->monomialInts[0];
  int n1y = psi->monomialInts[1];
  int n1z = psi->monomialInts[2];

  int noOfMonomials_1 = b->monomial_info.no_of_monomials_list[n1];
  int noOfMonomials_2 = b->monomial_info.no_of_monomials_list[n2];

  ergo_real dx0 = pointChargeCoords[0] - psi->centerCoords[0];
  ergo_real dx1 = pointChargeCoords[1] - psi->centerCoords[1];
  ergo_real dx2 = pointChargeCoords[2] - psi->centerCoords[2];

  ergo_real resultPreFactor = 2 * pi / alpha1;

  ergo_real primitiveIntegralList_h[noOfMonomials_1*noOfMonomials_2];
  ergo_real primitiveIntegralList_2[noOfMonomials_1*noOfMonomials_2];

  const JK::ExchWeights CAM_params_not_used;
  
  get_related_integrals_hermite(&b->monomial_info,
				CAM_params_not_used,
				n1, noOfMonomials_1,
				n2, noOfMonomials_2,
				dx0, 
				dx1, 
				dx2, 
				alpha0,
				resultPreFactor,
				primitiveIntegralList_h);
  
  b->hermite_conversion_info.multiply_by_hermite_conversion_matrix_from_right(n1,
									      n2,
									      1.0/alpha1,
									      primitiveIntegralList_h,
									      primitiveIntegralList_2);
  int monomialIndex = b->monomial_info.monomial_index_list[n1x][n1y][n1z];

  ergo_real result = psi->coeff * pointCharge * primitiveIntegralList_2[monomialIndex];

  return result;
}





ergo_real 
do_1e_repulsion_integral_using_symb_info(DistributionSpecStruct* psi,
					 ergo_real pointCharge,
					 const ergo_real* pointChargeCoords,
					 const IntegralInfo* b)
{
  return do_1e_repulsion_integral_using_symb_info_h(psi,
						    pointCharge,
						    pointChargeCoords,
						    b);
}



