/***************************************************************************
 *   Copyright (C) 2006 by Niklas Knutsson                                 *
 *   nq@altern.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "importcsvdialog.h"
#include "budget.h"
#include "kdateedit.h"

#include <klocale.h>
#include <kglobal.h>
#include <qlayout.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qlabel.h>
#include <qspinbox.h>
#include <knuminput.h>
#include <klineedit.h>
#include <kcombobox.h>
#include <kurlrequester.h>
#include <kmessagebox.h>
#include <qdatetime.h>
#include <qcheckbox.h>
#include <kio/netaccess.h>
#include <qmap.h>

#include <math.h>

ImportCSVDialog::ImportCSVDialog(Budget *budg, QWidget *parent) : KWizard(parent, 0, true), budget(budg) {

	setCaption(i18n("Import CSV file"));

	QWidget *page1 = new QWidget(this);
	addPage(page1, i18n("Transaction Type Selection"));
	QVBoxLayout *layout1 = new QVBoxLayout(page1, 6, 6);
	typeGroup = new QButtonGroup(5, Qt::Vertical, page1);
	new QRadioButton(i18n("Expenses"), typeGroup);
	new QRadioButton(i18n("Incomes"), typeGroup);
	new QRadioButton(i18n("Transfers"), typeGroup);
	new QRadioButton(i18n("Expenses and incomes"), typeGroup);
	new QRadioButton(i18n("All types"), typeGroup);
	layout1->addWidget(typeGroup);
	typeDescriptionLabel = new QLabel(page1);
	typeDescriptionLabel->setAlignment(Qt::WordBreak);
	layout1->addWidget(typeDescriptionLabel);
	typeGroup->setButton(0);

	QWidget *page2 = new QWidget(this);
	addPage(page2, i18n("File Selection"));
	QGridLayout *layout2 = new QGridLayout(page2, 5, 2, 6, 6);

	layout2->addWidget(new QLabel(i18n("File:"), page2), 0, 0);
	fileEdit = new KURLRequester(page2);
	fileEdit->setMode(KFile::File | KFile::ExistingOnly);
	fileEdit->setFilter("text/x-csv");
	layout2->addWidget(fileEdit, 0, 1);
	layout2->addWidget(new QLabel(i18n("First data row:"), page2), 1, 0);
	rowEdit = new QSpinBox(1, 1000, 1, page2);
	rowEdit->setValue(2);
	layout2->addWidget(rowEdit, 1, 1);
	layout2->addWidget(new QLabel(i18n("Column delimiter:"), page2), 2, 0);
	delimiterCombo = new KComboBox(page2);
	delimiterCombo->setEditable(false);
	delimiterCombo->insertItem(i18n("Comma"));
	delimiterCombo->insertItem(i18n("Tabulator"));
	delimiterCombo->insertItem(i18n("Semicolon"));
	delimiterCombo->insertItem(i18n("Space"));
	delimiterCombo->insertItem(i18n("Other"));
	layout2->addWidget(delimiterCombo, 2, 1);
	delimiterEdit = new KLineEdit(page2);
	delimiterEdit->setEnabled(false);
	layout2->addWidget(delimiterEdit, 3, 1);
	layout2->addMultiCell(new QSpacerItem(1, 1, QSizePolicy::Minimum, QSizePolicy::Expanding), 4, 4, 0, 1);

	QWidget *page3 = new QWidget(this);
	addPage(page3, i18n("Columns Specification"));
	QGridLayout *layout3 = new QGridLayout(page3, 8, 5, 6, 6);

	layout3->addWidget(new QLabel(i18n("Description:"), page3), 0, 0);
	descriptionGroup = new QButtonGroup();
	columnDescriptionButton = new QRadioButton(i18n("Column"), page3);
	descriptionGroup->insert(columnDescriptionButton);
	layout3->addWidget(columnDescriptionButton, 0, 1);
	columnDescriptionEdit = new QSpinBox(1, 100, 1, page3);
	columnDescriptionEdit->setValue(2);
	columnDescriptionButton->setChecked(true);
	layout3->addWidget(columnDescriptionEdit, 0, 2);
	valueDescriptionButton = new QRadioButton(i18n("Value"), page3);
	descriptionGroup->insert(valueDescriptionButton);
	layout3->addWidget(valueDescriptionButton, 0, 3);
	valueDescriptionEdit = new KLineEdit(page3);
	valueDescriptionEdit->setEnabled(false);
	layout3->addWidget(valueDescriptionEdit, 0, 4);

	valueLabel = new QLabel(i18n("Cost:"), page3);
	layout3->addWidget(valueLabel, 1, 0);
	valueGroup = new QButtonGroup();
	columnValueButton = new QRadioButton(i18n("Column"), page3);
	valueGroup->insert(columnValueButton);
	layout3->addWidget(columnValueButton, 1, 1);
	columnValueEdit = new QSpinBox(1, 100, 1, page3);
	columnValueEdit->setValue(3);
	columnValueButton->setChecked(true);
	layout3->addWidget(columnValueEdit, 1, 2);
	valueValueButton = new QRadioButton(i18n("Value"), page3);
	valueGroup->insert(valueValueButton);
	layout3->addWidget(valueValueButton, 1, 3);
	valueValueEdit = new KDoubleSpinBox(0.0, INT_MAX / pow(10, KGlobal::locale()->fracDigits()) - 1.0, 1.0, 0.0, KGlobal::locale()->fracDigits(), page3);
	if(KGlobal::locale()->positivePrefixCurrencySymbol() && KGlobal::locale()->negativePrefixCurrencySymbol()) {
		valueValueEdit->setPrefix(KGlobal::locale()->currencySymbol() + " ");
	} else {
		valueValueEdit->setSuffix(QString(" ") + KGlobal::locale()->currencySymbol());
	}
	valueValueEdit->setEnabled(false);
	layout3->addWidget(valueValueEdit, 1, 4);

	layout3->addWidget(new QLabel(i18n("Date:"), page3), 2, 0);
	dateGroup = new QButtonGroup();
	columnDateButton = new QRadioButton(i18n("Column"), page3);
	dateGroup->insert(columnDateButton);
	layout3->addWidget(columnDateButton, 2, 1);
	columnDateEdit = new QSpinBox(1, 100, 1, page3);
	columnDateEdit->setValue(1);
	columnDateButton->setChecked(true);
	layout3->addWidget(columnDateEdit, 2, 2);
	valueDateButton = new QRadioButton(i18n("Value"), page3);
	dateGroup->insert(valueDateButton);
	layout3->addWidget(valueDateButton, 2, 3);
	valueDateEdit = new KDateEdit(page3);
	valueDateEdit->setEnabled(false);
	layout3->addWidget(valueDateEdit, 2, 4);

	AC1Label = new QLabel(i18n("Category:"), page3);
	layout3->addWidget(AC1Label, 3, 0);
	AC1Group = new QButtonGroup();
	columnAC1Button = new QRadioButton(i18n("Column"), page3);
	AC1Group->insert(columnAC1Button);
	layout3->addWidget(columnAC1Button, 3, 1);
	columnAC1Edit = new QSpinBox(1, 100, 1, page3);
	columnAC1Edit->setValue(4);
	columnAC1Edit->setEnabled(false);
	layout3->addWidget(columnAC1Edit, 3, 2);
	valueAC1Button = new QRadioButton(i18n("Value"), page3);
	AC1Group->insert(valueAC1Button);
	valueAC1Button->setChecked(true);
	layout3->addWidget(valueAC1Button, 3, 3);
	valueAC1Edit = new KComboBox(page3);
	valueAC1Edit->setEditable(false);
	layout3->addWidget(valueAC1Edit, 3, 4);

	AC2Label = new QLabel(i18n("From account:"), page3);
	layout3->addWidget(AC2Label, 4, 0);
	AC2Group = new QButtonGroup();
	columnAC2Button = new QRadioButton(i18n("Column"), page3);
	AC2Group->insert(columnAC2Button);
	layout3->addWidget(columnAC2Button, 4, 1);
	columnAC2Edit = new QSpinBox(1, 100, 1, page3);
	columnAC2Edit->setValue(5);
	columnAC2Edit->setEnabled(false);
	layout3->addWidget(columnAC2Edit, 4, 2);
	valueAC2Button = new QRadioButton(i18n("Value"), page3);
	AC2Group->insert(valueAC2Button);
	valueAC2Button->setChecked(true);
	layout3->addWidget(valueAC2Button, 4, 3);
	valueAC2Edit = new KComboBox(page3);
	valueAC2Edit->setEditable(false);
	layout3->addWidget(valueAC2Edit, 4, 4);

	layout3->addWidget(new QLabel(i18n("Comments:"), page3), 5, 0);
	commentsGroup = new QButtonGroup();
	columnCommentsButton = new QRadioButton(i18n("Column"), page3);
	commentsGroup->insert(columnCommentsButton);
	layout3->addWidget(columnCommentsButton, 5, 1);
	columnCommentsEdit = new QSpinBox(1, 100, 1, page3);
	columnCommentsEdit->setValue(6);
	columnCommentsEdit->setEnabled(false);
	layout3->addWidget(columnCommentsEdit, 5, 2);
	valueCommentsButton = new QRadioButton(i18n("Value"), page3);
	commentsGroup->insert(valueCommentsButton);
	valueCommentsButton->setChecked(true);
	layout3->addWidget(valueCommentsButton, 5, 3);
	valueCommentsEdit = new KLineEdit(page3);
	layout3->addWidget(valueCommentsEdit, 5, 4);

	QHBoxLayout *layout3_cm = new QHBoxLayout();
	layout3_cm->addStretch(1);
	createMissingButton = new QCheckBox(i18n("Create missing categories and accounts"), page3);
	createMissingButton->setChecked(true);
	layout3_cm->addWidget(createMissingButton);
	layout3->addMultiCellLayout(layout3_cm, 6, 6, 0, 4);

	layout3->addMultiCell(new QSpacerItem(1, 1, QSizePolicy::Minimum, QSizePolicy::Expanding), 7, 7, 0, 4);

	setHelpEnabled(page1, false);
	setHelpEnabled(page2, false);
	setHelpEnabled(page3, false);
	setFinishEnabled(page3, true);
	
	typeChanged(0);

	typeGroup->setFocus();

	connect(delimiterCombo, SIGNAL(activated(int)), this, SLOT(delimiterChanged(int)));
	connect(columnDescriptionButton, SIGNAL(toggled(bool)), columnDescriptionEdit, SLOT(setEnabled(bool)));
	connect(valueDescriptionButton, SIGNAL(toggled(bool)), valueDescriptionEdit, SLOT(setEnabled(bool)));
	connect(columnValueButton, SIGNAL(toggled(bool)), columnValueEdit, SLOT(setEnabled(bool)));
	connect(valueValueButton, SIGNAL(toggled(bool)), valueValueEdit, SLOT(setEnabled(bool)));
	connect(columnDateButton, SIGNAL(toggled(bool)), columnDateEdit, SLOT(setEnabled(bool)));
	connect(valueDateButton, SIGNAL(toggled(bool)), valueDateEdit, SLOT(setEnabled(bool)));
	connect(columnAC1Button, SIGNAL(toggled(bool)), columnAC1Edit, SLOT(setEnabled(bool)));
	connect(valueAC1Button, SIGNAL(toggled(bool)), valueAC1Edit, SLOT(setEnabled(bool)));
	connect(columnAC2Button, SIGNAL(toggled(bool)), columnAC2Edit, SLOT(setEnabled(bool)));
	connect(valueAC2Button, SIGNAL(toggled(bool)), valueAC2Edit, SLOT(setEnabled(bool)));
	connect(columnCommentsButton, SIGNAL(toggled(bool)), columnCommentsEdit, SLOT(setEnabled(bool)));
	connect(valueCommentsButton, SIGNAL(toggled(bool)), valueCommentsEdit, SLOT(setEnabled(bool)));
	connect(typeGroup, SIGNAL(clicked(int)), this, SLOT(typeChanged(int)));
	
}

ImportCSVDialog::~ImportCSVDialog() {
	delete descriptionGroup;
	delete dateGroup;
	delete valueGroup;
	delete AC1Group;
	delete AC2Group;
	delete commentsGroup;
}

void ImportCSVDialog::delimiterChanged(int index) {
	delimiterEdit->setEnabled(index == 4);
}
void ImportCSVDialog::typeChanged(int id) {
	valueAC1Button->setChecked(true);
	columnAC1Edit->setEnabled(false);
	valueAC1Edit->setEnabled(true);
	valueAC1Button->setEnabled(true);
	valueAC2Button->setChecked(true);
	columnAC2Edit->setEnabled(false);
	valueAC2Edit->setEnabled(true);
	valueAC2Button->setEnabled(true);
	createMissingButton->setEnabled(id != 4);
	createMissingButton->setChecked(id != 4);
	valueAC1Edit->clear();
	valueAC2Edit->clear();
	if(id < 3) {
		AssetsAccount *aa = budget->assetsAccounts.first();
		while(aa) {if(aa != budget->balancingAccount && aa->accountType() != ASSETS_TYPE_SECURITIES) valueAC2Edit->insertItem(aa->name()); aa = budget->assetsAccounts.next();}
	}
	switch(id) {
		case 0: {
			typeDescriptionLabel->setText(i18n("Imports data as expenses. Costs has positive value. Value is the only required column."));
			valueLabel->setText(i18n("Cost:"));
			AC1Label->setText(i18n("Category:"));
			AC2Label->setText(i18n("From account:"));
			ExpensesAccount *ea = budget->expensesAccounts.first();
			while(ea) {valueAC1Edit->insertItem(ea->name()); ea = budget->expensesAccounts.next();}
			break;
		}
		case 1: {
			typeDescriptionLabel->setText(i18n("Imports data as incomes. Value is the only required column."));
			valueLabel->setText(i18n("Income:"));
			AC1Label->setText(i18n("Category:"));
			AC2Label->setText(i18n("To account:"));
			IncomesAccount *ia = budget->incomesAccounts.first();
			while(ia) {valueAC1Edit->insertItem(ia->name()); ia = budget->incomesAccounts.next();}
			break;
		}
		case 2: {
			typeDescriptionLabel->setText(i18n("Imports data as transfers. Value is the only required column."));
			valueLabel->setText(i18n("Amount:"));
			AC1Label->setText(i18n("From account:"));
			AC2Label->setText(i18n("To account:"));
			AssetsAccount *aa = budget->assetsAccounts.first();
			while(aa) {if(aa != budget->balancingAccount && aa->accountType() != ASSETS_TYPE_SECURITIES) valueAC1Edit->insertItem(aa->name()); aa = budget->assetsAccounts.next();}
			break;
		}
		case 3: {
			typeDescriptionLabel->setText(i18n("Imports data as expenses and incomes. Costs has negative value. Value and category are both required columns."));
			columnAC1Button->setChecked(true);
			columnAC1Edit->setEnabled(true);
			valueAC1Edit->setEnabled(false);
			valueAC1Button->setEnabled(false);
			valueLabel->setText(i18n("Value:"));
			AC1Label->setText(i18n("Category:"));
			AC2Label->setText(i18n("Account:"));
			break;
		}
		case 4: {
			typeDescriptionLabel->setText(i18n("Imports data as expenses, incomes, and transfers. Costs has negative or positive value. Value, to, and from are all required columns. Accounts and categories must be existing."));
			columnAC1Button->setChecked(true);
			columnAC1Edit->setEnabled(true);
			valueAC1Edit->setEnabled(false);
			valueAC1Button->setEnabled(false);
			columnAC2Button->setChecked(true);
			columnAC2Edit->setEnabled(true);
			valueAC2Edit->setEnabled(false);
			valueAC2Button->setEnabled(false);
			valueLabel->setText(i18n("Value:"));
			AC1Label->setText(i18n("From:"));
			AC2Label->setText(i18n("To:"));
			break;
		}
	}
	columnValueButton->setChecked(true);
	valueValueButton->setEnabled(false);
	if(valueAC1Edit->count() == 0) {
		columnAC1Button->setChecked(true);
		columnAC1Edit->setEnabled(true);
		valueAC1Edit->setEnabled(false);
		valueAC1Button->setEnabled(false);
	}
	if(valueAC2Edit->count() == 0) {
		columnAC2Button->setChecked(true);
		columnAC2Edit->setEnabled(true);
		valueAC2Edit->setEnabled(false);
		valueAC2Button->setEnabled(false);
	}
}
void ImportCSVDialog::next() {
	if(currentPage() == page(0)) {
		fileEdit->setFocus();
	} else if(currentPage() == page(1)) {
		const KURL &url = fileEdit->url();
		if(url.isEmpty()) {
			KMessageBox::error(this, i18n("A file must be selected."));
			fileEdit->setFocus();
			return;
		} else if(!url.isValid()) {
			QFileInfo info(fileEdit->lineEdit()->text());
			if(info.isDir()) {
				KMessageBox::error(this, i18n("Selected file is a directory."));
				fileEdit->setFocus();
				return;
			} else if(!info.exists()) {
				KMessageBox::error(this, i18n("Selected file does not exist."));
				fileEdit->setFocus();
				return;
			}
			fileEdit->setURL(info.absFilePath());
		} else if(url.isLocalFile()) {
			QFileInfo info(url.path());
			if(info.isDir()) {
				KMessageBox::error(this, i18n("Selected file is a directory."));
				fileEdit->setFocus();
				return;
			} else if(!info.exists()) {
				KMessageBox::error(this, i18n("Selected file does not exist."));
				fileEdit->setFocus();
				return;
			}
		}
		if(delimiterCombo->currentItem() == 4 && delimiterEdit->text().isEmpty()) {
			KMessageBox::error(this, i18n("Empty delimiter."));
			delimiterEdit->setFocus();
			return;
		}
		columnDescriptionEdit->setFocus();
	}
	KWizard::next();
}

QDate readCSVDate(const QString &str) {
	bool b = true;
	QDate date = KGlobal::locale()->readDate(str, &b);
	if(b && date.isValid()) return date;
	date = QDate::fromString(str, Qt::ISODate);
	if(date.isValid()) return date;
	date = QDate::fromString(str, Qt::TextDate);
	return date;
}

double readCSVValue(const QString &str, bool &ok) {
	return KGlobal::locale()->readMoney(str, &ok);
}

bool ImportCSVDialog::import() {

	int first_row = rowEdit->value();
	int type = typeGroup->selectedId();
	QString delimiter;
	switch(delimiterCombo->currentItem()) {
		case 0: {delimiter = ","; break;}
		case 1: {delimiter = "\t"; break;}
		case 2: {delimiter = ";"; break;}
		case 3: {delimiter = " "; break;}
		case 4: {delimiter = delimiterEdit->text(); break;}
	}
	int description_c = columnDescriptionButton->isChecked() ? columnDescriptionEdit->value() : -1;
	int value_c = columnValueButton->isChecked() ? columnValueEdit->value() : -1;
	int date_c = columnDateButton->isChecked() ? columnDateEdit->value() : -1;
	int AC1_c = columnAC1Button->isChecked() ? columnAC1Edit->value() : -1;
	int AC2_c = columnAC2Button->isChecked() ? columnAC2Edit->value() : -1;
	int comments_c = columnCommentsButton->isChecked() ? columnCommentsEdit->value() : -1;
	int ncolumns = 0, min_columns = 0;
	if(value_c > ncolumns) ncolumns = value_c;
	if(date_c > ncolumns) ncolumns = date_c;
	if(AC1_c > ncolumns) ncolumns = AC1_c;
	if(AC2_c > ncolumns) ncolumns = AC2_c;
	min_columns = ncolumns;
	if(description_c > ncolumns) ncolumns = description_c;
	if(comments_c > ncolumns) ncolumns = comments_c;
	if((description_c > 0 && (description_c == value_c || description_c == date_c || description_c == AC1_c || description_c == AC2_c || description_c == comments_c))
		   || (value_c > 0 && (value_c == date_c || value_c == AC1_c || value_c == AC2_c || value_c == comments_c))
		   || (date_c > 0 && (date_c == AC1_c || date_c == AC2_c || date_c == comments_c))
		   || (AC1_c > 0 && (AC1_c == AC2_c || AC1_c == comments_c))
		   || (AC2_c > 0 && AC2_c == comments_c)
	  ) {
		KMessageBox::error(this, i18n("The same column number is selected multiple times."));
		return false;
	}
	bool create_missing = createMissingButton->isChecked() && type != 4;
	QString description, comments;
	if(description_c < 0) description = valueDescriptionEdit->text();
	if(comments_c < 0) comments = valueCommentsEdit->text();
	QMap<QString, Account*> eaccounts, iaccounts, aaccounts;
	Account *ac1 = NULL, *ac2 = NULL;
	if(AC1_c < 0 || AC2_c < 0) {
		ExpensesAccount *ea = budget->expensesAccounts.first();
		while(ea) {eaccounts[ea->name()] = ea; ea = budget->expensesAccounts.next();}
		IncomesAccount *ia = budget->incomesAccounts.first();
		while(ia) {iaccounts[ia->name()] = ia; ia = budget->incomesAccounts.next();}
		AssetsAccount *aa = budget->assetsAccounts.first();
		while(aa) {aaccounts[aa->name()] = aa; aa = budget->assetsAccounts.next();}
	}
	if(AC1_c < 0) {
		int i = 0;
		Account *account = NULL;
		if(type == 0) account = budget->expensesAccounts.first();
		else if(type == 1) account = budget->incomesAccounts.first();
		else if(type == 2) account = budget->assetsAccounts.first();
		while(account) {
			if(type == 2) {
				while(account == budget->balancingAccount || ((AssetsAccount*) account)->accountType() == ASSETS_TYPE_SECURITIES) {
					account = budget->assetsAccounts.next();
					if(!account) break;
				}
			}
			if(i == valueAC1Edit->currentItem()) {
				ac1 = account;
				break;
			}
			if(type == 0) {
				account = budget->expensesAccounts.next();
			} else if(type == 1) {
				account = budget->incomesAccounts.next();
			} else if(type == 2) {
				account = budget->assetsAccounts.next();
			}
			i++;
		}
	}
	if(AC2_c < 0) {
		int i = 0;
		Account *account = budget->assetsAccounts.first();
		while(account) {
			while(account == budget->balancingAccount || ((AssetsAccount*) account)->accountType() == ASSETS_TYPE_SECURITIES) {
				account = budget->assetsAccounts.next();
				if(!account) break;
			}
			if(i == valueAC2Edit->currentItem()) {
				ac2 = account;
				break;
			}
			account = budget->assetsAccounts.next();
			i++;
		}
		if(ac1 == ac2) {
			KMessageBox::error(this, i18n("Selected from account is the same as the to account."));
			return false;
		}
	}
	QDate date;
	if(date_c < 0) {
		date = valueDateEdit->date();
		if(!date.isValid()) {
			KMessageBox::error(this, i18n("Invalid date."));
			return false;
		}
	}
	double value = 0.0;
	if(value_c < 0) {
		value = valueValueEdit->value();
	}

	KURL url = fileEdit->url();
	QString tmpfile;
	if(!url.isLocalFile()) {
		if(!KIO::NetAccess::download(url, tmpfile, this)) {
			KMessageBox::error(this, i18n("Couldn't fetch %1.").arg(url.prettyURL()));
			return false;
		}
	} else {
		tmpfile = url.path();
	}
	QFile file(tmpfile);
	if(!file.open(IO_ReadOnly) ) {
		KMessageBox::error(this, i18n("Couldn't open %1 for reading.").arg(url.prettyURL()));
		return false;
	} else if(!file.size()) {
		KMessageBox::error(this, i18n("Error reading reading %1.").arg(url.prettyURL()));
		return false;
	}
	QTextStream fstream(&file);
	fstream.setEncoding(QTextStream::Locale);

	bool had_data = false;
	int successes = 0;
	int failed = 0;
	bool missing_columns = false, value_error = false, date_error = false;
	bool AC1_empty = false, AC2_empty = false, AC1_missing = false, AC2_missing = false, AC_security = false, AC_balancing = false, AC_same = false;
	int AC1_c_bak = AC1_c;
	int AC2_c_bak = AC2_c;
	int row = 0;
	QString line = fstream.readLine();
	QString new_ac1 = "", new_ac2 = "";
	QDate curdate = QDate::currentDate();
	while(!line.isNull()) {
		row++;
		if(row >= first_row && !line.isEmpty()) {
			QStringList columns = QStringList::split(delimiter, line, true);
			for(QStringList::Iterator it = columns.begin(); it != columns.end(); ++it) {
				int i = 0;
				while(i < (int) (*it).length() && ((*it)[i] == ' ' || (*it)[i] == '\t')) {
					i++;
				}
				if(!(*it).isEmpty() && (*it)[i] == '\"') {
					(*it).remove(0, i + 1);
					i = (*it).length() - 1;
					while(i > 0 && ((*it)[i] == ' ' || (*it)[i] == '\t')) {
						i--;
					}
					if(i >= 0 && (*it)[i] == '\"') {
						(*it).truncate(i);
					} else {
						QStringList::Iterator it2 = it;
						++it2;
						while(it2 != columns.end()) {
							i = (*it2).length() - 1;
							while(i > 0 && ((*it2)[i] == ' ' || (*it2)[i] == '\t')) {
								i--;
							}
							if(i >= 0 && (*it2)[i] == '\"') {
								(*it2).truncate(i);
								*it += delimiter;
								*it += *it2;
								columns.erase(it2);
								break;
							}
							*it += delimiter;
							*it += *it2;
							columns.erase(it2);
							it2 = it;
							++it2;
						}
					}
				}
				*it = (*it).stripWhiteSpace();
			}
			if((int) columns.count() < min_columns) {
				missing_columns = true;
				failed++;
			} else {
				bool success = true;
				if(success && description_c > 0) {
					description = columns[description_c - 1];
				}
				if(success && value_c > 0) {
					bool ok = true;
					value = readCSVValue(columns[value_c - 1], ok);
					if(!ok || (type < 2 && value < 0.0)) {
						value_error = true;
						success = false;
					}
				}
				if(success && date_c > 0) {
					date = readCSVDate(columns[date_c - 1]);
					if(!date.isValid()) {
						date_error = true;
						success = false;
					}
				}
				if(success && type == 4 && value < 0.0) {
					AC1_c = AC2_c_bak;
					AC2_c = AC1_c_bak;
					value = -value;
				}
				if(success && AC1_c > 0) {
					QMap<QString, Account*>::iterator it_ac;
					bool found = false;
					if(type == 0 || (type == 3 && value < 0.0)) {
						it_ac = eaccounts.find(columns[AC1_c - 1]);
						found = (it_ac != eaccounts.end());
					} else if(type == 1 || type == 3) {
						it_ac = iaccounts.find(columns[AC1_c - 1]);
						found = (it_ac != iaccounts.end());
					} else if(type == 2) {
						it_ac = aaccounts.find(columns[AC1_c - 1]);
						found = (it_ac != aaccounts.end());
					} else if(type == 4) {
						it_ac = iaccounts.find(columns[AC1_c - 1]);
						found = (it_ac != iaccounts.end());
						if(!found) {
							it_ac = aaccounts.find(columns[AC1_c - 1]);
							found = (it_ac != aaccounts.end());
						}
					}
					if(found) {
						ac1 = it_ac.data();
						if(ac1->type() == ACCOUNT_TYPE_ASSETS && ((AssetsAccount*) ac1)->accountType() == ASSETS_TYPE_SECURITIES) {
							AC_security = true;
							success = false;
						} else if(type != 2 && type != 4 && ac1 == budget->balancingAccount) {
							AC_balancing = true;
							success = false;
						}
					} else if(columns[AC1_c - 1].isEmpty()) {
						AC1_empty = true;
						success = false;
					} else if(create_missing) {
						new_ac1 = columns[AC1_c - 1];
					} else {
						AC1_missing = true;
						success = false;
					}
				}
				if(success && AC2_c > 0) {
					QMap<QString, Account*>::iterator it_ac;
					bool found = false;
					if(type == 4) {
						it_ac = iaccounts.find(columns[AC2_c - 1]);
						found = (it_ac != iaccounts.end());
						if(!found) {
							it_ac = aaccounts.find(columns[AC2_c - 1]);
							found = (it_ac != aaccounts.end());
						}
					} else {
						it_ac = aaccounts.find(columns[AC2_c - 1]);
						found = (it_ac != aaccounts.end());
					}
					if(found) {
						ac2 = it_ac.data();
						if(ac1 == ac2) {
							AC_same = true;
							success = false;
						} else if(ac2->type() == ACCOUNT_TYPE_ASSETS && ((AssetsAccount*) ac2)->accountType() == ASSETS_TYPE_SECURITIES) {
							AC_security = true;
							success = false;
						} else if(ac2 == budget->balancingAccount) {
							if(type != 2) {
								AC_balancing = true;
								success = false;
							} else {
								if(type == 4 && ac1->type() != ACCOUNT_TYPE_ASSETS) {
									it_ac = aaccounts.find(columns[AC1_c - 1]);
									found = it_ac != aaccounts.end();
									if(found) {
										ac1 = it_ac.data();
										if(ac1->type() == ACCOUNT_TYPE_ASSETS && ((AssetsAccount*) ac1)->accountType() == ASSETS_TYPE_SECURITIES) {
											AC_security = true;
											success = false;
										} else if(ac1 == budget->balancingAccount) {
											AC_same = true;
											success = false;
										}
									} else {
										AC_balancing = true;
										success = false;
									}
								}
								if(success) {
									value = -value;
									Account *ac1_bak = ac1;
									ac1 = ac2;
									ac2 = ac1_bak;
								}
							}
						} else if(type == 4 && ac1 == budget->balancingAccount && ac2->type() != ACCOUNT_TYPE_ASSETS) {
							it_ac = aaccounts.find(columns[AC2_c - 1]);
							found = it_ac != aaccounts.end();
							if(found) {
								ac2 = it_ac.data();
								if(ac2->type() == ACCOUNT_TYPE_ASSETS && ((AssetsAccount*) ac2)->accountType() == ASSETS_TYPE_SECURITIES) {
									AC_security = true;
									success = false;
								} else if(ac2 == budget->balancingAccount) {
									AC_same = true;
									success = false;
								}
							} else {
								AC_balancing = true;
								success = false;
							}
						}
					} else if(columns[AC2_c - 1].isEmpty()) {
						AC2_empty = true;
						success = false;
					} else if(create_missing) {						
						new_ac2 = columns[AC2_c - 1];
						if(new_ac1 == new_ac2) {
							new_ac1 = "";
							new_ac2 = "";
							AC_same = true;
							success = false;
						}
					} else {
						AC2_missing = true;
						success = false;
					}
				}
				if(type == 4) {
					AC1_c = AC1_c_bak;
					AC2_c = AC2_c_bak;
				}
				if(success && comments_c > 0) {
					comments = columns[comments_c - 1];
				}
				printf("%s %s %s\n", line.ascii(), new_ac1.ascii(), new_ac2.ascii());
				if(success) {
					if(!new_ac1.isEmpty()) {
						if(type == 0 || (type == 3 && value < 0.0)) {ac1 = new ExpensesAccount(budget, new_ac1); budget->addAccount(ac1); eaccounts[ac1->name()] = ac1;}
						else if(type == 1 || type == 3) {ac1 = new IncomesAccount(budget, new_ac1); budget->addAccount(ac1); iaccounts[ac1->name()] = ac1;}
						else if(type == 2) {ac1 = new AssetsAccount(budget, ASSETS_TYPE_CASH, new_ac1); budget->addAccount(ac1); aaccounts[ac1->name()] = ac1;}
						new_ac1 = "";
					}
					if(!new_ac2.isEmpty()) {
						ac2 = new AssetsAccount(budget, ASSETS_TYPE_CASH, new_ac2);
						budget->addAccount(ac2);
						aaccounts[ac2->name()] = ac2;
						new_ac2 = "";
					}
					Transaction *trans = NULL;
					switch(type) {
						case 0: {
							trans = new Expense(budget, value, date, (ExpensesAccount*) ac1, (AssetsAccount*) ac2, description, comments);
							successes++;
							break;
						}
						case 1: {
							trans = new Income(budget, value, date, (IncomesAccount*) ac1, (AssetsAccount*) ac2, description, comments);
							successes++;
							break;
						}
						case 2: {
							if(ac1 == budget->balancingAccount) {
								trans = new Balancing(budget, value, date, (AssetsAccount*) ac2, description);
							} else if(value < 0.0) {
								trans = new Transfer(budget, -value, date, (AssetsAccount*) ac2, (AssetsAccount*) ac1, description, comments);
							} else {
								trans = new Transfer(budget, value, date, (AssetsAccount*) ac1, (AssetsAccount*) ac2, description, comments);
							}
							successes++;
							break;
						}
						case 3: {
							if(value < 0.0) {
								trans = new Expense(budget, -value, date, (ExpensesAccount*) ac1, (AssetsAccount*) ac2, description, comments);
							} else {
								trans = new Income(budget, value, date, (IncomesAccount*) ac1, (AssetsAccount*) ac2, description, comments);
							}
							successes++;
							break;
						}
						case 4: {
							if(ac1 == budget->balancingAccount) {
								trans = new Balancing(budget, value, date, (AssetsAccount*) ac2, description);
							} else if(ac1->type() == ACCOUNT_TYPE_INCOMES) {
								trans = new Income(budget, value, date, (IncomesAccount*) ac1, (AssetsAccount*) ac2, description, comments);
							} else if(ac2->type() == ACCOUNT_TYPE_EXPENSES) {
								trans = new Expense(budget, value, date, (ExpensesAccount*) ac2, (AssetsAccount*) ac1, description, comments);
							} else {
								trans = new Transfer(budget, value, date, (AssetsAccount*) ac1, (AssetsAccount*) ac2, description, comments);
							}
							successes++;
							break;
						}
					}
					if(trans) {
						if(trans->date() > curdate) {
							budget->addScheduledTransaction(new ScheduledTransaction(budget, trans, NULL));
						} else {
							budget->addTransaction(trans);
						}
					}
				} else {
					failed++;
				}
			}
			had_data = true;
		}
		line = fstream.readLine();
	}
	
	if(!url.isLocalFile()) {
		KIO::NetAccess::removeTempFile(tmpfile);
	}

	QString info = "", details = "";
	if(successes > 0) {
		info = i18n("Successfully imported 1 transaction.", "Successfully imported %n transactions.", successes);
	} else {
		info = i18n("Unable to import any transactions imported.");
	}
	if(failed > 0) {
		info += '\n';
		info += i18n("Failed to import 1 data row.", "Failed to import %n data rows.", failed);
		if(missing_columns) {details += "\n-"; details += i18n("Required columns missing.");}
		if(value_error) {details += "\n-"; details += i18n("Invalid value.");}
		if(date_error) {details += "\n-"; details += i18n("Invalid date.");}
		if(AC1_empty) {details += "\n-"; if(type == 0 || type == 1 || type == 2) {details += i18n("Empty category name.");} else {details += i18n("Empty account name.");}}
		if(AC2_empty) {details += "\n-"; details += i18n("Empty account name.");}
		if(AC1_missing) {details += "\n-"; if(type == 0 || type == 1 || type == 2) {details += i18n("Unknown category found.");} else {details += i18n("Unknown account found.");}}
		if(AC2_missing) {details += "\n-"; details += i18n("Unknown account found.");}
		if(AC_security) {details += "\n-"; details += i18n("Cannot import security transactions (to/from security accounts).");}
		if(AC_balancing) {details += "\n-"; details += i18n("Balancing account wrongly used.");}
		if(AC_same) {details += "\n-"; details += i18n("Same to and from account/category.");}
	} else if(successes == 0) {
		info = i18n("No data found.");
	}	
	if(failed > 0 || successes == 0) {
		KMessageBox::sorry(this, info + details);
	} else {
		KMessageBox::information(this, info);
	}
	return successes > 0;
}
void ImportCSVDialog::accept() {
	if(import()) KWizard::accept();
}

#include "importcsvdialog.moc"
