##
## close_other_tabs.py
##
## Provides menu command and keyboard shortcuts to close tabs
## other than the active tab
##
## Copyright (C) 2008 Sean Fulmer
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
##

import epiphany

import gtk

def _close_other_tabs_cb (action, window, close_left=True, close_right=True):
    atab = window.get_active_child()
    right=False
    for tab in window.get_notebook():
        if tab == atab:
            right=True
        elif (right and close_right) or (not right and close_left):
            tab.close()

def _close_left_tabs_cb(action, window):
    _close_other_tabs_cb (action, window, True, False)
def _close_right_tabs_cb(action, window):
    _close_other_tabs_cb (action, window, False, True)


# signal callbacks
def _switch_page_cb(notebook, page, page_num, window):
    """
    enables/disables menus when tab is changed
    """
    
    _set_menu_states(notebook, page_num, window)

def _page_added_removed_reordered_cb(notebook, child, page_num, window):
    """
    enables/disables menus when tab is added, removed, or re-ordered
    """
    
    page_num = window.get_notebook().get_current_page()
    _set_menu_states(notebook, page_num, window)

def _set_menu_states(notebook, page_num, window):
    """
    sets menu state based on presence of other tabs
    """

    page = page_num
    pages = notebook.get_n_pages()
    not_only = pages > 1
    not_first = page > 0
    not_last = page + 1 < pages

    action_group = window._close_other_tabs[0]

    action = action_group.get_action('CloseOtherTabs')
    action.set_sensitive(not_only)

    action = action_group.get_action('CloseLeftTabs')
    action.set_sensitive(not_first)

    action = action_group.get_action('CloseRightTabs')
    action.set_sensitive(not_last)



_ui_str = """
<ui>
<menubar name="menubar">
<menu name="TabsMenu" action="Tabs">
    <separator />
    <menuitem name="TabsCloseOtherTabsMenu" action="CloseOtherTabs"/>
    <menuitem name="TabsCloseLeftTabsMenu" action="CloseLeftTabs"/>
    <menuitem name="TabsCloseRightTabsMenu" action="CloseRightTabs"/>
</menu>
</menubar>
<popup name="EphyNotebookPopup" action="NotebookPopupAction">
    <separator />
    <menuitem name="TabsCloseOtherTabsMenuENP" action="CloseOtherTabs"/>
    <menuitem name="TabsCloseLeftTabsMenuENP" action="CloseLeftTabs"/>
    <menuitem name="TabsCloseRightTabsMenuENP" action="CloseRightTabs"/>
</popup>
</ui>
"""

_actions = [
            ('CloseOtherTabs', 'close_other_tabs', 'Close _Other Tabs',
                '<control><shift>W', None, _close_other_tabs_cb),
            ('CloseLeftTabs', 'close_left_tabs', 'Close _Left Tabs',
                None, None, _close_left_tabs_cb),
            ('CloseRightTabs', 'close_right_tabs', 'Close _Right Tabs',
                None, None, _close_right_tabs_cb),
           ]

def attach_window(window):

    # add ui stuff
    ui_manager = window.get_ui_manager()
    group = gtk.ActionGroup('CloseOtherTabs')
    group.add_actions(_actions, window)
    ui_manager.insert_action_group(group, 0)
    ui_id = ui_manager.add_ui_from_string(_ui_str)
    window._close_other_tabs = (group, ui_id)

    # connect switch_page signal
    notebook = window.get_notebook()
    signal_id = notebook.connect('switch_page', _switch_page_cb, window)
    notebook._switch_page_signal = signal_id

    # connect page_added signal
    signal_id = notebook.connect('page_added', _page_added_removed_reordered_cb, window)
    notebook._page_added_signal = signal_id

    # connect page_removed signal
    signal_id = notebook.connect('page_removed', _page_added_removed_reordered_cb, window)
    notebook._page_removed_signal = signal_id

    # connect page_added signal
    signal_id = notebook.connect('page_reordered', _page_added_removed_reordered_cb, window)
    notebook._page_reordered_signal = signal_id


def detach_window(window):

    # remove ui stuff
    group, ui_id = window._close_other_tabs
    del window._close_other_tabs
    ui_manager = window.get_ui_manager()
    ui_manager.remove_ui(ui_id)
    ui_manager.remove_action_group(group)
    ui_manager.ensure_update()

    # disconnect switch_page signal
    notebook = window.get_notebook()
    signal_id = notebook._switch_page_signal
    notebook.disconnect(signal_id)
    del notebook._switch_page_signal

    # disconnect page_added signal
    signal_id = notebook._page_added_signal
    notebook.disconnect(signal_id)
    del notebook._page_added_signal

    # disconnect page_removed signal
    signal_id = notebook._page_removed_signal
    notebook.disconnect(signal_id)
    del notebook._page_removed_signal

    # disconnect page_reordered signal
    signal_id = notebook._page_reordered_signal
    notebook.disconnect(signal_id)
    del notebook._page_reordered_signal
