/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: gui.hh,v 1.12.2.1 2003/09/28 20:43:30 dheck Exp $
 */
#ifndef GUI_HH_INCLUDED
#define GUI_HH_INCLUDED

#include "px/pxfwd.hh"
#include "px/geom.hh"
#include "SDL.h"
#include <vector>
#include <string>

//======================================================================
// EVENTS AND EVENT HANDLERS
//======================================================================
namespace gui
{
    class Widget;

    class ActionListener {
    public:
        virtual ~ActionListener() {}
        virtual void on_action(Widget */*w*/) {};
    };
}

//======================================================================
// GUI WIDGETS
//======================================================================
namespace gui
{
    class Menu;
    class Container;

    class Widget : public ActionListener {
    public:
        /*
        ** Widget interface
        */
        virtual void draw (px::GC &gc, const px::Rect &r) = 0;
        virtual void activate() {}
        virtual void deactivate() {}
        virtual bool on_event(const SDL_Event &/*e*/) { return false; }

        virtual void move(int x, int y) { area.x = x; area.y = y; }

        /*
        ** Accessors
        */
        void set_size(int w, int h) {area.w = w; area.h = h;}

        px::Rect get_area() const { return area; }
        void set_area(const px::Rect &r) { area = r; }
        int get_x() const { return area.x; }
        int get_y() const { return area.y; }
        int get_w() const { return area.w; }
        int get_h() const { return area.h; }
        Menu *get_menu() const { return menu; }

        void set_parent(Widget *parent) { m_parent = parent; }

        void set_listener(ActionListener *al) {
            listener = al;
        }
    protected:
        Widget(Widget *parent=0);
        void redraw();
        void redraw(const px::Rect &r);

        void invoke_listener();
    private:

        friend class Menu;
        void set_menu(Menu *m) { menu=m; }

        px::Rect        area;
        Menu           *menu;
        Widget         *m_parent;
        ActionListener *listener;
    };

//----------------------------------------
// Container, List, HList, VList
//----------------------------------------

    class Container : public Widget {
    public:
        Container() {}
        ~Container();

        void add_child (Widget *w);
        void redraw_child (Widget *w, const px::Rect &r);

        // Widget interface.
        void draw (px::GC& gc, const px::Rect &r);
        void move (int x, int y);

        Widget *find_widget(int x, int y);
        Widget *find_adjacent_widget(Widget *from, int x, int y);

    protected:
        typedef std::vector<Widget *> WidgetList;
        typedef WidgetList::iterator iterator;

        iterator begin() { return m_widgets.begin(); }
        iterator end() { return m_widgets.end(); }
        WidgetList m_widgets;

    private:
        px::Rect boundingbox();
    };

    class List : public Container {
    public:
        void set_spacing (int pixels);
    protected:
        List(int spacing=5) : m_spacing(spacing) {}
    private:
        int m_spacing;          // # of pixels between container items
    };

    class HList : public List {
    public:
        HList() : List() {}
#if 0 // currently unused
        void add_back(Widget *w) {
            Widget *lastw = 0;
            if (m_widgets.size() > 0)
                lastw = m_widgets[m_widgets.size()-1];
        }
#endif
    private:

    };

    class VList : public List {
    public:

    private:

    };

    class Image : public Widget {
    public:
        Image (const std::string &iname) : imgname(iname) {}
        void draw (px::GC &gc, const px::Rect &r);
    private:
        std::string imgname;
    };

    enum HAlignment {
        HALIGN_LEFT,
        HALIGN_CENTER,
        HALIGN_RIGHT
    };

    enum VAlignment {
        VALIGN_TOP,
        VALIGN_CENTER,
        VALIGN_BOTTOM
    };

    class Label : public Widget {
    public:
        Label (const std::string &text="");

        // Widget interface
        void draw (px::GC &gc, const px::Rect &r);

        // Methods
        void set_text (const std::string &text);
        void set_font (px::Font *font);
        void set_alignment (HAlignment halign, VAlignment valign=VALIGN_CENTER);
    private:
        // Variables.
        std::string m_text;
        px::Font   *m_font;
        HAlignment  m_halign;
        VAlignment  m_valign;
    };

    class Button : public Widget {
    protected:
        Button();

        // Widget interface.
        void draw(px::GC &gc, const px::Rect &r);
        void activate();
        void deactivate();
    private:

        bool m_activep;
    };


    class PushButton : public Button {
    public:
        PushButton();

        bool is_pressed() { return m_pressedp; }
    protected:
        bool on_event(const SDL_Event &e);
        void deactivate();
    private:
        bool m_pressedp;
    };

//----------------------------------------
// TextButton.
//----------------------------------------
    class TextButton : public PushButton {
    public:
        TextButton(const std::string &t, ActionListener *al=0);

        void set_text(const std::string &t);
    private:
        // Widget interface.
        void draw(px::GC &gc, const px::Rect &r);

        // Variables.
        static px::Font *menufont, *menufont_pressed;

        std::string text;
    };

//----------------------------------------
// ImageButton.
//----------------------------------------
    class ImageButton : public PushButton {
    public:
        ImageButton(const std::string &unselected,
                    const std::string &selected,
                    ActionListener    *al = 0);
    private:
        // Widget interface.
        void        draw(px::GC &gc, const px::Rect &r);
        std::string fname_sel, fname_unsel;
    };

//----------------------------------------
// Menu.
//----------------------------------------
    class Menu : public Container {
    public:
        Menu();

        //! true: ok, false: menu aborted by user
        bool manage(px::Screen *scr);

        void add(Widget *w);
        void add(Widget *w, px::Rect r);
        void center(px::Screen *scr);

        void draw_all(px::Screen *scr);

        void quit();
        void abort();

        void invalidate_area(const px::Rect &r);
        void invalidate_all();

    protected:
        // Menu interface.
        virtual void draw_background(px::GC &/*gc*/) {}
        virtual void tick (double /*dtime*/) {}

    private:
        void handle_event(const SDL_Event &e);
        void redraw_area(const px::Rect &r);

        void switch_active_widget(Widget *to_activate);

        void track_active_widget( int x, int y ); // used by mouse
        void goto_adjacent_widget(int xdir, int ydir); // used by keyboard

        // Variables.
        Widget *active_widget;
        bool quitp, abortp;
        px::RectList dirtyrects;
        px::Screen *screen;
    };
}
#endif
