#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/prochandle.h"

#include "cfg.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_interps.h"
#include "endeavour2.h"
#include "edv_help.h"
#include "edv_mime_type_install.h"
#include "edv_open.h"
#include "edv_cb.h"
#include "edv_op.h"
#include "edv_interps_op.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"


void EDVInterPSOPProcessCommand(
	edv_core_struct *core, const gchar *cmd
);
void EDVInterPSOPProcessCommands(
	edv_core_struct *core,
	gchar **cmd_list
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Processes the given InterPS command.
 */
void EDVInterPSOPProcessCommand(
	edv_core_struct *core, const gchar *cmd
)
{
	gint argc;
	gchar **argv;
	const gchar *op;
	cfg_item_struct *cfg_list = core->cfg_list;

	if(STRISEMPTY(cmd))
	    return;

	/* Explode command into arguments
	 *
	 * Note that we use ExecExplodeCommand() to handle arguments
	 * wraped in '"' characters
	 */
	argv = ExecExplodeCommand(cmd, &argc);
	if(argv == NULL)
	    return;

	/* Get operation argument (first argument) */
	op = (argc > 0) ? argv[0] : NULL;
	if(op == NULL)
	{
	    strlistfree(argv, argc);
	    return;
	}

	/* Begin handling operation */

	/* New Window */
	if(!g_strcasecmp(op, "new_window"))
	{
	    /* Note that the "new_window" command is sent by another
	     * Endeavour process (not another program) because otherwise
	     * if Endeavour is not running a new window cannot be
	     * created
	     *
	     * So other programs run Endeavour to instruct it to create
	     * a new window and Endeavour will check to see if it is
	     * already running and (if it is) then sends a "new_window"
	     * command which is handled here
	     */
	    if(argc >= 2)
	    {
		const gchar	*window_name = argv[1],
				*startup_path = (argc > 2) ? argv[2] : NULL,
				*window_extra = (argc > 3) ? argv[3] : NULL;

		/* Create the new window */
		EDVNewWindow(
		    core,
		    window_name,	/* Window Name */
		    startup_path,	/* Startup Path */
		    window_extra,	/* Extra */
		    NULL		/* Toplevel */
		);
	    }
	}
	/* Set Cfg Integer */
	else if(!g_strcasecmp(op, "set_cfg_i"))
	{
	    if(argc >= 3)
	    {
		const gchar *parm = argv[1];
		gint val = ATOI(argv[2]);
		EDV_SET_I(parm, val);
	    }
	}
	/* Set Cfg Unsigned Integer */
	else if(!g_strcasecmp(op, "set_cfg_ui"))
	{
	    if(argc >= 3)
	    {
		const gchar *parm = argv[1];
		guint val = (guint)ATOI(argv[2]);
		EDV_SET_I(parm, (gint)val);
	    }
	}
	/* Set Cfg Long */
	else if(!g_strcasecmp(op, "set_cfg_l"))
	{
	    if(argc >= 3)
	    {
		const gchar *parm = argv[1];
		glong val = ATOL(argv[2]);
		EDV_SET_L(parm, val);
	    }
	}
	/* Set Cfg Unsigned Long */
	else if(!g_strcasecmp(op, "set_cfg_ul"))
	{
	    if(argc >= 3)
	    {
		const gchar *parm = argv[1];
		gulong val = ATOL(argv[2]);
		EDV_SET_UL(parm, val);
	    }
	}
	/* Set Cfg Float */
	else if(!g_strcasecmp(op, "set_cfg_f"))
	{
	    if(argc >= 3)
	    {
		const gchar *parm = argv[1];
		gfloat val = ATOF(argv[2]);
		EDV_SET_F(parm, val);
	    }
	}
	/* Set Cfg Double */
	else if(!g_strcasecmp(op, "set_cfg_d"))
	{
	    if(argc >= 3)
	    {
		const gchar *parm = argv[1];
		gdouble val = (gdouble)ATOF(argv[2]);
		EDV_SET_D(parm, val);
	    }
	}
	/* Set Cfg String */
	else if(!g_strcasecmp(op, "set_cfg_s"))
	{
	    if(argc >= 3)
	    {
		const gchar *parm = argv[1];
		const gchar *val = argv[2];
		EDV_SET_S(parm, val);
	    }
	}
	/* Write Protect Changed Notify */
	else if(!g_strcasecmp(op, "write_protect_changed_notify"))
	{
	    EDVWriteProtectChangedEmit(
		core,
		EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT)
	    );
	}
	/* Object Added Notify */
	else if(!g_strcasecmp(op, "object_added_notify"))
	{
	    if(argc >= 2)
	    {
		const gchar *path = argv[1];
		if(!STRISEMPTY(path))
		{
		    struct stat lstat_buf;
		    if(!lstat(path, &lstat_buf))
			EDVObjectAddedEmit(core, path, &lstat_buf);
		}
	    }
	}
	/* Object Modified Notify */
	else if(!g_strcasecmp(op, "object_modified_notify"))
	{
	    if(argc >= 2)
	    {
		const gchar	*path = argv[1],
				*new_path = (argc > 2) ? argv[2] : NULL;
		if(!STRISEMPTY(path))
		{
		    struct stat lstat_buf;
		    if(!lstat(
			STRISEMPTY(new_path) ? path : new_path,
			&lstat_buf
		    ))
			EDVObjectModifiedEmit(
			    core, path, new_path, &lstat_buf
			);
		}
	    }
	}
	/* Object Removed Notify */
	else if(!g_strcasecmp(op, "object_removed_notify"))
	{
	    if(argc >= 2)
	    {
		const gchar *path = argv[1];
		if(!STRISEMPTY(path))
		    EDVObjectRemovedEmit(core, path);
	    }
	}
	/* Object Mounted Notify or Object Unmounted Notify */
	else if(!g_strcasecmp(op, "object_mounted_notify") ||
	        !g_strcasecmp(op, "object_unmounted_notify")
	)
	{
	    if(argc >= 2)
	    {
		const gchar *path = argv[1];
		gint dev_num;
		edv_device_struct *dev;

		/* The path could be a mount path or device path, see
		 * which one it is by matching it in our list of device
		 * references
		 */
		for(dev_num = 0; dev_num < core->total_devices; dev_num++)
		{
		    dev = core->device[dev_num];
		    if(dev == NULL)
			continue;

		    if(!STRISEMPTY(dev->mount_path) &&
		       !STRISEMPTY(path)
		    )
		    {
			if(!strcmp(dev->mount_path, path))
			    break;
		    }
		    if(!STRISEMPTY(dev->device_path) &&
		       !STRISEMPTY(path)
		    )
		    {
			if(!strcmp(dev->device_path, path))
			    break;
		    }
		}
		/* Got match? */
		if((dev_num < core->total_devices) &&
		   (dev != NULL)
		)
		{
		    /* Need to update mount states and device capacities */
		    EDVDevicesListUpdateMountStates(
			core->device, core->total_devices
		    );
		    EDVDevicesListUpdateStats(
			core->device, core->total_devices
		    );
		    /* Emit mount or unmount signal */
		    EDVObjectMountEmit(
			core, dev_num, dev,
			EDV_DEVICE_IS_MOUNTED(dev)
		    );
		}
	    }
	}
	/* Recycled Object Added Notify */
	else if(!g_strcasecmp(op, "recycled_object_added_notify"))
	{
	    if(argc >= 2)
	    {
		guint index = ATOI(argv[1]);
		EDVRecycledObjectAddedEmit(core, index);
	    }
	}
	/* Recycled Object Removed Notify */
	else if(!g_strcasecmp(op, "recycled_object_removed_notify"))
	{
	    if(argc >= 2)
	    {
		guint index = ATOI(argv[1]);
		EDVRecycledObjectRemovedEmit(core, index);
	    }
	}
	/* Reconfigured Notify */
	else if(!g_strcasecmp(op, "reconfigured_notify"))
	{
	    EDVReconfiguredEmit(core);
	}

	/* Append History */
	else if(!g_strcasecmp(op, "append_history"))
	{
	    /* Arguments:
	     *
	     * <type> <time_start> <time_end> <status>
	     * [source] [target] [comments]
	     */
	    if(argc >= 5)
	    {
		edv_history_type type = (edv_history_type)ATOI(argv[1]);
		gulong time_start = (gulong)ATOL(argv[2]);
		gulong time_end = (gulong)ATOL(argv[3]);
		gint status = ATOI(argv[4]);
		const gchar *source = (argc > 5) ? argv[5] : NULL;
		const gchar *target = (argc > 6) ? argv[6] : NULL;
		const gchar *comments = (argc > 7) ? argv[7] : NULL;
		/* Append history */
		EDVAppendHistory(
		    core,
		    type,
		    time_start,
		    time_end,
		    status,
		    source,
		    target,
		    comments
		);
	    }
	}
	/* Install MIME Type */
	else if(!g_strcasecmp(op, "install_mimetype"))
	{
	    /* Arguments:
	     *
	     * <class> <type>
	     * [value] [description]
	     * [handler]
	     * [icon_small_std_file] [icon_small_sel_file]
	     *  [icon_small_ext_file] [icon_small_hid_file]
	     * [icon_medium_std_file] [icon_medium_sel_file]
	     *  [icon_medium_ext_file] [icon_medium_hid_file]
	     * [icon_large_std_file] [icon_large_sel_file]
	     *  [icon_large_ext_file] [icon_large_hid_file]
	     * [command_name] [command] [...]
	     */
	    if(argc >= 3)
	    {
		GtkWidget *toplevel = NULL;
		edv_mime_types_list_win_struct *lw = core->mimetype_listwin;
		edv_mime_type_struct *m = EDVMimeTypeNew(
		    ATOI(argv[1]),			/* Class */
		    (argc > 3) ? argv[3] : NULL,	/* Value */
		    argv[2],				/* Type */
		    (argc > 4) ? argv[4] : NULL		/* Description */
		);

		/* Handler (arg 5) */
		m->handler = (argc > 5) ?
		    ATOI(argv[5]) : EDV_MIME_TYPE_HANDLER_COMMAND;

		/* Parse icon files (args 6 to 17) */
		if(argc > 17)
		{
		    m->small_icon_file[
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		    ] = STRDUP(argv[6]);
		    m->small_icon_file[
			EDV_MIME_TYPE_ICON_STATE_SELECTED
		    ] = STRDUP(argv[7]);
		    m->small_icon_file[
			EDV_MIME_TYPE_ICON_STATE_EXTENDED
		    ] = STRDUP(argv[8]);
		    m->small_icon_file[
			EDV_MIME_TYPE_ICON_STATE_HIDDEN
		    ] = STRDUP(argv[9]);

		    m->medium_icon_file[
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		    ] = STRDUP(argv[10]);
		    m->medium_icon_file[
			EDV_MIME_TYPE_ICON_STATE_SELECTED
		    ] = STRDUP(argv[11]);
		    m->medium_icon_file[
			EDV_MIME_TYPE_ICON_STATE_EXTENDED
		    ] = STRDUP(argv[12]);
		    m->medium_icon_file[
			EDV_MIME_TYPE_ICON_STATE_HIDDEN
		    ] = STRDUP(argv[13]);

		    m->large_icon_file[
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		    ] = STRDUP(argv[14]);
		    m->large_icon_file[
			EDV_MIME_TYPE_ICON_STATE_SELECTED
		    ] = STRDUP(argv[15]);
		    m->large_icon_file[
			EDV_MIME_TYPE_ICON_STATE_EXTENDED
		    ] = STRDUP(argv[16]);
		    m->large_icon_file[ 
			EDV_MIME_TYPE_ICON_STATE_HIDDEN
		    ] = STRDUP(argv[17]);
		}

		/* Parse commands (args 18 to ...) */
		if(argc > 18)
		{
		    gint i = 18;
		    edv_mime_type_command_struct *cmd;
		    while(i < argc)
		    {
			cmd = EDVMimeTypeCommandNew();
			if(cmd == NULL)
			    break;

			cmd->name = (argc > i) ?
                            STRDUP(argv[i]) : NULL;
                        i++;
                        cmd->command = (argc > i) ?
                            STRDUP(argv[i]) : NULL;
                        i++;

			m->commands_list = g_list_append(
			    m->commands_list, cmd
			);
		    }
		}

		if(EDVMimeTypesListWinIsMapped(lw))
		    toplevel = lw->toplevel;

		/* Install the MIME Type
		 *
		 * The MIME Type passed to this function will be
		 * installed (transfered) and should not be referenced
		 * again
		 */
		EDVMimeTypeInstall(
		    core, m,
		    -1,			/* Append */
		    TRUE,		/* Interactive */
		    TRUE,		/* Verbose */
		    toplevel
		);
	    }
	}

	/* Help */
	else if(!g_strcasecmp(op, "help"))
	{
	    const gchar *subject = (argc > 1) ? argv[1] : NULL;
	    EDVHelp(
		core,
		subject,		/* Subject */
	        NULL			/* Toplevel */
	    );
	}
	/* About */
	else if(!g_strcasecmp(op, "about"))
	{
	    const gchar *page_name = (argc > 1) ? argv[1] : NULL;
	    EDVAboutPage(
		core,
		page_name,	/* Page Name */
		NULL		/* Toplevel */
	    );
	}
	/* All else display error message */
	else
	{
	    gchar *buf = g_strdup_printf(
"Unsupported InterPS command:\n\
\n\
    %s\n\
\n\
Was received by this application.",
		op
	    );
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
"InterPS Command Warning",
buf,
"InterPS (Inter Process) commands are sent between different\n\
processes of this same application to ensure that only one copy of\n\
this application's process exists per user.\n\
\n\
If you are writing a program to send InterPS to this application,\n\
then you should contact the authors for assistance, otherwise you\n\
should regard this message as an internal programming error.\n",
		NULL
	    );
	    g_free(buf);
	}

	/* Delete arguments */
	strlistfree(argv, argc);
}

/*
 *	Processes the specified InterPS commands.
 *
 *	The cmd_list specifies the list of command strings. The last
 *	pointer in the array must be NULL to mark the end of the list.
 */
void EDVInterPSOPProcessCommands(
	edv_core_struct *core,
	gchar **cmd_list
)
{
	gint i;

	if((core == NULL) || (cmd_list == NULL))
	    return;

	for(i = 0; cmd_list[i] != NULL; i++)
	    EDVInterPSOPProcessCommand(core, cmd_list[i]);
}
