/* $Id: e2_complete__path.c 550 2007-07-22 10:30:40Z tpgww $

Copyright (C) 2004-2007 tooar <tooar@gmx.net>

This file is part of emelFM2.
emelFM2 is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

emelFM2 is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emelFM2; see the file GPL. If not, contact the Free Software
Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include "e2_complete.h"
#include <string.h>

/**
@brief
Note: to eliminate BGL-racing, no UI-change from here
@param parent absolute path of dir being processed, localised string
@param itemname name of discovered item
@param entries pointer to store for list of data items for @a parent
@param user_data pointer to data specified when the foreach was called, with members ->word and ->prefix

@return TRUE to signal the read has not been aborted
*/
static gboolean _e2_complete_path_drcb_check_dir (gchar *parent, gchar *itemname,
	GList **found, E2_Duo *user_data)
{
	gchar *utf, *localpath, *temp;
	gboolean is_exec, is_dir;

	utf = F_FILENAME_FROM_LOCALE (itemname);	//not DISPLAY
	if (g_str_has_prefix (utf, user_data->a))	//word
	{
#ifdef E2_VFSTMP
		//FIXME the path
#else
		localpath = g_build_filename (parent, itemname, NULL);
#endif
		is_exec = ! e2_fs_access (localpath, X_OK E2_ERR_NONE());
		is_dir = e2_fs_is_dir3 (localpath E2_ERR_NONE());
		g_free (localpath);
		if (is_exec || is_dir)
		{
			printd (DEBUG, "match: %s - %s", utf, user_data->a);
			if (user_data->b == NULL)	//prefix
			{
				temp = (is_dir) ?
					g_strconcat (utf, G_DIR_SEPARATOR_S, NULL):
					g_strdup (utf);
			}
			else
			{
				temp = (is_dir) ?
					g_strconcat (user_data->b, utf, G_DIR_SEPARATOR_S, NULL):
					g_strconcat (user_data->b, utf, NULL);
			}
			*found = g_list_append (*found, temp);
		}
	}
	F_FREE (utf);
	return TRUE;
}
/**
@brief add the sub-dirs and executables in @a dir that complete @a word, to list @a found
Downstream error handling expects BGL to be closed here.
@param localpath path of directory to scan, localised string, assumed no trailing /
@param word the text to be completed, utf8 string
@param prefix utf8 string to be prepended to each matching item, or NULL
@param found store of list to which matching items will be appended

@return the number of successful matches
*/
static gint _e2_complete_path_check_dir (gchar *localpath, gchar *word, gchar *prefix,
	GList **found)
{
	printd (DEBUG, "check_dir (dir:%s,word:%s,found:_,prefix:%s)", localpath, word, prefix);
	GList *checks = NULL;	//don't clobber callers list if error happens here
#ifdef E2_VFSTMP
	E2_FSType dirtype = ?;	//FIXME
#else
	E2_FSType dirtype = FS_LOCAL;	//FIXME
#endif
	E2_Duo pair = { word, prefix };

	checks = (GList *) e2_fs_dir_foreach (localpath, dirtype,
			_e2_complete_path_drcb_check_dir, &pair, NULL E2_ERR_NONE());

	if (E2DREAD_FAILED (checks))
		return 0;
	if (*found == NULL)
		*found = checks;
	else
		*found = g_list_concat (*found, checks);
	return (g_list_length (checks));
}
/**
@brief create list of executable items that are valid completions of @a line

The string to be completed extends from after the ' ' (if any) preceding
the cursor position @a pos, to that position. But if @a pos = 0, the string
will be empty
If the string starts with "./", the scanning occurs in the active directory
Otherwise the scan iterates over $PATH.

@param line utf-8 string containing the whole line to be completed
@param word utf-8 string, copy of part of @a line with the 'word' to be completed
@param pos 0-based index of cursor position (characters, not bytes) in @a line
@param found pointer to list to record matching items
@param flags pointer to bit-flags indicating type of match (dirs or files)
@param data discretionary data pointer for this method

@return the number of successful matches
*/
gint e2_complete_path (gchar *line, gchar *word, gint pos, GList **found,
	E2_CompleteFlags *flags, gpointer data)
{
	printd (DEBUG, "complete_path (line:%s,word:%s,pos:%d,found:_,flags:%d,data:_)",
		line, word, pos, *flags);

	if ((word == NULL) || (*word == '\0'))
		return 0;
/*
WRONG ! //the word to complete needs to be the whole line
	//(i.e. no ' ' in the line prior to the cursor position)
	//or else we can't match a path
	if (!g_str_equal (word, line))
	//error message in this case?
		return 0;
*/
	*found = NULL;
	gint ret = 0;

	if (g_str_has_prefix (word, E2_COMMAND_PREFIX))
	{	//current directory
		*flags |= E2_COMPLETE_FLAG_STOP;
		gchar *p1 = g_utf8_next_char (strrchr (word, G_DIR_SEPARATOR));
		gchar *p2 = e2_utf8_ndup (word, g_utf8_pointer_to_offset (word, p1));
		gchar *path;
#ifdef E2_VFSTMP
		path = ;//FIXME path and fstype for non-mounted dirs
#else
		//local CWD
		path = g_strconcat (curr_pane->path, p2, NULL);
#endif
		gchar *localpath = F_FILENAME_TO_LOCALE (path);
		ret += _e2_complete_path_check_dir (localpath, p1, p2, found);
		g_free (p2);
		g_free (path);
		F_FREE (localpath);
	}
	else
	{	//anywhere in $PATH
		printd (DEBUG, "complete from path");
		const gchar *p = g_getenv ("PATH");  //_I( = this ok for all languages?
		if (p != NULL) 	//CHECKME needed ?
		{
			gchar **paths = g_strsplit (p, G_SEARCHPATH_SEPARATOR_S, -1);
			//iterate over the path members
			gint i = 0;
			while (paths[i] != NULL)
			{
				ret += _e2_complete_path_check_dir (paths[i], word, NULL, found);
				i++;
			}
			g_strfreev (paths);
		}
	}

	if (ret > 0)
		*flags |= E2_COMPLETE_FLAG_STOP;
	return ret;
}
