package Emdebian::Tools;
#
#  Copyright (C) 2006, 2007  Neil Williams <codehelp@debian.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#

use Carp;
use warnings;
use strict;
use File::HomeDir;
use Cwd;
use Debian::Debhelper::Dh_Lib;
use Exporter;
use vars qw(@ISA @EXPORT);
@ISA=qw(Exporter);
@EXPORT=qw(emdeb_versionstring extract_emdebversion check_emdebian_control create_patches get_dpkg_cross_dir check_toolchains find_latest_gcc find_latest_libc check_ourrepo get_suite check_dist host_arch get_username get_workdir check_workdir prepare_checklist get_aptagent);

sub get_config
{
	my $dpkg_cross_dir = &get_dpkg_cross_dir;
	my $cfile = $dpkg_cross_dir . "/emsource";
	# otherwise use the debconf default.
	$cfile = "/etc/emsource.conf" if (! -f $cfile);
	# read emsource config file.
	my $config = Config::Auto::parse("$cfile", format => "colon");
	return $config;
}

sub get_username
{
	my $username = "";
	my $config = &get_config();
	$username = $config->{'username'} if ($config->{'username'} ne "");
	return $username;
}

sub get_workdir
{
	my $workdir = "";
	my $config = &get_config();
	$workdir = $config->{'workingdir'} if ($config->{'workingdir'} ne "");
	return $workdir;
}

sub get_aptagent
{
	my $aptagent = "apt-get";
	my $config = &get_config();
	croak ("missing aptagent entry in ~/.dpkg-cross/emsource - see /etc/emsource.conf")
		if (!defined($config->{'aptagent'}));
	$aptagent = "aptitude" if ($config->{'aptagent'} eq "false");
}

sub check_workdir
{
	my $workdir = $_[0];
	# ensure a new directory is writable by creating the directory ourselves
	system ("mkdir -p $workdir") if (! -d $workdir);
	my $msg = "Working directory has been specified as '$workdir' but it does not exist\n";
	$msg .= "and cannot be created using 'mkdir -p'!: $!\n";
	$msg .= "See emsource(1) for more information.\n";
	# avoid dying within the module, let the script decide whether to die.
	return $msg if (! -d $workdir);
	return "";
}

# Generates the emdebian version string, appended to the dv{VERSION}
# pass "new" for a new upstream package or 
# "next" for another release of the same upstream package or
# blank to get the complete version string.
sub emdeb_versionstring {
	my $emdebvers = "";
	my $debvers = $dh{VERSION};
	if ($debvers =~ /(.*)(em[0-9]+)$/) {
		$debvers = $1;
		$emdebvers = $2;
		$emdebvers =~ /^em([0-9]+)$/;
		my $emN = $1;
		$emN = 1 if (eval($emN) == 0);
		if ($_[0] eq "new") {
			$emN = 1;
			$emdebvers = "em$emN";
		}
		if ($_[0] eq "next") {
			$emN++;
			$emdebvers = "em$emN";
		}
	}
	else {
		$emdebvers="em1";
	}
	return "${debvers}${emdebvers}";
}

# Get just the emN part of the version string.
sub extract_emdebversion {
	if ($_[0] =~ /^Version: (.*)(em[0-9]+)$/) {
		return $2;
	}
	else { return ""; }
}

# check the current location is a debian package.
# leaves us in the directory above debian/ 
sub check_emdebian_control()
{
	my $pkg;
	# check this is a debian working directory
	# read debian/control
	# parse for locale packages.
	until (-f "debian/control")
	{
		chdir ".." or die "Cannot change directory ../ $!";
		if (cwd() eq '/') 
		{
			die "Cannot find debian/control anywhere!\nAre you in the source code tree?\n";
		}
	}
}

my @patchfiles = qw/rules control control.in changelog /;
# creates and updates patches for the defined @patchfiles.
sub create_patches
{
	my $package = shift;
	my $cwd = cwd;
	foreach my $file (@patchfiles)
	{
		if (! -f "debian/$file") { next; }
		chdir ("../");
		system "diff -u $package.old/debian/$file $cwd/debian/$file > emdebian-$file.patch";
		chdir ("$cwd");
	}
}

sub get_dpkg_cross_dir()
{
	my $home = File::HomeDir->my_home;
	# safeguard, just in case.
	$home = cwd if (!$home);
	my $path =  "$home/.dpkg-cross";
	mkdir $path if (! -d $path);
	return $path;
}

# prepares an EXACT string for each of all the required toolchain
# packages - returns an array of the package names.
sub prepare_checklist()
{
	my $arch = $_[0];
	my $target_gnu_type = $_[1];
	my $suite = &get_suite;
	my $gcc_latest =  &find_latest_gcc("gcc", $arch, $suite);
	my $gcc_vers = "gcc-" . $gcc_latest;
	my $libc_latest = &find_latest_libc("libc", $arch, $suite);
	my $libc_vers = "libc" . $libc_latest;
	push my @list ,"binutils-${target_gnu_type}";
	push @list, "${gcc_vers}-${target_gnu_type}";
	push @list, "${gcc_vers}-${target_gnu_type}-base";
	push @list, "${gcc_vers}-${target_gnu_type}";
	push @list, "cpp-${gcc_latest}-${target_gnu_type}";
	push @list, "g++-${gcc_latest}-${target_gnu_type}";
	push @list, "${libc_vers}-${arch}-cross";
	push @list, "${libc_vers}-dev-${arch}-cross";
	push @list, "libstdc++${libc_latest}-${arch}-cross";
	push @list, "libstdc++${libc_latest}-${gcc_latest}-dev-${arch}-cross";
	push @list, "libstdc++${libc_latest}-${gcc_latest}-dbg-${arch}-cross";
	push @list, "libstdc++${libc_latest}-${gcc_latest}-pic-${arch}-cross";
	push @list, "libgcc1-${arch}-cross";
	push @list, "linux-kernel-headers-${arch}-cross";
	return \@list;
}

# arg 1 : architecture. arg 2: target_gnu_type
sub check_toolchains()
{
	my $arch = $_[0];
	my $target = $_[1];
	my $list = &prepare_checklist($arch, $target);
	my $success = "";
	my $string = "";
	# keep these two formats in sync: space_packagename
	foreach my $pkg (@$list)
	{
		$string .= " $pkg";
	}
	$success = `dpkg-query -W -f=' \${Package}' $string 2>/dev/null`;
	return "true" if ($success eq $string);
	return "false" if ($success ne $string);
}

# Only use where the package NAME contains a mathematically correct
# version, e.g. gcc-3.4 vs gcc-4.1, libc6 vs libc7 etc.
# returns 0 if no gcc package can be found in the cache.
sub find_latest_gcc()
{
	my $arch = $_[1];
	my $suite = $_[2];
	my $dpkg_cross_dir = &get_dpkg_cross_dir;
	my $result = `apt-cache -o Apt::Architecture=$_[1] -c $dpkg_cross_dir/apt.conf-$_[2] pkgnames $_[0] 2>/dev/null`;
	my @list = split (/\n/, $result);
	my $choice = 0;
	foreach my $line (@list)
	{
		if ($line =~ /gcc-([0-9\.\-]*)$/)
		{
			if ($1 > $choice) { $choice = $1; }
		}
	}
	return $choice;
}

# Only use where the package NAME contains a mathematically correct
# version, e.g. gcc-3.4 vs gcc-4.1, libc6 vs libc7 etc.
sub find_latest_libc()
{
	my $arch = $_[1];
	my $suite = $_[2];
	my $dpkg_cross_dir = &get_dpkg_cross_dir;
	my $result = `apt-cache -o Apt::Architecture=$arch -c $dpkg_cross_dir/apt.conf-$suite pkgnames $_[0] 2>/dev/null`;
	my @list = split (/\n/, $result);
	my $choice = 0;
	foreach my $line (@list)
	{
		if ($line =~ /libc([0-9]*)$/)
		{
			if ($1 > $choice) { $choice = $1; }
		}
	}
	return $choice;
}

sub check_ourrepo()
{
	open (SOURCES, "/etc/apt/sources.list") or 
		croak ("Cannot read /etc/apt/sources.list: $!\n");
	my $mirror = "";
	my $checked = 0;
	while (<SOURCES>)
	{
		if (/^deb\s([a-z:\/\.]*)\s([a-z]*)\smain\n$/)
		{
			$mirror = $1;
		}
		$checked = 1 if ($mirror =~ /http:\/\/www\.emdebian\.org\/debian/);
	}
	close (SOURCES);
	return &get_suite if ($checked == 1);
	return '';
}

sub get_suite()
{
	# list of suites supported by Emdebian toolchain repository.
	my @suites = qw/unstable testing stable sid etch/;
	my $policy = `apt-cache policy 2>/dev/null | grep "l=Debian,c=main"`;
	croak ("Unable to determine apt-cache policy: $!") if (!$policy);
	$policy =~ /o=Debian,a=([a-z]*),l=Debian,c=main/;
	my $suite = $1;
	foreach my $s (@suites)
	{
		# check the matched value is sensible.
		return $s if ($suite eq $s);
	}
	$suite = &check_dist if ($suite eq "");
	return $suite;
}

# if suite is not found from policy, try to identify which suite is being used
sub check_dist
{
	open (SOURCES, "/etc/apt/sources.list") or 
		croak ("Cannot read /etc/apt/sources.list: $!\n");
	while (<SOURCES>)
	{
		if (/^deb\s[a-z:\/\.]*\s([a-z]*)\smain\n$/)
		{
			close (SOURCES);
			return $1;
		}
	}
	close (SOURCES);
	return "";
}

sub host_arch()
{
	my $result = `dpkg-architecture -qDEB_HOST_ARCH`;
	chomp ($result);
	return $result;
}

1;
