/* BTP cat - Banana Tree Protocol tool
 * Copyright (C) 2001-2002  The Regents of the University of Michigan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <limits.h>
#include <errno.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <time.h>
#include <glib.h>

#include "btp.h"
#include "btp_debug.h"
#include "b_conn.h"
#include "util.h"



static gboolean peer_dump (gpointer data);
static gboolean in_func (GIOChannel* source, GIOCondition condition, gpointer data);

static int join (char* url);
void packet_func (Btp* btp, const void* buffer, 
		  guint length, gpointer user_data);
void error_func (Btp* btp, gpointer user_data);
static void shutdown (void);

static void usage (char* prog, int exitval);
static void sig_int_cb (int ignore);
static void num_conns_zero_cb (void);
static ssize_t writen (int fd, const void* buf, size_t len);


static BPeer*     bpeer;
static Btp*       btp;
static GIOChannel* in_channel;
static guint      in_watch;

static gboolean	  loopback = FALSE;
static gboolean   quiet = FALSE;


int
main (int argc, char* argv[])
{
  extern char* optarg;
  extern int   optind;
  char c;

  gchar* url;
  GMainLoop* main_loop;

  /* Seed RNG */
  srand(time(NULL) + getpid());

  while ((c = getopt(argc, argv, "d:lqh")) != -1) 
    {
      switch (c) 
	{
	case 'd':
	  {
	    int ival;
	    unsigned int xval;

	    ival = atoi (optarg);
	    if (ival && optarg[0] != '0')
	      btp_debug_flags = ival;
	    else if (sscanf(optarg, "%x", &xval) == 1)
	      btp_debug_flags = xval;
	    else
	      usage (argv[0], EXIT_FAILURE);
	    break;
	  }	  

	case 'l':
	  loopback = 1;
	  break;

	case 'q':
	  quiet = 1;
	  break;

	case 'h':
	case '?':
	  usage (argv[0], EXIT_SUCCESS);
	  break;
	}
    }

  if (argc == optind)
    usage (argv[0], EXIT_FAILURE);
  url = argv[optind];

  /* Join tree */
  if (join (url))
    {
      fprintf (stderr, "%s: Error joining %s\n", argv[0], url);
      exit (EXIT_FAILURE);
    }
  
  /* On sig int, leave group */
  signal (SIGINT, sig_int_cb);

  /* Watch STDIN */
  if (!quiet)
    {
      in_channel = g_io_channel_unix_new (STDIN_FILENO);
      g_assert (in_channel);

      in_watch = g_io_add_watch (in_channel, 
				 G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
				 in_func, NULL);
    }

  /* Print peer occasionally */
  if (btp_debug_flags & (1<<8))
    g_timeout_add (500, peer_dump, NULL);

  main_loop = g_main_new(FALSE);
  g_main_run (main_loop);

  exit (EXIT_SUCCESS);
  return 0;
}



/* **************************************** */

static gboolean
peer_dump (gpointer data)
{
  time_t t;
  char* time_str;

  if (!(btp_debug_flags & (1<<8)))
    return FALSE;

  t = time (NULL);
  t = mktime (gmtime(&t));
  time_str = ctime(&t);         /* don't free string */
  time_str[strlen(time_str) - 1] = '\0';

  fprintf (stderr, "time: %s UTC\n", time_str);

  if (bpeer)
    btp_print (stderr, bpeer);

  return TRUE;
}



/* **************************************** */

gboolean
in_func (GIOChannel* source, GIOCondition condition, gpointer data)
{
  GIOError error;
  gchar buffer[G_MAXUINT16];
  guint bytes_read;

  /* Check for error */
  if (condition & (G_IO_ERR | G_IO_HUP | G_IO_NVAL))
    exit (EXIT_FAILURE);

  g_assert (condition & G_IO_IN);

  /* Read the data into our buffer */
  error = g_io_channel_read (source, buffer, sizeof(buffer), &bytes_read);

  /* Check for stdin error */
  if (error != G_IO_ERROR_NONE)
    exit (EXIT_FAILURE);
  
  /* Check for EOF */
  else if (bytes_read == 0)
    {
      shutdown();
      return FALSE;
    }

  /* Otherwise, send */
  else
    {
      btp_send (btp, buffer, bytes_read);
      if (loopback && !quiet && bytes_read)
	writen (STDOUT_FILENO, buffer, bytes_read);
    }

  return TRUE;
}


static void
shutdown (void)
{
  if (in_watch)
    {
      g_source_remove (in_watch);
      in_watch = 0;
    }

  if (in_channel)
    {
      g_io_channel_unref (in_channel);
      in_channel = NULL;
    }

  if (btp)
    {
      btp_leave (btp);
      btp = NULL;
    }

  if (bpeer)
    {
      b_peer_delete (bpeer);
      bpeer = NULL;
    }

  if (!b_conn_num_conns_zero)
    exit (EXIT_SUCCESS);

  b_conn_num_conns_zero = num_conns_zero_cb;
}


/* ******************** */

static int
join (char* url)
{
  GURL*   	gurl;
  GInetAddr*	iface;
  
  if (btp)
    return 1;

  gurl = gnet_url_new (url);
  if (!gurl)
    return 1;

  /* Attempt to create a peer named by the URL.  If it's local and the
     port isn't in use, it will succeed.  Otherwise, it will fail and
     we will attempt to join the group. */
  iface = gnet_inetaddr_new (gurl->hostname, gurl->port? gurl->port: BTP_PORT);
  if (!iface)
    return 1;
  bpeer = b_peer_new (gurl->hostname, iface, TRUE);
  gnet_inetaddr_delete (iface);

  /* Create */
  if (bpeer)
    {
      btp = btp_create (bpeer, gurl->resource);
    }
  /* Join */
  else
    {
      gchar*     my_hostname;
      GInetAddr* my_iface;
      gchar*     my_port_str;
      gint       my_port;
      gboolean	 my_force_port;

      /* Get my interface */
      my_iface = gnet_inetaddr_autodetect_internet_interface ();
      if (!my_iface)
	{
	  fprintf (stderr, "btpcat: Could not find Internet interface\n");
	  return 1;
	}

      /* Get my hostname */
      my_hostname = g_getenv ("BTP_DOMAINNAME");
      if (!my_hostname)
	my_hostname = g_getenv ("DOMAINNAME");
      if (!my_hostname)
	{
	  my_hostname = gnet_inetaddr_get_name (my_iface);
	  if (!my_hostname || !gnet_inetaddr_is_internet_domainname (my_hostname))
	    {
	      g_free (my_hostname);
	      my_hostname = gnet_inetaddr_get_canonical_name (my_iface);
	    }
	}
      if (!my_hostname)
	{
	  gnet_inetaddr_delete (my_iface);
	  fprintf (stderr, "btpcat: Could not get hostname\n");
	  return 1;
	}

      /* Get my port */
      my_port = 0;
      my_force_port = FALSE;
      my_port_str = g_getenv ("BTP_PORT");
      if (my_port_str && (my_port = atoi(my_port_str)) != 0)
	{
	  gnet_inetaddr_set_port (my_iface, my_port);
	  my_force_port = TRUE;
	}

      /* Create bpeer again.  Use any port. */
      bpeer = b_peer_new (my_hostname, my_iface, my_force_port);
      g_free (my_hostname);
      gnet_inetaddr_delete (my_iface);
      if (!bpeer)
	{
	  fprintf (stderr, "btpcat: Could not create peer\n");
	  return 1;
	}

      btp = btp_join (bpeer, gurl);
    }

  gnet_url_delete (gurl);
  if (btp == NULL)
    return 1;
  btp->packet_func = packet_func;
  btp->error_func =  error_func;

  return 0;
}


void
packet_func (Btp* btp, const void* buffer, guint length, gpointer user_data)
{
  if (!quiet && length)
    writen (STDOUT_FILENO, buffer, length);
}


void
error_func (Btp* btp, gpointer user_data)
{
  shutdown();
}


/* ******************** */

void
usage (char* prog, int exitval)
{
  fprintf (stderr, "Usage: %s [OPTION]... <URL>\n", prog);
  fprintf (stderr, "Banana Tree Protocol (BTP) utility\n");
  fprintf (stderr, "\t-d <value>        debug level\n");
  fprintf (stderr, "\t-h                display help\n");
  fprintf (stderr, "\t-l                turn loopback on\n");
  fprintf (stderr, "\t-q                quiet (no input/output)\n");
  exit (exitval);
}


static void
sig_int_cb (int ignore)
{
  /* Reset signal handler */
  signal (SIGINT, SIG_DFL);

  if (btp)
    {
      shutdown();
      return;
    }

  exit (EXIT_FAILURE);
}


static void
num_conns_zero_cb (void)
{
  exit (EXIT_SUCCESS);
}



static ssize_t
writen (int fd, const void* buf, size_t len)
{
  size_t nleft;
  ssize_t nwritten;
  char* ptr;

  nleft = len;
  ptr = (char*) buf;
  while (nleft > 0)
    {
      if ((nwritten = write(fd, ptr, nleft)) < 0)
	{
	  if (errno == EINTR)
	    nwritten = 0;
	  else
	    return -1;
	}
      
      nleft -= nwritten;
      ptr += nwritten;
    }

  return len;
}
