# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann


"""
Test whether the update_checker works fine
"""

from elisa.core.utils.update_checker import UpdateChecker, UPDATES_URL
from elisa.core import common

from twisted.trial.unittest import TestCase
from twisted.internet import defer, reactor


class FakeHttpClient(object):
    def __init__(self):
        self.requests = []
        self.responses = []

    def request(self, request):
        self.requests.append(request)
        return defer.succeed(self.responses.pop(0))


class PluginRegistryMock(object):
    def update_cache(self):
        return defer.succeed(None)

class ConfigMock(object):
    def get_option(self, *args, **kw):
        return False

class ApplicationMock(object):
    plugin_registry = PluginRegistryMock()
    config = ConfigMock()


class TestUpdateChecker(TestCase):

    install_date = 'test_date'
    user_id = 'superid'
    version = '0.5'
    
    def setUp(self):
        self._patch_application()
        self.checker = UpdateChecker(self.install_date, self.user_id,
                self.version)
        self.http_client = FakeHttpClient()

        # we overwrite the read_full method to adapt to our request answers of
        # the fake client
        self.checker.http_client = self.http_client
        self.checker._read_full = self._read_full

    def tearDown(self):
        self.checker.stop()
        self._unpatch_application()

    def _patch_application(self):
        self._old_application = common.application
        common.application = ApplicationMock()

    def _unpatch_application(self):
        common.application = self._old_application

    def _read_full(self, response):
        return response

    def test_parsing(self):
        
        data = "user_id:192834\nversion : 0.0.2\n installer_url:http://example.com"

        expected_result = {'user_id' : '192834', 'version' : '0.0.2',
                           'installer_url' : 'http://example.com'}

        result = self.checker.parse_result(data)
        self.assertEquals(expected_result, result)

    def test_makes_right_request(self):

        # we fake the result
        self.checker.operating_system = "linux"

        url = UPDATES_URL % {'os' : 'linux', 'date': self.install_date,
                             'user_id' : self.user_id, 'version' : self.version}

        self.http_client.responses.append('')

        def check(result, url):
            self.assertEquals(len(self.http_client.requests), 1)
            self.assertEquals(self.http_client.requests[0], url)

        dfr = self.checker.request()
        dfr.addCallback(check, url)
        return dfr

    def test_stop_cleans_up(self):
        # this is a bit luggish but we do a start request, wait for the callback
        # (aka we have a first result) and return. Now tearDown will call stop.
        # if the reactor is left in an unclean state we have an issue.

        # response that we will get
        self.http_client.responses.append('foo:bar')

        dfr = defer.Deferred()
        def callbacker(result):
            dfr.callback(result)

        self.checker.start(callbacker)
        return dfr

    # this is not allowed to pend a long time
    test_stop_cleans_up.timeout = 4

    def test_start_calls_a_loop(self):
        # response that we will get
        self.http_client.responses.append('foo:bar')
        self.http_client.responses.append('foo:bar')
        self.http_client.responses.append('foo:bar')

        self.counter = 0

        dfr = defer.Deferred()
        def callbacker(result):
            self.counter += 1
            if self.counter == 3:
                reactor.callLater(0.1, dfr.callback, result)

        # we make it run faster
        self.checker.check_interval = 1
        self.checker.start(callbacker)
        return dfr

    test_start_calls_a_loop.timeout = 4
