# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

__maintainer__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.core import plugin_registry

from pypgmtools.widgets.slideshow import Slideshow
from pypgmtools.widgets.player_osd import Dock

import pgm

SlideshowViewClass = plugin_registry.get_component_class('base:slideshow_view')

class SlideshowView(SlideshowViewClass):

    supported_controllers = ('base:slideshow_controller')

    SlideshowWidgetClass = Slideshow

    def __init__(self):
        SlideshowViewClass.__init__(self)

        # delay before the osd disappears automatically
        self._osd_duration = 3

        # position of the mouse when the dragging started
        self._drag_start_position = None


    def frontend_changed(self, previous_frontend, new_frontend):
        if new_frontend == None:
            return

        SlideshowViewClass.frontend_changed(self, previous_frontend, new_frontend)

        canvas = self.frontend.context.viewport_handle.get_canvas()

        # create go back button
        iw, ih = canvas.width, canvas.height
        back_size = ih * 0.10
        offset = ih * 0.02

        back_button = self.frontend.theme.get_media("back_button")
        self.back_button = Dock(canvas,
                                pgm.DRAWABLE_NEAR,
                                back_size, back_size,
                                back_button)
        self.back_button.position = (iw - back_size - offset, offset, 0)
        self.back_button.opacity = 0
        self.back_button.visible = True

        self.back_button.connect("clicked", self.back_button_clicked)


    def attribute_set(self, key, old_value, new_value):
        SlideshowViewClass.attribute_set(self, key, old_value, new_value)

        if key == 'focused':
             if new_value == True and self.frontend.context.touchscreen == True:
                 self.osd_show(self._osd_duration)

    def osd_show(self, time_visible=-1):
        self.back_button.show(time_visible)

    def back_button_clicked(self, drawable, x, y, z, button, time):
        if not self.controller.focused:
            return False
        self.back_button.hide()
        self.controller.parent.toggle_menu()
        return True

    def drag_begin(self, drawable, x, y, z, button, time):
        if not self.controller.focused:
            return False

        self._drag_start_position = (x, y)
        self.slideshow_widget.move(0.0)
        self.osd_show(-1)
        return True
           
    def drag_end(self, drawable, x, y, z, button, time):
        if self._drag_start_position != None:
            dx = self._drag_start_position[0] - x
            dy = self._drag_start_position[1] - y

            if dx > 0.3:
                if not self.controller.next_image():
                    self.slideshow_widget.release()
            elif dx < -0.3:
                if not self.controller.previous_image():
                    self.slideshow_widget.release()
            else:
                self.slideshow_widget.release()
            
            self._drag_start_position = None
            
            self.osd_show(self._osd_duration)
            
            return True

        return False

    def drag_motion(self, drawable, x, y, z, button, time):
        if self._drag_start_position != None:
            dx = x - self._drag_start_position[0]
            self.slideshow_widget.move(dx)
            return True

        return False

    def drag_clicked(self, drawable, x, y, z, button, time):
        if self.slideshow_widget.visible == False or self.slideshow_widget.opacity == 0.0:
            return False

        if not self.controller.focused:
            if len(self.controller.model.playlist) == 0:
                return False
            else:
                self.controller.parent.toggle_menu()
                return True

        if self._drag_start_position != None:
            dx = self._drag_start_position[0] - x
            if dx < -0.05 or dx > 0.05:
                return True
        
        self.osd_show(self._osd_duration)
        return True
