# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


import pygst
pygst.require('0.10')
import gst
from threading import Event

__maintainer__ = "Benjamin Kampmann <benjamin@fluendo.com>"

from elisa.core.utils.deferred_action import DeferredActionsManager

class MimeGetter:
    """
    Class for getting the type and mime info of a file.
    """

    def __init__(self):

        self._actions_manager = DeferredActionsManager()
        self._lock = Event()
        self._set_up_pipeline()

    def _set_up_pipeline(self, source='filesrc'):
        self._count = 0
        self._src = gst.element_factory_make(source)
        self._src.set_property("blocksize", 550000)

        self._dbin = gst.element_factory_make("decodebin")

        self._type = self._dbin.get_by_name("typefind")
        self._fakesink = []


        self._pipeline = gst.Pipeline('Typefinder')
        self._pipeline.add(self._src)
        self._pipeline.add(self._dbin)

        self._src.link(self._dbin)

        # connect to bus message
        self._pipeline.get_bus().connect("message", self._event_cb)
        self._pipeline.get_bus().add_signal_watch()

        # connect to typefind have-type signal
        self._type.connect("have-type", self._got_type)
        self._dbin.connect("new-decoded-pad", self._new_decoded_pad_cb)
        self._dbin.connect("unknown-type", self._unknown_type_cb)


    def get_type(self, source, location):
        """
        Return a defer which will look up the type of a file.  This defer might
        needs a lot of time, because we are currently only doing one lookup at
        a time and the other ones are put into a queue.

        The callback-result of the L{Deferred} then is a dictionary containing
        this:

         - file_type: mapped to a string, which is one of 'video', 'audio' or
         'image' or '' (if none of these three)
         - mime_type: mapped to a string, containing the mime_type

        @param source:          the name of the gstreamer-source element to use
                                for e.g. 'filesrc' or 'gnomevfssrc'
        @type source:           String

        @param location:        to which location should the source be set to?
        @type location:         String

        @rtype:                 L{Deferred}
        """

        ## We are enqueueing the action, because we don't won't to have more
        ## than one pipeline
        
        return self._actions_manager.enqueue_action(self._set_pipeline,
                                                    source, location)

    def _set_pipeline(self, source, location):
        ## We stop the pipeline
        self._pipeline.set_state(gst.STATE_NULL)
        self._pipeline.get_state()
        for sink in self._fakesink:
            self._dbin.unlink(sink)
            self._pipeline.remove(sink)
            sink.set_state(gst.STATE_NULL)

        if self._count >= 500:
            self._src.unlink(self._dbin)
            self._pipeline.remove(self._src, self._dbin)
            self._set_up_pipeline(source)

        self._count += 1
        self._fakesink = []
        self.is_video = False
        self.is_audio = False
        self._mime = None

        # If our existing source is different, we have to set a new one
        if self._src.get_factory().get_name() != source:
            # TODO: Do hard test of this part!
            #   - what if there are two different source are used alternating?
            #   - do we have the 'too much threads' problem again?

            # recreate the source element
            self._src.unlink(self._dbin)
            self._pipeline.remove(self._src)
            self._src = gst.element_factory_make(source)
            self._pipeline.add(self._src)
            self._src.link(self._dbin)
        
        # Set the location to the given one. Because set_property does not
        # handle unicode correctly!
        self._src.set_property("location", location)


        ### If the starting of playing went good, we'll wait and return the
        ### value of the lookup
        self._pipeline.set_state(gst.STATE_PLAYING)
        self._pipeline.get_state()
        self._lock.wait()

        if self.is_video:
            if self._mime.find('image') == -1:
                return {'file_type' : 'video', 'mime_type' : self._mime}

            return {'file_type' : 'image', 'mime_type' : self._mime}

        elif self.is_audio:
                return {'file_type' : 'audio', 'mime_type' : self._mime}

        return {'file_type' : '', 'mime_type' : self._mime}


    def _unknown_type_cb(self, dbin, pad, caps):
        self._lock.set() 

    def _notify_caps_cb(self, pad, args):
        self._lock.set()

    def _new_decoded_pad_cb(self, dbin, pad, is_last):
        # Does the file contain got audio or video ?
        caps = pad.get_caps()
        
        if "audio" in caps.to_string():
            self.is_audio = True
        elif "video" in caps.to_string():
            self.is_video = True
        else:
            return

        # we connect a fakesink to the new pad
        fakesink = gst.element_factory_make("fakesink")

        self._fakesink.append(fakesink)

        self._pipeline.add(fakesink)
        self._dbin.link(fakesink)
        sinkpad = fakesink.get_pad("sink")

        # and connect the callback to notify caps
        sinkpad.connect("notify::caps", self._notify_caps_cb)
        fakesink.set_state(gst.STATE_PLAYING)


    def _got_type(self, typefind, arg1, mime_type):
        self._mime = str(mime_type)

    def _event_cb(self, bus, msg):
        # FIXME
        t = msg.type
        if t == gst.MESSAGE_EOS:
            #print "Received EOS"
            pass
        elif t == gst.MESSAGE_ERROR:
            e, d = msg.parse_error()
            #print "ERROR:",e
            self._got_type(None, None, "")
        return True

if __name__ == '__main__':

    try:
        from twisted.internet import glib2reactor
        glib2reactor.install()
    except AssertionError:
        pass
    
    from twisted.internet import defer, threads, reactor
    from elisa.core.media_uri import MediaUri,quote, unquote

    import os

    #base_path = '/media/nfs/medias/videos/'
    base_path = '/home/ben/gr'
#    base_path = '/opt/elisa/sample_data/music'

    m = MimeGetter()
    list = []

    def got_type(mime, path, ind):
        list.append((path, mime))

    def stopped(arg, ind):
        global toto
        timer = time.time() - toto
        print list
        print "lookup of %s files took %s" % (ind, timer)

    ind = 1
    dfr = defer.Deferred()
    def walk_path(path):
            global ind, dfr
            
            for f in os.listdir(path): 
                loc = '%s/%s' % (path, f)
                if os.path.isdir(loc):
                    walk_path(loc)
                else:
                    dfr = m.get_type('filesrc', loc)
                    dfr.addCallback(got_type, f, ind)
                    ind = ind + 1
                    

    import time
    toto = time.time()
    walk_path(base_path)
    ### This is UGLY!
    dfr.addCallback(stopped, ind)

    reactor.run()
