# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'
__maintainer2__ = 'Benjamin Kampmann <benjamin@fluendo.com>'

from elisa.core import common
from elisa.core.utils import deferred_action

from elisa.extern.log import log

from twisted.internet import defer, threads
from twisted.python.failure import Failure

import PIL
from PIL import PngImagePlugin
import StringIO

import gst

import gobject
from mutex import mutex
import sys, os
import time
import md5 #FIXME: deprecated in 2.5, but hashlib is not available on 2.4
import Image, ImageStat

BORING_IMAGE_VARIANCE=2000

HOLES_SIZE= (9, 35)
HOLES_DATA='\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x9a\x9a\x9a\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x91\x91\x91\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x91\x91\x91\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x91\x91\x91\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x9a\x9a\x9a\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x91\x91\x91\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x91\x91\x91\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x91\x91\x91\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x9a\x9a\x9a\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x91\x91\x91\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x91\x91\x91\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x91\x91\x91\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x9a\x9a\x9a\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x91\x91\x91\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x91\x91\x91\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x91\x91\x91\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x9a\x9a\x9a\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x91\x91\x91\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x91\x91\x91\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\xff\xff\xff\xa6\x91\x91\x91\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6\x00\x00\x00\xa6'

class ThumbnailerError(Exception):

    def __init__(self, uri, error_message=None):
        if error_message == None:
            output = "Failed thumbnailing %r" % uri
        else:
            output = "Failed thumbnailing %r: %r" % (uri, error_message)

        Exception.__init__(self, output)
        self.uri = uri
        self.error_message = error_message

class NoThumbnailerFound(Exception):
    pass

class VideoSinkBin(gst.Bin):

    def __init__(self, needed_caps):
        self.reset()
        gst.Bin.__init__(self)
        self._capsfilter = gst.element_factory_make('capsfilter', 'capsfilter')

        self.set_caps(needed_caps)
        self.add(self._capsfilter)

        fakesink = gst.element_factory_make('fakesink', 'fakesink')
        fakesink.set_property("sync", False)
        self.add(fakesink)
        self._capsfilter.link(fakesink)

        pad = self._capsfilter.get_pad("sink")
        ghostpad = gst.GhostPad("sink", pad)

        pad2probe = fakesink.get_pad("sink")
        pad2probe.add_buffer_probe(self.buffer_probe)

        self.add_pad(ghostpad)
        self.sink = self._capsfilter

    def set_current_frame(self, value):
        self._current_frame = value

    def set_caps(self, caps):
        gst_caps = gst.caps_from_string(caps)
        self._capsfilter.set_property("caps", gst_caps)

    def get_current_frame(self):
        frame = self._current_frame
        self._current_frame = None
        return frame

    def buffer_probe(self, pad, buffer):
        caps = buffer.caps
        if caps != None:
            s = caps[0]
            self.width = s['width']
            self.height = s['height']
        if self.width != None and self.height != None and buffer != None:
            self.set_current_frame(buffer.data)
        return True

    def reset(self):
        self.width = None
        self.height = None
        self.set_current_frame(None)


gobject.type_register(VideoSinkBin)

from threading import Event

class VideoThumbnailer(log.Loggable):

    logCategory = "thumbnailer"

    def __init__(self):
        self._pipeline = gst.element_factory_make('playbin', 'playbin')
        caps = "video/x-raw-rgb,bpp=24,depth=24"

        self._sink = VideoSinkBin(caps)
        self._blocker = Event()

        self._pipeline.set_property("video-sink", self._sink)
        self._pipeline.set_property('volume', 0)


    def get_holes_img(self):
        img = Image.fromstring('RGBA', HOLES_SIZE, HOLES_DATA)
        return img

    def add_holes(self, img):
        holes = self.get_holes_img()
        holes_h = holes.size[1]
        remain = img.size[1] % holes_h

        i = 0
        nbands = 0
        while i < (img.size[1] - remain):
            left_box = (0, i, holes.size[0], (nbands+1) * holes.size[1])
            img.paste(holes, left_box)

            right_box = (img.size[0] - holes.size[0], i,
                         img.size[0], (nbands+1) * holes.size[1])
            img.paste(holes, right_box)

            i += holes_h
            nbands += 1

        remain_holes = holes.crop((0, 0, holes.size[0], remain))
        remain_holes.load()
        img.paste(remain_holes, (0, i, holes.size[0], img.size[1]))
        img.paste(remain_holes, (img.size[0] - holes.size[0], i,
                                 img.size[0], img.size[1]))
        return img

    def interesting_image(self, img):
        stat = ImageStat.Stat(img)
        return True in [ i > BORING_IMAGE_VARIANCE for i in stat.var ]

    def set_state_blocking(self, pipeline, state):
        status = pipeline.set_state(state)
        if status == gst.STATE_CHANGE_ASYNC:
            self.debug("Waiting for state change completion to %s..." % state)

            result = [False]
            max_try = 100
            nb_try = 0
            while not result[0] == gst.STATE_CHANGE_SUCCESS:
                if nb_try > max_try:
                    self.debug("State change failed: %s" % result[0])
                    return False
                nb_try += 1
                result = pipeline.get_state(50*gst.MSECOND)

            self.debug("State change completed.")
            return True
        elif status == gst.STATE_CHANGE_SUCCESS:
            self.debug("State change completed.")
            return True
        else:
            self.debug("State change failed")
            return False


    def generate_thumbnail(self, video_uri, size):
        """
        Try to generate a thumbnail for the video located at video_uri,
        of size width.

        @param video_uri: URI to make a thumbnail from
        @type video_uri:  L{elisa.core.media_uri.MediaUri}
        @param size:      size of the thumbnail in pixels
        @type size:       int

        @raise ThumbnailerError: if an error occurs when generating the thumbnail
        """

        self.set_state_blocking(self._pipeline, gst.STATE_NULL)
        self.debug("Generating thumbnail for file: %s" % video_uri)
        self._pipeline.set_property('uri', video_uri)

        """
        def bus_event(bus, message, pipeline):
            t = message.type
            if t == gst.MESSAGE_EOS:
                self.debug("End of Stream")
                pass
            elif t == gst.MESSAGE_ERROR:
                err, debug = message.parse_error()
                self.debug("Error: %s %s" % (err, debug))
                self.set_state_blocking(pipeline, gst.STATE_NULL)
            return True

        self._pipeline.get_bus().add_watch(bus_event, self._pipeline)
        """

        # start the pipeline
        if not self.set_state_blocking(self._pipeline, gst.STATE_PAUSED):
            self.debug("Cannot start the pipeline")
            self.set_state_blocking(self._pipeline, gst.STATE_NULL)
            raise ThumbnailerError(video_uri)

        if self._sink.width == None or self._sink.height == None:
            self.debug("Cannot determine media size")
            self.set_state_blocking(self._pipeline, gst.STATE_NULL)
            raise ThumbnailerError(video_uri)
        sink_size = (self._sink.width, self._sink.height)

        self.debug("width: %s; height: %s" % (self._sink.width, self._sink.height))
        try:
            duration, format = self._pipeline.query_duration(gst.FORMAT_TIME)
        except Exception, e:
        ## FIXME: precise this exception
            self.debug("Gstreamer cannot determine the media duration."
                         " using playing-thumbnailing for %s"  % video_uri)
            self.set_state_blocking(self._pipeline, gst.STATE_NULL)
            img = self._play_for_thumb(sink_size, size, 0)
            self.debug("play found %s" % img)
            if img:
                return img
        else:
            duration /= gst.NSECOND
            self.debug("duration: %s" % duration)
            try:
               img = self._seek_for_thumb(video_uri, duration, sink_size, size) 
               self.debug("seek found %s" % img)
               if img:
                   return img
            except ThumbnailerError, e:
                # Fallback: No Image found in seek_for
                self.debug("Using Fallback: play_for_thumb")
                self.set_state_blocking(self._pipeline, gst.STATE_NULL)
                img = self._play_for_thumb(sink_size, size, duration)
                self.debug("Fallback-Play found %s" % img)
                if img:
                    return img
        # stop the pipeline
        self.set_state_blocking(self._pipeline, gst.STATE_NULL)
        raise ThumbnailerError(video_uri)


    def _play_for_thumb(self, sink_size, size, duration=0):
        ## Maybe we should set a timer, so that we don't play the whole movie?
        self.debug("Doing play_for_thumb!")
        self.debug(duration)
        id = None
        self._img = None

        if duration >= 250000:
            self._every = 25
        elif duration >= 200000:
            self._every = 15
        elif duration >= 10000:
            self._every = 10
        elif duration >= 5000:
            self._every = 5
        else:
            self._every = 1

        self.debug("Setting every-frame to %s" % self._every)

        self._every_co = self._every

        ## How often Proceed?
        self._counter = 5
        
        def buffer_probe(pad, buffer):
            ## Proceed only every 5th frame!
            if self._every_co < self._every:
                self._every_co += 1
                return
            self._every_co = 0
            self.debug("Proceeding a Frame")

            try: 
                img = Image.frombuffer("RGB", sink_size, buffer,
                                                "raw", "RGB",0, 1)
            except Exception, e:
                self.debug("Invalid frame")
            else:
                self.debug("Found Frame")
                self._img = img
                if self.interesting_image(self._img):
                    self.debug("Intresting image found")

                    self._img.thumbnail((size, size), Image.BILINEAR)
                    if img.mode != 'RGBA':
                        img = img.convert(mode='RGBA')

                    self.debug("releasing %s" % self._img)
                    self._sink.reset()
                    pad.remove_buffer_probe(id)
                    self._blocker.set()
                    return

            self._counter -= 1
            if self._counter <= 0:
                self.debug("Counter off, resetting blocker!")
                # Is it better to return no image instead of a 'boring' one?
                if self._img:
                    self._img.thumbnail((size, size), Image.BILINEAR)
                    if img.mode != 'RGBA':
                        img = img.convert(mode='RGBA')

#                self._img = None
                self._sink.reset()
                pad.remove_buffer_probe(id)
                self._blocker.set()


        self.debug("Setting Pipeline")
        self.set_state_blocking(self._pipeline, gst.STATE_PLAYING)
#        self._pipeline.set_state(gst.STATE_PLAYING)
        pad = self._sink.get_pad('sink')
        id = pad.add_buffer_probe(buffer_probe)
        self.debug("Wait")
        self._blocker.wait()
        self.debug("Going on")
        self._pipeline.set_state(gst.STATE_NULL)
        self.debug("returning %s" % self._img)
        return self._img

    def _seek_for_thumb(self, video_uri, duration, sink_size, size):
        frame_locations = [ 1.0 / 3.0, 2.0 / 3.0, 0.1, 0.9, 0.5 ]
        self.debug('seeking')

        for location in frame_locations:
            abs_location = int(location * duration)
            self.debug("location: rel %s, abs %s" % (location, abs_location))

            if abs_location == 0:
                raise ThumbnailerError(video_uri, "Empty media")
            
            event = self._pipeline.seek(1.0, gst.FORMAT_TIME,
                                        gst.SEEK_FLAG_FLUSH | gst.SEEK_FLAG_KEY_UNIT,
                                        gst.SEEK_TYPE_SET, abs_location,
                                        gst.SEEK_TYPE_NONE, 0)
            if not event:
                raise ThumbnailerError(video_uri, "Not Seekable")

            if not self.set_state_blocking(self._pipeline, gst.STATE_PAUSED):
                raise ThumbnailerError(video_uri, "Not Pausable")

            frame = self._sink.get_current_frame()

            try:
                img = Image.frombuffer("RGB", sink_size, frame, "raw", "RGB", 0, 1)
            except:
                self.debug("Invalid frame")
                continue

            if self.interesting_image(img):
                self.debug("Interesting image found")
                break
            else:
                self.debug("Image not interesting")
                pass

        self._sink.reset()

        if img:
            img.thumbnail((size, size), Image.BILINEAR)
            if img.mode != 'RGBA':
                img = img.convert(mode='RGBA')
            self.set_state_blocking(self._pipeline, gst.STATE_NULL)
            return img
            

class Thumbnailer(deferred_action.DeferredActionsManager):

    def __init__(self, thumbnail_dir=None):
        """
        
        @param thumbnail_dir: abolute path to a directory where to store thumbnails
        @type thumbnail_dir:  string
        """
        deferred_action.DeferredActionsManager.__init__(self)

        if not thumbnail_dir:
            thumbnail_dir = os.path.join(os.path.expanduser("~"), ".thumbnails")
        self._thumbnail_dir = thumbnail_dir

        self._video_thumbnailer = VideoThumbnailer()


    def _save_thumbnail_as(self, file_uri, thumbnail, thumbnail_filename):
        """
        Save a thumbnail of an URI at a given location.

        @param file_uri:            URI the thumbnail was generated from
        @type file_uri:             L{elisa.core.media_uri.MediaUri}
        @param thumbnail:           Thumbnail instance
        @type thumbnail:            L{PIL.Image}
        @param thumbnail_filename:  filename to save to
        @type thumbnail_filename:   string
        @raises ThumbnailerError:   if the thumbnail couldn't be saved
        """

        # if the thumbnail directory does not exist yet, create it
        directory = os.path.dirname(thumbnail_filename)
        if not os.path.exists(directory):
            try:
                os.makedirs(directory, 0700)
            except OSError, e:
                msg = "Could not make directory %r: %s. Thumbnail not saved." % (directory, e)
                self.warning(msg)
                raise ThumbnailerError(file_uri, msg)
                
        info = PngImagePlugin.PngInfo()

        # required metadata
        info.add_text("Thumb::URI", str(file_uri))
        # FIXME: requires network transparent mtime
        #info.add_text("Thumb::MTime", os.path.getmtime(file_uri))

        # TODO: think about some useful (but optional) metadata to add
        # cf: http://jens.triq.net/thumbnailnail-spec/creation.html

        # FIXME: access permissions on thumbnail must be set to 600
        thumbnail.save(thumbnail_filename, "png", pnginfo=info)

    def _get_thumbnail_location(self, file_uri, size):
        """
        Return the thumbnail's location of a file for a particular size.

        @param file_uri:       URI of the file
        @type file_uri:        L{elisa.core.media_uri.MediaUri}
        @param size:           Size of the thumbnail (in pixels)
        @param type:           int
        @rtype:                tuple (path to thumbnail, maximum thumbnail size)

        @raise Exception:      Raise an exception if size is superior to 512
        """
        # FIXME: if file_uri refers to ~/.thumbnails/*, return str(file_uri)

        if size <= 128:
            thumbnail_dir_size = "normal"
            thumbnail_max_size = 128
        elif size <= 256:
            thumbnail_dir_size = "large"
            thumbnail_max_size = 256
        elif size <= 512:
            # NOTE: this is not supported by the freedesktop specification
            thumbnail_dir_size = "extra_large"
            thumbnail_max_size = 512
        else:
            raise Exception("ThumbnailManager._get_thumbnail_location(): size \
                             given too large: %d" % size)

        thumbnail_filename = md5.new(str(file_uri)).hexdigest() + ".png"

        return (os.path.join(self._thumbnail_dir, thumbnail_dir_size,
                             thumbnail_filename),
                thumbnail_max_size)


    def _get_thumbnail_from(self, file_uri, size):
        """
        Retrieve a thumbnail from a given URI.

        @param file_uri:       URI to get the thumbnail from
        @param file_uri:       L{elisa.core.media_uri.MediaUri}
        @param size:           size of the thumbnail in pixels
        @type size:            int

        @rtype:                L{PIL.Image} or None

        @raise:                Exception raise if an error occurs while loading
        """
        thumbnail_path, size = self._get_thumbnail_location(file_uri, size)

        if os.path.exists(thumbnail_path):
            # the thumbnail file exists and is valid, just load it
            try:
                thumbnail = image.Image(thumbnail_path)
                return thumbnail
            except IOError, error:
                raise Exception("Error loading %s: %s" % (thumbnail_path, error))
        else:
            return None


    def _retrieve_thumbnail(self, uri, size, media_type):
        """
        @raise ThumbnailerError: if an error occurs when generating the
                                 thumbnail
        @raise NoThumbnailerFound: when it is not possible to generate the
                                   thumbnail
        """
        self.debug("Generating thumbnail for %s of %s type" % (uri, media_type))
        if media_type == 'image':
            media_manager = common.application.media_manager
            fd = media_manager.blocking_open(uri, 'r')
            if fd == None:
                raise ThumbnailerError(uri)

            data = StringIO.StringIO(fd.blocking_read())
            fd.close()

            img = Image.open(data)

            img.thumbnail((size, size), Image.ANTIALIAS)
        elif media_type == 'video':
            img = self._video_thumbnailer.generate_thumbnail(uri, size)
        else:
            raise NoThumbnailerFound("No thumbnailer available for type %s\
                                          (%s)" % (media_type, uri))

        if img == None:
            raise ThumbnailerError(uri)

        # Save the thumbnail
        thumbnail_filename, thumbnail_max_size = self._get_thumbnail_location(uri, size)
        self._save_thumbnail_as(uri, img, thumbnail_filename)

        return (thumbnail_filename, thumbnail_max_size)
  

    def get_thumbnail(self, uri, size, media_type):
        """
        Creates a thumbnail of uri on a local location.
        Returns a deferred, it's callback is called with
        a tuple containing the local filename of the thumbnail
        and its size as an int.
        Calls the deferred error callback if an error occured.

        @param uri:     URI to generate a thumbnail from
        @type uri:      L{elisa.core.media_uri.MediaUri}
        @param size:    size of the desired thumbnail
        @type size:     int
        @rtype:         L{twisted.internet.defer.Deferred}
        """
        self.debug("Requesting thumbnail for %s of %s type" \
                    % (uri, media_type))
        uri = common.application.media_manager.get_real_uri(uri)
        f, size = self._get_thumbnail_location(uri, size)

        # Create the thumbnail if not already available
        if not os.path.exists(f):
            return self.enqueue_action(self._retrieve_thumbnail,
                                       uri,
                                       size,
                                       media_type)
        
        self.debug("Returning already existing thumbnail for %s" % uri)
        
        dfr = defer.Deferred()
        dfr.callback((f,size))
        return dfr


    def add_thumbnail(self, uri, size, thumbnail):
        """
        Adds to the thumbnail cache a thumbnail created by
        a 3rd party component (like a View)

        @param uri:        URI the thumbnail has been generated from
        @type uri:         L{elisa.core.media_uri.MediaUri}
        @param size:       size of the thumbnail
        @type size:        int
        @param thumbnail:  the Image
        @type thumbnail:   L{PIL.Image}
        """

        if thumbnail:
            thumbnail_filename, thumbnail_max_size = self._get_thumbnail_location(uri, size)
            self._save_thumbnail_as(uri, thumbnail, thumbnail_filename)



if __name__ == "__main__":

    from elisa.core import media_uri
    from elisa.core import common

    common.boot()


    t = Thumbnailer()
    path = u'/home/haiku/tmp/plier.mpeg'
    size = 512
    media_type = 'video'
    uri = media_uri.MediaUri(path)

    def done(pic):

        if pic:
            print "thumbnail generated !"
        else:
            print "the thumbnail has NOT been generated"

    def error(msg):

        print "error occured:", msg.getTraceback()

    dfr = t.get_thumbnail(uri, size, media_type)
    dfr.addCallback(done)
    dfr.addErrback(error)

    try:
        from twisted.internet import glib2reactor
        glib2reactor.install()
    except AssertionError:
        # already installed...
        pass

    from twisted.internet import reactor

    reactor.run()
