# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Lionel Martin <lionel@fluendo.com>'


from elisa.core.utils import classinit, signal
from elisa.core import log
from elisa.base_components.theme import Theme
from elisa.core import common

from twisted.internet import defer


class Frontend(log.Loggable):
    """
    A frontend can be connected to a L{elisa.core.backend.Backend} in order to
    render its content. It does so by holding the root of a tree
    L{elisa.base_components.view.View}s that render the
    L{elisa.base_components.controller.Controller}s hold by the backend. The
    rendering is done in a context which is unique to a frontend.

    A L{elisa.base_components.theme.Theme} can be used in order to change the
    set of icons used by the views for their rendering.

    @ivar view:         root view associated to the frontend
    @type view:         L{elisa.base_components.view.View}
    @ivar context:      context in which the root view is rendered
    @type context:      L{elisa.base_components.context.Context}
    @ivar theme:        theme used by the frontend at rendering
    @type theme:        L{elisa.base_components.theme.Theme}

    @ivar languages:    the translation to use
    @type languages:    list

    DOCME: ivar name; see below.
    """

    __metaclass__ = classinit.ClassInitMeta
    __classinit__ = classinit.build_properties
    
    theme_changed = signal.Signal('theme_changed', Theme)

    # FIXME: what is name used for? It needs to be documented.
    name = None

    def __init__(self, context, view, theme):
        log.Loggable.__init__(self)
        self._view = view
        self._context = context
        self._theme = theme
        self._languages = []

    def view__get(self):
        return self._view

    def context__get(self):
        return self._context

    def theme__get(self):
        return self._theme

    def theme__set(self, theme):
        self._theme = theme
        self.theme_changed.emit(theme)

        # save theme path in config
        application = common.application
        application.config.set_option('theme', theme.path, section=self.name)
       
    def languages__get(self):
        return self._languages

    def languages__set(self, languages):
        self.debug("Setting languages to %r", languages)
        if isinstance(languages, list):
            self._languages = languages
        elif isinstance(languages, base_string):
            self._languages = [languages]
        else:
            self.warning("Could not set the language to '%s'. Please use a"
                         " list of strings instead!" % languages)

    def update(self):
        """
        Refresh the display by updating the views in their context.
        """
        if self._context:
            self._context.update()

    def clean(self):
        """
        Cleanup the frontend. Once done, no rendering will be possible.

        @rtype: L{twisted.internet.defer.Deferred}
        """
        # the cleanups have to be chained (in a sequence)
        dfr = defer.Deferred()
        dfr.addCallback(lambda x: self._view.clean())
        dfr.addCallback(lambda x: self._theme.clean())
        dfr.addCallback(lambda x: self._context.clean())
        dfr.callback(None)
        return dfr
