# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'
__maintainer2__ = 'Lionel Martin <lionel@fluendo.com>'


from elisa.core.component import Component, ComponentError
from elisa.base_components.controller import Controller
from elisa.core.observers.observer import Observer
from elisa.core import common


class ControllerNotSupported(Exception):
    pass

class View(Component, Observer):
    """
    A View is responsible for rendering a Model. It can be of various nature
    such as graphics, audio or even network output. A View does not hold a
    direct reference to the Model it renders but uses a Controller to access it.

    A View is also responsible for translating InputEvents into messages to the
    Controller and the underlying Model.

    A View is a local object that cannot be exposed on the network.

    DOCME:
     - supported_controllers
     - context_handle
     - frontend
     - parent
     - controller (controller that the view has to render)
    """

    supported_controllers = ()

    default_associations = (
    ('base:controller','base:view'),
    )

    # FIXME: why is that here and not in __init__ ?
    context_path = None

    def __init__(self):
        Component.__init__(self)
        Observer.__init__(self)

        self._controller = None
        self.context_handle = None
        self._frontend = None
        self.parent = None

    def frontend__get(self):
        return self._frontend

    def frontend__set(self, new_frontend):
        previous_frontend, self._frontend = self._frontend, new_frontend
        self.frontend_changed(previous_frontend, new_frontend)

    def frontend_changed(self, previous_frontend, new_frontend):
        """
        DOCME
        """
        pass

    def controller__get(self):
        return self._controller

    def controller__set(self, controller):  
        if self._controller == controller:
            return

        if controller == None:
            self.stop_observing()
        else:
            if not controller.path in self.supported_controllers:
                self.warning("View %r does not support controller %r" % (self.path, controller.path))
                raise ControllerNotSupported("View %r does not support controller %r" % (self.path, controller.path))
            self.observe(controller)

        self._controller = controller
        # FIXME: should pass old controller and new controller
        self.controller_changed()

    def controller_changed(self):
        """
        This method is called by the controller property when set.
        View can at this time securely initialize controller's
        properties.
        # FIXME: should take old controller and new controller as parameters
        """

    def create_child_view(self, controller):
        """
        DOCME

        @raise TypeError: if controller is not a Controller object
        """

        # FIXME: needs comments
        if isinstance(controller, Controller):
            view_path_list = self.get_view_paths(controller.path)
            if len(view_path_list) == 0:
                self.warning("Cannot create view for controller %r (with view %r)" % (controller.path, self.path))
                raise ComponentError(controller.path)
            else:
                plugin_registry = common.application.plugin_registry
                view = plugin_registry.create_component(view_path_list[0])
                view.parent = self
                view.frontend = self.frontend
                self.child_view_creating(view, controller)
                view.controller = controller
                self.child_view_created(view)
                return view
        else:
            self.warning("Cannot create view for controller %r (with view %r)" % (controller.path, self.path))
            raise TypeError


    def child_view_creating(self, view, controller):
        """
        called when a child view is being created before its controller is set

        DOCME
        """

    def child_view_created(self, view):
        """
        called when a child view has been created

        DOCME
        """

    def get_view_paths(self, controller_path):
        """
        DOCME
        """
        view_paths = []
        for association in self.default_associations:
            if association[0] == controller_path:
                view_paths.append(association[1])
        return view_paths

    def update(self):
        """
        Refresh completely the rendering of the associated Controller. This should
        be called when the Controller has changed.
        """
