# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.theme import Theme
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.poblesec.poblesec_browser_controller import PoblesecBrowserController
from elisa.plugins.poblesec.widgets.button import StateButton
from elisa.core import common

import pgm
from twisted.internet import reactor, defer

class AutohideButtons(Widget):
    max_opacity = 255
    min_opacity = 0

    def __init__(self, frontend):
        super(AutohideButtons, self).__init__()
        self._frontend = frontend
        self._buttons = []
        self.time_before_hiding = 3.0
        self._hiding_dfr = None

    def _create_button(self, unfocused_resource, focused_resource):
        button = StateButton(unfocused_resource, focused_resource)
        button.bg_focused.bg_a = 0
        button.bg_unfocused.bg_a = 0
        self.add(button, pgm.DRAWABLE_NEAR)
        self._buttons.append(button)
        button.visible = True
        button.text.visible = False
        return button

    def hide(self):
        if self._hiding_dfr and self._hiding_dfr.active():
            self._hiding_dfr.cancel()
        for button in self._buttons:
            button.opacity = self.min_opacity

    def show(self):
        if self._hiding_dfr and self._hiding_dfr.active():
            self._hiding_dfr.cancel()

        for button in self._buttons:
            button.opacity = self.max_opacity

        if self.time_before_hiding > 0:
            self._hiding_dfr = reactor.callLater(self.time_before_hiding, self.hide)

    def update_style_properties(self, props=None):
        if props is None:
            return

        remaining_props = {}

        for key, value in props.iteritems():
            if key == 'buttons-bg_a':
                for button in self._buttons:
                    button.bg_a = value
            else:
                remaining_props[key] = value

        if self.time_before_hiding == -1:
            self.show()

        if len(remaining_props) > 0:
            return super(AutohideButtons, self).update_style_properties(remaining_props)


class HomeButton(AutohideButtons):
    max_opacity = 255
    min_opacity = 80

    def __init__(self, frontend):
        super(HomeButton, self).__init__(frontend)

        self.home_button = Image()
        self.add(self.home_button)
        self._buttons.append(self.home_button)
        self.home_button.layout = pgm.IMAGE_SCALED
        self.home_button.bg_a = 0
        self.home_button.fg_a = self.min_opacity
        self.home_button.opacity = 255
        self.home_button.visible = True
        self._clicked_handler_id = self.home_button.connect('clicked',
                                                            self._home_clicked)
        self._focused_entered_id = self.home_button.connect('entered',
                                                            self._hover_cb,
                                                            True)
        self._focused_left_id = self.home_button.connect('left',
                                                         self._hover_cb,
                                                         False)

        self.hide()


    def clean(self):
        self.home_button.disconnect(self._clicked_handler_id)
        self.home_button.disconnect(self._focused_entered_id)
        self.home_button.disconnect(self._focused_left_id)
        super(HomeButton, self).clean()

    def _hover_cb(self, button, x, y, z, time, hover):
        if hover:
            self.home_button.fg_a = self.max_opacity
        else:
            self.home_button.fg_a = self.min_opacity

    def _home_clicked(self, drawable, x, y, z, button, time, pressure):
        self._frontend.viewport.hide()

    def update_style_properties(self, props=None):
        if props is None:
            return

        theme = Theme.get_default()

        remaining_props = {}

        for key, value in props.iteritems():
            if key == 'button-height':
                width = props.get('button-width', self.home_button.width)
                self.home_button.width, self.home_button.height = width, value
                self.home_button.x = 0.
                self.home_button.y = 0.
            elif key == 'button-width':
                continue
            elif key == 'home-button':
                self.home_button.set_from_file(theme.get_resource(value))
            else:
                remaining_props[key] = value

        if len(remaining_props) > 0:
            return super(HomeButton, self).update_style_properties(remaining_props)


class MinMaxCloseButtons(AutohideButtons):

    def __init__(self, frontend):
        super(MinMaxCloseButtons, self).__init__(frontend)

        self.minimize_button = self._create_button("elisa.plugins.poblesec.viewport_min_button_normal",
                                                   "elisa.plugins.poblesec.viewport_min_button_hover")
        self.maximize_button = self._create_button("elisa.plugins.poblesec.viewport_max_button_normal",
                                                   "elisa.plugins.poblesec.viewport_max_button_hover")
        self.close_button = self._create_button("elisa.plugins.poblesec.viewport_close_button_normal",
                                                "elisa.plugins.poblesec.viewport_close_button_hover")

        self._close_handler_id = self.close_button.connect('clicked',
                                                           self._close_clicked)
        self._max_handler_id = self.maximize_button.connect('clicked',
                                                            self._maximize_clicked)
        self._min_handler_id = self.minimize_button.connect('clicked',
                                                            self._minimize_clicked)


    def clean(self):
        self.minimize_button.disconnect(self._min_handler_id)
        self.maximize_button.disconnect(self._max_handler_id)
        self.close_button.disconnect(self._close_handler_id)
        super(MinMaxCloseButtons, self).clean()

    def _close_clicked(self, drawable, x, y, z, button, time, pressure):
        common.application.stop()

    def _maximize_clicked(self, drawable, x, y, z, button, time, pressure):
        # toggle fullscreen state of the viewport
        self._frontend.viewport.fullscreen = not self._frontend.viewport.fullscreen

    def _minimize_clicked(self, drawable, x, y, z, button, time, pressure):
        # Cope with old versions of Pigment
        if hasattr(self._frontend.viewport, 'iconified'):
            self._frontend.viewport.iconified = True

    def update_style_properties(self, props=None):
        if props is None:
            return

        theme = Theme.get_default()

        remaining_props = {}

        for key, value in props.iteritems():
            if key == 'button-height':
                width = props.get('button-width', self.minimize_button.width)
                space = props.get('space', 0)

                x_offset = 0

                self.minimize_button.width, self.minimize_button.height = width, value
                self.minimize_button.x = x_offset
                self.minimize_button.y = 0.
                x_offset += width + space

                self.maximize_button.width, self.maximize_button.height = width, value
                self.maximize_button.x = x_offset
                self.maximize_button.y = 0.
                x_offset += width + space

                self.close_button.width, self.close_button.height = width, value
                self.close_button.x = x_offset
                self.close_button.y = 0.
            elif key in ('button-width', 'space'):
                continue
            else:
                remaining_props[key] = value

        if len(remaining_props) > 0:
            return super(MinMaxCloseButtons, self).update_style_properties(remaining_props)


def buttons_decorator(controller):

    # retrieve frontend so that our buttons can act on it
    frontend = controller.frontend

    buttons_mode = controller.config.get('viewport_buttons_mode', 'desktop')
    is_desktop = buttons_mode == 'desktop'
    if is_desktop:
        buttons = MinMaxCloseButtons(frontend)
    else:
        buttons = HomeButton(frontend)
    buttons.visible = True
    controller.widget.add(buttons, pgm.DRAWABLE_NEAR)
    buttons.update_style_properties(buttons.style.get_items())

    if is_desktop:
        buttons.hide()
    else:
        buttons.show()

    def _mouse_motion_cb(viewport, event):
        # don't show buttons if a player is currently focused
        if controller.has_focus() and isinstance(controller.focused_controller,
                                                 PoblesecBrowserController):
            buttons.show()
        else:
            buttons.hide()

    def on_clean(controller, handler_id):
        controller.frontend.viewport.disconnect(handler_id)
        return defer.succeed(None)

    handler_id = controller.frontend.viewport.connect('motion-notify-event',
                                                      _mouse_motion_cb)
    controller.connect('clean', on_clean, handler_id)

    return defer.succeed(None)
